/**
 * @popperjs/core v2.11.6 - MIT License
 */

(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
        typeof define === 'function' && define.amd ? define(['exports'], factory) :
            (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.Popper = {}));
}(this, (function (exports) { 'use strict';

    function getWindow(node) {
        if (node == null) {
            return window;
        }

        if (node.toString() !== '[object Window]') {
            var ownerDocument = node.ownerDocument;
            return ownerDocument ? ownerDocument.defaultView || window : window;
        }

        return node;
    }

    function isElement(node) {
        var OwnElement = getWindow(node).Element;
        return node instanceof OwnElement || node instanceof Element;
    }

    function isHTMLElement(node) {
        var OwnElement = getWindow(node).HTMLElement;
        return node instanceof OwnElement || node instanceof HTMLElement;
    }

    function isShadowRoot(node) {
        // IE 11 has no ShadowRoot
        if (typeof ShadowRoot === 'undefined') {
            return false;
        }

        var OwnElement = getWindow(node).ShadowRoot;
        return node instanceof OwnElement || node instanceof ShadowRoot;
    }

    var max = Math.max;
    var min = Math.min;
    var round = Math.round;

    function getUAString() {
        var uaData = navigator.userAgentData;

        if (uaData != null && uaData.brands) {
            return uaData.brands.map(function (item) {
                return item.brand + "/" + item.version;
            }).join(' ');
        }

        return navigator.userAgent;
    }

    function isLayoutViewport() {
        return !/^((?!chrome|android).)*safari/i.test(getUAString());
    }

    function getBoundingClientRect(element, includeScale, isFixedStrategy) {
        if (includeScale === void 0) {
            includeScale = false;
        }

        if (isFixedStrategy === void 0) {
            isFixedStrategy = false;
        }

        var clientRect = element.getBoundingClientRect();
        var scaleX = 1;
        var scaleY = 1;

        if (includeScale && isHTMLElement(element)) {
            scaleX = element.offsetWidth > 0 ? round(clientRect.width) / element.offsetWidth || 1 : 1;
            scaleY = element.offsetHeight > 0 ? round(clientRect.height) / element.offsetHeight || 1 : 1;
        }

        var _ref = isElement(element) ? getWindow(element) : window,
            visualViewport = _ref.visualViewport;

        var addVisualOffsets = !isLayoutViewport() && isFixedStrategy;
        var x = (clientRect.left + (addVisualOffsets && visualViewport ? visualViewport.offsetLeft : 0)) / scaleX;
        var y = (clientRect.top + (addVisualOffsets && visualViewport ? visualViewport.offsetTop : 0)) / scaleY;
        var width = clientRect.width / scaleX;
        var height = clientRect.height / scaleY;
        return {
            width: width,
            height: height,
            top: y,
            right: x + width,
            bottom: y + height,
            left: x,
            x: x,
            y: y
        };
    }

    function getWindowScroll(node) {
        var win = getWindow(node);
        var scrollLeft = win.pageXOffset;
        var scrollTop = win.pageYOffset;
        return {
            scrollLeft: scrollLeft,
            scrollTop: scrollTop
        };
    }

    function getHTMLElementScroll(element) {
        return {
            scrollLeft: element.scrollLeft,
            scrollTop: element.scrollTop
        };
    }

    function getNodeScroll(node) {
        if (node === getWindow(node) || !isHTMLElement(node)) {
            return getWindowScroll(node);
        } else {
            return getHTMLElementScroll(node);
        }
    }

    function getNodeName(element) {
        return element ? (element.nodeName || '').toLowerCase() : null;
    }

    function getDocumentElement(element) {
        // $FlowFixMe[incompatible-return]: assume body is always available
        return ((isElement(element) ? element.ownerDocument : // $FlowFixMe[prop-missing]
            element.document) || window.document).documentElement;
    }

    function getWindowScrollBarX(element) {
        // If <html> has a CSS width greater than the viewport, then this will be
        // incorrect for RTL.
        // Popper 1 is broken in this case and never had a bug report so let's assume
        // it's not an issue. I don't think anyone ever specifies width on <html>
        // anyway.
        // Browsers where the left scrollbar doesn't cause an issue report `0` for
        // this (e.g. Edge 2019, IE11, Safari)
        return getBoundingClientRect(getDocumentElement(element)).left + getWindowScroll(element).scrollLeft;
    }

    function getComputedStyle(element) {
        return getWindow(element).getComputedStyle(element);
    }

    function isScrollParent(element) {
        // Firefox wants us to check `-x` and `-y` variations as well
        var _getComputedStyle = getComputedStyle(element),
            overflow = _getComputedStyle.overflow,
            overflowX = _getComputedStyle.overflowX,
            overflowY = _getComputedStyle.overflowY;

        return /auto|scroll|overlay|hidden/.test(overflow + overflowY + overflowX);
    }

    function isElementScaled(element) {
        var rect = element.getBoundingClientRect();
        var scaleX = round(rect.width) / element.offsetWidth || 1;
        var scaleY = round(rect.height) / element.offsetHeight || 1;
        return scaleX !== 1 || scaleY !== 1;
    } // Returns the composite rect of an element relative to its offsetParent.
    // Composite means it takes into account transforms as well as layout.


    function getCompositeRect(elementOrVirtualElement, offsetParent, isFixed) {
        if (isFixed === void 0) {
            isFixed = false;
        }

        var isOffsetParentAnElement = isHTMLElement(offsetParent);
        var offsetParentIsScaled = isHTMLElement(offsetParent) && isElementScaled(offsetParent);
        var documentElement = getDocumentElement(offsetParent);
        var rect = getBoundingClientRect(elementOrVirtualElement, offsetParentIsScaled, isFixed);
        var scroll = {
            scrollLeft: 0,
            scrollTop: 0
        };
        var offsets = {
            x: 0,
            y: 0
        };

        if (isOffsetParentAnElement || !isOffsetParentAnElement && !isFixed) {
            if (getNodeName(offsetParent) !== 'body' || // https://github.com/popperjs/popper-core/issues/1078
                isScrollParent(documentElement)) {
                scroll = getNodeScroll(offsetParent);
            }

            if (isHTMLElement(offsetParent)) {
                offsets = getBoundingClientRect(offsetParent, true);
                offsets.x += offsetParent.clientLeft;
                offsets.y += offsetParent.clientTop;
            } else if (documentElement) {
                offsets.x = getWindowScrollBarX(documentElement);
            }
        }

        return {
            x: rect.left + scroll.scrollLeft - offsets.x,
            y: rect.top + scroll.scrollTop - offsets.y,
            width: rect.width,
            height: rect.height
        };
    }

    // means it doesn't take into account transforms.

    function getLayoutRect(element) {
        var clientRect = getBoundingClientRect(element); // Use the clientRect sizes if it's not been transformed.
        // Fixes https://github.com/popperjs/popper-core/issues/1223

        var width = element.offsetWidth;
        var height = element.offsetHeight;

        if (Math.abs(clientRect.width - width) <= 1) {
            width = clientRect.width;
        }

        if (Math.abs(clientRect.height - height) <= 1) {
            height = clientRect.height;
        }

        return {
            x: element.offsetLeft,
            y: element.offsetTop,
            width: width,
            height: height
        };
    }

    function getParentNode(element) {
        if (getNodeName(element) === 'html') {
            return element;
        }

        return (// this is a quicker (but less type safe) way to save quite some bytes from the bundle
            // $FlowFixMe[incompatible-return]
            // $FlowFixMe[prop-missing]
            element.assignedSlot || // step into the shadow DOM of the parent of a slotted node
            element.parentNode || ( // DOM Element detected
                isShadowRoot(element) ? element.host : null) || // ShadowRoot detected
            // $FlowFixMe[incompatible-call]: HTMLElement is a Node
            getDocumentElement(element) // fallback

        );
    }

    function getScrollParent(node) {
        if (['html', 'body', '#document'].indexOf(getNodeName(node)) >= 0) {
            // $FlowFixMe[incompatible-return]: assume body is always available
            return node.ownerDocument.body;
        }

        if (isHTMLElement(node) && isScrollParent(node)) {
            return node;
        }

        return getScrollParent(getParentNode(node));
    }

    /*
  given a DOM element, return the list of all scroll parents, up the list of ancesors
  until we get to the top window object. This list is what we attach scroll listeners
  to, because if any of these parent elements scroll, we'll need to re-calculate the
  reference element's position.
  */

    function listScrollParents(element, list) {
        var _element$ownerDocumen;

        if (list === void 0) {
            list = [];
        }

        var scrollParent = getScrollParent(element);
        var isBody = scrollParent === ((_element$ownerDocumen = element.ownerDocument) == null ? void 0 : _element$ownerDocumen.body);
        var win = getWindow(scrollParent);
        var target = isBody ? [win].concat(win.visualViewport || [], isScrollParent(scrollParent) ? scrollParent : []) : scrollParent;
        var updatedList = list.concat(target);
        return isBody ? updatedList : // $FlowFixMe[incompatible-call]: isBody tells us target will be an HTMLElement here
            updatedList.concat(listScrollParents(getParentNode(target)));
    }

    function isTableElement(element) {
        return ['table', 'td', 'th'].indexOf(getNodeName(element)) >= 0;
    }

    function getTrueOffsetParent(element) {
        if (!isHTMLElement(element) || // https://github.com/popperjs/popper-core/issues/837
            getComputedStyle(element).position === 'fixed') {
            return null;
        }

        return element.offsetParent;
    } // `.offsetParent` reports `null` for fixed elements, while absolute elements
    // return the containing block


    function getContainingBlock(element) {
        var isFirefox = /firefox/i.test(getUAString());
        var isIE = /Trident/i.test(getUAString());

        if (isIE && isHTMLElement(element)) {
            // In IE 9, 10 and 11 fixed elements containing block is always established by the viewport
            var elementCss = getComputedStyle(element);

            if (elementCss.position === 'fixed') {
                return null;
            }
        }

        var currentNode = getParentNode(element);

        if (isShadowRoot(currentNode)) {
            currentNode = currentNode.host;
        }

        while (isHTMLElement(currentNode) && ['html', 'body'].indexOf(getNodeName(currentNode)) < 0) {
            var css = getComputedStyle(currentNode); // This is non-exhaustive but covers the most common CSS properties that
            // create a containing block.
            // https://developer.mozilla.org/en-US/docs/Web/CSS/Containing_block#identifying_the_containing_block

            if (css.transform !== 'none' || css.perspective !== 'none' || css.contain === 'paint' || ['transform', 'perspective'].indexOf(css.willChange) !== -1 || isFirefox && css.willChange === 'filter' || isFirefox && css.filter && css.filter !== 'none') {
                return currentNode;
            } else {
                currentNode = currentNode.parentNode;
            }
        }

        return null;
    } // Gets the closest ancestor positioned element. Handles some edge cases,
    // such as table ancestors and cross browser bugs.


    function getOffsetParent(element) {
        var window = getWindow(element);
        var offsetParent = getTrueOffsetParent(element);

        while (offsetParent && isTableElement(offsetParent) && getComputedStyle(offsetParent).position === 'static') {
            offsetParent = getTrueOffsetParent(offsetParent);
        }

        if (offsetParent && (getNodeName(offsetParent) === 'html' || getNodeName(offsetParent) === 'body' && getComputedStyle(offsetParent).position === 'static')) {
            return window;
        }

        return offsetParent || getContainingBlock(element) || window;
    }

    var top = 'top';
    var bottom = 'bottom';
    var right = 'right';
    var left = 'left';
    var auto = 'auto';
    var basePlacements = [top, bottom, right, left];
    var start = 'start';
    var end = 'end';
    var clippingParents = 'clippingParents';
    var viewport = 'viewport';
    var popper = 'popper';
    var reference = 'reference';
    var variationPlacements = /*#__PURE__*/basePlacements.reduce(function (acc, placement) {
        return acc.concat([placement + "-" + start, placement + "-" + end]);
    }, []);
    var placements = /*#__PURE__*/[].concat(basePlacements, [auto]).reduce(function (acc, placement) {
        return acc.concat([placement, placement + "-" + start, placement + "-" + end]);
    }, []); // modifiers that need to read the DOM

    var beforeRead = 'beforeRead';
    var read = 'read';
    var afterRead = 'afterRead'; // pure-logic modifiers

    var beforeMain = 'beforeMain';
    var main = 'main';
    var afterMain = 'afterMain'; // modifier with the purpose to write to the DOM (or write into a framework state)

    var beforeWrite = 'beforeWrite';
    var write = 'write';
    var afterWrite = 'afterWrite';
    var modifierPhases = [beforeRead, read, afterRead, beforeMain, main, afterMain, beforeWrite, write, afterWrite];

    function order(modifiers) {
        var map = new Map();
        var visited = new Set();
        var result = [];
        modifiers.forEach(function (modifier) {
            map.set(modifier.name, modifier);
        }); // On visiting object, check for its dependencies and visit them recursively

        function sort(modifier) {
            visited.add(modifier.name);
            var requires = [].concat(modifier.requires || [], modifier.requiresIfExists || []);
            requires.forEach(function (dep) {
                if (!visited.has(dep)) {
                    var depModifier = map.get(dep);

                    if (depModifier) {
                        sort(depModifier);
                    }
                }
            });
            result.push(modifier);
        }

        modifiers.forEach(function (modifier) {
            if (!visited.has(modifier.name)) {
                // check for visited object
                sort(modifier);
            }
        });
        return result;
    }

    function orderModifiers(modifiers) {
        // order based on dependencies
        var orderedModifiers = order(modifiers); // order based on phase

        return modifierPhases.reduce(function (acc, phase) {
            return acc.concat(orderedModifiers.filter(function (modifier) {
                return modifier.phase === phase;
            }));
        }, []);
    }

    function debounce(fn) {
        var pending;
        return function () {
            if (!pending) {
                pending = new Promise(function (resolve) {
                    Promise.resolve().then(function () {
                        pending = undefined;
                        resolve(fn());
                    });
                });
            }

            return pending;
        };
    }

    function format(str) {
        for (var _len = arguments.length, args = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
            args[_key - 1] = arguments[_key];
        }

        return [].concat(args).reduce(function (p, c) {
            return p.replace(/%s/, c);
        }, str);
    }

    var INVALID_MODIFIER_ERROR = 'Popper: modifier "%s" provided an invalid %s property, expected %s but got %s';
    var MISSING_DEPENDENCY_ERROR = 'Popper: modifier "%s" requires "%s", but "%s" modifier is not available';
    var VALID_PROPERTIES = ['name', 'enabled', 'phase', 'fn', 'effect', 'requires', 'options'];
    function validateModifiers(modifiers) {
        modifiers.forEach(function (modifier) {
            [].concat(Object.keys(modifier), VALID_PROPERTIES) // IE11-compatible replacement for `new Set(iterable)`
                .filter(function (value, index, self) {
                    return self.indexOf(value) === index;
                }).forEach(function (key) {
                switch (key) {
                    case 'name':
                        if (typeof modifier.name !== 'string') {
                            console.error(format(INVALID_MODIFIER_ERROR, String(modifier.name), '"name"', '"string"', "\"" + String(modifier.name) + "\""));
                        }

                        break;

                    case 'enabled':
                        if (typeof modifier.enabled !== 'boolean') {
                            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"enabled"', '"boolean"', "\"" + String(modifier.enabled) + "\""));
                        }

                        break;

                    case 'phase':
                        if (modifierPhases.indexOf(modifier.phase) < 0) {
                            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"phase"', "either " + modifierPhases.join(', '), "\"" + String(modifier.phase) + "\""));
                        }

                        break;

                    case 'fn':
                        if (typeof modifier.fn !== 'function') {
                            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"fn"', '"function"', "\"" + String(modifier.fn) + "\""));
                        }

                        break;

                    case 'effect':
                        if (modifier.effect != null && typeof modifier.effect !== 'function') {
                            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"effect"', '"function"', "\"" + String(modifier.fn) + "\""));
                        }

                        break;

                    case 'requires':
                        if (modifier.requires != null && !Array.isArray(modifier.requires)) {
                            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"requires"', '"array"', "\"" + String(modifier.requires) + "\""));
                        }

                        break;

                    case 'requiresIfExists':
                        if (!Array.isArray(modifier.requiresIfExists)) {
                            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"requiresIfExists"', '"array"', "\"" + String(modifier.requiresIfExists) + "\""));
                        }

                        break;

                    case 'options':
                    case 'data':
                        break;

                    default:
                        console.error("PopperJS: an invalid property has been provided to the \"" + modifier.name + "\" modifier, valid properties are " + VALID_PROPERTIES.map(function (s) {
                            return "\"" + s + "\"";
                        }).join(', ') + "; but \"" + key + "\" was provided.");
                }

                modifier.requires && modifier.requires.forEach(function (requirement) {
                    if (modifiers.find(function (mod) {
                        return mod.name === requirement;
                    }) == null) {
                        console.error(format(MISSING_DEPENDENCY_ERROR, String(modifier.name), requirement, requirement));
                    }
                });
            });
        });
    }

    function uniqueBy(arr, fn) {
        var identifiers = new Set();
        return arr.filter(function (item) {
            var identifier = fn(item);

            if (!identifiers.has(identifier)) {
                identifiers.add(identifier);
                return true;
            }
        });
    }

    function getBasePlacement(placement) {
        return placement.split('-')[0];
    }

    function mergeByName(modifiers) {
        var merged = modifiers.reduce(function (merged, current) {
            var existing = merged[current.name];
            merged[current.name] = existing ? Object.assign({}, existing, current, {
                options: Object.assign({}, existing.options, current.options),
                data: Object.assign({}, existing.data, current.data)
            }) : current;
            return merged;
        }, {}); // IE11 does not support Object.values

        return Object.keys(merged).map(function (key) {
            return merged[key];
        });
    }

    function getViewportRect(element, strategy) {
        var win = getWindow(element);
        var html = getDocumentElement(element);
        var visualViewport = win.visualViewport;
        var width = html.clientWidth;
        var height = html.clientHeight;
        var x = 0;
        var y = 0;

        if (visualViewport) {
            width = visualViewport.width;
            height = visualViewport.height;
            var layoutViewport = isLayoutViewport();

            if (layoutViewport || !layoutViewport && strategy === 'fixed') {
                x = visualViewport.offsetLeft;
                y = visualViewport.offsetTop;
            }
        }

        return {
            width: width,
            height: height,
            x: x + getWindowScrollBarX(element),
            y: y
        };
    }

    // of the `<html>` and `<body>` rect bounds if horizontally scrollable

    function getDocumentRect(element) {
        var _element$ownerDocumen;

        var html = getDocumentElement(element);
        var winScroll = getWindowScroll(element);
        var body = (_element$ownerDocumen = element.ownerDocument) == null ? void 0 : _element$ownerDocumen.body;
        var width = max(html.scrollWidth, html.clientWidth, body ? body.scrollWidth : 0, body ? body.clientWidth : 0);
        var height = max(html.scrollHeight, html.clientHeight, body ? body.scrollHeight : 0, body ? body.clientHeight : 0);
        var x = -winScroll.scrollLeft + getWindowScrollBarX(element);
        var y = -winScroll.scrollTop;

        if (getComputedStyle(body || html).direction === 'rtl') {
            x += max(html.clientWidth, body ? body.clientWidth : 0) - width;
        }

        return {
            width: width,
            height: height,
            x: x,
            y: y
        };
    }

    function contains(parent, child) {
        var rootNode = child.getRootNode && child.getRootNode(); // First, attempt with faster native method

        if (parent.contains(child)) {
            return true;
        } // then fallback to custom implementation with Shadow DOM support
        else if (rootNode && isShadowRoot(rootNode)) {
            var next = child;

            do {
                if (next && parent.isSameNode(next)) {
                    return true;
                } // $FlowFixMe[prop-missing]: need a better way to handle this...


                next = next.parentNode || next.host;
            } while (next);
        } // Give up, the result is false


        return false;
    }

    function rectToClientRect(rect) {
        return Object.assign({}, rect, {
            left: rect.x,
            top: rect.y,
            right: rect.x + rect.width,
            bottom: rect.y + rect.height
        });
    }

    function getInnerBoundingClientRect(element, strategy) {
        var rect = getBoundingClientRect(element, false, strategy === 'fixed');
        rect.top = rect.top + element.clientTop;
        rect.left = rect.left + element.clientLeft;
        rect.bottom = rect.top + element.clientHeight;
        rect.right = rect.left + element.clientWidth;
        rect.width = element.clientWidth;
        rect.height = element.clientHeight;
        rect.x = rect.left;
        rect.y = rect.top;
        return rect;
    }

    function getClientRectFromMixedType(element, clippingParent, strategy) {
        return clippingParent === viewport ? rectToClientRect(getViewportRect(element, strategy)) : isElement(clippingParent) ? getInnerBoundingClientRect(clippingParent, strategy) : rectToClientRect(getDocumentRect(getDocumentElement(element)));
    } // A "clipping parent" is an overflowable container with the characteristic of
    // clipping (or hiding) overflowing elements with a position different from
    // `initial`


    function getClippingParents(element) {
        var clippingParents = listScrollParents(getParentNode(element));
        var canEscapeClipping = ['absolute', 'fixed'].indexOf(getComputedStyle(element).position) >= 0;
        var clipperElement = canEscapeClipping && isHTMLElement(element) ? getOffsetParent(element) : element;

        if (!isElement(clipperElement)) {
            return [];
        } // $FlowFixMe[incompatible-return]: https://github.com/facebook/flow/issues/1414


        return clippingParents.filter(function (clippingParent) {
            return isElement(clippingParent) && contains(clippingParent, clipperElement) && getNodeName(clippingParent) !== 'body';
        });
    } // Gets the maximum area that the element is visible in due to any number of
    // clipping parents


    function getClippingRect(element, boundary, rootBoundary, strategy) {
        var mainClippingParents = boundary === 'clippingParents' ? getClippingParents(element) : [].concat(boundary);
        var clippingParents = [].concat(mainClippingParents, [rootBoundary]);
        var firstClippingParent = clippingParents[0];
        var clippingRect = clippingParents.reduce(function (accRect, clippingParent) {
            var rect = getClientRectFromMixedType(element, clippingParent, strategy);
            accRect.top = max(rect.top, accRect.top);
            accRect.right = min(rect.right, accRect.right);
            accRect.bottom = min(rect.bottom, accRect.bottom);
            accRect.left = max(rect.left, accRect.left);
            return accRect;
        }, getClientRectFromMixedType(element, firstClippingParent, strategy));
        clippingRect.width = clippingRect.right - clippingRect.left;
        clippingRect.height = clippingRect.bottom - clippingRect.top;
        clippingRect.x = clippingRect.left;
        clippingRect.y = clippingRect.top;
        return clippingRect;
    }

    function getVariation(placement) {
        return placement.split('-')[1];
    }

    function getMainAxisFromPlacement(placement) {
        return ['top', 'bottom'].indexOf(placement) >= 0 ? 'x' : 'y';
    }

    function computeOffsets(_ref) {
        var reference = _ref.reference,
            element = _ref.element,
            placement = _ref.placement;
        var basePlacement = placement ? getBasePlacement(placement) : null;
        var variation = placement ? getVariation(placement) : null;
        var commonX = reference.x + reference.width / 2 - element.width / 2;
        var commonY = reference.y + reference.height / 2 - element.height / 2;
        var offsets;

        switch (basePlacement) {
            case top:
                offsets = {
                    x: commonX,
                    y: reference.y - element.height
                };
                break;

            case bottom:
                offsets = {
                    x: commonX,
                    y: reference.y + reference.height
                };
                break;

            case right:
                offsets = {
                    x: reference.x + reference.width,
                    y: commonY
                };
                break;

            case left:
                offsets = {
                    x: reference.x - element.width,
                    y: commonY
                };
                break;

            default:
                offsets = {
                    x: reference.x,
                    y: reference.y
                };
        }

        var mainAxis = basePlacement ? getMainAxisFromPlacement(basePlacement) : null;

        if (mainAxis != null) {
            var len = mainAxis === 'y' ? 'height' : 'width';

            switch (variation) {
                case start:
                    offsets[mainAxis] = offsets[mainAxis] - (reference[len] / 2 - element[len] / 2);
                    break;

                case end:
                    offsets[mainAxis] = offsets[mainAxis] + (reference[len] / 2 - element[len] / 2);
                    break;
            }
        }

        return offsets;
    }

    function getFreshSideObject() {
        return {
            top: 0,
            right: 0,
            bottom: 0,
            left: 0
        };
    }

    function mergePaddingObject(paddingObject) {
        return Object.assign({}, getFreshSideObject(), paddingObject);
    }

    function expandToHashMap(value, keys) {
        return keys.reduce(function (hashMap, key) {
            hashMap[key] = value;
            return hashMap;
        }, {});
    }

    function detectOverflow(state, options) {
        if (options === void 0) {
            options = {};
        }

        var _options = options,
            _options$placement = _options.placement,
            placement = _options$placement === void 0 ? state.placement : _options$placement,
            _options$strategy = _options.strategy,
            strategy = _options$strategy === void 0 ? state.strategy : _options$strategy,
            _options$boundary = _options.boundary,
            boundary = _options$boundary === void 0 ? clippingParents : _options$boundary,
            _options$rootBoundary = _options.rootBoundary,
            rootBoundary = _options$rootBoundary === void 0 ? viewport : _options$rootBoundary,
            _options$elementConte = _options.elementContext,
            elementContext = _options$elementConte === void 0 ? popper : _options$elementConte,
            _options$altBoundary = _options.altBoundary,
            altBoundary = _options$altBoundary === void 0 ? false : _options$altBoundary,
            _options$padding = _options.padding,
            padding = _options$padding === void 0 ? 0 : _options$padding;
        var paddingObject = mergePaddingObject(typeof padding !== 'number' ? padding : expandToHashMap(padding, basePlacements));
        var altContext = elementContext === popper ? reference : popper;
        var popperRect = state.rects.popper;
        var element = state.elements[altBoundary ? altContext : elementContext];
        var clippingClientRect = getClippingRect(isElement(element) ? element : element.contextElement || getDocumentElement(state.elements.popper), boundary, rootBoundary, strategy);
        var referenceClientRect = getBoundingClientRect(state.elements.reference);
        var popperOffsets = computeOffsets({
            reference: referenceClientRect,
            element: popperRect,
            strategy: 'absolute',
            placement: placement
        });
        var popperClientRect = rectToClientRect(Object.assign({}, popperRect, popperOffsets));
        var elementClientRect = elementContext === popper ? popperClientRect : referenceClientRect; // positive = overflowing the clipping rect
        // 0 or negative = within the clipping rect

        var overflowOffsets = {
            top: clippingClientRect.top - elementClientRect.top + paddingObject.top,
            bottom: elementClientRect.bottom - clippingClientRect.bottom + paddingObject.bottom,
            left: clippingClientRect.left - elementClientRect.left + paddingObject.left,
            right: elementClientRect.right - clippingClientRect.right + paddingObject.right
        };
        var offsetData = state.modifiersData.offset; // Offsets can be applied only to the popper element

        if (elementContext === popper && offsetData) {
            var offset = offsetData[placement];
            Object.keys(overflowOffsets).forEach(function (key) {
                var multiply = [right, bottom].indexOf(key) >= 0 ? 1 : -1;
                var axis = [top, bottom].indexOf(key) >= 0 ? 'y' : 'x';
                overflowOffsets[key] += offset[axis] * multiply;
            });
        }

        return overflowOffsets;
    }

    var INVALID_ELEMENT_ERROR = 'Popper: Invalid reference or popper argument provided. They must be either a DOM element or virtual element.';
    var INFINITE_LOOP_ERROR = 'Popper: An infinite loop in the modifiers cycle has been detected! The cycle has been interrupted to prevent a browser crash.';
    var DEFAULT_OPTIONS = {
        placement: 'bottom',
        modifiers: [],
        strategy: 'absolute'
    };

    function areValidElements() {
        for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
            args[_key] = arguments[_key];
        }

        return !args.some(function (element) {
            return !(element && typeof element.getBoundingClientRect === 'function');
        });
    }

    function popperGenerator(generatorOptions) {
        if (generatorOptions === void 0) {
            generatorOptions = {};
        }

        var _generatorOptions = generatorOptions,
            _generatorOptions$def = _generatorOptions.defaultModifiers,
            defaultModifiers = _generatorOptions$def === void 0 ? [] : _generatorOptions$def,
            _generatorOptions$def2 = _generatorOptions.defaultOptions,
            defaultOptions = _generatorOptions$def2 === void 0 ? DEFAULT_OPTIONS : _generatorOptions$def2;
        return function createPopper(reference, popper, options) {
            if (options === void 0) {
                options = defaultOptions;
            }

            var state = {
                placement: 'bottom',
                orderedModifiers: [],
                options: Object.assign({}, DEFAULT_OPTIONS, defaultOptions),
                modifiersData: {},
                elements: {
                    reference: reference,
                    popper: popper
                },
                attributes: {},
                styles: {}
            };
            var effectCleanupFns = [];
            var isDestroyed = false;
            var instance = {
                state: state,
                setOptions: function setOptions(setOptionsAction) {
                    var options = typeof setOptionsAction === 'function' ? setOptionsAction(state.options) : setOptionsAction;
                    cleanupModifierEffects();
                    state.options = Object.assign({}, defaultOptions, state.options, options);
                    state.scrollParents = {
                        reference: isElement(reference) ? listScrollParents(reference) : reference.contextElement ? listScrollParents(reference.contextElement) : [],
                        popper: listScrollParents(popper)
                    }; // Orders the modifiers based on their dependencies and `phase`
                    // properties

                    var orderedModifiers = orderModifiers(mergeByName([].concat(defaultModifiers, state.options.modifiers))); // Strip out disabled modifiers

                    state.orderedModifiers = orderedModifiers.filter(function (m) {
                        return m.enabled;
                    }); // Validate the provided modifiers so that the consumer will get warned
                    // if one of the modifiers is invalid for any reason

                    {
                        var modifiers = uniqueBy([].concat(orderedModifiers, state.options.modifiers), function (_ref) {
                            var name = _ref.name;
                            return name;
                        });
                        validateModifiers(modifiers);

                        if (getBasePlacement(state.options.placement) === auto) {
                            var flipModifier = state.orderedModifiers.find(function (_ref2) {
                                var name = _ref2.name;
                                return name === 'flip';
                            });

                            if (!flipModifier) {
                                console.error(['Popper: "auto" placements require the "flip" modifier be', 'present and enabled to work.'].join(' '));
                            }
                        }

                        var _getComputedStyle = getComputedStyle(popper),
                            marginTop = _getComputedStyle.marginTop,
                            marginRight = _getComputedStyle.marginRight,
                            marginBottom = _getComputedStyle.marginBottom,
                            marginLeft = _getComputedStyle.marginLeft; // We no longer take into account `margins` on the popper, and it can
                        // cause bugs with positioning, so we'll warn the consumer


                        if ([marginTop, marginRight, marginBottom, marginLeft].some(function (margin) {
                            return parseFloat(margin);
                        })) {
                            console.warn(['Popper: CSS "margin" styles cannot be used to apply padding', 'between the popper and its reference element or boundary.', 'To replicate margin, use the `offset` modifier, as well as', 'the `padding` option in the `preventOverflow` and `flip`', 'modifiers.'].join(' '));
                        }
                    }

                    runModifierEffects();
                    return instance.update();
                },
                // Sync update – it will always be executed, even if not necessary. This
                // is useful for low frequency updates where sync behavior simplifies the
                // logic.
                // For high frequency updates (e.g. `resize` and `scroll` events), always
                // prefer the async Popper#update method
                forceUpdate: function forceUpdate() {
                    if (isDestroyed) {
                        return;
                    }

                    var _state$elements = state.elements,
                        reference = _state$elements.reference,
                        popper = _state$elements.popper; // Don't proceed if `reference` or `popper` are not valid elements
                    // anymore

                    if (!areValidElements(reference, popper)) {
                        {
                            console.error(INVALID_ELEMENT_ERROR);
                        }

                        return;
                    } // Store the reference and popper rects to be read by modifiers


                    state.rects = {
                        reference: getCompositeRect(reference, getOffsetParent(popper), state.options.strategy === 'fixed'),
                        popper: getLayoutRect(popper)
                    }; // Modifiers have the ability to reset the current update cycle. The
                    // most common use case for this is the `flip` modifier changing the
                    // placement, which then needs to re-run all the modifiers, because the
                    // logic was previously ran for the previous placement and is therefore
                    // stale/incorrect

                    state.reset = false;
                    state.placement = state.options.placement; // On each update cycle, the `modifiersData` property for each modifier
                    // is filled with the initial data specified by the modifier. This means
                    // it doesn't persist and is fresh on each update.
                    // To ensure persistent data, use `${name}#persistent`

                    state.orderedModifiers.forEach(function (modifier) {
                        return state.modifiersData[modifier.name] = Object.assign({}, modifier.data);
                    });
                    var __debug_loops__ = 0;

                    for (var index = 0; index < state.orderedModifiers.length; index++) {
                        {
                            __debug_loops__ += 1;

                            if (__debug_loops__ > 100) {
                                console.error(INFINITE_LOOP_ERROR);
                                break;
                            }
                        }

                        if (state.reset === true) {
                            state.reset = false;
                            index = -1;
                            continue;
                        }

                        var _state$orderedModifie = state.orderedModifiers[index],
                            fn = _state$orderedModifie.fn,
                            _state$orderedModifie2 = _state$orderedModifie.options,
                            _options = _state$orderedModifie2 === void 0 ? {} : _state$orderedModifie2,
                            name = _state$orderedModifie.name;

                        if (typeof fn === 'function') {
                            state = fn({
                                state: state,
                                options: _options,
                                name: name,
                                instance: instance
                            }) || state;
                        }
                    }
                },
                // Async and optimistically optimized update – it will not be executed if
                // not necessary (debounced to run at most once-per-tick)
                update: debounce(function () {
                    return new Promise(function (resolve) {
                        instance.forceUpdate();
                        resolve(state);
                    });
                }),
                destroy: function destroy() {
                    cleanupModifierEffects();
                    isDestroyed = true;
                }
            };

            if (!areValidElements(reference, popper)) {
                {
                    console.error(INVALID_ELEMENT_ERROR);
                }

                return instance;
            }

            instance.setOptions(options).then(function (state) {
                if (!isDestroyed && options.onFirstUpdate) {
                    options.onFirstUpdate(state);
                }
            }); // Modifiers have the ability to execute arbitrary code before the first
            // update cycle runs. They will be executed in the same order as the update
            // cycle. This is useful when a modifier adds some persistent data that
            // other modifiers need to use, but the modifier is run after the dependent
            // one.

            function runModifierEffects() {
                state.orderedModifiers.forEach(function (_ref3) {
                    var name = _ref3.name,
                        _ref3$options = _ref3.options,
                        options = _ref3$options === void 0 ? {} : _ref3$options,
                        effect = _ref3.effect;

                    if (typeof effect === 'function') {
                        var cleanupFn = effect({
                            state: state,
                            name: name,
                            instance: instance,
                            options: options
                        });

                        var noopFn = function noopFn() {};

                        effectCleanupFns.push(cleanupFn || noopFn);
                    }
                });
            }

            function cleanupModifierEffects() {
                effectCleanupFns.forEach(function (fn) {
                    return fn();
                });
                effectCleanupFns = [];
            }

            return instance;
        };
    }

    var passive = {
        passive: true
    };

    function effect$2(_ref) {
        var state = _ref.state,
            instance = _ref.instance,
            options = _ref.options;
        var _options$scroll = options.scroll,
            scroll = _options$scroll === void 0 ? true : _options$scroll,
            _options$resize = options.resize,
            resize = _options$resize === void 0 ? true : _options$resize;
        var window = getWindow(state.elements.popper);
        var scrollParents = [].concat(state.scrollParents.reference, state.scrollParents.popper);

        if (scroll) {
            scrollParents.forEach(function (scrollParent) {
                scrollParent.addEventListener('scroll', instance.update, passive);
            });
        }

        if (resize) {
            window.addEventListener('resize', instance.update, passive);
        }

        return function () {
            if (scroll) {
                scrollParents.forEach(function (scrollParent) {
                    scrollParent.removeEventListener('scroll', instance.update, passive);
                });
            }

            if (resize) {
                window.removeEventListener('resize', instance.update, passive);
            }
        };
    } // eslint-disable-next-line import/no-unused-modules


    var eventListeners = {
        name: 'eventListeners',
        enabled: true,
        phase: 'write',
        fn: function fn() {},
        effect: effect$2,
        data: {}
    };

    function popperOffsets(_ref) {
        var state = _ref.state,
            name = _ref.name;
        // Offsets are the actual position the popper needs to have to be
        // properly positioned near its reference element
        // This is the most basic placement, and will be adjusted by
        // the modifiers in the next step
        state.modifiersData[name] = computeOffsets({
            reference: state.rects.reference,
            element: state.rects.popper,
            strategy: 'absolute',
            placement: state.placement
        });
    } // eslint-disable-next-line import/no-unused-modules


    var popperOffsets$1 = {
        name: 'popperOffsets',
        enabled: true,
        phase: 'read',
        fn: popperOffsets,
        data: {}
    };

    var unsetSides = {
        top: 'auto',
        right: 'auto',
        bottom: 'auto',
        left: 'auto'
    }; // Round the offsets to the nearest suitable subpixel based on the DPR.
    // Zooming can change the DPR, but it seems to report a value that will
    // cleanly divide the values into the appropriate subpixels.

    function roundOffsetsByDPR(_ref) {
        var x = _ref.x,
            y = _ref.y;
        var win = window;
        var dpr = win.devicePixelRatio || 1;
        return {
            x: round(x * dpr) / dpr || 0,
            y: round(y * dpr) / dpr || 0
        };
    }

    function mapToStyles(_ref2) {
        var _Object$assign2;

        var popper = _ref2.popper,
            popperRect = _ref2.popperRect,
            placement = _ref2.placement,
            variation = _ref2.variation,
            offsets = _ref2.offsets,
            position = _ref2.position,
            gpuAcceleration = _ref2.gpuAcceleration,
            adaptive = _ref2.adaptive,
            roundOffsets = _ref2.roundOffsets,
            isFixed = _ref2.isFixed;
        var _offsets$x = offsets.x,
            x = _offsets$x === void 0 ? 0 : _offsets$x,
            _offsets$y = offsets.y,
            y = _offsets$y === void 0 ? 0 : _offsets$y;

        var _ref3 = typeof roundOffsets === 'function' ? roundOffsets({
            x: x,
            y: y
        }) : {
            x: x,
            y: y
        };

        x = _ref3.x;
        y = _ref3.y;
        var hasX = offsets.hasOwnProperty('x');
        var hasY = offsets.hasOwnProperty('y');
        var sideX = left;
        var sideY = top;
        var win = window;

        if (adaptive) {
            var offsetParent = getOffsetParent(popper);
            var heightProp = 'clientHeight';
            var widthProp = 'clientWidth';

            if (offsetParent === getWindow(popper)) {
                offsetParent = getDocumentElement(popper);

                if (getComputedStyle(offsetParent).position !== 'static' && position === 'absolute') {
                    heightProp = 'scrollHeight';
                    widthProp = 'scrollWidth';
                }
            } // $FlowFixMe[incompatible-cast]: force type refinement, we compare offsetParent with window above, but Flow doesn't detect it


            offsetParent = offsetParent;

            if (placement === top || (placement === left || placement === right) && variation === end) {
                sideY = bottom;
                var offsetY = isFixed && offsetParent === win && win.visualViewport ? win.visualViewport.height : // $FlowFixMe[prop-missing]
                    offsetParent[heightProp];
                y -= offsetY - popperRect.height;
                y *= gpuAcceleration ? 1 : -1;
            }

            if (placement === left || (placement === top || placement === bottom) && variation === end) {
                sideX = right;
                var offsetX = isFixed && offsetParent === win && win.visualViewport ? win.visualViewport.width : // $FlowFixMe[prop-missing]
                    offsetParent[widthProp];
                x -= offsetX - popperRect.width;
                x *= gpuAcceleration ? 1 : -1;
            }
        }

        var commonStyles = Object.assign({
            position: position
        }, adaptive && unsetSides);

        var _ref4 = roundOffsets === true ? roundOffsetsByDPR({
            x: x,
            y: y
        }) : {
            x: x,
            y: y
        };

        x = _ref4.x;
        y = _ref4.y;

        if (gpuAcceleration) {
            var _Object$assign;

            return Object.assign({}, commonStyles, (_Object$assign = {}, _Object$assign[sideY] = hasY ? '0' : '', _Object$assign[sideX] = hasX ? '0' : '', _Object$assign.transform = (win.devicePixelRatio || 1) <= 1 ? "translate(" + x + "px, " + y + "px)" : "translate3d(" + x + "px, " + y + "px, 0)", _Object$assign));
        }

        return Object.assign({}, commonStyles, (_Object$assign2 = {}, _Object$assign2[sideY] = hasY ? y + "px" : '', _Object$assign2[sideX] = hasX ? x + "px" : '', _Object$assign2.transform = '', _Object$assign2));
    }

    function computeStyles(_ref5) {
        var state = _ref5.state,
            options = _ref5.options;
        var _options$gpuAccelerat = options.gpuAcceleration,
            gpuAcceleration = _options$gpuAccelerat === void 0 ? true : _options$gpuAccelerat,
            _options$adaptive = options.adaptive,
            adaptive = _options$adaptive === void 0 ? true : _options$adaptive,
            _options$roundOffsets = options.roundOffsets,
            roundOffsets = _options$roundOffsets === void 0 ? true : _options$roundOffsets;

        {
            var transitionProperty = getComputedStyle(state.elements.popper).transitionProperty || '';

            if (adaptive && ['transform', 'top', 'right', 'bottom', 'left'].some(function (property) {
                return transitionProperty.indexOf(property) >= 0;
            })) {
                console.warn(['Popper: Detected CSS transitions on at least one of the following', 'CSS properties: "transform", "top", "right", "bottom", "left".', '\n\n', 'Disable the "computeStyles" modifier\'s `adaptive` option to allow', 'for smooth transitions, or remove these properties from the CSS', 'transition declaration on the popper element if only transitioning', 'opacity or background-color for example.', '\n\n', 'We recommend using the popper element as a wrapper around an inner', 'element that can have any CSS property transitioned for animations.'].join(' '));
            }
        }

        var commonStyles = {
            placement: getBasePlacement(state.placement),
            variation: getVariation(state.placement),
            popper: state.elements.popper,
            popperRect: state.rects.popper,
            gpuAcceleration: gpuAcceleration,
            isFixed: state.options.strategy === 'fixed'
        };

        if (state.modifiersData.popperOffsets != null) {
            state.styles.popper = Object.assign({}, state.styles.popper, mapToStyles(Object.assign({}, commonStyles, {
                offsets: state.modifiersData.popperOffsets,
                position: state.options.strategy,
                adaptive: adaptive,
                roundOffsets: roundOffsets
            })));
        }

        if (state.modifiersData.arrow != null) {
            state.styles.arrow = Object.assign({}, state.styles.arrow, mapToStyles(Object.assign({}, commonStyles, {
                offsets: state.modifiersData.arrow,
                position: 'absolute',
                adaptive: false,
                roundOffsets: roundOffsets
            })));
        }

        state.attributes.popper = Object.assign({}, state.attributes.popper, {
            'data-popper-placement': state.placement
        });
    } // eslint-disable-next-line import/no-unused-modules


    var computeStyles$1 = {
        name: 'computeStyles',
        enabled: true,
        phase: 'beforeWrite',
        fn: computeStyles,
        data: {}
    };

    // and applies them to the HTMLElements such as popper and arrow

    function applyStyles(_ref) {
        var state = _ref.state;
        Object.keys(state.elements).forEach(function (name) {
            var style = state.styles[name] || {};
            var attributes = state.attributes[name] || {};
            var element = state.elements[name]; // arrow is optional + virtual elements

            if (!isHTMLElement(element) || !getNodeName(element)) {
                return;
            } // Flow doesn't support to extend this property, but it's the most
            // effective way to apply styles to an HTMLElement
            // $FlowFixMe[cannot-write]


            Object.assign(element.style, style);
            Object.keys(attributes).forEach(function (name) {
                var value = attributes[name];

                if (value === false) {
                    element.removeAttribute(name);
                } else {
                    element.setAttribute(name, value === true ? '' : value);
                }
            });
        });
    }

    function effect$1(_ref2) {
        var state = _ref2.state;
        var initialStyles = {
            popper: {
                position: state.options.strategy,
                left: '0',
                top: '0',
                margin: '0'
            },
            arrow: {
                position: 'absolute'
            },
            reference: {}
        };
        Object.assign(state.elements.popper.style, initialStyles.popper);
        state.styles = initialStyles;

        if (state.elements.arrow) {
            Object.assign(state.elements.arrow.style, initialStyles.arrow);
        }

        return function () {
            Object.keys(state.elements).forEach(function (name) {
                var element = state.elements[name];
                var attributes = state.attributes[name] || {};
                var styleProperties = Object.keys(state.styles.hasOwnProperty(name) ? state.styles[name] : initialStyles[name]); // Set all values to an empty string to unset them

                var style = styleProperties.reduce(function (style, property) {
                    style[property] = '';
                    return style;
                }, {}); // arrow is optional + virtual elements

                if (!isHTMLElement(element) || !getNodeName(element)) {
                    return;
                }

                Object.assign(element.style, style);
                Object.keys(attributes).forEach(function (attribute) {
                    element.removeAttribute(attribute);
                });
            });
        };
    } // eslint-disable-next-line import/no-unused-modules


    var applyStyles$1 = {
        name: 'applyStyles',
        enabled: true,
        phase: 'write',
        fn: applyStyles,
        effect: effect$1,
        requires: ['computeStyles']
    };

    function distanceAndSkiddingToXY(placement, rects, offset) {
        var basePlacement = getBasePlacement(placement);
        var invertDistance = [left, top].indexOf(basePlacement) >= 0 ? -1 : 1;

        var _ref = typeof offset === 'function' ? offset(Object.assign({}, rects, {
                placement: placement
            })) : offset,
            skidding = _ref[0],
            distance = _ref[1];

        skidding = skidding || 0;
        distance = (distance || 0) * invertDistance;
        return [left, right].indexOf(basePlacement) >= 0 ? {
            x: distance,
            y: skidding
        } : {
            x: skidding,
            y: distance
        };
    }

    function offset(_ref2) {
        var state = _ref2.state,
            options = _ref2.options,
            name = _ref2.name;
        var _options$offset = options.offset,
            offset = _options$offset === void 0 ? [0, 0] : _options$offset;
        var data = placements.reduce(function (acc, placement) {
            acc[placement] = distanceAndSkiddingToXY(placement, state.rects, offset);
            return acc;
        }, {});
        var _data$state$placement = data[state.placement],
            x = _data$state$placement.x,
            y = _data$state$placement.y;

        if (state.modifiersData.popperOffsets != null) {
            state.modifiersData.popperOffsets.x += x;
            state.modifiersData.popperOffsets.y += y;
        }

        state.modifiersData[name] = data;
    } // eslint-disable-next-line import/no-unused-modules


    var offset$1 = {
        name: 'offset',
        enabled: true,
        phase: 'main',
        requires: ['popperOffsets'],
        fn: offset
    };

    var hash$1 = {
        left: 'right',
        right: 'left',
        bottom: 'top',
        top: 'bottom'
    };
    function getOppositePlacement(placement) {
        return placement.replace(/left|right|bottom|top/g, function (matched) {
            return hash$1[matched];
        });
    }

    var hash = {
        start: 'end',
        end: 'start'
    };
    function getOppositeVariationPlacement(placement) {
        return placement.replace(/start|end/g, function (matched) {
            return hash[matched];
        });
    }

    function computeAutoPlacement(state, options) {
        if (options === void 0) {
            options = {};
        }

        var _options = options,
            placement = _options.placement,
            boundary = _options.boundary,
            rootBoundary = _options.rootBoundary,
            padding = _options.padding,
            flipVariations = _options.flipVariations,
            _options$allowedAutoP = _options.allowedAutoPlacements,
            allowedAutoPlacements = _options$allowedAutoP === void 0 ? placements : _options$allowedAutoP;
        var variation = getVariation(placement);
        var placements$1 = variation ? flipVariations ? variationPlacements : variationPlacements.filter(function (placement) {
            return getVariation(placement) === variation;
        }) : basePlacements;
        var allowedPlacements = placements$1.filter(function (placement) {
            return allowedAutoPlacements.indexOf(placement) >= 0;
        });

        if (allowedPlacements.length === 0) {
            allowedPlacements = placements$1;

            {
                console.error(['Popper: The `allowedAutoPlacements` option did not allow any', 'placements. Ensure the `placement` option matches the variation', 'of the allowed placements.', 'For example, "auto" cannot be used to allow "bottom-start".', 'Use "auto-start" instead.'].join(' '));
            }
        } // $FlowFixMe[incompatible-type]: Flow seems to have problems with two array unions...


        var overflows = allowedPlacements.reduce(function (acc, placement) {
            acc[placement] = detectOverflow(state, {
                placement: placement,
                boundary: boundary,
                rootBoundary: rootBoundary,
                padding: padding
            })[getBasePlacement(placement)];
            return acc;
        }, {});
        return Object.keys(overflows).sort(function (a, b) {
            return overflows[a] - overflows[b];
        });
    }

    function getExpandedFallbackPlacements(placement) {
        if (getBasePlacement(placement) === auto) {
            return [];
        }

        var oppositePlacement = getOppositePlacement(placement);
        return [getOppositeVariationPlacement(placement), oppositePlacement, getOppositeVariationPlacement(oppositePlacement)];
    }

    function flip(_ref) {
        var state = _ref.state,
            options = _ref.options,
            name = _ref.name;

        if (state.modifiersData[name]._skip) {
            return;
        }

        var _options$mainAxis = options.mainAxis,
            checkMainAxis = _options$mainAxis === void 0 ? true : _options$mainAxis,
            _options$altAxis = options.altAxis,
            checkAltAxis = _options$altAxis === void 0 ? true : _options$altAxis,
            specifiedFallbackPlacements = options.fallbackPlacements,
            padding = options.padding,
            boundary = options.boundary,
            rootBoundary = options.rootBoundary,
            altBoundary = options.altBoundary,
            _options$flipVariatio = options.flipVariations,
            flipVariations = _options$flipVariatio === void 0 ? true : _options$flipVariatio,
            allowedAutoPlacements = options.allowedAutoPlacements;
        var preferredPlacement = state.options.placement;
        var basePlacement = getBasePlacement(preferredPlacement);
        var isBasePlacement = basePlacement === preferredPlacement;
        var fallbackPlacements = specifiedFallbackPlacements || (isBasePlacement || !flipVariations ? [getOppositePlacement(preferredPlacement)] : getExpandedFallbackPlacements(preferredPlacement));
        var placements = [preferredPlacement].concat(fallbackPlacements).reduce(function (acc, placement) {
            return acc.concat(getBasePlacement(placement) === auto ? computeAutoPlacement(state, {
                placement: placement,
                boundary: boundary,
                rootBoundary: rootBoundary,
                padding: padding,
                flipVariations: flipVariations,
                allowedAutoPlacements: allowedAutoPlacements
            }) : placement);
        }, []);
        var referenceRect = state.rects.reference;
        var popperRect = state.rects.popper;
        var checksMap = new Map();
        var makeFallbackChecks = true;
        var firstFittingPlacement = placements[0];

        for (var i = 0; i < placements.length; i++) {
            var placement = placements[i];

            var _basePlacement = getBasePlacement(placement);

            var isStartVariation = getVariation(placement) === start;
            var isVertical = [top, bottom].indexOf(_basePlacement) >= 0;
            var len = isVertical ? 'width' : 'height';
            var overflow = detectOverflow(state, {
                placement: placement,
                boundary: boundary,
                rootBoundary: rootBoundary,
                altBoundary: altBoundary,
                padding: padding
            });
            var mainVariationSide = isVertical ? isStartVariation ? right : left : isStartVariation ? bottom : top;

            if (referenceRect[len] > popperRect[len]) {
                mainVariationSide = getOppositePlacement(mainVariationSide);
            }

            var altVariationSide = getOppositePlacement(mainVariationSide);
            var checks = [];

            if (checkMainAxis) {
                checks.push(overflow[_basePlacement] <= 0);
            }

            if (checkAltAxis) {
                checks.push(overflow[mainVariationSide] <= 0, overflow[altVariationSide] <= 0);
            }

            if (checks.every(function (check) {
                return check;
            })) {
                firstFittingPlacement = placement;
                makeFallbackChecks = false;
                break;
            }

            checksMap.set(placement, checks);
        }

        if (makeFallbackChecks) {
            // `2` may be desired in some cases – research later
            var numberOfChecks = flipVariations ? 3 : 1;

            var _loop = function _loop(_i) {
                var fittingPlacement = placements.find(function (placement) {
                    var checks = checksMap.get(placement);

                    if (checks) {
                        return checks.slice(0, _i).every(function (check) {
                            return check;
                        });
                    }
                });

                if (fittingPlacement) {
                    firstFittingPlacement = fittingPlacement;
                    return "break";
                }
            };

            for (var _i = numberOfChecks; _i > 0; _i--) {
                var _ret = _loop(_i);

                if (_ret === "break") break;
            }
        }

        if (state.placement !== firstFittingPlacement) {
            state.modifiersData[name]._skip = true;
            state.placement = firstFittingPlacement;
            state.reset = true;
        }
    } // eslint-disable-next-line import/no-unused-modules


    var flip$1 = {
        name: 'flip',
        enabled: true,
        phase: 'main',
        fn: flip,
        requiresIfExists: ['offset'],
        data: {
            _skip: false
        }
    };

    function getAltAxis(axis) {
        return axis === 'x' ? 'y' : 'x';
    }

    function within(min$1, value, max$1) {
        return max(min$1, min(value, max$1));
    }
    function withinMaxClamp(min, value, max) {
        var v = within(min, value, max);
        return v > max ? max : v;
    }

    function preventOverflow(_ref) {
        var state = _ref.state,
            options = _ref.options,
            name = _ref.name;
        var _options$mainAxis = options.mainAxis,
            checkMainAxis = _options$mainAxis === void 0 ? true : _options$mainAxis,
            _options$altAxis = options.altAxis,
            checkAltAxis = _options$altAxis === void 0 ? false : _options$altAxis,
            boundary = options.boundary,
            rootBoundary = options.rootBoundary,
            altBoundary = options.altBoundary,
            padding = options.padding,
            _options$tether = options.tether,
            tether = _options$tether === void 0 ? true : _options$tether,
            _options$tetherOffset = options.tetherOffset,
            tetherOffset = _options$tetherOffset === void 0 ? 0 : _options$tetherOffset;
        var overflow = detectOverflow(state, {
            boundary: boundary,
            rootBoundary: rootBoundary,
            padding: padding,
            altBoundary: altBoundary
        });
        var basePlacement = getBasePlacement(state.placement);
        var variation = getVariation(state.placement);
        var isBasePlacement = !variation;
        var mainAxis = getMainAxisFromPlacement(basePlacement);
        var altAxis = getAltAxis(mainAxis);
        var popperOffsets = state.modifiersData.popperOffsets;
        var referenceRect = state.rects.reference;
        var popperRect = state.rects.popper;
        var tetherOffsetValue = typeof tetherOffset === 'function' ? tetherOffset(Object.assign({}, state.rects, {
            placement: state.placement
        })) : tetherOffset;
        var normalizedTetherOffsetValue = typeof tetherOffsetValue === 'number' ? {
            mainAxis: tetherOffsetValue,
            altAxis: tetherOffsetValue
        } : Object.assign({
            mainAxis: 0,
            altAxis: 0
        }, tetherOffsetValue);
        var offsetModifierState = state.modifiersData.offset ? state.modifiersData.offset[state.placement] : null;
        var data = {
            x: 0,
            y: 0
        };

        if (!popperOffsets) {
            return;
        }

        if (checkMainAxis) {
            var _offsetModifierState$;

            var mainSide = mainAxis === 'y' ? top : left;
            var altSide = mainAxis === 'y' ? bottom : right;
            var len = mainAxis === 'y' ? 'height' : 'width';
            var offset = popperOffsets[mainAxis];
            var min$1 = offset + overflow[mainSide];
            var max$1 = offset - overflow[altSide];
            var additive = tether ? -popperRect[len] / 2 : 0;
            var minLen = variation === start ? referenceRect[len] : popperRect[len];
            var maxLen = variation === start ? -popperRect[len] : -referenceRect[len]; // We need to include the arrow in the calculation so the arrow doesn't go
            // outside the reference bounds

            var arrowElement = state.elements.arrow;
            var arrowRect = tether && arrowElement ? getLayoutRect(arrowElement) : {
                width: 0,
                height: 0
            };
            var arrowPaddingObject = state.modifiersData['arrow#persistent'] ? state.modifiersData['arrow#persistent'].padding : getFreshSideObject();
            var arrowPaddingMin = arrowPaddingObject[mainSide];
            var arrowPaddingMax = arrowPaddingObject[altSide]; // If the reference length is smaller than the arrow length, we don't want
            // to include its full size in the calculation. If the reference is small
            // and near the edge of a boundary, the popper can overflow even if the
            // reference is not overflowing as well (e.g. virtual elements with no
            // width or height)

            var arrowLen = within(0, referenceRect[len], arrowRect[len]);
            var minOffset = isBasePlacement ? referenceRect[len] / 2 - additive - arrowLen - arrowPaddingMin - normalizedTetherOffsetValue.mainAxis : minLen - arrowLen - arrowPaddingMin - normalizedTetherOffsetValue.mainAxis;
            var maxOffset = isBasePlacement ? -referenceRect[len] / 2 + additive + arrowLen + arrowPaddingMax + normalizedTetherOffsetValue.mainAxis : maxLen + arrowLen + arrowPaddingMax + normalizedTetherOffsetValue.mainAxis;
            var arrowOffsetParent = state.elements.arrow && getOffsetParent(state.elements.arrow);
            var clientOffset = arrowOffsetParent ? mainAxis === 'y' ? arrowOffsetParent.clientTop || 0 : arrowOffsetParent.clientLeft || 0 : 0;
            var offsetModifierValue = (_offsetModifierState$ = offsetModifierState == null ? void 0 : offsetModifierState[mainAxis]) != null ? _offsetModifierState$ : 0;
            var tetherMin = offset + minOffset - offsetModifierValue - clientOffset;
            var tetherMax = offset + maxOffset - offsetModifierValue;
            var preventedOffset = within(tether ? min(min$1, tetherMin) : min$1, offset, tether ? max(max$1, tetherMax) : max$1);
            popperOffsets[mainAxis] = preventedOffset;
            data[mainAxis] = preventedOffset - offset;
        }

        if (checkAltAxis) {
            var _offsetModifierState$2;

            var _mainSide = mainAxis === 'x' ? top : left;

            var _altSide = mainAxis === 'x' ? bottom : right;

            var _offset = popperOffsets[altAxis];

            var _len = altAxis === 'y' ? 'height' : 'width';

            var _min = _offset + overflow[_mainSide];

            var _max = _offset - overflow[_altSide];

            var isOriginSide = [top, left].indexOf(basePlacement) !== -1;

            var _offsetModifierValue = (_offsetModifierState$2 = offsetModifierState == null ? void 0 : offsetModifierState[altAxis]) != null ? _offsetModifierState$2 : 0;

            var _tetherMin = isOriginSide ? _min : _offset - referenceRect[_len] - popperRect[_len] - _offsetModifierValue + normalizedTetherOffsetValue.altAxis;

            var _tetherMax = isOriginSide ? _offset + referenceRect[_len] + popperRect[_len] - _offsetModifierValue - normalizedTetherOffsetValue.altAxis : _max;

            var _preventedOffset = tether && isOriginSide ? withinMaxClamp(_tetherMin, _offset, _tetherMax) : within(tether ? _tetherMin : _min, _offset, tether ? _tetherMax : _max);

            popperOffsets[altAxis] = _preventedOffset;
            data[altAxis] = _preventedOffset - _offset;
        }

        state.modifiersData[name] = data;
    } // eslint-disable-next-line import/no-unused-modules


    var preventOverflow$1 = {
        name: 'preventOverflow',
        enabled: true,
        phase: 'main',
        fn: preventOverflow,
        requiresIfExists: ['offset']
    };

    var toPaddingObject = function toPaddingObject(padding, state) {
        padding = typeof padding === 'function' ? padding(Object.assign({}, state.rects, {
            placement: state.placement
        })) : padding;
        return mergePaddingObject(typeof padding !== 'number' ? padding : expandToHashMap(padding, basePlacements));
    };

    function arrow(_ref) {
        var _state$modifiersData$;

        var state = _ref.state,
            name = _ref.name,
            options = _ref.options;
        var arrowElement = state.elements.arrow;
        var popperOffsets = state.modifiersData.popperOffsets;
        var basePlacement = getBasePlacement(state.placement);
        var axis = getMainAxisFromPlacement(basePlacement);
        var isVertical = [left, right].indexOf(basePlacement) >= 0;
        var len = isVertical ? 'height' : 'width';

        if (!arrowElement || !popperOffsets) {
            return;
        }

        var paddingObject = toPaddingObject(options.padding, state);
        var arrowRect = getLayoutRect(arrowElement);
        var minProp = axis === 'y' ? top : left;
        var maxProp = axis === 'y' ? bottom : right;
        var endDiff = state.rects.reference[len] + state.rects.reference[axis] - popperOffsets[axis] - state.rects.popper[len];
        var startDiff = popperOffsets[axis] - state.rects.reference[axis];
        var arrowOffsetParent = getOffsetParent(arrowElement);
        var clientSize = arrowOffsetParent ? axis === 'y' ? arrowOffsetParent.clientHeight || 0 : arrowOffsetParent.clientWidth || 0 : 0;
        var centerToReference = endDiff / 2 - startDiff / 2; // Make sure the arrow doesn't overflow the popper if the center point is
        // outside of the popper bounds

        var min = paddingObject[minProp];
        var max = clientSize - arrowRect[len] - paddingObject[maxProp];
        var center = clientSize / 2 - arrowRect[len] / 2 + centerToReference;
        var offset = within(min, center, max); // Prevents breaking syntax highlighting...

        var axisProp = axis;
        state.modifiersData[name] = (_state$modifiersData$ = {}, _state$modifiersData$[axisProp] = offset, _state$modifiersData$.centerOffset = offset - center, _state$modifiersData$);
    }

    function effect(_ref2) {
        var state = _ref2.state,
            options = _ref2.options;
        var _options$element = options.element,
            arrowElement = _options$element === void 0 ? '[data-popper-arrow]' : _options$element;

        if (arrowElement == null) {
            return;
        } // CSS selector


        if (typeof arrowElement === 'string') {
            arrowElement = state.elements.popper.querySelector(arrowElement);

            if (!arrowElement) {
                return;
            }
        }

        {
            if (!isHTMLElement(arrowElement)) {
                console.error(['Popper: "arrow" element must be an HTMLElement (not an SVGElement).', 'To use an SVG arrow, wrap it in an HTMLElement that will be used as', 'the arrow.'].join(' '));
            }
        }

        if (!contains(state.elements.popper, arrowElement)) {
            {
                console.error(['Popper: "arrow" modifier\'s `element` must be a child of the popper', 'element.'].join(' '));
            }

            return;
        }

        state.elements.arrow = arrowElement;
    } // eslint-disable-next-line import/no-unused-modules


    var arrow$1 = {
        name: 'arrow',
        enabled: true,
        phase: 'main',
        fn: arrow,
        effect: effect,
        requires: ['popperOffsets'],
        requiresIfExists: ['preventOverflow']
    };

    function getSideOffsets(overflow, rect, preventedOffsets) {
        if (preventedOffsets === void 0) {
            preventedOffsets = {
                x: 0,
                y: 0
            };
        }

        return {
            top: overflow.top - rect.height - preventedOffsets.y,
            right: overflow.right - rect.width + preventedOffsets.x,
            bottom: overflow.bottom - rect.height + preventedOffsets.y,
            left: overflow.left - rect.width - preventedOffsets.x
        };
    }

    function isAnySideFullyClipped(overflow) {
        return [top, right, bottom, left].some(function (side) {
            return overflow[side] >= 0;
        });
    }

    function hide(_ref) {
        var state = _ref.state,
            name = _ref.name;
        var referenceRect = state.rects.reference;
        var popperRect = state.rects.popper;
        var preventedOffsets = state.modifiersData.preventOverflow;
        var referenceOverflow = detectOverflow(state, {
            elementContext: 'reference'
        });
        var popperAltOverflow = detectOverflow(state, {
            altBoundary: true
        });
        var referenceClippingOffsets = getSideOffsets(referenceOverflow, referenceRect);
        var popperEscapeOffsets = getSideOffsets(popperAltOverflow, popperRect, preventedOffsets);
        var isReferenceHidden = isAnySideFullyClipped(referenceClippingOffsets);
        var hasPopperEscaped = isAnySideFullyClipped(popperEscapeOffsets);
        state.modifiersData[name] = {
            referenceClippingOffsets: referenceClippingOffsets,
            popperEscapeOffsets: popperEscapeOffsets,
            isReferenceHidden: isReferenceHidden,
            hasPopperEscaped: hasPopperEscaped
        };
        state.attributes.popper = Object.assign({}, state.attributes.popper, {
            'data-popper-reference-hidden': isReferenceHidden,
            'data-popper-escaped': hasPopperEscaped
        });
    } // eslint-disable-next-line import/no-unused-modules


    var hide$1 = {
        name: 'hide',
        enabled: true,
        phase: 'main',
        requiresIfExists: ['preventOverflow'],
        fn: hide
    };

    var defaultModifiers$1 = [eventListeners, popperOffsets$1, computeStyles$1, applyStyles$1];
    var createPopper$1 = /*#__PURE__*/popperGenerator({
        defaultModifiers: defaultModifiers$1
    }); // eslint-disable-next-line import/no-unused-modules

    var defaultModifiers = [eventListeners, popperOffsets$1, computeStyles$1, applyStyles$1, offset$1, flip$1, preventOverflow$1, arrow$1, hide$1];
    var createPopper = /*#__PURE__*/popperGenerator({
        defaultModifiers: defaultModifiers
    }); // eslint-disable-next-line import/no-unused-modules

    exports.applyStyles = applyStyles$1;
    exports.arrow = arrow$1;
    exports.computeStyles = computeStyles$1;
    exports.createPopper = createPopper;
    exports.createPopperLite = createPopper$1;
    exports.defaultModifiers = defaultModifiers;
    exports.detectOverflow = detectOverflow;
    exports.eventListeners = eventListeners;
    exports.flip = flip$1;
    exports.hide = hide$1;
    exports.offset = offset$1;
    exports.popperGenerator = popperGenerator;
    exports.popperOffsets = popperOffsets$1;
    exports.preventOverflow = preventOverflow$1;

    Object.defineProperty(exports, '__esModule', { value: true });

})));
//# sourceMappingURL=popper.js.map

/*!
  * Bootstrap v5.3.0-alpha1 (https://getbootstrap.com/)
  * Copyright 2011-2022 The Bootstrap Authors (https://github.com/twbs/bootstrap/graphs/contributors)
  * Licensed under MIT (https://github.com/twbs/bootstrap/blob/main/LICENSE)
  */
!function(t,e){"object"==typeof exports&&"undefined"!=typeof module?module.exports=e(require("@popperjs/core")):"function"==typeof define&&define.amd?define(["@popperjs/core"],e):(t="undefined"!=typeof globalThis?globalThis:t||self).bootstrap=e(t.Popper)}(this,(function(t){"use strict";function e(t){const e=Object.create(null,{[Symbol.toStringTag]:{value:"Module"}});if(t)for(const s in t)if("default"!==s){const i=Object.getOwnPropertyDescriptor(t,s);Object.defineProperty(e,s,i.get?i:{enumerable:!0,get:()=>t[s]})}return e.default=t,Object.freeze(e)}const s=e(t),i="transitionend",n=t=>(t&&window.CSS&&window.CSS.escape&&(t=t.replace(/#([^\s"#']+)/g,((t,e)=>`#${CSS.escape(e)}`))),t),o=t=>{t.dispatchEvent(new Event(i))},r=t=>!(!t||"object"!=typeof t)&&(void 0!==t.jquery&&(t=t[0]),void 0!==t.nodeType),a=t=>r(t)?t.jquery?t[0]:t:"string"==typeof t&&t.length>0?document.querySelector(n(t)):null,l=t=>{if(!r(t)||0===t.getClientRects().length)return!1;const e="visible"===getComputedStyle(t).getPropertyValue("visibility"),s=t.closest("details:not([open])");if(!s)return e;if(s!==t){const e=t.closest("summary");if(e&&e.parentNode!==s)return!1;if(null===e)return!1}return e},c=t=>!t||t.nodeType!==Node.ELEMENT_NODE||!!t.classList.contains("disabled")||(void 0!==t.disabled?t.disabled:t.hasAttribute("disabled")&&"false"!==t.getAttribute("disabled")),h=t=>{if(!document.documentElement.attachShadow)return null;if("function"==typeof t.getRootNode){const e=t.getRootNode();return e instanceof ShadowRoot?e:null}return t instanceof ShadowRoot?t:t.parentNode?h(t.parentNode):null},d=()=>{},u=t=>{t.offsetHeight},_=()=>window.jQuery&&!document.body.hasAttribute("data-bs-no-jquery")?window.jQuery:null,g=[],f=()=>"rtl"===document.documentElement.dir,m=t=>{var e;e=()=>{const e=_();if(e){const s=t.NAME,i=e.fn[s];e.fn[s]=t.jQueryInterface,e.fn[s].Constructor=t,e.fn[s].noConflict=()=>(e.fn[s]=i,t.jQueryInterface)}},"loading"===document.readyState?(g.length||document.addEventListener("DOMContentLoaded",(()=>{for(const t of g)t()})),g.push(e)):e()},p=(t,e=[],s=t)=>"function"==typeof t?t(...e):s,b=(t,e,s=!0)=>{if(!s)return void p(t);const n=(t=>{if(!t)return 0;let{transitionDuration:e,transitionDelay:s}=window.getComputedStyle(t);const i=Number.parseFloat(e),n=Number.parseFloat(s);return i||n?(e=e.split(",")[0],s=s.split(",")[0],1e3*(Number.parseFloat(e)+Number.parseFloat(s))):0})(e)+5;let r=!1;const a=({target:s})=>{s===e&&(r=!0,e.removeEventListener(i,a),p(t))};e.addEventListener(i,a),setTimeout((()=>{r||o(e)}),n)},v=(t,e,s,i)=>{const n=t.length;let o=t.indexOf(e);return-1===o?!s&&i?t[n-1]:t[0]:(o+=s?1:-1,i&&(o=(o+n)%n),t[Math.max(0,Math.min(o,n-1))])},y=/[^.]*(?=\..*)\.|.*/,w=/\..*/,A=/::\d+$/,E={};let C=1;const T={mouseenter:"mouseover",mouseleave:"mouseout"},k=new Set(["click","dblclick","mouseup","mousedown","contextmenu","mousewheel","DOMMouseScroll","mouseover","mouseout","mousemove","selectstart","selectend","keydown","keypress","keyup","orientationchange","touchstart","touchmove","touchend","touchcancel","pointerdown","pointermove","pointerup","pointerleave","pointercancel","gesturestart","gesturechange","gestureend","focus","blur","change","reset","select","submit","focusin","focusout","load","unload","beforeunload","resize","move","DOMContentLoaded","readystatechange","error","abort","scroll"]);function S(t,e){return e&&`${e}::${C++}`||t.uidEvent||C++}function L(t){const e=S(t);return t.uidEvent=e,E[e]=E[e]||{},E[e]}function O(t,e,s=null){return Object.values(t).find((t=>t.callable===e&&t.delegationSelector===s))}function I(t,e,s){const i="string"==typeof e,n=i?s:e||s;let o=x(t);return k.has(o)||(o=t),[i,n,o]}function D(t,e,s,i,n){if("string"!=typeof e||!t)return;let[o,r,a]=I(e,s,i);if(e in T){const t=t=>function(e){if(!e.relatedTarget||e.relatedTarget!==e.delegateTarget&&!e.delegateTarget.contains(e.relatedTarget))return t.call(this,e)};r=t(r)}const l=L(t),c=l[a]||(l[a]={}),h=O(c,r,o?s:null);if(h)return void(h.oneOff=h.oneOff&&n);const d=S(r,e.replace(y,"")),u=o?function(t,e,s){return function i(n){const o=t.querySelectorAll(e);for(let{target:r}=n;r&&r!==this;r=r.parentNode)for(const a of o)if(a===r)return j(n,{delegateTarget:r}),i.oneOff&&M.off(t,n.type,e,s),s.apply(r,[n])}}(t,s,r):function(t,e){return function s(i){return j(i,{delegateTarget:t}),s.oneOff&&M.off(t,i.type,e),e.apply(t,[i])}}(t,r);u.delegationSelector=o?s:null,u.callable=r,u.oneOff=n,u.uidEvent=d,c[d]=u,t.addEventListener(a,u,o)}function N(t,e,s,i,n){const o=O(e[s],i,n);o&&(t.removeEventListener(s,o,Boolean(n)),delete e[s][o.uidEvent])}function P(t,e,s,i){const n=e[s]||{};for(const[o,r]of Object.entries(n))o.includes(i)&&N(t,e,s,r.callable,r.delegationSelector)}function x(t){return t=t.replace(w,""),T[t]||t}const M={on(t,e,s,i){D(t,e,s,i,!1)},one(t,e,s,i){D(t,e,s,i,!0)},off(t,e,s,i){if("string"!=typeof e||!t)return;const[n,o,r]=I(e,s,i),a=r!==e,l=L(t),c=l[r]||{},h=e.startsWith(".");if(void 0===o){if(h)for(const s of Object.keys(l))P(t,l,s,e.slice(1));for(const[s,i]of Object.entries(c)){const n=s.replace(A,"");a&&!e.includes(n)||N(t,l,r,i.callable,i.delegationSelector)}}else{if(!Object.keys(c).length)return;N(t,l,r,o,n?s:null)}},trigger(t,e,s){if("string"!=typeof e||!t)return null;const i=_();let n=null,o=!0,r=!0,a=!1;e!==x(e)&&i&&(n=i.Event(e,s),i(t).trigger(n),o=!n.isPropagationStopped(),r=!n.isImmediatePropagationStopped(),a=n.isDefaultPrevented());let l=new Event(e,{bubbles:o,cancelable:!0});return l=j(l,s),a&&l.preventDefault(),r&&t.dispatchEvent(l),l.defaultPrevented&&n&&n.preventDefault(),l}};function j(t,e={}){for(const[s,i]of Object.entries(e))try{t[s]=i}catch(e){Object.defineProperty(t,s,{configurable:!0,get:()=>i})}return t}const F=new Map,$={set(t,e,s){F.has(t)||F.set(t,new Map);const i=F.get(t);i.has(e)||0===i.size?i.set(e,s):console.error(`Bootstrap doesn't allow more than one instance per element. Bound instance: ${Array.from(i.keys())[0]}.`)},get:(t,e)=>F.has(t)&&F.get(t).get(e)||null,remove(t,e){if(!F.has(t))return;const s=F.get(t);s.delete(e),0===s.size&&F.delete(t)}};function z(t){if("true"===t)return!0;if("false"===t)return!1;if(t===Number(t).toString())return Number(t);if(""===t||"null"===t)return null;if("string"!=typeof t)return t;try{return JSON.parse(decodeURIComponent(t))}catch(e){return t}}function H(t){return t.replace(/[A-Z]/g,(t=>`-${t.toLowerCase()}`))}const B={setDataAttribute(t,e,s){t.setAttribute(`data-bs-${H(e)}`,s)},removeDataAttribute(t,e){t.removeAttribute(`data-bs-${H(e)}`)},getDataAttributes(t){if(!t)return{};const e={},s=Object.keys(t.dataset).filter((t=>t.startsWith("bs")&&!t.startsWith("bsConfig")));for(const i of s){let s=i.replace(/^bs/,"");s=s.charAt(0).toLowerCase()+s.slice(1,s.length),e[s]=z(t.dataset[i])}return e},getDataAttribute:(t,e)=>z(t.getAttribute(`data-bs-${H(e)}`))};class q{static get Default(){return{}}static get DefaultType(){return{}}static get NAME(){throw new Error('You have to implement the static method "NAME", for each component!')}_getConfig(t){return t=this._mergeConfigObj(t),t=this._configAfterMerge(t),this._typeCheckConfig(t),t}_configAfterMerge(t){return t}_mergeConfigObj(t,e){const s=r(e)?B.getDataAttribute(e,"config"):{};return{...this.constructor.Default,..."object"==typeof s?s:{},...r(e)?B.getDataAttributes(e):{},..."object"==typeof t?t:{}}}_typeCheckConfig(t,e=this.constructor.DefaultType){for(const[i,n]of Object.entries(e)){const e=t[i],o=r(e)?"element":null==(s=e)?`${s}`:Object.prototype.toString.call(s).match(/\s([a-z]+)/i)[1].toLowerCase();if(!new RegExp(n).test(o))throw new TypeError(`${this.constructor.NAME.toUpperCase()}: Option "${i}" provided type "${o}" but expected type "${n}".`)}var s}}class W extends q{constructor(t,e){super(),(t=a(t))&&(this._element=t,this._config=this._getConfig(e),$.set(this._element,this.constructor.DATA_KEY,this))}dispose(){$.remove(this._element,this.constructor.DATA_KEY),M.off(this._element,this.constructor.EVENT_KEY);for(const t of Object.getOwnPropertyNames(this))this[t]=null}_queueCallback(t,e,s=!0){b(t,e,s)}_getConfig(t){return t=this._mergeConfigObj(t,this._element),t=this._configAfterMerge(t),this._typeCheckConfig(t),t}static getInstance(t){return $.get(a(t),this.DATA_KEY)}static getOrCreateInstance(t,e={}){return this.getInstance(t)||new this(t,"object"==typeof e?e:null)}static get VERSION(){return"5.3.0-alpha1"}static get DATA_KEY(){return`bs.${this.NAME}`}static get EVENT_KEY(){return`.${this.DATA_KEY}`}static eventName(t){return`${t}${this.EVENT_KEY}`}}const R=t=>{let e=t.getAttribute("data-bs-target");if(!e||"#"===e){let s=t.getAttribute("href");if(!s||!s.includes("#")&&!s.startsWith("."))return null;s.includes("#")&&!s.startsWith("#")&&(s=`#${s.split("#")[1]}`),e=s&&"#"!==s?s.trim():null}return n(e)},V={find:(t,e=document.documentElement)=>[].concat(...Element.prototype.querySelectorAll.call(e,t)),findOne:(t,e=document.documentElement)=>Element.prototype.querySelector.call(e,t),children:(t,e)=>[].concat(...t.children).filter((t=>t.matches(e))),parents(t,e){const s=[];let i=t.parentNode.closest(e);for(;i;)s.push(i),i=i.parentNode.closest(e);return s},prev(t,e){let s=t.previousElementSibling;for(;s;){if(s.matches(e))return[s];s=s.previousElementSibling}return[]},next(t,e){let s=t.nextElementSibling;for(;s;){if(s.matches(e))return[s];s=s.nextElementSibling}return[]},focusableChildren(t){const e=["a","button","input","textarea","select","details","[tabindex]",'[contenteditable="true"]'].map((t=>`${t}:not([tabindex^="-"])`)).join(",");return this.find(e,t).filter((t=>!c(t)&&l(t)))},getSelectorFromElement(t){const e=R(t);return e&&V.findOne(e)?e:null},getElementFromSelector(t){const e=R(t);return e?V.findOne(e):null},getMultipleElementsFromSelector(t){const e=R(t);return e?V.find(e):[]}},K=(t,e="hide")=>{const s=`click.dismiss${t.EVENT_KEY}`,i=t.NAME;M.on(document,s,`[data-bs-dismiss="${i}"]`,(function(s){if(["A","AREA"].includes(this.tagName)&&s.preventDefault(),c(this))return;const n=V.getElementFromSelector(this)||this.closest(`.${i}`);t.getOrCreateInstance(n)[e]()}))};class Q extends W{static get NAME(){return"alert"}close(){if(M.trigger(this._element,"close.bs.alert").defaultPrevented)return;this._element.classList.remove("show");const t=this._element.classList.contains("fade");this._queueCallback((()=>this._destroyElement()),this._element,t)}_destroyElement(){this._element.remove(),M.trigger(this._element,"closed.bs.alert"),this.dispose()}static jQueryInterface(t){return this.each((function(){const e=Q.getOrCreateInstance(this);if("string"==typeof t){if(void 0===e[t]||t.startsWith("_")||"constructor"===t)throw new TypeError(`No method named "${t}"`);e[t](this)}}))}}K(Q,"close"),m(Q);const X='[data-bs-toggle="button"]';class Y extends W{static get NAME(){return"button"}toggle(){this._element.setAttribute("aria-pressed",this._element.classList.toggle("active"))}static jQueryInterface(t){return this.each((function(){const e=Y.getOrCreateInstance(this);"toggle"===t&&e[t]()}))}}M.on(document,"click.bs.button.data-api",X,(t=>{t.preventDefault();const e=t.target.closest(X);Y.getOrCreateInstance(e).toggle()})),m(Y);const U={endCallback:null,leftCallback:null,rightCallback:null},G={endCallback:"(function|null)",leftCallback:"(function|null)",rightCallback:"(function|null)"};class J extends q{constructor(t,e){super(),this._element=t,t&&J.isSupported()&&(this._config=this._getConfig(e),this._deltaX=0,this._supportPointerEvents=Boolean(window.PointerEvent),this._initEvents())}static get Default(){return U}static get DefaultType(){return G}static get NAME(){return"swipe"}dispose(){M.off(this._element,".bs.swipe")}_start(t){this._supportPointerEvents?this._eventIsPointerPenTouch(t)&&(this._deltaX=t.clientX):this._deltaX=t.touches[0].clientX}_end(t){this._eventIsPointerPenTouch(t)&&(this._deltaX=t.clientX-this._deltaX),this._handleSwipe(),p(this._config.endCallback)}_move(t){this._deltaX=t.touches&&t.touches.length>1?0:t.touches[0].clientX-this._deltaX}_handleSwipe(){const t=Math.abs(this._deltaX);if(t<=40)return;const e=t/this._deltaX;this._deltaX=0,e&&p(e>0?this._config.rightCallback:this._config.leftCallback)}_initEvents(){this._supportPointerEvents?(M.on(this._element,"pointerdown.bs.swipe",(t=>this._start(t))),M.on(this._element,"pointerup.bs.swipe",(t=>this._end(t))),this._element.classList.add("pointer-event")):(M.on(this._element,"touchstart.bs.swipe",(t=>this._start(t))),M.on(this._element,"touchmove.bs.swipe",(t=>this._move(t))),M.on(this._element,"touchend.bs.swipe",(t=>this._end(t))))}_eventIsPointerPenTouch(t){return this._supportPointerEvents&&("pen"===t.pointerType||"touch"===t.pointerType)}static isSupported(){return"ontouchstart"in document.documentElement||navigator.maxTouchPoints>0}}const Z="next",tt="prev",et="left",st="right",it="slid.bs.carousel",nt="carousel",ot="active",rt={ArrowLeft:st,ArrowRight:et},at={interval:5e3,keyboard:!0,pause:"hover",ride:!1,touch:!0,wrap:!0},lt={interval:"(number|boolean)",keyboard:"boolean",pause:"(string|boolean)",ride:"(boolean|string)",touch:"boolean",wrap:"boolean"};class ct extends W{constructor(t,e){super(t,e),this._interval=null,this._activeElement=null,this._isSliding=!1,this.touchTimeout=null,this._swipeHelper=null,this._indicatorsElement=V.findOne(".carousel-indicators",this._element),this._addEventListeners(),this._config.ride===nt&&this.cycle()}static get Default(){return at}static get DefaultType(){return lt}static get NAME(){return"carousel"}next(){this._slide(Z)}nextWhenVisible(){!document.hidden&&l(this._element)&&this.next()}prev(){this._slide(tt)}pause(){this._isSliding&&o(this._element),this._clearInterval()}cycle(){this._clearInterval(),this._updateInterval(),this._interval=setInterval((()=>this.nextWhenVisible()),this._config.interval)}_maybeEnableCycle(){this._config.ride&&(this._isSliding?M.one(this._element,it,(()=>this.cycle())):this.cycle())}to(t){const e=this._getItems();if(t>e.length-1||t<0)return;if(this._isSliding)return void M.one(this._element,it,(()=>this.to(t)));const s=this._getItemIndex(this._getActive());if(s===t)return;const i=t>s?Z:tt;this._slide(i,e[t])}dispose(){this._swipeHelper&&this._swipeHelper.dispose(),super.dispose()}_configAfterMerge(t){return t.defaultInterval=t.interval,t}_addEventListeners(){this._config.keyboard&&M.on(this._element,"keydown.bs.carousel",(t=>this._keydown(t))),"hover"===this._config.pause&&(M.on(this._element,"mouseenter.bs.carousel",(()=>this.pause())),M.on(this._element,"mouseleave.bs.carousel",(()=>this._maybeEnableCycle()))),this._config.touch&&J.isSupported()&&this._addTouchEventListeners()}_addTouchEventListeners(){for(const t of V.find(".carousel-item img",this._element))M.on(t,"dragstart.bs.carousel",(t=>t.preventDefault()));const t={leftCallback:()=>this._slide(this._directionToOrder(et)),rightCallback:()=>this._slide(this._directionToOrder(st)),endCallback:()=>{"hover"===this._config.pause&&(this.pause(),this.touchTimeout&&clearTimeout(this.touchTimeout),this.touchTimeout=setTimeout((()=>this._maybeEnableCycle()),500+this._config.interval))}};this._swipeHelper=new J(this._element,t)}_keydown(t){if(/input|textarea/i.test(t.target.tagName))return;const e=rt[t.key];e&&(t.preventDefault(),this._slide(this._directionToOrder(e)))}_getItemIndex(t){return this._getItems().indexOf(t)}_setActiveIndicatorElement(t){if(!this._indicatorsElement)return;const e=V.findOne(".active",this._indicatorsElement);e.classList.remove(ot),e.removeAttribute("aria-current");const s=V.findOne(`[data-bs-slide-to="${t}"]`,this._indicatorsElement);s&&(s.classList.add(ot),s.setAttribute("aria-current","true"))}_updateInterval(){const t=this._activeElement||this._getActive();if(!t)return;const e=Number.parseInt(t.getAttribute("data-bs-interval"),10);this._config.interval=e||this._config.defaultInterval}_slide(t,e=null){if(this._isSliding)return;const s=this._getActive(),i=t===Z,n=e||v(this._getItems(),s,i,this._config.wrap);if(n===s)return;const o=this._getItemIndex(n),r=e=>M.trigger(this._element,e,{relatedTarget:n,direction:this._orderToDirection(t),from:this._getItemIndex(s),to:o});if(r("slide.bs.carousel").defaultPrevented)return;if(!s||!n)return;const a=Boolean(this._interval);this.pause(),this._isSliding=!0,this._setActiveIndicatorElement(o),this._activeElement=n;const l=i?"carousel-item-start":"carousel-item-end",c=i?"carousel-item-next":"carousel-item-prev";n.classList.add(c),u(n),s.classList.add(l),n.classList.add(l),this._queueCallback((()=>{n.classList.remove(l,c),n.classList.add(ot),s.classList.remove(ot,c,l),this._isSliding=!1,r(it)}),s,this._isAnimated()),a&&this.cycle()}_isAnimated(){return this._element.classList.contains("slide")}_getActive(){return V.findOne(".active.carousel-item",this._element)}_getItems(){return V.find(".carousel-item",this._element)}_clearInterval(){this._interval&&(clearInterval(this._interval),this._interval=null)}_directionToOrder(t){return f()?t===et?tt:Z:t===et?Z:tt}_orderToDirection(t){return f()?t===tt?et:st:t===tt?st:et}static jQueryInterface(t){return this.each((function(){const e=ct.getOrCreateInstance(this,t);if("number"!=typeof t){if("string"==typeof t){if(void 0===e[t]||t.startsWith("_")||"constructor"===t)throw new TypeError(`No method named "${t}"`);e[t]()}}else e.to(t)}))}}M.on(document,"click.bs.carousel.data-api","[data-bs-slide], [data-bs-slide-to]",(function(t){const e=V.getElementFromSelector(this);if(!e||!e.classList.contains(nt))return;t.preventDefault();const s=ct.getOrCreateInstance(e),i=this.getAttribute("data-bs-slide-to");return i?(s.to(i),void s._maybeEnableCycle()):"next"===B.getDataAttribute(this,"slide")?(s.next(),void s._maybeEnableCycle()):(s.prev(),void s._maybeEnableCycle())})),M.on(window,"load.bs.carousel.data-api",(()=>{const t=V.find('[data-bs-ride="carousel"]');for(const e of t)ct.getOrCreateInstance(e)})),m(ct);const ht="show",dt="collapse",ut="collapsing",_t='[data-bs-toggle="collapse"]',gt={parent:null,toggle:!0},ft={parent:"(null|element)",toggle:"boolean"};class mt extends W{constructor(t,e){super(t,e),this._isTransitioning=!1,this._triggerArray=[];const s=V.find(_t);for(const t of s){const e=V.getSelectorFromElement(t),s=V.find(e).filter((t=>t===this._element));null!==e&&s.length&&this._triggerArray.push(t)}this._initializeChildren(),this._config.parent||this._addAriaAndCollapsedClass(this._triggerArray,this._isShown()),this._config.toggle&&this.toggle()}static get Default(){return gt}static get DefaultType(){return ft}static get NAME(){return"collapse"}toggle(){this._isShown()?this.hide():this.show()}show(){if(this._isTransitioning||this._isShown())return;let t=[];if(this._config.parent&&(t=this._getFirstLevelChildren(".collapse.show, .collapse.collapsing").filter((t=>t!==this._element)).map((t=>mt.getOrCreateInstance(t,{toggle:!1})))),t.length&&t[0]._isTransitioning)return;if(M.trigger(this._element,"show.bs.collapse").defaultPrevented)return;for(const e of t)e.hide();const e=this._getDimension();this._element.classList.remove(dt),this._element.classList.add(ut),this._element.style[e]=0,this._addAriaAndCollapsedClass(this._triggerArray,!0),this._isTransitioning=!0;const s=`scroll${e[0].toUpperCase()+e.slice(1)}`;this._queueCallback((()=>{this._isTransitioning=!1,this._element.classList.remove(ut),this._element.classList.add(dt,ht),this._element.style[e]="",M.trigger(this._element,"shown.bs.collapse")}),this._element,!0),this._element.style[e]=`${this._element[s]}px`}hide(){if(this._isTransitioning||!this._isShown())return;if(M.trigger(this._element,"hide.bs.collapse").defaultPrevented)return;const t=this._getDimension();this._element.style[t]=`${this._element.getBoundingClientRect()[t]}px`,u(this._element),this._element.classList.add(ut),this._element.classList.remove(dt,ht);for(const t of this._triggerArray){const e=V.getElementFromSelector(t);e&&!this._isShown(e)&&this._addAriaAndCollapsedClass([t],!1)}this._isTransitioning=!0,this._element.style[t]="",this._queueCallback((()=>{this._isTransitioning=!1,this._element.classList.remove(ut),this._element.classList.add(dt),M.trigger(this._element,"hidden.bs.collapse")}),this._element,!0)}_isShown(t=this._element){return t.classList.contains(ht)}_configAfterMerge(t){return t.toggle=Boolean(t.toggle),t.parent=a(t.parent),t}_getDimension(){return this._element.classList.contains("collapse-horizontal")?"width":"height"}_initializeChildren(){if(!this._config.parent)return;const t=this._getFirstLevelChildren(_t);for(const e of t){const t=V.getElementFromSelector(e);t&&this._addAriaAndCollapsedClass([e],this._isShown(t))}}_getFirstLevelChildren(t){const e=V.find(":scope .collapse .collapse",this._config.parent);return V.find(t,this._config.parent).filter((t=>!e.includes(t)))}_addAriaAndCollapsedClass(t,e){if(t.length)for(const s of t)s.classList.toggle("collapsed",!e),s.setAttribute("aria-expanded",e)}static jQueryInterface(t){const e={};return"string"==typeof t&&/show|hide/.test(t)&&(e.toggle=!1),this.each((function(){const s=mt.getOrCreateInstance(this,e);if("string"==typeof t){if(void 0===s[t])throw new TypeError(`No method named "${t}"`);s[t]()}}))}}M.on(document,"click.bs.collapse.data-api",_t,(function(t){("A"===t.target.tagName||t.delegateTarget&&"A"===t.delegateTarget.tagName)&&t.preventDefault();for(const t of V.getMultipleElementsFromSelector(this))mt.getOrCreateInstance(t,{toggle:!1}).toggle()})),m(mt);const pt="dropdown",bt="ArrowUp",vt="ArrowDown",yt="click.bs.dropdown.data-api",wt="keydown.bs.dropdown.data-api",At="show",Et='[data-bs-toggle="dropdown"]:not(.disabled):not(:disabled)',Ct=`${Et}.show`,Tt=".dropdown-menu",kt=f()?"top-end":"top-start",St=f()?"top-start":"top-end",Lt=f()?"bottom-end":"bottom-start",Ot=f()?"bottom-start":"bottom-end",It=f()?"left-start":"right-start",Dt=f()?"right-start":"left-start",Nt={autoClose:!0,boundary:"clippingParents",display:"dynamic",offset:[0,2],popperConfig:null,reference:"toggle"},Pt={autoClose:"(boolean|string)",boundary:"(string|element)",display:"string",offset:"(array|string|function)",popperConfig:"(null|object|function)",reference:"(string|element|object)"};class xt extends W{constructor(t,e){super(t,e),this._popper=null,this._parent=this._element.parentNode,this._menu=V.next(this._element,Tt)[0]||V.prev(this._element,Tt)[0]||V.findOne(Tt,this._parent),this._inNavbar=this._detectNavbar()}static get Default(){return Nt}static get DefaultType(){return Pt}static get NAME(){return pt}toggle(){return this._isShown()?this.hide():this.show()}show(){if(c(this._element)||this._isShown())return;const t={relatedTarget:this._element};if(!M.trigger(this._element,"show.bs.dropdown",t).defaultPrevented){if(this._createPopper(),"ontouchstart"in document.documentElement&&!this._parent.closest(".navbar-nav"))for(const t of[].concat(...document.body.children))M.on(t,"mouseover",d);this._element.focus(),this._element.setAttribute("aria-expanded",!0),this._menu.classList.add(At),this._element.classList.add(At),M.trigger(this._element,"shown.bs.dropdown",t)}}hide(){if(c(this._element)||!this._isShown())return;const t={relatedTarget:this._element};this._completeHide(t)}dispose(){this._popper&&this._popper.destroy(),super.dispose()}update(){this._inNavbar=this._detectNavbar(),this._popper&&this._popper.update()}_completeHide(t){if(!M.trigger(this._element,"hide.bs.dropdown",t).defaultPrevented){if("ontouchstart"in document.documentElement)for(const t of[].concat(...document.body.children))M.off(t,"mouseover",d);this._popper&&this._popper.destroy(),this._menu.classList.remove(At),this._element.classList.remove(At),this._element.setAttribute("aria-expanded","false"),B.removeDataAttribute(this._menu,"popper"),M.trigger(this._element,"hidden.bs.dropdown",t)}}_getConfig(t){if("object"==typeof(t=super._getConfig(t)).reference&&!r(t.reference)&&"function"!=typeof t.reference.getBoundingClientRect)throw new TypeError(`${pt.toUpperCase()}: Option "reference" provided type "object" without a required "getBoundingClientRect" method.`);return t}_createPopper(){if(void 0===s)throw new TypeError("Bootstrap's dropdowns require Popper (https://popper.js.org)");let t=this._element;"parent"===this._config.reference?t=this._parent:r(this._config.reference)?t=a(this._config.reference):"object"==typeof this._config.reference&&(t=this._config.reference);const e=this._getPopperConfig();this._popper=s.createPopper(t,this._menu,e)}_isShown(){return this._menu.classList.contains(At)}_getPlacement(){const t=this._parent;if(t.classList.contains("dropend"))return It;if(t.classList.contains("dropstart"))return Dt;if(t.classList.contains("dropup-center"))return"top";if(t.classList.contains("dropdown-center"))return"bottom";const e="end"===getComputedStyle(this._menu).getPropertyValue("--bs-position").trim();return t.classList.contains("dropup")?e?St:kt:e?Ot:Lt}_detectNavbar(){return null!==this._element.closest(".navbar")}_getOffset(){const{offset:t}=this._config;return"string"==typeof t?t.split(",").map((t=>Number.parseInt(t,10))):"function"==typeof t?e=>t(e,this._element):t}_getPopperConfig(){const t={placement:this._getPlacement(),modifiers:[{name:"preventOverflow",options:{boundary:this._config.boundary}},{name:"offset",options:{offset:this._getOffset()}}]};return(this._inNavbar||"static"===this._config.display)&&(B.setDataAttribute(this._menu,"popper","static"),t.modifiers=[{name:"applyStyles",enabled:!1}]),{...t,...p(this._config.popperConfig,[t])}}_selectMenuItem({key:t,target:e}){const s=V.find(".dropdown-menu .dropdown-item:not(.disabled):not(:disabled)",this._menu).filter((t=>l(t)));s.length&&v(s,e,t===vt,!s.includes(e)).focus()}static jQueryInterface(t){return this.each((function(){const e=xt.getOrCreateInstance(this,t);if("string"==typeof t){if(void 0===e[t])throw new TypeError(`No method named "${t}"`);e[t]()}}))}static clearMenus(t){if(2===t.button||"keyup"===t.type&&"Tab"!==t.key)return;const e=V.find(Ct);for(const s of e){const e=xt.getInstance(s);if(!e||!1===e._config.autoClose)continue;const i=t.composedPath(),n=i.includes(e._menu);if(i.includes(e._element)||"inside"===e._config.autoClose&&!n||"outside"===e._config.autoClose&&n)continue;if(e._menu.contains(t.target)&&("keyup"===t.type&&"Tab"===t.key||/input|select|option|textarea|form/i.test(t.target.tagName)))continue;const o={relatedTarget:e._element};"click"===t.type&&(o.clickEvent=t),e._completeHide(o)}}static dataApiKeydownHandler(t){const e=/input|textarea/i.test(t.target.tagName),s="Escape"===t.key,i=[bt,vt].includes(t.key);if(!i&&!s)return;if(e&&!s)return;t.preventDefault();const n=this.matches(Et)?this:V.prev(this,Et)[0]||V.next(this,Et)[0]||V.findOne(Et,t.delegateTarget.parentNode),o=xt.getOrCreateInstance(n);if(i)return t.stopPropagation(),o.show(),void o._selectMenuItem(t);o._isShown()&&(t.stopPropagation(),o.hide(),n.focus())}}M.on(document,wt,Et,xt.dataApiKeydownHandler),M.on(document,wt,Tt,xt.dataApiKeydownHandler),M.on(document,yt,xt.clearMenus),M.on(document,"keyup.bs.dropdown.data-api",xt.clearMenus),M.on(document,yt,Et,(function(t){t.preventDefault(),xt.getOrCreateInstance(this).toggle()})),m(xt);const Mt=".fixed-top, .fixed-bottom, .is-fixed, .sticky-top",jt=".sticky-top",Ft="padding-right",$t="margin-right";class zt{constructor(){this._element=document.body}getWidth(){const t=document.documentElement.clientWidth;return Math.abs(window.innerWidth-t)}hide(){const t=this.getWidth();this._disableOverFlow(),this._setElementAttributes(this._element,Ft,(e=>e+t)),this._setElementAttributes(Mt,Ft,(e=>e+t)),this._setElementAttributes(jt,$t,(e=>e-t))}reset(){this._resetElementAttributes(this._element,"overflow"),this._resetElementAttributes(this._element,Ft),this._resetElementAttributes(Mt,Ft),this._resetElementAttributes(jt,$t)}isOverflowing(){return this.getWidth()>0}_disableOverFlow(){this._saveInitialAttribute(this._element,"overflow"),this._element.style.overflow="hidden"}_setElementAttributes(t,e,s){const i=this.getWidth();this._applyManipulationCallback(t,(t=>{if(t!==this._element&&window.innerWidth>t.clientWidth+i)return;this._saveInitialAttribute(t,e);const n=window.getComputedStyle(t).getPropertyValue(e);t.style.setProperty(e,`${s(Number.parseFloat(n))}px`)}))}_saveInitialAttribute(t,e){const s=t.style.getPropertyValue(e);s&&B.setDataAttribute(t,e,s)}_resetElementAttributes(t,e){this._applyManipulationCallback(t,(t=>{const s=B.getDataAttribute(t,e);null!==s?(B.removeDataAttribute(t,e),t.style.setProperty(e,s)):t.style.removeProperty(e)}))}_applyManipulationCallback(t,e){if(r(t))e(t);else for(const s of V.find(t,this._element))e(s)}}const Ht="show",Bt="mousedown.bs.backdrop",qt={className:"modal-backdrop",clickCallback:null,isAnimated:!1,isVisible:!0,rootElement:"body"},Wt={className:"string",clickCallback:"(function|null)",isAnimated:"boolean",isVisible:"boolean",rootElement:"(element|string)"};class Rt extends q{constructor(t){super(),this._config=this._getConfig(t),this._isAppended=!1,this._element=null}static get Default(){return qt}static get DefaultType(){return Wt}static get NAME(){return"backdrop"}show(t){if(!this._config.isVisible)return void p(t);this._append();const e=this._getElement();this._config.isAnimated&&u(e),e.classList.add(Ht),this._emulateAnimation((()=>{p(t)}))}hide(t){this._config.isVisible?(this._getElement().classList.remove(Ht),this._emulateAnimation((()=>{this.dispose(),p(t)}))):p(t)}dispose(){this._isAppended&&(M.off(this._element,Bt),this._element.remove(),this._isAppended=!1)}_getElement(){if(!this._element){const t=document.createElement("div");t.className=this._config.className,this._config.isAnimated&&t.classList.add("fade"),this._element=t}return this._element}_configAfterMerge(t){return t.rootElement=a(t.rootElement),t}_append(){if(this._isAppended)return;const t=this._getElement();this._config.rootElement.append(t),M.on(t,Bt,(()=>{p(this._config.clickCallback)})),this._isAppended=!0}_emulateAnimation(t){b(t,this._getElement(),this._config.isAnimated)}}const Vt=".bs.focustrap",Kt="backward",Qt={autofocus:!0,trapElement:null},Xt={autofocus:"boolean",trapElement:"element"};class Yt extends q{constructor(t){super(),this._config=this._getConfig(t),this._isActive=!1,this._lastTabNavDirection=null}static get Default(){return Qt}static get DefaultType(){return Xt}static get NAME(){return"focustrap"}activate(){this._isActive||(this._config.autofocus&&this._config.trapElement.focus(),M.off(document,Vt),M.on(document,"focusin.bs.focustrap",(t=>this._handleFocusin(t))),M.on(document,"keydown.tab.bs.focustrap",(t=>this._handleKeydown(t))),this._isActive=!0)}deactivate(){this._isActive&&(this._isActive=!1,M.off(document,Vt))}_handleFocusin(t){const{trapElement:e}=this._config;if(t.target===document||t.target===e||e.contains(t.target))return;const s=V.focusableChildren(e);0===s.length?e.focus():this._lastTabNavDirection===Kt?s[s.length-1].focus():s[0].focus()}_handleKeydown(t){"Tab"===t.key&&(this._lastTabNavDirection=t.shiftKey?Kt:"forward")}}const Ut="hidden.bs.modal",Gt="show.bs.modal",Jt="modal-open",Zt="show",te="modal-static",ee={backdrop:!0,focus:!0,keyboard:!0},se={backdrop:"(boolean|string)",focus:"boolean",keyboard:"boolean"};class ie extends W{constructor(t,e){super(t,e),this._dialog=V.findOne(".modal-dialog",this._element),this._backdrop=this._initializeBackDrop(),this._focustrap=this._initializeFocusTrap(),this._isShown=!1,this._isTransitioning=!1,this._scrollBar=new zt,this._addEventListeners()}static get Default(){return ee}static get DefaultType(){return se}static get NAME(){return"modal"}toggle(t){return this._isShown?this.hide():this.show(t)}show(t){this._isShown||this._isTransitioning||M.trigger(this._element,Gt,{relatedTarget:t}).defaultPrevented||(this._isShown=!0,this._isTransitioning=!0,this._scrollBar.hide(),document.body.classList.add(Jt),this._adjustDialog(),this._backdrop.show((()=>this._showElement(t))))}hide(){this._isShown&&!this._isTransitioning&&(M.trigger(this._element,"hide.bs.modal").defaultPrevented||(this._isShown=!1,this._isTransitioning=!0,this._focustrap.deactivate(),this._element.classList.remove(Zt),this._queueCallback((()=>this._hideModal()),this._element,this._isAnimated())))}dispose(){for(const t of[window,this._dialog])M.off(t,".bs.modal");this._backdrop.dispose(),this._focustrap.deactivate(),super.dispose()}handleUpdate(){this._adjustDialog()}_initializeBackDrop(){return new Rt({isVisible:Boolean(this._config.backdrop),isAnimated:this._isAnimated()})}_initializeFocusTrap(){return new Yt({trapElement:this._element})}_showElement(t){document.body.contains(this._element)||document.body.append(this._element),this._element.style.display="block",this._element.removeAttribute("aria-hidden"),this._element.setAttribute("aria-modal",!0),this._element.setAttribute("role","dialog"),this._element.scrollTop=0;const e=V.findOne(".modal-body",this._dialog);e&&(e.scrollTop=0),u(this._element),this._element.classList.add(Zt),this._queueCallback((()=>{this._config.focus&&this._focustrap.activate(),this._isTransitioning=!1,M.trigger(this._element,"shown.bs.modal",{relatedTarget:t})}),this._dialog,this._isAnimated())}_addEventListeners(){M.on(this._element,"keydown.dismiss.bs.modal",(t=>{if("Escape"===t.key)return this._config.keyboard?(t.preventDefault(),void this.hide()):void this._triggerBackdropTransition()})),M.on(window,"resize.bs.modal",(()=>{this._isShown&&!this._isTransitioning&&this._adjustDialog()})),M.on(this._element,"mousedown.dismiss.bs.modal",(t=>{M.one(this._element,"click.dismiss.bs.modal",(e=>{this._element===t.target&&this._element===e.target&&("static"!==this._config.backdrop?this._config.backdrop&&this.hide():this._triggerBackdropTransition())}))}))}_hideModal(){this._element.style.display="none",this._element.setAttribute("aria-hidden",!0),this._element.removeAttribute("aria-modal"),this._element.removeAttribute("role"),this._isTransitioning=!1,this._backdrop.hide((()=>{document.body.classList.remove(Jt),this._resetAdjustments(),this._scrollBar.reset(),M.trigger(this._element,Ut)}))}_isAnimated(){return this._element.classList.contains("fade")}_triggerBackdropTransition(){if(M.trigger(this._element,"hidePrevented.bs.modal").defaultPrevented)return;const t=this._element.scrollHeight>document.documentElement.clientHeight,e=this._element.style.overflowY;"hidden"===e||this._element.classList.contains(te)||(t||(this._element.style.overflowY="hidden"),this._element.classList.add(te),this._queueCallback((()=>{this._element.classList.remove(te),this._queueCallback((()=>{this._element.style.overflowY=e}),this._dialog)}),this._dialog),this._element.focus())}_adjustDialog(){const t=this._element.scrollHeight>document.documentElement.clientHeight,e=this._scrollBar.getWidth(),s=e>0;if(s&&!t){const t=f()?"paddingLeft":"paddingRight";this._element.style[t]=`${e}px`}if(!s&&t){const t=f()?"paddingRight":"paddingLeft";this._element.style[t]=`${e}px`}}_resetAdjustments(){this._element.style.paddingLeft="",this._element.style.paddingRight=""}static jQueryInterface(t,e){return this.each((function(){const s=ie.getOrCreateInstance(this,t);if("string"==typeof t){if(void 0===s[t])throw new TypeError(`No method named "${t}"`);s[t](e)}}))}}M.on(document,"click.bs.modal.data-api",'[data-bs-toggle="modal"]',(function(t){const e=V.getElementFromSelector(this);["A","AREA"].includes(this.tagName)&&t.preventDefault(),M.one(e,Gt,(t=>{t.defaultPrevented||M.one(e,Ut,(()=>{l(this)&&this.focus()}))}));const s=V.findOne(".modal.show");s&&ie.getInstance(s).hide(),ie.getOrCreateInstance(e).toggle(this)})),K(ie),m(ie);const ne="show",oe="showing",re="hiding",ae=".offcanvas.show",le="hidePrevented.bs.offcanvas",ce="hidden.bs.offcanvas",he={backdrop:!0,keyboard:!0,scroll:!1},de={backdrop:"(boolean|string)",keyboard:"boolean",scroll:"boolean"};class ue extends W{constructor(t,e){super(t,e),this._isShown=!1,this._backdrop=this._initializeBackDrop(),this._focustrap=this._initializeFocusTrap(),this._addEventListeners()}static get Default(){return he}static get DefaultType(){return de}static get NAME(){return"offcanvas"}toggle(t){return this._isShown?this.hide():this.show(t)}show(t){this._isShown||M.trigger(this._element,"show.bs.offcanvas",{relatedTarget:t}).defaultPrevented||(this._isShown=!0,this._backdrop.show(),this._config.scroll||(new zt).hide(),this._element.setAttribute("aria-modal",!0),this._element.setAttribute("role","dialog"),this._element.classList.add(oe),this._queueCallback((()=>{this._config.scroll&&!this._config.backdrop||this._focustrap.activate(),this._element.classList.add(ne),this._element.classList.remove(oe),M.trigger(this._element,"shown.bs.offcanvas",{relatedTarget:t})}),this._element,!0))}hide(){this._isShown&&(M.trigger(this._element,"hide.bs.offcanvas").defaultPrevented||(this._focustrap.deactivate(),this._element.blur(),this._isShown=!1,this._element.classList.add(re),this._backdrop.hide(),this._queueCallback((()=>{this._element.classList.remove(ne,re),this._element.removeAttribute("aria-modal"),this._element.removeAttribute("role"),this._config.scroll||(new zt).reset(),M.trigger(this._element,ce)}),this._element,!0)))}dispose(){this._backdrop.dispose(),this._focustrap.deactivate(),super.dispose()}_initializeBackDrop(){const t=Boolean(this._config.backdrop);return new Rt({className:"offcanvas-backdrop",isVisible:t,isAnimated:!0,rootElement:this._element.parentNode,clickCallback:t?()=>{"static"!==this._config.backdrop?this.hide():M.trigger(this._element,le)}:null})}_initializeFocusTrap(){return new Yt({trapElement:this._element})}_addEventListeners(){M.on(this._element,"keydown.dismiss.bs.offcanvas",(t=>{"Escape"===t.key&&(this._config.keyboard?this.hide():M.trigger(this._element,le))}))}static jQueryInterface(t){return this.each((function(){const e=ue.getOrCreateInstance(this,t);if("string"==typeof t){if(void 0===e[t]||t.startsWith("_")||"constructor"===t)throw new TypeError(`No method named "${t}"`);e[t](this)}}))}}M.on(document,"click.bs.offcanvas.data-api",'[data-bs-toggle="offcanvas"]',(function(t){const e=V.getElementFromSelector(this);if(["A","AREA"].includes(this.tagName)&&t.preventDefault(),c(this))return;M.one(e,ce,(()=>{l(this)&&this.focus()}));const s=V.findOne(ae);s&&s!==e&&ue.getInstance(s).hide(),ue.getOrCreateInstance(e).toggle(this)})),M.on(window,"load.bs.offcanvas.data-api",(()=>{for(const t of V.find(ae))ue.getOrCreateInstance(t).show()})),M.on(window,"resize.bs.offcanvas",(()=>{for(const t of V.find("[aria-modal][class*=show][class*=offcanvas-]"))"fixed"!==getComputedStyle(t).position&&ue.getOrCreateInstance(t).hide()})),K(ue),m(ue);const _e=new Set(["background","cite","href","itemtype","longdesc","poster","src","xlink:href"]),ge=/^(?:(?:https?|mailto|ftp|tel|file|sms):|[^#&/:?]*(?:[#/?]|$))/i,fe=/^data:(?:image\/(?:bmp|gif|jpeg|jpg|png|tiff|webp)|video\/(?:mpeg|mp4|ogg|webm)|audio\/(?:mp3|oga|ogg|opus));base64,[\d+/a-z]+=*$/i,me=(t,e)=>{const s=t.nodeName.toLowerCase();return e.includes(s)?!_e.has(s)||Boolean(ge.test(t.nodeValue)||fe.test(t.nodeValue)):e.filter((t=>t instanceof RegExp)).some((t=>t.test(s)))},pe={"*":["class","dir","id","lang","role",/^aria-[\w-]*$/i],a:["target","href","title","rel"],area:[],b:[],br:[],col:[],code:[],div:[],em:[],hr:[],h1:[],h2:[],h3:[],h4:[],h5:[],h6:[],i:[],img:["src","srcset","alt","title","width","height"],li:[],ol:[],p:[],pre:[],s:[],small:[],span:[],sub:[],sup:[],strong:[],u:[],ul:[]},be={allowList:pe,content:{},extraClass:"",html:!1,sanitize:!0,sanitizeFn:null,template:"<div></div>"},ve={allowList:"object",content:"object",extraClass:"(string|function)",html:"boolean",sanitize:"boolean",sanitizeFn:"(null|function)",template:"string"},ye={entry:"(string|element|function|null)",selector:"(string|element)"};class we extends q{constructor(t){super(),this._config=this._getConfig(t)}static get Default(){return be}static get DefaultType(){return ve}static get NAME(){return"TemplateFactory"}getContent(){return Object.values(this._config.content).map((t=>this._resolvePossibleFunction(t))).filter(Boolean)}hasContent(){return this.getContent().length>0}changeContent(t){return this._checkContent(t),this._config.content={...this._config.content,...t},this}toHtml(){const t=document.createElement("div");t.innerHTML=this._maybeSanitize(this._config.template);for(const[e,s]of Object.entries(this._config.content))this._setContent(t,s,e);const e=t.children[0],s=this._resolvePossibleFunction(this._config.extraClass);return s&&e.classList.add(...s.split(" ")),e}_typeCheckConfig(t){super._typeCheckConfig(t),this._checkContent(t.content)}_checkContent(t){for(const[e,s]of Object.entries(t))super._typeCheckConfig({selector:e,entry:s},ye)}_setContent(t,e,s){const i=V.findOne(s,t);i&&((e=this._resolvePossibleFunction(e))?r(e)?this._putElementInTemplate(a(e),i):this._config.html?i.innerHTML=this._maybeSanitize(e):i.textContent=e:i.remove())}_maybeSanitize(t){return this._config.sanitize?function(t,e,s){if(!t.length)return t;if(s&&"function"==typeof s)return s(t);const i=(new window.DOMParser).parseFromString(t,"text/html"),n=[].concat(...i.body.querySelectorAll("*"));for(const t of n){const s=t.nodeName.toLowerCase();if(!Object.keys(e).includes(s)){t.remove();continue}const i=[].concat(...t.attributes),n=[].concat(e["*"]||[],e[s]||[]);for(const e of i)me(e,n)||t.removeAttribute(e.nodeName)}return i.body.innerHTML}(t,this._config.allowList,this._config.sanitizeFn):t}_resolvePossibleFunction(t){return p(t,[this])}_putElementInTemplate(t,e){if(this._config.html)return e.innerHTML="",void e.append(t);e.textContent=t.textContent}}const Ae=new Set(["sanitize","allowList","sanitizeFn"]),Ee="fade",Ce="show",Te=".modal",ke="hide.bs.modal",Se="hover",Le="focus",Oe={AUTO:"auto",TOP:"top",RIGHT:f()?"left":"right",BOTTOM:"bottom",LEFT:f()?"right":"left"},Ie={allowList:pe,animation:!0,boundary:"clippingParents",container:!1,customClass:"",delay:0,fallbackPlacements:["top","right","bottom","left"],html:!1,offset:[0,0],placement:"top",popperConfig:null,sanitize:!0,sanitizeFn:null,selector:!1,template:'<div class="tooltip" role="tooltip"><div class="tooltip-arrow"></div><div class="tooltip-inner"></div></div>',title:"",trigger:"hover focus"},De={allowList:"object",animation:"boolean",boundary:"(string|element)",container:"(string|element|boolean)",customClass:"(string|function)",delay:"(number|object)",fallbackPlacements:"array",html:"boolean",offset:"(array|string|function)",placement:"(string|function)",popperConfig:"(null|object|function)",sanitize:"boolean",sanitizeFn:"(null|function)",selector:"(string|boolean)",template:"string",title:"(string|element|function)",trigger:"string"};class Ne extends W{constructor(t,e){if(void 0===s)throw new TypeError("Bootstrap's tooltips require Popper (https://popper.js.org)");super(t,e),this._isEnabled=!0,this._timeout=0,this._isHovered=null,this._activeTrigger={},this._popper=null,this._templateFactory=null,this._newContent=null,this.tip=null,this._setListeners(),this._config.selector||this._fixTitle()}static get Default(){return Ie}static get DefaultType(){return De}static get NAME(){return"tooltip"}enable(){this._isEnabled=!0}disable(){this._isEnabled=!1}toggleEnabled(){this._isEnabled=!this._isEnabled}toggle(){this._isEnabled&&(this._activeTrigger.click=!this._activeTrigger.click,this._isShown()?this._leave():this._enter())}dispose(){clearTimeout(this._timeout),M.off(this._element.closest(Te),ke,this._hideModalHandler),this._element.getAttribute("data-bs-original-title")&&this._element.setAttribute("title",this._element.getAttribute("data-bs-original-title")),this._disposePopper(),super.dispose()}show(){if("none"===this._element.style.display)throw new Error("Please use show on visible elements");if(!this._isWithContent()||!this._isEnabled)return;const t=M.trigger(this._element,this.constructor.eventName("show")),e=(h(this._element)||this._element.ownerDocument.documentElement).contains(this._element);if(t.defaultPrevented||!e)return;this._disposePopper();const s=this._getTipElement();this._element.setAttribute("aria-describedby",s.getAttribute("id"));const{container:i}=this._config;if(this._element.ownerDocument.documentElement.contains(this.tip)||(i.append(s),M.trigger(this._element,this.constructor.eventName("inserted"))),this._popper=this._createPopper(s),s.classList.add(Ce),"ontouchstart"in document.documentElement)for(const t of[].concat(...document.body.children))M.on(t,"mouseover",d);this._queueCallback((()=>{M.trigger(this._element,this.constructor.eventName("shown")),!1===this._isHovered&&this._leave(),this._isHovered=!1}),this.tip,this._isAnimated())}hide(){if(this._isShown()&&!M.trigger(this._element,this.constructor.eventName("hide")).defaultPrevented){if(this._getTipElement().classList.remove(Ce),"ontouchstart"in document.documentElement)for(const t of[].concat(...document.body.children))M.off(t,"mouseover",d);this._activeTrigger.click=!1,this._activeTrigger.focus=!1,this._activeTrigger.hover=!1,this._isHovered=null,this._queueCallback((()=>{this._isWithActiveTrigger()||(this._isHovered||this._disposePopper(),this._element.removeAttribute("aria-describedby"),M.trigger(this._element,this.constructor.eventName("hidden")))}),this.tip,this._isAnimated())}}update(){this._popper&&this._popper.update()}_isWithContent(){return Boolean(this._getTitle())}_getTipElement(){return this.tip||(this.tip=this._createTipElement(this._newContent||this._getContentForTemplate())),this.tip}_createTipElement(t){const e=this._getTemplateFactory(t).toHtml();if(!e)return null;e.classList.remove(Ee,Ce),e.classList.add(`bs-${this.constructor.NAME}-auto`);const s=(t=>{do{t+=Math.floor(1e6*Math.random())}while(document.getElementById(t));return t})(this.constructor.NAME).toString();return e.setAttribute("id",s),this._isAnimated()&&e.classList.add(Ee),e}setContent(t){this._newContent=t,this._isShown()&&(this._disposePopper(),this.show())}_getTemplateFactory(t){return this._templateFactory?this._templateFactory.changeContent(t):this._templateFactory=new we({...this._config,content:t,extraClass:this._resolvePossibleFunction(this._config.customClass)}),this._templateFactory}_getContentForTemplate(){return{".tooltip-inner":this._getTitle()}}_getTitle(){return this._resolvePossibleFunction(this._config.title)||this._element.getAttribute("data-bs-original-title")}_initializeOnDelegatedTarget(t){return this.constructor.getOrCreateInstance(t.delegateTarget,this._getDelegateConfig())}_isAnimated(){return this._config.animation||this.tip&&this.tip.classList.contains(Ee)}_isShown(){return this.tip&&this.tip.classList.contains(Ce)}_createPopper(t){const e=p(this._config.placement,[this,t,this._element]),i=Oe[e.toUpperCase()];return s.createPopper(this._element,t,this._getPopperConfig(i))}_getOffset(){const{offset:t}=this._config;return"string"==typeof t?t.split(",").map((t=>Number.parseInt(t,10))):"function"==typeof t?e=>t(e,this._element):t}_resolvePossibleFunction(t){return p(t,[this._element])}_getPopperConfig(t){const e={placement:t,modifiers:[{name:"flip",options:{fallbackPlacements:this._config.fallbackPlacements}},{name:"offset",options:{offset:this._getOffset()}},{name:"preventOverflow",options:{boundary:this._config.boundary}},{name:"arrow",options:{element:`.${this.constructor.NAME}-arrow`}},{name:"preSetPlacement",enabled:!0,phase:"beforeMain",fn:t=>{this._getTipElement().setAttribute("data-popper-placement",t.state.placement)}}]};return{...e,...p(this._config.popperConfig,[e])}}_setListeners(){const t=this._config.trigger.split(" ");for(const e of t)if("click"===e)M.on(this._element,this.constructor.eventName("click"),this._config.selector,(t=>{this._initializeOnDelegatedTarget(t).toggle()}));else if("manual"!==e){const t=e===Se?this.constructor.eventName("mouseenter"):this.constructor.eventName("focusin"),s=e===Se?this.constructor.eventName("mouseleave"):this.constructor.eventName("focusout");M.on(this._element,t,this._config.selector,(t=>{const e=this._initializeOnDelegatedTarget(t);e._activeTrigger["focusin"===t.type?Le:Se]=!0,e._enter()})),M.on(this._element,s,this._config.selector,(t=>{const e=this._initializeOnDelegatedTarget(t);e._activeTrigger["focusout"===t.type?Le:Se]=e._element.contains(t.relatedTarget),e._leave()}))}this._hideModalHandler=()=>{this._element&&this.hide()},M.on(this._element.closest(Te),ke,this._hideModalHandler)}_fixTitle(){const t=this._element.getAttribute("title");t&&(this._element.getAttribute("aria-label")||this._element.textContent.trim()||this._element.setAttribute("aria-label",t),this._element.setAttribute("data-bs-original-title",t),this._element.removeAttribute("title"))}_enter(){this._isShown()||this._isHovered?this._isHovered=!0:(this._isHovered=!0,this._setTimeout((()=>{this._isHovered&&this.show()}),this._config.delay.show))}_leave(){this._isWithActiveTrigger()||(this._isHovered=!1,this._setTimeout((()=>{this._isHovered||this.hide()}),this._config.delay.hide))}_setTimeout(t,e){clearTimeout(this._timeout),this._timeout=setTimeout(t,e)}_isWithActiveTrigger(){return Object.values(this._activeTrigger).includes(!0)}_getConfig(t){const e=B.getDataAttributes(this._element);for(const t of Object.keys(e))Ae.has(t)&&delete e[t];return t={...e,..."object"==typeof t&&t?t:{}},t=this._mergeConfigObj(t),t=this._configAfterMerge(t),this._typeCheckConfig(t),t}_configAfterMerge(t){return t.container=!1===t.container?document.body:a(t.container),"number"==typeof t.delay&&(t.delay={show:t.delay,hide:t.delay}),"number"==typeof t.title&&(t.title=t.title.toString()),"number"==typeof t.content&&(t.content=t.content.toString()),t}_getDelegateConfig(){const t={};for(const[e,s]of Object.entries(this._config))this.constructor.Default[e]!==s&&(t[e]=s);return t.selector=!1,t.trigger="manual",t}_disposePopper(){this._popper&&(this._popper.destroy(),this._popper=null),this.tip&&(this.tip.remove(),this.tip=null)}static jQueryInterface(t){return this.each((function(){const e=Ne.getOrCreateInstance(this,t);if("string"==typeof t){if(void 0===e[t])throw new TypeError(`No method named "${t}"`);e[t]()}}))}}m(Ne);const Pe={...Ne.Default,content:"",offset:[0,8],placement:"right",template:'<div class="popover" role="tooltip"><div class="popover-arrow"></div><h3 class="popover-header"></h3><div class="popover-body"></div></div>',trigger:"click"},xe={...Ne.DefaultType,content:"(null|string|element|function)"};class Me extends Ne{static get Default(){return Pe}static get DefaultType(){return xe}static get NAME(){return"popover"}_isWithContent(){return this._getTitle()||this._getContent()}_getContentForTemplate(){return{".popover-header":this._getTitle(),".popover-body":this._getContent()}}_getContent(){return this._resolvePossibleFunction(this._config.content)}static jQueryInterface(t){return this.each((function(){const e=Me.getOrCreateInstance(this,t);if("string"==typeof t){if(void 0===e[t])throw new TypeError(`No method named "${t}"`);e[t]()}}))}}m(Me);const je="click.bs.scrollspy",Fe="active",$e="[href]",ze={offset:null,rootMargin:"0px 0px -25%",smoothScroll:!1,target:null,threshold:[.1,.5,1]},He={offset:"(number|null)",rootMargin:"string",smoothScroll:"boolean",target:"element",threshold:"array"};class Be extends W{constructor(t,e){super(t,e),this._targetLinks=new Map,this._observableSections=new Map,this._rootElement="visible"===getComputedStyle(this._element).overflowY?null:this._element,this._activeTarget=null,this._observer=null,this._previousScrollData={visibleEntryTop:0,parentScrollTop:0},this.refresh()}static get Default(){return ze}static get DefaultType(){return He}static get NAME(){return"scrollspy"}refresh(){this._initializeTargetsAndObservables(),this._maybeEnableSmoothScroll(),this._observer?this._observer.disconnect():this._observer=this._getNewObserver();for(const t of this._observableSections.values())this._observer.observe(t)}dispose(){this._observer.disconnect(),super.dispose()}_configAfterMerge(t){return t.target=a(t.target)||document.body,t.rootMargin=t.offset?`${t.offset}px 0px -30%`:t.rootMargin,"string"==typeof t.threshold&&(t.threshold=t.threshold.split(",").map((t=>Number.parseFloat(t)))),t}_maybeEnableSmoothScroll(){this._config.smoothScroll&&(M.off(this._config.target,je),M.on(this._config.target,je,$e,(t=>{const e=this._observableSections.get(t.target.hash);if(e){t.preventDefault();const s=this._rootElement||window,i=e.offsetTop-this._element.offsetTop;if(s.scrollTo)return void s.scrollTo({top:i,behavior:"smooth"});s.scrollTop=i}})))}_getNewObserver(){const t={root:this._rootElement,threshold:this._config.threshold,rootMargin:this._config.rootMargin};return new IntersectionObserver((t=>this._observerCallback(t)),t)}_observerCallback(t){const e=t=>this._targetLinks.get(`#${t.target.id}`),s=t=>{this._previousScrollData.visibleEntryTop=t.target.offsetTop,this._process(e(t))},i=(this._rootElement||document.documentElement).scrollTop,n=i>=this._previousScrollData.parentScrollTop;this._previousScrollData.parentScrollTop=i;for(const o of t){if(!o.isIntersecting){this._activeTarget=null,this._clearActiveClass(e(o));continue}const t=o.target.offsetTop>=this._previousScrollData.visibleEntryTop;if(n&&t){if(s(o),!i)return}else n||t||s(o)}}_initializeTargetsAndObservables(){this._targetLinks=new Map,this._observableSections=new Map;const t=V.find($e,this._config.target);for(const e of t){if(!e.hash||c(e))continue;const t=V.findOne(e.hash,this._element);l(t)&&(this._targetLinks.set(e.hash,e),this._observableSections.set(e.hash,t))}}_process(t){this._activeTarget!==t&&(this._clearActiveClass(this._config.target),this._activeTarget=t,t.classList.add(Fe),this._activateParents(t),M.trigger(this._element,"activate.bs.scrollspy",{relatedTarget:t}))}_activateParents(t){if(t.classList.contains("dropdown-item"))V.findOne(".dropdown-toggle",t.closest(".dropdown")).classList.add(Fe);else for(const e of V.parents(t,".nav, .list-group"))for(const t of V.prev(e,".nav-link, .nav-item > .nav-link, .list-group-item"))t.classList.add(Fe)}_clearActiveClass(t){t.classList.remove(Fe);const e=V.find("[href].active",t);for(const t of e)t.classList.remove(Fe)}static jQueryInterface(t){return this.each((function(){const e=Be.getOrCreateInstance(this,t);if("string"==typeof t){if(void 0===e[t]||t.startsWith("_")||"constructor"===t)throw new TypeError(`No method named "${t}"`);e[t]()}}))}}M.on(window,"load.bs.scrollspy.data-api",(()=>{for(const t of V.find('[data-bs-spy="scroll"]'))Be.getOrCreateInstance(t)})),m(Be);const qe="ArrowLeft",We="ArrowRight",Re="ArrowUp",Ve="ArrowDown",Ke="active",Qe="fade",Xe="show",Ye='[data-bs-toggle="tab"], [data-bs-toggle="pill"], [data-bs-toggle="list"]',Ue=`.nav-link:not(.dropdown-toggle), .list-group-item:not(.dropdown-toggle), [role="tab"]:not(.dropdown-toggle), ${Ye}`;class Ge extends W{constructor(t){super(t),this._parent=this._element.closest('.list-group, .nav, [role="tablist"]'),this._parent&&(this._setInitialAttributes(this._parent,this._getChildren()),M.on(this._element,"keydown.bs.tab",(t=>this._keydown(t))))}static get NAME(){return"tab"}show(){const t=this._element;if(this._elemIsActive(t))return;const e=this._getActiveElem(),s=e?M.trigger(e,"hide.bs.tab",{relatedTarget:t}):null;M.trigger(t,"show.bs.tab",{relatedTarget:e}).defaultPrevented||s&&s.defaultPrevented||(this._deactivate(e,t),this._activate(t,e))}_activate(t,e){t&&(t.classList.add(Ke),this._activate(V.getElementFromSelector(t)),this._queueCallback((()=>{"tab"===t.getAttribute("role")?(t.removeAttribute("tabindex"),t.setAttribute("aria-selected",!0),this._toggleDropDown(t,!0),M.trigger(t,"shown.bs.tab",{relatedTarget:e})):t.classList.add(Xe)}),t,t.classList.contains(Qe)))}_deactivate(t,e){t&&(t.classList.remove(Ke),t.blur(),this._deactivate(V.getElementFromSelector(t)),this._queueCallback((()=>{"tab"===t.getAttribute("role")?(t.setAttribute("aria-selected",!1),t.setAttribute("tabindex","-1"),this._toggleDropDown(t,!1),M.trigger(t,"hidden.bs.tab",{relatedTarget:e})):t.classList.remove(Xe)}),t,t.classList.contains(Qe)))}_keydown(t){if(![qe,We,Re,Ve].includes(t.key))return;t.stopPropagation(),t.preventDefault();const e=[We,Ve].includes(t.key),s=v(this._getChildren().filter((t=>!c(t))),t.target,e,!0);s&&(s.focus({preventScroll:!0}),Ge.getOrCreateInstance(s).show())}_getChildren(){return V.find(Ue,this._parent)}_getActiveElem(){return this._getChildren().find((t=>this._elemIsActive(t)))||null}_setInitialAttributes(t,e){this._setAttributeIfNotExists(t,"role","tablist");for(const t of e)this._setInitialAttributesOnChild(t)}_setInitialAttributesOnChild(t){t=this._getInnerElement(t);const e=this._elemIsActive(t),s=this._getOuterElement(t);t.setAttribute("aria-selected",e),s!==t&&this._setAttributeIfNotExists(s,"role","presentation"),e||t.setAttribute("tabindex","-1"),this._setAttributeIfNotExists(t,"role","tab"),this._setInitialAttributesOnTargetPanel(t)}_setInitialAttributesOnTargetPanel(t){const e=V.getElementFromSelector(t);e&&(this._setAttributeIfNotExists(e,"role","tabpanel"),t.id&&this._setAttributeIfNotExists(e,"aria-labelledby",`#${t.id}`))}_toggleDropDown(t,e){const s=this._getOuterElement(t);if(!s.classList.contains("dropdown"))return;const i=(t,i)=>{const n=V.findOne(t,s);n&&n.classList.toggle(i,e)};i(".dropdown-toggle",Ke),i(".dropdown-menu",Xe),s.setAttribute("aria-expanded",e)}_setAttributeIfNotExists(t,e,s){t.hasAttribute(e)||t.setAttribute(e,s)}_elemIsActive(t){return t.classList.contains(Ke)}_getInnerElement(t){return t.matches(Ue)?t:V.findOne(Ue,t)}_getOuterElement(t){return t.closest(".nav-item, .list-group-item")||t}static jQueryInterface(t){return this.each((function(){const e=Ge.getOrCreateInstance(this);if("string"==typeof t){if(void 0===e[t]||t.startsWith("_")||"constructor"===t)throw new TypeError(`No method named "${t}"`);e[t]()}}))}}M.on(document,"click.bs.tab",Ye,(function(t){["A","AREA"].includes(this.tagName)&&t.preventDefault(),c(this)||Ge.getOrCreateInstance(this).show()})),M.on(window,"load.bs.tab",(()=>{for(const t of V.find('.active[data-bs-toggle="tab"], .active[data-bs-toggle="pill"], .active[data-bs-toggle="list"]'))Ge.getOrCreateInstance(t)})),m(Ge);const Je="hide",Ze="show",ts="showing",es={animation:"boolean",autohide:"boolean",delay:"number"},ss={animation:!0,autohide:!0,delay:5e3};class is extends W{constructor(t,e){super(t,e),this._timeout=null,this._hasMouseInteraction=!1,this._hasKeyboardInteraction=!1,this._setListeners()}static get Default(){return ss}static get DefaultType(){return es}static get NAME(){return"toast"}show(){M.trigger(this._element,"show.bs.toast").defaultPrevented||(this._clearTimeout(),this._config.animation&&this._element.classList.add("fade"),this._element.classList.remove(Je),u(this._element),this._element.classList.add(Ze,ts),this._queueCallback((()=>{this._element.classList.remove(ts),M.trigger(this._element,"shown.bs.toast"),this._maybeScheduleHide()}),this._element,this._config.animation))}hide(){this.isShown()&&(M.trigger(this._element,"hide.bs.toast").defaultPrevented||(this._element.classList.add(ts),this._queueCallback((()=>{this._element.classList.add(Je),this._element.classList.remove(ts,Ze),M.trigger(this._element,"hidden.bs.toast")}),this._element,this._config.animation)))}dispose(){this._clearTimeout(),this.isShown()&&this._element.classList.remove(Ze),super.dispose()}isShown(){return this._element.classList.contains(Ze)}_maybeScheduleHide(){this._config.autohide&&(this._hasMouseInteraction||this._hasKeyboardInteraction||(this._timeout=setTimeout((()=>{this.hide()}),this._config.delay)))}_onInteraction(t,e){switch(t.type){case"mouseover":case"mouseout":this._hasMouseInteraction=e;break;case"focusin":case"focusout":this._hasKeyboardInteraction=e}if(e)return void this._clearTimeout();const s=t.relatedTarget;this._element===s||this._element.contains(s)||this._maybeScheduleHide()}_setListeners(){M.on(this._element,"mouseover.bs.toast",(t=>this._onInteraction(t,!0))),M.on(this._element,"mouseout.bs.toast",(t=>this._onInteraction(t,!1))),M.on(this._element,"focusin.bs.toast",(t=>this._onInteraction(t,!0))),M.on(this._element,"focusout.bs.toast",(t=>this._onInteraction(t,!1)))}_clearTimeout(){clearTimeout(this._timeout),this._timeout=null}static jQueryInterface(t){return this.each((function(){const e=is.getOrCreateInstance(this,t);if("string"==typeof t){if(void 0===e[t])throw new TypeError(`No method named "${t}"`);e[t](this)}}))}}return K(is),m(is),{Alert:Q,Button:Y,Carousel:ct,Collapse:mt,Dropdown:xt,Modal:ie,Offcanvas:ue,Popover:Me,ScrollSpy:Be,Tab:Ge,Toast:is,Tooltip:Ne}}));

(function(factory) {
    if (typeof define === "function" && define.amd) {
        // AMD. Register as an anonymous module.
        define([], factory);
    } else if (typeof exports === "object") {
        // Node/CommonJS
        module.exports = factory();
    } else {
        // Browser globals
        window.wNumb = factory();
    }
})(function() {
    "use strict";

    var FormatOptions = [
        "decimals",
        "thousand",
        "mark",
        "prefix",
        "suffix",
        "encoder",
        "decoder",
        "negativeBefore",
        "negative",
        "edit",
        "undo"
    ];

    // General

    // Reverse a string
    function strReverse(a) {
        return a
            .split("")
            .reverse()
            .join("");
    }

    // Check if a string starts with a specified prefix.
    function strStartsWith(input, match) {
        return input.substring(0, match.length) === match;
    }

    // Check is a string ends in a specified suffix.
    function strEndsWith(input, match) {
        return input.slice(-1 * match.length) === match;
    }

    // Throw an error if formatting options are incompatible.
    function throwEqualError(F, a, b) {
        if ((F[a] || F[b]) && F[a] === F[b]) {
            throw new Error(a);
        }
    }

    // Check if a number is finite and not NaN
    function isValidNumber(input) {
        return typeof input === "number" && isFinite(input);
    }

    // Provide rounding-accurate toFixed method.
    // Borrowed: http://stackoverflow.com/a/21323330/775265
    function toFixed(value, exp) {
        value = value.toString().split("e");
        value = Math.round(+(value[0] + "e" + (value[1] ? +value[1] + exp : exp)));
        value = value.toString().split("e");
        return (+(value[0] + "e" + (value[1] ? +value[1] - exp : -exp))).toFixed(exp);
    }

    // Formatting

    // Accept a number as input, output formatted string.
    function formatTo(
        decimals,
        thousand,
        mark,
        prefix,
        suffix,
        encoder,
        decoder,
        negativeBefore,
        negative,
        edit,
        undo,
        input
    ) {
        var originalInput = input,
            inputIsNegative,
            inputPieces,
            inputBase,
            inputDecimals = "",
            output = "";

        // Apply user encoder to the input.
        // Expected outcome: number.
        if (encoder) {
            input = encoder(input);
        }

        // Stop if no valid number was provided, the number is infinite or NaN.
        if (!isValidNumber(input)) {
            return false;
        }

        // Rounding away decimals might cause a value of -0
        // when using very small ranges. Remove those cases.
        if (decimals !== false && parseFloat(input.toFixed(decimals)) === 0) {
            input = 0;
        }

        // Formatting is done on absolute numbers,
        // decorated by an optional negative symbol.
        if (input < 0) {
            inputIsNegative = true;
            input = Math.abs(input);
        }

        // Reduce the number of decimals to the specified option.
        if (decimals !== false) {
            input = toFixed(input, decimals);
        }

        // Transform the number into a string, so it can be split.
        input = input.toString();

        // Break the number on the decimal separator.
        if (input.indexOf(".") !== -1) {
            inputPieces = input.split(".");

            inputBase = inputPieces[0];

            if (mark) {
                inputDecimals = mark + inputPieces[1];
            }
        } else {
            // If it isn't split, the entire number will do.
            inputBase = input;
        }

        // Group numbers in sets of three.
        if (thousand) {
            inputBase = strReverse(inputBase).match(/.{1,3}/g);
            inputBase = strReverse(inputBase.join(strReverse(thousand)));
        }

        // If the number is negative, prefix with negation symbol.
        if (inputIsNegative && negativeBefore) {
            output += negativeBefore;
        }

        // Prefix the number
        if (prefix) {
            output += prefix;
        }

        // Normal negative option comes after the prefix. Defaults to '-'.
        if (inputIsNegative && negative) {
            output += negative;
        }

        // Append the actual number.
        output += inputBase;
        output += inputDecimals;

        // Apply the suffix.
        if (suffix) {
            output += suffix;
        }

        // Run the output through a user-specified post-formatter.
        if (edit) {
            output = edit(output, originalInput);
        }

        // All done.
        return output;
    }

    // Accept a sting as input, output decoded number.
    function formatFrom(
        decimals,
        thousand,
        mark,
        prefix,
        suffix,
        encoder,
        decoder,
        negativeBefore,
        negative,
        edit,
        undo,
        input
    ) {
        var originalInput = input,
            inputIsNegative,
            output = "";

        // User defined pre-decoder. Result must be a non empty string.
        if (undo) {
            input = undo(input);
        }

        // Test the input. Can't be empty.
        if (!input || typeof input !== "string") {
            return false;
        }

        // If the string starts with the negativeBefore value: remove it.
        // Remember is was there, the number is negative.
        if (negativeBefore && strStartsWith(input, negativeBefore)) {
            input = input.replace(negativeBefore, "");
            inputIsNegative = true;
        }

        // Repeat the same procedure for the prefix.
        if (prefix && strStartsWith(input, prefix)) {
            input = input.replace(prefix, "");
        }

        // And again for negative.
        if (negative && strStartsWith(input, negative)) {
            input = input.replace(negative, "");
            inputIsNegative = true;
        }

        // Remove the suffix.
        // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/slice
        if (suffix && strEndsWith(input, suffix)) {
            input = input.slice(0, -1 * suffix.length);
        }

        // Remove the thousand grouping.
        if (thousand) {
            input = input.split(thousand).join("");
        }

        // Set the decimal separator back to period.
        if (mark) {
            input = input.replace(mark, ".");
        }

        // Prepend the negative symbol.
        if (inputIsNegative) {
            output += "-";
        }

        // Add the number
        output += input;

        // Trim all non-numeric characters (allow '.' and '-');
        output = output.replace(/[^0-9\.\-.]/g, "");

        // The value contains no parse-able number.
        if (output === "") {
            return false;
        }

        // Covert to number.
        output = Number(output);

        // Run the user-specified post-decoder.
        if (decoder) {
            output = decoder(output);
        }

        // Check is the output is valid, otherwise: return false.
        if (!isValidNumber(output)) {
            return false;
        }

        return output;
    }

    // Framework

    // Validate formatting options
    function validate(inputOptions) {
        var i,
            optionName,
            optionValue,
            filteredOptions = {};

        if (inputOptions["suffix"] === undefined) {
            inputOptions["suffix"] = inputOptions["postfix"];
        }

        for (i = 0; i < FormatOptions.length; i += 1) {
            optionName = FormatOptions[i];
            optionValue = inputOptions[optionName];

            if (optionValue === undefined) {
                // Only default if negativeBefore isn't set.
                if (optionName === "negative" && !filteredOptions.negativeBefore) {
                    filteredOptions[optionName] = "-";
                    // Don't set a default for mark when 'thousand' is set.
                } else if (optionName === "mark" && filteredOptions.thousand !== ".") {
                    filteredOptions[optionName] = ".";
                } else {
                    filteredOptions[optionName] = false;
                }

                // Floating points in JS are stable up to 7 decimals.
            } else if (optionName === "decimals") {
                if (optionValue >= 0 && optionValue < 8) {
                    filteredOptions[optionName] = optionValue;
                } else {
                    throw new Error(optionName);
                }

                // These options, when provided, must be functions.
            } else if (
                optionName === "encoder" ||
                optionName === "decoder" ||
                optionName === "edit" ||
                optionName === "undo"
            ) {
                if (typeof optionValue === "function") {
                    filteredOptions[optionName] = optionValue;
                } else {
                    throw new Error(optionName);
                }

                // Other options are strings.
            } else {
                if (typeof optionValue === "string") {
                    filteredOptions[optionName] = optionValue;
                } else {
                    throw new Error(optionName);
                }
            }
        }

        // Some values can't be extracted from a
        // string if certain combinations are present.
        throwEqualError(filteredOptions, "mark", "thousand");
        throwEqualError(filteredOptions, "prefix", "negative");
        throwEqualError(filteredOptions, "prefix", "negativeBefore");

        return filteredOptions;
    }

    // Pass all options as function arguments
    function passAll(options, method, input) {
        var i,
            args = [];

        // Add all options in order of FormatOptions
        for (i = 0; i < FormatOptions.length; i += 1) {
            args.push(options[FormatOptions[i]]);
        }

        // Append the input, then call the method, presenting all
        // options as arguments.
        args.push(input);
        return method.apply("", args);
    }

    function wNumb(options) {
        if (!(this instanceof wNumb)) {
            return new wNumb(options);
        }

        if (typeof options !== "object") {
            return;
        }

        options = validate(options);

        // Call 'formatTo' with proper arguments.
        this.to = function(input) {
            return passAll(options, formatTo, input);
        };

        // Call 'formatFrom' with proper arguments.
        this.from = function(input) {
            return passAll(options, formatFrom, input);
        };
    }

    return wNumb;
});

/* axios v0.21.4 | (c) 2021 by Matt Zabriskie */
!function(e,t){"object"==typeof exports&&"object"==typeof module?module.exports=t():"function"==typeof define&&define.amd?define([],t):"object"==typeof exports?exports.axios=t():e.axios=t()}(window,(function(){return function(e){var t={};function r(n){if(t[n])return t[n].exports;var o=t[n]={i:n,l:!1,exports:{}};return e[n].call(o.exports,o,o.exports,r),o.l=!0,o.exports}return r.m=e,r.c=t,r.d=function(e,t,n){r.o(e,t)||Object.defineProperty(e,t,{enumerable:!0,get:n})},r.r=function(e){"undefined"!=typeof Symbol&&Symbol.toStringTag&&Object.defineProperty(e,Symbol.toStringTag,{value:"Module"}),Object.defineProperty(e,"__esModule",{value:!0})},r.t=function(e,t){if(1&t&&(e=r(e)),8&t)return e;if(4&t&&"object"==typeof e&&e&&e.__esModule)return e;var n=Object.create(null);if(r.r(n),Object.defineProperty(n,"default",{enumerable:!0,value:e}),2&t&&"string"!=typeof e)for(var o in e)r.d(n,o,function(t){return e[t]}.bind(null,o));return n},r.n=function(e){var t=e&&e.__esModule?function(){return e.default}:function(){return e};return r.d(t,"a",t),t},r.o=function(e,t){return Object.prototype.hasOwnProperty.call(e,t)},r.p="",r(r.s=10)}([function(e,t,r){"use strict";var n=r(2),o=Object.prototype.toString;function i(e){return"[object Array]"===o.call(e)}function s(e){return void 0===e}function a(e){return null!==e&&"object"==typeof e}function u(e){if("[object Object]"!==o.call(e))return!1;var t=Object.getPrototypeOf(e);return null===t||t===Object.prototype}function c(e){return"[object Function]"===o.call(e)}function f(e,t){if(null!=e)if("object"!=typeof e&&(e=[e]),i(e))for(var r=0,n=e.length;r<n;r++)t.call(null,e[r],r,e);else for(var o in e)Object.prototype.hasOwnProperty.call(e,o)&&t.call(null,e[o],o,e)}e.exports={isArray:i,isArrayBuffer:function(e){return"[object ArrayBuffer]"===o.call(e)},isBuffer:function(e){return null!==e&&!s(e)&&null!==e.constructor&&!s(e.constructor)&&"function"==typeof e.constructor.isBuffer&&e.constructor.isBuffer(e)},isFormData:function(e){return"undefined"!=typeof FormData&&e instanceof FormData},isArrayBufferView:function(e){return"undefined"!=typeof ArrayBuffer&&ArrayBuffer.isView?ArrayBuffer.isView(e):e&&e.buffer&&e.buffer instanceof ArrayBuffer},isString:function(e){return"string"==typeof e},isNumber:function(e){return"number"==typeof e},isObject:a,isPlainObject:u,isUndefined:s,isDate:function(e){return"[object Date]"===o.call(e)},isFile:function(e){return"[object File]"===o.call(e)},isBlob:function(e){return"[object Blob]"===o.call(e)},isFunction:c,isStream:function(e){return a(e)&&c(e.pipe)},isURLSearchParams:function(e){return"undefined"!=typeof URLSearchParams&&e instanceof URLSearchParams},isStandardBrowserEnv:function(){return("undefined"==typeof navigator||"ReactNative"!==navigator.product&&"NativeScript"!==navigator.product&&"NS"!==navigator.product)&&("undefined"!=typeof window&&"undefined"!=typeof document)},forEach:f,merge:function e(){var t={};function r(r,n){u(t[n])&&u(r)?t[n]=e(t[n],r):u(r)?t[n]=e({},r):i(r)?t[n]=r.slice():t[n]=r}for(var n=0,o=arguments.length;n<o;n++)f(arguments[n],r);return t},extend:function(e,t,r){return f(t,(function(t,o){e[o]=r&&"function"==typeof t?n(t,r):t})),e},trim:function(e){return e.trim?e.trim():e.replace(/^\s+|\s+$/g,"")},stripBOM:function(e){return 65279===e.charCodeAt(0)&&(e=e.slice(1)),e}}},function(e,t,r){"use strict";var n=r(0),o=r(16),i=r(4),s={"Content-Type":"application/x-www-form-urlencoded"};function a(e,t){!n.isUndefined(e)&&n.isUndefined(e["Content-Type"])&&(e["Content-Type"]=t)}var u,c={transitional:{silentJSONParsing:!0,forcedJSONParsing:!0,clarifyTimeoutError:!1},adapter:(("undefined"!=typeof XMLHttpRequest||"undefined"!=typeof process&&"[object process]"===Object.prototype.toString.call(process))&&(u=r(5)),u),transformRequest:[function(e,t){return o(t,"Accept"),o(t,"Content-Type"),n.isFormData(e)||n.isArrayBuffer(e)||n.isBuffer(e)||n.isStream(e)||n.isFile(e)||n.isBlob(e)?e:n.isArrayBufferView(e)?e.buffer:n.isURLSearchParams(e)?(a(t,"application/x-www-form-urlencoded;charset=utf-8"),e.toString()):n.isObject(e)||t&&"application/json"===t["Content-Type"]?(a(t,"application/json"),function(e,t,r){if(n.isString(e))try{return(t||JSON.parse)(e),n.trim(e)}catch(e){if("SyntaxError"!==e.name)throw e}return(r||JSON.stringify)(e)}(e)):e}],transformResponse:[function(e){var t=this.transitional,r=t&&t.silentJSONParsing,o=t&&t.forcedJSONParsing,s=!r&&"json"===this.responseType;if(s||o&&n.isString(e)&&e.length)try{return JSON.parse(e)}catch(e){if(s){if("SyntaxError"===e.name)throw i(e,this,"E_JSON_PARSE");throw e}}return e}],timeout:0,xsrfCookieName:"XSRF-TOKEN",xsrfHeaderName:"X-XSRF-TOKEN",maxContentLength:-1,maxBodyLength:-1,validateStatus:function(e){return e>=200&&e<300}};c.headers={common:{Accept:"application/json, text/plain, */*"}},n.forEach(["delete","get","head"],(function(e){c.headers[e]={}})),n.forEach(["post","put","patch"],(function(e){c.headers[e]=n.merge(s)})),e.exports=c},function(e,t,r){"use strict";e.exports=function(e,t){return function(){for(var r=new Array(arguments.length),n=0;n<r.length;n++)r[n]=arguments[n];return e.apply(t,r)}}},function(e,t,r){"use strict";var n=r(0);function o(e){return encodeURIComponent(e).replace(/%3A/gi,":").replace(/%24/g,"$").replace(/%2C/gi,",").replace(/%20/g,"+").replace(/%5B/gi,"[").replace(/%5D/gi,"]")}e.exports=function(e,t,r){if(!t)return e;var i;if(r)i=r(t);else if(n.isURLSearchParams(t))i=t.toString();else{var s=[];n.forEach(t,(function(e,t){null!=e&&(n.isArray(e)?t+="[]":e=[e],n.forEach(e,(function(e){n.isDate(e)?e=e.toISOString():n.isObject(e)&&(e=JSON.stringify(e)),s.push(o(t)+"="+o(e))})))})),i=s.join("&")}if(i){var a=e.indexOf("#");-1!==a&&(e=e.slice(0,a)),e+=(-1===e.indexOf("?")?"?":"&")+i}return e}},function(e,t,r){"use strict";e.exports=function(e,t,r,n,o){return e.config=t,r&&(e.code=r),e.request=n,e.response=o,e.isAxiosError=!0,e.toJSON=function(){return{message:this.message,name:this.name,description:this.description,number:this.number,fileName:this.fileName,lineNumber:this.lineNumber,columnNumber:this.columnNumber,stack:this.stack,config:this.config,code:this.code}},e}},function(e,t,r){"use strict";var n=r(0),o=r(17),i=r(18),s=r(3),a=r(19),u=r(22),c=r(23),f=r(6);e.exports=function(e){return new Promise((function(t,r){var p=e.data,l=e.headers,d=e.responseType;n.isFormData(p)&&delete l["Content-Type"];var h=new XMLHttpRequest;if(e.auth){var m=e.auth.username||"",g=e.auth.password?unescape(encodeURIComponent(e.auth.password)):"";l.Authorization="Basic "+btoa(m+":"+g)}var v=a(e.baseURL,e.url);function y(){if(h){var n="getAllResponseHeaders"in h?u(h.getAllResponseHeaders()):null,i={data:d&&"text"!==d&&"json"!==d?h.response:h.responseText,status:h.status,statusText:h.statusText,headers:n,config:e,request:h};o(t,r,i),h=null}}if(h.open(e.method.toUpperCase(),s(v,e.params,e.paramsSerializer),!0),h.timeout=e.timeout,"onloadend"in h?h.onloadend=y:h.onreadystatechange=function(){h&&4===h.readyState&&(0!==h.status||h.responseURL&&0===h.responseURL.indexOf("file:"))&&setTimeout(y)},h.onabort=function(){h&&(r(f("Request aborted",e,"ECONNABORTED",h)),h=null)},h.onerror=function(){r(f("Network Error",e,null,h)),h=null},h.ontimeout=function(){var t="timeout of "+e.timeout+"ms exceeded";e.timeoutErrorMessage&&(t=e.timeoutErrorMessage),r(f(t,e,e.transitional&&e.transitional.clarifyTimeoutError?"ETIMEDOUT":"ECONNABORTED",h)),h=null},n.isStandardBrowserEnv()){var b=(e.withCredentials||c(v))&&e.xsrfCookieName?i.read(e.xsrfCookieName):void 0;b&&(l[e.xsrfHeaderName]=b)}"setRequestHeader"in h&&n.forEach(l,(function(e,t){void 0===p&&"content-type"===t.toLowerCase()?delete l[t]:h.setRequestHeader(t,e)})),n.isUndefined(e.withCredentials)||(h.withCredentials=!!e.withCredentials),d&&"json"!==d&&(h.responseType=e.responseType),"function"==typeof e.onDownloadProgress&&h.addEventListener("progress",e.onDownloadProgress),"function"==typeof e.onUploadProgress&&h.upload&&h.upload.addEventListener("progress",e.onUploadProgress),e.cancelToken&&e.cancelToken.promise.then((function(e){h&&(h.abort(),r(e),h=null)})),p||(p=null),h.send(p)}))}},function(e,t,r){"use strict";var n=r(4);e.exports=function(e,t,r,o,i){var s=new Error(e);return n(s,t,r,o,i)}},function(e,t,r){"use strict";e.exports=function(e){return!(!e||!e.__CANCEL__)}},function(e,t,r){"use strict";var n=r(0);e.exports=function(e,t){t=t||{};var r={},o=["url","method","data"],i=["headers","auth","proxy","params"],s=["baseURL","transformRequest","transformResponse","paramsSerializer","timeout","timeoutMessage","withCredentials","adapter","responseType","xsrfCookieName","xsrfHeaderName","onUploadProgress","onDownloadProgress","decompress","maxContentLength","maxBodyLength","maxRedirects","transport","httpAgent","httpsAgent","cancelToken","socketPath","responseEncoding"],a=["validateStatus"];function u(e,t){return n.isPlainObject(e)&&n.isPlainObject(t)?n.merge(e,t):n.isPlainObject(t)?n.merge({},t):n.isArray(t)?t.slice():t}function c(o){n.isUndefined(t[o])?n.isUndefined(e[o])||(r[o]=u(void 0,e[o])):r[o]=u(e[o],t[o])}n.forEach(o,(function(e){n.isUndefined(t[e])||(r[e]=u(void 0,t[e]))})),n.forEach(i,c),n.forEach(s,(function(o){n.isUndefined(t[o])?n.isUndefined(e[o])||(r[o]=u(void 0,e[o])):r[o]=u(void 0,t[o])})),n.forEach(a,(function(n){n in t?r[n]=u(e[n],t[n]):n in e&&(r[n]=u(void 0,e[n]))}));var f=o.concat(i).concat(s).concat(a),p=Object.keys(e).concat(Object.keys(t)).filter((function(e){return-1===f.indexOf(e)}));return n.forEach(p,c),r}},function(e,t,r){"use strict";function n(e){this.message=e}n.prototype.toString=function(){return"Cancel"+(this.message?": "+this.message:"")},n.prototype.__CANCEL__=!0,e.exports=n},function(e,t,r){e.exports=r(11)},function(e,t,r){"use strict";var n=r(0),o=r(2),i=r(12),s=r(8);function a(e){var t=new i(e),r=o(i.prototype.request,t);return n.extend(r,i.prototype,t),n.extend(r,t),r}var u=a(r(1));u.Axios=i,u.create=function(e){return a(s(u.defaults,e))},u.Cancel=r(9),u.CancelToken=r(26),u.isCancel=r(7),u.all=function(e){return Promise.all(e)},u.spread=r(27),u.isAxiosError=r(28),e.exports=u,e.exports.default=u},function(e,t,r){"use strict";var n=r(0),o=r(3),i=r(13),s=r(14),a=r(8),u=r(24),c=u.validators;function f(e){this.defaults=e,this.interceptors={request:new i,response:new i}}f.prototype.request=function(e){"string"==typeof e?(e=arguments[1]||{}).url=arguments[0]:e=e||{},(e=a(this.defaults,e)).method?e.method=e.method.toLowerCase():this.defaults.method?e.method=this.defaults.method.toLowerCase():e.method="get";var t=e.transitional;void 0!==t&&u.assertOptions(t,{silentJSONParsing:c.transitional(c.boolean,"1.0.0"),forcedJSONParsing:c.transitional(c.boolean,"1.0.0"),clarifyTimeoutError:c.transitional(c.boolean,"1.0.0")},!1);var r=[],n=!0;this.interceptors.request.forEach((function(t){"function"==typeof t.runWhen&&!1===t.runWhen(e)||(n=n&&t.synchronous,r.unshift(t.fulfilled,t.rejected))}));var o,i=[];if(this.interceptors.response.forEach((function(e){i.push(e.fulfilled,e.rejected)})),!n){var f=[s,void 0];for(Array.prototype.unshift.apply(f,r),f=f.concat(i),o=Promise.resolve(e);f.length;)o=o.then(f.shift(),f.shift());return o}for(var p=e;r.length;){var l=r.shift(),d=r.shift();try{p=l(p)}catch(e){d(e);break}}try{o=s(p)}catch(e){return Promise.reject(e)}for(;i.length;)o=o.then(i.shift(),i.shift());return o},f.prototype.getUri=function(e){return e=a(this.defaults,e),o(e.url,e.params,e.paramsSerializer).replace(/^\?/,"")},n.forEach(["delete","get","head","options"],(function(e){f.prototype[e]=function(t,r){return this.request(a(r||{},{method:e,url:t,data:(r||{}).data}))}})),n.forEach(["post","put","patch"],(function(e){f.prototype[e]=function(t,r,n){return this.request(a(n||{},{method:e,url:t,data:r}))}})),e.exports=f},function(e,t,r){"use strict";var n=r(0);function o(){this.handlers=[]}o.prototype.use=function(e,t,r){return this.handlers.push({fulfilled:e,rejected:t,synchronous:!!r&&r.synchronous,runWhen:r?r.runWhen:null}),this.handlers.length-1},o.prototype.eject=function(e){this.handlers[e]&&(this.handlers[e]=null)},o.prototype.forEach=function(e){n.forEach(this.handlers,(function(t){null!==t&&e(t)}))},e.exports=o},function(e,t,r){"use strict";var n=r(0),o=r(15),i=r(7),s=r(1);function a(e){e.cancelToken&&e.cancelToken.throwIfRequested()}e.exports=function(e){return a(e),e.headers=e.headers||{},e.data=o.call(e,e.data,e.headers,e.transformRequest),e.headers=n.merge(e.headers.common||{},e.headers[e.method]||{},e.headers),n.forEach(["delete","get","head","post","put","patch","common"],(function(t){delete e.headers[t]})),(e.adapter||s.adapter)(e).then((function(t){return a(e),t.data=o.call(e,t.data,t.headers,e.transformResponse),t}),(function(t){return i(t)||(a(e),t&&t.response&&(t.response.data=o.call(e,t.response.data,t.response.headers,e.transformResponse))),Promise.reject(t)}))}},function(e,t,r){"use strict";var n=r(0),o=r(1);e.exports=function(e,t,r){var i=this||o;return n.forEach(r,(function(r){e=r.call(i,e,t)})),e}},function(e,t,r){"use strict";var n=r(0);e.exports=function(e,t){n.forEach(e,(function(r,n){n!==t&&n.toUpperCase()===t.toUpperCase()&&(e[t]=r,delete e[n])}))}},function(e,t,r){"use strict";var n=r(6);e.exports=function(e,t,r){var o=r.config.validateStatus;r.status&&o&&!o(r.status)?t(n("Request failed with status code "+r.status,r.config,null,r.request,r)):e(r)}},function(e,t,r){"use strict";var n=r(0);e.exports=n.isStandardBrowserEnv()?{write:function(e,t,r,o,i,s){var a=[];a.push(e+"="+encodeURIComponent(t)),n.isNumber(r)&&a.push("expires="+new Date(r).toGMTString()),n.isString(o)&&a.push("path="+o),n.isString(i)&&a.push("domain="+i),!0===s&&a.push("secure"),document.cookie=a.join("; ")},read:function(e){var t=document.cookie.match(new RegExp("(^|;\\s*)("+e+")=([^;]*)"));return t?decodeURIComponent(t[3]):null},remove:function(e){this.write(e,"",Date.now()-864e5)}}:{write:function(){},read:function(){return null},remove:function(){}}},function(e,t,r){"use strict";var n=r(20),o=r(21);e.exports=function(e,t){return e&&!n(t)?o(e,t):t}},function(e,t,r){"use strict";e.exports=function(e){return/^([a-z][a-z\d\+\-\.]*:)?\/\//i.test(e)}},function(e,t,r){"use strict";e.exports=function(e,t){return t?e.replace(/\/+$/,"")+"/"+t.replace(/^\/+/,""):e}},function(e,t,r){"use strict";var n=r(0),o=["age","authorization","content-length","content-type","etag","expires","from","host","if-modified-since","if-unmodified-since","last-modified","location","max-forwards","proxy-authorization","referer","retry-after","user-agent"];e.exports=function(e){var t,r,i,s={};return e?(n.forEach(e.split("\n"),(function(e){if(i=e.indexOf(":"),t=n.trim(e.substr(0,i)).toLowerCase(),r=n.trim(e.substr(i+1)),t){if(s[t]&&o.indexOf(t)>=0)return;s[t]="set-cookie"===t?(s[t]?s[t]:[]).concat([r]):s[t]?s[t]+", "+r:r}})),s):s}},function(e,t,r){"use strict";var n=r(0);e.exports=n.isStandardBrowserEnv()?function(){var e,t=/(msie|trident)/i.test(navigator.userAgent),r=document.createElement("a");function o(e){var n=e;return t&&(r.setAttribute("href",n),n=r.href),r.setAttribute("href",n),{href:r.href,protocol:r.protocol?r.protocol.replace(/:$/,""):"",host:r.host,search:r.search?r.search.replace(/^\?/,""):"",hash:r.hash?r.hash.replace(/^#/,""):"",hostname:r.hostname,port:r.port,pathname:"/"===r.pathname.charAt(0)?r.pathname:"/"+r.pathname}}return e=o(window.location.href),function(t){var r=n.isString(t)?o(t):t;return r.protocol===e.protocol&&r.host===e.host}}():function(){return!0}},function(e,t,r){"use strict";var n=r(25),o={};["object","boolean","number","function","string","symbol"].forEach((function(e,t){o[e]=function(r){return typeof r===e||"a"+(t<1?"n ":" ")+e}}));var i={},s=n.version.split(".");function a(e,t){for(var r=t?t.split("."):s,n=e.split("."),o=0;o<3;o++){if(r[o]>n[o])return!0;if(r[o]<n[o])return!1}return!1}o.transitional=function(e,t,r){var o=t&&a(t);function s(e,t){return"[Axios v"+n.version+"] Transitional option '"+e+"'"+t+(r?". "+r:"")}return function(r,n,a){if(!1===e)throw new Error(s(n," has been removed in "+t));return o&&!i[n]&&(i[n]=!0,console.warn(s(n," has been deprecated since v"+t+" and will be removed in the near future"))),!e||e(r,n,a)}},e.exports={isOlderVersion:a,assertOptions:function(e,t,r){if("object"!=typeof e)throw new TypeError("options must be an object");for(var n=Object.keys(e),o=n.length;o-- >0;){var i=n[o],s=t[i];if(s){var a=e[i],u=void 0===a||s(a,i,e);if(!0!==u)throw new TypeError("option "+i+" must be "+u)}else if(!0!==r)throw Error("Unknown option "+i)}},validators:o}},function(e){e.exports=JSON.parse('{"name":"axios","version":"0.21.4","description":"Promise based HTTP client for the browser and node.js","main":"index.js","scripts":{"test":"grunt test","start":"node ./sandbox/server.js","build":"NODE_ENV=production grunt build","preversion":"npm test","version":"npm run build && grunt version && git add -A dist && git add CHANGELOG.md bower.json package.json","postversion":"git push && git push --tags","examples":"node ./examples/server.js","coveralls":"cat coverage/lcov.info | ./node_modules/coveralls/bin/coveralls.js","fix":"eslint --fix lib/**/*.js"},"repository":{"type":"git","url":"https://github.com/axios/axios.git"},"keywords":["xhr","http","ajax","promise","node"],"author":"Matt Zabriskie","license":"MIT","bugs":{"url":"https://github.com/axios/axios/issues"},"homepage":"https://axios-http.com","devDependencies":{"coveralls":"^3.0.0","es6-promise":"^4.2.4","grunt":"^1.3.0","grunt-banner":"^0.6.0","grunt-cli":"^1.2.0","grunt-contrib-clean":"^1.1.0","grunt-contrib-watch":"^1.0.0","grunt-eslint":"^23.0.0","grunt-karma":"^4.0.0","grunt-mocha-test":"^0.13.3","grunt-ts":"^6.0.0-beta.19","grunt-webpack":"^4.0.2","istanbul-instrumenter-loader":"^1.0.0","jasmine-core":"^2.4.1","karma":"^6.3.2","karma-chrome-launcher":"^3.1.0","karma-firefox-launcher":"^2.1.0","karma-jasmine":"^1.1.1","karma-jasmine-ajax":"^0.1.13","karma-safari-launcher":"^1.0.0","karma-sauce-launcher":"^4.3.6","karma-sinon":"^1.0.5","karma-sourcemap-loader":"^0.3.8","karma-webpack":"^4.0.2","load-grunt-tasks":"^3.5.2","minimist":"^1.2.0","mocha":"^8.2.1","sinon":"^4.5.0","terser-webpack-plugin":"^4.2.3","typescript":"^4.0.5","url-search-params":"^0.10.0","webpack":"^4.44.2","webpack-dev-server":"^3.11.0"},"browser":{"./lib/adapters/http.js":"./lib/adapters/xhr.js"},"jsdelivr":"dist/axios.min.js","unpkg":"dist/axios.min.js","typings":"./index.d.ts","dependencies":{"follow-redirects":"^1.14.0"},"bundlesize":[{"path":"./dist/axios.min.js","threshold":"5kB"}]}')},function(e,t,r){"use strict";var n=r(9);function o(e){if("function"!=typeof e)throw new TypeError("executor must be a function.");var t;this.promise=new Promise((function(e){t=e}));var r=this;e((function(e){r.reason||(r.reason=new n(e),t(r.reason))}))}o.prototype.throwIfRequested=function(){if(this.reason)throw this.reason},o.source=function(){var e;return{token:new o((function(t){e=t})),cancel:e}},e.exports=o},function(e,t,r){"use strict";e.exports=function(e){return function(t){return e.apply(null,t)}}},function(e,t,r){"use strict";e.exports=function(e){return"object"==typeof e&&!0===e.isAxiosError}}])}));
//# sourceMappingURL=axios.min.map
/*! lozad.js - v1.16.0 - 2020-09-06
* https://github.com/ApoorvSaxena/lozad.js
* Copyright (c) 2020 Apoorv Saxena; Licensed MIT */
!function(t,e){"object"==typeof exports&&"undefined"!=typeof module?module.exports=e():"function"==typeof define&&define.amd?define(e):t.lozad=e()}(this,function(){"use strict";
    /**
     * Detect IE browser
     * @const {boolean}
     * @private
     */var g="undefined"!=typeof document&&document.documentMode,f={rootMargin:"0px",threshold:0,load:function(t){if("picture"===t.nodeName.toLowerCase()){var e=t.querySelector("img"),r=!1;null===e&&(e=document.createElement("img"),r=!0),g&&t.getAttribute("data-iesrc")&&(e.src=t.getAttribute("data-iesrc")),t.getAttribute("data-alt")&&(e.alt=t.getAttribute("data-alt")),r&&t.append(e)}if("video"===t.nodeName.toLowerCase()&&!t.getAttribute("data-src")&&t.children){for(var a=t.children,o=void 0,i=0;i<=a.length-1;i++)(o=a[i].getAttribute("data-src"))&&(a[i].src=o);t.load()}t.getAttribute("data-poster")&&(t.poster=t.getAttribute("data-poster")),t.getAttribute("data-src")&&(t.src=t.getAttribute("data-src")),t.getAttribute("data-srcset")&&t.setAttribute("srcset",t.getAttribute("data-srcset"));var n=",";if(t.getAttribute("data-background-delimiter")&&(n=t.getAttribute("data-background-delimiter")),t.getAttribute("data-background-image"))t.style.backgroundImage="url('"+t.getAttribute("data-background-image").split(n).join("'),url('")+"')";else if(t.getAttribute("data-background-image-set")){var d=t.getAttribute("data-background-image-set").split(n),u=d[0].substr(0,d[0].indexOf(" "))||d[0];// Substring before ... 1x
            u=-1===u.indexOf("url(")?"url("+u+")":u,1===d.length?t.style.backgroundImage=u:t.setAttribute("style",(t.getAttribute("style")||"")+"background-image: "+u+"; background-image: -webkit-image-set("+d+"); background-image: image-set("+d+")")}t.getAttribute("data-toggle-class")&&t.classList.toggle(t.getAttribute("data-toggle-class"))},loaded:function(){}};function A(t){t.setAttribute("data-loaded",!0)}var m=function(t){return"true"===t.getAttribute("data-loaded")},v=function(t){var e=1<arguments.length&&void 0!==arguments[1]?arguments[1]:document;return t instanceof Element?[t]:t instanceof NodeList?t:e.querySelectorAll(t)};return function(){var r,a,o=0<arguments.length&&void 0!==arguments[0]?arguments[0]:".lozad",t=1<arguments.length&&void 0!==arguments[1]?arguments[1]:{},e=Object.assign({},f,t),i=e.root,n=e.rootMargin,d=e.threshold,u=e.load,g=e.loaded,s=void 0;"undefined"!=typeof window&&window.IntersectionObserver&&(s=new IntersectionObserver((r=u,a=g,function(t,e){t.forEach(function(t){(0<t.intersectionRatio||t.isIntersecting)&&(e.unobserve(t.target),m(t.target)||(r(t.target),A(t.target),a(t.target)))})}),{root:i,rootMargin:n,threshold:d}));for(var c,l=v(o,i),b=0;b<l.length;b++)(c=l[b]).getAttribute("data-placeholder-background")&&(c.style.background=c.getAttribute("data-placeholder-background"));return{observe:function(){for(var t=v(o,i),e=0;e<t.length;e++)m(t[e])||(s?s.observe(t[e]):(u(t[e]),A(t[e]),g(t[e])))},triggerLoad:function(t){m(t)||(u(t),A(t),g(t))},observer:s}}});


/*!
  * Tempus Dominus v6.2.10 (https://getdatepicker.com/)
  * Copyright 2013-2022 Jonathan Peterson
  * Licensed under MIT (https://github.com/Eonasdan/tempus-dominus/blob/master/LICENSE)
  */
!function(t,e){"object"==typeof exports&&"undefined"!=typeof module?e(exports):"function"==typeof define&&define.amd?define(["exports"],e):e((t="undefined"!=typeof globalThis?globalThis:t||self).tempusDominus={})}(this,(function(t){"use strict";var e;t.Unit=void 0,(e=t.Unit||(t.Unit={})).seconds="seconds",e.minutes="minutes",e.hours="hours",e.date="date",e.month="month",e.year="year";const s={month:"2-digit",day:"2-digit",year:"numeric",hour:"2-digit",minute:"2-digit",second:"2-digit"},i=t=>{switch(t){case"date":return{dateStyle:"short"};case"month":return{month:"numeric",year:"numeric"};case"year":return{year:"numeric"}}};class o extends Date{constructor(){super(...arguments),this.locale="default",this.nonLeapLadder=[0,31,59,90,120,151,181,212,243,273,304,334],this.leapLadder=[0,31,60,91,121,152,182,213,244,274,305,335]}setLocale(t){return this.locale=t,this}static convert(t,e="default"){if(!t)throw new Error("A date is required");return new o(t.getFullYear(),t.getMonth(),t.getDate(),t.getHours(),t.getMinutes(),t.getSeconds(),t.getMilliseconds()).setLocale(e)}static fromString(t,e){return new o(t)}get clone(){return new o(this.year,this.month,this.date,this.hours,this.minutes,this.seconds,this.getMilliseconds()).setLocale(this.locale)}startOf(e,s=0){if(void 0===this[e])throw new Error(`Unit '${e}' is not valid`);switch(e){case"seconds":this.setMilliseconds(0);break;case"minutes":this.setSeconds(0,0);break;case"hours":this.setMinutes(0,0,0);break;case"date":this.setHours(0,0,0,0);break;case"weekDay":{if(this.startOf(t.Unit.date),this.weekDay===s)break;let e=this.weekDay;0!==s&&0===this.weekDay&&(e=8-s),this.manipulate(s-e,t.Unit.date);break}case"month":this.startOf(t.Unit.date),this.setDate(1);break;case"year":this.startOf(t.Unit.date),this.setMonth(0,1)}return this}endOf(e,s=0){if(void 0===this[e])throw new Error(`Unit '${e}' is not valid`);switch(e){case"seconds":this.setMilliseconds(999);break;case"minutes":this.setSeconds(59,999);break;case"hours":this.setMinutes(59,59,999);break;case"date":this.setHours(23,59,59,999);break;case"weekDay":{this.endOf(t.Unit.date);const e=6+s;if(this.weekDay===e)break;this.manipulate(e-this.weekDay,t.Unit.date);break}case"month":this.endOf(t.Unit.date),this.manipulate(1,t.Unit.month),this.setDate(0);break;case"year":this.endOf(t.Unit.date),this.setMonth(11,31)}return this}manipulate(t,e){if(void 0===this[e])throw new Error(`Unit '${e}' is not valid`);return this[e]+=t,this}format(t,e=this.locale){return new Intl.DateTimeFormat(e,t).format(this)}isBefore(t,e){if(!e)return this.valueOf()<t.valueOf();if(void 0===this[e])throw new Error(`Unit '${e}' is not valid`);return this.clone.startOf(e).valueOf()<t.clone.startOf(e).valueOf()}isAfter(t,e){if(!e)return this.valueOf()>t.valueOf();if(void 0===this[e])throw new Error(`Unit '${e}' is not valid`);return this.clone.startOf(e).valueOf()>t.clone.startOf(e).valueOf()}isSame(t,e){if(!e)return this.valueOf()===t.valueOf();if(void 0===this[e])throw new Error(`Unit '${e}' is not valid`);return t=o.convert(t),this.clone.startOf(e).valueOf()===t.startOf(e).valueOf()}isBetween(t,e,s,i="()"){if(s&&void 0===this[s])throw new Error(`Unit '${s}' is not valid`);const o="("===i[0],a=")"===i[1];return(o?this.isAfter(t,s):!this.isBefore(t,s))&&(a?this.isBefore(e,s):!this.isAfter(e,s))||(o?this.isBefore(t,s):!this.isAfter(t,s))&&(a?this.isAfter(e,s):!this.isBefore(e,s))}parts(t=this.locale,e={dateStyle:"full",timeStyle:"long"}){const s={};return new Intl.DateTimeFormat(t,e).formatToParts(this).filter((t=>"literal"!==t.type)).forEach((t=>s[t.type]=t.value)),s}get seconds(){return this.getSeconds()}set seconds(t){this.setSeconds(t)}get secondsFormatted(){return this.parts(void 0,s).second}get minutes(){return this.getMinutes()}set minutes(t){this.setMinutes(t)}get minutesFormatted(){return this.parts(void 0,s).minute}get hours(){return this.getHours()}set hours(t){this.setHours(t)}getHoursFormatted(t="h12"){return this.parts(void 0,{...s,hourCycle:t}).hour}meridiem(t=this.locale){return new Intl.DateTimeFormat(t,{hour:"numeric",hour12:!0}).formatToParts(this).find((t=>"dayPeriod"===t.type))?.value}get date(){return this.getDate()}set date(t){this.setDate(t)}get dateFormatted(){return this.parts(void 0,s).day}get weekDay(){return this.getDay()}get month(){return this.getMonth()}set month(t){const e=new Date(this.year,t+1);e.setDate(0);const s=e.getDate();this.date>s&&(this.date=s),this.setMonth(t)}get monthFormatted(){return this.parts(void 0,s).month}get year(){return this.getFullYear()}set year(t){this.setFullYear(t)}get week(){const t=this.computeOrdinal(),e=this.getUTCDay();let s=Math.floor((t-e+10)/7);return s<1?s=this.weeksInWeekYear(this.year-1):s>this.weeksInWeekYear(this.year)&&(s=1),s}weeksInWeekYear(t){const e=(t+Math.floor(t/4)-Math.floor(t/100)+Math.floor(t/400))%7,s=t-1,i=(s+Math.floor(s/4)-Math.floor(s/100)+Math.floor(s/400))%7;return 4===e||3===i?53:52}get isLeapYear(){return this.year%4==0&&(this.year%100!=0||this.year%400==0)}computeOrdinal(){return this.date+(this.isLeapYear?this.leapLadder:this.nonLeapLadder)[this.month]}}class a extends Error{}const n="tempus-dominus";class r{}r.NAME=n,r.dataKey="td",r.events=new class{constructor(){this.key=".td",this.change=`change${this.key}`,this.update=`update${this.key}`,this.error=`error${this.key}`,this.show=`show${this.key}`,this.hide=`hide${this.key}`,this.blur=`blur${this.key}`,this.focus=`focus${this.key}`,this.keyup=`keyup${this.key}`,this.keydown=`keydown${this.key}`}},r.css=new class{constructor(){this.widget=`${n}-widget`,this.calendarHeader="calendar-header",this.switch="picker-switch",this.toolbar="toolbar",this.noHighlight="no-highlight",this.sideBySide="timepicker-sbs",this.previous="previous",this.next="next",this.disabled="disabled",this.old="old",this.new="new",this.active="active",this.dateContainer="date-container",this.decadesContainer=`${this.dateContainer}-decades`,this.decade="decade",this.yearsContainer=`${this.dateContainer}-years`,this.year="year",this.monthsContainer=`${this.dateContainer}-months`,this.month="month",this.daysContainer=`${this.dateContainer}-days`,this.day="day",this.calendarWeeks="cw",this.dayOfTheWeek="dow",this.today="today",this.weekend="weekend",this.timeContainer="time-container",this.separator="separator",this.clockContainer=`${this.timeContainer}-clock`,this.hourContainer=`${this.timeContainer}-hour`,this.minuteContainer=`${this.timeContainer}-minute`,this.secondContainer=`${this.timeContainer}-second`,this.hour="hour",this.minute="minute",this.second="second",this.toggleMeridiem="toggleMeridiem",this.show="show",this.collapsing="td-collapsing",this.collapse="td-collapse",this.inline="inline",this.lightTheme="light",this.darkTheme="dark",this.isDarkPreferredQuery="(prefers-color-scheme: dark)"}},r.errorMessages=new class{constructor(){this.base="TD:",this.failedToSetInvalidDate="Failed to set invalid date",this.failedToParseInput="Failed parse input field"}unexpectedOption(t){const e=new a(`${this.base} Unexpected option: ${t} does not match a known option.`);throw e.code=1,e}unexpectedOptions(t){const e=new a(`${this.base}: ${t.join(", ")}`);throw e.code=1,e}unexpectedOptionValue(t,e,s){const i=new a(`${this.base} Unexpected option value: ${t} does not accept a value of "${e}". Valid values are: ${s.join(", ")}`);throw i.code=2,i}typeMismatch(t,e,s){const i=new a(`${this.base} Mismatch types: ${t} has a type of ${e} instead of the required ${s}`);throw i.code=3,i}numbersOutOfRange(t,e,s){const i=new a(`${this.base} ${t} expected an array of number between ${e} and ${s}.`);throw i.code=4,i}failedToParseDate(t,e,s=!1){const i=new a(`${this.base} Could not correctly parse "${e}" to a date for ${t}.`);if(i.code=5,!s)throw i;console.warn(i)}mustProvideElement(){const t=new a(`${this.base} No element was provided.`);throw t.code=6,t}subscribeMismatch(){const t=new a(`${this.base} The subscribed events does not match the number of callbacks`);throw t.code=7,t}conflictingConfiguration(t){const e=new a(`${this.base} A configuration value conflicts with another rule. ${t}`);throw e.code=8,e}customDateFormatError(t){const e=new a(`${this.base} customDateFormat: ${t}`);throw e.code=9,e}dateString(){console.warn(`${this.base} Using a string for date options is not recommended unless you specify an ISO string or use the customDateFormat plugin.`)}deprecatedWarning(t,e){console.warn(`${this.base} Warning ${t} is deprecated and will be removed in a future version. ${e}`)}throwError(t){const e=new a(`${this.base} ${t}`);throw e.code=9,e}};class d{constructor(){this.cache=new Map}locate(t){const e=this.cache.get(t);if(e)return e;const s=new t;return this.cache.set(t,s),s}}let c;const l=[{name:"calendar",className:r.css.daysContainer,unit:t.Unit.month,step:1},{name:"months",className:r.css.monthsContainer,unit:t.Unit.year,step:1},{name:"years",className:r.css.yearsContainer,unit:t.Unit.year,step:10},{name:"decades",className:r.css.decadesContainer,unit:t.Unit.year,step:100}];class h{constructor(){this._currentCalendarViewMode=0,this._viewDate=new o,this.minimumCalendarViewMode=0,this.currentView="calendar"}get currentCalendarViewMode(){return this._currentCalendarViewMode}set currentCalendarViewMode(t){this._currentCalendarViewMode=t,this.currentView=l[t].name}get viewDate(){return this._viewDate}set viewDate(t){this._viewDate=t,this.options&&(this.options.viewDate=t)}refreshCurrentView(){this.currentView=l[this.currentCalendarViewMode].name}get isTwelveHour(){return["h12","h11"].includes(this.options.localization.hourCycle)}}class p{constructor(){this.optionsStore=c.locate(h)}isValid(e,s){if(s!==t.Unit.month&&this.optionsStore.options.restrictions.disabledDates.length>0&&this._isInDisabledDates(e))return!1;if(s!==t.Unit.month&&this.optionsStore.options.restrictions.enabledDates.length>0&&!this._isInEnabledDates(e))return!1;if(s!==t.Unit.month&&s!==t.Unit.year&&this.optionsStore.options.restrictions.daysOfWeekDisabled?.length>0&&-1!==this.optionsStore.options.restrictions.daysOfWeekDisabled.indexOf(e.weekDay))return!1;if(this.optionsStore.options.restrictions.minDate&&e.isBefore(this.optionsStore.options.restrictions.minDate,s))return!1;if(this.optionsStore.options.restrictions.maxDate&&e.isAfter(this.optionsStore.options.restrictions.maxDate,s))return!1;if(s===t.Unit.hours||s===t.Unit.minutes||s===t.Unit.seconds){if(this.optionsStore.options.restrictions.disabledHours.length>0&&this._isInDisabledHours(e))return!1;if(this.optionsStore.options.restrictions.enabledHours.length>0&&!this._isInEnabledHours(e))return!1;if(this.optionsStore.options.restrictions.disabledTimeIntervals.length>0)for(const t of this.optionsStore.options.restrictions.disabledTimeIntervals)if(e.isBetween(t.from,t.to))return!1}return!0}_isInDisabledDates(e){return!(!this.optionsStore.options.restrictions.disabledDates||0===this.optionsStore.options.restrictions.disabledDates.length)&&this.optionsStore.options.restrictions.disabledDates.find((s=>s.isSame(e,t.Unit.date)))}_isInEnabledDates(e){return!this.optionsStore.options.restrictions.enabledDates||0===this.optionsStore.options.restrictions.enabledDates.length||this.optionsStore.options.restrictions.enabledDates.find((s=>s.isSame(e,t.Unit.date)))}_isInDisabledHours(t){if(!this.optionsStore.options.restrictions.disabledHours||0===this.optionsStore.options.restrictions.disabledHours.length)return!1;const e=t.hours;return this.optionsStore.options.restrictions.disabledHours.find((t=>t===e))}_isInEnabledHours(t){if(!this.optionsStore.options.restrictions.enabledHours||0===this.optionsStore.options.restrictions.enabledHours.length)return!0;const e=t.hours;return this.optionsStore.options.restrictions.enabledHours.find((t=>t===e))}}class u{constructor(){this.subscribers=[]}subscribe(t){return this.subscribers.push(t),this.unsubscribe.bind(this,this.subscribers.length-1)}unsubscribe(t){this.subscribers.splice(t,1)}emit(t){this.subscribers.forEach((e=>{e(t)}))}destroy(){this.subscribers=null,this.subscribers=[]}}class m{constructor(){this.triggerEvent=new u,this.viewUpdate=new u,this.updateDisplay=new u,this.action=new u,this.updateViewDate=new u}destroy(){this.triggerEvent.destroy(),this.viewUpdate.destroy(),this.updateDisplay.destroy(),this.action.destroy(),this.updateViewDate.destroy()}}const y={restrictions:{minDate:void 0,maxDate:void 0,disabledDates:[],enabledDates:[],daysOfWeekDisabled:[],disabledTimeIntervals:[],disabledHours:[],enabledHours:[]},display:{icons:{type:"icons",time:"fa-solid fa-clock",date:"fa-solid fa-calendar",up:"fa-solid fa-arrow-up",down:"fa-solid fa-arrow-down",previous:"fa-solid fa-chevron-left",next:"fa-solid fa-chevron-right",today:"fa-solid fa-calendar-check",clear:"fa-solid fa-trash",close:"fa-solid fa-xmark"},sideBySide:!1,calendarWeeks:!1,viewMode:"calendar",toolbarPlacement:"bottom",keepOpen:!1,buttons:{today:!1,clear:!1,close:!1},components:{calendar:!0,date:!0,month:!0,year:!0,decades:!0,clock:!0,hours:!0,minutes:!0,seconds:!1,useTwentyfourHour:void 0},inline:!1,theme:"auto"},stepping:1,useCurrent:!0,defaultDate:void 0,localization:{today:"Go to today",clear:"Clear selection",close:"Close the picker",selectMonth:"Select Month",previousMonth:"Previous Month",nextMonth:"Next Month",selectYear:"Select Year",previousYear:"Previous Year",nextYear:"Next Year",selectDecade:"Select Decade",previousDecade:"Previous Decade",nextDecade:"Next Decade",previousCentury:"Previous Century",nextCentury:"Next Century",pickHour:"Pick Hour",incrementHour:"Increment Hour",decrementHour:"Decrement Hour",pickMinute:"Pick Minute",incrementMinute:"Increment Minute",decrementMinute:"Decrement Minute",pickSecond:"Pick Second",incrementSecond:"Increment Second",decrementSecond:"Decrement Second",toggleMeridiem:"Toggle Meridiem",selectTime:"Select Time",selectDate:"Select Date",dayViewHeaderFormat:{month:"long",year:"2-digit"},locale:"default",hourCycle:void 0,startOfTheWeek:0,dateFormats:{LTS:"h:mm:ss T",LT:"h:mm T",L:"MM/dd/yyyy",LL:"MMMM d, yyyy",LLL:"MMMM d, yyyy h:mm T",LLLL:"dddd, MMMM d, yyyy h:mm T"},ordinal:t=>t,format:"L LT"},keepInvalid:!1,debug:!1,allowInputToggle:!1,viewDate:new o,multipleDates:!1,multipleDatesSeparator:"; ",promptTimeOnDateChange:!1,promptTimeOnDateChangeTransitionDelay:200,meta:{},container:void 0};function g(t,e){if(t.constructor.name===o.name)return t;if(t.constructor.name===Date.name)return o.convert(t);if("string"==typeof t){const s=o.fromString(t,e);return"null"===JSON.stringify(s)?null:s}return null}function v(t,e,s){"string"==typeof t&&"input"!==e&&r.errorMessages.dateString();const i=g(t,s);return i||r.errorMessages.failedToParseDate(e,t,"input"===e),i}function S(t,e,s,i){Array.isArray(e)||r.errorMessages.typeMismatch(t,s,"array of DateTime or Date");for(let s=0;s<e.length;s++){const o=e[s],a=v(o,t,i);a||r.errorMessages.typeMismatch(t,typeof o,"DateTime or Date"),a.setLocale(i?.locale??"default"),e[s]=a}}function w(t,e,s){Array.isArray(e)&&!e.some((t=>"number"!=typeof t))||r.errorMessages.typeMismatch(t,s,"array of numbers")}function f(t){return({value:e,providedType:s,localization:i})=>{const o=v(e,t,i);if(void 0!==o)return o.setLocale(i.locale),o;r.errorMessages.typeMismatch(t,s,"DateTime or Date")}}function b(t){const e=f(t);return t=>void 0===t.value?t.value:e(t)}function D(t,e,s){return({value:i,providedType:o})=>void 0===i?[]:(w(t,i,o),i.some((t=>t<e||t>s))&&r.errorMessages.numbersOutOfRange(t,e,s),i)}function _(t){return D(t,0,23)}function k(t){return({value:e,providedType:s,localization:i})=>void 0===e?[]:(S(t,e,s,i),e)}function C(t){return({value:e,path:s})=>(t.includes(e)||r.errorMessages.unexpectedOptionValue(s.substring(1),e,t),e)}const E=Object.freeze({defaultDate:f("defaultDate"),viewDate:f("viewDate"),minDate:b("restrictions.minDate"),maxDate:b("restrictions.maxDate"),disabledHours:_("restrictions.disabledHours"),enabledHours:_("restrictions.enabledHours"),disabledDates:k("restrictions.disabledDates"),enabledDates:k("restrictions.enabledDates"),daysOfWeekDisabled:D("restrictions.daysOfWeekDisabled",0,6),disabledTimeIntervals:({key:t,value:e,providedType:s,localization:i})=>{if(void 0===e)return[];Array.isArray(e)||r.errorMessages.typeMismatch(t,s,"array of { from: DateTime|Date, to: DateTime|Date }");const o=e;for(let e=0;e<o.length;e++)Object.keys(o[e]).forEach((s=>{const a=`${t}[${e}].${s}`,n=o[e][s],d=v(n,a,i);d||r.errorMessages.typeMismatch(a,typeof n,"DateTime or Date"),d.setLocale(i.locale),o[e][s]=d}));return o},toolbarPlacement:C(["top","bottom","default"]),type:C(["icons","sprites"]),viewMode:C(["clock","calendar","months","years","decades"]),theme:C(["light","dark","auto"]),meta:({value:t})=>t,dayViewHeaderFormat:({value:t})=>t,container:({value:t,path:e})=>(t&&!(t instanceof HTMLElement||t instanceof Element||t?.appendChild)&&r.errorMessages.typeMismatch(e.substring(1),typeof t,"HTMLElement"),t),useTwentyfourHour:({value:t,path:e,providedType:s,defaultType:i})=>{if(r.errorMessages.deprecatedWarning("useTwentyfourHour",'Please use "options.localization.hourCycle" instead'),void 0===t||"boolean"===s)return t;r.errorMessages.typeMismatch(e,s,i)},hourCycle:C(["h11","h12","h23","h24"])}),M=({value:t,defaultType:e,providedType:s,path:i})=>{switch(e){case"boolean":return"true"===t||!0===t;case"number":return+t;case"string":return t.toString();case"object":return{};case"function":return t;default:r.errorMessages.typeMismatch(i,s,e)}};function T(t){return(E[t.key]||M)(t)}class L{static deepCopy(t){const e={};return Object.keys(t).forEach((s=>{const i=t[s];i instanceof o?e[s]=i.clone:i instanceof Date?e[s]=new Date(i.valueOf()):(e[s]=i,"object"!=typeof i||i instanceof HTMLElement||i instanceof Element||Array.isArray(i)||(e[s]=L.deepCopy(i)))})),e}static objectPath(t,e){return"."===t.charAt(0)&&(t=t.slice(1)),t?t.split(".").reduce(((t,e)=>L.isValue(t)||L.isValue(t[e])?t[e]:void 0),e):e}static spread(t,e,s="",i){const o=L.objectPath(s,y),a=Object.keys(t).filter((t=>!Object.keys(o).includes(t)));if(a.length>0){const t=L.getFlattenDefaultOptions(),e=a.map((e=>{let i=`"${s}.${e}" in not a known option.`;const o=t.find((t=>t.includes(e)));return o&&(i+=` Did you mean "${o}"?`),i}));r.errorMessages.unexpectedOptions(e)}Object.keys(t).filter((t=>"__proto__"!==t&&"constructor"!==t)).forEach((a=>{"."===(s+=`.${a}`).charAt(0)&&(s=s.slice(1));const n=o[a],r=typeof t[a],d=typeof n,c=t[a];if(null==c)return e[a]=c,void(s=s.substring(0,s.lastIndexOf(`.${a}`)));"object"!=typeof n||Array.isArray(t[a])||n instanceof Date||L.ignoreProperties.includes(a)?e[a]=L.processKey(a,c,r,d,s,i):L.spread(t[a],e[a],s,i),s=s.substring(0,s.lastIndexOf(`.${a}`))}))}static processKey(t,e,s,i,o,a){return T({key:t,value:e,providedType:s,defaultType:i,path:o,localization:a})}static _mergeOptions(t,e){const s=L.deepCopy(e),i="default"!==e.localization?.locale?e.localization:t?.localization||y.localization;return L.spread(t,s,"",i),s}static _dataToOptions(t,e){const s=JSON.parse(JSON.stringify(t.dataset));if(s?.tdTargetInput&&delete s.tdTargetInput,s?.tdTargetToggle&&delete s.tdTargetToggle,!s||0===Object.keys(s).length||s.constructor!==DOMStringMap)return e;const i={},o=t=>{const e={};return Object.keys(t).forEach((t=>{e[t.toLowerCase()]=t})),e},a=(t,e,s,i)=>{const n=o(s)[t[e].toLowerCase()],r={};return void 0===n||(s[n].constructor===Object?(e++,r[n]=a(t,e,s[n],i)):r[n]=i),r},n=o(e);return Object.keys(s).filter((t=>t.startsWith(r.dataKey))).map((t=>t.substring(2))).forEach((t=>{let o=n[t.toLowerCase()];if(t.includes("_")){const r=t.split("_");o=n[r[0].toLowerCase()],void 0!==o&&e[o].constructor===Object&&(i[o]=a(r,1,e[o],s[`td${t}`]))}else void 0!==o&&(i[o]=s[`td${t}`])})),this._mergeOptions(i,e)}static _dateTypeCheck(t,e){return g(t,e)}static _typeCheckDateArray(t,e,s,i){return S(t,e,s,i)}static _typeCheckNumberArray(t,e,s){return w(t,e,s)}static dateConversion(t,e,s){return v(t,e,s)}static getFlattenDefaultOptions(){if(this._flattenDefaults)return this._flattenDefaults;const t=(e,s=[])=>Array.isArray(e)?[]:Object(e)===e?Object.entries(e).flatMap((([e,i])=>t(i,[...s,e]))):s.join(".");return this._flattenDefaults=t(y),this._flattenDefaults}static _validateConflicts(t){!t.display.sideBySide||t.display.components.clock&&(t.display.components.hours||t.display.components.minutes||t.display.components.seconds)||r.errorMessages.conflictingConfiguration("Cannot use side by side mode without the clock components"),t.restrictions.minDate&&t.restrictions.maxDate&&(t.restrictions.minDate.isAfter(t.restrictions.maxDate)&&r.errorMessages.conflictingConfiguration("minDate is after maxDate"),t.restrictions.maxDate.isBefore(t.restrictions.minDate)&&r.errorMessages.conflictingConfiguration("maxDate is before minDate"))}}L.ignoreProperties=["meta","dayViewHeaderFormat","container","dateForms","ordinal"],L.isValue=t=>null!=t;class U{constructor(){this._dates=[],this.optionsStore=c.locate(h),this.validation=c.locate(p),this._eventEmitters=c.locate(m)}get picked(){return this._dates}get lastPicked(){return this._dates[this.lastPickedIndex]}get lastPickedIndex(){return 0===this._dates.length?0:this._dates.length-1}formatInput(t){const e=this.optionsStore.options.display.components;return t?t.format({year:e.calendar&&e.year?"numeric":void 0,month:e.calendar&&e.month?"2-digit":void 0,day:e.calendar&&e.date?"2-digit":void 0,hour:e.clock&&e.hours?"2-digit":void 0,minute:e.clock&&e.minutes?"2-digit":void 0,second:e.clock&&e.seconds?"2-digit":void 0,hourCycle:this.optionsStore.options.localization.hourCycle}):""}parseInput(t){return L.dateConversion(t,"input",this.optionsStore.options.localization)}setFromInput(t,e){if(!t)return void this.setValue(void 0,e);const s=this.parseInput(t);s&&(s.setLocale(this.optionsStore.options.localization.locale),this.setValue(s,e))}add(t){this._dates.push(t)}isPicked(t,e){if(!e)return void 0!==this._dates.find((e=>e===t));const s=i(e),o=t.format(s);return void 0!==this._dates.map((t=>t.format(s))).find((t=>t===o))}pickedIndex(t,e){if(!e)return this._dates.indexOf(t);const s=i(e),o=t.format(s);return this._dates.map((t=>t.format(s))).indexOf(o)}clear(){this.optionsStore.unset=!0,this._eventEmitters.triggerEvent.emit({type:r.events.change,date:void 0,oldDate:this.lastPicked,isClear:!0,isValid:!0}),this._dates=[]}static getStartEndYear(t,e){const s=t/10,i=Math.floor(e/t)*t;return[i,i+9*s,Math.floor(e/s)*s]}updateInput(t){if(!this.optionsStore.input)return;let e=this.formatInput(t);this.optionsStore.options.multipleDates&&(e=this._dates.map((t=>this.formatInput(t))).join(this.optionsStore.options.multipleDatesSeparator)),this.optionsStore.input.value!=e&&(this.optionsStore.input.value=e)}setValue(t,e){const s=void 0===e,i=!t&&s;let o=this.optionsStore.unset?null:this._dates[e];if(!o&&!this.optionsStore.unset&&s&&i&&(o=this.lastPicked),t&&o?.isSame(t))this.updateInput(t);else{if(!t)return!this.optionsStore.options.multipleDates||1===this._dates.length||i?(this.optionsStore.unset=!0,this._dates=[]):this._dates.splice(e,1),this.updateInput(),this._eventEmitters.triggerEvent.emit({type:r.events.change,date:void 0,oldDate:o,isClear:i,isValid:!0}),void this._eventEmitters.updateDisplay.emit("all");if(e=e||0,t=t.clone,1!==this.optionsStore.options.stepping&&(t.minutes=Math.round(t.minutes/this.optionsStore.options.stepping)*this.optionsStore.options.stepping,t.seconds=0),this.validation.isValid(t))return this._dates[e]=t,this._eventEmitters.updateViewDate.emit(t.clone),this.updateInput(t),this.optionsStore.unset=!1,this._eventEmitters.updateDisplay.emit("all"),void this._eventEmitters.triggerEvent.emit({type:r.events.change,date:t,oldDate:o,isClear:i,isValid:!0});this.optionsStore.options.keepInvalid&&(this._dates[e]=t,this._eventEmitters.updateViewDate.emit(t.clone),this.updateInput(t),this._eventEmitters.triggerEvent.emit({type:r.events.change,date:t,oldDate:o,isClear:i,isValid:!1})),this._eventEmitters.triggerEvent.emit({type:r.events.error,reason:r.errorMessages.failedToSetInvalidDate,date:t,oldDate:o})}}}var A;!function(t){t.next="next",t.previous="previous",t.changeCalendarView="changeCalendarView",t.selectMonth="selectMonth",t.selectYear="selectYear",t.selectDecade="selectDecade",t.selectDay="selectDay",t.selectHour="selectHour",t.selectMinute="selectMinute",t.selectSecond="selectSecond",t.incrementHours="incrementHours",t.incrementMinutes="incrementMinutes",t.incrementSeconds="incrementSeconds",t.decrementHours="decrementHours",t.decrementMinutes="decrementMinutes",t.decrementSeconds="decrementSeconds",t.toggleMeridiem="toggleMeridiem",t.togglePicker="togglePicker",t.showClock="showClock",t.showHours="showHours",t.showMinutes="showMinutes",t.showSeconds="showSeconds",t.clear="clear",t.close="close",t.today="today"}(A||(A={}));var $=A;class V{constructor(){this.optionsStore=c.locate(h),this.dates=c.locate(U),this.validation=c.locate(p)}getPicker(){const t=document.createElement("div");if(t.classList.add(r.css.daysContainer),t.append(...this._daysOfTheWeek()),this.optionsStore.options.display.calendarWeeks){const e=document.createElement("div");e.classList.add(r.css.calendarWeeks,r.css.noHighlight),t.appendChild(e)}for(let e=0;e<42;e++){if(0!==e&&e%7==0&&this.optionsStore.options.display.calendarWeeks){const e=document.createElement("div");e.classList.add(r.css.calendarWeeks,r.css.noHighlight),t.appendChild(e)}const s=document.createElement("div");s.setAttribute("data-action",$.selectDay),t.appendChild(s)}return t}_update(e,s){const i=e.getElementsByClassName(r.css.daysContainer)[0];if("calendar"===this.optionsStore.currentView){const[e,s,o]=i.parentElement.getElementsByClassName(r.css.calendarHeader)[0].getElementsByTagName("div");s.setAttribute(r.css.daysContainer,this.optionsStore.viewDate.format(this.optionsStore.options.localization.dayViewHeaderFormat)),this.optionsStore.options.display.components.month?s.classList.remove(r.css.disabled):s.classList.add(r.css.disabled),this.validation.isValid(this.optionsStore.viewDate.clone.manipulate(-1,t.Unit.month),t.Unit.month)?e.classList.remove(r.css.disabled):e.classList.add(r.css.disabled),this.validation.isValid(this.optionsStore.viewDate.clone.manipulate(1,t.Unit.month),t.Unit.month)?o.classList.remove(r.css.disabled):o.classList.add(r.css.disabled)}const a=this.optionsStore.viewDate.clone.startOf(t.Unit.month).startOf("weekDay",this.optionsStore.options.localization.startOfTheWeek).manipulate(12,t.Unit.hours);i.querySelectorAll(`[data-action="${$.selectDay}"], .${r.css.calendarWeeks}`).forEach((e=>{if(this.optionsStore.options.display.calendarWeeks&&e.classList.contains(r.css.calendarWeeks)){if("#"===e.innerText)return;return void(e.innerText=`${a.week}`)}const i=[];i.push(r.css.day),a.isBefore(this.optionsStore.viewDate,t.Unit.month)&&i.push(r.css.old),a.isAfter(this.optionsStore.viewDate,t.Unit.month)&&i.push(r.css.new),!this.optionsStore.unset&&this.dates.isPicked(a,t.Unit.date)&&i.push(r.css.active),this.validation.isValid(a,t.Unit.date)||i.push(r.css.disabled),a.isSame(new o,t.Unit.date)&&i.push(r.css.today),0!==a.weekDay&&6!==a.weekDay||i.push(r.css.weekend),s(t.Unit.date,a,i,e),e.classList.remove(...e.classList),e.classList.add(...i),e.setAttribute("data-value",`${a.year}-${a.monthFormatted}-${a.dateFormatted}`),e.setAttribute("data-day",`${a.date}`),e.innerText=a.format({day:"numeric"}),a.manipulate(1,t.Unit.date)}))}_daysOfTheWeek(){const e=this.optionsStore.viewDate.clone.startOf("weekDay",this.optionsStore.options.localization.startOfTheWeek).startOf(t.Unit.date),s=[];if(document.createElement("div"),this.optionsStore.options.display.calendarWeeks){const t=document.createElement("div");t.classList.add(r.css.calendarWeeks,r.css.noHighlight),t.innerText="#",s.push(t)}for(let i=0;i<7;i++){const i=document.createElement("div");i.classList.add(r.css.dayOfTheWeek,r.css.noHighlight),i.innerText=e.format({weekday:"short"}),e.manipulate(1,t.Unit.date),s.push(i)}return s}}class O{constructor(){this.optionsStore=c.locate(h),this.dates=c.locate(U),this.validation=c.locate(p)}getPicker(){const t=document.createElement("div");t.classList.add(r.css.monthsContainer);for(let e=0;e<12;e++){const e=document.createElement("div");e.setAttribute("data-action",$.selectMonth),t.appendChild(e)}return t}_update(e,s){const i=e.getElementsByClassName(r.css.monthsContainer)[0];if("months"===this.optionsStore.currentView){const[e,s,o]=i.parentElement.getElementsByClassName(r.css.calendarHeader)[0].getElementsByTagName("div");s.setAttribute(r.css.monthsContainer,this.optionsStore.viewDate.format({year:"numeric"})),this.optionsStore.options.display.components.year?s.classList.remove(r.css.disabled):s.classList.add(r.css.disabled),this.validation.isValid(this.optionsStore.viewDate.clone.manipulate(-1,t.Unit.year),t.Unit.year)?e.classList.remove(r.css.disabled):e.classList.add(r.css.disabled),this.validation.isValid(this.optionsStore.viewDate.clone.manipulate(1,t.Unit.year),t.Unit.year)?o.classList.remove(r.css.disabled):o.classList.add(r.css.disabled)}const o=this.optionsStore.viewDate.clone.startOf(t.Unit.year);i.querySelectorAll(`[data-action="${$.selectMonth}"]`).forEach(((e,i)=>{const a=[];a.push(r.css.month),!this.optionsStore.unset&&this.dates.isPicked(o,t.Unit.month)&&a.push(r.css.active),this.validation.isValid(o,t.Unit.month)||a.push(r.css.disabled),s(t.Unit.month,o,a,e),e.classList.remove(...e.classList),e.classList.add(...a),e.setAttribute("data-value",`${i}`),e.innerText=`${o.format({month:"short"})}`,o.manipulate(1,t.Unit.month)}))}}class H{constructor(){this.optionsStore=c.locate(h),this.dates=c.locate(U),this.validation=c.locate(p)}getPicker(){const t=document.createElement("div");t.classList.add(r.css.yearsContainer);for(let e=0;e<12;e++){const e=document.createElement("div");e.setAttribute("data-action",$.selectYear),t.appendChild(e)}return t}_update(e,s){this._startYear=this.optionsStore.viewDate.clone.manipulate(-1,t.Unit.year),this._endYear=this.optionsStore.viewDate.clone.manipulate(10,t.Unit.year);const i=e.getElementsByClassName(r.css.yearsContainer)[0];if("years"===this.optionsStore.currentView){const[e,s,o]=i.parentElement.getElementsByClassName(r.css.calendarHeader)[0].getElementsByTagName("div");s.setAttribute(r.css.yearsContainer,`${this._startYear.format({year:"numeric"})}-${this._endYear.format({year:"numeric"})}`),this.optionsStore.options.display.components.decades?s.classList.remove(r.css.disabled):s.classList.add(r.css.disabled),this.validation.isValid(this._startYear,t.Unit.year)?e.classList.remove(r.css.disabled):e.classList.add(r.css.disabled),this.validation.isValid(this._endYear,t.Unit.year)?o.classList.remove(r.css.disabled):o.classList.add(r.css.disabled)}const o=this.optionsStore.viewDate.clone.startOf(t.Unit.year).manipulate(-1,t.Unit.year);i.querySelectorAll(`[data-action="${$.selectYear}"]`).forEach((e=>{const i=[];i.push(r.css.year),!this.optionsStore.unset&&this.dates.isPicked(o,t.Unit.year)&&i.push(r.css.active),this.validation.isValid(o,t.Unit.year)||i.push(r.css.disabled),s(t.Unit.year,o,i,e),e.classList.remove(...e.classList),e.classList.add(...i),e.setAttribute("data-value",`${o.year}`),e.innerText=o.format({year:"numeric"}),o.manipulate(1,t.Unit.year)}))}}class x{constructor(){this.optionsStore=c.locate(h),this.dates=c.locate(U),this.validation=c.locate(p)}getPicker(){const t=document.createElement("div");t.classList.add(r.css.decadesContainer);for(let e=0;e<12;e++){const e=document.createElement("div");e.setAttribute("data-action",$.selectDecade),t.appendChild(e)}return t}_update(e,s){const[i,o]=U.getStartEndYear(100,this.optionsStore.viewDate.year);this._startDecade=this.optionsStore.viewDate.clone.startOf(t.Unit.year),this._startDecade.year=i,this._endDecade=this.optionsStore.viewDate.clone.startOf(t.Unit.year),this._endDecade.year=o;const a=e.getElementsByClassName(r.css.decadesContainer)[0],[n,d,c]=a.parentElement.getElementsByClassName(r.css.calendarHeader)[0].getElementsByTagName("div");"decades"===this.optionsStore.currentView&&(d.setAttribute(r.css.decadesContainer,`${this._startDecade.format({year:"numeric"})}-${this._endDecade.format({year:"numeric"})}`),this.validation.isValid(this._startDecade,t.Unit.year)?n.classList.remove(r.css.disabled):n.classList.add(r.css.disabled),this.validation.isValid(this._endDecade,t.Unit.year)?c.classList.remove(r.css.disabled):c.classList.add(r.css.disabled));const l=this.dates.picked.map((t=>t.year));a.querySelectorAll(`[data-action="${$.selectDecade}"]`).forEach(((e,i)=>{if(0===i)return e.classList.add(r.css.old),this._startDecade.year-10<0?(e.textContent=" ",n.classList.add(r.css.disabled),e.classList.add(r.css.disabled),void e.setAttribute("data-value","")):(e.innerText=this._startDecade.clone.manipulate(-10,t.Unit.year).format({year:"numeric"}),void e.setAttribute("data-value",`${this._startDecade.year}`));const o=[];o.push(r.css.decade);const a=this._startDecade.year,d=this._startDecade.year+9;!this.optionsStore.unset&&l.filter((t=>t>=a&&t<=d)).length>0&&o.push(r.css.active),s("decade",this._startDecade,o,e),e.classList.remove(...e.classList),e.classList.add(...o),e.setAttribute("data-value",`${this._startDecade.year}`),e.innerText=`${this._startDecade.format({year:"numeric"})}`,this._startDecade.manipulate(10,t.Unit.year)}))}}class P{constructor(){this._gridColumns="",this.optionsStore=c.locate(h),this.dates=c.locate(U),this.validation=c.locate(p)}getPicker(t){const e=document.createElement("div");return e.classList.add(r.css.clockContainer),e.append(...this._grid(t)),e}_update(e){const s=e.getElementsByClassName(r.css.clockContainer)[0],i=(this.dates.lastPicked||this.optionsStore.viewDate).clone;if(s.querySelectorAll(".disabled").forEach((t=>t.classList.remove(r.css.disabled))),this.optionsStore.options.display.components.hours&&(this.validation.isValid(this.optionsStore.viewDate.clone.manipulate(1,t.Unit.hours),t.Unit.hours)||s.querySelector(`[data-action=${$.incrementHours}]`).classList.add(r.css.disabled),this.validation.isValid(this.optionsStore.viewDate.clone.manipulate(-1,t.Unit.hours),t.Unit.hours)||s.querySelector(`[data-action=${$.decrementHours}]`).classList.add(r.css.disabled),s.querySelector(`[data-time-component=${t.Unit.hours}]`).innerText=i.getHoursFormatted(this.optionsStore.options.localization.hourCycle)),this.optionsStore.options.display.components.minutes&&(this.validation.isValid(this.optionsStore.viewDate.clone.manipulate(1,t.Unit.minutes),t.Unit.minutes)||s.querySelector(`[data-action=${$.incrementMinutes}]`).classList.add(r.css.disabled),this.validation.isValid(this.optionsStore.viewDate.clone.manipulate(-1,t.Unit.minutes),t.Unit.minutes)||s.querySelector(`[data-action=${$.decrementMinutes}]`).classList.add(r.css.disabled),s.querySelector(`[data-time-component=${t.Unit.minutes}]`).innerText=i.minutesFormatted),this.optionsStore.options.display.components.seconds&&(this.validation.isValid(this.optionsStore.viewDate.clone.manipulate(1,t.Unit.seconds),t.Unit.seconds)||s.querySelector(`[data-action=${$.incrementSeconds}]`).classList.add(r.css.disabled),this.validation.isValid(this.optionsStore.viewDate.clone.manipulate(-1,t.Unit.seconds),t.Unit.seconds)||s.querySelector(`[data-action=${$.decrementSeconds}]`).classList.add(r.css.disabled),s.querySelector(`[data-time-component=${t.Unit.seconds}]`).innerText=i.secondsFormatted),this.optionsStore.isTwelveHour){const e=s.querySelector(`[data-action=${$.toggleMeridiem}]`);e.innerText=i.meridiem(),this.validation.isValid(i.clone.manipulate(i.hours>=12?-12:12,t.Unit.hours))?e.classList.remove(r.css.disabled):e.classList.add(r.css.disabled)}s.style.gridTemplateAreas=`"${this._gridColumns}"`}_grid(e){this._gridColumns="";const s=[],i=[],o=[],a=document.createElement("div"),n=e(this.optionsStore.options.display.icons.up),d=e(this.optionsStore.options.display.icons.down);a.classList.add(r.css.separator,r.css.noHighlight);const c=a.cloneNode(!0);c.innerHTML=":";const l=(t=!1)=>t?c.cloneNode(!0):a.cloneNode(!0);if(this.optionsStore.options.display.components.hours){let e=document.createElement("div");e.setAttribute("title",this.optionsStore.options.localization.incrementHour),e.setAttribute("data-action",$.incrementHours),e.appendChild(n.cloneNode(!0)),s.push(e),e=document.createElement("div"),e.setAttribute("title",this.optionsStore.options.localization.pickHour),e.setAttribute("data-action",$.showHours),e.setAttribute("data-time-component",t.Unit.hours),i.push(e),e=document.createElement("div"),e.setAttribute("title",this.optionsStore.options.localization.decrementHour),e.setAttribute("data-action",$.decrementHours),e.appendChild(d.cloneNode(!0)),o.push(e),this._gridColumns+="a"}if(this.optionsStore.options.display.components.minutes){this._gridColumns+=" a",this.optionsStore.options.display.components.hours&&(s.push(l()),i.push(l(!0)),o.push(l()),this._gridColumns+=" a");let e=document.createElement("div");e.setAttribute("title",this.optionsStore.options.localization.incrementMinute),e.setAttribute("data-action",$.incrementMinutes),e.appendChild(n.cloneNode(!0)),s.push(e),e=document.createElement("div"),e.setAttribute("title",this.optionsStore.options.localization.pickMinute),e.setAttribute("data-action",$.showMinutes),e.setAttribute("data-time-component",t.Unit.minutes),i.push(e),e=document.createElement("div"),e.setAttribute("title",this.optionsStore.options.localization.decrementMinute),e.setAttribute("data-action",$.decrementMinutes),e.appendChild(d.cloneNode(!0)),o.push(e)}if(this.optionsStore.options.display.components.seconds){this._gridColumns+=" a",this.optionsStore.options.display.components.minutes&&(s.push(l()),i.push(l(!0)),o.push(l()),this._gridColumns+=" a");let e=document.createElement("div");e.setAttribute("title",this.optionsStore.options.localization.incrementSecond),e.setAttribute("data-action",$.incrementSeconds),e.appendChild(n.cloneNode(!0)),s.push(e),e=document.createElement("div"),e.setAttribute("title",this.optionsStore.options.localization.pickSecond),e.setAttribute("data-action",$.showSeconds),e.setAttribute("data-time-component",t.Unit.seconds),i.push(e),e=document.createElement("div"),e.setAttribute("title",this.optionsStore.options.localization.decrementSecond),e.setAttribute("data-action",$.decrementSeconds),e.appendChild(d.cloneNode(!0)),o.push(e)}if(this.optionsStore.isTwelveHour){this._gridColumns+=" a";let t=l();s.push(t);const e=document.createElement("button");e.setAttribute("title",this.optionsStore.options.localization.toggleMeridiem),e.setAttribute("data-action",$.toggleMeridiem),e.setAttribute("tabindex","-1"),r.css.toggleMeridiem.includes(",")?e.classList.add(...r.css.toggleMeridiem.split(",")):e.classList.add(r.css.toggleMeridiem),t=document.createElement("div"),t.classList.add(r.css.noHighlight),t.appendChild(e),i.push(t),t=l(),o.push(t)}return this._gridColumns=this._gridColumns.trim(),[...s,...i,...o]}}class I{constructor(){this.optionsStore=c.locate(h),this.validation=c.locate(p)}getPicker(){const t=document.createElement("div");t.classList.add(r.css.hourContainer);for(let e=0;e<(this.optionsStore.isTwelveHour?12:24);e++){const e=document.createElement("div");e.setAttribute("data-action",$.selectHour),t.appendChild(e)}return t}_update(e,s){const i=e.getElementsByClassName(r.css.hourContainer)[0],o=this.optionsStore.viewDate.clone.startOf(t.Unit.date);i.querySelectorAll(`[data-action="${$.selectHour}"]`).forEach((e=>{const i=[];i.push(r.css.hour),this.validation.isValid(o,t.Unit.hours)||i.push(r.css.disabled),s(t.Unit.hours,o,i,e),e.classList.remove(...e.classList),e.classList.add(...i),e.setAttribute("data-value",`${o.hours}`),e.innerText=o.getHoursFormatted(this.optionsStore.options.localization.hourCycle),o.manipulate(1,t.Unit.hours)}))}}class z{constructor(){this.optionsStore=c.locate(h),this.validation=c.locate(p)}getPicker(){const t=document.createElement("div");t.classList.add(r.css.minuteContainer);const e=1===this.optionsStore.options.stepping?5:this.optionsStore.options.stepping;for(let s=0;s<60/e;s++){const e=document.createElement("div");e.setAttribute("data-action",$.selectMinute),t.appendChild(e)}return t}_update(e,s){const i=e.getElementsByClassName(r.css.minuteContainer)[0],o=this.optionsStore.viewDate.clone.startOf(t.Unit.hours),a=1===this.optionsStore.options.stepping?5:this.optionsStore.options.stepping;i.querySelectorAll(`[data-action="${$.selectMinute}"]`).forEach((e=>{const i=[];i.push(r.css.minute),this.validation.isValid(o,t.Unit.minutes)||i.push(r.css.disabled),s(t.Unit.minutes,o,i,e),e.classList.remove(...e.classList),e.classList.add(...i),e.setAttribute("data-value",`${o.minutes}`),e.innerText=o.minutesFormatted,o.manipulate(a,t.Unit.minutes)}))}}class N{constructor(){this.optionsStore=c.locate(h),this.validation=c.locate(p)}getPicker(){const t=document.createElement("div");t.classList.add(r.css.secondContainer);for(let e=0;e<12;e++){const e=document.createElement("div");e.setAttribute("data-action",$.selectSecond),t.appendChild(e)}return t}_update(e,s){const i=e.getElementsByClassName(r.css.secondContainer)[0],o=this.optionsStore.viewDate.clone.startOf(t.Unit.minutes);i.querySelectorAll(`[data-action="${$.selectSecond}"]`).forEach((e=>{const i=[];i.push(r.css.second),this.validation.isValid(o,t.Unit.seconds)||i.push(r.css.disabled),s(t.Unit.seconds,o,i,e),e.classList.remove(...e.classList),e.classList.add(...i),e.setAttribute("data-value",`${o.seconds}`),e.innerText=o.secondsFormatted,o.manipulate(5,t.Unit.seconds)}))}}class B{static toggle(t){t.classList.contains(r.css.show)?this.hide(t):this.show(t)}static showImmediately(t){t.classList.remove(r.css.collapsing),t.classList.add(r.css.collapse,r.css.show),t.style.height=""}static show(t){if(t.classList.contains(r.css.collapsing)||t.classList.contains(r.css.show))return;t.style.height="0",t.classList.remove(r.css.collapse),t.classList.add(r.css.collapsing),setTimeout((()=>{B.showImmediately(t)}),this.getTransitionDurationFromElement(t)),t.style.height=`${t.scrollHeight}px`}static hideImmediately(t){t&&(t.classList.remove(r.css.collapsing,r.css.show),t.classList.add(r.css.collapse))}static hide(t){if(t.classList.contains(r.css.collapsing)||!t.classList.contains(r.css.show))return;t.style.height=`${t.getBoundingClientRect().height}px`;t.offsetHeight,t.classList.remove(r.css.collapse,r.css.show),t.classList.add(r.css.collapsing),t.style.height="",setTimeout((()=>{B.hideImmediately(t)}),this.getTransitionDurationFromElement(t))}}B.getTransitionDurationFromElement=t=>{if(!t)return 0;let{transitionDuration:e,transitionDelay:s}=window.getComputedStyle(t);const i=Number.parseFloat(e),o=Number.parseFloat(s);return i||o?(e=e.split(",")[0],s=s.split(",")[0],1e3*(Number.parseFloat(e)+Number.parseFloat(s))):0};class F{constructor(){this._isVisible=!1,this._documentClickEvent=t=>{this.optionsStore.options.debug||window.debug||!this._isVisible||t.composedPath().includes(this.widget)||t.composedPath()?.includes(this.optionsStore.element)||this.hide()},this._actionsClickEvent=t=>{this._eventEmitters.action.emit({e:t})},this.optionsStore=c.locate(h),this.validation=c.locate(p),this.dates=c.locate(U),this.dateDisplay=c.locate(V),this.monthDisplay=c.locate(O),this.yearDisplay=c.locate(H),this.decadeDisplay=c.locate(x),this.timeDisplay=c.locate(P),this.hourDisplay=c.locate(I),this.minuteDisplay=c.locate(z),this.secondDisplay=c.locate(N),this._eventEmitters=c.locate(m),this._widget=void 0,this._eventEmitters.updateDisplay.subscribe((t=>{this._update(t)}))}get widget(){return this._widget}get isVisible(){return this._isVisible}_update(e){if(this.widget)switch(e){case t.Unit.seconds:this.secondDisplay._update(this.widget,this.paint);break;case t.Unit.minutes:this.minuteDisplay._update(this.widget,this.paint);break;case t.Unit.hours:this.hourDisplay._update(this.widget,this.paint);break;case t.Unit.date:this.dateDisplay._update(this.widget,this.paint);break;case t.Unit.month:this.monthDisplay._update(this.widget,this.paint);break;case t.Unit.year:this.yearDisplay._update(this.widget,this.paint);break;case"clock":if(!this._hasTime)break;this.timeDisplay._update(this.widget),this._update(t.Unit.hours),this._update(t.Unit.minutes),this._update(t.Unit.seconds);break;case"calendar":this._update(t.Unit.date),this._update(t.Unit.year),this._update(t.Unit.month),this.decadeDisplay._update(this.widget,this.paint),this._updateCalendarHeader();break;case"all":this._hasTime&&this._update("clock"),this._hasDate&&this._update("calendar")}}paint(t,e,s,i){}show(){if(null==this.widget){if(0==this.dates.picked.length){if(this.optionsStore.options.useCurrent&&!this.optionsStore.options.defaultDate){const e=(new o).setLocale(this.optionsStore.options.localization.locale);if(!this.optionsStore.options.keepInvalid){let s=0,i=1;for(this.optionsStore.options.restrictions.maxDate?.isBefore(e)&&(i=-1);!(this.validation.isValid(e)||(e.manipulate(i,t.Unit.date),s>31));)s++}this.dates.setValue(e)}this.optionsStore.options.defaultDate&&this.dates.setValue(this.optionsStore.options.defaultDate)}this._buildWidget(),this._updateTheme();const e=this._hasTime&&!this._hasDate;if(e?(this.optionsStore.currentView="clock",this._eventEmitters.action.emit({e:null,action:$.showClock})):this.optionsStore.currentCalendarViewMode||(this.optionsStore.currentCalendarViewMode=this.optionsStore.minimumCalendarViewMode),e||"clock"===this.optionsStore.options.display.viewMode||(this._hasTime&&(this.optionsStore.options.display.sideBySide?B.show(this.widget.querySelector(`div.${r.css.timeContainer}`)):B.hideImmediately(this.widget.querySelector(`div.${r.css.timeContainer}`))),B.show(this.widget.querySelector(`div.${r.css.dateContainer}`))),this._hasDate&&this._showMode(),this.optionsStore.options.display.inline)this.optionsStore.element.appendChild(this.widget);else{(this.optionsStore.options?.container||document.body).appendChild(this.widget),this.createPopup(this.optionsStore.element,this.widget,{modifiers:[{name:"eventListeners",enabled:!0}],placement:"rtl"===document.documentElement.dir?"bottom-end":"bottom-start"}).then()}"clock"==this.optionsStore.options.display.viewMode&&this._eventEmitters.action.emit({e:null,action:$.showClock}),this.widget.querySelectorAll("[data-action]").forEach((t=>t.addEventListener("click",this._actionsClickEvent))),this._hasTime&&this.optionsStore.options.display.sideBySide&&(this.timeDisplay._update(this.widget),this.widget.getElementsByClassName(r.css.clockContainer)[0].style.display="grid")}this.widget.classList.add(r.css.show),this.optionsStore.options.display.inline||(this.updatePopup(),document.addEventListener("click",this._documentClickEvent)),this._eventEmitters.triggerEvent.emit({type:r.events.show}),this._isVisible=!0}async createPopup(t,e,s){let i;if(window?.Popper)i=window?.Popper?.createPopper;else{const{createPopper:t}=await import("@popperjs/core");i=t}i&&(this._popperInstance=i(t,e,s))}updatePopup(){this._popperInstance?.update()}_showMode(t){if(!this.widget)return;if(t){const e=Math.max(this.optionsStore.minimumCalendarViewMode,Math.min(3,this.optionsStore.currentCalendarViewMode+t));if(this.optionsStore.currentCalendarViewMode==e)return;this.optionsStore.currentCalendarViewMode=e}this.widget.querySelectorAll(`.${r.css.dateContainer} > div:not(.${r.css.calendarHeader}), .${r.css.timeContainer} > div:not(.${r.css.clockContainer})`).forEach((t=>t.style.display="none"));const e=l[this.optionsStore.currentCalendarViewMode],s=this.widget.querySelector(`.${e.className}`);switch(e.className){case r.css.decadesContainer:this.decadeDisplay._update(this.widget,this.paint);break;case r.css.yearsContainer:this.yearDisplay._update(this.widget,this.paint);break;case r.css.monthsContainer:this.monthDisplay._update(this.widget,this.paint);break;case r.css.daysContainer:this.dateDisplay._update(this.widget,this.paint)}s.style.display="grid",this.optionsStore.options.display.sideBySide&&(this.widget.querySelectorAll(`.${r.css.clockContainer}`)[0].style.display="grid"),this._updateCalendarHeader(),this._eventEmitters.viewUpdate.emit()}_updateTheme(t){if(this.widget){if(t){if(this.optionsStore.options.display.theme===t)return;this.optionsStore.options.display.theme=t}this.widget.classList.remove("light","dark"),this.widget.classList.add(this._getThemeClass()),"auto"===this.optionsStore.options.display.theme?window.matchMedia(r.css.isDarkPreferredQuery).addEventListener("change",(()=>this._updateTheme())):window.matchMedia(r.css.isDarkPreferredQuery).removeEventListener("change",(()=>this._updateTheme()))}}_getThemeClass(){const t=this.optionsStore.options.display.theme||"auto",e=window.matchMedia&&window.matchMedia(r.css.isDarkPreferredQuery).matches;switch(t){case"light":return r.css.lightTheme;case"dark":return r.css.darkTheme;case"auto":return e?r.css.darkTheme:r.css.lightTheme}}_updateCalendarHeader(){if(!this._hasDate)return;const t=[...this.widget.querySelector(`.${r.css.dateContainer} div[style*="display: grid"]`).classList].find((t=>t.startsWith(r.css.dateContainer))),[e,s,i]=this.widget.getElementsByClassName(r.css.calendarHeader)[0].getElementsByTagName("div");switch(t){case r.css.decadesContainer:e.setAttribute("title",this.optionsStore.options.localization.previousCentury),s.setAttribute("title",""),i.setAttribute("title",this.optionsStore.options.localization.nextCentury);break;case r.css.yearsContainer:e.setAttribute("title",this.optionsStore.options.localization.previousDecade),s.setAttribute("title",this.optionsStore.options.localization.selectDecade),i.setAttribute("title",this.optionsStore.options.localization.nextDecade);break;case r.css.monthsContainer:e.setAttribute("title",this.optionsStore.options.localization.previousYear),s.setAttribute("title",this.optionsStore.options.localization.selectYear),i.setAttribute("title",this.optionsStore.options.localization.nextYear);break;case r.css.daysContainer:e.setAttribute("title",this.optionsStore.options.localization.previousMonth),s.setAttribute("title",this.optionsStore.options.localization.selectMonth),i.setAttribute("title",this.optionsStore.options.localization.nextMonth),s.setAttribute(t,this.optionsStore.viewDate.format(this.optionsStore.options.localization.dayViewHeaderFormat))}s.innerText=s.getAttribute(t)}hide(){this.widget&&this._isVisible&&(this.widget.classList.remove(r.css.show),this._isVisible&&(this._eventEmitters.triggerEvent.emit({type:r.events.hide,date:this.optionsStore.unset?null:this.dates.lastPicked?this.dates.lastPicked.clone:void 0}),this._isVisible=!1),document.removeEventListener("click",this._documentClickEvent))}toggle(){return this._isVisible?this.hide():this.show()}_dispose(){document.removeEventListener("click",this._documentClickEvent),this.widget&&(this.widget.querySelectorAll("[data-action]").forEach((t=>t.removeEventListener("click",this._actionsClickEvent))),this.widget.parentNode.removeChild(this.widget),this._widget=void 0)}_buildWidget(){const t=document.createElement("div");t.classList.add(r.css.widget);const e=document.createElement("div");e.classList.add(r.css.dateContainer),e.append(this.getHeadTemplate(),this.decadeDisplay.getPicker(),this.yearDisplay.getPicker(),this.monthDisplay.getPicker(),this.dateDisplay.getPicker());const s=document.createElement("div");s.classList.add(r.css.timeContainer),s.appendChild(this.timeDisplay.getPicker(this._iconTag.bind(this))),s.appendChild(this.hourDisplay.getPicker()),s.appendChild(this.minuteDisplay.getPicker()),s.appendChild(this.secondDisplay.getPicker());const i=document.createElement("div");if(i.classList.add(r.css.toolbar),i.append(...this.getToolbarElements()),this.optionsStore.options.display.inline&&t.classList.add(r.css.inline),this.optionsStore.options.display.calendarWeeks&&t.classList.add("calendarWeeks"),this.optionsStore.options.display.sideBySide&&this._hasDate&&this._hasTime){t.classList.add(r.css.sideBySide),"top"===this.optionsStore.options.display.toolbarPlacement&&t.appendChild(i);const o=document.createElement("div");return o.classList.add("td-row"),e.classList.add("td-half"),s.classList.add("td-half"),o.appendChild(e),o.appendChild(s),t.appendChild(o),"bottom"===this.optionsStore.options.display.toolbarPlacement&&t.appendChild(i),void(this._widget=t)}"top"===this.optionsStore.options.display.toolbarPlacement&&t.appendChild(i),this._hasDate&&(this._hasTime&&(e.classList.add(r.css.collapse),"clock"!==this.optionsStore.options.display.viewMode&&e.classList.add(r.css.show)),t.appendChild(e)),this._hasTime&&(this._hasDate&&(s.classList.add(r.css.collapse),"clock"===this.optionsStore.options.display.viewMode&&s.classList.add(r.css.show)),t.appendChild(s)),"bottom"===this.optionsStore.options.display.toolbarPlacement&&t.appendChild(i);const o=document.createElement("div");o.classList.add("arrow"),o.setAttribute("data-popper-arrow",""),t.appendChild(o),this._widget=t}get _hasTime(){return this.optionsStore.options.display.components.clock&&(this.optionsStore.options.display.components.hours||this.optionsStore.options.display.components.minutes||this.optionsStore.options.display.components.seconds)}get _hasDate(){return this.optionsStore.options.display.components.calendar&&(this.optionsStore.options.display.components.year||this.optionsStore.options.display.components.month||this.optionsStore.options.display.components.date)}getToolbarElements(){const t=[];if(this.optionsStore.options.display.buttons.today){const e=document.createElement("div");e.setAttribute("data-action",$.today),e.setAttribute("title",this.optionsStore.options.localization.today),e.appendChild(this._iconTag(this.optionsStore.options.display.icons.today)),t.push(e)}if(!this.optionsStore.options.display.sideBySide&&this._hasDate&&this._hasTime){let e,s;"clock"===this.optionsStore.options.display.viewMode?(e=this.optionsStore.options.localization.selectDate,s=this.optionsStore.options.display.icons.date):(e=this.optionsStore.options.localization.selectTime,s=this.optionsStore.options.display.icons.time);const i=document.createElement("div");i.setAttribute("data-action",$.togglePicker),i.setAttribute("title",e),i.appendChild(this._iconTag(s)),t.push(i)}if(this.optionsStore.options.display.buttons.clear){const e=document.createElement("div");e.setAttribute("data-action",$.clear),e.setAttribute("title",this.optionsStore.options.localization.clear),e.appendChild(this._iconTag(this.optionsStore.options.display.icons.clear)),t.push(e)}if(this.optionsStore.options.display.buttons.close){const e=document.createElement("div");e.setAttribute("data-action",$.close),e.setAttribute("title",this.optionsStore.options.localization.close),e.appendChild(this._iconTag(this.optionsStore.options.display.icons.close)),t.push(e)}return t}getHeadTemplate(){const t=document.createElement("div");t.classList.add(r.css.calendarHeader);const e=document.createElement("div");e.classList.add(r.css.previous),e.setAttribute("data-action",$.previous),e.appendChild(this._iconTag(this.optionsStore.options.display.icons.previous));const s=document.createElement("div");s.classList.add(r.css.switch),s.setAttribute("data-action",$.changeCalendarView);const i=document.createElement("div");return i.classList.add(r.css.next),i.setAttribute("data-action",$.next),i.appendChild(this._iconTag(this.optionsStore.options.display.icons.next)),t.append(e,s,i),t}_iconTag(t){if("sprites"===this.optionsStore.options.display.icons.type){const e=document.createElementNS("http://www.w3.org/2000/svg","svg"),s=document.createElementNS("http://www.w3.org/2000/svg","use");return s.setAttribute("xlink:href",t),s.setAttribute("href",t),e.appendChild(s),e}const e=document.createElement("i");return e.classList.add(...t.split(" ")),e}_rebuild(){const t=this._isVisible;this._dispose(),t&&this.show()}}class q{constructor(){this.optionsStore=c.locate(h),this.dates=c.locate(U),this.validation=c.locate(p),this.display=c.locate(F),this._eventEmitters=c.locate(m),this._eventEmitters.action.subscribe((t=>{this.do(t.e,t.action)}))}do(e,s){const i=e?.currentTarget;if(i?.classList?.contains(r.css.disabled))return!1;s=s||i?.dataset?.action;const a=(this.dates.lastPicked||this.optionsStore.viewDate).clone;switch(s){case $.next:case $.previous:this.handleNextPrevious(s);break;case $.changeCalendarView:this.display._showMode(1),this.display._updateCalendarHeader();break;case $.selectMonth:case $.selectYear:case $.selectDecade:this.handleSelectCalendarMode(s,i);break;case $.selectDay:this.handleSelectDay(i);break;case $.selectHour:{let t=+i.dataset.value;a.hours>=12&&this.optionsStore.isTwelveHour&&(t+=12),a.hours=t,this.dates.setValue(a,this.dates.lastPickedIndex),this.hideOrClock(e);break}case $.selectMinute:a.minutes=+i.dataset.value,this.dates.setValue(a,this.dates.lastPickedIndex),this.hideOrClock(e);break;case $.selectSecond:a.seconds=+i.dataset.value,this.dates.setValue(a,this.dates.lastPickedIndex),this.hideOrClock(e);break;case $.incrementHours:this.manipulateAndSet(a,t.Unit.hours);break;case $.incrementMinutes:this.manipulateAndSet(a,t.Unit.minutes,this.optionsStore.options.stepping);break;case $.incrementSeconds:this.manipulateAndSet(a,t.Unit.seconds);break;case $.decrementHours:this.manipulateAndSet(a,t.Unit.hours,-1);break;case $.decrementMinutes:this.manipulateAndSet(a,t.Unit.minutes,-1*this.optionsStore.options.stepping);break;case $.decrementSeconds:this.manipulateAndSet(a,t.Unit.seconds,-1);break;case $.toggleMeridiem:this.manipulateAndSet(a,t.Unit.hours,this.dates.lastPicked.hours>=12?-12:12);break;case $.togglePicker:this.handleToggle(i);break;case $.showClock:case $.showHours:case $.showMinutes:case $.showSeconds:this.optionsStore.options.display.sideBySide||"clock"===this.optionsStore.currentView||(B.hideImmediately(this.display.widget.querySelector(`div.${r.css.dateContainer}`)),B.showImmediately(this.display.widget.querySelector(`div.${r.css.timeContainer}`))),this.handleShowClockContainers(s);break;case $.clear:this.dates.setValue(null),this.display._updateCalendarHeader();break;case $.close:this.display.hide();break;case $.today:{const e=(new o).setLocale(this.optionsStore.options.localization.locale);this._eventEmitters.updateViewDate.emit(e),this.validation.isValid(e,t.Unit.date)&&this.dates.setValue(e,this.dates.lastPickedIndex);break}}}handleShowClockContainers(e){if(!this.display._hasTime)return void r.errorMessages.throwError("Cannot show clock containers when time is disabled.");this.optionsStore.currentView="clock",this.display.widget.querySelectorAll(`.${r.css.timeContainer} > div`).forEach((t=>t.style.display="none"));let s="";switch(e){case $.showClock:s=r.css.clockContainer,this.display._update("clock");break;case $.showHours:s=r.css.hourContainer,this.display._update(t.Unit.hours);break;case $.showMinutes:s=r.css.minuteContainer,this.display._update(t.Unit.minutes);break;case $.showSeconds:s=r.css.secondContainer,this.display._update(t.Unit.seconds)}this.display.widget.getElementsByClassName(s)[0].style.display="grid"}handleNextPrevious(t){const{unit:e,step:s}=l[this.optionsStore.currentCalendarViewMode];t===$.next?this.optionsStore.viewDate.manipulate(s,e):this.optionsStore.viewDate.manipulate(-1*s,e),this._eventEmitters.viewUpdate.emit(),this.display._showMode()}hideOrClock(t){this.optionsStore.isTwelveHour||this.optionsStore.options.display.components.minutes||this.optionsStore.options.display.keepOpen||this.optionsStore.options.display.inline?this.do(t,$.showClock):this.display.hide()}manipulateAndSet(t,e,s=1){const i=t.manipulate(s,e);this.validation.isValid(i,e)&&this.dates.setValue(i,this.dates.lastPickedIndex)}handleSelectCalendarMode(t,e){const s=+e.dataset.value;switch(t){case $.selectMonth:this.optionsStore.viewDate.month=s;break;case $.selectYear:case $.selectDecade:this.optionsStore.viewDate.year=s}this.optionsStore.currentCalendarViewMode===this.optionsStore.minimumCalendarViewMode?(this.dates.setValue(this.optionsStore.viewDate,this.dates.lastPickedIndex),this.optionsStore.options.display.inline||this.display.hide()):this.display._showMode(-1)}handleToggle(t){t.getAttribute("title")===this.optionsStore.options.localization.selectDate?(t.setAttribute("title",this.optionsStore.options.localization.selectTime),t.innerHTML=this.display._iconTag(this.optionsStore.options.display.icons.time).outerHTML,this.display._updateCalendarHeader(),this.optionsStore.refreshCurrentView()):(t.setAttribute("title",this.optionsStore.options.localization.selectDate),t.innerHTML=this.display._iconTag(this.optionsStore.options.display.icons.date).outerHTML,this.display._hasTime&&(this.handleShowClockContainers($.showClock),this.display._update("clock"))),this.display.widget.querySelectorAll(`.${r.css.dateContainer}, .${r.css.timeContainer}`).forEach((t=>B.toggle(t))),this._eventEmitters.viewUpdate.emit()}handleSelectDay(e){const s=this.optionsStore.viewDate.clone;e.classList.contains(r.css.old)&&s.manipulate(-1,t.Unit.month),e.classList.contains(r.css.new)&&s.manipulate(1,t.Unit.month),s.date=+e.dataset.day;let i=0;this.optionsStore.options.multipleDates?(i=this.dates.pickedIndex(s,t.Unit.date),-1!==i?this.dates.setValue(null,i):this.dates.setValue(s,this.dates.lastPickedIndex+1)):this.dates.setValue(s,this.dates.lastPickedIndex),this.display._hasTime||this.optionsStore.options.display.keepOpen||this.optionsStore.options.display.inline||this.optionsStore.options.multipleDates||this.display.hide()}}class W{constructor(t,e={}){this._subscribers={},this._isDisabled=!1,this._inputChangeEvent=t=>{const e=t?.detail;if(e)return;const s=()=>{this.dates.lastPicked&&(this.optionsStore.viewDate=this.dates.lastPicked.clone)},i=this.optionsStore.input.value;if(this.optionsStore.options.multipleDates)try{const t=i.split(this.optionsStore.options.multipleDatesSeparator);for(let e=0;e<t.length;e++)this.dates.setFromInput(t[e],e);s()}catch{console.warn("TD: Something went wrong trying to set the multipleDates values from the input field.")}else this.dates.setFromInput(i,0),s()},this._toggleClickEvent=()=>{this.optionsStore.element?.disabled||this.optionsStore.input?.disabled||this.toggle()},c=new d,this._eventEmitters=c.locate(m),this.optionsStore=c.locate(h),this.display=c.locate(F),this.dates=c.locate(U),this.actions=c.locate(q),t||r.errorMessages.mustProvideElement(),this.optionsStore.element=t,this._initializeOptions(e,y,!0),this.optionsStore.viewDate.setLocale(this.optionsStore.options.localization.locale),this.optionsStore.unset=!0,this._initializeInput(),this._initializeToggle(),this.optionsStore.options.display.inline&&this.display.show(),this._eventEmitters.triggerEvent.subscribe((t=>{this._triggerEvent(t)})),this._eventEmitters.viewUpdate.subscribe((()=>{this._viewUpdate()})),this._eventEmitters.updateViewDate.subscribe((t=>{this.viewDate=t}))}get viewDate(){return this.optionsStore.viewDate}set viewDate(t){this.optionsStore.viewDate=t,this.optionsStore.viewDate.setLocale(this.optionsStore.options.localization.locale),this.display._update("clock"===this.optionsStore.currentView?"clock":"calendar")}updateOptions(t,e=!1){e?this._initializeOptions(t,y):this._initializeOptions(t,this.optionsStore.options),this.display._rebuild()}toggle(){this._isDisabled||this.display.toggle()}show(){this._isDisabled||this.display.show()}hide(){this.display.hide()}disable(){this._isDisabled=!0,this.optionsStore.input?.setAttribute("disabled","disabled"),this.display.hide()}enable(){this._isDisabled=!1,this.optionsStore.input?.removeAttribute("disabled")}clear(){this.optionsStore.input.value="",this.dates.clear()}subscribe(t,e){let s;"string"==typeof t&&(t=[t]),s=Array.isArray(e)?e:[e],t.length!==s.length&&r.errorMessages.subscribeMismatch();const i=[];for(let e=0;e<t.length;e++){const o=t[e];if(Array.isArray(this._subscribers[o])||(this._subscribers[o]=[]),this._subscribers[o].push(s[e]),i.push({unsubscribe:this._unsubscribe.bind(this,o,this._subscribers[o].length-1)}),1===t.length)return i[0]}return i}dispose(){this.display.hide(),this.display._dispose(),this._eventEmitters.destroy(),this.optionsStore.input?.removeEventListener("change",this._inputChangeEvent),this.optionsStore.options.allowInputToggle&&this.optionsStore.input?.removeEventListener("click",this._toggleClickEvent),this._toggle?.removeEventListener("click",this._toggleClickEvent),this._subscribers={}}locale(t){const e=Y[t];e&&this.updateOptions({localization:e})}_triggerEvent(t){t.viewMode=this.optionsStore.currentView;const e=t.type===r.events.change;if(e){const{date:e,oldDate:s,isClear:i}=t;if(e&&s&&e.isSame(s)||!i&&!e&&!s)return;this._handleAfterChangeEvent(t),this.optionsStore.input?.dispatchEvent(new CustomEvent(t.type,{detail:t})),this.optionsStore.input?.dispatchEvent(new CustomEvent("change",{detail:t}))}if(this.optionsStore.element.dispatchEvent(new CustomEvent(t.type,{detail:t})),window.jQuery){const s=window.jQuery;e&&this.optionsStore.input?s(this.optionsStore.input).trigger(t):s(this.optionsStore.element).trigger(t)}this._publish(t)}_publish(t){Array.isArray(this._subscribers[t.type])&&this._subscribers[t.type].forEach((e=>{e(t)}))}_viewUpdate(){this._triggerEvent({type:r.events.update,viewDate:this.optionsStore.viewDate.clone})}_unsubscribe(t,e){this._subscribers[t].splice(e,1)}_initializeOptions(t,e,s=!1){let i=L.deepCopy(t);i=L._mergeOptions(i,e),s&&(i=L._dataToOptions(this.optionsStore.element,i)),L._validateConflicts(i),i.viewDate=i.viewDate.setLocale(i.localization.locale),this.optionsStore.viewDate.isSame(i.viewDate)||(this.optionsStore.viewDate=i.viewDate),i.display.components.year&&(this.optionsStore.minimumCalendarViewMode=2),i.display.components.month&&(this.optionsStore.minimumCalendarViewMode=1),i.display.components.date&&(this.optionsStore.minimumCalendarViewMode=0),this.optionsStore.currentCalendarViewMode=Math.max(this.optionsStore.minimumCalendarViewMode,this.optionsStore.currentCalendarViewMode),l[this.optionsStore.currentCalendarViewMode].name!==i.display.viewMode&&(this.optionsStore.currentCalendarViewMode=Math.max(l.findIndex((t=>t.name===i.display.viewMode)),this.optionsStore.minimumCalendarViewMode)),this.display?.isVisible&&this.display._update("all"),i.display.components.useTwentyfourHour&&void 0===i.localization.hourCycle?i.localization.hourCycle="h24":void 0===i.localization.hourCycle&&(i.localization.hourCycle=(t=>{if(!t)return"h12";const e={hour:"2-digit",minute:"2-digit",numberingSystem:"latn"},s=(new o).setLocale(t);s.hours=0;const i=s.parts(void 0,e).hour;if("12"===i)return"h12";if("24"===i)return"h24";s.hours=23;const a=s.parts(void 0,e).hour;return"00"===i&&"11"===a?"h11":"00"===i&&"23"===a?"h23":void console.warn(`couldn't determine hour cycle for ${t}. start: ${i}. end: ${a}`)})(i.localization.locale)),this.optionsStore.options=i}_initializeInput(){if("INPUT"==this.optionsStore.element.tagName)this.optionsStore.input=this.optionsStore.element;else{const t=this.optionsStore.element.dataset.tdTargetInput;this.optionsStore.input=null==t||"nearest"==t?this.optionsStore.element.querySelector("input"):this.optionsStore.element.querySelector(t)}this.optionsStore.input&&(!this.optionsStore.input.value&&this.optionsStore.options.defaultDate&&(this.optionsStore.input.value=this.dates.formatInput(this.optionsStore.options.defaultDate)),this.optionsStore.input.addEventListener("change",this._inputChangeEvent),this.optionsStore.options.allowInputToggle&&this.optionsStore.input.addEventListener("click",this._toggleClickEvent),this.optionsStore.input.value&&this._inputChangeEvent())}_initializeToggle(){if(this.optionsStore.options.display.inline)return;let t=this.optionsStore.element.dataset.tdTargetToggle;"nearest"==t&&(t='[data-td-toggle="datetimepicker"]'),this._toggle=null==t?this.optionsStore.element:this.optionsStore.element.querySelector(t),this._toggle.addEventListener("click",this._toggleClickEvent)}_handleAfterChangeEvent(t){!this.optionsStore.options.promptTimeOnDateChange||this.optionsStore.options.display.inline||this.optionsStore.options.display.sideBySide||!this.display._hasTime||this.display.widget?.getElementsByClassName(r.css.show)[0].classList.contains(r.css.timeContainer)||!t.oldDate&&this.optionsStore.options.useCurrent||t.oldDate&&t.date?.isSame(t.oldDate)||(clearTimeout(this._currentPromptTimeTimeout),this._currentPromptTimeTimeout=setTimeout((()=>{this.display.widget&&this._eventEmitters.action.emit({e:{currentTarget:this.display.widget.querySelector(`.${r.css.switch}`)},action:$.togglePicker})}),this.optionsStore.options.promptTimeOnDateChangeTransitionDelay))}}const Y={},j=t=>{Y[t.name]||(Y[t.name]=t.localization)},Q=t=>{const e=Y[t];e&&(y.localization=e)},K=function(t,e){return t?(t.installed||(t(e,{TempusDominus:W,Dates:U,Display:F,DateTime:o,Namespace:r},R),t.installed=!0),R):R},J="6.2.10",R={TempusDominus:W,extend:K,loadLocale:j,locale:Q,Namespace:r,DefaultOptions:y,DateTime:o,Unit:t.Unit,version:J};t.DateTime=o,t.DefaultOptions=y,t.Namespace=r,t.TempusDominus=W,t.extend=K,t.loadLocale=j,t.locale=Q,t.version=J,Object.defineProperty(t,"__esModule",{value:!0})}));

/*!
  * Tempus Dominus v6.2.10 (https://getdatepicker.com/)
  * Copyright 2013-2022 Jonathan Peterson
  * Licensed under MIT (https://github.com/Eonasdan/tempus-dominus/blob/master/LICENSE)
  */
(function(g,f){typeof exports==='object'&&typeof module!=='undefined'?module.exports=f():typeof define==='function'&&define.amd?define(f):(g=typeof globalThis!=='undefined'?globalThis:g||self,(g.tempusDominus=g.tempusDominus||{},g.tempusDominus.plugins=g.tempusDominus.plugins||{},g.tempusDominus.plugins.customDateFormat=f()));})(this,(function(){'use strict';class CustomDateFormat {
    constructor(dateTime, errorMessages) {
        this.REGEX_FORMAT = /\[([^\]]+)]|y{1,4}|M{1,4}|d{1,4}|H{1,2}|h{1,2}|t|T|m{1,2}|s{1,2}|Z{1,2}/g;
        // noinspection SpellCheckingInspection
        this.englishFormats = {
            LTS: 'h:mm:ss T',
            LT: 'h:mm T',
            L: 'MM/dd/yyyy',
            LL: 'MMMM d, yyyy',
            LLL: 'MMMM d, yyyy h:mm T',
            LLLL: 'dddd, MMMM d, yyyy h:mm T',
        };
        this.formattingTokens = /(\[[^[]*])|([-_:/.,()\s]+)|(T|t|yyyy|yy?|MM?M?M?|Do|dd?|hh?|HH?|mm?|ss?|z|zz?z?)/g;
        this.match1 = /\d/; // 0 - 9
        this.match2 = /\d\d/; // 00 - 99
        this.match3 = /\d{3}/; // 000 - 999
        this.match4 = /\d{4}/; // 0000 - 9999
        this.match1to2 = /\d\d?/; // 0 - 99
        this.matchSigned = /[+-]?\d+/; // -inf - inf
        this.matchOffset = /[+-]\d\d:?(\d\d)?|Z/; // +00:00 -00:00 +0000 or -0000 +00 or Z
        this.matchWord = /\d*[^-_:/,()\s\d]+/; // Word
        this.zoneExpressions = [
            this.matchOffset,
            (obj, input) => {
                obj.offset = this.offsetFromString(input);
            },
        ];
        this.expressions = {
            t: [
                this.matchWord,
                (ojb, input) => {
                    ojb.afternoon = this.meridiemMatch(input);
                },
            ],
            T: [
                this.matchWord,
                (ojb, input) => {
                    ojb.afternoon = this.meridiemMatch(input);
                },
            ],
            fff: [
                this.match3,
                (ojb, input) => {
                    ojb.milliseconds = +input;
                },
            ],
            s: [this.match1to2, this.addInput('seconds')],
            ss: [this.match1to2, this.addInput('seconds')],
            m: [this.match1to2, this.addInput('minutes')],
            mm: [this.match1to2, this.addInput('minutes')],
            H: [this.match1to2, this.addInput('hours')],
            h: [this.match1to2, this.addInput('hours')],
            HH: [this.match1to2, this.addInput('hours')],
            hh: [this.match1to2, this.addInput('hours')],
            d: [this.match1to2, this.addInput('day')],
            dd: [this.match2, this.addInput('day')],
            Do: [
                this.matchWord,
                (ojb, input) => {
                    [ojb.day] = input.match(/\d+/);
                    if (!this.localization.ordinal)
                        return;
                    for (let i = 1; i <= 31; i += 1) {
                        if (this.localization.ordinal(i).replace(/[[\]]/g, '') === input) {
                            ojb.day = i;
                        }
                    }
                },
            ],
            M: [this.match1to2, this.addInput('month')],
            MM: [this.match2, this.addInput('month')],
            MMM: [
                this.matchWord,
                (obj, input) => {
                    const months = this.getAllMonths();
                    const monthsShort = this.getAllMonths('short');
                    const matchIndex = (monthsShort || months.map((_) => _.slice(0, 3))).indexOf(input) + 1;
                    if (matchIndex < 1) {
                        throw new Error();
                    }
                    obj.month = matchIndex % 12 || matchIndex;
                },
            ],
            MMMM: [
                this.matchWord,
                (obj, input) => {
                    const months = this.getAllMonths();
                    const matchIndex = months.indexOf(input) + 1;
                    if (matchIndex < 1) {
                        throw new Error();
                    }
                    obj.month = matchIndex % 12 || matchIndex;
                },
            ],
            y: [this.matchSigned, this.addInput('year')],
            yy: [
                this.match2,
                (obj, input) => {
                    obj.year = this.parseTwoDigitYear(input);
                },
            ],
            yyyy: [this.match4, this.addInput('year')],
            // z: this.zoneExpressions,
            // zz: this.zoneExpressions,
            // zzz: this.zoneExpressions
        };
        this.parseFormattedInput = (input) => {
            if (!this.localization.format) {
                this.errorMessages.customDateFormatError('No format was provided');
            }
            try {
                if (['x', 'X'].indexOf(this.localization.format) > -1)
                    return new this.DateTime((this.localization.format === 'X' ? 1000 : 1) * input);
                const parser = this.makeParser(this.localization.format);
                const { year, month, day, hours, minutes, seconds, milliseconds, zone } = parser(input);
                const now = new this.DateTime();
                const d = day || (!year && !month ? now.getDate() : 1);
                const y = year || now.getFullYear();
                let M = 0;
                if (!(year && !month)) {
                    M = month > 0 ? month - 1 : now.getMonth();
                }
                const h = hours || 0;
                const m = minutes || 0;
                const s = seconds || 0;
                const ms = milliseconds || 0;
                if (zone) {
                    return new this.DateTime(Date.UTC(y, M, d, h, m, s, ms + zone.offset * 60 * 1000));
                }
                return new this.DateTime(y, M, d, h, m, s, ms);
            }
            catch (e) {
                this.errorMessages.customDateFormatError(`Unable to parse provided input: ${input}, format: ${this.localization.format}`);
                return new this.DateTime(''); // Invalid Date
            }
        };
        this.DateTime = dateTime;
        this.errorMessages = errorMessages;
    }
    getAllMonths(format = 'long') {
        const applyFormat = new Intl.DateTimeFormat(this.localization.locale, {
            month: format,
        }).format;
        return [...Array(12).keys()].map((m) => applyFormat(new Date(2021, m)));
    }
    replaceTokens(formatStr, formats) {
        return formatStr.replace(/(\[[^\]]+])|(LTS?|l{1,4}|L{1,4})/g, (_, a, b) => {
            const B = b && b.toUpperCase();
            return a || formats[B] || this.englishFormats[B];
        });
    }
    parseTwoDigitYear(input) {
        input = +input;
        return input + (input > 68 ? 1900 : 2000);
    }
    offsetFromString(string) {
        if (!string)
            return 0;
        if (string === 'Z')
            return 0;
        const parts = string.match(/([+-]|\d\d)/g);
        const minutes = +(parts[1] * 60) + (+parts[2] || 0);
        return minutes === 0 ? 0 : parts[0] === '+' ? -minutes : minutes; // eslint-disable-line no-nested-ternary
    }
    addInput(property) {
        return (time, input) => {
            time[property] = +input;
        };
    }
    /**
     * z = -4, zz = -04, zzz = -0400
     * @param date
     * @param style
     * @private
     */
    zoneInformation(date, style) {
        let name = date
            .parts(this.localization.locale, { timeZoneName: 'longOffset' })
            .timeZoneName.replace('GMT', '')
            .replace(':', '');
        const negative = name.includes('-');
        name = name.replace('-', '');
        if (style === 'z')
            name = name.substring(1, 2);
        else if (style === 'zz')
            name = name.substring(0, 2);
        return `${negative ? '-' : ''}${name}`;
    }
    meridiemMatch(input) {
        const meridiem = new Intl.DateTimeFormat(this.localization.locale, {
            hour: 'numeric',
            hour12: true,
        })
            .formatToParts(new Date(2022, 3, 4, 13))
            .find((p) => p.type === 'dayPeriod')?.value;
        return input.toLowerCase() === meridiem.toLowerCase();
    }
    correctHours(time) {
        const { afternoon } = time;
        if (afternoon !== undefined) {
            const { hours } = time;
            if (afternoon) {
                if (hours < 12) {
                    time.hours += 12;
                }
            }
            else if (hours === 12) {
                time.hours = 0;
            }
            delete time.afternoon;
        }
    }
    makeParser(format) {
        format = this.replaceTokens(format, this.localization.dateFormats);
        const array = format.match(this.formattingTokens);
        const { length } = array;
        for (let i = 0; i < length; i += 1) {
            const token = array[i];
            const parseTo = this.expressions[token];
            const regex = parseTo && parseTo[0];
            const parser = parseTo && parseTo[1];
            if (parser) {
                array[i] = { regex, parser };
            }
            else {
                array[i] = token.replace(/^\[|]$/g, '');
            }
        }
        return (input) => {
            const time = {};
            for (let i = 0, start = 0; i < length; i += 1) {
                const token = array[i];
                if (typeof token === 'string') {
                    start += token.length;
                }
                else {
                    const { regex, parser } = token;
                    const part = input.slice(start);
                    const match = regex.exec(part);
                    const value = match[0];
                    parser.call(this, time, value);
                    input = input.replace(value, '');
                }
            }
            this.correctHours(time);
            return time;
        };
    }
    format(dateTime) {
        if (!dateTime)
            return dateTime;
        if (JSON.stringify(dateTime) === 'null')
            return 'Invalid Date';
        const format = this.replaceTokens(this.localization.format ||
            `${this.englishFormats.L}, ${this.englishFormats.LT}`, this.localization.dateFormats);
        const formatter = (template) => new Intl.DateTimeFormat(this.localization.locale, template).format(dateTime);
        //if the format asks for a twenty-four-hour string but the hour cycle is not, then make a base guess
        const HHCycle = this.localization.hourCycle.startsWith('h1')
            ? 'h24'
            : this.localization.hourCycle;
        const hhCycle = this.localization.hourCycle.startsWith('h2')
            ? 'h12'
            : this.localization.hourCycle;
        const matches = {
            yy: formatter({ year: '2-digit' }),
            yyyy: dateTime.year,
            M: formatter({ month: 'numeric' }),
            MM: dateTime.monthFormatted,
            MMM: this.getAllMonths('short')[dateTime.getMonth()],
            MMMM: this.getAllMonths()[dateTime.getMonth()],
            d: dateTime.date,
            dd: dateTime.dateFormatted,
            ddd: formatter({ weekday: 'short' }),
            dddd: formatter({ weekday: 'long' }),
            H: dateTime.getHours(),
            HH: dateTime.getHoursFormatted(HHCycle),
            h: dateTime.hours > 12 ? dateTime.hours - 12 : dateTime.hours,
            hh: dateTime.getHoursFormatted(hhCycle),
            t: dateTime.meridiem(),
            T: dateTime.meridiem().toUpperCase(),
            m: dateTime.minutes,
            mm: dateTime.minutesFormatted,
            s: dateTime.seconds,
            ss: dateTime.secondsFormatted,
            fff: dateTime.getMilliseconds(),
            // z: this.zoneInformation(dateTime, 'z'), //-4
            // zz: this.zoneInformation(dateTime, 'zz'), //-04
            // zzz: this.zoneInformation(dateTime, 'zzz') //-0400
        };
        return format.replace(this.REGEX_FORMAT, (match, $1) => {
            return $1 || matches[match];
        });
    }
}
    var index = (_, tdClasses) => {
        const customDateFormat = new CustomDateFormat(tdClasses.DateTime, tdClasses.Namespace.errorMessages);
        // noinspection JSUnusedGlobalSymbols
        tdClasses.Dates.prototype.formatInput = function (date) {
            if (!date)
                return '';
            customDateFormat.localization = this.optionsStore.options.localization;
            return customDateFormat.format(date);
        };
        // noinspection JSUnusedGlobalSymbols
        tdClasses.Dates.prototype.parseInput = function (input) {
            customDateFormat.localization = this.optionsStore.options.localization;
            return customDateFormat.parseFormattedInput(input);
        };
        tdClasses.DateTime.fromString = function (input, localization) {
            customDateFormat.localization = localization;
            return customDateFormat.parseFormattedInput(input);
        };
    };return index;}));
/* flatpickr v4.6.13, @license MIT */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
        typeof define === 'function' && define.amd ? define(factory) :
            (global = typeof globalThis !== 'undefined' ? globalThis : global || self, global.flatpickr = factory());
}(this, (function () { 'use strict';

    /*! *****************************************************************************
    Copyright (c) Microsoft Corporation.

    Permission to use, copy, modify, and/or distribute this software for any
    purpose with or without fee is hereby granted.

    THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
    REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
    AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
    INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
    LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
    OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
    PERFORMANCE OF THIS SOFTWARE.
    ***************************************************************************** */

    var __assign = function() {
        __assign = Object.assign || function __assign(t) {
            for (var s, i = 1, n = arguments.length; i < n; i++) {
                s = arguments[i];
                for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p)) t[p] = s[p];
            }
            return t;
        };
        return __assign.apply(this, arguments);
    };

    function __spreadArrays() {
        for (var s = 0, i = 0, il = arguments.length; i < il; i++) s += arguments[i].length;
        for (var r = Array(s), k = 0, i = 0; i < il; i++)
            for (var a = arguments[i], j = 0, jl = a.length; j < jl; j++, k++)
                r[k] = a[j];
        return r;
    }

    var HOOKS = [
        "onChange",
        "onClose",
        "onDayCreate",
        "onDestroy",
        "onKeyDown",
        "onMonthChange",
        "onOpen",
        "onParseConfig",
        "onReady",
        "onValueUpdate",
        "onYearChange",
        "onPreCalendarPosition",
    ];
    var defaults = {
        _disable: [],
        allowInput: false,
        allowInvalidPreload: false,
        altFormat: "F j, Y",
        altInput: false,
        altInputClass: "form-control input",
        animate: typeof window === "object" &&
            window.navigator.userAgent.indexOf("MSIE") === -1,
        ariaDateFormat: "F j, Y",
        autoFillDefaultTime: true,
        clickOpens: true,
        closeOnSelect: true,
        conjunction: ", ",
        dateFormat: "Y-m-d",
        defaultHour: 12,
        defaultMinute: 0,
        defaultSeconds: 0,
        disable: [],
        disableMobile: false,
        enableSeconds: false,
        enableTime: false,
        errorHandler: function (err) {
            return typeof console !== "undefined" && console.warn(err);
        },
        getWeek: function (givenDate) {
            var date = new Date(givenDate.getTime());
            date.setHours(0, 0, 0, 0);
            // Thursday in current week decides the year.
            date.setDate(date.getDate() + 3 - ((date.getDay() + 6) % 7));
            // January 4 is always in week 1.
            var week1 = new Date(date.getFullYear(), 0, 4);
            // Adjust to Thursday in week 1 and count number of weeks from date to week1.
            return (1 +
                Math.round(((date.getTime() - week1.getTime()) / 86400000 -
                        3 +
                        ((week1.getDay() + 6) % 7)) /
                    7));
        },
        hourIncrement: 1,
        ignoredFocusElements: [],
        inline: false,
        locale: "default",
        minuteIncrement: 5,
        mode: "single",
        monthSelectorType: "dropdown",
        nextArrow: "<svg version='1.1' xmlns='http://www.w3.org/2000/svg' xmlns:xlink='http://www.w3.org/1999/xlink' viewBox='0 0 17 17'><g></g><path d='M13.207 8.472l-7.854 7.854-0.707-0.707 7.146-7.146-7.146-7.148 0.707-0.707 7.854 7.854z' /></svg>",
        noCalendar: false,
        now: new Date(),
        onChange: [],
        onClose: [],
        onDayCreate: [],
        onDestroy: [],
        onKeyDown: [],
        onMonthChange: [],
        onOpen: [],
        onParseConfig: [],
        onReady: [],
        onValueUpdate: [],
        onYearChange: [],
        onPreCalendarPosition: [],
        plugins: [],
        position: "auto",
        positionElement: undefined,
        prevArrow: "<svg version='1.1' xmlns='http://www.w3.org/2000/svg' xmlns:xlink='http://www.w3.org/1999/xlink' viewBox='0 0 17 17'><g></g><path d='M5.207 8.471l7.146 7.147-0.707 0.707-7.853-7.854 7.854-7.853 0.707 0.707-7.147 7.146z' /></svg>",
        shorthandCurrentMonth: false,
        showMonths: 1,
        static: false,
        time_24hr: false,
        weekNumbers: false,
        wrap: false,
    };

    var english = {
        weekdays: {
            shorthand: ["Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"],
            longhand: [
                "Sunday",
                "Monday",
                "Tuesday",
                "Wednesday",
                "Thursday",
                "Friday",
                "Saturday",
            ],
        },
        months: {
            shorthand: [
                "Jan",
                "Feb",
                "Mar",
                "Apr",
                "May",
                "Jun",
                "Jul",
                "Aug",
                "Sep",
                "Oct",
                "Nov",
                "Dec",
            ],
            longhand: [
                "January",
                "February",
                "March",
                "April",
                "May",
                "June",
                "July",
                "August",
                "September",
                "October",
                "November",
                "December",
            ],
        },
        daysInMonth: [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31],
        firstDayOfWeek: 0,
        ordinal: function (nth) {
            var s = nth % 100;
            if (s > 3 && s < 21)
                return "th";
            switch (s % 10) {
                case 1:
                    return "st";
                case 2:
                    return "nd";
                case 3:
                    return "rd";
                default:
                    return "th";
            }
        },
        rangeSeparator: " to ",
        weekAbbreviation: "Wk",
        scrollTitle: "Scroll to increment",
        toggleTitle: "Click to toggle",
        amPM: ["AM", "PM"],
        yearAriaLabel: "Year",
        monthAriaLabel: "Month",
        hourAriaLabel: "Hour",
        minuteAriaLabel: "Minute",
        time_24hr: false,
    };

    var pad = function (number, length) {
        if (length === void 0) { length = 2; }
        return ("000" + number).slice(length * -1);
    };
    var int = function (bool) { return (bool === true ? 1 : 0); };
    /* istanbul ignore next */
    function debounce(fn, wait) {
        var t;
        return function () {
            var _this = this;
            var args = arguments;
            clearTimeout(t);
            t = setTimeout(function () { return fn.apply(_this, args); }, wait);
        };
    }
    var arrayify = function (obj) {
        return obj instanceof Array ? obj : [obj];
    };

    function toggleClass(elem, className, bool) {
        if (bool === true)
            return elem.classList.add(className);
        elem.classList.remove(className);
    }
    function createElement(tag, className, content) {
        var e = window.document.createElement(tag);
        className = className || "";
        content = content || "";
        e.className = className;
        if (content !== undefined)
            e.textContent = content;
        return e;
    }
    function clearNode(node) {
        while (node.firstChild)
            node.removeChild(node.firstChild);
    }
    function findParent(node, condition) {
        if (condition(node))
            return node;
        else if (node.parentNode)
            return findParent(node.parentNode, condition);
        return undefined; // nothing found
    }
    function createNumberInput(inputClassName, opts) {
        var wrapper = createElement("div", "numInputWrapper"), numInput = createElement("input", "numInput " + inputClassName), arrowUp = createElement("span", "arrowUp"), arrowDown = createElement("span", "arrowDown");
        if (navigator.userAgent.indexOf("MSIE 9.0") === -1) {
            numInput.type = "number";
        }
        else {
            numInput.type = "text";
            numInput.pattern = "\\d*";
        }
        if (opts !== undefined)
            for (var key in opts)
                numInput.setAttribute(key, opts[key]);
        wrapper.appendChild(numInput);
        wrapper.appendChild(arrowUp);
        wrapper.appendChild(arrowDown);
        return wrapper;
    }
    function getEventTarget(event) {
        try {
            if (typeof event.composedPath === "function") {
                var path = event.composedPath();
                return path[0];
            }
            return event.target;
        }
        catch (error) {
            return event.target;
        }
    }

    var doNothing = function () { return undefined; };
    var monthToStr = function (monthNumber, shorthand, locale) { return locale.months[shorthand ? "shorthand" : "longhand"][monthNumber]; };
    var revFormat = {
        D: doNothing,
        F: function (dateObj, monthName, locale) {
            dateObj.setMonth(locale.months.longhand.indexOf(monthName));
        },
        G: function (dateObj, hour) {
            dateObj.setHours((dateObj.getHours() >= 12 ? 12 : 0) + parseFloat(hour));
        },
        H: function (dateObj, hour) {
            dateObj.setHours(parseFloat(hour));
        },
        J: function (dateObj, day) {
            dateObj.setDate(parseFloat(day));
        },
        K: function (dateObj, amPM, locale) {
            dateObj.setHours((dateObj.getHours() % 12) +
                12 * int(new RegExp(locale.amPM[1], "i").test(amPM)));
        },
        M: function (dateObj, shortMonth, locale) {
            dateObj.setMonth(locale.months.shorthand.indexOf(shortMonth));
        },
        S: function (dateObj, seconds) {
            dateObj.setSeconds(parseFloat(seconds));
        },
        U: function (_, unixSeconds) { return new Date(parseFloat(unixSeconds) * 1000); },
        W: function (dateObj, weekNum, locale) {
            var weekNumber = parseInt(weekNum);
            var date = new Date(dateObj.getFullYear(), 0, 2 + (weekNumber - 1) * 7, 0, 0, 0, 0);
            date.setDate(date.getDate() - date.getDay() + locale.firstDayOfWeek);
            return date;
        },
        Y: function (dateObj, year) {
            dateObj.setFullYear(parseFloat(year));
        },
        Z: function (_, ISODate) { return new Date(ISODate); },
        d: function (dateObj, day) {
            dateObj.setDate(parseFloat(day));
        },
        h: function (dateObj, hour) {
            dateObj.setHours((dateObj.getHours() >= 12 ? 12 : 0) + parseFloat(hour));
        },
        i: function (dateObj, minutes) {
            dateObj.setMinutes(parseFloat(minutes));
        },
        j: function (dateObj, day) {
            dateObj.setDate(parseFloat(day));
        },
        l: doNothing,
        m: function (dateObj, month) {
            dateObj.setMonth(parseFloat(month) - 1);
        },
        n: function (dateObj, month) {
            dateObj.setMonth(parseFloat(month) - 1);
        },
        s: function (dateObj, seconds) {
            dateObj.setSeconds(parseFloat(seconds));
        },
        u: function (_, unixMillSeconds) {
            return new Date(parseFloat(unixMillSeconds));
        },
        w: doNothing,
        y: function (dateObj, year) {
            dateObj.setFullYear(2000 + parseFloat(year));
        },
    };
    var tokenRegex = {
        D: "",
        F: "",
        G: "(\\d\\d|\\d)",
        H: "(\\d\\d|\\d)",
        J: "(\\d\\d|\\d)\\w+",
        K: "",
        M: "",
        S: "(\\d\\d|\\d)",
        U: "(.+)",
        W: "(\\d\\d|\\d)",
        Y: "(\\d{4})",
        Z: "(.+)",
        d: "(\\d\\d|\\d)",
        h: "(\\d\\d|\\d)",
        i: "(\\d\\d|\\d)",
        j: "(\\d\\d|\\d)",
        l: "",
        m: "(\\d\\d|\\d)",
        n: "(\\d\\d|\\d)",
        s: "(\\d\\d|\\d)",
        u: "(.+)",
        w: "(\\d\\d|\\d)",
        y: "(\\d{2})",
    };
    var formats = {
        // get the date in UTC
        Z: function (date) { return date.toISOString(); },
        // weekday name, short, e.g. Thu
        D: function (date, locale, options) {
            return locale.weekdays.shorthand[formats.w(date, locale, options)];
        },
        // full month name e.g. January
        F: function (date, locale, options) {
            return monthToStr(formats.n(date, locale, options) - 1, false, locale);
        },
        // padded hour 1-12
        G: function (date, locale, options) {
            return pad(formats.h(date, locale, options));
        },
        // hours with leading zero e.g. 03
        H: function (date) { return pad(date.getHours()); },
        // day (1-30) with ordinal suffix e.g. 1st, 2nd
        J: function (date, locale) {
            return locale.ordinal !== undefined
                ? date.getDate() + locale.ordinal(date.getDate())
                : date.getDate();
        },
        // AM/PM
        K: function (date, locale) { return locale.amPM[int(date.getHours() > 11)]; },
        // shorthand month e.g. Jan, Sep, Oct, etc
        M: function (date, locale) {
            return monthToStr(date.getMonth(), true, locale);
        },
        // seconds 00-59
        S: function (date) { return pad(date.getSeconds()); },
        // unix timestamp
        U: function (date) { return date.getTime() / 1000; },
        W: function (date, _, options) {
            return options.getWeek(date);
        },
        // full year e.g. 2016, padded (0001-9999)
        Y: function (date) { return pad(date.getFullYear(), 4); },
        // day in month, padded (01-30)
        d: function (date) { return pad(date.getDate()); },
        // hour from 1-12 (am/pm)
        h: function (date) { return (date.getHours() % 12 ? date.getHours() % 12 : 12); },
        // minutes, padded with leading zero e.g. 09
        i: function (date) { return pad(date.getMinutes()); },
        // day in month (1-30)
        j: function (date) { return date.getDate(); },
        // weekday name, full, e.g. Thursday
        l: function (date, locale) {
            return locale.weekdays.longhand[date.getDay()];
        },
        // padded month number (01-12)
        m: function (date) { return pad(date.getMonth() + 1); },
        // the month number (1-12)
        n: function (date) { return date.getMonth() + 1; },
        // seconds 0-59
        s: function (date) { return date.getSeconds(); },
        // Unix Milliseconds
        u: function (date) { return date.getTime(); },
        // number of the day of the week
        w: function (date) { return date.getDay(); },
        // last two digits of year e.g. 16 for 2016
        y: function (date) { return String(date.getFullYear()).substring(2); },
    };

    var createDateFormatter = function (_a) {
        var _b = _a.config, config = _b === void 0 ? defaults : _b, _c = _a.l10n, l10n = _c === void 0 ? english : _c, _d = _a.isMobile, isMobile = _d === void 0 ? false : _d;
        return function (dateObj, frmt, overrideLocale) {
            var locale = overrideLocale || l10n;
            if (config.formatDate !== undefined && !isMobile) {
                return config.formatDate(dateObj, frmt, locale);
            }
            return frmt
                .split("")
                .map(function (c, i, arr) {
                    return formats[c] && arr[i - 1] !== "\\"
                        ? formats[c](dateObj, locale, config)
                        : c !== "\\"
                            ? c
                            : "";
                })
                .join("");
        };
    };
    var createDateParser = function (_a) {
        var _b = _a.config, config = _b === void 0 ? defaults : _b, _c = _a.l10n, l10n = _c === void 0 ? english : _c;
        return function (date, givenFormat, timeless, customLocale) {
            if (date !== 0 && !date)
                return undefined;
            var locale = customLocale || l10n;
            var parsedDate;
            var dateOrig = date;
            if (date instanceof Date)
                parsedDate = new Date(date.getTime());
            else if (typeof date !== "string" &&
                date.toFixed !== undefined // timestamp
            )
                // create a copy
                parsedDate = new Date(date);
            else if (typeof date === "string") {
                // date string
                var format = givenFormat || (config || defaults).dateFormat;
                var datestr = String(date).trim();
                if (datestr === "today") {
                    parsedDate = new Date();
                    timeless = true;
                }
                else if (config && config.parseDate) {
                    parsedDate = config.parseDate(date, format);
                }
                else if (/Z$/.test(datestr) ||
                    /GMT$/.test(datestr) // datestrings w/ timezone
                ) {
                    parsedDate = new Date(date);
                }
                else {
                    var matched = void 0, ops = [];
                    for (var i = 0, matchIndex = 0, regexStr = ""; i < format.length; i++) {
                        var token_1 = format[i];
                        var isBackSlash = token_1 === "\\";
                        var escaped = format[i - 1] === "\\" || isBackSlash;
                        if (tokenRegex[token_1] && !escaped) {
                            regexStr += tokenRegex[token_1];
                            var match = new RegExp(regexStr).exec(date);
                            if (match && (matched = true)) {
                                ops[token_1 !== "Y" ? "push" : "unshift"]({
                                    fn: revFormat[token_1],
                                    val: match[++matchIndex],
                                });
                            }
                        }
                        else if (!isBackSlash)
                            regexStr += "."; // don't really care
                    }
                    parsedDate =
                        !config || !config.noCalendar
                            ? new Date(new Date().getFullYear(), 0, 1, 0, 0, 0, 0)
                            : new Date(new Date().setHours(0, 0, 0, 0));
                    ops.forEach(function (_a) {
                        var fn = _a.fn, val = _a.val;
                        return (parsedDate = fn(parsedDate, val, locale) || parsedDate);
                    });
                    parsedDate = matched ? parsedDate : undefined;
                }
            }
            /* istanbul ignore next */
            if (!(parsedDate instanceof Date && !isNaN(parsedDate.getTime()))) {
                config.errorHandler(new Error("Invalid date provided: " + dateOrig));
                return undefined;
            }
            if (timeless === true)
                parsedDate.setHours(0, 0, 0, 0);
            return parsedDate;
        };
    };
    /**
     * Compute the difference in dates, measured in ms
     */
    function compareDates(date1, date2, timeless) {
        if (timeless === void 0) { timeless = true; }
        if (timeless !== false) {
            return (new Date(date1.getTime()).setHours(0, 0, 0, 0) -
                new Date(date2.getTime()).setHours(0, 0, 0, 0));
        }
        return date1.getTime() - date2.getTime();
    }
    var isBetween = function (ts, ts1, ts2) {
        return ts > Math.min(ts1, ts2) && ts < Math.max(ts1, ts2);
    };
    var calculateSecondsSinceMidnight = function (hours, minutes, seconds) {
        return hours * 3600 + minutes * 60 + seconds;
    };
    var parseSeconds = function (secondsSinceMidnight) {
        var hours = Math.floor(secondsSinceMidnight / 3600), minutes = (secondsSinceMidnight - hours * 3600) / 60;
        return [hours, minutes, secondsSinceMidnight - hours * 3600 - minutes * 60];
    };
    var duration = {
        DAY: 86400000,
    };
    function getDefaultHours(config) {
        var hours = config.defaultHour;
        var minutes = config.defaultMinute;
        var seconds = config.defaultSeconds;
        if (config.minDate !== undefined) {
            var minHour = config.minDate.getHours();
            var minMinutes = config.minDate.getMinutes();
            var minSeconds = config.minDate.getSeconds();
            if (hours < minHour) {
                hours = minHour;
            }
            if (hours === minHour && minutes < minMinutes) {
                minutes = minMinutes;
            }
            if (hours === minHour && minutes === minMinutes && seconds < minSeconds)
                seconds = config.minDate.getSeconds();
        }
        if (config.maxDate !== undefined) {
            var maxHr = config.maxDate.getHours();
            var maxMinutes = config.maxDate.getMinutes();
            hours = Math.min(hours, maxHr);
            if (hours === maxHr)
                minutes = Math.min(maxMinutes, minutes);
            if (hours === maxHr && minutes === maxMinutes)
                seconds = config.maxDate.getSeconds();
        }
        return { hours: hours, minutes: minutes, seconds: seconds };
    }

    if (typeof Object.assign !== "function") {
        Object.assign = function (target) {
            var args = [];
            for (var _i = 1; _i < arguments.length; _i++) {
                args[_i - 1] = arguments[_i];
            }
            if (!target) {
                throw TypeError("Cannot convert undefined or null to object");
            }
            var _loop_1 = function (source) {
                if (source) {
                    Object.keys(source).forEach(function (key) { return (target[key] = source[key]); });
                }
            };
            for (var _a = 0, args_1 = args; _a < args_1.length; _a++) {
                var source = args_1[_a];
                _loop_1(source);
            }
            return target;
        };
    }

    var DEBOUNCED_CHANGE_MS = 300;
    function FlatpickrInstance(element, instanceConfig) {
        var self = {
            config: __assign(__assign({}, defaults), flatpickr.defaultConfig),
            l10n: english,
        };
        self.parseDate = createDateParser({ config: self.config, l10n: self.l10n });
        self._handlers = [];
        self.pluginElements = [];
        self.loadedPlugins = [];
        self._bind = bind;
        self._setHoursFromDate = setHoursFromDate;
        self._positionCalendar = positionCalendar;
        self.changeMonth = changeMonth;
        self.changeYear = changeYear;
        self.clear = clear;
        self.close = close;
        self.onMouseOver = onMouseOver;
        self._createElement = createElement;
        self.createDay = createDay;
        self.destroy = destroy;
        self.isEnabled = isEnabled;
        self.jumpToDate = jumpToDate;
        self.updateValue = updateValue;
        self.open = open;
        self.redraw = redraw;
        self.set = set;
        self.setDate = setDate;
        self.toggle = toggle;
        function setupHelperFunctions() {
            self.utils = {
                getDaysInMonth: function (month, yr) {
                    if (month === void 0) { month = self.currentMonth; }
                    if (yr === void 0) { yr = self.currentYear; }
                    if (month === 1 && ((yr % 4 === 0 && yr % 100 !== 0) || yr % 400 === 0))
                        return 29;
                    return self.l10n.daysInMonth[month];
                },
            };
        }
        function init() {
            self.element = self.input = element;
            self.isOpen = false;
            parseConfig();
            setupLocale();
            setupInputs();
            setupDates();
            setupHelperFunctions();
            if (!self.isMobile)
                build();
            bindEvents();
            if (self.selectedDates.length || self.config.noCalendar) {
                if (self.config.enableTime) {
                    setHoursFromDate(self.config.noCalendar ? self.latestSelectedDateObj : undefined);
                }
                updateValue(false);
            }
            setCalendarWidth();
            var isSafari = /^((?!chrome|android).)*safari/i.test(navigator.userAgent);
            /* TODO: investigate this further

              Currently, there is weird positioning behavior in safari causing pages
              to scroll up. https://github.com/chmln/flatpickr/issues/563

              However, most browsers are not Safari and positioning is expensive when used
              in scale. https://github.com/chmln/flatpickr/issues/1096
            */
            if (!self.isMobile && isSafari) {
                positionCalendar();
            }
            triggerEvent("onReady");
        }
        function getClosestActiveElement() {
            var _a;
            return (((_a = self.calendarContainer) === null || _a === void 0 ? void 0 : _a.getRootNode())
                .activeElement || document.activeElement);
        }
        function bindToInstance(fn) {
            return fn.bind(self);
        }
        function setCalendarWidth() {
            var config = self.config;
            if (config.weekNumbers === false && config.showMonths === 1) {
                return;
            }
            else if (config.noCalendar !== true) {
                window.requestAnimationFrame(function () {
                    if (self.calendarContainer !== undefined) {
                        self.calendarContainer.style.visibility = "hidden";
                        self.calendarContainer.style.display = "block";
                    }
                    if (self.daysContainer !== undefined) {
                        var daysWidth = (self.days.offsetWidth + 1) * config.showMonths;
                        self.daysContainer.style.width = daysWidth + "px";
                        self.calendarContainer.style.width =
                            daysWidth +
                            (self.weekWrapper !== undefined
                                ? self.weekWrapper.offsetWidth
                                : 0) +
                            "px";
                        self.calendarContainer.style.removeProperty("visibility");
                        self.calendarContainer.style.removeProperty("display");
                    }
                });
            }
        }
        /**
         * The handler for all events targeting the time inputs
         */
        function updateTime(e) {
            if (self.selectedDates.length === 0) {
                var defaultDate = self.config.minDate === undefined ||
                compareDates(new Date(), self.config.minDate) >= 0
                    ? new Date()
                    : new Date(self.config.minDate.getTime());
                var defaults = getDefaultHours(self.config);
                defaultDate.setHours(defaults.hours, defaults.minutes, defaults.seconds, defaultDate.getMilliseconds());
                self.selectedDates = [defaultDate];
                self.latestSelectedDateObj = defaultDate;
            }
            if (e !== undefined && e.type !== "blur") {
                timeWrapper(e);
            }
            var prevValue = self._input.value;
            setHoursFromInputs();
            updateValue();
            if (self._input.value !== prevValue) {
                self._debouncedChange();
            }
        }
        function ampm2military(hour, amPM) {
            return (hour % 12) + 12 * int(amPM === self.l10n.amPM[1]);
        }
        function military2ampm(hour) {
            switch (hour % 24) {
                case 0:
                case 12:
                    return 12;
                default:
                    return hour % 12;
            }
        }
        /**
         * Syncs the selected date object time with user's time input
         */
        function setHoursFromInputs() {
            if (self.hourElement === undefined || self.minuteElement === undefined)
                return;
            var hours = (parseInt(self.hourElement.value.slice(-2), 10) || 0) % 24, minutes = (parseInt(self.minuteElement.value, 10) || 0) % 60, seconds = self.secondElement !== undefined
                ? (parseInt(self.secondElement.value, 10) || 0) % 60
                : 0;
            if (self.amPM !== undefined) {
                hours = ampm2military(hours, self.amPM.textContent);
            }
            var limitMinHours = self.config.minTime !== undefined ||
                (self.config.minDate &&
                    self.minDateHasTime &&
                    self.latestSelectedDateObj &&
                    compareDates(self.latestSelectedDateObj, self.config.minDate, true) ===
                    0);
            var limitMaxHours = self.config.maxTime !== undefined ||
                (self.config.maxDate &&
                    self.maxDateHasTime &&
                    self.latestSelectedDateObj &&
                    compareDates(self.latestSelectedDateObj, self.config.maxDate, true) ===
                    0);
            if (self.config.maxTime !== undefined &&
                self.config.minTime !== undefined &&
                self.config.minTime > self.config.maxTime) {
                var minBound = calculateSecondsSinceMidnight(self.config.minTime.getHours(), self.config.minTime.getMinutes(), self.config.minTime.getSeconds());
                var maxBound = calculateSecondsSinceMidnight(self.config.maxTime.getHours(), self.config.maxTime.getMinutes(), self.config.maxTime.getSeconds());
                var currentTime = calculateSecondsSinceMidnight(hours, minutes, seconds);
                if (currentTime > maxBound && currentTime < minBound) {
                    var result = parseSeconds(minBound);
                    hours = result[0];
                    minutes = result[1];
                    seconds = result[2];
                }
            }
            else {
                if (limitMaxHours) {
                    var maxTime = self.config.maxTime !== undefined
                        ? self.config.maxTime
                        : self.config.maxDate;
                    hours = Math.min(hours, maxTime.getHours());
                    if (hours === maxTime.getHours())
                        minutes = Math.min(minutes, maxTime.getMinutes());
                    if (minutes === maxTime.getMinutes())
                        seconds = Math.min(seconds, maxTime.getSeconds());
                }
                if (limitMinHours) {
                    var minTime = self.config.minTime !== undefined
                        ? self.config.minTime
                        : self.config.minDate;
                    hours = Math.max(hours, minTime.getHours());
                    if (hours === minTime.getHours() && minutes < minTime.getMinutes())
                        minutes = minTime.getMinutes();
                    if (minutes === minTime.getMinutes())
                        seconds = Math.max(seconds, minTime.getSeconds());
                }
            }
            setHours(hours, minutes, seconds);
        }
        /**
         * Syncs time input values with a date
         */
        function setHoursFromDate(dateObj) {
            var date = dateObj || self.latestSelectedDateObj;
            if (date && date instanceof Date) {
                setHours(date.getHours(), date.getMinutes(), date.getSeconds());
            }
        }
        /**
         * Sets the hours, minutes, and optionally seconds
         * of the latest selected date object and the
         * corresponding time inputs
         * @param {Number} hours the hour. whether its military
         *                 or am-pm gets inferred from config
         * @param {Number} minutes the minutes
         * @param {Number} seconds the seconds (optional)
         */
        function setHours(hours, minutes, seconds) {
            if (self.latestSelectedDateObj !== undefined) {
                self.latestSelectedDateObj.setHours(hours % 24, minutes, seconds || 0, 0);
            }
            if (!self.hourElement || !self.minuteElement || self.isMobile)
                return;
            self.hourElement.value = pad(!self.config.time_24hr
                ? ((12 + hours) % 12) + 12 * int(hours % 12 === 0)
                : hours);
            self.minuteElement.value = pad(minutes);
            if (self.amPM !== undefined)
                self.amPM.textContent = self.l10n.amPM[int(hours >= 12)];
            if (self.secondElement !== undefined)
                self.secondElement.value = pad(seconds);
        }
        /**
         * Handles the year input and incrementing events
         * @param {Event} event the keyup or increment event
         */
        function onYearInput(event) {
            var eventTarget = getEventTarget(event);
            var year = parseInt(eventTarget.value) + (event.delta || 0);
            if (year / 1000 > 1 ||
                (event.key === "Enter" && !/[^\d]/.test(year.toString()))) {
                changeYear(year);
            }
        }
        /**
         * Essentially addEventListener + tracking
         * @param {Element} element the element to addEventListener to
         * @param {String} event the event name
         * @param {Function} handler the event handler
         */
        function bind(element, event, handler, options) {
            if (event instanceof Array)
                return event.forEach(function (ev) { return bind(element, ev, handler, options); });
            if (element instanceof Array)
                return element.forEach(function (el) { return bind(el, event, handler, options); });
            element.addEventListener(event, handler, options);
            self._handlers.push({
                remove: function () { return element.removeEventListener(event, handler, options); },
            });
        }
        function triggerChange() {
            triggerEvent("onChange");
        }
        /**
         * Adds all the necessary event listeners
         */
        function bindEvents() {
            if (self.config.wrap) {
                ["open", "close", "toggle", "clear"].forEach(function (evt) {
                    Array.prototype.forEach.call(self.element.querySelectorAll("[data-" + evt + "]"), function (el) {
                        return bind(el, "click", self[evt]);
                    });
                });
            }
            if (self.isMobile) {
                setupMobile();
                return;
            }
            var debouncedResize = debounce(onResize, 50);
            self._debouncedChange = debounce(triggerChange, DEBOUNCED_CHANGE_MS);
            if (self.daysContainer && !/iPhone|iPad|iPod/i.test(navigator.userAgent))
                bind(self.daysContainer, "mouseover", function (e) {
                    if (self.config.mode === "range")
                        onMouseOver(getEventTarget(e));
                });
            bind(self._input, "keydown", onKeyDown);
            if (self.calendarContainer !== undefined) {
                bind(self.calendarContainer, "keydown", onKeyDown);
            }
            if (!self.config.inline && !self.config.static)
                bind(window, "resize", debouncedResize);
            if (window.ontouchstart !== undefined)
                bind(window.document, "touchstart", documentClick);
            else
                bind(window.document, "mousedown", documentClick);
            bind(window.document, "focus", documentClick, { capture: true });
            if (self.config.clickOpens === true) {
                bind(self._input, "focus", self.open);
                bind(self._input, "click", self.open);
            }
            if (self.daysContainer !== undefined) {
                bind(self.monthNav, "click", onMonthNavClick);
                bind(self.monthNav, ["keyup", "increment"], onYearInput);
                bind(self.daysContainer, "click", selectDate);
            }
            if (self.timeContainer !== undefined &&
                self.minuteElement !== undefined &&
                self.hourElement !== undefined) {
                var selText = function (e) {
                    return getEventTarget(e).select();
                };
                bind(self.timeContainer, ["increment"], updateTime);
                bind(self.timeContainer, "blur", updateTime, { capture: true });
                bind(self.timeContainer, "click", timeIncrement);
                bind([self.hourElement, self.minuteElement], ["focus", "click"], selText);
                if (self.secondElement !== undefined)
                    bind(self.secondElement, "focus", function () { return self.secondElement && self.secondElement.select(); });
                if (self.amPM !== undefined) {
                    bind(self.amPM, "click", function (e) {
                        updateTime(e);
                    });
                }
            }
            if (self.config.allowInput) {
                bind(self._input, "blur", onBlur);
            }
        }
        /**
         * Set the calendar view to a particular date.
         * @param {Date} jumpDate the date to set the view to
         * @param {boolean} triggerChange if change events should be triggered
         */
        function jumpToDate(jumpDate, triggerChange) {
            var jumpTo = jumpDate !== undefined
                ? self.parseDate(jumpDate)
                : self.latestSelectedDateObj ||
                (self.config.minDate && self.config.minDate > self.now
                    ? self.config.minDate
                    : self.config.maxDate && self.config.maxDate < self.now
                        ? self.config.maxDate
                        : self.now);
            var oldYear = self.currentYear;
            var oldMonth = self.currentMonth;
            try {
                if (jumpTo !== undefined) {
                    self.currentYear = jumpTo.getFullYear();
                    self.currentMonth = jumpTo.getMonth();
                }
            }
            catch (e) {
                /* istanbul ignore next */
                e.message = "Invalid date supplied: " + jumpTo;
                self.config.errorHandler(e);
            }
            if (triggerChange && self.currentYear !== oldYear) {
                triggerEvent("onYearChange");
                buildMonthSwitch();
            }
            if (triggerChange &&
                (self.currentYear !== oldYear || self.currentMonth !== oldMonth)) {
                triggerEvent("onMonthChange");
            }
            self.redraw();
        }
        /**
         * The up/down arrow handler for time inputs
         * @param {Event} e the click event
         */
        function timeIncrement(e) {
            var eventTarget = getEventTarget(e);
            if (~eventTarget.className.indexOf("arrow"))
                incrementNumInput(e, eventTarget.classList.contains("arrowUp") ? 1 : -1);
        }
        /**
         * Increments/decrements the value of input associ-
         * ated with the up/down arrow by dispatching an
         * "increment" event on the input.
         *
         * @param {Event} e the click event
         * @param {Number} delta the diff (usually 1 or -1)
         * @param {Element} inputElem the input element
         */
        function incrementNumInput(e, delta, inputElem) {
            var target = e && getEventTarget(e);
            var input = inputElem ||
                (target && target.parentNode && target.parentNode.firstChild);
            var event = createEvent("increment");
            event.delta = delta;
            input && input.dispatchEvent(event);
        }
        function build() {
            var fragment = window.document.createDocumentFragment();
            self.calendarContainer = createElement("div", "flatpickr-calendar");
            self.calendarContainer.tabIndex = -1;
            if (!self.config.noCalendar) {
                fragment.appendChild(buildMonthNav());
                self.innerContainer = createElement("div", "flatpickr-innerContainer");
                if (self.config.weekNumbers) {
                    var _a = buildWeeks(), weekWrapper = _a.weekWrapper, weekNumbers = _a.weekNumbers;
                    self.innerContainer.appendChild(weekWrapper);
                    self.weekNumbers = weekNumbers;
                    self.weekWrapper = weekWrapper;
                }
                self.rContainer = createElement("div", "flatpickr-rContainer");
                self.rContainer.appendChild(buildWeekdays());
                if (!self.daysContainer) {
                    self.daysContainer = createElement("div", "flatpickr-days");
                    self.daysContainer.tabIndex = -1;
                }
                buildDays();
                self.rContainer.appendChild(self.daysContainer);
                self.innerContainer.appendChild(self.rContainer);
                fragment.appendChild(self.innerContainer);
            }
            if (self.config.enableTime) {
                fragment.appendChild(buildTime());
            }
            toggleClass(self.calendarContainer, "rangeMode", self.config.mode === "range");
            toggleClass(self.calendarContainer, "animate", self.config.animate === true);
            toggleClass(self.calendarContainer, "multiMonth", self.config.showMonths > 1);
            self.calendarContainer.appendChild(fragment);
            var customAppend = self.config.appendTo !== undefined &&
                self.config.appendTo.nodeType !== undefined;
            if (self.config.inline || self.config.static) {
                self.calendarContainer.classList.add(self.config.inline ? "inline" : "static");
                if (self.config.inline) {
                    if (!customAppend && self.element.parentNode)
                        self.element.parentNode.insertBefore(self.calendarContainer, self._input.nextSibling);
                    else if (self.config.appendTo !== undefined)
                        self.config.appendTo.appendChild(self.calendarContainer);
                }
                if (self.config.static) {
                    var wrapper = createElement("div", "flatpickr-wrapper");
                    if (self.element.parentNode)
                        self.element.parentNode.insertBefore(wrapper, self.element);
                    wrapper.appendChild(self.element);
                    if (self.altInput)
                        wrapper.appendChild(self.altInput);
                    wrapper.appendChild(self.calendarContainer);
                }
            }
            if (!self.config.static && !self.config.inline)
                (self.config.appendTo !== undefined
                    ? self.config.appendTo
                    : window.document.body).appendChild(self.calendarContainer);
        }
        function createDay(className, date, _dayNumber, i) {
            var dateIsEnabled = isEnabled(date, true), dayElement = createElement("span", className, date.getDate().toString());
            dayElement.dateObj = date;
            dayElement.$i = i;
            dayElement.setAttribute("aria-label", self.formatDate(date, self.config.ariaDateFormat));
            if (className.indexOf("hidden") === -1 &&
                compareDates(date, self.now) === 0) {
                self.todayDateElem = dayElement;
                dayElement.classList.add("today");
                dayElement.setAttribute("aria-current", "date");
            }
            if (dateIsEnabled) {
                dayElement.tabIndex = -1;
                if (isDateSelected(date)) {
                    dayElement.classList.add("selected");
                    self.selectedDateElem = dayElement;
                    if (self.config.mode === "range") {
                        toggleClass(dayElement, "startRange", self.selectedDates[0] &&
                            compareDates(date, self.selectedDates[0], true) === 0);
                        toggleClass(dayElement, "endRange", self.selectedDates[1] &&
                            compareDates(date, self.selectedDates[1], true) === 0);
                        if (className === "nextMonthDay")
                            dayElement.classList.add("inRange");
                    }
                }
            }
            else {
                dayElement.classList.add("flatpickr-disabled");
            }
            if (self.config.mode === "range") {
                if (isDateInRange(date) && !isDateSelected(date))
                    dayElement.classList.add("inRange");
            }
            if (self.weekNumbers &&
                self.config.showMonths === 1 &&
                className !== "prevMonthDay" &&
                i % 7 === 6) {
                self.weekNumbers.insertAdjacentHTML("beforeend", "<span class='flatpickr-day'>" + self.config.getWeek(date) + "</span>");
            }
            triggerEvent("onDayCreate", dayElement);
            return dayElement;
        }
        function focusOnDayElem(targetNode) {
            targetNode.focus();
            if (self.config.mode === "range")
                onMouseOver(targetNode);
        }
        function getFirstAvailableDay(delta) {
            var startMonth = delta > 0 ? 0 : self.config.showMonths - 1;
            var endMonth = delta > 0 ? self.config.showMonths : -1;
            for (var m = startMonth; m != endMonth; m += delta) {
                var month = self.daysContainer.children[m];
                var startIndex = delta > 0 ? 0 : month.children.length - 1;
                var endIndex = delta > 0 ? month.children.length : -1;
                for (var i = startIndex; i != endIndex; i += delta) {
                    var c = month.children[i];
                    if (c.className.indexOf("hidden") === -1 && isEnabled(c.dateObj))
                        return c;
                }
            }
            return undefined;
        }
        function getNextAvailableDay(current, delta) {
            var givenMonth = current.className.indexOf("Month") === -1
                ? current.dateObj.getMonth()
                : self.currentMonth;
            var endMonth = delta > 0 ? self.config.showMonths : -1;
            var loopDelta = delta > 0 ? 1 : -1;
            for (var m = givenMonth - self.currentMonth; m != endMonth; m += loopDelta) {
                var month = self.daysContainer.children[m];
                var startIndex = givenMonth - self.currentMonth === m
                    ? current.$i + delta
                    : delta < 0
                        ? month.children.length - 1
                        : 0;
                var numMonthDays = month.children.length;
                for (var i = startIndex; i >= 0 && i < numMonthDays && i != (delta > 0 ? numMonthDays : -1); i += loopDelta) {
                    var c = month.children[i];
                    if (c.className.indexOf("hidden") === -1 &&
                        isEnabled(c.dateObj) &&
                        Math.abs(current.$i - i) >= Math.abs(delta))
                        return focusOnDayElem(c);
                }
            }
            self.changeMonth(loopDelta);
            focusOnDay(getFirstAvailableDay(loopDelta), 0);
            return undefined;
        }
        function focusOnDay(current, offset) {
            var activeElement = getClosestActiveElement();
            var dayFocused = isInView(activeElement || document.body);
            var startElem = current !== undefined
                ? current
                : dayFocused
                    ? activeElement
                    : self.selectedDateElem !== undefined && isInView(self.selectedDateElem)
                        ? self.selectedDateElem
                        : self.todayDateElem !== undefined && isInView(self.todayDateElem)
                            ? self.todayDateElem
                            : getFirstAvailableDay(offset > 0 ? 1 : -1);
            if (startElem === undefined) {
                self._input.focus();
            }
            else if (!dayFocused) {
                focusOnDayElem(startElem);
            }
            else {
                getNextAvailableDay(startElem, offset);
            }
        }
        function buildMonthDays(year, month) {
            var firstOfMonth = (new Date(year, month, 1).getDay() - self.l10n.firstDayOfWeek + 7) % 7;
            var prevMonthDays = self.utils.getDaysInMonth((month - 1 + 12) % 12, year);
            var daysInMonth = self.utils.getDaysInMonth(month, year), days = window.document.createDocumentFragment(), isMultiMonth = self.config.showMonths > 1, prevMonthDayClass = isMultiMonth ? "prevMonthDay hidden" : "prevMonthDay", nextMonthDayClass = isMultiMonth ? "nextMonthDay hidden" : "nextMonthDay";
            var dayNumber = prevMonthDays + 1 - firstOfMonth, dayIndex = 0;
            // prepend days from the ending of previous month
            for (; dayNumber <= prevMonthDays; dayNumber++, dayIndex++) {
                days.appendChild(createDay("flatpickr-day " + prevMonthDayClass, new Date(year, month - 1, dayNumber), dayNumber, dayIndex));
            }
            // Start at 1 since there is no 0th day
            for (dayNumber = 1; dayNumber <= daysInMonth; dayNumber++, dayIndex++) {
                days.appendChild(createDay("flatpickr-day", new Date(year, month, dayNumber), dayNumber, dayIndex));
            }
            // append days from the next month
            for (var dayNum = daysInMonth + 1; dayNum <= 42 - firstOfMonth &&
            (self.config.showMonths === 1 || dayIndex % 7 !== 0); dayNum++, dayIndex++) {
                days.appendChild(createDay("flatpickr-day " + nextMonthDayClass, new Date(year, month + 1, dayNum % daysInMonth), dayNum, dayIndex));
            }
            //updateNavigationCurrentMonth();
            var dayContainer = createElement("div", "dayContainer");
            dayContainer.appendChild(days);
            return dayContainer;
        }
        function buildDays() {
            if (self.daysContainer === undefined) {
                return;
            }
            clearNode(self.daysContainer);
            // TODO: week numbers for each month
            if (self.weekNumbers)
                clearNode(self.weekNumbers);
            var frag = document.createDocumentFragment();
            for (var i = 0; i < self.config.showMonths; i++) {
                var d = new Date(self.currentYear, self.currentMonth, 1);
                d.setMonth(self.currentMonth + i);
                frag.appendChild(buildMonthDays(d.getFullYear(), d.getMonth()));
            }
            self.daysContainer.appendChild(frag);
            self.days = self.daysContainer.firstChild;
            if (self.config.mode === "range" && self.selectedDates.length === 1) {
                onMouseOver();
            }
        }
        function buildMonthSwitch() {
            if (self.config.showMonths > 1 ||
                self.config.monthSelectorType !== "dropdown")
                return;
            var shouldBuildMonth = function (month) {
                if (self.config.minDate !== undefined &&
                    self.currentYear === self.config.minDate.getFullYear() &&
                    month < self.config.minDate.getMonth()) {
                    return false;
                }
                return !(self.config.maxDate !== undefined &&
                    self.currentYear === self.config.maxDate.getFullYear() &&
                    month > self.config.maxDate.getMonth());
            };
            self.monthsDropdownContainer.tabIndex = -1;
            self.monthsDropdownContainer.innerHTML = "";
            for (var i = 0; i < 12; i++) {
                if (!shouldBuildMonth(i))
                    continue;
                var month = createElement("option", "flatpickr-monthDropdown-month");
                month.value = new Date(self.currentYear, i).getMonth().toString();
                month.textContent = monthToStr(i, self.config.shorthandCurrentMonth, self.l10n);
                month.tabIndex = -1;
                if (self.currentMonth === i) {
                    month.selected = true;
                }
                self.monthsDropdownContainer.appendChild(month);
            }
        }
        function buildMonth() {
            var container = createElement("div", "flatpickr-month");
            var monthNavFragment = window.document.createDocumentFragment();
            var monthElement;
            if (self.config.showMonths > 1 ||
                self.config.monthSelectorType === "static") {
                monthElement = createElement("span", "cur-month");
            }
            else {
                self.monthsDropdownContainer = createElement("select", "flatpickr-monthDropdown-months");
                self.monthsDropdownContainer.setAttribute("aria-label", self.l10n.monthAriaLabel);
                bind(self.monthsDropdownContainer, "change", function (e) {
                    var target = getEventTarget(e);
                    var selectedMonth = parseInt(target.value, 10);
                    self.changeMonth(selectedMonth - self.currentMonth);
                    triggerEvent("onMonthChange");
                });
                buildMonthSwitch();
                monthElement = self.monthsDropdownContainer;
            }
            var yearInput = createNumberInput("cur-year", { tabindex: "-1" });
            var yearElement = yearInput.getElementsByTagName("input")[0];
            yearElement.setAttribute("aria-label", self.l10n.yearAriaLabel);
            if (self.config.minDate) {
                yearElement.setAttribute("min", self.config.minDate.getFullYear().toString());
            }
            if (self.config.maxDate) {
                yearElement.setAttribute("max", self.config.maxDate.getFullYear().toString());
                yearElement.disabled =
                    !!self.config.minDate &&
                    self.config.minDate.getFullYear() === self.config.maxDate.getFullYear();
            }
            var currentMonth = createElement("div", "flatpickr-current-month");
            currentMonth.appendChild(monthElement);
            currentMonth.appendChild(yearInput);
            monthNavFragment.appendChild(currentMonth);
            container.appendChild(monthNavFragment);
            return {
                container: container,
                yearElement: yearElement,
                monthElement: monthElement,
            };
        }
        function buildMonths() {
            clearNode(self.monthNav);
            self.monthNav.appendChild(self.prevMonthNav);
            if (self.config.showMonths) {
                self.yearElements = [];
                self.monthElements = [];
            }
            for (var m = self.config.showMonths; m--;) {
                var month = buildMonth();
                self.yearElements.push(month.yearElement);
                self.monthElements.push(month.monthElement);
                self.monthNav.appendChild(month.container);
            }
            self.monthNav.appendChild(self.nextMonthNav);
        }
        function buildMonthNav() {
            self.monthNav = createElement("div", "flatpickr-months");
            self.yearElements = [];
            self.monthElements = [];
            self.prevMonthNav = createElement("span", "flatpickr-prev-month");
            self.prevMonthNav.innerHTML = self.config.prevArrow;
            self.nextMonthNav = createElement("span", "flatpickr-next-month");
            self.nextMonthNav.innerHTML = self.config.nextArrow;
            buildMonths();
            Object.defineProperty(self, "_hidePrevMonthArrow", {
                get: function () { return self.__hidePrevMonthArrow; },
                set: function (bool) {
                    if (self.__hidePrevMonthArrow !== bool) {
                        toggleClass(self.prevMonthNav, "flatpickr-disabled", bool);
                        self.__hidePrevMonthArrow = bool;
                    }
                },
            });
            Object.defineProperty(self, "_hideNextMonthArrow", {
                get: function () { return self.__hideNextMonthArrow; },
                set: function (bool) {
                    if (self.__hideNextMonthArrow !== bool) {
                        toggleClass(self.nextMonthNav, "flatpickr-disabled", bool);
                        self.__hideNextMonthArrow = bool;
                    }
                },
            });
            self.currentYearElement = self.yearElements[0];
            updateNavigationCurrentMonth();
            return self.monthNav;
        }
        function buildTime() {
            self.calendarContainer.classList.add("hasTime");
            if (self.config.noCalendar)
                self.calendarContainer.classList.add("noCalendar");
            var defaults = getDefaultHours(self.config);
            self.timeContainer = createElement("div", "flatpickr-time");
            self.timeContainer.tabIndex = -1;
            var separator = createElement("span", "flatpickr-time-separator", ":");
            var hourInput = createNumberInput("flatpickr-hour", {
                "aria-label": self.l10n.hourAriaLabel,
            });
            self.hourElement = hourInput.getElementsByTagName("input")[0];
            var minuteInput = createNumberInput("flatpickr-minute", {
                "aria-label": self.l10n.minuteAriaLabel,
            });
            self.minuteElement = minuteInput.getElementsByTagName("input")[0];
            self.hourElement.tabIndex = self.minuteElement.tabIndex = -1;
            self.hourElement.value = pad(self.latestSelectedDateObj
                ? self.latestSelectedDateObj.getHours()
                : self.config.time_24hr
                    ? defaults.hours
                    : military2ampm(defaults.hours));
            self.minuteElement.value = pad(self.latestSelectedDateObj
                ? self.latestSelectedDateObj.getMinutes()
                : defaults.minutes);
            self.hourElement.setAttribute("step", self.config.hourIncrement.toString());
            self.minuteElement.setAttribute("step", self.config.minuteIncrement.toString());
            self.hourElement.setAttribute("min", self.config.time_24hr ? "0" : "1");
            self.hourElement.setAttribute("max", self.config.time_24hr ? "23" : "12");
            self.hourElement.setAttribute("maxlength", "2");
            self.minuteElement.setAttribute("min", "0");
            self.minuteElement.setAttribute("max", "59");
            self.minuteElement.setAttribute("maxlength", "2");
            self.timeContainer.appendChild(hourInput);
            self.timeContainer.appendChild(separator);
            self.timeContainer.appendChild(minuteInput);
            if (self.config.time_24hr)
                self.timeContainer.classList.add("time24hr");
            if (self.config.enableSeconds) {
                self.timeContainer.classList.add("hasSeconds");
                var secondInput = createNumberInput("flatpickr-second");
                self.secondElement = secondInput.getElementsByTagName("input")[0];
                self.secondElement.value = pad(self.latestSelectedDateObj
                    ? self.latestSelectedDateObj.getSeconds()
                    : defaults.seconds);
                self.secondElement.setAttribute("step", self.minuteElement.getAttribute("step"));
                self.secondElement.setAttribute("min", "0");
                self.secondElement.setAttribute("max", "59");
                self.secondElement.setAttribute("maxlength", "2");
                self.timeContainer.appendChild(createElement("span", "flatpickr-time-separator", ":"));
                self.timeContainer.appendChild(secondInput);
            }
            if (!self.config.time_24hr) {
                // add self.amPM if appropriate
                self.amPM = createElement("span", "flatpickr-am-pm", self.l10n.amPM[int((self.latestSelectedDateObj
                    ? self.hourElement.value
                    : self.config.defaultHour) > 11)]);
                self.amPM.title = self.l10n.toggleTitle;
                self.amPM.tabIndex = -1;
                self.timeContainer.appendChild(self.amPM);
            }
            return self.timeContainer;
        }
        function buildWeekdays() {
            if (!self.weekdayContainer)
                self.weekdayContainer = createElement("div", "flatpickr-weekdays");
            else
                clearNode(self.weekdayContainer);
            for (var i = self.config.showMonths; i--;) {
                var container = createElement("div", "flatpickr-weekdaycontainer");
                self.weekdayContainer.appendChild(container);
            }
            updateWeekdays();
            return self.weekdayContainer;
        }
        function updateWeekdays() {
            if (!self.weekdayContainer) {
                return;
            }
            var firstDayOfWeek = self.l10n.firstDayOfWeek;
            var weekdays = __spreadArrays(self.l10n.weekdays.shorthand);
            if (firstDayOfWeek > 0 && firstDayOfWeek < weekdays.length) {
                weekdays = __spreadArrays(weekdays.splice(firstDayOfWeek, weekdays.length), weekdays.splice(0, firstDayOfWeek));
            }
            for (var i = self.config.showMonths; i--;) {
                self.weekdayContainer.children[i].innerHTML = "\n      <span class='flatpickr-weekday'>\n        " + weekdays.join("</span><span class='flatpickr-weekday'>") + "\n      </span>\n      ";
            }
        }
        /* istanbul ignore next */
        function buildWeeks() {
            self.calendarContainer.classList.add("hasWeeks");
            var weekWrapper = createElement("div", "flatpickr-weekwrapper");
            weekWrapper.appendChild(createElement("span", "flatpickr-weekday", self.l10n.weekAbbreviation));
            var weekNumbers = createElement("div", "flatpickr-weeks");
            weekWrapper.appendChild(weekNumbers);
            return {
                weekWrapper: weekWrapper,
                weekNumbers: weekNumbers,
            };
        }
        function changeMonth(value, isOffset) {
            if (isOffset === void 0) { isOffset = true; }
            var delta = isOffset ? value : value - self.currentMonth;
            if ((delta < 0 && self._hidePrevMonthArrow === true) ||
                (delta > 0 && self._hideNextMonthArrow === true))
                return;
            self.currentMonth += delta;
            if (self.currentMonth < 0 || self.currentMonth > 11) {
                self.currentYear += self.currentMonth > 11 ? 1 : -1;
                self.currentMonth = (self.currentMonth + 12) % 12;
                triggerEvent("onYearChange");
                buildMonthSwitch();
            }
            buildDays();
            triggerEvent("onMonthChange");
            updateNavigationCurrentMonth();
        }
        function clear(triggerChangeEvent, toInitial) {
            if (triggerChangeEvent === void 0) { triggerChangeEvent = true; }
            if (toInitial === void 0) { toInitial = true; }
            self.input.value = "";
            if (self.altInput !== undefined)
                self.altInput.value = "";
            if (self.mobileInput !== undefined)
                self.mobileInput.value = "";
            self.selectedDates = [];
            self.latestSelectedDateObj = undefined;
            if (toInitial === true) {
                self.currentYear = self._initialDate.getFullYear();
                self.currentMonth = self._initialDate.getMonth();
            }
            if (self.config.enableTime === true) {
                var _a = getDefaultHours(self.config), hours = _a.hours, minutes = _a.minutes, seconds = _a.seconds;
                setHours(hours, minutes, seconds);
            }
            self.redraw();
            if (triggerChangeEvent)
                // triggerChangeEvent is true (default) or an Event
                triggerEvent("onChange");
        }
        function close() {
            self.isOpen = false;
            if (!self.isMobile) {
                if (self.calendarContainer !== undefined) {
                    self.calendarContainer.classList.remove("open");
                }
                if (self._input !== undefined) {
                    self._input.classList.remove("active");
                }
            }
            triggerEvent("onClose");
        }
        function destroy() {
            if (self.config !== undefined)
                triggerEvent("onDestroy");
            for (var i = self._handlers.length; i--;) {
                self._handlers[i].remove();
            }
            self._handlers = [];
            if (self.mobileInput) {
                if (self.mobileInput.parentNode)
                    self.mobileInput.parentNode.removeChild(self.mobileInput);
                self.mobileInput = undefined;
            }
            else if (self.calendarContainer && self.calendarContainer.parentNode) {
                if (self.config.static && self.calendarContainer.parentNode) {
                    var wrapper = self.calendarContainer.parentNode;
                    wrapper.lastChild && wrapper.removeChild(wrapper.lastChild);
                    if (wrapper.parentNode) {
                        while (wrapper.firstChild)
                            wrapper.parentNode.insertBefore(wrapper.firstChild, wrapper);
                        wrapper.parentNode.removeChild(wrapper);
                    }
                }
                else
                    self.calendarContainer.parentNode.removeChild(self.calendarContainer);
            }
            if (self.altInput) {
                self.input.type = "text";
                if (self.altInput.parentNode)
                    self.altInput.parentNode.removeChild(self.altInput);
                delete self.altInput;
            }
            if (self.input) {
                self.input.type = self.input._type;
                self.input.classList.remove("flatpickr-input");
                self.input.removeAttribute("readonly");
            }
            [
                "_showTimeInput",
                "latestSelectedDateObj",
                "_hideNextMonthArrow",
                "_hidePrevMonthArrow",
                "__hideNextMonthArrow",
                "__hidePrevMonthArrow",
                "isMobile",
                "isOpen",
                "selectedDateElem",
                "minDateHasTime",
                "maxDateHasTime",
                "days",
                "daysContainer",
                "_input",
                "_positionElement",
                "innerContainer",
                "rContainer",
                "monthNav",
                "todayDateElem",
                "calendarContainer",
                "weekdayContainer",
                "prevMonthNav",
                "nextMonthNav",
                "monthsDropdownContainer",
                "currentMonthElement",
                "currentYearElement",
                "navigationCurrentMonth",
                "selectedDateElem",
                "config",
            ].forEach(function (k) {
                try {
                    delete self[k];
                }
                catch (_) { }
            });
        }
        function isCalendarElem(elem) {
            return self.calendarContainer.contains(elem);
        }
        function documentClick(e) {
            if (self.isOpen && !self.config.inline) {
                var eventTarget_1 = getEventTarget(e);
                var isCalendarElement = isCalendarElem(eventTarget_1);
                var isInput = eventTarget_1 === self.input ||
                    eventTarget_1 === self.altInput ||
                    self.element.contains(eventTarget_1) ||
                    // web components
                    // e.path is not present in all browsers. circumventing typechecks
                    (e.path &&
                        e.path.indexOf &&
                        (~e.path.indexOf(self.input) ||
                            ~e.path.indexOf(self.altInput)));
                var lostFocus = !isInput &&
                    !isCalendarElement &&
                    !isCalendarElem(e.relatedTarget);
                var isIgnored = !self.config.ignoredFocusElements.some(function (elem) {
                    return elem.contains(eventTarget_1);
                });
                if (lostFocus && isIgnored) {
                    if (self.config.allowInput) {
                        self.setDate(self._input.value, false, self.config.altInput
                            ? self.config.altFormat
                            : self.config.dateFormat);
                    }
                    if (self.timeContainer !== undefined &&
                        self.minuteElement !== undefined &&
                        self.hourElement !== undefined &&
                        self.input.value !== "" &&
                        self.input.value !== undefined) {
                        updateTime();
                    }
                    self.close();
                    if (self.config &&
                        self.config.mode === "range" &&
                        self.selectedDates.length === 1)
                        self.clear(false);
                }
            }
        }
        function changeYear(newYear) {
            if (!newYear ||
                (self.config.minDate && newYear < self.config.minDate.getFullYear()) ||
                (self.config.maxDate && newYear > self.config.maxDate.getFullYear()))
                return;
            var newYearNum = newYear, isNewYear = self.currentYear !== newYearNum;
            self.currentYear = newYearNum || self.currentYear;
            if (self.config.maxDate &&
                self.currentYear === self.config.maxDate.getFullYear()) {
                self.currentMonth = Math.min(self.config.maxDate.getMonth(), self.currentMonth);
            }
            else if (self.config.minDate &&
                self.currentYear === self.config.minDate.getFullYear()) {
                self.currentMonth = Math.max(self.config.minDate.getMonth(), self.currentMonth);
            }
            if (isNewYear) {
                self.redraw();
                triggerEvent("onYearChange");
                buildMonthSwitch();
            }
        }
        function isEnabled(date, timeless) {
            var _a;
            if (timeless === void 0) { timeless = true; }
            var dateToCheck = self.parseDate(date, undefined, timeless); // timeless
            if ((self.config.minDate &&
                    dateToCheck &&
                    compareDates(dateToCheck, self.config.minDate, timeless !== undefined ? timeless : !self.minDateHasTime) < 0) ||
                (self.config.maxDate &&
                    dateToCheck &&
                    compareDates(dateToCheck, self.config.maxDate, timeless !== undefined ? timeless : !self.maxDateHasTime) > 0))
                return false;
            if (!self.config.enable && self.config.disable.length === 0)
                return true;
            if (dateToCheck === undefined)
                return false;
            var bool = !!self.config.enable, array = (_a = self.config.enable) !== null && _a !== void 0 ? _a : self.config.disable;
            for (var i = 0, d = void 0; i < array.length; i++) {
                d = array[i];
                if (typeof d === "function" &&
                    d(dateToCheck) // disabled by function
                )
                    return bool;
                else if (d instanceof Date &&
                    dateToCheck !== undefined &&
                    d.getTime() === dateToCheck.getTime())
                    // disabled by date
                    return bool;
                else if (typeof d === "string") {
                    // disabled by date string
                    var parsed = self.parseDate(d, undefined, true);
                    return parsed && parsed.getTime() === dateToCheck.getTime()
                        ? bool
                        : !bool;
                }
                else if (
                    // disabled by range
                    typeof d === "object" &&
                    dateToCheck !== undefined &&
                    d.from &&
                    d.to &&
                    dateToCheck.getTime() >= d.from.getTime() &&
                    dateToCheck.getTime() <= d.to.getTime())
                    return bool;
            }
            return !bool;
        }
        function isInView(elem) {
            if (self.daysContainer !== undefined)
                return (elem.className.indexOf("hidden") === -1 &&
                    elem.className.indexOf("flatpickr-disabled") === -1 &&
                    self.daysContainer.contains(elem));
            return false;
        }
        function onBlur(e) {
            var isInput = e.target === self._input;
            var valueChanged = self._input.value.trimEnd() !== getDateStr();
            if (isInput &&
                valueChanged &&
                !(e.relatedTarget && isCalendarElem(e.relatedTarget))) {
                self.setDate(self._input.value, true, e.target === self.altInput
                    ? self.config.altFormat
                    : self.config.dateFormat);
            }
        }
        function onKeyDown(e) {
            // e.key                      e.keyCode
            // "Backspace"                        8
            // "Tab"                              9
            // "Enter"                           13
            // "Escape"     (IE "Esc")           27
            // "ArrowLeft"  (IE "Left")          37
            // "ArrowUp"    (IE "Up")            38
            // "ArrowRight" (IE "Right")         39
            // "ArrowDown"  (IE "Down")          40
            // "Delete"     (IE "Del")           46
            var eventTarget = getEventTarget(e);
            var isInput = self.config.wrap
                ? element.contains(eventTarget)
                : eventTarget === self._input;
            var allowInput = self.config.allowInput;
            var allowKeydown = self.isOpen && (!allowInput || !isInput);
            var allowInlineKeydown = self.config.inline && isInput && !allowInput;
            if (e.keyCode === 13 && isInput) {
                if (allowInput) {
                    self.setDate(self._input.value, true, eventTarget === self.altInput
                        ? self.config.altFormat
                        : self.config.dateFormat);
                    self.close();
                    return eventTarget.blur();
                }
                else {
                    self.open();
                }
            }
            else if (isCalendarElem(eventTarget) ||
                allowKeydown ||
                allowInlineKeydown) {
                var isTimeObj = !!self.timeContainer &&
                    self.timeContainer.contains(eventTarget);
                switch (e.keyCode) {
                    case 13:
                        if (isTimeObj) {
                            e.preventDefault();
                            updateTime();
                            focusAndClose();
                        }
                        else
                            selectDate(e);
                        break;
                    case 27: // escape
                        e.preventDefault();
                        focusAndClose();
                        break;
                    case 8:
                    case 46:
                        if (isInput && !self.config.allowInput) {
                            e.preventDefault();
                            self.clear();
                        }
                        break;
                    case 37:
                    case 39:
                        if (!isTimeObj && !isInput) {
                            e.preventDefault();
                            var activeElement = getClosestActiveElement();
                            if (self.daysContainer !== undefined &&
                                (allowInput === false ||
                                    (activeElement && isInView(activeElement)))) {
                                var delta_1 = e.keyCode === 39 ? 1 : -1;
                                if (!e.ctrlKey)
                                    focusOnDay(undefined, delta_1);
                                else {
                                    e.stopPropagation();
                                    changeMonth(delta_1);
                                    focusOnDay(getFirstAvailableDay(1), 0);
                                }
                            }
                        }
                        else if (self.hourElement)
                            self.hourElement.focus();
                        break;
                    case 38:
                    case 40:
                        e.preventDefault();
                        var delta = e.keyCode === 40 ? 1 : -1;
                        if ((self.daysContainer &&
                                eventTarget.$i !== undefined) ||
                            eventTarget === self.input ||
                            eventTarget === self.altInput) {
                            if (e.ctrlKey) {
                                e.stopPropagation();
                                changeYear(self.currentYear - delta);
                                focusOnDay(getFirstAvailableDay(1), 0);
                            }
                            else if (!isTimeObj)
                                focusOnDay(undefined, delta * 7);
                        }
                        else if (eventTarget === self.currentYearElement) {
                            changeYear(self.currentYear - delta);
                        }
                        else if (self.config.enableTime) {
                            if (!isTimeObj && self.hourElement)
                                self.hourElement.focus();
                            updateTime(e);
                            self._debouncedChange();
                        }
                        break;
                    case 9:
                        if (isTimeObj) {
                            var elems = [
                                self.hourElement,
                                self.minuteElement,
                                self.secondElement,
                                self.amPM,
                            ]
                                .concat(self.pluginElements)
                                .filter(function (x) { return x; });
                            var i = elems.indexOf(eventTarget);
                            if (i !== -1) {
                                var target = elems[i + (e.shiftKey ? -1 : 1)];
                                e.preventDefault();
                                (target || self._input).focus();
                            }
                        }
                        else if (!self.config.noCalendar &&
                            self.daysContainer &&
                            self.daysContainer.contains(eventTarget) &&
                            e.shiftKey) {
                            e.preventDefault();
                            self._input.focus();
                        }
                        break;
                }
            }
            if (self.amPM !== undefined && eventTarget === self.amPM) {
                switch (e.key) {
                    case self.l10n.amPM[0].charAt(0):
                    case self.l10n.amPM[0].charAt(0).toLowerCase():
                        self.amPM.textContent = self.l10n.amPM[0];
                        setHoursFromInputs();
                        updateValue();
                        break;
                    case self.l10n.amPM[1].charAt(0):
                    case self.l10n.amPM[1].charAt(0).toLowerCase():
                        self.amPM.textContent = self.l10n.amPM[1];
                        setHoursFromInputs();
                        updateValue();
                        break;
                }
            }
            if (isInput || isCalendarElem(eventTarget)) {
                triggerEvent("onKeyDown", e);
            }
        }
        function onMouseOver(elem, cellClass) {
            if (cellClass === void 0) { cellClass = "flatpickr-day"; }
            if (self.selectedDates.length !== 1 ||
                (elem &&
                    (!elem.classList.contains(cellClass) ||
                        elem.classList.contains("flatpickr-disabled"))))
                return;
            var hoverDate = elem
                ? elem.dateObj.getTime()
                : self.days.firstElementChild.dateObj.getTime(), initialDate = self.parseDate(self.selectedDates[0], undefined, true).getTime(), rangeStartDate = Math.min(hoverDate, self.selectedDates[0].getTime()), rangeEndDate = Math.max(hoverDate, self.selectedDates[0].getTime());
            var containsDisabled = false;
            var minRange = 0, maxRange = 0;
            for (var t = rangeStartDate; t < rangeEndDate; t += duration.DAY) {
                if (!isEnabled(new Date(t), true)) {
                    containsDisabled =
                        containsDisabled || (t > rangeStartDate && t < rangeEndDate);
                    if (t < initialDate && (!minRange || t > minRange))
                        minRange = t;
                    else if (t > initialDate && (!maxRange || t < maxRange))
                        maxRange = t;
                }
            }
            var hoverableCells = Array.from(self.rContainer.querySelectorAll("*:nth-child(-n+" + self.config.showMonths + ") > ." + cellClass));
            hoverableCells.forEach(function (dayElem) {
                var date = dayElem.dateObj;
                var timestamp = date.getTime();
                var outOfRange = (minRange > 0 && timestamp < minRange) ||
                    (maxRange > 0 && timestamp > maxRange);
                if (outOfRange) {
                    dayElem.classList.add("notAllowed");
                    ["inRange", "startRange", "endRange"].forEach(function (c) {
                        dayElem.classList.remove(c);
                    });
                    return;
                }
                else if (containsDisabled && !outOfRange)
                    return;
                ["startRange", "inRange", "endRange", "notAllowed"].forEach(function (c) {
                    dayElem.classList.remove(c);
                });
                if (elem !== undefined) {
                    elem.classList.add(hoverDate <= self.selectedDates[0].getTime()
                        ? "startRange"
                        : "endRange");
                    if (initialDate < hoverDate && timestamp === initialDate)
                        dayElem.classList.add("startRange");
                    else if (initialDate > hoverDate && timestamp === initialDate)
                        dayElem.classList.add("endRange");
                    if (timestamp >= minRange &&
                        (maxRange === 0 || timestamp <= maxRange) &&
                        isBetween(timestamp, initialDate, hoverDate))
                        dayElem.classList.add("inRange");
                }
            });
        }
        function onResize() {
            if (self.isOpen && !self.config.static && !self.config.inline)
                positionCalendar();
        }
        function open(e, positionElement) {
            if (positionElement === void 0) { positionElement = self._positionElement; }
            if (self.isMobile === true) {
                if (e) {
                    e.preventDefault();
                    var eventTarget = getEventTarget(e);
                    if (eventTarget) {
                        eventTarget.blur();
                    }
                }
                if (self.mobileInput !== undefined) {
                    self.mobileInput.focus();
                    self.mobileInput.click();
                }
                triggerEvent("onOpen");
                return;
            }
            else if (self._input.disabled || self.config.inline) {
                return;
            }
            var wasOpen = self.isOpen;
            self.isOpen = true;
            if (!wasOpen) {
                self.calendarContainer.classList.add("open");
                self._input.classList.add("active");
                triggerEvent("onOpen");
                positionCalendar(positionElement);
            }
            if (self.config.enableTime === true && self.config.noCalendar === true) {
                if (self.config.allowInput === false &&
                    (e === undefined ||
                        !self.timeContainer.contains(e.relatedTarget))) {
                    setTimeout(function () { return self.hourElement.select(); }, 50);
                }
            }
        }
        function minMaxDateSetter(type) {
            return function (date) {
                var dateObj = (self.config["_" + type + "Date"] = self.parseDate(date, self.config.dateFormat));
                var inverseDateObj = self.config["_" + (type === "min" ? "max" : "min") + "Date"];
                if (dateObj !== undefined) {
                    self[type === "min" ? "minDateHasTime" : "maxDateHasTime"] =
                        dateObj.getHours() > 0 ||
                        dateObj.getMinutes() > 0 ||
                        dateObj.getSeconds() > 0;
                }
                if (self.selectedDates) {
                    self.selectedDates = self.selectedDates.filter(function (d) { return isEnabled(d); });
                    if (!self.selectedDates.length && type === "min")
                        setHoursFromDate(dateObj);
                    updateValue();
                }
                if (self.daysContainer) {
                    redraw();
                    if (dateObj !== undefined)
                        self.currentYearElement[type] = dateObj.getFullYear().toString();
                    else
                        self.currentYearElement.removeAttribute(type);
                    self.currentYearElement.disabled =
                        !!inverseDateObj &&
                        dateObj !== undefined &&
                        inverseDateObj.getFullYear() === dateObj.getFullYear();
                }
            };
        }
        function parseConfig() {
            var boolOpts = [
                "wrap",
                "weekNumbers",
                "allowInput",
                "allowInvalidPreload",
                "clickOpens",
                "time_24hr",
                "enableTime",
                "noCalendar",
                "altInput",
                "shorthandCurrentMonth",
                "inline",
                "static",
                "enableSeconds",
                "disableMobile",
            ];
            var userConfig = __assign(__assign({}, JSON.parse(JSON.stringify(element.dataset || {}))), instanceConfig);
            var formats = {};
            self.config.parseDate = userConfig.parseDate;
            self.config.formatDate = userConfig.formatDate;
            Object.defineProperty(self.config, "enable", {
                get: function () { return self.config._enable; },
                set: function (dates) {
                    self.config._enable = parseDateRules(dates);
                },
            });
            Object.defineProperty(self.config, "disable", {
                get: function () { return self.config._disable; },
                set: function (dates) {
                    self.config._disable = parseDateRules(dates);
                },
            });
            var timeMode = userConfig.mode === "time";
            if (!userConfig.dateFormat && (userConfig.enableTime || timeMode)) {
                var defaultDateFormat = flatpickr.defaultConfig.dateFormat || defaults.dateFormat;
                formats.dateFormat =
                    userConfig.noCalendar || timeMode
                        ? "H:i" + (userConfig.enableSeconds ? ":S" : "")
                        : defaultDateFormat + " H:i" + (userConfig.enableSeconds ? ":S" : "");
            }
            if (userConfig.altInput &&
                (userConfig.enableTime || timeMode) &&
                !userConfig.altFormat) {
                var defaultAltFormat = flatpickr.defaultConfig.altFormat || defaults.altFormat;
                formats.altFormat =
                    userConfig.noCalendar || timeMode
                        ? "h:i" + (userConfig.enableSeconds ? ":S K" : " K")
                        : defaultAltFormat + (" h:i" + (userConfig.enableSeconds ? ":S" : "") + " K");
            }
            Object.defineProperty(self.config, "minDate", {
                get: function () { return self.config._minDate; },
                set: minMaxDateSetter("min"),
            });
            Object.defineProperty(self.config, "maxDate", {
                get: function () { return self.config._maxDate; },
                set: minMaxDateSetter("max"),
            });
            var minMaxTimeSetter = function (type) { return function (val) {
                self.config[type === "min" ? "_minTime" : "_maxTime"] = self.parseDate(val, "H:i:S");
            }; };
            Object.defineProperty(self.config, "minTime", {
                get: function () { return self.config._minTime; },
                set: minMaxTimeSetter("min"),
            });
            Object.defineProperty(self.config, "maxTime", {
                get: function () { return self.config._maxTime; },
                set: minMaxTimeSetter("max"),
            });
            if (userConfig.mode === "time") {
                self.config.noCalendar = true;
                self.config.enableTime = true;
            }
            Object.assign(self.config, formats, userConfig);
            for (var i = 0; i < boolOpts.length; i++)
                // https://github.com/microsoft/TypeScript/issues/31663
                self.config[boolOpts[i]] =
                    self.config[boolOpts[i]] === true ||
                    self.config[boolOpts[i]] === "true";
            HOOKS.filter(function (hook) { return self.config[hook] !== undefined; }).forEach(function (hook) {
                self.config[hook] = arrayify(self.config[hook] || []).map(bindToInstance);
            });
            self.isMobile =
                !self.config.disableMobile &&
                !self.config.inline &&
                self.config.mode === "single" &&
                !self.config.disable.length &&
                !self.config.enable &&
                !self.config.weekNumbers &&
                /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
            for (var i = 0; i < self.config.plugins.length; i++) {
                var pluginConf = self.config.plugins[i](self) || {};
                for (var key in pluginConf) {
                    if (HOOKS.indexOf(key) > -1) {
                        self.config[key] = arrayify(pluginConf[key])
                            .map(bindToInstance)
                            .concat(self.config[key]);
                    }
                    else if (typeof userConfig[key] === "undefined")
                        self.config[key] = pluginConf[key];
                }
            }
            if (!userConfig.altInputClass) {
                self.config.altInputClass =
                    getInputElem().className + " " + self.config.altInputClass;
            }
            triggerEvent("onParseConfig");
        }
        function getInputElem() {
            return self.config.wrap
                ? element.querySelector("[data-input]")
                : element;
        }
        function setupLocale() {
            if (typeof self.config.locale !== "object" &&
                typeof flatpickr.l10ns[self.config.locale] === "undefined")
                self.config.errorHandler(new Error("flatpickr: invalid locale " + self.config.locale));
            self.l10n = __assign(__assign({}, flatpickr.l10ns.default), (typeof self.config.locale === "object"
                ? self.config.locale
                : self.config.locale !== "default"
                    ? flatpickr.l10ns[self.config.locale]
                    : undefined));
            tokenRegex.D = "(" + self.l10n.weekdays.shorthand.join("|") + ")";
            tokenRegex.l = "(" + self.l10n.weekdays.longhand.join("|") + ")";
            tokenRegex.M = "(" + self.l10n.months.shorthand.join("|") + ")";
            tokenRegex.F = "(" + self.l10n.months.longhand.join("|") + ")";
            tokenRegex.K = "(" + self.l10n.amPM[0] + "|" + self.l10n.amPM[1] + "|" + self.l10n.amPM[0].toLowerCase() + "|" + self.l10n.amPM[1].toLowerCase() + ")";
            var userConfig = __assign(__assign({}, instanceConfig), JSON.parse(JSON.stringify(element.dataset || {})));
            if (userConfig.time_24hr === undefined &&
                flatpickr.defaultConfig.time_24hr === undefined) {
                self.config.time_24hr = self.l10n.time_24hr;
            }
            self.formatDate = createDateFormatter(self);
            self.parseDate = createDateParser({ config: self.config, l10n: self.l10n });
        }
        function positionCalendar(customPositionElement) {
            if (typeof self.config.position === "function") {
                return void self.config.position(self, customPositionElement);
            }
            if (self.calendarContainer === undefined)
                return;
            triggerEvent("onPreCalendarPosition");
            var positionElement = customPositionElement || self._positionElement;
            var calendarHeight = Array.prototype.reduce.call(self.calendarContainer.children, (function (acc, child) { return acc + child.offsetHeight; }), 0), calendarWidth = self.calendarContainer.offsetWidth, configPos = self.config.position.split(" "), configPosVertical = configPos[0], configPosHorizontal = configPos.length > 1 ? configPos[1] : null, inputBounds = positionElement.getBoundingClientRect(), distanceFromBottom = window.innerHeight - inputBounds.bottom, showOnTop = configPosVertical === "above" ||
                (configPosVertical !== "below" &&
                    distanceFromBottom < calendarHeight &&
                    inputBounds.top > calendarHeight);
            var top = window.pageYOffset +
                inputBounds.top +
                (!showOnTop ? positionElement.offsetHeight + 2 : -calendarHeight - 2);
            toggleClass(self.calendarContainer, "arrowTop", !showOnTop);
            toggleClass(self.calendarContainer, "arrowBottom", showOnTop);
            if (self.config.inline)
                return;
            var left = window.pageXOffset + inputBounds.left;
            var isCenter = false;
            var isRight = false;
            if (configPosHorizontal === "center") {
                left -= (calendarWidth - inputBounds.width) / 2;
                isCenter = true;
            }
            else if (configPosHorizontal === "right") {
                left -= calendarWidth - inputBounds.width;
                isRight = true;
            }
            toggleClass(self.calendarContainer, "arrowLeft", !isCenter && !isRight);
            toggleClass(self.calendarContainer, "arrowCenter", isCenter);
            toggleClass(self.calendarContainer, "arrowRight", isRight);
            var right = window.document.body.offsetWidth -
                (window.pageXOffset + inputBounds.right);
            var rightMost = left + calendarWidth > window.document.body.offsetWidth;
            var centerMost = right + calendarWidth > window.document.body.offsetWidth;
            toggleClass(self.calendarContainer, "rightMost", rightMost);
            if (self.config.static)
                return;
            self.calendarContainer.style.top = top + "px";
            if (!rightMost) {
                self.calendarContainer.style.left = left + "px";
                self.calendarContainer.style.right = "auto";
            }
            else if (!centerMost) {
                self.calendarContainer.style.left = "auto";
                self.calendarContainer.style.right = right + "px";
            }
            else {
                var doc = getDocumentStyleSheet();
                // some testing environments don't have css support
                if (doc === undefined)
                    return;
                var bodyWidth = window.document.body.offsetWidth;
                var centerLeft = Math.max(0, bodyWidth / 2 - calendarWidth / 2);
                var centerBefore = ".flatpickr-calendar.centerMost:before";
                var centerAfter = ".flatpickr-calendar.centerMost:after";
                var centerIndex = doc.cssRules.length;
                var centerStyle = "{left:" + inputBounds.left + "px;right:auto;}";
                toggleClass(self.calendarContainer, "rightMost", false);
                toggleClass(self.calendarContainer, "centerMost", true);
                doc.insertRule(centerBefore + "," + centerAfter + centerStyle, centerIndex);
                self.calendarContainer.style.left = centerLeft + "px";
                self.calendarContainer.style.right = "auto";
            }
        }
        function getDocumentStyleSheet() {
            var editableSheet = null;
            for (var i = 0; i < document.styleSheets.length; i++) {
                var sheet = document.styleSheets[i];
                if (!sheet.cssRules)
                    continue;
                try {
                    sheet.cssRules;
                }
                catch (err) {
                    continue;
                }
                editableSheet = sheet;
                break;
            }
            return editableSheet != null ? editableSheet : createStyleSheet();
        }
        function createStyleSheet() {
            var style = document.createElement("style");
            document.head.appendChild(style);
            return style.sheet;
        }
        function redraw() {
            if (self.config.noCalendar || self.isMobile)
                return;
            buildMonthSwitch();
            updateNavigationCurrentMonth();
            buildDays();
        }
        function focusAndClose() {
            self._input.focus();
            if (window.navigator.userAgent.indexOf("MSIE") !== -1 ||
                navigator.msMaxTouchPoints !== undefined) {
                // hack - bugs in the way IE handles focus keeps the calendar open
                setTimeout(self.close, 0);
            }
            else {
                self.close();
            }
        }
        function selectDate(e) {
            e.preventDefault();
            e.stopPropagation();
            var isSelectable = function (day) {
                return day.classList &&
                    day.classList.contains("flatpickr-day") &&
                    !day.classList.contains("flatpickr-disabled") &&
                    !day.classList.contains("notAllowed");
            };
            var t = findParent(getEventTarget(e), isSelectable);
            if (t === undefined)
                return;
            var target = t;
            var selectedDate = (self.latestSelectedDateObj = new Date(target.dateObj.getTime()));
            var shouldChangeMonth = (selectedDate.getMonth() < self.currentMonth ||
                    selectedDate.getMonth() >
                    self.currentMonth + self.config.showMonths - 1) &&
                self.config.mode !== "range";
            self.selectedDateElem = target;
            if (self.config.mode === "single")
                self.selectedDates = [selectedDate];
            else if (self.config.mode === "multiple") {
                var selectedIndex = isDateSelected(selectedDate);
                if (selectedIndex)
                    self.selectedDates.splice(parseInt(selectedIndex), 1);
                else
                    self.selectedDates.push(selectedDate);
            }
            else if (self.config.mode === "range") {
                if (self.selectedDates.length === 2) {
                    self.clear(false, false);
                }
                self.latestSelectedDateObj = selectedDate;
                self.selectedDates.push(selectedDate);
                // unless selecting same date twice, sort ascendingly
                if (compareDates(selectedDate, self.selectedDates[0], true) !== 0)
                    self.selectedDates.sort(function (a, b) { return a.getTime() - b.getTime(); });
            }
            setHoursFromInputs();
            if (shouldChangeMonth) {
                var isNewYear = self.currentYear !== selectedDate.getFullYear();
                self.currentYear = selectedDate.getFullYear();
                self.currentMonth = selectedDate.getMonth();
                if (isNewYear) {
                    triggerEvent("onYearChange");
                    buildMonthSwitch();
                }
                triggerEvent("onMonthChange");
            }
            updateNavigationCurrentMonth();
            buildDays();
            updateValue();
            // maintain focus
            if (!shouldChangeMonth &&
                self.config.mode !== "range" &&
                self.config.showMonths === 1)
                focusOnDayElem(target);
            else if (self.selectedDateElem !== undefined &&
                self.hourElement === undefined) {
                self.selectedDateElem && self.selectedDateElem.focus();
            }
            if (self.hourElement !== undefined)
                self.hourElement !== undefined && self.hourElement.focus();
            if (self.config.closeOnSelect) {
                var single = self.config.mode === "single" && !self.config.enableTime;
                var range = self.config.mode === "range" &&
                    self.selectedDates.length === 2 &&
                    !self.config.enableTime;
                if (single || range) {
                    focusAndClose();
                }
            }
            triggerChange();
        }
        var CALLBACKS = {
            locale: [setupLocale, updateWeekdays],
            showMonths: [buildMonths, setCalendarWidth, buildWeekdays],
            minDate: [jumpToDate],
            maxDate: [jumpToDate],
            positionElement: [updatePositionElement],
            clickOpens: [
                function () {
                    if (self.config.clickOpens === true) {
                        bind(self._input, "focus", self.open);
                        bind(self._input, "click", self.open);
                    }
                    else {
                        self._input.removeEventListener("focus", self.open);
                        self._input.removeEventListener("click", self.open);
                    }
                },
            ],
        };
        function set(option, value) {
            if (option !== null && typeof option === "object") {
                Object.assign(self.config, option);
                for (var key in option) {
                    if (CALLBACKS[key] !== undefined)
                        CALLBACKS[key].forEach(function (x) { return x(); });
                }
            }
            else {
                self.config[option] = value;
                if (CALLBACKS[option] !== undefined)
                    CALLBACKS[option].forEach(function (x) { return x(); });
                else if (HOOKS.indexOf(option) > -1)
                    self.config[option] = arrayify(value);
            }
            self.redraw();
            updateValue(true);
        }
        function setSelectedDate(inputDate, format) {
            var dates = [];
            if (inputDate instanceof Array)
                dates = inputDate.map(function (d) { return self.parseDate(d, format); });
            else if (inputDate instanceof Date || typeof inputDate === "number")
                dates = [self.parseDate(inputDate, format)];
            else if (typeof inputDate === "string") {
                switch (self.config.mode) {
                    case "single":
                    case "time":
                        dates = [self.parseDate(inputDate, format)];
                        break;
                    case "multiple":
                        dates = inputDate
                            .split(self.config.conjunction)
                            .map(function (date) { return self.parseDate(date, format); });
                        break;
                    case "range":
                        dates = inputDate
                            .split(self.l10n.rangeSeparator)
                            .map(function (date) { return self.parseDate(date, format); });
                        break;
                }
            }
            else
                self.config.errorHandler(new Error("Invalid date supplied: " + JSON.stringify(inputDate)));
            self.selectedDates = (self.config.allowInvalidPreload
                ? dates
                : dates.filter(function (d) { return d instanceof Date && isEnabled(d, false); }));
            if (self.config.mode === "range")
                self.selectedDates.sort(function (a, b) { return a.getTime() - b.getTime(); });
        }
        function setDate(date, triggerChange, format) {
            if (triggerChange === void 0) { triggerChange = false; }
            if (format === void 0) { format = self.config.dateFormat; }
            if ((date !== 0 && !date) || (date instanceof Array && date.length === 0))
                return self.clear(triggerChange);
            setSelectedDate(date, format);
            self.latestSelectedDateObj =
                self.selectedDates[self.selectedDates.length - 1];
            self.redraw();
            jumpToDate(undefined, triggerChange);
            setHoursFromDate();
            if (self.selectedDates.length === 0) {
                self.clear(false);
            }
            updateValue(triggerChange);
            if (triggerChange)
                triggerEvent("onChange");
        }
        function parseDateRules(arr) {
            return arr
                .slice()
                .map(function (rule) {
                    if (typeof rule === "string" ||
                        typeof rule === "number" ||
                        rule instanceof Date) {
                        return self.parseDate(rule, undefined, true);
                    }
                    else if (rule &&
                        typeof rule === "object" &&
                        rule.from &&
                        rule.to)
                        return {
                            from: self.parseDate(rule.from, undefined),
                            to: self.parseDate(rule.to, undefined),
                        };
                    return rule;
                })
                .filter(function (x) { return x; }); // remove falsy values
        }
        function setupDates() {
            self.selectedDates = [];
            self.now = self.parseDate(self.config.now) || new Date();
            // Workaround IE11 setting placeholder as the input's value
            var preloadedDate = self.config.defaultDate ||
                ((self.input.nodeName === "INPUT" ||
                    self.input.nodeName === "TEXTAREA") &&
                self.input.placeholder &&
                self.input.value === self.input.placeholder
                    ? null
                    : self.input.value);
            if (preloadedDate)
                setSelectedDate(preloadedDate, self.config.dateFormat);
            self._initialDate =
                self.selectedDates.length > 0
                    ? self.selectedDates[0]
                    : self.config.minDate &&
                    self.config.minDate.getTime() > self.now.getTime()
                        ? self.config.minDate
                        : self.config.maxDate &&
                        self.config.maxDate.getTime() < self.now.getTime()
                            ? self.config.maxDate
                            : self.now;
            self.currentYear = self._initialDate.getFullYear();
            self.currentMonth = self._initialDate.getMonth();
            if (self.selectedDates.length > 0)
                self.latestSelectedDateObj = self.selectedDates[0];
            if (self.config.minTime !== undefined)
                self.config.minTime = self.parseDate(self.config.minTime, "H:i");
            if (self.config.maxTime !== undefined)
                self.config.maxTime = self.parseDate(self.config.maxTime, "H:i");
            self.minDateHasTime =
                !!self.config.minDate &&
                (self.config.minDate.getHours() > 0 ||
                    self.config.minDate.getMinutes() > 0 ||
                    self.config.minDate.getSeconds() > 0);
            self.maxDateHasTime =
                !!self.config.maxDate &&
                (self.config.maxDate.getHours() > 0 ||
                    self.config.maxDate.getMinutes() > 0 ||
                    self.config.maxDate.getSeconds() > 0);
        }
        function setupInputs() {
            self.input = getInputElem();
            /* istanbul ignore next */
            if (!self.input) {
                self.config.errorHandler(new Error("Invalid input element specified"));
                return;
            }
            // hack: store previous type to restore it after destroy()
            self.input._type = self.input.type;
            self.input.type = "text";
            self.input.classList.add("flatpickr-input");
            self._input = self.input;
            if (self.config.altInput) {
                // replicate self.element
                self.altInput = createElement(self.input.nodeName, self.config.altInputClass);
                self._input = self.altInput;
                self.altInput.placeholder = self.input.placeholder;
                self.altInput.disabled = self.input.disabled;
                self.altInput.required = self.input.required;
                self.altInput.tabIndex = self.input.tabIndex;
                self.altInput.type = "text";
                self.input.setAttribute("type", "hidden");
                if (!self.config.static && self.input.parentNode)
                    self.input.parentNode.insertBefore(self.altInput, self.input.nextSibling);
            }
            if (!self.config.allowInput)
                self._input.setAttribute("readonly", "readonly");
            updatePositionElement();
        }
        function updatePositionElement() {
            self._positionElement = self.config.positionElement || self._input;
        }
        function setupMobile() {
            var inputType = self.config.enableTime
                ? self.config.noCalendar
                    ? "time"
                    : "datetime-local"
                : "date";
            self.mobileInput = createElement("input", self.input.className + " flatpickr-mobile");
            self.mobileInput.tabIndex = 1;
            self.mobileInput.type = inputType;
            self.mobileInput.disabled = self.input.disabled;
            self.mobileInput.required = self.input.required;
            self.mobileInput.placeholder = self.input.placeholder;
            self.mobileFormatStr =
                inputType === "datetime-local"
                    ? "Y-m-d\\TH:i:S"
                    : inputType === "date"
                        ? "Y-m-d"
                        : "H:i:S";
            if (self.selectedDates.length > 0) {
                self.mobileInput.defaultValue = self.mobileInput.value = self.formatDate(self.selectedDates[0], self.mobileFormatStr);
            }
            if (self.config.minDate)
                self.mobileInput.min = self.formatDate(self.config.minDate, "Y-m-d");
            if (self.config.maxDate)
                self.mobileInput.max = self.formatDate(self.config.maxDate, "Y-m-d");
            if (self.input.getAttribute("step"))
                self.mobileInput.step = String(self.input.getAttribute("step"));
            self.input.type = "hidden";
            if (self.altInput !== undefined)
                self.altInput.type = "hidden";
            try {
                if (self.input.parentNode)
                    self.input.parentNode.insertBefore(self.mobileInput, self.input.nextSibling);
            }
            catch (_a) { }
            bind(self.mobileInput, "change", function (e) {
                self.setDate(getEventTarget(e).value, false, self.mobileFormatStr);
                triggerEvent("onChange");
                triggerEvent("onClose");
            });
        }
        function toggle(e) {
            if (self.isOpen === true)
                return self.close();
            self.open(e);
        }
        function triggerEvent(event, data) {
            // If the instance has been destroyed already, all hooks have been removed
            if (self.config === undefined)
                return;
            var hooks = self.config[event];
            if (hooks !== undefined && hooks.length > 0) {
                for (var i = 0; hooks[i] && i < hooks.length; i++)
                    hooks[i](self.selectedDates, self.input.value, self, data);
            }
            if (event === "onChange") {
                self.input.dispatchEvent(createEvent("change"));
                // many front-end frameworks bind to the input event
                self.input.dispatchEvent(createEvent("input"));
            }
        }
        function createEvent(name) {
            var e = document.createEvent("Event");
            e.initEvent(name, true, true);
            return e;
        }
        function isDateSelected(date) {
            for (var i = 0; i < self.selectedDates.length; i++) {
                var selectedDate = self.selectedDates[i];
                if (selectedDate instanceof Date &&
                    compareDates(selectedDate, date) === 0)
                    return "" + i;
            }
            return false;
        }
        function isDateInRange(date) {
            if (self.config.mode !== "range" || self.selectedDates.length < 2)
                return false;
            return (compareDates(date, self.selectedDates[0]) >= 0 &&
                compareDates(date, self.selectedDates[1]) <= 0);
        }
        function updateNavigationCurrentMonth() {
            if (self.config.noCalendar || self.isMobile || !self.monthNav)
                return;
            self.yearElements.forEach(function (yearElement, i) {
                var d = new Date(self.currentYear, self.currentMonth, 1);
                d.setMonth(self.currentMonth + i);
                if (self.config.showMonths > 1 ||
                    self.config.monthSelectorType === "static") {
                    self.monthElements[i].textContent =
                        monthToStr(d.getMonth(), self.config.shorthandCurrentMonth, self.l10n) + " ";
                }
                else {
                    self.monthsDropdownContainer.value = d.getMonth().toString();
                }
                yearElement.value = d.getFullYear().toString();
            });
            self._hidePrevMonthArrow =
                self.config.minDate !== undefined &&
                (self.currentYear === self.config.minDate.getFullYear()
                    ? self.currentMonth <= self.config.minDate.getMonth()
                    : self.currentYear < self.config.minDate.getFullYear());
            self._hideNextMonthArrow =
                self.config.maxDate !== undefined &&
                (self.currentYear === self.config.maxDate.getFullYear()
                    ? self.currentMonth + 1 > self.config.maxDate.getMonth()
                    : self.currentYear > self.config.maxDate.getFullYear());
        }
        function getDateStr(specificFormat) {
            var format = specificFormat ||
                (self.config.altInput ? self.config.altFormat : self.config.dateFormat);
            return self.selectedDates
                .map(function (dObj) { return self.formatDate(dObj, format); })
                .filter(function (d, i, arr) {
                    return self.config.mode !== "range" ||
                        self.config.enableTime ||
                        arr.indexOf(d) === i;
                })
                .join(self.config.mode !== "range"
                    ? self.config.conjunction
                    : self.l10n.rangeSeparator);
        }
        /**
         * Updates the values of inputs associated with the calendar
         */
        function updateValue(triggerChange) {
            if (triggerChange === void 0) { triggerChange = true; }
            if (self.mobileInput !== undefined && self.mobileFormatStr) {
                self.mobileInput.value =
                    self.latestSelectedDateObj !== undefined
                        ? self.formatDate(self.latestSelectedDateObj, self.mobileFormatStr)
                        : "";
            }
            self.input.value = getDateStr(self.config.dateFormat);
            if (self.altInput !== undefined) {
                self.altInput.value = getDateStr(self.config.altFormat);
            }
            if (triggerChange !== false)
                triggerEvent("onValueUpdate");
        }
        function onMonthNavClick(e) {
            var eventTarget = getEventTarget(e);
            var isPrevMonth = self.prevMonthNav.contains(eventTarget);
            var isNextMonth = self.nextMonthNav.contains(eventTarget);
            if (isPrevMonth || isNextMonth) {
                changeMonth(isPrevMonth ? -1 : 1);
            }
            else if (self.yearElements.indexOf(eventTarget) >= 0) {
                eventTarget.select();
            }
            else if (eventTarget.classList.contains("arrowUp")) {
                self.changeYear(self.currentYear + 1);
            }
            else if (eventTarget.classList.contains("arrowDown")) {
                self.changeYear(self.currentYear - 1);
            }
        }
        function timeWrapper(e) {
            e.preventDefault();
            var isKeyDown = e.type === "keydown", eventTarget = getEventTarget(e), input = eventTarget;
            if (self.amPM !== undefined && eventTarget === self.amPM) {
                self.amPM.textContent =
                    self.l10n.amPM[int(self.amPM.textContent === self.l10n.amPM[0])];
            }
            var min = parseFloat(input.getAttribute("min")), max = parseFloat(input.getAttribute("max")), step = parseFloat(input.getAttribute("step")), curValue = parseInt(input.value, 10), delta = e.delta ||
                (isKeyDown ? (e.which === 38 ? 1 : -1) : 0);
            var newValue = curValue + step * delta;
            if (typeof input.value !== "undefined" && input.value.length === 2) {
                var isHourElem = input === self.hourElement, isMinuteElem = input === self.minuteElement;
                if (newValue < min) {
                    newValue =
                        max +
                        newValue +
                        int(!isHourElem) +
                        (int(isHourElem) && int(!self.amPM));
                    if (isMinuteElem)
                        incrementNumInput(undefined, -1, self.hourElement);
                }
                else if (newValue > max) {
                    newValue =
                        input === self.hourElement ? newValue - max - int(!self.amPM) : min;
                    if (isMinuteElem)
                        incrementNumInput(undefined, 1, self.hourElement);
                }
                if (self.amPM &&
                    isHourElem &&
                    (step === 1
                        ? newValue + curValue === 23
                        : Math.abs(newValue - curValue) > step)) {
                    self.amPM.textContent =
                        self.l10n.amPM[int(self.amPM.textContent === self.l10n.amPM[0])];
                }
                input.value = pad(newValue);
            }
        }
        init();
        return self;
    }
    /* istanbul ignore next */
    function _flatpickr(nodeList, config) {
        // static list
        var nodes = Array.prototype.slice
            .call(nodeList)
            .filter(function (x) { return x instanceof HTMLElement; });
        var instances = [];
        for (var i = 0; i < nodes.length; i++) {
            var node = nodes[i];
            try {
                if (node.getAttribute("data-fp-omit") !== null)
                    continue;
                if (node._flatpickr !== undefined) {
                    node._flatpickr.destroy();
                    node._flatpickr = undefined;
                }
                node._flatpickr = FlatpickrInstance(node, config || {});
                instances.push(node._flatpickr);
            }
            catch (e) {
                console.error(e);
            }
        }
        return instances.length === 1 ? instances[0] : instances;
    }
    /* istanbul ignore next */
    if (typeof HTMLElement !== "undefined" &&
        typeof HTMLCollection !== "undefined" &&
        typeof NodeList !== "undefined") {
        // browser env
        HTMLCollection.prototype.flatpickr = NodeList.prototype.flatpickr = function (config) {
            return _flatpickr(this, config);
        };
        HTMLElement.prototype.flatpickr = function (config) {
            return _flatpickr([this], config);
        };
    }
    /* istanbul ignore next */
    var flatpickr = function (selector, config) {
        if (typeof selector === "string") {
            return _flatpickr(window.document.querySelectorAll(selector), config);
        }
        else if (selector instanceof Node) {
            return _flatpickr([selector], config);
        }
        else {
            return _flatpickr(selector, config);
        }
    };
    /* istanbul ignore next */
    flatpickr.defaultConfig = {};
    flatpickr.l10ns = {
        en: __assign({}, english),
        default: __assign({}, english),
    };
    flatpickr.localize = function (l10n) {
        flatpickr.l10ns.default = __assign(__assign({}, flatpickr.l10ns.default), l10n);
    };
    flatpickr.setDefaults = function (config) {
        flatpickr.defaultConfig = __assign(__assign({}, flatpickr.defaultConfig), config);
    };
    flatpickr.parseDate = createDateParser({});
    flatpickr.formatDate = createDateFormatter({});
    flatpickr.compareDates = compareDates;
    /* istanbul ignore next */
    if (typeof jQuery !== "undefined" && typeof jQuery.fn !== "undefined") {
        jQuery.fn.flatpickr = function (config) {
            return _flatpickr(this, config);
        };
    }
    Date.prototype.fp_incr = function (days) {
        return new Date(this.getFullYear(), this.getMonth(), this.getDate() + (typeof days === "string" ? parseInt(days, 10) : days));
    };
    if (typeof window !== "undefined") {
        window.flatpickr = flatpickr;
    }

    return flatpickr;

})));

(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
        typeof define === 'function' && define.amd ? define(['exports'], factory) :
            (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.ar = {}));
}(this, (function (exports) { 'use strict';

    var fp = typeof window !== "undefined" && window.flatpickr !== undefined
        ? window.flatpickr
        : {
            l10ns: {},
        };
    var Arabic = {
        weekdays: {
            shorthand: ["أحد", "اثنين", "ثلاثاء", "أربعاء", "خميس", "جمعة", "سبت"],
            longhand: [
                "الأحد",
                "الاثنين",
                "الثلاثاء",
                "الأربعاء",
                "الخميس",
                "الجمعة",
                "السبت",
            ],
        },
        months: {
            shorthand: ["1", "2", "3", "4", "5", "6", "7", "8", "9", "10", "11", "12"],
            longhand: [
                "يناير",
                "فبراير",
                "مارس",
                "أبريل",
                "مايو",
                "يونيو",
                "يوليو",
                "أغسطس",
                "سبتمبر",
                "أكتوبر",
                "نوفمبر",
                "ديسمبر",
            ],
        },
        firstDayOfWeek: 6,
        rangeSeparator: " إلى ",
        weekAbbreviation: "Wk",
        scrollTitle: "قم بالتمرير للزيادة",
        toggleTitle: "اضغط للتبديل",
        amPM: ["ص", "م"],
        yearAriaLabel: "سنة",
        monthAriaLabel: "شهر",
        hourAriaLabel: "ساعة",
        minuteAriaLabel: "دقيقة",
        time_24hr: false,
    };
    fp.l10ns.ar = Arabic;
    var ar = fp.l10ns;

    exports.Arabic = Arabic;
    exports.default = ar;

    Object.defineProperty(exports, '__esModule', { value: true });

})));

/*!
 * https://github.com/paulmillr/es6-shim
 * @license es6-shim Copyright 2013-2016 by Paul Miller (http://paulmillr.com)
 *   and contributors,  MIT License
 * es6-shim: v0.35.4
 * see https://github.com/paulmillr/es6-shim/blob/0.35.3/LICENSE
 * Details and documentation:
 * https://github.com/paulmillr/es6-shim/
 */

// UMD (Universal Module Definition)
// see https://github.com/umdjs/umd/blob/master/returnExports.js
(function (root, factory) {
    /*global define */
    if (typeof define === 'function' && define.amd) {
        // AMD. Register as an anonymous module.
        define(factory);
    } else if (typeof exports === 'object') {
        // Node. Does not work with strict CommonJS, but
        // only CommonJS-like environments that support module.exports,
        // like Node.
        module.exports = factory();
    } else {
        // Browser globals (root is window)
        root.returnExports = factory();
    }
}(this, function () {
    'use strict';

    var _apply = Function.call.bind(Function.apply);
    var _call = Function.call.bind(Function.call);
    var isArray = Array.isArray;
    var keys = Object.keys;

    var not = function notThunker(func) {
        return function notThunk() {
            return !_apply(func, this, arguments);
        };
    };
    var throwsError = function (func) {
        try {
            func();
            return false;
        } catch (e) {
            return true;
        }
    };
    var valueOrFalseIfThrows = function valueOrFalseIfThrows(func) {
        try {
            return func();
        } catch (e) {
            return false;
        }
    };

    var isCallableWithoutNew = not(throwsError);
    var arePropertyDescriptorsSupported = function () {
        // if Object.defineProperty exists but throws, it's IE 8
        return !throwsError(function () {
            return Object.defineProperty({}, 'x', { get: function () { } }); // eslint-disable-line getter-return
        });
    };
    var supportsDescriptors = !!Object.defineProperty && arePropertyDescriptorsSupported();
    var functionsHaveNames = (function foo() {}).name === 'foo'; // eslint-disable-line no-extra-parens

    var _forEach = Function.call.bind(Array.prototype.forEach);
    var _reduce = Function.call.bind(Array.prototype.reduce);
    var _filter = Function.call.bind(Array.prototype.filter);
    var _some = Function.call.bind(Array.prototype.some);

    var defineProperty = function (object, name, value, force) {
        if (!force && name in object) { return; }
        if (supportsDescriptors) {
            Object.defineProperty(object, name, {
                configurable: true,
                enumerable: false,
                writable: true,
                value: value
            });
        } else {
            object[name] = value;
        }
    };

    // Define configurable, writable and non-enumerable props
    // if they don’t exist.
    var defineProperties = function (object, map, forceOverride) {
        _forEach(keys(map), function (name) {
            var method = map[name];
            defineProperty(object, name, method, !!forceOverride);
        });
    };

    var _toString = Function.call.bind(Object.prototype.toString);
    var isCallable = typeof /abc/ === 'function' ? function IsCallableSlow(x) {
        // Some old browsers (IE, FF) say that typeof /abc/ === 'function'
        return typeof x === 'function' && _toString(x) === '[object Function]';
    } : function IsCallableFast(x) { return typeof x === 'function'; };

    var Value = {
        getter: function (object, name, getter) {
            if (!supportsDescriptors) {
                throw new TypeError('getters require true ES5 support');
            }
            Object.defineProperty(object, name, {
                configurable: true,
                enumerable: false,
                get: getter
            });
        },
        proxy: function (originalObject, key, targetObject) {
            if (!supportsDescriptors) {
                throw new TypeError('getters require true ES5 support');
            }
            var originalDescriptor = Object.getOwnPropertyDescriptor(originalObject, key);
            Object.defineProperty(targetObject, key, {
                configurable: originalDescriptor.configurable,
                enumerable: originalDescriptor.enumerable,
                get: function getKey() { return originalObject[key]; },
                set: function setKey(value) { originalObject[key] = value; }
            });
        },
        redefine: function (object, property, newValue) {
            if (supportsDescriptors) {
                var descriptor = Object.getOwnPropertyDescriptor(object, property);
                descriptor.value = newValue;
                Object.defineProperty(object, property, descriptor);
            } else {
                object[property] = newValue;
            }
        },
        defineByDescriptor: function (object, property, descriptor) {
            if (supportsDescriptors) {
                Object.defineProperty(object, property, descriptor);
            } else if ('value' in descriptor) {
                object[property] = descriptor.value;
            }
        },
        preserveToString: function (target, source) {
            if (source && isCallable(source.toString)) {
                defineProperty(target, 'toString', source.toString.bind(source), true);
            }
        }
    };

    // Simple shim for Object.create on ES3 browsers
    // (unlike real shim, no attempt to support `prototype === null`)
    var create = Object.create || function (prototype, properties) {
        var Prototype = function Prototype() {};
        Prototype.prototype = prototype;
        var object = new Prototype();
        if (typeof properties !== 'undefined') {
            keys(properties).forEach(function (key) {
                Value.defineByDescriptor(object, key, properties[key]);
            });
        }
        return object;
    };

    var supportsSubclassing = function (C, f) {
        if (!Object.setPrototypeOf) { return false; /* skip test on IE < 11 */ }
        return valueOrFalseIfThrows(function () {
            var Sub = function Subclass(arg) {
                var o = new C(arg);
                Object.setPrototypeOf(o, Subclass.prototype);
                return o;
            };
            Object.setPrototypeOf(Sub, C);
            Sub.prototype = create(C.prototype, {
                constructor: { value: Sub }
            });
            return f(Sub);
        });
    };

    var getGlobal = function () {
        /* global self, window */
        // the only reliable means to get the global object is
        // `Function('return this')()`
        // However, this causes CSP violations in Chrome apps.
        if (typeof self !== 'undefined') { return self; }
        if (typeof window !== 'undefined') { return window; }
        if (typeof global !== 'undefined') { return global; }
        throw new Error('unable to locate global object');
    };

    var globals = getGlobal();
    var globalIsFinite = globals.isFinite;
    var _indexOf = Function.call.bind(String.prototype.indexOf);
    var _arrayIndexOfApply = Function.apply.bind(Array.prototype.indexOf);
    var _concat = Function.call.bind(Array.prototype.concat);
    // var _sort = Function.call.bind(Array.prototype.sort);
    var _strSlice = Function.call.bind(String.prototype.slice);
    var _push = Function.call.bind(Array.prototype.push);
    var _pushApply = Function.apply.bind(Array.prototype.push);
    var _join = Function.call.bind(Array.prototype.join);
    var _shift = Function.call.bind(Array.prototype.shift);
    var _max = Math.max;
    var _min = Math.min;
    var _floor = Math.floor;
    var _abs = Math.abs;
    var _exp = Math.exp;
    var _log = Math.log;
    var _sqrt = Math.sqrt;
    var _hasOwnProperty = Function.call.bind(Object.prototype.hasOwnProperty);
    var ArrayIterator; // make our implementation private
    var noop = function () {};

    var OrigMap = globals.Map;
    var origMapDelete = OrigMap && OrigMap.prototype['delete'];
    var origMapGet = OrigMap && OrigMap.prototype.get;
    var origMapHas = OrigMap && OrigMap.prototype.has;
    var origMapSet = OrigMap && OrigMap.prototype.set;

    var Symbol = globals.Symbol || {};
    var symbolSpecies = Symbol.species || '@@species';

    var numberIsNaN = Number.isNaN || function isNaN(value) {
        // NaN !== NaN, but they are identical.
        // NaNs are the only non-reflexive value, i.e., if x !== x,
        // then x is NaN.
        // isNaN is broken: it converts its argument to number, so
        // isNaN('foo') => true
        return value !== value;
    };
    var numberIsFinite = Number.isFinite || function isFinite(value) {
        return typeof value === 'number' && globalIsFinite(value);
    };
    var _sign = isCallable(Math.sign) ? Math.sign : function sign(value) {
        var number = Number(value);
        if (number === 0) { return number; }
        if (numberIsNaN(number)) { return number; }
        return number < 0 ? -1 : 1;
    };
    var _log1p = function log1p(value) {
        var x = Number(value);
        if (x < -1 || numberIsNaN(x)) { return NaN; }
        if (x === 0 || x === Infinity) { return x; }
        if (x === -1) { return -Infinity; }

        return (1 + x) - 1 === 0 ? x : x * (_log(1 + x) / ((1 + x) - 1));
    };

    // taken directly from https://github.com/ljharb/is-arguments/blob/master/index.js
    // can be replaced with require('is-arguments') if we ever use a build process instead
    var isStandardArguments = function isArguments(value) {
        return _toString(value) === '[object Arguments]';
    };
    var isLegacyArguments = function isArguments(value) {
        return value !== null &&
            typeof value === 'object' &&
            typeof value.length === 'number' &&
            value.length >= 0 &&
            _toString(value) !== '[object Array]' &&
            _toString(value.callee) === '[object Function]';
    };
    var isArguments = isStandardArguments(arguments) ? isStandardArguments : isLegacyArguments;

    var Type = {
        primitive: function (x) { return x === null || (typeof x !== 'function' && typeof x !== 'object'); },
        string: function (x) { return _toString(x) === '[object String]'; },
        regex: function (x) { return _toString(x) === '[object RegExp]'; },
        symbol: function (x) {
            return typeof globals.Symbol === 'function' && typeof x === 'symbol';
        }
    };

    var overrideNative = function overrideNative(object, property, replacement) {
        var original = object[property];
        defineProperty(object, property, replacement, true);
        Value.preserveToString(object[property], original);
    };

    // eslint-disable-next-line no-restricted-properties
    var hasSymbols = typeof Symbol === 'function' && typeof Symbol['for'] === 'function' && Type.symbol(Symbol());

    // This is a private name in the es6 spec, equal to '[Symbol.iterator]'
    // we're going to use an arbitrary _-prefixed name to make our shims
    // work properly with each other, even though we don't have full Iterator
    // support.  That is, `Array.from(map.keys())` will work, but we don't
    // pretend to export a "real" Iterator interface.
    var $iterator$ = Type.symbol(Symbol.iterator) ? Symbol.iterator : '_es6-shim iterator_';
    // Firefox ships a partial implementation using the name @@iterator.
    // https://bugzilla.mozilla.org/show_bug.cgi?id=907077#c14
    // So use that name if we detect it.
    if (globals.Set && typeof new globals.Set()['@@iterator'] === 'function') {
        $iterator$ = '@@iterator';
    }

    // Reflect
    if (!globals.Reflect) {
        defineProperty(globals, 'Reflect', {}, true);
    }
    var Reflect = globals.Reflect;

    var $String = String;

    /* global document */
    var domAll = (typeof document === 'undefined' || !document) ? null : document.all;
    var isNullOrUndefined = domAll == null ? function isNullOrUndefined(x) {
        return x == null;
    } : function isNullOrUndefinedAndNotDocumentAll(x) {
        return x == null && x !== domAll;
    };

    var ES = {
        // http://www.ecma-international.org/ecma-262/6.0/#sec-call
        Call: function Call(F, V) {
            var args = arguments.length > 2 ? arguments[2] : [];
            if (!ES.IsCallable(F)) {
                throw new TypeError(F + ' is not a function');
            }
            return _apply(F, V, args);
        },

        RequireObjectCoercible: function (x, optMessage) {
            if (isNullOrUndefined(x)) {
                throw new TypeError(optMessage || 'Cannot call method on ' + x);
            }
            return x;
        },

        // This might miss the "(non-standard exotic and does not implement
        // [[Call]])" case from
        // http://www.ecma-international.org/ecma-262/6.0/#sec-typeof-operator-runtime-semantics-evaluation
        // but we can't find any evidence these objects exist in practice.
        // If we find some in the future, you could test `Object(x) === x`,
        // which is reliable according to
        // http://www.ecma-international.org/ecma-262/6.0/#sec-toobject
        // but is not well optimized by runtimes and creates an object
        // whenever it returns false, and thus is very slow.
        TypeIsObject: function (x) {
            if (x === void 0 || x === null || x === true || x === false) {
                return false;
            }
            return typeof x === 'function' || typeof x === 'object' || x === domAll;
        },

        ToObject: function (o, optMessage) {
            return Object(ES.RequireObjectCoercible(o, optMessage));
        },

        IsCallable: isCallable,

        IsConstructor: function (x) {
            // We can't tell callables from constructors in ES5
            return ES.IsCallable(x);
        },

        ToInt32: function (x) {
            return ES.ToNumber(x) >> 0;
        },

        ToUint32: function (x) {
            return ES.ToNumber(x) >>> 0;
        },

        ToNumber: function (value) {
            if (hasSymbols && _toString(value) === '[object Symbol]') {
                throw new TypeError('Cannot convert a Symbol value to a number');
            }
            return +value;
        },

        ToInteger: function (value) {
            var number = ES.ToNumber(value);
            if (numberIsNaN(number)) { return 0; }
            if (number === 0 || !numberIsFinite(number)) { return number; }
            return (number > 0 ? 1 : -1) * _floor(_abs(number));
        },

        ToLength: function (value) {
            var len = ES.ToInteger(value);
            if (len <= 0) { return 0; } // includes converting -0 to +0
            if (len > Number.MAX_SAFE_INTEGER) { return Number.MAX_SAFE_INTEGER; }
            return len;
        },

        SameValue: function (a, b) {
            if (a === b) {
                // 0 === -0, but they are not identical.
                if (a === 0) { return 1 / a === 1 / b; }
                return true;
            }
            return numberIsNaN(a) && numberIsNaN(b);
        },

        SameValueZero: function (a, b) {
            // same as SameValue except for SameValueZero(+0, -0) == true
            return (a === b) || (numberIsNaN(a) && numberIsNaN(b));
        },

        IsIterable: function (o) {
            return ES.TypeIsObject(o) && (typeof o[$iterator$] !== 'undefined' || isArguments(o));
        },

        GetIterator: function (o) {
            if (isArguments(o)) {
                // special case support for `arguments`
                return new ArrayIterator(o, 'value');
            }
            var itFn = ES.GetMethod(o, $iterator$);
            if (!ES.IsCallable(itFn)) {
                // Better diagnostics if itFn is null or undefined
                throw new TypeError('value is not an iterable');
            }
            var it = ES.Call(itFn, o);
            if (!ES.TypeIsObject(it)) {
                throw new TypeError('bad iterator');
            }
            return it;
        },

        GetMethod: function (o, p) {
            var func = ES.ToObject(o)[p];
            if (isNullOrUndefined(func)) {
                return void 0;
            }
            if (!ES.IsCallable(func)) {
                throw new TypeError('Method not callable: ' + p);
            }
            return func;
        },

        IteratorComplete: function (iterResult) {
            return !!iterResult.done;
        },

        IteratorClose: function (iterator, completionIsThrow) {
            var returnMethod = ES.GetMethod(iterator, 'return');
            if (returnMethod === void 0) {
                return;
            }
            var innerResult, innerException;
            try {
                innerResult = ES.Call(returnMethod, iterator);
            } catch (e) {
                innerException = e;
            }
            if (completionIsThrow) {
                return;
            }
            if (innerException) {
                throw innerException;
            }
            if (!ES.TypeIsObject(innerResult)) {
                throw new TypeError("Iterator's return method returned a non-object.");
            }
        },

        IteratorNext: function (it) {
            var result = arguments.length > 1 ? it.next(arguments[1]) : it.next();
            if (!ES.TypeIsObject(result)) {
                throw new TypeError('bad iterator');
            }
            return result;
        },

        IteratorStep: function (it) {
            var result = ES.IteratorNext(it);
            var done = ES.IteratorComplete(result);
            return done ? false : result;
        },

        Construct: function (C, args, newTarget, isES6internal) {
            var target = typeof newTarget === 'undefined' ? C : newTarget;

            if (!isES6internal && Reflect.construct) {
                // Try to use Reflect.construct if available
                return Reflect.construct(C, args, target);
            }
            // OK, we have to fake it.  This will only work if the
            // C.[[ConstructorKind]] == "base" -- but that's the only
            // kind we can make in ES5 code anyway.

            // OrdinaryCreateFromConstructor(target, "%ObjectPrototype%")
            var proto = target.prototype;
            if (!ES.TypeIsObject(proto)) {
                proto = Object.prototype;
            }
            var obj = create(proto);
            // Call the constructor.
            var result = ES.Call(C, obj, args);
            return ES.TypeIsObject(result) ? result : obj;
        },

        SpeciesConstructor: function (O, defaultConstructor) {
            var C = O.constructor;
            if (C === void 0) {
                return defaultConstructor;
            }
            if (!ES.TypeIsObject(C)) {
                throw new TypeError('Bad constructor');
            }
            var S = C[symbolSpecies];
            if (isNullOrUndefined(S)) {
                return defaultConstructor;
            }
            if (!ES.IsConstructor(S)) {
                throw new TypeError('Bad @@species');
            }
            return S;
        },

        CreateHTML: function (string, tag, attribute, value) {
            var S = ES.ToString(string);
            var p1 = '<' + tag;
            if (attribute !== '') {
                var V = ES.ToString(value);
                var escapedV = V.replace(/"/g, '&quot;');
                p1 += ' ' + attribute + '="' + escapedV + '"';
            }
            var p2 = p1 + '>';
            var p3 = p2 + S;
            return p3 + '</' + tag + '>';
        },

        IsRegExp: function IsRegExp(argument) {
            if (!ES.TypeIsObject(argument)) {
                return false;
            }
            var isRegExp = argument[Symbol.match];
            if (typeof isRegExp !== 'undefined') {
                return !!isRegExp;
            }
            return Type.regex(argument);
        },

        ToString: function ToString(string) {
            if (hasSymbols && _toString(string) === '[object Symbol]') {
                throw new TypeError('Cannot convert a Symbol value to a number');
            }
            return $String(string);
        }
    };

    // Well-known Symbol shims
    if (supportsDescriptors && hasSymbols) {
        var defineWellKnownSymbol = function defineWellKnownSymbol(name) {
            if (Type.symbol(Symbol[name])) {
                return Symbol[name];
            }
            // eslint-disable-next-line no-restricted-properties
            var sym = Symbol['for']('Symbol.' + name);
            Object.defineProperty(Symbol, name, {
                configurable: false,
                enumerable: false,
                writable: false,
                value: sym
            });
            return sym;
        };
        if (!Type.symbol(Symbol.search)) {
            var symbolSearch = defineWellKnownSymbol('search');
            var originalSearch = String.prototype.search;
            defineProperty(RegExp.prototype, symbolSearch, function search(string) {
                return ES.Call(originalSearch, string, [this]);
            });
            var searchShim = function search(regexp) {
                var O = ES.RequireObjectCoercible(this);
                if (!isNullOrUndefined(regexp)) {
                    var searcher = ES.GetMethod(regexp, symbolSearch);
                    if (typeof searcher !== 'undefined') {
                        return ES.Call(searcher, regexp, [O]);
                    }
                }
                return ES.Call(originalSearch, O, [ES.ToString(regexp)]);
            };
            overrideNative(String.prototype, 'search', searchShim);
        }
        if (!Type.symbol(Symbol.replace)) {
            var symbolReplace = defineWellKnownSymbol('replace');
            var originalReplace = String.prototype.replace;
            defineProperty(RegExp.prototype, symbolReplace, function replace(string, replaceValue) {
                return ES.Call(originalReplace, string, [this, replaceValue]);
            });
            var replaceShim = function replace(searchValue, replaceValue) {
                var O = ES.RequireObjectCoercible(this);
                if (!isNullOrUndefined(searchValue)) {
                    var replacer = ES.GetMethod(searchValue, symbolReplace);
                    if (typeof replacer !== 'undefined') {
                        return ES.Call(replacer, searchValue, [O, replaceValue]);
                    }
                }
                return ES.Call(originalReplace, O, [ES.ToString(searchValue), replaceValue]);
            };
            overrideNative(String.prototype, 'replace', replaceShim);
        }
        if (!Type.symbol(Symbol.split)) {
            var symbolSplit = defineWellKnownSymbol('split');
            var originalSplit = String.prototype.split;
            defineProperty(RegExp.prototype, symbolSplit, function split(string, limit) {
                return ES.Call(originalSplit, string, [this, limit]);
            });
            var splitShim = function split(separator, limit) {
                var O = ES.RequireObjectCoercible(this);
                if (!isNullOrUndefined(separator)) {
                    var splitter = ES.GetMethod(separator, symbolSplit);
                    if (typeof splitter !== 'undefined') {
                        return ES.Call(splitter, separator, [O, limit]);
                    }
                }
                return ES.Call(originalSplit, O, [ES.ToString(separator), limit]);
            };
            overrideNative(String.prototype, 'split', splitShim);
        }
        var symbolMatchExists = Type.symbol(Symbol.match);
        var stringMatchIgnoresSymbolMatch = symbolMatchExists && (function () {
            // Firefox 41, through Nightly 45 has Symbol.match, but String#match ignores it.
            // Firefox 40 and below have Symbol.match but String#match works fine.
            var o = {};
            o[Symbol.match] = function () { return 42; };
            return 'a'.match(o) !== 42;
        }());
        if (!symbolMatchExists || stringMatchIgnoresSymbolMatch) {
            var symbolMatch = defineWellKnownSymbol('match');

            var originalMatch = String.prototype.match;
            defineProperty(RegExp.prototype, symbolMatch, function match(string) {
                return ES.Call(originalMatch, string, [this]);
            });

            var matchShim = function match(regexp) {
                var O = ES.RequireObjectCoercible(this);
                if (!isNullOrUndefined(regexp)) {
                    var matcher = ES.GetMethod(regexp, symbolMatch);
                    if (typeof matcher !== 'undefined') {
                        return ES.Call(matcher, regexp, [O]);
                    }
                }
                return ES.Call(originalMatch, O, [ES.ToString(regexp)]);
            };
            overrideNative(String.prototype, 'match', matchShim);
        }
    }

    var wrapConstructor = function wrapConstructor(original, replacement, keysToSkip) {
        Value.preserveToString(replacement, original);
        if (Object.setPrototypeOf) {
            // sets up proper prototype chain where possible
            Object.setPrototypeOf(original, replacement);
        }
        if (supportsDescriptors) {
            _forEach(Object.getOwnPropertyNames(original), function (key) {
                if (key in noop || keysToSkip[key]) { return; }
                Value.proxy(original, key, replacement);
            });
        } else {
            _forEach(Object.keys(original), function (key) {
                if (key in noop || keysToSkip[key]) { return; }
                replacement[key] = original[key];
            });
        }
        replacement.prototype = original.prototype;
        Value.redefine(original.prototype, 'constructor', replacement);
    };

    var defaultSpeciesGetter = function () { return this; };
    var addDefaultSpecies = function (C) {
        if (supportsDescriptors && !_hasOwnProperty(C, symbolSpecies)) {
            Value.getter(C, symbolSpecies, defaultSpeciesGetter);
        }
    };

    var addIterator = function (prototype, impl) {
        var implementation = impl || function iterator() { return this; };
        defineProperty(prototype, $iterator$, implementation);
        if (!prototype[$iterator$] && Type.symbol($iterator$)) {
            // implementations are buggy when $iterator$ is a Symbol
            prototype[$iterator$] = implementation;
        }
    };

    var createDataProperty = function createDataProperty(object, name, value) {
        if (supportsDescriptors) {
            Object.defineProperty(object, name, {
                configurable: true,
                enumerable: true,
                writable: true,
                value: value
            });
        } else {
            object[name] = value;
        }
    };
    var createDataPropertyOrThrow = function createDataPropertyOrThrow(object, name, value) {
        createDataProperty(object, name, value);
        if (!ES.SameValue(object[name], value)) {
            throw new TypeError('property is nonconfigurable');
        }
    };

    var emulateES6construct = function (o, defaultNewTarget, defaultProto, slots) {
        // This is an es5 approximation to es6 construct semantics.  in es6,
        // 'new Foo' invokes Foo.[[Construct]] which (for almost all objects)
        // just sets the internal variable NewTarget (in es6 syntax `new.target`)
        // to Foo and then returns Foo().

        // Many ES6 object then have constructors of the form:
        // 1. If NewTarget is undefined, throw a TypeError exception
        // 2. Let xxx by OrdinaryCreateFromConstructor(NewTarget, yyy, zzz)

        // So we're going to emulate those first two steps.
        if (!ES.TypeIsObject(o)) {
            throw new TypeError('Constructor requires `new`: ' + defaultNewTarget.name);
        }
        var proto = defaultNewTarget.prototype;
        if (!ES.TypeIsObject(proto)) {
            proto = defaultProto;
        }
        var obj = create(proto);
        for (var name in slots) {
            if (_hasOwnProperty(slots, name)) {
                var value = slots[name];
                defineProperty(obj, name, value, true);
            }
        }
        return obj;
    };

    // Firefox 31 reports this function's length as 0
    // https://bugzilla.mozilla.org/show_bug.cgi?id=1062484
    if (String.fromCodePoint && String.fromCodePoint.length !== 1) {
        var originalFromCodePoint = String.fromCodePoint;
        overrideNative(String, 'fromCodePoint', function fromCodePoint(codePoints) {
            return ES.Call(originalFromCodePoint, this, arguments);
        });
    }

    var StringShims = {
        fromCodePoint: function fromCodePoint(codePoints) {
            var result = [];
            var next;
            for (var i = 0, length = arguments.length; i < length; i++) {
                next = Number(arguments[i]);
                if (!ES.SameValue(next, ES.ToInteger(next)) || next < 0 || next > 0x10FFFF) {
                    throw new RangeError('Invalid code point ' + next);
                }

                if (next < 0x10000) {
                    _push(result, String.fromCharCode(next));
                } else {
                    next -= 0x10000;
                    _push(result, String.fromCharCode((next >> 10) + 0xD800));
                    _push(result, String.fromCharCode((next % 0x400) + 0xDC00));
                }
            }
            return _join(result, '');
        },

        raw: function raw(template) {
            var numberOfSubstitutions = arguments.length - 1;
            var cooked = ES.ToObject(template, 'bad template');
            var raw = ES.ToObject(cooked.raw, 'bad raw value');
            var len = raw.length;
            var literalSegments = ES.ToLength(len);
            if (literalSegments <= 0) {
                return '';
            }

            var stringElements = [];
            var nextIndex = 0;
            var nextKey, next, nextSeg, nextSub;
            while (nextIndex < literalSegments) {
                nextKey = ES.ToString(nextIndex);
                nextSeg = ES.ToString(raw[nextKey]);
                _push(stringElements, nextSeg);
                if (nextIndex + 1 >= literalSegments) {
                    break;
                }
                next = nextIndex + 1 < arguments.length ? arguments[nextIndex + 1] : '';
                nextSub = ES.ToString(next);
                _push(stringElements, nextSub);
                nextIndex += 1;
            }
            return _join(stringElements, '');
        }
    };
    if (String.raw && String.raw({ raw: { 0: 'x', 1: 'y', length: 2 } }) !== 'xy') {
        // IE 11 TP has a broken String.raw implementation
        overrideNative(String, 'raw', StringShims.raw);
    }
    defineProperties(String, StringShims);

    // Fast repeat, uses the `Exponentiation by squaring` algorithm.
    // Perf: http://jsperf.com/string-repeat2/2
    var stringRepeat = function repeat(s, times) {
        if (times < 1) { return ''; }
        if (times % 2) { return repeat(s, times - 1) + s; }
        var half = repeat(s, times / 2);
        return half + half;
    };
    var stringMaxLength = Infinity;

    var StringPrototypeShims = {
        repeat: function repeat(times) {
            var thisStr = ES.ToString(ES.RequireObjectCoercible(this));
            var numTimes = ES.ToInteger(times);
            if (numTimes < 0 || numTimes >= stringMaxLength) {
                throw new RangeError('repeat count must be less than infinity and not overflow maximum string size');
            }
            return stringRepeat(thisStr, numTimes);
        },

        startsWith: function startsWith(searchString) {
            var S = ES.ToString(ES.RequireObjectCoercible(this));
            if (ES.IsRegExp(searchString)) {
                throw new TypeError('Cannot call method "startsWith" with a regex');
            }
            var searchStr = ES.ToString(searchString);
            var position;
            if (arguments.length > 1) {
                position = arguments[1];
            }
            var start = _max(ES.ToInteger(position), 0);
            return _strSlice(S, start, start + searchStr.length) === searchStr;
        },

        endsWith: function endsWith(searchString) {
            var S = ES.ToString(ES.RequireObjectCoercible(this));
            if (ES.IsRegExp(searchString)) {
                throw new TypeError('Cannot call method "endsWith" with a regex');
            }
            var searchStr = ES.ToString(searchString);
            var len = S.length;
            var endPosition;
            if (arguments.length > 1) {
                endPosition = arguments[1];
            }
            var pos = typeof endPosition === 'undefined' ? len : ES.ToInteger(endPosition);
            var end = _min(_max(pos, 0), len);
            return _strSlice(S, end - searchStr.length, end) === searchStr;
        },

        includes: function includes(searchString) {
            if (ES.IsRegExp(searchString)) {
                throw new TypeError('"includes" does not accept a RegExp');
            }
            var searchStr = ES.ToString(searchString);
            var position;
            if (arguments.length > 1) {
                position = arguments[1];
            }
            // Somehow this trick makes method 100% compat with the spec.
            return _indexOf(this, searchStr, position) !== -1;
        },

        codePointAt: function codePointAt(pos) {
            var thisStr = ES.ToString(ES.RequireObjectCoercible(this));
            var position = ES.ToInteger(pos);
            var length = thisStr.length;
            if (position >= 0 && position < length) {
                var first = thisStr.charCodeAt(position);
                var isEnd = position + 1 === length;
                if (first < 0xD800 || first > 0xDBFF || isEnd) { return first; }
                var second = thisStr.charCodeAt(position + 1);
                if (second < 0xDC00 || second > 0xDFFF) { return first; }
                return ((first - 0xD800) * 1024) + (second - 0xDC00) + 0x10000;
            }
        }
    };
    if (String.prototype.includes && 'a'.includes('a', Infinity) !== false) {
        overrideNative(String.prototype, 'includes', StringPrototypeShims.includes);
    }

    if (String.prototype.startsWith && String.prototype.endsWith) {
        var startsWithRejectsRegex = throwsError(function () {
            /* throws if spec-compliant */
            return '/a/'.startsWith(/a/);
        });
        var startsWithHandlesInfinity = valueOrFalseIfThrows(function () {
            return 'abc'.startsWith('a', Infinity) === false;
        });
        if (!startsWithRejectsRegex || !startsWithHandlesInfinity) {
            // Firefox (< 37?) and IE 11 TP have a noncompliant startsWith implementation
            overrideNative(String.prototype, 'startsWith', StringPrototypeShims.startsWith);
            overrideNative(String.prototype, 'endsWith', StringPrototypeShims.endsWith);
        }
    }
    if (hasSymbols) {
        var startsWithSupportsSymbolMatch = valueOrFalseIfThrows(function () {
            var re = /a/;
            re[Symbol.match] = false;
            return '/a/'.startsWith(re);
        });
        if (!startsWithSupportsSymbolMatch) {
            overrideNative(String.prototype, 'startsWith', StringPrototypeShims.startsWith);
        }
        var endsWithSupportsSymbolMatch = valueOrFalseIfThrows(function () {
            var re = /a/;
            re[Symbol.match] = false;
            return '/a/'.endsWith(re);
        });
        if (!endsWithSupportsSymbolMatch) {
            overrideNative(String.prototype, 'endsWith', StringPrototypeShims.endsWith);
        }
        var includesSupportsSymbolMatch = valueOrFalseIfThrows(function () {
            var re = /a/;
            re[Symbol.match] = false;
            return '/a/'.includes(re);
        });
        if (!includesSupportsSymbolMatch) {
            overrideNative(String.prototype, 'includes', StringPrototypeShims.includes);
        }
    }

    defineProperties(String.prototype, StringPrototypeShims);

    // whitespace from: http://es5.github.io/#x15.5.4.20
    // implementation from https://github.com/es-shims/es5-shim/blob/v3.4.0/es5-shim.js#L1304-L1324
    var ws = [
        '\x09\x0A\x0B\x0C\x0D\x20\xA0\u1680\u180E\u2000\u2001\u2002\u2003',
        '\u2004\u2005\u2006\u2007\u2008\u2009\u200A\u202F\u205F\u3000\u2028',
        '\u2029\uFEFF'
    ].join('');
    var trimRegexp = new RegExp('(^[' + ws + ']+)|([' + ws + ']+$)', 'g');
    var trimShim = function trim() {
        return ES.ToString(ES.RequireObjectCoercible(this)).replace(trimRegexp, '');
    };
    var nonWS = ['\u0085', '\u200b', '\ufffe'].join('');
    var nonWSregex = new RegExp('[' + nonWS + ']', 'g');
    var isBadHexRegex = /^[-+]0x[0-9a-f]+$/i;
    var hasStringTrimBug = nonWS.trim().length !== nonWS.length;
    defineProperty(String.prototype, 'trim', trimShim, hasStringTrimBug);

    // Given an argument x, it will return an IteratorResult object,
    // with value set to x and done to false.
    // Given no arguments, it will return an iterator completion object.
    var iteratorResult = function (x) {
        return { value: x, done: arguments.length === 0 };
    };

    // see http://www.ecma-international.org/ecma-262/6.0/#sec-string.prototype-@@iterator
    var StringIterator = function (s) {
        ES.RequireObjectCoercible(s);
        this._s = ES.ToString(s);
        this._i = 0;
    };
    StringIterator.prototype.next = function () {
        var s = this._s;
        var i = this._i;
        if (typeof s === 'undefined' || i >= s.length) {
            this._s = void 0;
            return iteratorResult();
        }
        var first = s.charCodeAt(i);
        var second, len;
        if (first < 0xD800 || first > 0xDBFF || (i + 1) === s.length) {
            len = 1;
        } else {
            second = s.charCodeAt(i + 1);
            len = (second < 0xDC00 || second > 0xDFFF) ? 1 : 2;
        }
        this._i = i + len;
        return iteratorResult(s.substr(i, len));
    };
    addIterator(StringIterator.prototype);
    addIterator(String.prototype, function () {
        return new StringIterator(this);
    });

    var ArrayShims = {
        from: function from(items) {
            var C = this;
            var mapFn;
            if (arguments.length > 1) {
                mapFn = arguments[1];
            }
            var mapping, T;
            if (typeof mapFn === 'undefined') {
                mapping = false;
            } else {
                if (!ES.IsCallable(mapFn)) {
                    throw new TypeError('Array.from: when provided, the second argument must be a function');
                }
                if (arguments.length > 2) {
                    T = arguments[2];
                }
                mapping = true;
            }

            // Note that that Arrays will use ArrayIterator:
            // https://bugs.ecmascript.org/show_bug.cgi?id=2416
            var usingIterator = typeof (isArguments(items) || ES.GetMethod(items, $iterator$)) !== 'undefined';

            var length, result, i;
            if (usingIterator) {
                result = ES.IsConstructor(C) ? Object(new C()) : [];
                var iterator = ES.GetIterator(items);
                var next, nextValue;

                i = 0;
                while (true) {
                    next = ES.IteratorStep(iterator);
                    if (next === false) {
                        break;
                    }
                    nextValue = next.value;
                    try {
                        if (mapping) {
                            nextValue = typeof T === 'undefined' ? mapFn(nextValue, i) : _call(mapFn, T, nextValue, i);
                        }
                        result[i] = nextValue;
                    } catch (e) {
                        ES.IteratorClose(iterator, true);
                        throw e;
                    }
                    i += 1;
                }
                length = i;
            } else {
                var arrayLike = ES.ToObject(items);
                length = ES.ToLength(arrayLike.length);
                result = ES.IsConstructor(C) ? Object(new C(length)) : new Array(length);
                var value;
                for (i = 0; i < length; ++i) {
                    value = arrayLike[i];
                    if (mapping) {
                        value = typeof T === 'undefined' ? mapFn(value, i) : _call(mapFn, T, value, i);
                    }
                    createDataPropertyOrThrow(result, i, value);
                }
            }

            result.length = length;
            return result;
        },

        of: function of() {
            var len = arguments.length;
            var C = this;
            var A = isArray(C) || !ES.IsCallable(C) ? new Array(len) : ES.Construct(C, [len]);
            for (var k = 0; k < len; ++k) {
                createDataPropertyOrThrow(A, k, arguments[k]);
            }
            A.length = len;
            return A;
        }
    };
    defineProperties(Array, ArrayShims);
    addDefaultSpecies(Array);

    // Our ArrayIterator is private; see
    // https://github.com/paulmillr/es6-shim/issues/252
    ArrayIterator = function (array, kind) {
        this.i = 0;
        this.array = array;
        this.kind = kind;
    };

    defineProperties(ArrayIterator.prototype, {
        next: function () {
            var i = this.i;
            var array = this.array;
            if (!(this instanceof ArrayIterator)) {
                throw new TypeError('Not an ArrayIterator');
            }
            if (typeof array !== 'undefined') {
                var len = ES.ToLength(array.length);
                if (i < len) {
                    //for (; i < len; i++) {
                    var kind = this.kind;
                    var retval;
                    if (kind === 'key') {
                        retval = i;
                    } else if (kind === 'value') {
                        retval = array[i];
                    } else if (kind === 'entry') {
                        retval = [i, array[i]];
                    }
                    this.i = i + 1;
                    return iteratorResult(retval);
                }
            }
            this.array = void 0;
            return iteratorResult();
        }
    });
    addIterator(ArrayIterator.prototype);

    /*
  var orderKeys = function orderKeys(a, b) {
    var aNumeric = String(ES.ToInteger(a)) === a;
    var bNumeric = String(ES.ToInteger(b)) === b;
    if (aNumeric && bNumeric) {
      return b - a;
    } else if (aNumeric && !bNumeric) {
      return -1;
    } else if (!aNumeric && bNumeric) {
      return 1;
    } else {
      return a.localeCompare(b);
    }
  };

  var getAllKeys = function getAllKeys(object) {
    var ownKeys = [];
    var keys = [];

    for (var key in object) {
      _push(_hasOwnProperty(object, key) ? ownKeys : keys, key);
    }
    _sort(ownKeys, orderKeys);
    _sort(keys, orderKeys);

    return _concat(ownKeys, keys);
  };
  */

    // note: this is positioned here because it depends on ArrayIterator
    var arrayOfSupportsSubclassing = Array.of === ArrayShims.of || (function () {
        // Detects a bug in Webkit nightly r181886
        var Foo = function Foo(len) { this.length = len; };
        Foo.prototype = [];
        var fooArr = Array.of.apply(Foo, [1, 2]);
        return fooArr instanceof Foo && fooArr.length === 2;
    }());
    if (!arrayOfSupportsSubclassing) {
        overrideNative(Array, 'of', ArrayShims.of);
    }

    var ArrayPrototypeShims = {
        copyWithin: function copyWithin(target, start) {
            var o = ES.ToObject(this);
            var len = ES.ToLength(o.length);
            var relativeTarget = ES.ToInteger(target);
            var relativeStart = ES.ToInteger(start);
            var to = relativeTarget < 0 ? _max(len + relativeTarget, 0) : _min(relativeTarget, len);
            var from = relativeStart < 0 ? _max(len + relativeStart, 0) : _min(relativeStart, len);
            var end;
            if (arguments.length > 2) {
                end = arguments[2];
            }
            var relativeEnd = typeof end === 'undefined' ? len : ES.ToInteger(end);
            var finalItem = relativeEnd < 0 ? _max(len + relativeEnd, 0) : _min(relativeEnd, len);
            var count = _min(finalItem - from, len - to);
            var direction = 1;
            if (from < to && to < (from + count)) {
                direction = -1;
                from += count - 1;
                to += count - 1;
            }
            while (count > 0) {
                if (from in o) {
                    o[to] = o[from];
                } else {
                    delete o[to];
                }
                from += direction;
                to += direction;
                count -= 1;
            }
            return o;
        },

        fill: function fill(value) {
            var start;
            if (arguments.length > 1) {
                start = arguments[1];
            }
            var end;
            if (arguments.length > 2) {
                end = arguments[2];
            }
            var O = ES.ToObject(this);
            var len = ES.ToLength(O.length);
            start = ES.ToInteger(typeof start === 'undefined' ? 0 : start);
            end = ES.ToInteger(typeof end === 'undefined' ? len : end);

            var relativeStart = start < 0 ? _max(len + start, 0) : _min(start, len);
            var relativeEnd = end < 0 ? len + end : end;

            for (var i = relativeStart; i < len && i < relativeEnd; ++i) {
                O[i] = value;
            }
            return O;
        },

        find: function find(predicate) {
            var list = ES.ToObject(this);
            var length = ES.ToLength(list.length);
            if (!ES.IsCallable(predicate)) {
                throw new TypeError('Array#find: predicate must be a function');
            }
            var thisArg = arguments.length > 1 ? arguments[1] : null;
            for (var i = 0, value; i < length; i++) {
                value = list[i];
                if (thisArg) {
                    if (_call(predicate, thisArg, value, i, list)) {
                        return value;
                    }
                } else if (predicate(value, i, list)) {
                    return value;
                }
            }
        },

        findIndex: function findIndex(predicate) {
            var list = ES.ToObject(this);
            var length = ES.ToLength(list.length);
            if (!ES.IsCallable(predicate)) {
                throw new TypeError('Array#findIndex: predicate must be a function');
            }
            var thisArg = arguments.length > 1 ? arguments[1] : null;
            for (var i = 0; i < length; i++) {
                if (thisArg) {
                    if (_call(predicate, thisArg, list[i], i, list)) {
                        return i;
                    }
                } else if (predicate(list[i], i, list)) {
                    return i;
                }
            }
            return -1;
        },

        keys: function keys() {
            return new ArrayIterator(this, 'key');
        },

        values: function values() {
            return new ArrayIterator(this, 'value');
        },

        entries: function entries() {
            return new ArrayIterator(this, 'entry');
        }
    };
    // Safari 7.1 defines Array#keys and Array#entries natively,
    // but the resulting ArrayIterator objects don't have a "next" method.
    if (Array.prototype.keys && !ES.IsCallable([1].keys().next)) {
        delete Array.prototype.keys;
    }
    if (Array.prototype.entries && !ES.IsCallable([1].entries().next)) {
        delete Array.prototype.entries;
    }

    // Chrome 38 defines Array#keys and Array#entries, and Array#@@iterator, but not Array#values
    if (Array.prototype.keys && Array.prototype.entries && !Array.prototype.values && Array.prototype[$iterator$]) {
        defineProperties(Array.prototype, {
            values: Array.prototype[$iterator$]
        });
        if (Type.symbol(Symbol.unscopables)) {
            Array.prototype[Symbol.unscopables].values = true;
        }
    }
    // Chrome 40 defines Array#values with the incorrect name, although Array#{keys,entries} have the correct name
    if (functionsHaveNames && Array.prototype.values && Array.prototype.values.name !== 'values') {
        var originalArrayPrototypeValues = Array.prototype.values;
        overrideNative(Array.prototype, 'values', function values() { return ES.Call(originalArrayPrototypeValues, this, arguments); });
        defineProperty(Array.prototype, $iterator$, Array.prototype.values, true);
    }
    defineProperties(Array.prototype, ArrayPrototypeShims);

    if (1 / [true].indexOf(true, -0) < 0) {
        // indexOf when given a position arg of -0 should return +0.
        // https://github.com/tc39/ecma262/pull/316
        defineProperty(Array.prototype, 'indexOf', function indexOf(searchElement) {
            var value = _arrayIndexOfApply(this, arguments);
            if (value === 0 && (1 / value) < 0) {
                return 0;
            }
            return value;
        }, true);
    }

    addIterator(Array.prototype, function () { return this.values(); });
    // Chrome defines keys/values/entries on Array, but doesn't give us
    // any way to identify its iterator.  So add our own shimmed field.
    if (Object.getPrototypeOf) {
        addIterator(Object.getPrototypeOf([].values()));
    }

    // note: this is positioned here because it relies on Array#entries
    var arrayFromSwallowsNegativeLengths = (function () {
        // Detects a Firefox bug in v32
        // https://bugzilla.mozilla.org/show_bug.cgi?id=1063993
        return valueOrFalseIfThrows(function () {
            return Array.from({ length: -1 }).length === 0;
        });
    }());
    var arrayFromHandlesIterables = (function () {
        // Detects a bug in Webkit nightly r181886
        var arr = Array.from([0].entries());
        return arr.length === 1 && isArray(arr[0]) && arr[0][0] === 0 && arr[0][1] === 0;
    }());
    if (!arrayFromSwallowsNegativeLengths || !arrayFromHandlesIterables) {
        overrideNative(Array, 'from', ArrayShims.from);
    }
    var arrayFromHandlesUndefinedMapFunction = (function () {
        // Microsoft Edge v0.11 throws if the mapFn argument is *provided* but undefined,
        // but the spec doesn't care if it's provided or not - undefined doesn't throw.
        return valueOrFalseIfThrows(function () {
            return Array.from([0], void 0);
        });
    }());
    if (!arrayFromHandlesUndefinedMapFunction) {
        var origArrayFrom = Array.from;
        overrideNative(Array, 'from', function from(items) {
            if (arguments.length > 1 && typeof arguments[1] !== 'undefined') {
                return ES.Call(origArrayFrom, this, arguments);
            } else {
                return _call(origArrayFrom, this, items);
            }
        });
    }

    var int32sAsOne = -(Math.pow(2, 32) - 1);
    var toLengthsCorrectly = function (method, reversed) {
        var obj = { length: int32sAsOne };
        obj[reversed ? (obj.length >>> 0) - 1 : 0] = true;
        return valueOrFalseIfThrows(function () {
            _call(method, obj, function () {
                // note: in nonconforming browsers, this will be called
                // -1 >>> 0 times, which is 4294967295, so the throw matters.
                throw new RangeError('should not reach here');
            }, []);
            return true;
        });
    };
    if (!toLengthsCorrectly(Array.prototype.forEach)) {
        var originalForEach = Array.prototype.forEach;
        overrideNative(Array.prototype, 'forEach', function forEach(callbackFn) {
            return ES.Call(originalForEach, this.length >= 0 ? this : [], arguments);
        });
    }
    if (!toLengthsCorrectly(Array.prototype.map)) {
        var originalMap = Array.prototype.map;
        overrideNative(Array.prototype, 'map', function map(callbackFn) {
            return ES.Call(originalMap, this.length >= 0 ? this : [], arguments);
        });
    }
    if (!toLengthsCorrectly(Array.prototype.filter)) {
        var originalFilter = Array.prototype.filter;
        overrideNative(Array.prototype, 'filter', function filter(callbackFn) {
            return ES.Call(originalFilter, this.length >= 0 ? this : [], arguments);
        });
    }
    if (!toLengthsCorrectly(Array.prototype.some)) {
        var originalSome = Array.prototype.some;
        overrideNative(Array.prototype, 'some', function some(callbackFn) {
            return ES.Call(originalSome, this.length >= 0 ? this : [], arguments);
        });
    }
    if (!toLengthsCorrectly(Array.prototype.every)) {
        var originalEvery = Array.prototype.every;
        overrideNative(Array.prototype, 'every', function every(callbackFn) {
            return ES.Call(originalEvery, this.length >= 0 ? this : [], arguments);
        });
    }
    if (!toLengthsCorrectly(Array.prototype.reduce)) {
        var originalReduce = Array.prototype.reduce;
        overrideNative(Array.prototype, 'reduce', function reduce(callbackFn) {
            return ES.Call(originalReduce, this.length >= 0 ? this : [], arguments);
        });
    }
    if (!toLengthsCorrectly(Array.prototype.reduceRight, true)) {
        var originalReduceRight = Array.prototype.reduceRight;
        overrideNative(Array.prototype, 'reduceRight', function reduceRight(callbackFn) {
            return ES.Call(originalReduceRight, this.length >= 0 ? this : [], arguments);
        });
    }

    var lacksOctalSupport = Number('0o10') !== 8;
    var lacksBinarySupport = Number('0b10') !== 2;
    var trimsNonWhitespace = _some(nonWS, function (c) {
        return Number(c + 0 + c) === 0;
    });
    if (lacksOctalSupport || lacksBinarySupport || trimsNonWhitespace) {
        var OrigNumber = Number;
        var binaryRegex = /^0b[01]+$/i;
        var octalRegex = /^0o[0-7]+$/i;
        // Note that in IE 8, RegExp.prototype.test doesn't seem to exist: ie, "test" is an own property of regexes. wtf.
        var isBinary = binaryRegex.test.bind(binaryRegex);
        var isOctal = octalRegex.test.bind(octalRegex);
        var toPrimitive = function (O, hint) { // need to replace this with `es-to-primitive/es6`
            var result;
            if (typeof O.valueOf === 'function') {
                result = O.valueOf();
                if (Type.primitive(result)) {
                    return result;
                }
            }
            if (typeof O.toString === 'function') {
                result = O.toString();
                if (Type.primitive(result)) {
                    return result;
                }
            }
            throw new TypeError('No default value');
        };
        var hasNonWS = nonWSregex.test.bind(nonWSregex);
        var isBadHex = isBadHexRegex.test.bind(isBadHexRegex);
        var NumberShim = (function () {
            // this is wrapped in an IIFE because of IE 6-8's wacky scoping issues with named function expressions.
            var NumberShim = function Number(value) {
                var primValue;
                if (arguments.length > 0) {
                    primValue = Type.primitive(value) ? value : toPrimitive(value, 'number');
                } else {
                    primValue = 0;
                }
                if (typeof primValue === 'string') {
                    primValue = ES.Call(trimShim, primValue);
                    if (isBinary(primValue)) {
                        primValue = parseInt(_strSlice(primValue, 2), 2);
                    } else if (isOctal(primValue)) {
                        primValue = parseInt(_strSlice(primValue, 2), 8);
                    } else if (hasNonWS(primValue) || isBadHex(primValue)) {
                        primValue = NaN;
                    }
                }
                var receiver = this;
                var valueOfSucceeds = valueOrFalseIfThrows(function () {
                    OrigNumber.prototype.valueOf.call(receiver);
                    return true;
                });
                if (receiver instanceof NumberShim && !valueOfSucceeds) {
                    return new OrigNumber(primValue);
                }
                return OrigNumber(primValue);
            };
            return NumberShim;
        }());
        wrapConstructor(OrigNumber, NumberShim, {});
        // this is necessary for ES3 browsers, where these properties are non-enumerable.
        defineProperties(NumberShim, {
            NaN: OrigNumber.NaN,
            MAX_VALUE: OrigNumber.MAX_VALUE,
            MIN_VALUE: OrigNumber.MIN_VALUE,
            NEGATIVE_INFINITY: OrigNumber.NEGATIVE_INFINITY,
            POSITIVE_INFINITY: OrigNumber.POSITIVE_INFINITY
        });
        /* eslint-disable no-undef, no-global-assign */
        Number = NumberShim;
        Value.redefine(globals, 'Number', NumberShim);
        /* eslint-enable no-undef, no-global-assign */
    }

    var maxSafeInteger = Math.pow(2, 53) - 1;
    defineProperties(Number, {
        MAX_SAFE_INTEGER: maxSafeInteger,
        MIN_SAFE_INTEGER: -maxSafeInteger,
        EPSILON: 2.220446049250313e-16,

        parseInt: globals.parseInt,
        parseFloat: globals.parseFloat,

        isFinite: numberIsFinite,

        isInteger: function isInteger(value) {
            return numberIsFinite(value) && ES.ToInteger(value) === value;
        },

        isSafeInteger: function isSafeInteger(value) {
            return Number.isInteger(value) && _abs(value) <= Number.MAX_SAFE_INTEGER;
        },

        isNaN: numberIsNaN
    });
    // Firefox 37 has a conforming Number.parseInt, but it's not === to the global parseInt (fixed in v40)
    defineProperty(Number, 'parseInt', globals.parseInt, Number.parseInt !== globals.parseInt);

    // Work around bugs in Array#find and Array#findIndex -- early
    // implementations skipped holes in sparse arrays. (Note that the
    // implementations of find/findIndex indirectly use shimmed
    // methods of Number, so this test has to happen down here.)
    /* eslint-disable no-sparse-arrays */
    if ([, 1].find(function () { return true; }) === 1) {
        overrideNative(Array.prototype, 'find', ArrayPrototypeShims.find);
    }
    if ([, 1].findIndex(function () { return true; }) !== 0) {
        overrideNative(Array.prototype, 'findIndex', ArrayPrototypeShims.findIndex);
    }
    /* eslint-enable no-sparse-arrays */

    var isEnumerableOn = Function.bind.call(Function.bind, Object.prototype.propertyIsEnumerable);
    var ensureEnumerable = function ensureEnumerable(obj, prop) {
        if (supportsDescriptors && isEnumerableOn(obj, prop)) {
            Object.defineProperty(obj, prop, { enumerable: false });
        }
    };
    var sliceArgs = function sliceArgs() {
        // per https://github.com/petkaantonov/bluebird/wiki/Optimization-killers#32-leaking-arguments
        // and https://gist.github.com/WebReflection/4327762cb87a8c634a29
        var initial = Number(this);
        var len = arguments.length;
        var desiredArgCount = len - initial;
        var args = new Array(desiredArgCount < 0 ? 0 : desiredArgCount);
        for (var i = initial; i < len; ++i) {
            args[i - initial] = arguments[i];
        }
        return args;
    };
    var assignTo = function assignTo(source) {
        return function assignToSource(target, key) {
            target[key] = source[key];
            return target;
        };
    };
    var assignReducer = function (target, source) {
        var sourceKeys = keys(Object(source));
        var symbols;
        if (ES.IsCallable(Object.getOwnPropertySymbols)) {
            symbols = _filter(Object.getOwnPropertySymbols(Object(source)), isEnumerableOn(source));
        }
        return _reduce(_concat(sourceKeys, symbols || []), assignTo(source), target);
    };

    var ObjectShims = {
        // 19.1.3.1
        assign: function (target, source) {
            var to = ES.ToObject(target, 'Cannot convert undefined or null to object');
            return _reduce(ES.Call(sliceArgs, 1, arguments), assignReducer, to);
        },

        // Added in WebKit in https://bugs.webkit.org/show_bug.cgi?id=143865
        is: function is(a, b) {
            return ES.SameValue(a, b);
        }
    };
    var assignHasPendingExceptions = Object.assign && Object.preventExtensions && (function () {
        // Firefox 37 still has "pending exception" logic in its Object.assign implementation,
        // which is 72% slower than our shim, and Firefox 40's native implementation.
        var thrower = Object.preventExtensions({ 1: 2 });
        try {
            Object.assign(thrower, 'xy');
        } catch (e) {
            return thrower[1] === 'y';
        }
    }());
    if (assignHasPendingExceptions) {
        overrideNative(Object, 'assign', ObjectShims.assign);
    }
    defineProperties(Object, ObjectShims);

    if (supportsDescriptors) {
        var ES5ObjectShims = {
            // 19.1.3.9
            // shim from https://gist.github.com/WebReflection/5593554
            setPrototypeOf: (function (Object, magic) {
                var set;

                var checkArgs = function (O, proto) {
                    if (!ES.TypeIsObject(O)) {
                        throw new TypeError('cannot set prototype on a non-object');
                    }
                    if (!(proto === null || ES.TypeIsObject(proto))) {
                        throw new TypeError('can only set prototype to an object or null' + proto);
                    }
                };

                var setPrototypeOf = function (O, proto) {
                    checkArgs(O, proto);
                    _call(set, O, proto);
                    return O;
                };

                try {
                    // this works already in Firefox and Safari
                    set = Object.getOwnPropertyDescriptor(Object.prototype, magic).set;
                    _call(set, {}, null);
                } catch (e) {
                    if (Object.prototype !== {}[magic]) {
                        // IE < 11 cannot be shimmed
                        return;
                    }
                    // probably Chrome or some old Mobile stock browser
                    set = function (proto) {
                        this[magic] = proto;
                    };
                    // please note that this will **not** work
                    // in those browsers that do not inherit
                    // __proto__ by mistake from Object.prototype
                    // in these cases we should probably throw an error
                    // or at least be informed about the issue
                    setPrototypeOf.polyfill = setPrototypeOf(
                        setPrototypeOf({}, null),
                        Object.prototype
                    ) instanceof Object;
                    // setPrototypeOf.polyfill === true means it works as meant
                    // setPrototypeOf.polyfill === false means it's not 100% reliable
                    // setPrototypeOf.polyfill === undefined
                    // or
                    // setPrototypeOf.polyfill ==  null means it's not a polyfill
                    // which means it works as expected
                    // we can even delete Object.prototype.__proto__;
                }
                return setPrototypeOf;
            }(Object, '__proto__'))
        };

        defineProperties(Object, ES5ObjectShims);
    }

    // Workaround bug in Opera 12 where setPrototypeOf(x, null) doesn't work,
    // but Object.create(null) does.
    if (Object.setPrototypeOf && Object.getPrototypeOf &&
        Object.getPrototypeOf(Object.setPrototypeOf({}, null)) !== null &&
        Object.getPrototypeOf(Object.create(null)) === null) {
        (function () {
            var FAKENULL = Object.create(null);
            var gpo = Object.getPrototypeOf;
            var spo = Object.setPrototypeOf;
            Object.getPrototypeOf = function (o) {
                var result = gpo(o);
                return result === FAKENULL ? null : result;
            };
            Object.setPrototypeOf = function (o, p) {
                var proto = p === null ? FAKENULL : p;
                return spo(o, proto);
            };
            Object.setPrototypeOf.polyfill = false;
        }());
    }

    var objectKeysAcceptsPrimitives = !throwsError(function () { return Object.keys('foo'); });
    if (!objectKeysAcceptsPrimitives) {
        var originalObjectKeys = Object.keys;
        overrideNative(Object, 'keys', function keys(value) {
            return originalObjectKeys(ES.ToObject(value));
        });
        keys = Object.keys;
    }
    var objectKeysRejectsRegex = throwsError(function () { return Object.keys(/a/g); });
    if (objectKeysRejectsRegex) {
        var regexRejectingObjectKeys = Object.keys;
        overrideNative(Object, 'keys', function keys(value) {
            if (Type.regex(value)) {
                var regexKeys = [];
                for (var k in value) {
                    if (_hasOwnProperty(value, k)) {
                        _push(regexKeys, k);
                    }
                }
                return regexKeys;
            }
            return regexRejectingObjectKeys(value);
        });
        keys = Object.keys;
    }

    if (Object.getOwnPropertyNames) {
        var objectGOPNAcceptsPrimitives = !throwsError(function () { return Object.getOwnPropertyNames('foo'); });
        if (!objectGOPNAcceptsPrimitives) {
            var cachedWindowNames = typeof window === 'object' ? Object.getOwnPropertyNames(window) : [];
            var originalObjectGetOwnPropertyNames = Object.getOwnPropertyNames;
            overrideNative(Object, 'getOwnPropertyNames', function getOwnPropertyNames(value) {
                var val = ES.ToObject(value);
                if (_toString(val) === '[object Window]') {
                    try {
                        return originalObjectGetOwnPropertyNames(val);
                    } catch (e) {
                        // IE bug where layout engine calls userland gOPN for cross-domain `window` objects
                        return _concat([], cachedWindowNames);
                    }
                }
                return originalObjectGetOwnPropertyNames(val);
            });
        }
    }
    if (Object.getOwnPropertyDescriptor) {
        var objectGOPDAcceptsPrimitives = !throwsError(function () { return Object.getOwnPropertyDescriptor('foo', 'bar'); });
        if (!objectGOPDAcceptsPrimitives) {
            var originalObjectGetOwnPropertyDescriptor = Object.getOwnPropertyDescriptor;
            overrideNative(Object, 'getOwnPropertyDescriptor', function getOwnPropertyDescriptor(value, property) {
                return originalObjectGetOwnPropertyDescriptor(ES.ToObject(value), property);
            });
        }
    }
    if (Object.seal) {
        var objectSealAcceptsPrimitives = !throwsError(function () { return Object.seal('foo'); });
        if (!objectSealAcceptsPrimitives) {
            var originalObjectSeal = Object.seal;
            overrideNative(Object, 'seal', function seal(value) {
                if (!ES.TypeIsObject(value)) { return value; }
                return originalObjectSeal(value);
            });
        }
    }
    if (Object.isSealed) {
        var objectIsSealedAcceptsPrimitives = !throwsError(function () { return Object.isSealed('foo'); });
        if (!objectIsSealedAcceptsPrimitives) {
            var originalObjectIsSealed = Object.isSealed;
            overrideNative(Object, 'isSealed', function isSealed(value) {
                if (!ES.TypeIsObject(value)) { return true; }
                return originalObjectIsSealed(value);
            });
        }
    }
    if (Object.freeze) {
        var objectFreezeAcceptsPrimitives = !throwsError(function () { return Object.freeze('foo'); });
        if (!objectFreezeAcceptsPrimitives) {
            var originalObjectFreeze = Object.freeze;
            overrideNative(Object, 'freeze', function freeze(value) {
                if (!ES.TypeIsObject(value)) { return value; }
                return originalObjectFreeze(value);
            });
        }
    }
    if (Object.isFrozen) {
        var objectIsFrozenAcceptsPrimitives = !throwsError(function () { return Object.isFrozen('foo'); });
        if (!objectIsFrozenAcceptsPrimitives) {
            var originalObjectIsFrozen = Object.isFrozen;
            overrideNative(Object, 'isFrozen', function isFrozen(value) {
                if (!ES.TypeIsObject(value)) { return true; }
                return originalObjectIsFrozen(value);
            });
        }
    }
    if (Object.preventExtensions) {
        var objectPreventExtensionsAcceptsPrimitives = !throwsError(function () { return Object.preventExtensions('foo'); });
        if (!objectPreventExtensionsAcceptsPrimitives) {
            var originalObjectPreventExtensions = Object.preventExtensions;
            overrideNative(Object, 'preventExtensions', function preventExtensions(value) {
                if (!ES.TypeIsObject(value)) { return value; }
                return originalObjectPreventExtensions(value);
            });
        }
    }
    if (Object.isExtensible) {
        var objectIsExtensibleAcceptsPrimitives = !throwsError(function () { return Object.isExtensible('foo'); });
        if (!objectIsExtensibleAcceptsPrimitives) {
            var originalObjectIsExtensible = Object.isExtensible;
            overrideNative(Object, 'isExtensible', function isExtensible(value) {
                if (!ES.TypeIsObject(value)) { return false; }
                return originalObjectIsExtensible(value);
            });
        }
    }
    if (Object.getPrototypeOf) {
        var objectGetProtoAcceptsPrimitives = !throwsError(function () { return Object.getPrototypeOf('foo'); });
        if (!objectGetProtoAcceptsPrimitives) {
            var originalGetProto = Object.getPrototypeOf;
            overrideNative(Object, 'getPrototypeOf', function getPrototypeOf(value) {
                return originalGetProto(ES.ToObject(value));
            });
        }
    }

    var hasFlags = supportsDescriptors && (function () {
        var desc = Object.getOwnPropertyDescriptor(RegExp.prototype, 'flags');
        return desc && ES.IsCallable(desc.get);
    }());
    if (supportsDescriptors && !hasFlags) {
        var regExpFlagsGetter = function flags() {
            if (!ES.TypeIsObject(this)) {
                throw new TypeError('Method called on incompatible type: must be an object.');
            }
            var result = '';
            if (this.global) {
                result += 'g';
            }
            if (this.ignoreCase) {
                result += 'i';
            }
            if (this.multiline) {
                result += 'm';
            }
            if (this.unicode) {
                result += 'u';
            }
            if (this.sticky) {
                result += 'y';
            }
            return result;
        };

        Value.getter(RegExp.prototype, 'flags', regExpFlagsGetter);
    }

    var regExpSupportsFlagsWithRegex = supportsDescriptors && valueOrFalseIfThrows(function () {
        return String(new RegExp(/a/g, 'i')) === '/a/i';
    });
    var regExpNeedsToSupportSymbolMatch = hasSymbols && supportsDescriptors && (function () {
        // Edge 0.12 supports flags fully, but does not support Symbol.match
        var regex = /./;
        regex[Symbol.match] = false;
        return RegExp(regex) === regex;
    }());

    var regexToStringIsGeneric = valueOrFalseIfThrows(function () {
        return RegExp.prototype.toString.call({ source: 'abc' }) === '/abc/';
    });
    var regexToStringSupportsGenericFlags = regexToStringIsGeneric && valueOrFalseIfThrows(function () {
        return RegExp.prototype.toString.call({ source: 'a', flags: 'b' }) === '/a/b';
    });
    if (!regexToStringIsGeneric || !regexToStringSupportsGenericFlags) {
        var origRegExpToString = RegExp.prototype.toString;
        defineProperty(RegExp.prototype, 'toString', function toString() {
            var R = ES.RequireObjectCoercible(this);
            if (Type.regex(R)) {
                return _call(origRegExpToString, R);
            }
            var pattern = $String(R.source);
            var flags = $String(R.flags);
            return '/' + pattern + '/' + flags;
        }, true);
        Value.preserveToString(RegExp.prototype.toString, origRegExpToString);
    }

    if (supportsDescriptors && (!regExpSupportsFlagsWithRegex || regExpNeedsToSupportSymbolMatch)) {
        var flagsGetter = Object.getOwnPropertyDescriptor(RegExp.prototype, 'flags').get;
        var sourceDesc = Object.getOwnPropertyDescriptor(RegExp.prototype, 'source') || {};
        var legacySourceGetter = function () {
            // prior to it being a getter, it's own + nonconfigurable
            return this.source;
        };
        var sourceGetter = ES.IsCallable(sourceDesc.get) ? sourceDesc.get : legacySourceGetter;

        var OrigRegExp = RegExp;
        var RegExpShim = (function () {
            return function RegExp(pattern, flags) {
                var patternIsRegExp = ES.IsRegExp(pattern);
                var calledWithNew = this instanceof RegExp;
                if (!calledWithNew && patternIsRegExp && typeof flags === 'undefined' && pattern.constructor === RegExp) {
                    return pattern;
                }

                var P = pattern;
                var F = flags;
                if (Type.regex(pattern)) {
                    P = ES.Call(sourceGetter, pattern);
                    F = typeof flags === 'undefined' ? ES.Call(flagsGetter, pattern) : flags;
                    return new RegExp(P, F);
                } else if (patternIsRegExp) {
                    P = pattern.source;
                    F = typeof flags === 'undefined' ? pattern.flags : flags;
                }
                return new OrigRegExp(pattern, flags);
            };
        }());
        wrapConstructor(OrigRegExp, RegExpShim, {
            $input: true // Chrome < v39 & Opera < 26 have a nonstandard "$input" property
        });
        /* eslint-disable no-undef, no-global-assign */
        RegExp = RegExpShim;
        Value.redefine(globals, 'RegExp', RegExpShim);
        /* eslint-enable no-undef, no-global-assign */
    }

    if (supportsDescriptors) {
        var regexGlobals = {
            input: '$_',
            lastMatch: '$&',
            lastParen: '$+',
            leftContext: '$`',
            rightContext: '$\''
        };
        _forEach(keys(regexGlobals), function (prop) {
            if (prop in RegExp && !(regexGlobals[prop] in RegExp)) {
                Value.getter(RegExp, regexGlobals[prop], function get() {
                    return RegExp[prop];
                });
            }
        });
    }
    addDefaultSpecies(RegExp);

    var inverseEpsilon = 1 / Number.EPSILON;
    var roundTiesToEven = function roundTiesToEven(n) {
        // Even though this reduces down to `return n`, it takes advantage of built-in rounding.
        return (n + inverseEpsilon) - inverseEpsilon;
    };
    var BINARY_32_EPSILON = Math.pow(2, -23);
    var BINARY_32_MAX_VALUE = Math.pow(2, 127) * (2 - BINARY_32_EPSILON);
    var BINARY_32_MIN_VALUE = Math.pow(2, -126);
    var E = Math.E;
    var LOG2E = Math.LOG2E;
    var LOG10E = Math.LOG10E;
    var numberCLZ = Number.prototype.clz;
    delete Number.prototype.clz; // Safari 8 has Number#clz

    var MathShims = {
        acosh: function acosh(value) {
            var x = Number(value);
            if (numberIsNaN(x) || value < 1) { return NaN; }
            if (x === 1) { return 0; }
            if (x === Infinity) { return x; }

            var xInvSquared = 1 / (x * x);
            if (x < 2) {
                return _log1p(x - 1 + (_sqrt(1 - xInvSquared) * x));
            }
            var halfX = x / 2;
            return _log1p(halfX + (_sqrt(1 - xInvSquared) * halfX) - 1) + (1 / LOG2E);
        },

        asinh: function asinh(value) {
            var x = Number(value);
            if (x === 0 || !globalIsFinite(x)) {
                return x;
            }

            var a = _abs(x);
            var aSquared = a * a;
            var s = _sign(x);
            if (a < 1) {
                return s * _log1p(a + (aSquared / (_sqrt(aSquared + 1) + 1)));
            }
            return s * (_log1p((a / 2) + (_sqrt(1 + (1 / aSquared)) * a / 2) - 1) + (1 / LOG2E));
        },

        atanh: function atanh(value) {
            var x = Number(value);

            if (x === 0) { return x; }
            if (x === -1) { return -Infinity; }
            if (x === 1) { return Infinity; }
            if (numberIsNaN(x) || x < -1 || x > 1) {
                return NaN;
            }

            var a = _abs(x);
            return _sign(x) * _log1p(2 * a / (1 - a)) / 2;
        },

        cbrt: function cbrt(value) {
            var x = Number(value);
            if (x === 0) { return x; }
            var negate = x < 0;
            var result;
            if (negate) { x = -x; }
            if (x === Infinity) {
                result = Infinity;
            } else {
                result = _exp(_log(x) / 3);
                // from http://en.wikipedia.org/wiki/Cube_root#Numerical_methods
                result = ((x / (result * result)) + (2 * result)) / 3;
            }
            return negate ? -result : result;
        },

        clz32: function clz32(value) {
            // See https://bugs.ecmascript.org/show_bug.cgi?id=2465
            var x = Number(value);
            var number = ES.ToUint32(x);
            if (number === 0) {
                return 32;
            }
            return numberCLZ ? ES.Call(numberCLZ, number) : 31 - _floor(_log(number + 0.5) * LOG2E);
        },

        cosh: function cosh(value) {
            var x = Number(value);
            if (x === 0) { return 1; } // +0 or -0
            if (numberIsNaN(x)) { return NaN; }
            if (!globalIsFinite(x)) { return Infinity; }

            var t = _exp(_abs(x) - 1);
            return (t + (1 / (t * E * E))) * (E / 2);
        },

        expm1: function expm1(value) {
            var x = Number(value);
            if (x === -Infinity) { return -1; }
            if (!globalIsFinite(x) || x === 0) { return x; }
            if (_abs(x) > 0.5) {
                return _exp(x) - 1;
            }
            // A more precise approximation using Taylor series expansion
            // from https://github.com/paulmillr/es6-shim/issues/314#issuecomment-70293986
            var t = x;
            var sum = 0;
            var n = 1;
            while (sum + t !== sum) {
                sum += t;
                n += 1;
                t *= x / n;
            }
            return sum;
        },

        hypot: function hypot(x, y) {
            var result = 0;
            var largest = 0;
            for (var i = 0; i < arguments.length; ++i) {
                var value = _abs(Number(arguments[i]));
                if (largest < value) {
                    result *= (largest / value) * (largest / value);
                    result += 1;
                    largest = value;
                } else {
                    result += value > 0 ? (value / largest) * (value / largest) : value;
                }
            }
            return largest === Infinity ? Infinity : largest * _sqrt(result);
        },

        log2: function log2(value) {
            return _log(value) * LOG2E;
        },

        log10: function log10(value) {
            return _log(value) * LOG10E;
        },

        log1p: _log1p,

        sign: _sign,

        sinh: function sinh(value) {
            var x = Number(value);
            if (!globalIsFinite(x) || x === 0) { return x; }

            var a = _abs(x);
            if (a < 1) {
                var u = Math.expm1(a);
                return _sign(x) * u * (1 + (1 / (u + 1))) / 2;
            }
            var t = _exp(a - 1);
            return _sign(x) * (t - (1 / (t * E * E))) * (E / 2);
        },

        tanh: function tanh(value) {
            var x = Number(value);
            if (numberIsNaN(x) || x === 0) { return x; }
            // can exit early at +-20 as JS loses precision for true value at this integer
            if (x >= 20) { return 1; }
            if (x <= -20) { return -1; }

            return (Math.expm1(x) - Math.expm1(-x)) / (_exp(x) + _exp(-x));
        },

        trunc: function trunc(value) {
            var x = Number(value);
            return x < 0 ? -_floor(-x) : _floor(x);
        },

        imul: function imul(x, y) {
            // taken from https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Math/imul
            var a = ES.ToUint32(x);
            var b = ES.ToUint32(y);
            var ah = (a >>> 16) & 0xffff;
            var al = a & 0xffff;
            var bh = (b >>> 16) & 0xffff;
            var bl = b & 0xffff;
            // the shift by 0 fixes the sign on the high part
            // the final |0 converts the unsigned value into a signed value
            return (al * bl) + ((((ah * bl) + (al * bh)) << 16) >>> 0) | 0;
        },

        fround: function fround(x) {
            var v = Number(x);
            if (v === 0 || v === Infinity || v === -Infinity || numberIsNaN(v)) {
                return v;
            }
            var sign = _sign(v);
            var abs = _abs(v);
            if (abs < BINARY_32_MIN_VALUE) {
                return sign * roundTiesToEven(abs / BINARY_32_MIN_VALUE / BINARY_32_EPSILON) * BINARY_32_MIN_VALUE * BINARY_32_EPSILON;
            }
            // Veltkamp's splitting (?)
            var a = (1 + (BINARY_32_EPSILON / Number.EPSILON)) * abs;
            var result = a - (a - abs);
            if (result > BINARY_32_MAX_VALUE || numberIsNaN(result)) {
                return sign * Infinity;
            }
            return sign * result;
        }
    };

    var withinULPDistance = function withinULPDistance(result, expected, distance) {
        return _abs(1 - (result / expected)) / Number.EPSILON < (distance || 8);
    };

    defineProperties(Math, MathShims);
    // Chrome < 40 sinh returns ∞ for large numbers
    defineProperty(Math, 'sinh', MathShims.sinh, Math.sinh(710) === Infinity);
    // Chrome < 40 cosh returns ∞ for large numbers
    defineProperty(Math, 'cosh', MathShims.cosh, Math.cosh(710) === Infinity);
    // IE 11 TP has an imprecise log1p: reports Math.log1p(-1e-17) as 0
    defineProperty(Math, 'log1p', MathShims.log1p, Math.log1p(-1e-17) !== -1e-17);
    // IE 11 TP has an imprecise asinh: reports Math.asinh(-1e7) as not exactly equal to -Math.asinh(1e7)
    defineProperty(Math, 'asinh', MathShims.asinh, Math.asinh(-1e7) !== -Math.asinh(1e7));
    // Chrome < 54 asinh returns ∞ for large numbers and should not
    defineProperty(Math, 'asinh', MathShims.asinh, Math.asinh(1e+300) === Infinity);
    // Chrome < 54 atanh incorrectly returns 0 for large numbers
    defineProperty(Math, 'atanh', MathShims.atanh, Math.atanh(1e-300) === 0);
    // Chrome 40 has an imprecise Math.tanh with very small numbers
    defineProperty(Math, 'tanh', MathShims.tanh, Math.tanh(-2e-17) !== -2e-17);
    // Chrome 40 loses Math.acosh precision with high numbers
    defineProperty(Math, 'acosh', MathShims.acosh, Math.acosh(Number.MAX_VALUE) === Infinity);
    // Chrome < 54 has an inaccurate acosh for EPSILON deltas
    defineProperty(Math, 'acosh', MathShims.acosh, !withinULPDistance(Math.acosh(1 + Number.EPSILON), Math.sqrt(2 * Number.EPSILON)));
    // Firefox 38 on Windows
    defineProperty(Math, 'cbrt', MathShims.cbrt, !withinULPDistance(Math.cbrt(1e-300), 1e-100));
    // node 0.11 has an imprecise Math.sinh with very small numbers
    defineProperty(Math, 'sinh', MathShims.sinh, Math.sinh(-2e-17) !== -2e-17);
    // FF 35 on Linux reports 22025.465794806725 for Math.expm1(10)
    var expm1OfTen = Math.expm1(10);
    defineProperty(Math, 'expm1', MathShims.expm1, expm1OfTen > 22025.465794806719 || expm1OfTen < 22025.4657948067165168);
    // node v12.11 - v12.15 report NaN
    defineProperty(Math, 'hypot', MathShims.hypot, Math.hypot(Infinity, NaN) !== Infinity);

    var origMathRound = Math.round;
    // breaks in e.g. Safari 8, Internet Explorer 11, Opera 12
    var roundHandlesBoundaryConditions = Math.round(0.5 - (Number.EPSILON / 4)) === 0 &&
        Math.round(-0.5 + (Number.EPSILON / 3.99)) === 1;

    // When engines use Math.floor(x + 0.5) internally, Math.round can be buggy for large integers.
    // This behavior should be governed by "round to nearest, ties to even mode"
    // see http://www.ecma-international.org/ecma-262/6.0/#sec-terms-and-definitions-number-type
    // These are the boundary cases where it breaks.
    var smallestPositiveNumberWhereRoundBreaks = inverseEpsilon + 1;
    var largestPositiveNumberWhereRoundBreaks = (2 * inverseEpsilon) - 1;
    var roundDoesNotIncreaseIntegers = [
        smallestPositiveNumberWhereRoundBreaks,
        largestPositiveNumberWhereRoundBreaks
    ].every(function (num) {
        return Math.round(num) === num;
    });
    defineProperty(Math, 'round', function round(x) {
        var floor = _floor(x);
        var ceil = floor === -1 ? -0 : floor + 1;
        return x - floor < 0.5 ? floor : ceil;
    }, !roundHandlesBoundaryConditions || !roundDoesNotIncreaseIntegers);
    Value.preserveToString(Math.round, origMathRound);

    var origImul = Math.imul;
    if (Math.imul(0xffffffff, 5) !== -5) {
        // Safari 6.1, at least, reports "0" for this value
        Math.imul = MathShims.imul;
        Value.preserveToString(Math.imul, origImul);
    }
    if (Math.imul.length !== 2) {
        // Safari 8.0.4 has a length of 1
        // fixed in https://bugs.webkit.org/show_bug.cgi?id=143658
        overrideNative(Math, 'imul', function imul(x, y) {
            return ES.Call(origImul, Math, arguments);
        });
    }

    // Promises
    // Simplest possible implementation; use a 3rd-party library if you
    // want the best possible speed and/or long stack traces.
    var PromiseShim = (function () {
        var setTimeout = globals.setTimeout;
        // some environments don't have setTimeout - no way to shim here.
        if (typeof setTimeout !== 'function' && typeof setTimeout !== 'object') { return; }

        ES.IsPromise = function (promise) {
            if (!ES.TypeIsObject(promise)) {
                return false;
            }
            if (typeof promise._promise === 'undefined') {
                return false; // uninitialized, or missing our hidden field.
            }
            return true;
        };

        // "PromiseCapability" in the spec is what most promise implementations
        // call a "deferred".
        var PromiseCapability = function (C) {
            if (!ES.IsConstructor(C)) {
                throw new TypeError('Bad promise constructor');
            }
            var capability = this;
            var resolver = function (resolve, reject) {
                if (capability.resolve !== void 0 || capability.reject !== void 0) {
                    throw new TypeError('Bad Promise implementation!');
                }
                capability.resolve = resolve;
                capability.reject = reject;
            };
            // Initialize fields to inform optimizers about the object shape.
            capability.resolve = void 0;
            capability.reject = void 0;
            capability.promise = new C(resolver);
            if (!(ES.IsCallable(capability.resolve) && ES.IsCallable(capability.reject))) {
                throw new TypeError('Bad promise constructor');
            }
        };

        // find an appropriate setImmediate-alike
        var makeZeroTimeout;
        if (typeof window !== 'undefined' && ES.IsCallable(window.postMessage)) {
            makeZeroTimeout = function () {
                // from http://dbaron.org/log/20100309-faster-timeouts
                var timeouts = [];
                var messageName = 'zero-timeout-message';
                var setZeroTimeout = function (fn) {
                    _push(timeouts, fn);
                    window.postMessage(messageName, '*');
                };
                var handleMessage = function (event) {
                    if (event.source === window && event.data === messageName) {
                        event.stopPropagation();
                        if (timeouts.length === 0) { return; }
                        var fn = _shift(timeouts);
                        fn();
                    }
                };
                window.addEventListener('message', handleMessage, true);
                return setZeroTimeout;
            };
        }
        var makePromiseAsap = function () {
            // An efficient task-scheduler based on a pre-existing Promise
            // implementation, which we can use even if we override the
            // global Promise below (in order to workaround bugs)
            // https://github.com/Raynos/observ-hash/issues/2#issuecomment-35857671
            var P = globals.Promise;
            var pr = P && P.resolve && P.resolve();
            return pr && function (task) {
                return pr.then(task);
            };
        };
        var enqueue = ES.IsCallable(globals.setImmediate) ?
            globals.setImmediate :
            typeof process === 'object' && process.nextTick ? process.nextTick : makePromiseAsap() ||
                (ES.IsCallable(makeZeroTimeout) ? makeZeroTimeout() : function (task) { setTimeout(task, 0); }); // fallback

        // Constants for Promise implementation
        var PROMISE_IDENTITY = function (x) { return x; };
        var PROMISE_THROWER = function (e) { throw e; };
        var PROMISE_PENDING = 0;
        var PROMISE_FULFILLED = 1;
        var PROMISE_REJECTED = 2;
        // We store fulfill/reject handlers and capabilities in a single array.
        var PROMISE_FULFILL_OFFSET = 0;
        var PROMISE_REJECT_OFFSET = 1;
        var PROMISE_CAPABILITY_OFFSET = 2;
        // This is used in an optimization for chaining promises via then.
        var PROMISE_FAKE_CAPABILITY = {};

        var enqueuePromiseReactionJob = function (handler, capability, argument) {
            enqueue(function () {
                promiseReactionJob(handler, capability, argument);
            });
        };

        var promiseReactionJob = function (handler, promiseCapability, argument) {
            var handlerResult, f;
            if (promiseCapability === PROMISE_FAKE_CAPABILITY) {
                // Fast case, when we don't actually need to chain through to a
                // (real) promiseCapability.
                return handler(argument);
            }
            try {
                handlerResult = handler(argument);
                f = promiseCapability.resolve;
            } catch (e) {
                handlerResult = e;
                f = promiseCapability.reject;
            }
            f(handlerResult);
        };

        var fulfillPromise = function (promise, value) {
            var _promise = promise._promise;
            var length = _promise.reactionLength;
            if (length > 0) {
                enqueuePromiseReactionJob(
                    _promise.fulfillReactionHandler0,
                    _promise.reactionCapability0,
                    value
                );
                _promise.fulfillReactionHandler0 = void 0;
                _promise.rejectReactions0 = void 0;
                _promise.reactionCapability0 = void 0;
                if (length > 1) {
                    for (var i = 1, idx = 0; i < length; i++, idx += 3) {
                        enqueuePromiseReactionJob(
                            _promise[idx + PROMISE_FULFILL_OFFSET],
                            _promise[idx + PROMISE_CAPABILITY_OFFSET],
                            value
                        );
                        promise[idx + PROMISE_FULFILL_OFFSET] = void 0;
                        promise[idx + PROMISE_REJECT_OFFSET] = void 0;
                        promise[idx + PROMISE_CAPABILITY_OFFSET] = void 0;
                    }
                }
            }
            _promise.result = value;
            _promise.state = PROMISE_FULFILLED;
            _promise.reactionLength = 0;
        };

        var rejectPromise = function (promise, reason) {
            var _promise = promise._promise;
            var length = _promise.reactionLength;
            if (length > 0) {
                enqueuePromiseReactionJob(
                    _promise.rejectReactionHandler0,
                    _promise.reactionCapability0,
                    reason
                );
                _promise.fulfillReactionHandler0 = void 0;
                _promise.rejectReactions0 = void 0;
                _promise.reactionCapability0 = void 0;
                if (length > 1) {
                    for (var i = 1, idx = 0; i < length; i++, idx += 3) {
                        enqueuePromiseReactionJob(
                            _promise[idx + PROMISE_REJECT_OFFSET],
                            _promise[idx + PROMISE_CAPABILITY_OFFSET],
                            reason
                        );
                        promise[idx + PROMISE_FULFILL_OFFSET] = void 0;
                        promise[idx + PROMISE_REJECT_OFFSET] = void 0;
                        promise[idx + PROMISE_CAPABILITY_OFFSET] = void 0;
                    }
                }
            }
            _promise.result = reason;
            _promise.state = PROMISE_REJECTED;
            _promise.reactionLength = 0;
        };

        var createResolvingFunctions = function (promise) {
            var alreadyResolved = false;
            var resolve = function (resolution) {
                var then;
                if (alreadyResolved) { return; }
                alreadyResolved = true;
                if (resolution === promise) {
                    return rejectPromise(promise, new TypeError('Self resolution'));
                }
                if (!ES.TypeIsObject(resolution)) {
                    return fulfillPromise(promise, resolution);
                }
                try {
                    then = resolution.then;
                } catch (e) {
                    return rejectPromise(promise, e);
                }
                if (!ES.IsCallable(then)) {
                    return fulfillPromise(promise, resolution);
                }
                enqueue(function () {
                    promiseResolveThenableJob(promise, resolution, then);
                });
            };
            var reject = function (reason) {
                if (alreadyResolved) { return; }
                alreadyResolved = true;
                return rejectPromise(promise, reason);
            };
            return { resolve: resolve, reject: reject };
        };

        var optimizedThen = function (then, thenable, resolve, reject) {
            // Optimization: since we discard the result, we can pass our
            // own then implementation a special hint to let it know it
            // doesn't have to create it.  (The PROMISE_FAKE_CAPABILITY
            // object is local to this implementation and unforgeable outside.)
            if (then === Promise$prototype$then) {
                _call(then, thenable, resolve, reject, PROMISE_FAKE_CAPABILITY);
            } else {
                _call(then, thenable, resolve, reject);
            }
        };
        var promiseResolveThenableJob = function (promise, thenable, then) {
            var resolvingFunctions = createResolvingFunctions(promise);
            var resolve = resolvingFunctions.resolve;
            var reject = resolvingFunctions.reject;
            try {
                optimizedThen(then, thenable, resolve, reject);
            } catch (e) {
                reject(e);
            }
        };

        var Promise$prototype, Promise$prototype$then;
        var Promise = (function () {
            var PromiseShim = function Promise(resolver) {
                if (!(this instanceof PromiseShim)) {
                    throw new TypeError('Constructor Promise requires "new"');
                }
                if (this && this._promise) {
                    throw new TypeError('Bad construction');
                }
                // see https://bugs.ecmascript.org/show_bug.cgi?id=2482
                if (!ES.IsCallable(resolver)) {
                    throw new TypeError('not a valid resolver');
                }
                var promise = emulateES6construct(this, PromiseShim, Promise$prototype, {
                    _promise: {
                        result: void 0,
                        state: PROMISE_PENDING,
                        // The first member of the "reactions" array is inlined here,
                        // since most promises only have one reaction.
                        // We've also exploded the 'reaction' object to inline the
                        // "handler" and "capability" fields, since both fulfill and
                        // reject reactions share the same capability.
                        reactionLength: 0,
                        fulfillReactionHandler0: void 0,
                        rejectReactionHandler0: void 0,
                        reactionCapability0: void 0
                    }
                });
                var resolvingFunctions = createResolvingFunctions(promise);
                var reject = resolvingFunctions.reject;
                try {
                    resolver(resolvingFunctions.resolve, reject);
                } catch (e) {
                    reject(e);
                }
                return promise;
            };
            return PromiseShim;
        }());
        Promise$prototype = Promise.prototype;

        var _promiseAllResolver = function (index, values, capability, remaining) {
            var alreadyCalled = false;
            return function (x) {
                if (alreadyCalled) { return; }
                alreadyCalled = true;
                values[index] = x;
                if ((--remaining.count) === 0) {
                    var resolve = capability.resolve;
                    resolve(values); // call w/ this===undefined
                }
            };
        };

        var performPromiseAll = function (iteratorRecord, C, resultCapability) {
            var it = iteratorRecord.iterator;
            var values = [];
            var remaining = { count: 1 };
            var next, nextValue;
            var index = 0;
            while (true) {
                try {
                    next = ES.IteratorStep(it);
                    if (next === false) {
                        iteratorRecord.done = true;
                        break;
                    }
                    nextValue = next.value;
                } catch (e) {
                    iteratorRecord.done = true;
                    throw e;
                }
                values[index] = void 0;
                var nextPromise = C.resolve(nextValue);
                var resolveElement = _promiseAllResolver(
                    index,
                    values,
                    resultCapability,
                    remaining
                );
                remaining.count += 1;
                optimizedThen(nextPromise.then, nextPromise, resolveElement, resultCapability.reject);
                index += 1;
            }
            if ((--remaining.count) === 0) {
                var resolve = resultCapability.resolve;
                resolve(values); // call w/ this===undefined
            }
            return resultCapability.promise;
        };

        var performPromiseRace = function (iteratorRecord, C, resultCapability) {
            var it = iteratorRecord.iterator;
            var next, nextValue, nextPromise;
            while (true) {
                try {
                    next = ES.IteratorStep(it);
                    if (next === false) {
                        // NOTE: If iterable has no items, resulting promise will never
                        // resolve; see:
                        // https://github.com/domenic/promises-unwrapping/issues/75
                        // https://bugs.ecmascript.org/show_bug.cgi?id=2515
                        iteratorRecord.done = true;
                        break;
                    }
                    nextValue = next.value;
                } catch (e) {
                    iteratorRecord.done = true;
                    throw e;
                }
                nextPromise = C.resolve(nextValue);
                optimizedThen(nextPromise.then, nextPromise, resultCapability.resolve, resultCapability.reject);
            }
            return resultCapability.promise;
        };

        defineProperties(Promise, {
            all: function all(iterable) {
                var C = this;
                if (!ES.TypeIsObject(C)) {
                    throw new TypeError('Promise is not object');
                }
                var capability = new PromiseCapability(C);
                var iterator, iteratorRecord;
                try {
                    iterator = ES.GetIterator(iterable);
                    iteratorRecord = { iterator: iterator, done: false };
                    return performPromiseAll(iteratorRecord, C, capability);
                } catch (e) {
                    var exception = e;
                    if (iteratorRecord && !iteratorRecord.done) {
                        try {
                            ES.IteratorClose(iterator, true);
                        } catch (ee) {
                            exception = ee;
                        }
                    }
                    var reject = capability.reject;
                    reject(exception);
                    return capability.promise;
                }
            },

            race: function race(iterable) {
                var C = this;
                if (!ES.TypeIsObject(C)) {
                    throw new TypeError('Promise is not object');
                }
                var capability = new PromiseCapability(C);
                var iterator, iteratorRecord;
                try {
                    iterator = ES.GetIterator(iterable);
                    iteratorRecord = { iterator: iterator, done: false };
                    return performPromiseRace(iteratorRecord, C, capability);
                } catch (e) {
                    var exception = e;
                    if (iteratorRecord && !iteratorRecord.done) {
                        try {
                            ES.IteratorClose(iterator, true);
                        } catch (ee) {
                            exception = ee;
                        }
                    }
                    var reject = capability.reject;
                    reject(exception);
                    return capability.promise;
                }
            },

            reject: function reject(reason) {
                var C = this;
                if (!ES.TypeIsObject(C)) {
                    throw new TypeError('Bad promise constructor');
                }
                var capability = new PromiseCapability(C);
                var rejectFunc = capability.reject;
                rejectFunc(reason); // call with this===undefined
                return capability.promise;
            },

            resolve: function resolve(v) {
                // See https://esdiscuss.org/topic/fixing-promise-resolve for spec
                var C = this;
                if (!ES.TypeIsObject(C)) {
                    throw new TypeError('Bad promise constructor');
                }
                if (ES.IsPromise(v)) {
                    var constructor = v.constructor;
                    if (constructor === C) {
                        return v;
                    }
                }
                var capability = new PromiseCapability(C);
                var resolveFunc = capability.resolve;
                resolveFunc(v); // call with this===undefined
                return capability.promise;
            }
        });

        defineProperties(Promise$prototype, {
            'catch': function (onRejected) {
                return this.then(null, onRejected);
            },

            then: function then(onFulfilled, onRejected) {
                var promise = this;
                if (!ES.IsPromise(promise)) { throw new TypeError('not a promise'); }
                var C = ES.SpeciesConstructor(promise, Promise);
                var resultCapability;
                var returnValueIsIgnored = arguments.length > 2 && arguments[2] === PROMISE_FAKE_CAPABILITY;
                if (returnValueIsIgnored && C === Promise) {
                    resultCapability = PROMISE_FAKE_CAPABILITY;
                } else {
                    resultCapability = new PromiseCapability(C);
                }
                // PerformPromiseThen(promise, onFulfilled, onRejected, resultCapability)
                // Note that we've split the 'reaction' object into its two
                // components, "capabilities" and "handler"
                // "capabilities" is always equal to `resultCapability`
                var fulfillReactionHandler = ES.IsCallable(onFulfilled) ? onFulfilled : PROMISE_IDENTITY;
                var rejectReactionHandler = ES.IsCallable(onRejected) ? onRejected : PROMISE_THROWER;
                var _promise = promise._promise;
                var value;
                if (_promise.state === PROMISE_PENDING) {
                    if (_promise.reactionLength === 0) {
                        _promise.fulfillReactionHandler0 = fulfillReactionHandler;
                        _promise.rejectReactionHandler0 = rejectReactionHandler;
                        _promise.reactionCapability0 = resultCapability;
                    } else {
                        var idx = 3 * (_promise.reactionLength - 1);
                        _promise[idx + PROMISE_FULFILL_OFFSET] = fulfillReactionHandler;
                        _promise[idx + PROMISE_REJECT_OFFSET] = rejectReactionHandler;
                        _promise[idx + PROMISE_CAPABILITY_OFFSET] = resultCapability;
                    }
                    _promise.reactionLength += 1;
                } else if (_promise.state === PROMISE_FULFILLED) {
                    value = _promise.result;
                    enqueuePromiseReactionJob(
                        fulfillReactionHandler,
                        resultCapability,
                        value
                    );
                } else if (_promise.state === PROMISE_REJECTED) {
                    value = _promise.result;
                    enqueuePromiseReactionJob(
                        rejectReactionHandler,
                        resultCapability,
                        value
                    );
                } else {
                    throw new TypeError('unexpected Promise state');
                }
                return resultCapability.promise;
            }
        });
        // This helps the optimizer by ensuring that methods which take
        // capabilities aren't polymorphic.
        PROMISE_FAKE_CAPABILITY = new PromiseCapability(Promise);
        Promise$prototype$then = Promise$prototype.then;

        return Promise;
    }());

    // Chrome's native Promise has extra methods that it shouldn't have. Let's remove them.
    if (globals.Promise) {
        delete globals.Promise.accept;
        delete globals.Promise.defer;
        delete globals.Promise.prototype.chain;
    }

    if (typeof PromiseShim === 'function') {
        // export the Promise constructor.
        defineProperties(globals, { Promise: PromiseShim });
        // In Chrome 33 (and thereabouts) Promise is defined, but the
        // implementation is buggy in a number of ways.  Let's check subclassing
        // support to see if we have a buggy implementation.
        var promiseSupportsSubclassing = supportsSubclassing(globals.Promise, function (S) {
            return S.resolve(42).then(function () {}) instanceof S;
        });
        var promiseIgnoresNonFunctionThenCallbacks = !throwsError(function () {
            return globals.Promise.reject(42).then(null, 5).then(null, noop);
        });
        var promiseRequiresObjectContext = throwsError(function () { return globals.Promise.call(3, noop); });
        // Promise.resolve() was errata'ed late in the ES6 process.
        // See: https://bugzilla.mozilla.org/show_bug.cgi?id=1170742
        //      https://code.google.com/p/v8/issues/detail?id=4161
        // It serves as a proxy for a number of other bugs in early Promise
        // implementations.
        var promiseResolveBroken = (function (Promise) {
            var p = Promise.resolve(5);
            p.constructor = {};
            var p2 = Promise.resolve(p);
            try {
                p2.then(null, noop).then(null, noop); // avoid "uncaught rejection" warnings in console
            } catch (e) {
                return true; // v8 native Promises break here https://code.google.com/p/chromium/issues/detail?id=575314
            }
            return p === p2; // This *should* be false!
        }(globals.Promise));

        // Chrome 46 (probably older too) does not retrieve a thenable's .then synchronously
        var getsThenSynchronously = supportsDescriptors && (function () {
            var count = 0;
            // eslint-disable-next-line getter-return
            var thenable = Object.defineProperty({}, 'then', { get: function () { count += 1; } });
            Promise.resolve(thenable);
            return count === 1;
        }());

        var BadResolverPromise = function BadResolverPromise(executor) {
            var p = new Promise(executor);
            executor(3, function () {});
            this.then = p.then;
            this.constructor = BadResolverPromise;
        };
        BadResolverPromise.prototype = Promise.prototype;
        BadResolverPromise.all = Promise.all;
        // Chrome Canary 49 (probably older too) has some implementation bugs
        var hasBadResolverPromise = valueOrFalseIfThrows(function () {
            return !!BadResolverPromise.all([1, 2]);
        });

        if (!promiseSupportsSubclassing || !promiseIgnoresNonFunctionThenCallbacks ||
            !promiseRequiresObjectContext || promiseResolveBroken ||
            !getsThenSynchronously || hasBadResolverPromise) {
            /* globals Promise: true */
            /* eslint-disable no-undef, no-global-assign */
            Promise = PromiseShim;
            /* eslint-enable no-undef, no-global-assign */
            overrideNative(globals, 'Promise', PromiseShim);
        }
        if (Promise.all.length !== 1) {
            var origAll = Promise.all;
            overrideNative(Promise, 'all', function all(iterable) {
                return ES.Call(origAll, this, arguments);
            });
        }
        if (Promise.race.length !== 1) {
            var origRace = Promise.race;
            overrideNative(Promise, 'race', function race(iterable) {
                return ES.Call(origRace, this, arguments);
            });
        }
        if (Promise.resolve.length !== 1) {
            var origResolve = Promise.resolve;
            overrideNative(Promise, 'resolve', function resolve(x) {
                return ES.Call(origResolve, this, arguments);
            });
        }
        if (Promise.reject.length !== 1) {
            var origReject = Promise.reject;
            overrideNative(Promise, 'reject', function reject(r) {
                return ES.Call(origReject, this, arguments);
            });
        }
        ensureEnumerable(Promise, 'all');
        ensureEnumerable(Promise, 'race');
        ensureEnumerable(Promise, 'resolve');
        ensureEnumerable(Promise, 'reject');
        addDefaultSpecies(Promise);
    }

    // Map and Set require a true ES5 environment
    // Their fast path also requires that the environment preserve
    // property insertion order, which is not guaranteed by the spec.
    var testOrder = function (a) {
        var b = keys(_reduce(a, function (o, k) {
            o[k] = true;
            return o;
        }, {}));
        return a.join(':') === b.join(':');
    };
    var preservesInsertionOrder = testOrder(['z', 'a', 'bb']);
    // some engines (eg, Chrome) only preserve insertion order for string keys
    var preservesNumericInsertionOrder = testOrder(['z', 1, 'a', '3', 2]);

    if (supportsDescriptors) {

        var fastkey = function fastkey(key, skipInsertionOrderCheck) {
            if (!skipInsertionOrderCheck && !preservesInsertionOrder) {
                return null;
            }
            if (isNullOrUndefined(key)) {
                return '^' + ES.ToString(key);
            } else if (typeof key === 'string') {
                return '$' + key;
            } else if (typeof key === 'number') {
                // note that -0 will get coerced to "0" when used as a property key
                if (!preservesNumericInsertionOrder) {
                    return 'n' + key;
                }
                return key;
            } else if (typeof key === 'boolean') {
                return 'b' + key;
            }
            return null;
        };

        var emptyObject = function emptyObject() {
            // accomodate some older not-quite-ES5 browsers
            return Object.create ? Object.create(null) : {};
        };

        var addIterableToMap = function addIterableToMap(MapConstructor, map, iterable) {
            if (isArray(iterable) || Type.string(iterable)) {
                _forEach(iterable, function (entry) {
                    if (!ES.TypeIsObject(entry)) {
                        throw new TypeError('Iterator value ' + entry + ' is not an entry object');
                    }
                    map.set(entry[0], entry[1]);
                });
            } else if (iterable instanceof MapConstructor) {
                _call(MapConstructor.prototype.forEach, iterable, function (value, key) {
                    map.set(key, value);
                });
            } else {
                var iter, adder;
                if (!isNullOrUndefined(iterable)) {
                    adder = map.set;
                    if (!ES.IsCallable(adder)) { throw new TypeError('bad map'); }
                    iter = ES.GetIterator(iterable);
                }
                if (typeof iter !== 'undefined') {
                    while (true) {
                        var next = ES.IteratorStep(iter);
                        if (next === false) { break; }
                        var nextItem = next.value;
                        try {
                            if (!ES.TypeIsObject(nextItem)) {
                                throw new TypeError('Iterator value ' + nextItem + ' is not an entry object');
                            }
                            _call(adder, map, nextItem[0], nextItem[1]);
                        } catch (e) {
                            ES.IteratorClose(iter, true);
                            throw e;
                        }
                    }
                }
            }
        };
        var addIterableToSet = function addIterableToSet(SetConstructor, set, iterable) {
            if (isArray(iterable) || Type.string(iterable)) {
                _forEach(iterable, function (value) {
                    set.add(value);
                });
            } else if (iterable instanceof SetConstructor) {
                _call(SetConstructor.prototype.forEach, iterable, function (value) {
                    set.add(value);
                });
            } else {
                var iter, adder;
                if (!isNullOrUndefined(iterable)) {
                    adder = set.add;
                    if (!ES.IsCallable(adder)) { throw new TypeError('bad set'); }
                    iter = ES.GetIterator(iterable);
                }
                if (typeof iter !== 'undefined') {
                    while (true) {
                        var next = ES.IteratorStep(iter);
                        if (next === false) { break; }
                        var nextValue = next.value;
                        try {
                            _call(adder, set, nextValue);
                        } catch (e) {
                            ES.IteratorClose(iter, true);
                            throw e;
                        }
                    }
                }
            }
        };

        var collectionShims = {
            Map: (function () {

                var empty = {};

                var MapEntry = function MapEntry(key, value) {
                    this.key = key;
                    this.value = value;
                    this.next = null;
                    this.prev = null;
                };

                MapEntry.prototype.isRemoved = function isRemoved() {
                    return this.key === empty;
                };

                var isMap = function isMap(map) {
                    return !!map._es6map;
                };

                var requireMapSlot = function requireMapSlot(map, method) {
                    if (!ES.TypeIsObject(map) || !isMap(map)) {
                        throw new TypeError('Method Map.prototype.' + method + ' called on incompatible receiver ' + ES.ToString(map));
                    }
                };

                var MapIterator = function MapIterator(map, kind) {
                    requireMapSlot(map, '[[MapIterator]]');
                    this.head = map._head;
                    this.i = this.head;
                    this.kind = kind;
                };

                MapIterator.prototype = {
                    isMapIterator: true,
                    next: function next() {
                        if (!this.isMapIterator) {
                            throw new TypeError('Not a MapIterator');
                        }
                        var i = this.i;
                        var kind = this.kind;
                        var head = this.head;
                        if (typeof this.i === 'undefined') {
                            return iteratorResult();
                        }
                        while (i.isRemoved() && i !== head) {
                            // back up off of removed entries
                            i = i.prev;
                        }
                        // advance to next unreturned element.
                        var result;
                        while (i.next !== head) {
                            i = i.next;
                            if (!i.isRemoved()) {
                                if (kind === 'key') {
                                    result = i.key;
                                } else if (kind === 'value') {
                                    result = i.value;
                                } else {
                                    result = [i.key, i.value];
                                }
                                this.i = i;
                                return iteratorResult(result);
                            }
                        }
                        // once the iterator is done, it is done forever.
                        this.i = void 0;
                        return iteratorResult();
                    }
                };
                addIterator(MapIterator.prototype);

                var Map$prototype;
                var MapShim = function Map() {
                    if (!(this instanceof Map)) {
                        throw new TypeError('Constructor Map requires "new"');
                    }
                    if (this && this._es6map) {
                        throw new TypeError('Bad construction');
                    }
                    var map = emulateES6construct(this, Map, Map$prototype, {
                        _es6map: true,
                        _head: null,
                        _map: OrigMap ? new OrigMap() : null,
                        _size: 0,
                        _storage: emptyObject()
                    });

                    var head = new MapEntry(null, null);
                    // circular doubly-linked list.
                    /* eslint no-multi-assign: 1 */
                    head.next = head.prev = head;
                    map._head = head;

                    // Optionally initialize map from iterable
                    if (arguments.length > 0) {
                        addIterableToMap(Map, map, arguments[0]);
                    }
                    return map;
                };
                Map$prototype = MapShim.prototype;

                Value.getter(Map$prototype, 'size', function () {
                    if (typeof this._size === 'undefined') {
                        throw new TypeError('size method called on incompatible Map');
                    }
                    return this._size;
                });

                defineProperties(Map$prototype, {
                    get: function get(key) {
                        requireMapSlot(this, 'get');
                        var entry;
                        var fkey = fastkey(key, true);
                        if (fkey !== null) {
                            // fast O(1) path
                            entry = this._storage[fkey];
                            if (entry) {
                                return entry.value;
                            } else {
                                return;
                            }
                        }
                        if (this._map) {
                            // fast object key path
                            entry = origMapGet.call(this._map, key);
                            if (entry) {
                                return entry.value;
                            } else {
                                return;
                            }
                        }
                        var head = this._head;
                        var i = head;
                        while ((i = i.next) !== head) {
                            if (ES.SameValueZero(i.key, key)) {
                                return i.value;
                            }
                        }
                    },

                    has: function has(key) {
                        requireMapSlot(this, 'has');
                        var fkey = fastkey(key, true);
                        if (fkey !== null) {
                            // fast O(1) path
                            return typeof this._storage[fkey] !== 'undefined';
                        }
                        if (this._map) {
                            // fast object key path
                            return origMapHas.call(this._map, key);
                        }
                        var head = this._head;
                        var i = head;
                        while ((i = i.next) !== head) {
                            if (ES.SameValueZero(i.key, key)) {
                                return true;
                            }
                        }
                        return false;
                    },

                    set: function set(key, value) {
                        requireMapSlot(this, 'set');
                        var head = this._head;
                        var i = head;
                        var entry;
                        var fkey = fastkey(key, true);
                        if (fkey !== null) {
                            // fast O(1) path
                            if (typeof this._storage[fkey] !== 'undefined') {
                                this._storage[fkey].value = value;
                                return this;
                            } else {
                                entry = this._storage[fkey] = new MapEntry(key, value); /* eslint no-multi-assign: 1 */
                                i = head.prev;
                                // fall through
                            }
                        } else if (this._map) {
                            // fast object key path
                            if (origMapHas.call(this._map, key)) {
                                origMapGet.call(this._map, key).value = value;
                            } else {
                                entry = new MapEntry(key, value);
                                origMapSet.call(this._map, key, entry);
                                i = head.prev;
                                // fall through
                            }
                        }
                        while ((i = i.next) !== head) {
                            if (ES.SameValueZero(i.key, key)) {
                                i.value = value;
                                return this;
                            }
                        }
                        entry = entry || new MapEntry(key, value);
                        if (ES.SameValue(-0, key)) {
                            entry.key = +0; // coerce -0 to +0 in entry
                        }
                        entry.next = this._head;
                        entry.prev = this._head.prev;
                        entry.prev.next = entry;
                        entry.next.prev = entry;
                        this._size += 1;
                        return this;
                    },

                    'delete': function (key) {
                        requireMapSlot(this, 'delete');
                        var head = this._head;
                        var i = head;
                        var fkey = fastkey(key, true);
                        if (fkey !== null) {
                            // fast O(1) path
                            if (typeof this._storage[fkey] === 'undefined') {
                                return false;
                            }
                            i = this._storage[fkey].prev;
                            delete this._storage[fkey];
                            // fall through
                        } else if (this._map) {
                            // fast object key path
                            if (!origMapHas.call(this._map, key)) {
                                return false;
                            }
                            i = origMapGet.call(this._map, key).prev;
                            origMapDelete.call(this._map, key);
                            // fall through
                        }
                        while ((i = i.next) !== head) {
                            if (ES.SameValueZero(i.key, key)) {
                                i.key = empty;
                                i.value = empty;
                                i.prev.next = i.next;
                                i.next.prev = i.prev;
                                this._size -= 1;
                                return true;
                            }
                        }
                        return false;
                    },

                    clear: function clear() {
                        /* eslint no-multi-assign: 1 */
                        requireMapSlot(this, 'clear');
                        this._map = OrigMap ? new OrigMap() : null;
                        this._size = 0;
                        this._storage = emptyObject();
                        var head = this._head;
                        var i = head;
                        var p = i.next;
                        while ((i = p) !== head) {
                            i.key = empty;
                            i.value = empty;
                            p = i.next;
                            i.next = i.prev = head;
                        }
                        head.next = head.prev = head;
                    },

                    keys: function keys() {
                        requireMapSlot(this, 'keys');
                        return new MapIterator(this, 'key');
                    },

                    values: function values() {
                        requireMapSlot(this, 'values');
                        return new MapIterator(this, 'value');
                    },

                    entries: function entries() {
                        requireMapSlot(this, 'entries');
                        return new MapIterator(this, 'key+value');
                    },

                    forEach: function forEach(callback) {
                        requireMapSlot(this, 'forEach');
                        var context = arguments.length > 1 ? arguments[1] : null;
                        var it = this.entries();
                        for (var entry = it.next(); !entry.done; entry = it.next()) {
                            if (context) {
                                _call(callback, context, entry.value[1], entry.value[0], this);
                            } else {
                                callback(entry.value[1], entry.value[0], this);
                            }
                        }
                    }
                });
                addIterator(Map$prototype, Map$prototype.entries);

                return MapShim;
            }()),

            Set: (function () {
                var isSet = function isSet(set) {
                    return set._es6set && typeof set._storage !== 'undefined';
                };
                var requireSetSlot = function requireSetSlot(set, method) {
                    if (!ES.TypeIsObject(set) || !isSet(set)) {
                        // https://github.com/paulmillr/es6-shim/issues/176
                        throw new TypeError('Set.prototype.' + method + ' called on incompatible receiver ' + ES.ToString(set));
                    }
                };

                // Creating a Map is expensive.  To speed up the common case of
                // Sets containing only string or numeric keys, we use an object
                // as backing storage and lazily create a full Map only when
                // required.
                var Set$prototype;
                var SetShim = function Set() {
                    if (!(this instanceof Set)) {
                        throw new TypeError('Constructor Set requires "new"');
                    }
                    if (this && this._es6set) {
                        throw new TypeError('Bad construction');
                    }
                    var set = emulateES6construct(this, Set, Set$prototype, {
                        _es6set: true,
                        '[[SetData]]': null,
                        _storage: emptyObject()
                    });
                    if (!set._es6set) {
                        throw new TypeError('bad set');
                    }

                    // Optionally initialize Set from iterable
                    if (arguments.length > 0) {
                        addIterableToSet(Set, set, arguments[0]);
                    }
                    return set;
                };
                Set$prototype = SetShim.prototype;

                var decodeKey = function (key) {
                    var k = key;
                    if (k === '^null') {
                        return null;
                    } else if (k === '^undefined') {
                        return void 0;
                    } else {
                        var first = k.charAt(0);
                        if (first === '$') {
                            return _strSlice(k, 1);
                        } else if (first === 'n') {
                            return +_strSlice(k, 1);
                        } else if (first === 'b') {
                            return k === 'btrue';
                        }
                    }
                    return +k;
                };
                // Switch from the object backing storage to a full Map.
                var ensureMap = function ensureMap(set) {
                    if (!set['[[SetData]]']) {
                        var m = new collectionShims.Map();
                        set['[[SetData]]'] = m;
                        _forEach(keys(set._storage), function (key) {
                            var k = decodeKey(key);
                            m.set(k, k);
                        });
                        set['[[SetData]]'] = m;
                    }
                    set._storage = null; // free old backing storage
                };

                Value.getter(SetShim.prototype, 'size', function () {
                    requireSetSlot(this, 'size');
                    if (this._storage) {
                        return keys(this._storage).length;
                    }
                    ensureMap(this);
                    return this['[[SetData]]'].size;
                });

                defineProperties(SetShim.prototype, {
                    has: function has(key) {
                        requireSetSlot(this, 'has');
                        var fkey;
                        if (this._storage && (fkey = fastkey(key)) !== null) {
                            return !!this._storage[fkey];
                        }
                        ensureMap(this);
                        return this['[[SetData]]'].has(key);
                    },

                    add: function add(key) {
                        requireSetSlot(this, 'add');
                        var fkey;
                        if (this._storage && (fkey = fastkey(key)) !== null) {
                            this._storage[fkey] = true;
                            return this;
                        }
                        ensureMap(this);
                        this['[[SetData]]'].set(key, key);
                        return this;
                    },

                    'delete': function (key) {
                        requireSetSlot(this, 'delete');
                        var fkey;
                        if (this._storage && (fkey = fastkey(key)) !== null) {
                            var hasFKey = _hasOwnProperty(this._storage, fkey);
                            return (delete this._storage[fkey]) && hasFKey;
                        }
                        ensureMap(this);
                        return this['[[SetData]]']['delete'](key);
                    },

                    clear: function clear() {
                        requireSetSlot(this, 'clear');
                        if (this._storage) {
                            this._storage = emptyObject();
                        }
                        if (this['[[SetData]]']) {
                            this['[[SetData]]'].clear();
                        }
                    },

                    values: function values() {
                        requireSetSlot(this, 'values');
                        ensureMap(this);
                        return new SetIterator(this['[[SetData]]'].values());
                    },

                    entries: function entries() {
                        requireSetSlot(this, 'entries');
                        ensureMap(this);
                        return new SetIterator(this['[[SetData]]'].entries());
                    },

                    forEach: function forEach(callback) {
                        requireSetSlot(this, 'forEach');
                        var context = arguments.length > 1 ? arguments[1] : null;
                        var entireSet = this;
                        ensureMap(entireSet);
                        this['[[SetData]]'].forEach(function (value, key) {
                            if (context) {
                                _call(callback, context, key, key, entireSet);
                            } else {
                                callback(key, key, entireSet);
                            }
                        });
                    }
                });
                defineProperty(SetShim.prototype, 'keys', SetShim.prototype.values, true);
                addIterator(SetShim.prototype, SetShim.prototype.values);

                var SetIterator = function SetIterator(it) {
                    this.it = it;
                };
                SetIterator.prototype = {
                    isSetIterator: true,
                    next: function next() {
                        if (!this.isSetIterator) {
                            throw new TypeError('Not a SetIterator');
                        }
                        return this.it.next();
                    }
                };
                addIterator(SetIterator.prototype);

                return SetShim;
            }())
        };

        var isGoogleTranslate = globals.Set && !Set.prototype['delete'] && Set.prototype.remove && Set.prototype.items && Set.prototype.map && Array.isArray(new Set().keys);
        if (isGoogleTranslate) {
            // special-case force removal of wildly invalid Set implementation in Google Translate iframes
            // see https://github.com/paulmillr/es6-shim/issues/438 / https://twitter.com/ljharb/status/849335573114363904
            globals.Set = collectionShims.Set;
        }
        if (globals.Map || globals.Set) {
            // Safari 8, for example, doesn't accept an iterable.
            var mapAcceptsArguments = valueOrFalseIfThrows(function () { return new Map([[1, 2]]).get(1) === 2; });
            if (!mapAcceptsArguments) {
                globals.Map = function Map() {
                    if (!(this instanceof Map)) {
                        throw new TypeError('Constructor Map requires "new"');
                    }
                    var m = new OrigMap();
                    if (arguments.length > 0) {
                        addIterableToMap(Map, m, arguments[0]);
                    }
                    delete m.constructor;
                    Object.setPrototypeOf(m, globals.Map.prototype);
                    return m;
                };
                globals.Map.prototype = create(OrigMap.prototype);
                defineProperty(globals.Map.prototype, 'constructor', globals.Map, true);
                Value.preserveToString(globals.Map, OrigMap);
            }
            var testMap = new Map();
            var mapUsesSameValueZero = (function () {
                // Chrome 38-42, node 0.11/0.12, iojs 1/2 also have a bug when the Map has a size > 4
                var m = new Map([[1, 0], [2, 0], [3, 0], [4, 0]]);
                m.set(-0, m);
                return m.get(0) === m && m.get(-0) === m && m.has(0) && m.has(-0);
            }());
            var mapSupportsChaining = testMap.set(1, 2) === testMap;
            if (!mapUsesSameValueZero || !mapSupportsChaining) {
                overrideNative(Map.prototype, 'set', function set(k, v) {
                    _call(origMapSet, this, k === 0 ? 0 : k, v);
                    return this;
                });
            }
            if (!mapUsesSameValueZero) {
                defineProperties(Map.prototype, {
                    get: function get(k) {
                        return _call(origMapGet, this, k === 0 ? 0 : k);
                    },
                    has: function has(k) {
                        return _call(origMapHas, this, k === 0 ? 0 : k);
                    }
                }, true);
                Value.preserveToString(Map.prototype.get, origMapGet);
                Value.preserveToString(Map.prototype.has, origMapHas);
            }
            var testSet = new Set();
            var setUsesSameValueZero = Set.prototype['delete'] && Set.prototype.add && Set.prototype.has && (function (s) {
                s['delete'](0);
                s.add(-0);
                return !s.has(0);
            }(testSet));
            var setSupportsChaining = testSet.add(1) === testSet;
            if (!setUsesSameValueZero || !setSupportsChaining) {
                var origSetAdd = Set.prototype.add;
                Set.prototype.add = function add(v) {
                    _call(origSetAdd, this, v === 0 ? 0 : v);
                    return this;
                };
                Value.preserveToString(Set.prototype.add, origSetAdd);
            }
            if (!setUsesSameValueZero) {
                var origSetHas = Set.prototype.has;
                Set.prototype.has = function has(v) {
                    return _call(origSetHas, this, v === 0 ? 0 : v);
                };
                Value.preserveToString(Set.prototype.has, origSetHas);
                var origSetDel = Set.prototype['delete'];
                Set.prototype['delete'] = function SetDelete(v) {
                    return _call(origSetDel, this, v === 0 ? 0 : v);
                };
                Value.preserveToString(Set.prototype['delete'], origSetDel);
            }
            var mapSupportsSubclassing = supportsSubclassing(globals.Map, function (M) {
                var m = new M([]);
                // Firefox 32 is ok with the instantiating the subclass but will
                // throw when the map is used.
                m.set(42, 42);
                return m instanceof M;
            });
            // without Object.setPrototypeOf, subclassing is not possible
            var mapFailsToSupportSubclassing = Object.setPrototypeOf && !mapSupportsSubclassing;
            var mapRequiresNew = (function () {
                try {
                    return !(globals.Map() instanceof globals.Map);
                } catch (e) {
                    return e instanceof TypeError;
                }
            }());
            if (globals.Map.length !== 0 || mapFailsToSupportSubclassing || !mapRequiresNew) {
                globals.Map = function Map() {
                    if (!(this instanceof Map)) {
                        throw new TypeError('Constructor Map requires "new"');
                    }
                    var m = new OrigMap();
                    if (arguments.length > 0) {
                        addIterableToMap(Map, m, arguments[0]);
                    }
                    delete m.constructor;
                    Object.setPrototypeOf(m, Map.prototype);
                    return m;
                };
                globals.Map.prototype = OrigMap.prototype;
                defineProperty(globals.Map.prototype, 'constructor', globals.Map, true);
                Value.preserveToString(globals.Map, OrigMap);
            }
            var setSupportsSubclassing = supportsSubclassing(globals.Set, function (S) {
                var s = new S([]);
                s.add(42, 42);
                return s instanceof S;
            });
            // without Object.setPrototypeOf, subclassing is not possible
            var setFailsToSupportSubclassing = Object.setPrototypeOf && !setSupportsSubclassing;
            var setRequiresNew = (function () {
                try {
                    return !(globals.Set() instanceof globals.Set);
                } catch (e) {
                    return e instanceof TypeError;
                }
            }());
            if (globals.Set.length !== 0 || setFailsToSupportSubclassing || !setRequiresNew) {
                var OrigSet = globals.Set;
                globals.Set = function Set() {
                    if (!(this instanceof Set)) {
                        throw new TypeError('Constructor Set requires "new"');
                    }
                    var s = new OrigSet();
                    if (arguments.length > 0) {
                        addIterableToSet(Set, s, arguments[0]);
                    }
                    delete s.constructor;
                    Object.setPrototypeOf(s, Set.prototype);
                    return s;
                };
                globals.Set.prototype = OrigSet.prototype;
                defineProperty(globals.Set.prototype, 'constructor', globals.Set, true);
                Value.preserveToString(globals.Set, OrigSet);
            }
            var newMap = new globals.Map();
            var mapIterationThrowsStopIterator = !valueOrFalseIfThrows(function () {
                return newMap.keys().next().done;
            });
            /*
        - In Firefox < 23, Map#size is a function.
        - In all current Firefox, Set#entries/keys/values & Map#clear do not exist
        - https://bugzilla.mozilla.org/show_bug.cgi?id=869996
        - In Firefox 24, Map and Set do not implement forEach
        - In Firefox 25 at least, Map and Set are callable without "new"
      */
            if (
                typeof globals.Map.prototype.clear !== 'function' ||
                new globals.Set().size !== 0 ||
                newMap.size !== 0 ||
                typeof globals.Map.prototype.keys !== 'function' ||
                typeof globals.Set.prototype.keys !== 'function' ||
                typeof globals.Map.prototype.forEach !== 'function' ||
                typeof globals.Set.prototype.forEach !== 'function' ||
                isCallableWithoutNew(globals.Map) ||
                isCallableWithoutNew(globals.Set) ||
                typeof newMap.keys().next !== 'function' || // Safari 8
                mapIterationThrowsStopIterator || // Firefox 25
                !mapSupportsSubclassing
            ) {
                defineProperties(globals, {
                    Map: collectionShims.Map,
                    Set: collectionShims.Set
                }, true);
            }

            if (globals.Set.prototype.keys !== globals.Set.prototype.values) {
                // Fixed in WebKit with https://bugs.webkit.org/show_bug.cgi?id=144190
                defineProperty(globals.Set.prototype, 'keys', globals.Set.prototype.values, true);
            }

            // Shim incomplete iterator implementations.
            addIterator(Object.getPrototypeOf((new globals.Map()).keys()));
            addIterator(Object.getPrototypeOf((new globals.Set()).keys()));

            if (functionsHaveNames && globals.Set.prototype.has.name !== 'has') {
                // Microsoft Edge v0.11.10074.0 is missing a name on Set#has
                var anonymousSetHas = globals.Set.prototype.has;
                overrideNative(globals.Set.prototype, 'has', function has(key) {
                    return _call(anonymousSetHas, this, key);
                });
            }
        }
        defineProperties(globals, collectionShims);
        addDefaultSpecies(globals.Map);
        addDefaultSpecies(globals.Set);
    }

    var throwUnlessTargetIsObject = function throwUnlessTargetIsObject(target) {
        if (!ES.TypeIsObject(target)) {
            throw new TypeError('target must be an object');
        }
    };

    // Some Reflect methods are basically the same as
    // those on the Object global, except that a TypeError is thrown if
    // target isn't an object. As well as returning a boolean indicating
    // the success of the operation.
    var ReflectShims = {
        // Apply method in a functional form.
        apply: function apply() {
            return ES.Call(ES.Call, null, arguments);
        },

        // New operator in a functional form.
        construct: function construct(constructor, args) {
            if (!ES.IsConstructor(constructor)) {
                throw new TypeError('First argument must be a constructor.');
            }
            var newTarget = arguments.length > 2 ? arguments[2] : constructor;
            if (!ES.IsConstructor(newTarget)) {
                throw new TypeError('new.target must be a constructor.');
            }
            return ES.Construct(constructor, args, newTarget, 'internal');
        },

        // When deleting a non-existent or configurable property,
        // true is returned.
        // When attempting to delete a non-configurable property,
        // it will return false.
        deleteProperty: function deleteProperty(target, key) {
            throwUnlessTargetIsObject(target);
            if (supportsDescriptors) {
                var desc = Object.getOwnPropertyDescriptor(target, key);

                if (desc && !desc.configurable) {
                    return false;
                }
            }

            // Will return true.
            return delete target[key];
        },

        has: function has(target, key) {
            throwUnlessTargetIsObject(target);
            return key in target;
        }
    };

    if (Object.getOwnPropertyNames) {
        Object.assign(ReflectShims, {
            // Basically the result of calling the internal [[OwnPropertyKeys]].
            // Concatenating propertyNames and propertySymbols should do the trick.
            // This should continue to work together with a Symbol shim
            // which overrides Object.getOwnPropertyNames and implements
            // Object.getOwnPropertySymbols.
            ownKeys: function ownKeys(target) {
                throwUnlessTargetIsObject(target);
                var keys = Object.getOwnPropertyNames(target);

                if (ES.IsCallable(Object.getOwnPropertySymbols)) {
                    _pushApply(keys, Object.getOwnPropertySymbols(target));
                }

                return keys;
            }
        });
    }

    var callAndCatchException = function ConvertExceptionToBoolean(func) {
        return !throwsError(func);
    };

    if (Object.preventExtensions) {
        Object.assign(ReflectShims, {
            isExtensible: function isExtensible(target) {
                throwUnlessTargetIsObject(target);
                return Object.isExtensible(target);
            },
            preventExtensions: function preventExtensions(target) {
                throwUnlessTargetIsObject(target);
                return callAndCatchException(function () {
                    return Object.preventExtensions(target);
                });
            }
        });
    }

    if (supportsDescriptors) {
        var internalGet = function get(target, key, receiver) {
            var desc = Object.getOwnPropertyDescriptor(target, key);

            if (!desc) {
                var parent = Object.getPrototypeOf(target);

                if (parent === null) {
                    return void 0;
                }

                return internalGet(parent, key, receiver);
            }

            if ('value' in desc) {
                return desc.value;
            }

            if (desc.get) {
                return ES.Call(desc.get, receiver);
            }

            return void 0;
        };

        var internalSet = function set(target, key, value, receiver) {
            var desc = Object.getOwnPropertyDescriptor(target, key);

            if (!desc) {
                var parent = Object.getPrototypeOf(target);

                if (parent !== null) {
                    return internalSet(parent, key, value, receiver);
                }

                desc = {
                    value: void 0,
                    writable: true,
                    enumerable: true,
                    configurable: true
                };
            }

            if ('value' in desc) {
                if (!desc.writable) {
                    return false;
                }

                if (!ES.TypeIsObject(receiver)) {
                    return false;
                }

                var existingDesc = Object.getOwnPropertyDescriptor(receiver, key);

                if (existingDesc) {
                    return Reflect.defineProperty(receiver, key, {
                        value: value
                    });
                } else {
                    return Reflect.defineProperty(receiver, key, {
                        value: value,
                        writable: true,
                        enumerable: true,
                        configurable: true
                    });
                }
            }

            if (desc.set) {
                _call(desc.set, receiver, value);
                return true;
            }

            return false;
        };

        Object.assign(ReflectShims, {
            defineProperty: function defineProperty(target, propertyKey, attributes) {
                throwUnlessTargetIsObject(target);
                return callAndCatchException(function () {
                    return Object.defineProperty(target, propertyKey, attributes);
                });
            },

            getOwnPropertyDescriptor: function getOwnPropertyDescriptor(target, propertyKey) {
                throwUnlessTargetIsObject(target);
                return Object.getOwnPropertyDescriptor(target, propertyKey);
            },

            // Syntax in a functional form.
            get: function get(target, key) {
                throwUnlessTargetIsObject(target);
                var receiver = arguments.length > 2 ? arguments[2] : target;

                return internalGet(target, key, receiver);
            },

            set: function set(target, key, value) {
                throwUnlessTargetIsObject(target);
                var receiver = arguments.length > 3 ? arguments[3] : target;

                return internalSet(target, key, value, receiver);
            }
        });
    }

    if (Object.getPrototypeOf) {
        var objectDotGetPrototypeOf = Object.getPrototypeOf;
        ReflectShims.getPrototypeOf = function getPrototypeOf(target) {
            throwUnlessTargetIsObject(target);
            return objectDotGetPrototypeOf(target);
        };
    }

    if (Object.setPrototypeOf && ReflectShims.getPrototypeOf) {
        var willCreateCircularPrototype = function (object, lastProto) {
            var proto = lastProto;
            while (proto) {
                if (object === proto) {
                    return true;
                }
                proto = ReflectShims.getPrototypeOf(proto);
            }
            return false;
        };

        Object.assign(ReflectShims, {
            // Sets the prototype of the given object.
            // Returns true on success, otherwise false.
            setPrototypeOf: function setPrototypeOf(object, proto) {
                throwUnlessTargetIsObject(object);
                if (proto !== null && !ES.TypeIsObject(proto)) {
                    throw new TypeError('proto must be an object or null');
                }

                // If they already are the same, we're done.
                if (proto === Reflect.getPrototypeOf(object)) {
                    return true;
                }

                // Cannot alter prototype if object not extensible.
                if (Reflect.isExtensible && !Reflect.isExtensible(object)) {
                    return false;
                }

                // Ensure that we do not create a circular prototype chain.
                if (willCreateCircularPrototype(object, proto)) {
                    return false;
                }

                Object.setPrototypeOf(object, proto);

                return true;
            }
        });
    }
    var defineOrOverrideReflectProperty = function (key, shim) {
        if (!ES.IsCallable(globals.Reflect[key])) {
            defineProperty(globals.Reflect, key, shim);
        } else {
            var acceptsPrimitives = valueOrFalseIfThrows(function () {
                globals.Reflect[key](1);
                globals.Reflect[key](NaN);
                globals.Reflect[key](true);
                return true;
            });
            if (acceptsPrimitives) {
                overrideNative(globals.Reflect, key, shim);
            }
        }
    };
    Object.keys(ReflectShims).forEach(function (key) {
        defineOrOverrideReflectProperty(key, ReflectShims[key]);
    });
    var originalReflectGetProto = globals.Reflect.getPrototypeOf;
    if (functionsHaveNames && originalReflectGetProto && originalReflectGetProto.name !== 'getPrototypeOf') {
        overrideNative(globals.Reflect, 'getPrototypeOf', function getPrototypeOf(target) {
            return _call(originalReflectGetProto, globals.Reflect, target);
        });
    }
    if (globals.Reflect.setPrototypeOf) {
        if (valueOrFalseIfThrows(function () {
            globals.Reflect.setPrototypeOf(1, {});
            return true;
        })) {
            overrideNative(globals.Reflect, 'setPrototypeOf', ReflectShims.setPrototypeOf);
        }
    }
    if (globals.Reflect.defineProperty) {
        if (!valueOrFalseIfThrows(function () {
            var basic = !globals.Reflect.defineProperty(1, 'test', { value: 1 });
            // "extensible" fails on Edge 0.12
            var extensible = typeof Object.preventExtensions !== 'function' || !globals.Reflect.defineProperty(Object.preventExtensions({}), 'test', {});
            return basic && extensible;
        })) {
            overrideNative(globals.Reflect, 'defineProperty', ReflectShims.defineProperty);
        }
    }
    if (globals.Reflect.construct) {
        if (!valueOrFalseIfThrows(function () {
            var F = function F() {};
            return globals.Reflect.construct(function () {}, [], F) instanceof F;
        })) {
            overrideNative(globals.Reflect, 'construct', ReflectShims.construct);
        }
    }

    if (String(new Date(NaN)) !== 'Invalid Date') {
        var dateToString = Date.prototype.toString;
        var shimmedDateToString = function toString() {
            var valueOf = +this;
            if (valueOf !== valueOf) {
                return 'Invalid Date';
            }
            return ES.Call(dateToString, this);
        };
        overrideNative(Date.prototype, 'toString', shimmedDateToString);
    }

    // Annex B HTML methods
    // http://www.ecma-international.org/ecma-262/6.0/#sec-additional-properties-of-the-string.prototype-object
    var stringHTMLshims = {
        anchor: function anchor(name) { return ES.CreateHTML(this, 'a', 'name', name); },
        big: function big() { return ES.CreateHTML(this, 'big', '', ''); },
        blink: function blink() { return ES.CreateHTML(this, 'blink', '', ''); },
        bold: function bold() { return ES.CreateHTML(this, 'b', '', ''); },
        fixed: function fixed() { return ES.CreateHTML(this, 'tt', '', ''); },
        fontcolor: function fontcolor(color) { return ES.CreateHTML(this, 'font', 'color', color); },
        fontsize: function fontsize(size) { return ES.CreateHTML(this, 'font', 'size', size); },
        italics: function italics() { return ES.CreateHTML(this, 'i', '', ''); },
        link: function link(url) { return ES.CreateHTML(this, 'a', 'href', url); },
        small: function small() { return ES.CreateHTML(this, 'small', '', ''); },
        strike: function strike() { return ES.CreateHTML(this, 'strike', '', ''); },
        sub: function sub() { return ES.CreateHTML(this, 'sub', '', ''); },
        sup: function sub() { return ES.CreateHTML(this, 'sup', '', ''); }
    };
    _forEach(Object.keys(stringHTMLshims), function (key) {
        var method = String.prototype[key];
        var shouldOverwrite = false;
        if (ES.IsCallable(method)) {
            var output = _call(method, '', ' " ');
            var quotesCount = _concat([], output.match(/"/g)).length;
            shouldOverwrite = output !== output.toLowerCase() || quotesCount > 2;
        } else {
            shouldOverwrite = true;
        }
        if (shouldOverwrite) {
            overrideNative(String.prototype, key, stringHTMLshims[key]);
        }
    });

    var JSONstringifiesSymbols = (function () {
        // Microsoft Edge v0.12 stringifies Symbols incorrectly
        if (!hasSymbols) { return false; } // Symbols are not supported
        var stringify = typeof JSON === 'object' && typeof JSON.stringify === 'function' ? JSON.stringify : null;
        if (!stringify) { return false; } // JSON.stringify is not supported
        if (typeof stringify(Symbol()) !== 'undefined') { return true; } // Symbols should become `undefined`
        if (stringify([Symbol()]) !== '[null]') { return true; } // Symbols in arrays should become `null`
        var obj = { a: Symbol() };
        obj[Symbol()] = true;
        if (stringify(obj) !== '{}') { return true; } // Symbol-valued keys *and* Symbol-valued properties should be omitted
        return false;
    }());
    var JSONstringifyAcceptsObjectSymbol = valueOrFalseIfThrows(function () {
        // Chrome 45 throws on stringifying object symbols
        if (!hasSymbols) { return true; } // Symbols are not supported
        return JSON.stringify(Object(Symbol())) === '{}' && JSON.stringify([Object(Symbol())]) === '[{}]';
    });
    if (JSONstringifiesSymbols || !JSONstringifyAcceptsObjectSymbol) {
        var origStringify = JSON.stringify;
        overrideNative(JSON, 'stringify', function stringify(value) {
            if (typeof value === 'symbol') { return; }
            var replacer;
            if (arguments.length > 1) {
                replacer = arguments[1];
            }
            var args = [value];
            if (!isArray(replacer)) {
                var replaceFn = ES.IsCallable(replacer) ? replacer : null;
                var wrappedReplacer = function (key, val) {
                    var parsedValue = replaceFn ? _call(replaceFn, this, key, val) : val;
                    if (typeof parsedValue !== 'symbol') {
                        if (Type.symbol(parsedValue)) {
                            return assignTo({})(parsedValue);
                        } else {
                            return parsedValue;
                        }
                    }
                };
                args.push(wrappedReplacer);
            } else {
                // create wrapped replacer that handles an array replacer?
                args.push(replacer);
            }
            if (arguments.length > 2) {
                args.push(arguments[2]);
            }
            return origStringify.apply(this, args);
        });
    }

    return globals;
}));

(function(global,factory){typeof exports==="object"&&typeof module!=="undefined"?factory(exports):typeof define==="function"&&define.amd?define(["exports"],factory):(global=typeof globalThis!=="undefined"?globalThis:global||self,factory(global.FormValidation={}))})(this,(function(exports){"use strict";function t$15(t){var e=t.length;var l=[[0,1,2,3,4,5,6,7,8,9],[0,2,4,6,8,1,3,5,7,9]];var n=0;var r=0;while(e--){r+=l[n][parseInt(t.charAt(e),10)];n=1-n}return r%10===0&&r>0}function t$14(t){var e=t.length;var n=5;for(var r=0;r<e;r++){n=((n||10)*2%11+parseInt(t.charAt(r),10))%10}return n===1}function t$13(t){var e=arguments.length>1&&arguments[1]!==undefined?arguments[1]:"0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";var n=t.length;var o=e.length;var l=Math.floor(o/2);for(var r=0;r<n;r++){l=((l||o)*2%(o+1)+e.indexOf(t.charAt(r)))%o}return l===1}function t$12(t){var e=[[0,1,2,3,4,5,6,7,8,9],[1,2,3,4,0,6,7,8,9,5],[2,3,4,0,1,7,8,9,5,6],[3,4,0,1,2,8,9,5,6,7],[4,0,1,2,3,9,5,6,7,8],[5,9,8,7,6,0,4,3,2,1],[6,5,9,8,7,1,0,4,3,2],[7,6,5,9,8,2,1,0,4,3],[8,7,6,5,9,3,2,1,0,4],[9,8,7,6,5,4,3,2,1,0]];var n=[[0,1,2,3,4,5,6,7,8,9],[1,5,7,6,2,8,3,0,9,4],[5,8,0,3,7,9,6,1,4,2],[8,9,1,6,0,4,3,5,2,7],[9,4,5,3,1,2,6,8,7,0],[4,2,8,6,5,7,3,9,0,1],[2,7,9,3,8,0,6,4,1,5],[7,0,4,6,9,1,3,2,5,8]];var o=t.reverse();var r=0;for(var _t=0;_t<o.length;_t++){r=e[r][n[_t%8][o[_t]]]}return r===0}var index$3={luhn:t$15,mod11And10:t$14,mod37And36:t$13,verhoeff:t$12};function _classCallCheck(instance,Constructor){if(!(instance instanceof Constructor)){throw new TypeError("Cannot call a class as a function")}}function _defineProperties(target,props){for(var i=0;i<props.length;i++){var descriptor=props[i];descriptor.enumerable=descriptor.enumerable||false;descriptor.configurable=true;if("value"in descriptor)descriptor.writable=true;Object.defineProperty(target,descriptor.key,descriptor)}}function _createClass(Constructor,protoProps,staticProps){if(protoProps)_defineProperties(Constructor.prototype,protoProps);if(staticProps)_defineProperties(Constructor,staticProps);Object.defineProperty(Constructor,"prototype",{writable:false});return Constructor}function _defineProperty(obj,key,value){if(key in obj){Object.defineProperty(obj,key,{value:value,enumerable:true,configurable:true,writable:true})}else{obj[key]=value}return obj}function _inherits(subClass,superClass){if(typeof superClass!=="function"&&superClass!==null){throw new TypeError("Super expression must either be null or a function")}subClass.prototype=Object.create(superClass&&superClass.prototype,{constructor:{value:subClass,writable:true,configurable:true}});Object.defineProperty(subClass,"prototype",{writable:false});if(superClass)_setPrototypeOf(subClass,superClass)}function _getPrototypeOf(o){_getPrototypeOf=Object.setPrototypeOf?Object.getPrototypeOf.bind():function _getPrototypeOf(o){return o.__proto__||Object.getPrototypeOf(o)};return _getPrototypeOf(o)}function _setPrototypeOf(o,p){_setPrototypeOf=Object.setPrototypeOf?Object.setPrototypeOf.bind():function _setPrototypeOf(o,p){o.__proto__=p;return o};return _setPrototypeOf(o,p)}function _isNativeReflectConstruct(){if(typeof Reflect==="undefined"||!Reflect.construct)return false;if(Reflect.construct.sham)return false;if(typeof Proxy==="function")return true;try{Boolean.prototype.valueOf.call(Reflect.construct(Boolean,[],(function(){})));return true}catch(e){return false}}function _assertThisInitialized(self){if(self===void 0){throw new ReferenceError("this hasn't been initialised - super() hasn't been called")}return self}function _possibleConstructorReturn(self,call){if(call&&(typeof call==="object"||typeof call==="function")){return call}else if(call!==void 0){throw new TypeError("Derived constructors may only return object or undefined")}return _assertThisInitialized(self)}function _createSuper(Derived){var hasNativeReflectConstruct=_isNativeReflectConstruct();return function _createSuperInternal(){var Super=_getPrototypeOf(Derived),result;if(hasNativeReflectConstruct){var NewTarget=_getPrototypeOf(this).constructor;result=Reflect.construct(Super,arguments,NewTarget)}else{result=Super.apply(this,arguments)}return _possibleConstructorReturn(this,result)}}function _unsupportedIterableToArray(o,minLen){if(!o)return;if(typeof o==="string")return _arrayLikeToArray(o,minLen);var n=Object.prototype.toString.call(o).slice(8,-1);if(n==="Object"&&o.constructor)n=o.constructor.name;if(n==="Map"||n==="Set")return Array.from(o);if(n==="Arguments"||/^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(n))return _arrayLikeToArray(o,minLen)}function _arrayLikeToArray(arr,len){if(len==null||len>arr.length)len=arr.length;for(var i=0,arr2=new Array(len);i<len;i++)arr2[i]=arr[i];return arr2}function _createForOfIteratorHelper(o,allowArrayLike){var it=typeof Symbol!=="undefined"&&o[Symbol.iterator]||o["@@iterator"];if(!it){if(Array.isArray(o)||(it=_unsupportedIterableToArray(o))||allowArrayLike&&o&&typeof o.length==="number"){if(it)o=it;var i=0;var F=function(){};return{s:F,n:function(){if(i>=o.length)return{done:true};return{done:false,value:o[i++]}},e:function(e){throw e},f:F}}throw new TypeError("Invalid attempt to iterate non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}var normalCompletion=true,didErr=false,err;return{s:function(){it=it.call(o)},n:function(){var step=it.next();normalCompletion=step.done;return step},e:function(e){didErr=true;err=e},f:function(){try{if(!normalCompletion&&it.return!=null)it.return()}finally{if(didErr)throw err}}}}function s$a(){return{fns:{},clear:function clear(){this.fns={}},emit:function emit(s){for(var _len=arguments.length,f=new Array(_len>1?_len-1:0),_key=1;_key<_len;_key++){f[_key-1]=arguments[_key]}(this.fns[s]||[]).map((function(s){return s.apply(s,f)}))},off:function off(s,f){if(this.fns[s]){var n=this.fns[s].indexOf(f);if(n>=0){this.fns[s].splice(n,1)}}},on:function on(s,f){(this.fns[s]=this.fns[s]||[]).push(f)}}}function t$11(){return{filters:{},add:function add(t,e){(this.filters[t]=this.filters[t]||[]).push(e)},clear:function clear(){this.filters={}},execute:function execute(t,e,i){if(!this.filters[t]||!this.filters[t].length){return e}var s=e;var r=this.filters[t];var l=r.length;for(var _t=0;_t<l;_t++){s=r[_t].apply(s,i)}return s},remove:function remove(t,e){if(this.filters[t]){this.filters[t]=this.filters[t].filter((function(t){return t!==e}))}}}}function e$H(e,t,r,n){var o=(r.getAttribute("type")||"").toLowerCase();var c=r.tagName.toLowerCase();if(c==="textarea"){return r.value}if(c==="select"){var _e=r;var _t=_e.selectedIndex;return _t>=0?_e.options.item(_t).value:""}if(c==="input"){if("radio"===o||"checkbox"===o){var _e2=n.filter((function(e){return e.checked})).length;return _e2===0?"":_e2+""}else{return r.value}}return""}function r$d(r,e){var t=Array.isArray(e)?e:[e];var a=r;t.forEach((function(r){a=a.replace("%s",r)}));return a}function s$9(){var s=function s(e){return parseFloat("".concat(e).replace(",","."))};return{validate:function validate(a){var t=a.value;if(t===""){return{valid:true}}var n=Object.assign({},{inclusive:true,message:""},a.options);var l=s(n.min);var o=s(n.max);return n.inclusive?{message:r$d(a.l10n?n.message||a.l10n.between["default"]:n.message,["".concat(l),"".concat(o)]),valid:parseFloat(t)>=l&&parseFloat(t)<=o}:{message:r$d(a.l10n?n.message||a.l10n.between.notInclusive:n.message,["".concat(l),"".concat(o)]),valid:parseFloat(t)>l&&parseFloat(t)<o}}}}function t$10(){return{validate:function validate(t){return{valid:true}}}}function t$$(t,n){if("function"===typeof t){return t.apply(this,n)}else if("string"===typeof t){var e=t;if("()"===e.substring(e.length-2)){e=e.substring(0,e.length-2)}var i=e.split(".");var o=i.pop();var f=window;var _iterator=_createForOfIteratorHelper(i),_step;try{for(_iterator.s();!(_step=_iterator.n()).done;){var _t=_step.value;f=f[_t]}}catch(err){_iterator.e(err)}finally{_iterator.f()}return typeof f[o]==="undefined"?null:f[o].apply(this,n)}}function o$4(){return{validate:function validate(o){var l=t$$(o.options.callback,[o]);return"boolean"===typeof l?{valid:l}:l}}}function t$_(){return{validate:function validate(t){var o="select"===t.element.tagName.toLowerCase()?t.element.querySelectorAll("option:checked").length:t.elements.filter((function(e){return e.checked})).length;var s=t.options.min?"".concat(t.options.min):"";var n=t.options.max?"".concat(t.options.max):"";var a=t.l10n?t.options.message||t.l10n.choice["default"]:t.options.message;var l=!(s&&o<parseInt(s,10)||n&&o>parseInt(n,10));switch(true){case!!s&&!!n:a=r$d(t.l10n?t.l10n.choice.between:t.options.message,[s,n]);break;case!!s:a=r$d(t.l10n?t.l10n.choice.more:t.options.message,s);break;case!!n:a=r$d(t.l10n?t.l10n.choice.less:t.options.message,n);break}return{message:a,valid:l}}}}var t$Z={AMERICAN_EXPRESS:{length:[15],prefix:["34","37"]},DANKORT:{length:[16],prefix:["5019"]},DINERS_CLUB:{length:[14],prefix:["300","301","302","303","304","305","36"]},DINERS_CLUB_US:{length:[16],prefix:["54","55"]},DISCOVER:{length:[16],prefix:["6011","622126","622127","622128","622129","62213","62214","62215","62216","62217","62218","62219","6222","6223","6224","6225","6226","6227","6228","62290","62291","622920","622921","622922","622923","622924","622925","644","645","646","647","648","649","65"]},ELO:{length:[16],prefix:["4011","4312","4389","4514","4573","4576","5041","5066","5067","509","6277","6362","6363","650","6516","6550"]},FORBRUGSFORENINGEN:{length:[16],prefix:["600722"]},JCB:{length:[16],prefix:["3528","3529","353","354","355","356","357","358"]},LASER:{length:[16,17,18,19],prefix:["6304","6706","6771","6709"]},MAESTRO:{length:[12,13,14,15,16,17,18,19],prefix:["5018","5020","5038","5868","6304","6759","6761","6762","6763","6764","6765","6766"]},MASTERCARD:{length:[16],prefix:["51","52","53","54","55"]},SOLO:{length:[16,18,19],prefix:["6334","6767"]},UNIONPAY:{length:[16,17,18,19],prefix:["622126","622127","622128","622129","62213","62214","62215","62216","62217","62218","62219","6222","6223","6224","6225","6226","6227","6228","62290","62291","622920","622921","622922","622923","622924","622925"]},VISA:{length:[16],prefix:["4"]},VISA_ELECTRON:{length:[16],prefix:["4026","417500","4405","4508","4844","4913","4917"]}};function l$2(){return{validate:function validate(l){if(l.value===""){return{meta:{type:null},valid:true}}if(/[^0-9-\s]+/.test(l.value)){return{meta:{type:null},valid:false}}var r=l.value.replace(/\D/g,"");if(!t$15(r)){return{meta:{type:null},valid:false}}for(var _i=0,_Object$keys=Object.keys(t$Z);_i<_Object$keys.length;_i++){var _e=_Object$keys[_i];for(var n in t$Z[_e].prefix){if(l.value.substr(0,t$Z[_e].prefix[n].length)===t$Z[_e].prefix[n]&&t$Z[_e].length.indexOf(r.length)!==-1){return{meta:{type:_e},valid:true}}}}return{meta:{type:null},valid:false}}}}function t$Y(t,e,n,r){if(isNaN(t)||isNaN(e)||isNaN(n)){return false}if(t<1e3||t>9999||e<=0||e>12){return false}var s=[31,t%400===0||t%100!==0&&t%4===0?29:28,31,30,31,30,31,31,30,31,30,31];if(n<=0||n>s[e-1]){return false}if(r===true){var _r=new Date;var _s=_r.getFullYear();var a=_r.getMonth();var u=_r.getDate();return t<_s||t===_s&&e-1<a||t===_s&&e-1===a&&n<u}return true}function n$1(){var n=function n(t,e,_n){var s=e.indexOf("YYYY");var a=e.indexOf("MM");var l=e.indexOf("DD");if(s===-1||a===-1||l===-1){return null}var o=t.split(" ");var r=o[0].split(_n);if(r.length<3){return null}var c=new Date(parseInt(r[s],10),parseInt(r[a],10)-1,parseInt(r[l],10));if(o.length>1){var _t=o[1].split(":");c.setHours(_t.length>0?parseInt(_t[0],10):0);c.setMinutes(_t.length>1?parseInt(_t[1],10):0);c.setSeconds(_t.length>2?parseInt(_t[2],10):0)}return c};var s=function s(t,e){var n=e.replace(/Y/g,"y").replace(/M/g,"m").replace(/D/g,"d").replace(/:m/g,":M").replace(/:mm/g,":MM").replace(/:S/,":s").replace(/:SS/,":ss");var s=t.getDate();var a=s<10?"0".concat(s):s;var l=t.getMonth()+1;var o=l<10?"0".concat(l):l;var r="".concat(t.getFullYear()).substr(2);var c=t.getFullYear();var i=t.getHours()%12||12;var g=i<10?"0".concat(i):i;var u=t.getHours();var m=u<10?"0".concat(u):u;var d=t.getMinutes();var f=d<10?"0".concat(d):d;var p=t.getSeconds();var h=p<10?"0".concat(p):p;var $={H:"".concat(u),HH:"".concat(m),M:"".concat(d),MM:"".concat(f),d:"".concat(s),dd:"".concat(a),h:"".concat(i),hh:"".concat(g),m:"".concat(l),mm:"".concat(o),s:"".concat(p),ss:"".concat(h),yy:"".concat(r),yyyy:"".concat(c)};return n.replace(/d{1,4}|m{1,4}|yy(?:yy)?|([HhMs])\1?|"[^"]*"|'[^']*'/g,(function(t){return $[t]?$[t]:t.slice(1,t.length-1)}))};return{validate:function validate(a){if(a.value===""){return{meta:{date:null},valid:true}}var l=Object.assign({},{format:a.element&&a.element.getAttribute("type")==="date"?"YYYY-MM-DD":"MM/DD/YYYY",message:""},a.options);var o=a.l10n?a.l10n.date["default"]:l.message;var r={message:"".concat(o),meta:{date:null},valid:false};var c=l.format.split(" ");var i=c.length>1?c[1]:null;var g=c.length>2?c[2]:null;var u=a.value.split(" ");var m=u[0];var d=u.length>1?u[1]:null;if(c.length!==u.length){return r}var f=l.separator||(m.indexOf("/")!==-1?"/":m.indexOf("-")!==-1?"-":m.indexOf(".")!==-1?".":"/");if(f===null||m.indexOf(f)===-1){return r}var p=m.split(f);var h=c[0].split(f);if(p.length!==h.length){return r}var $=p[h.indexOf("YYYY")];var M=p[h.indexOf("MM")];var Y=p[h.indexOf("DD")];if(!/^\d+$/.test($)||!/^\d+$/.test(M)||!/^\d+$/.test(Y)||$.length>4||M.length>2||Y.length>2){return r}var D=parseInt($,10);var x=parseInt(M,10);var y=parseInt(Y,10);if(!t$Y(D,x,y)){return r}var I=new Date(D,x-1,y);if(i){var _t2=d.split(":");if(i.split(":").length!==_t2.length){return r}var _e=_t2.length>0?_t2[0].length<=2&&/^\d+$/.test(_t2[0])?parseInt(_t2[0],10):-1:0;var _n2=_t2.length>1?_t2[1].length<=2&&/^\d+$/.test(_t2[1])?parseInt(_t2[1],10):-1:0;var _s=_t2.length>2?_t2[2].length<=2&&/^\d+$/.test(_t2[2])?parseInt(_t2[2],10):-1:0;if(_e===-1||_n2===-1||_s===-1){return r}if(_s<0||_s>60){return r}if(_e<0||_e>=24||g&&_e>12){return r}if(_n2<0||_n2>59){return r}I.setHours(_e);I.setMinutes(_n2);I.setSeconds(_s)}var O=typeof l.min==="function"?l.min():l.min;var v=O instanceof Date?O:O?n(O,h,f):I;var H=typeof l.max==="function"?l.max():l.max;var T=H instanceof Date?H:H?n(H,h,f):I;var S=O instanceof Date?s(v,l.format):O;var b=H instanceof Date?s(T,l.format):H;switch(true){case!!S&&!b:return{message:r$d(a.l10n?a.l10n.date.min:o,S),meta:{date:I},valid:I.getTime()>=v.getTime()};case!!b&&!S:return{message:r$d(a.l10n?a.l10n.date.max:o,b),meta:{date:I},valid:I.getTime()<=T.getTime()};case!!b&&!!S:return{message:r$d(a.l10n?a.l10n.date.range:o,[S,b]),meta:{date:I},valid:I.getTime()<=T.getTime()&&I.getTime()>=v.getTime()};default:return{message:"".concat(o),meta:{date:I},valid:true}}}}}function o$3(){return{validate:function validate(o){var t="function"===typeof o.options.compare?o.options.compare.call(this):o.options.compare;return{valid:t===""||o.value!==t}}}}function e$G(){return{validate:function validate(e){return{valid:e.value===""||/^\d+$/.test(e.value)}}}}function t$X(){var t=function t(_t3,e){var s=_t3.split(/"/);var l=s.length;var n=[];var r="";for(var _t=0;_t<l;_t++){if(_t%2===0){var _l=s[_t].split(e);var a=_l.length;if(a===1){r+=_l[0]}else{n.push(r+_l[0]);for(var _t2=1;_t2<a-1;_t2++){n.push(_l[_t2])}r=_l[a-1]}}else{r+='"'+s[_t];if(_t<l-1){r+='"'}}}n.push(r);return n};return{validate:function validate(e){if(e.value===""){return{valid:true}}var s=Object.assign({},{multiple:false,separator:/[,;]/},e.options);var l=/^(([^<>()[\]\\.,;:\s@"]+(\.[^<>()[\]\\.,;:\s@"]+)*)|(".+"))@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*$/;var n=s.multiple===true||"".concat(s.multiple)==="true";if(n){var _n=s.separator||/[,;]/;var r=t(e.value,_n);var a=r.length;for(var _t4=0;_t4<a;_t4++){if(!l.test(r[_t4])){return{valid:false}}}return{valid:true}}else{return{valid:l.test(e.value)}}}}}function e$F(){return{validate:function validate(e){if(e.value===""){return{valid:true}}var t;var i=e.options.extension?e.options.extension.toLowerCase().split(","):null;var s=e.options.type?e.options.type.toLowerCase().split(","):null;var n=window["File"]&&window["FileList"]&&window["FileReader"];if(n){var _n=e.element.files;var o=_n.length;var a=0;if(e.options.maxFiles&&o>parseInt("".concat(e.options.maxFiles),10)){return{meta:{error:"INVALID_MAX_FILES"},valid:false}}if(e.options.minFiles&&o<parseInt("".concat(e.options.minFiles),10)){return{meta:{error:"INVALID_MIN_FILES"},valid:false}}var r={};for(var l=0;l<o;l++){a+=_n[l].size;t=_n[l].name.substr(_n[l].name.lastIndexOf(".")+1);r={ext:t,file:_n[l],size:_n[l].size,type:_n[l].type};if(e.options.minSize&&_n[l].size<parseInt("".concat(e.options.minSize),10)){return{meta:Object.assign({},{error:"INVALID_MIN_SIZE"},r),valid:false}}if(e.options.maxSize&&_n[l].size>parseInt("".concat(e.options.maxSize),10)){return{meta:Object.assign({},{error:"INVALID_MAX_SIZE"},r),valid:false}}if(i&&i.indexOf(t.toLowerCase())===-1){return{meta:Object.assign({},{error:"INVALID_EXTENSION"},r),valid:false}}if(_n[l].type&&s&&s.indexOf(_n[l].type.toLowerCase())===-1){return{meta:Object.assign({},{error:"INVALID_TYPE"},r),valid:false}}}if(e.options.maxTotalSize&&a>parseInt("".concat(e.options.maxTotalSize),10)){return{meta:Object.assign({},{error:"INVALID_MAX_TOTAL_SIZE",totalSize:a},r),valid:false}}if(e.options.minTotalSize&&a<parseInt("".concat(e.options.minTotalSize),10)){return{meta:Object.assign({},{error:"INVALID_MIN_TOTAL_SIZE",totalSize:a},r),valid:false}}}else{t=e.value.substr(e.value.lastIndexOf(".")+1);if(i&&i.indexOf(t.toLowerCase())===-1){return{meta:{error:"INVALID_EXTENSION",ext:t},valid:false}}}return{valid:true}}}}function a$7(){return{validate:function validate(a){if(a.value===""){return{valid:true}}var s=Object.assign({},{inclusive:true,message:""},a.options);var t=parseFloat("".concat(s.min).replace(",","."));return s.inclusive?{message:r$d(a.l10n?s.message||a.l10n.greaterThan["default"]:s.message,"".concat(t)),valid:parseFloat(a.value)>=t}:{message:r$d(a.l10n?s.message||a.l10n.greaterThan.notInclusive:s.message,"".concat(t)),valid:parseFloat(a.value)>t}}}}function o$2(){return{validate:function validate(o){var t="function"===typeof o.options.compare?o.options.compare.call(this):o.options.compare;return{valid:t===""||o.value===t}}}}function a$6(){return{validate:function validate(a){if(a.value===""){return{valid:true}}var e=Object.assign({},{decimalSeparator:".",thousandsSeparator:""},a.options);var t=e.decimalSeparator==="."?"\\.":e.decimalSeparator;var r=e.thousandsSeparator==="."?"\\.":e.thousandsSeparator;var o=new RegExp("^-?[0-9]{1,3}(".concat(r,"[0-9]{3})*(").concat(t,"[0-9]+)?$"));var n=new RegExp(r,"g");var s="".concat(a.value);if(!o.test(s)){return{valid:false}}if(r){s=s.replace(n,"")}if(t){s=s.replace(t,".")}var i=parseFloat(s);return{valid:!isNaN(i)&&isFinite(i)&&Math.floor(i)===i}}}}function d(){return{validate:function validate(d){if(d.value===""){return{valid:true}}var a=Object.assign({},{ipv4:true,ipv6:true},d.options);var e=/^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)(\/([0-9]|[1-2][0-9]|3[0-2]))?$/;var s=/^\s*((([0-9A-Fa-f]{1,4}:){7}([0-9A-Fa-f]{1,4}|:))|(([0-9A-Fa-f]{1,4}:){6}(:[0-9A-Fa-f]{1,4}|((25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(\.(25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3})|:))|(([0-9A-Fa-f]{1,4}:){5}(((:[0-9A-Fa-f]{1,4}){1,2})|:((25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(\.(25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3})|:))|(([0-9A-Fa-f]{1,4}:){4}(((:[0-9A-Fa-f]{1,4}){1,3})|((:[0-9A-Fa-f]{1,4})?:((25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(\.(25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){3}(((:[0-9A-Fa-f]{1,4}){1,4})|((:[0-9A-Fa-f]{1,4}){0,2}:((25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(\.(25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){2}(((:[0-9A-Fa-f]{1,4}){1,5})|((:[0-9A-Fa-f]{1,4}){0,3}:((25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(\.(25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){1}(((:[0-9A-Fa-f]{1,4}){1,6})|((:[0-9A-Fa-f]{1,4}){0,4}:((25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(\.(25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:))|(:(((:[0-9A-Fa-f]{1,4}){1,7})|((:[0-9A-Fa-f]{1,4}){0,5}:((25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(\.(25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:)))(%.+)?\s*(\/(\d|\d\d|1[0-1]\d|12[0-8]))?$/;switch(true){case a.ipv4&&!a.ipv6:return{message:d.l10n?a.message||d.l10n.ip.ipv4:a.message,valid:e.test(d.value)};case!a.ipv4&&a.ipv6:return{message:d.l10n?a.message||d.l10n.ip.ipv6:a.message,valid:s.test(d.value)};case a.ipv4&&a.ipv6:default:return{message:d.l10n?a.message||d.l10n.ip["default"]:a.message,valid:e.test(d.value)||s.test(d.value)}}}}}function s$8(){return{validate:function validate(s){if(s.value===""){return{valid:true}}var a=Object.assign({},{inclusive:true,message:""},s.options);var l=parseFloat("".concat(a.max).replace(",","."));return a.inclusive?{message:r$d(s.l10n?a.message||s.l10n.lessThan["default"]:a.message,"".concat(l)),valid:parseFloat(s.value)<=l}:{message:r$d(s.l10n?a.message||s.l10n.lessThan.notInclusive:a.message,"".concat(l)),valid:parseFloat(s.value)<l}}}}function t$W(){return{validate:function validate(t){var n=!!t.options&&!!t.options.trim;var o=t.value;return{valid:!n&&o!==""||n&&o!==""&&o.trim()!==""}}}}function a$5(){return{validate:function validate(a){if(a.value===""){return{valid:true}}var e=Object.assign({},{decimalSeparator:".",thousandsSeparator:""},a.options);var t="".concat(a.value);if(t.substr(0,1)===e.decimalSeparator){t="0".concat(e.decimalSeparator).concat(t.substr(1))}else if(t.substr(0,2)==="-".concat(e.decimalSeparator)){t="-0".concat(e.decimalSeparator).concat(t.substr(2))}var r=e.decimalSeparator==="."?"\\.":e.decimalSeparator;var s=e.thousandsSeparator==="."?"\\.":e.thousandsSeparator;var i=new RegExp("^-?[0-9]{1,3}(".concat(s,"[0-9]{3})*(").concat(r,"[0-9]+)?$"));var o=new RegExp(s,"g");if(!i.test(t)){return{valid:false}}if(s){t=t.replace(o,"")}if(r){t=t.replace(r,".")}var l=parseFloat(t);return{valid:!isNaN(l)&&isFinite(l)}}}}function r$c(){return{validate:function validate(r){return t$$(r.options.promise,[r])}}}function e$E(){return{validate:function validate(e){if(e.value===""){return{valid:true}}var t=e.options.regexp;if(t instanceof RegExp){return{valid:t.test(e.value)}}else{var n=t.toString();var o=e.options.flags?new RegExp(n,e.options.flags):new RegExp(n);return{valid:o.test(e.value)}}}}}var e$D=function e(_e){return Array(_e).fill("").map((function(e){return Math.random().toString(36).charAt(2)})).join("")};function t$V(t,n){var o=function o(e){return Object.keys(e).map((function(t){return"".concat(encodeURIComponent(t),"=").concat(encodeURIComponent(e[t]))})).join("&")};return new Promise((function(s,a){var r=Object.assign({},{crossDomain:false,headers:{},method:"GET",params:{}},n);var d=Object.keys(r.params).map((function(e){return"".concat(encodeURIComponent(e),"=").concat(encodeURIComponent(r.params[e]))})).join("&");var c=t.indexOf("?");var i="GET"===r.method?"".concat(t).concat(c?"?":"&").concat(d):t;if(r.crossDomain){var _t=document.createElement("script");var _n="___FormValidationFetch_".concat(e$D(12),"___");window[_n]=function(e){delete window[_n];s(e)};_t.src="".concat(i).concat(c?"&":"?","callback=").concat(_n);_t.async=true;_t.addEventListener("load",(function(){_t.parentNode.removeChild(_t)}));_t.addEventListener("error",(function(){return a}));document.head.appendChild(_t)}else{var _e2=new XMLHttpRequest;_e2.open(r.method,i);_e2.setRequestHeader("X-Requested-With","XMLHttpRequest");if("POST"===r.method){_e2.setRequestHeader("Content-Type","application/x-www-form-urlencoded")}Object.keys(r.headers).forEach((function(t){return _e2.setRequestHeader(t,r.headers[t])}));_e2.addEventListener("load",(function(){s(JSON.parse(this.responseText))}));_e2.addEventListener("error",(function(){return a}));_e2.send(o(r.params))}}))}function a$4(){var a={crossDomain:false,data:{},headers:{},method:"GET",validKey:"valid"};return{validate:function validate(t){if(t.value===""){return Promise.resolve({valid:true})}var s=Object.assign({},a,t.options);var r=s.data;if("function"===typeof s.data){r=s.data.call(this,t)}if("string"===typeof r){r=JSON.parse(r)}r[s.name||t.field]=t.value;var o="function"===typeof s.url?s.url.call(this,t):s.url;return t$V(o,{crossDomain:s.crossDomain,headers:s.headers,method:s.method,params:r}).then((function(e){return Promise.resolve({message:e["message"],meta:e,valid:"".concat(e[s.validKey])==="true"})}))["catch"]((function(e){return Promise.reject({valid:false})}))}}}function e$C(){return{validate:function validate(e){if(e.value===""){return{valid:true}}var a=Object.assign({},{case:"lower"},e.options);var s=(a["case"]||"lower").toLowerCase();return{message:a.message||(e.l10n?"upper"===s?e.l10n.stringCase.upper:e.l10n.stringCase["default"]:a.message),valid:"upper"===s?e.value===e.value.toUpperCase():e.value===e.value.toLowerCase()}}}}function t$U(){var t=function t(e){var t=e.length;for(var s=e.length-1;s>=0;s--){var n=e.charCodeAt(s);if(n>127&&n<=2047){t++}else if(n>2047&&n<=65535){t+=2}if(n>=56320&&n<=57343){s--}}return"".concat(t)};return{validate:function validate(s){var n=Object.assign({},{message:"",trim:false,utf8Bytes:false},s.options);var a=n.trim===true||"".concat(n.trim)==="true"?s.value.trim():s.value;if(a===""){return{valid:true}}var r=n.min?"".concat(n.min):"";var l=n.max?"".concat(n.max):"";var i=n.utf8Bytes?t(a):a.length;var g=true;var m=s.l10n?n.message||s.l10n.stringLength["default"]:n.message;if(r&&i<parseInt(r,10)||l&&i>parseInt(l,10)){g=false}switch(true){case!!r&&!!l:m=r$d(s.l10n?n.message||s.l10n.stringLength.between:n.message,[r,l]);break;case!!r:m=r$d(s.l10n?n.message||s.l10n.stringLength.more:n.message,"".concat(parseInt(r,10)));break;case!!l:m=r$d(s.l10n?n.message||s.l10n.stringLength.less:n.message,"".concat(parseInt(l,10)));break}return{message:m,valid:g}}}}function t$T(){var t={allowEmptyProtocol:false,allowLocal:false,protocol:"http, https, ftp"};return{validate:function validate(o){if(o.value===""){return{valid:true}}var a=Object.assign({},t,o.options);var l=a.allowLocal===true||"".concat(a.allowLocal)==="true";var f=a.allowEmptyProtocol===true||"".concat(a.allowEmptyProtocol)==="true";var u=a.protocol.split(",").join("|").replace(/\s/g,"");var e=new RegExp("^"+"(?:(?:"+u+")://)"+(f?"?":"")+"(?:\\S+(?::\\S*)?@)?"+"(?:"+(l?"":"(?!(?:10|127)(?:\\.\\d{1,3}){3})"+"(?!(?:169\\.254|192\\.168)(?:\\.\\d{1,3}){2})"+"(?!172\\.(?:1[6-9]|2\\d|3[0-1])(?:\\.\\d{1,3}){2})")+"(?:[1-9]\\d?|1\\d\\d|2[01]\\d|22[0-3])"+"(?:\\.(?:1?\\d{1,2}|2[0-4]\\d|25[0-5])){2}"+"(?:\\.(?:[1-9]\\d?|1\\d\\d|2[0-4]\\d|25[0-4]))"+"|"+"(?:(?:[a-z\\u00a1-\\uffff0-9]-?)*[a-z\\u00a1-\\uffff0-9]+)"+"(?:\\.(?:[a-z\\u00a1-\\uffff0-9]-?)*[a-z\\u00a1-\\uffff0-9])*"+"(?:\\.(?:[a-z\\u00a1-\\uffff]{2,}))"+(l?"?":"")+")"+"(?::\\d{2,5})?"+"(?:/[^\\s]*)?$","i");return{valid:e.test(o.value)}}}}function a$3(){return{validate:function validate(a){return{valid:a.value===""||/^(?:[A-Za-z0-9+/]{4})*(?:[A-Za-z0-9+/]{2}==|[A-Za-z0-9+/]{3}=|[A-Za-z0-9+/]{4})$/.test(a.value)}}}}function a$2(){return{validate:function validate(a){return{valid:a.value===""||/^[a-zA-Z]{6}[a-zA-Z0-9]{2}([a-zA-Z0-9]{3})?$/.test(a.value)}}}}function e$B(){var e=["hex","rgb","rgba","hsl","hsla","keyword"];var a=["aliceblue","antiquewhite","aqua","aquamarine","azure","beige","bisque","black","blanchedalmond","blue","blueviolet","brown","burlywood","cadetblue","chartreuse","chocolate","coral","cornflowerblue","cornsilk","crimson","cyan","darkblue","darkcyan","darkgoldenrod","darkgray","darkgreen","darkgrey","darkkhaki","darkmagenta","darkolivegreen","darkorange","darkorchid","darkred","darksalmon","darkseagreen","darkslateblue","darkslategray","darkslategrey","darkturquoise","darkviolet","deeppink","deepskyblue","dimgray","dimgrey","dodgerblue","firebrick","floralwhite","forestgreen","fuchsia","gainsboro","ghostwhite","gold","goldenrod","gray","green","greenyellow","grey","honeydew","hotpink","indianred","indigo","ivory","khaki","lavender","lavenderblush","lawngreen","lemonchiffon","lightblue","lightcoral","lightcyan","lightgoldenrodyellow","lightgray","lightgreen","lightgrey","lightpink","lightsalmon","lightseagreen","lightskyblue","lightslategray","lightslategrey","lightsteelblue","lightyellow","lime","limegreen","linen","magenta","maroon","mediumaquamarine","mediumblue","mediumorchid","mediumpurple","mediumseagreen","mediumslateblue","mediumspringgreen","mediumturquoise","mediumvioletred","midnightblue","mintcream","mistyrose","moccasin","navajowhite","navy","oldlace","olive","olivedrab","orange","orangered","orchid","palegoldenrod","palegreen","paleturquoise","palevioletred","papayawhip","peachpuff","peru","pink","plum","powderblue","purple","red","rosybrown","royalblue","saddlebrown","salmon","sandybrown","seagreen","seashell","sienna","silver","skyblue","slateblue","slategray","slategrey","snow","springgreen","steelblue","tan","teal","thistle","tomato","transparent","turquoise","violet","wheat","white","whitesmoke","yellow","yellowgreen"];var r=function r(e){return/(^#[0-9A-F]{6}$)|(^#[0-9A-F]{3}$)/i.test(e)};var l=function l(e){return/^hsl\((\s*(-?\d+)\s*,)(\s*(\b(0?\d{1,2}|100)\b%)\s*,)(\s*(\b(0?\d{1,2}|100)\b%)\s*)\)$/.test(e)};var s=function s(e){return/^hsla\((\s*(-?\d+)\s*,)(\s*(\b(0?\d{1,2}|100)\b%)\s*,){2}(\s*(0?(\.\d+)?|1(\.0+)?)\s*)\)$/.test(e)};var t=function t(e){return a.indexOf(e)>=0};var i=function i(e){return/^rgb\((\s*(\b([01]?\d{1,2}|2[0-4]\d|25[0-5])\b)\s*,){2}(\s*(\b([01]?\d{1,2}|2[0-4]\d|25[0-5])\b)\s*)\)$/.test(e)||/^rgb\((\s*(\b(0?\d{1,2}|100)\b%)\s*,){2}(\s*(\b(0?\d{1,2}|100)\b%)\s*)\)$/.test(e)};var o=function o(e){return/^rgba\((\s*(\b([01]?\d{1,2}|2[0-4]\d|25[0-5])\b)\s*,){3}(\s*(0?(\.\d+)?|1(\.0+)?)\s*)\)$/.test(e)||/^rgba\((\s*(\b(0?\d{1,2}|100)\b%)\s*,){3}(\s*(0?(\.\d+)?|1(\.0+)?)\s*)\)$/.test(e)};return{validate:function validate(a){if(a.value===""){return{valid:true}}var n=typeof a.options.type==="string"?a.options.type.toString().replace(/s/g,"").split(","):a.options.type||e;var _iterator=_createForOfIteratorHelper(n),_step;try{for(_iterator.s();!(_step=_iterator.n()).done;){var d=_step.value;var _n=d.toLowerCase();if(e.indexOf(_n)===-1){continue}var g=true;switch(_n){case"hex":g=r(a.value);break;case"hsl":g=l(a.value);break;case"hsla":g=s(a.value);break;case"keyword":g=t(a.value);break;case"rgb":g=i(a.value);break;case"rgba":g=o(a.value);break}if(g){return{valid:true}}}}catch(err){_iterator.e(err)}finally{_iterator.f()}return{valid:false}}}}function t$S(){return{validate:function validate(t){if(t.value===""){return{valid:true}}var e=t.value.toUpperCase();if(!/^[0123456789ABCDEFGHJKLMNPQRSTUVWXYZ*@#]{9}$/.test(e)){return{valid:false}}var r=e.split("");var a=r.pop();var n=r.map((function(t){var e=t.charCodeAt(0);switch(true){case t==="*":return 36;case t==="@":return 37;case t==="#":return 38;case e>="A".charCodeAt(0)&&e<="Z".charCodeAt(0):return e-"A".charCodeAt(0)+10;default:return parseInt(t,10)}}));var c=n.map((function(t,e){var r=e%2===0?t:2*t;return Math.floor(r/10)+r%10})).reduce((function(t,e){return t+e}),0);var o=(10-c%10)%10;return{valid:a==="".concat(o)}}}}function e$A(){return{validate:function validate(e){if(e.value===""){return{valid:true}}if(!/^(\d{8}|\d{12}|\d{13}|\d{14})$/.test(e.value)){return{valid:false}}var t=e.value.length;var a=0;var l=t===8?[3,1]:[1,3];for(var r=0;r<t-1;r++){a+=parseInt(e.value.charAt(r),10)*l[r%2]}a=(10-a%10)%10;return{valid:"".concat(a)===e.value.charAt(t-1)}}}}function e$z(){var e={ANDOVER:["10","12"],ATLANTA:["60","67"],AUSTIN:["50","53"],BROOKHAVEN:["01","02","03","04","05","06","11","13","14","16","21","22","23","25","34","51","52","54","55","56","57","58","59","65"],CINCINNATI:["30","32","35","36","37","38","61"],FRESNO:["15","24"],INTERNET:["20","26","27","45","46","47"],KANSAS_CITY:["40","44"],MEMPHIS:["94","95"],OGDEN:["80","90"],PHILADELPHIA:["33","39","41","42","43","48","62","63","64","66","68","71","72","73","74","75","76","77","81","82","83","84","85","86","87","88","91","92","93","98","99"],SMALL_BUSINESS_ADMINISTRATION:["31"]};return{validate:function validate(t){if(t.value===""){return{meta:null,valid:true}}if(!/^[0-9]{2}-?[0-9]{7}$/.test(t.value)){return{meta:null,valid:false}}var a="".concat(t.value.substr(0,2));for(var _t in e){if(e[_t].indexOf(a)!==-1){return{meta:{campus:_t},valid:true}}}return{meta:null,valid:false}}}}function r$b(){return{validate:function validate(r){if(r.value===""){return{valid:true}}var t=r.value.toUpperCase();if(!/^[GRID:]*([0-9A-Z]{2})[-\s]*([0-9A-Z]{5})[-\s]*([0-9A-Z]{10})[-\s]*([0-9A-Z]{1})$/g.test(t)){return{valid:false}}t=t.replace(/\s/g,"").replace(/-/g,"");if("GRID:"===t.substr(0,5)){t=t.substr(5)}return{valid:t$13(t)}}}}function e$y(){return{validate:function validate(e){return{valid:e.value===""||/^[0-9a-fA-F]+$/.test(e.value)}}}}function Z(){var Z={AD:"AD[0-9]{2}[0-9]{4}[0-9]{4}[A-Z0-9]{12}",AE:"AE[0-9]{2}[0-9]{3}[0-9]{16}",AL:"AL[0-9]{2}[0-9]{8}[A-Z0-9]{16}",AO:"AO[0-9]{2}[0-9]{21}",AT:"AT[0-9]{2}[0-9]{5}[0-9]{11}",AZ:"AZ[0-9]{2}[A-Z]{4}[A-Z0-9]{20}",BA:"BA[0-9]{2}[0-9]{3}[0-9]{3}[0-9]{8}[0-9]{2}",BE:"BE[0-9]{2}[0-9]{3}[0-9]{7}[0-9]{2}",BF:"BF[0-9]{2}[0-9]{23}",BG:"BG[0-9]{2}[A-Z]{4}[0-9]{4}[0-9]{2}[A-Z0-9]{8}",BH:"BH[0-9]{2}[A-Z]{4}[A-Z0-9]{14}",BI:"BI[0-9]{2}[0-9]{12}",BJ:"BJ[0-9]{2}[A-Z]{1}[0-9]{23}",BR:"BR[0-9]{2}[0-9]{8}[0-9]{5}[0-9]{10}[A-Z][A-Z0-9]",CH:"CH[0-9]{2}[0-9]{5}[A-Z0-9]{12}",CI:"CI[0-9]{2}[A-Z]{1}[0-9]{23}",CM:"CM[0-9]{2}[0-9]{23}",CR:"CR[0-9]{2}[0-9][0-9]{3}[0-9]{14}",CV:"CV[0-9]{2}[0-9]{21}",CY:"CY[0-9]{2}[0-9]{3}[0-9]{5}[A-Z0-9]{16}",CZ:"CZ[0-9]{2}[0-9]{20}",DE:"DE[0-9]{2}[0-9]{8}[0-9]{10}",DK:"DK[0-9]{2}[0-9]{14}",DO:"DO[0-9]{2}[A-Z0-9]{4}[0-9]{20}",DZ:"DZ[0-9]{2}[0-9]{20}",EE:"EE[0-9]{2}[0-9]{2}[0-9]{2}[0-9]{11}[0-9]{1}",ES:"ES[0-9]{2}[0-9]{4}[0-9]{4}[0-9]{1}[0-9]{1}[0-9]{10}",FI:"FI[0-9]{2}[0-9]{6}[0-9]{7}[0-9]{1}",FO:"FO[0-9]{2}[0-9]{4}[0-9]{9}[0-9]{1}",FR:"FR[0-9]{2}[0-9]{5}[0-9]{5}[A-Z0-9]{11}[0-9]{2}",GB:"GB[0-9]{2}[A-Z]{4}[0-9]{6}[0-9]{8}",GE:"GE[0-9]{2}[A-Z]{2}[0-9]{16}",GI:"GI[0-9]{2}[A-Z]{4}[A-Z0-9]{15}",GL:"GL[0-9]{2}[0-9]{4}[0-9]{9}[0-9]{1}",GR:"GR[0-9]{2}[0-9]{3}[0-9]{4}[A-Z0-9]{16}",GT:"GT[0-9]{2}[A-Z0-9]{4}[A-Z0-9]{20}",HR:"HR[0-9]{2}[0-9]{7}[0-9]{10}",HU:"HU[0-9]{2}[0-9]{3}[0-9]{4}[0-9]{1}[0-9]{15}[0-9]{1}",IE:"IE[0-9]{2}[A-Z]{4}[0-9]{6}[0-9]{8}",IL:"IL[0-9]{2}[0-9]{3}[0-9]{3}[0-9]{13}",IR:"IR[0-9]{2}[0-9]{22}",IS:"IS[0-9]{2}[0-9]{4}[0-9]{2}[0-9]{6}[0-9]{10}",IT:"IT[0-9]{2}[A-Z]{1}[0-9]{5}[0-9]{5}[A-Z0-9]{12}",JO:"JO[0-9]{2}[A-Z]{4}[0-9]{4}[0]{8}[A-Z0-9]{10}",KW:"KW[0-9]{2}[A-Z]{4}[0-9]{22}",KZ:"KZ[0-9]{2}[0-9]{3}[A-Z0-9]{13}",LB:"LB[0-9]{2}[0-9]{4}[A-Z0-9]{20}",LI:"LI[0-9]{2}[0-9]{5}[A-Z0-9]{12}",LT:"LT[0-9]{2}[0-9]{5}[0-9]{11}",LU:"LU[0-9]{2}[0-9]{3}[A-Z0-9]{13}",LV:"LV[0-9]{2}[A-Z]{4}[A-Z0-9]{13}",MC:"MC[0-9]{2}[0-9]{5}[0-9]{5}[A-Z0-9]{11}[0-9]{2}",MD:"MD[0-9]{2}[A-Z0-9]{20}",ME:"ME[0-9]{2}[0-9]{3}[0-9]{13}[0-9]{2}",MG:"MG[0-9]{2}[0-9]{23}",MK:"MK[0-9]{2}[0-9]{3}[A-Z0-9]{10}[0-9]{2}",ML:"ML[0-9]{2}[A-Z]{1}[0-9]{23}",MR:"MR13[0-9]{5}[0-9]{5}[0-9]{11}[0-9]{2}",MT:"MT[0-9]{2}[A-Z]{4}[0-9]{5}[A-Z0-9]{18}",MU:"MU[0-9]{2}[A-Z]{4}[0-9]{2}[0-9]{2}[0-9]{12}[0-9]{3}[A-Z]{3}",MZ:"MZ[0-9]{2}[0-9]{21}",NL:"NL[0-9]{2}[A-Z]{4}[0-9]{10}",NO:"NO[0-9]{2}[0-9]{4}[0-9]{6}[0-9]{1}",PK:"PK[0-9]{2}[A-Z]{4}[A-Z0-9]{16}",PL:"PL[0-9]{2}[0-9]{8}[0-9]{16}",PS:"PS[0-9]{2}[A-Z]{4}[A-Z0-9]{21}",PT:"PT[0-9]{2}[0-9]{4}[0-9]{4}[0-9]{11}[0-9]{2}",QA:"QA[0-9]{2}[A-Z]{4}[A-Z0-9]{21}",RO:"RO[0-9]{2}[A-Z]{4}[A-Z0-9]{16}",RS:"RS[0-9]{2}[0-9]{3}[0-9]{13}[0-9]{2}",SA:"SA[0-9]{2}[0-9]{2}[A-Z0-9]{18}",SE:"SE[0-9]{2}[0-9]{3}[0-9]{16}[0-9]{1}",SI:"SI[0-9]{2}[0-9]{5}[0-9]{8}[0-9]{2}",SK:"SK[0-9]{2}[0-9]{4}[0-9]{6}[0-9]{10}",SM:"SM[0-9]{2}[A-Z]{1}[0-9]{5}[0-9]{5}[A-Z0-9]{12}",SN:"SN[0-9]{2}[A-Z]{1}[0-9]{23}",TL:"TL38[0-9]{3}[0-9]{14}[0-9]{2}",TN:"TN59[0-9]{2}[0-9]{3}[0-9]{13}[0-9]{2}",TR:"TR[0-9]{2}[0-9]{5}[A-Z0-9]{1}[A-Z0-9]{16}",VG:"VG[0-9]{2}[A-Z]{4}[0-9]{16}",XK:"XK[0-9]{2}[0-9]{4}[0-9]{10}[0-9]{2}"};var e=["AT","BE","BG","CH","CY","CZ","DE","DK","EE","ES","FI","FR","GB","GI","GR","HR","HU","IE","IS","IT","LI","LT","LU","LV","MC","MT","NL","NO","PL","PT","RO","SE","SI","SK","SM"];return{validate:function validate(s){if(s.value===""){return{valid:true}}var t=Object.assign({},{message:""},s.options);var a=s.value.replace(/[^a-zA-Z0-9]/g,"").toUpperCase();var r=t.country||a.substr(0,2);if(!Z[r]){return{message:t.message,valid:false}}if(t.sepa!==undefined){var _A=e.indexOf(r)!==-1;if((t.sepa==="true"||t.sepa===true)&&!_A||(t.sepa==="false"||t.sepa===false)&&_A){return{message:t.message,valid:false}}}var n=r$d(s.l10n?t.message||s.l10n.iban.country:t.message,s.l10n?s.l10n.iban.countries[r]:r);if(!new RegExp("^".concat(Z[r],"$")).test(s.value)){return{message:n,valid:false}}a="".concat(a.substr(4)).concat(a.substr(0,4));a=a.split("").map((function(A){var Z=A.charCodeAt(0);return Z>="A".charCodeAt(0)&&Z<="Z".charCodeAt(0)?Z-"A".charCodeAt(0)+10:A})).join("");var I=parseInt(a.substr(0,1),10);var L=a.length;for(var _A2=1;_A2<L;++_A2){I=(I*10+parseInt(a.substr(_A2,1),10))%97}return{message:n,valid:I===1}}}}function t$R(t){var e=t.replace(/\./g,"");return{meta:{},valid:/^\d{7,8}$/.test(e)}}function t$Q(t,r){if(!/^\d{13}$/.test(t)){return false}var e=parseInt(t.substr(0,2),10);var s=parseInt(t.substr(2,2),10);var n=parseInt(t.substr(7,2),10);var a=parseInt(t.substr(12,1),10);if(e>31||s>12){return false}var u=0;for(var _r=0;_r<6;_r++){u+=(7-_r)*(parseInt(t.charAt(_r),10)+parseInt(t.charAt(_r+6),10))}u=11-u%11;if(u===10||u===11){u=0}if(u!==a){return false}switch(r.toUpperCase()){case"BA":return 10<=n&&n<=19;case"MK":return 41<=n&&n<=49;case"ME":return 20<=n&&n<=29;case"RS":return 70<=n&&n<=99;case"SI":return 50<=n&&n<=59;default:return true}}function r$a(r){return{meta:{},valid:t$Q(r,"BA")}}function e$x(e){if(!/^\d{10}$/.test(e)&&!/^\d{6}\s\d{3}\s\d{1}$/.test(e)){return{meta:{},valid:false}}var s=e.replace(/\s/g,"");var r=parseInt(s.substr(0,2),10)+1900;var a=parseInt(s.substr(2,2),10);var l=parseInt(s.substr(4,2),10);if(a>40){r+=100;a-=40}else if(a>20){r-=100;a-=20}if(!t$Y(r,a,l)){return{meta:{},valid:false}}var i=0;var n=[2,4,8,5,10,9,7,3,6];for(var _t=0;_t<9;_t++){i+=parseInt(s.charAt(_t),10)*n[_t]}i=i%11%10;return{meta:{},valid:"".concat(i)===s.substr(9,1)}}function t$P(t){var e=t.replace(/\D/g,"");if(!/^\d{11}$/.test(e)||/^1{11}|2{11}|3{11}|4{11}|5{11}|6{11}|7{11}|8{11}|9{11}|0{11}$/.test(e)){return{meta:{},valid:false}}var a=0;var r;for(r=0;r<9;r++){a+=(10-r)*parseInt(e.charAt(r),10)}a=11-a%11;if(a===10||a===11){a=0}if("".concat(a)!==e.charAt(9)){return{meta:{},valid:false}}var f=0;for(r=0;r<10;r++){f+=(11-r)*parseInt(e.charAt(r),10)}f=11-f%11;if(f===10||f===11){f=0}return{meta:{},valid:"".concat(f)===e.charAt(10)}}function t$O(t){if(!/^756[.]{0,1}[0-9]{4}[.]{0,1}[0-9]{4}[.]{0,1}[0-9]{2}$/.test(t)){return{meta:{},valid:false}}var e=t.replace(/\D/g,"").substr(3);var r=e.length;var a=r===8?[3,1]:[1,3];var n=0;for(var _t=0;_t<r-1;_t++){n+=parseInt(e.charAt(_t),10)*a[_t%2]}n=10-n%10;return{meta:{},valid:"".concat(n)===e.charAt(r-1)}}function e$w(e){if(!/^\d{7,8}[-]{0,1}[0-9K]$/i.test(e)){return{meta:{},valid:false}}var t=e.replace(/-/g,"");while(t.length<9){t="0".concat(t)}var l=[3,2,7,6,5,4,3,2];var a=0;for(var _e=0;_e<8;_e++){a+=parseInt(t.charAt(_e),10)*l[_e]}a=11-a%11;var r="".concat(a);if(a===11){r="0"}else if(a===10){r="K"}return{meta:{},valid:r===t.charAt(8).toUpperCase()}}function r$9(r){var s=r.trim();if(!/^\d{15}$/.test(s)&&!/^\d{17}[\dXx]{1}$/.test(s)){return{meta:{},valid:false}}var e={11:{0:[0],1:[[0,9],[11,17]],2:[0,28,29]},12:{0:[0],1:[[0,16]],2:[0,21,23,25]},13:{0:[0],1:[[0,5],7,8,21,[23,33],[81,85]],2:[[0,5],[7,9],[23,25],27,29,30,81,83],3:[[0,4],[21,24]],4:[[0,4],6,21,[23,35],81],5:[[0,3],[21,35],81,82],6:[[0,4],[21,38],[81,84]],7:[[0,3],5,6,[21,33]],8:[[0,4],[21,28]],9:[[0,3],[21,30],[81,84]],10:[[0,3],[22,26],28,81,82],11:[[0,2],[21,28],81,82]},14:{0:[0],1:[0,1,[5,10],[21,23],81],2:[[0,3],11,12,[21,27]],3:[[0,3],11,21,22],4:[[0,2],11,21,[23,31],81],5:[[0,2],21,22,24,25,81],6:[[0,3],[21,24]],7:[[0,2],[21,29],81],8:[[0,2],[21,30],81,82],9:[[0,2],[21,32],81],10:[[0,2],[21,34],81,82],11:[[0,2],[21,30],81,82],23:[[0,3],22,23,[25,30],32,33]},15:{0:[0],1:[[0,5],[21,25]],2:[[0,7],[21,23]],3:[[0,4]],4:[[0,4],[21,26],[28,30]],5:[[0,2],[21,26],81],6:[[0,2],[21,27]],7:[[0,3],[21,27],[81,85]],8:[[0,2],[21,26]],9:[[0,2],[21,29],81],22:[[0,2],[21,24]],25:[[0,2],[22,31]],26:[[0,2],[24,27],[29,32],34],28:[0,1,[22,27]],29:[0,[21,23]]},21:{0:[0],1:[[0,6],[11,14],[22,24],81],2:[[0,4],[11,13],24,[81,83]],3:[[0,4],11,21,23,81],4:[[0,4],11,[21,23]],5:[[0,5],21,22],6:[[0,4],24,81,82],7:[[0,3],11,26,27,81,82],8:[[0,4],11,81,82],9:[[0,5],11,21,22],10:[[0,5],11,21,81],11:[[0,3],21,22],12:[[0,2],4,21,23,24,81,82],13:[[0,3],21,22,24,81,82],14:[[0,4],21,22,81]},22:{0:[0],1:[[0,6],12,22,[81,83]],2:[[0,4],11,21,[81,84]],3:[[0,3],22,23,81,82],4:[[0,3],21,22],5:[[0,3],21,23,24,81,82],6:[[0,2],4,5,[21,23],25,81],7:[[0,2],[21,24],81],8:[[0,2],21,22,81,82],24:[[0,6],24,26]},23:{0:[0],1:[[0,12],21,[23,29],[81,84]],2:[[0,8],21,[23,25],27,[29,31],81],3:[[0,7],21,81,82],4:[[0,7],21,22],5:[[0,3],5,6,[21,24]],6:[[0,6],[21,24]],7:[[0,16],22,81],8:[[0,5],11,22,26,28,33,81,82],9:[[0,4],21],10:[[0,5],24,25,81,[83,85]],11:[[0,2],21,23,24,81,82],12:[[0,2],[21,26],[81,83]],27:[[0,4],[21,23]]},31:{0:[0],1:[0,1,[3,10],[12,20]],2:[0,30]},32:{0:[0],1:[[0,7],11,[13,18],24,25],2:[[0,6],11,81,82],3:[[0,5],11,12,[21,24],81,82],4:[[0,2],4,5,11,12,81,82],5:[[0,9],[81,85]],6:[[0,2],11,12,21,23,[81,84]],7:[0,1,3,5,6,[21,24]],8:[[0,4],11,26,[29,31]],9:[[0,3],[21,25],28,81,82],10:[[0,3],11,12,23,81,84,88],11:[[0,2],11,12,[81,83]],12:[[0,4],[81,84]],13:[[0,2],11,[21,24]]},33:{0:[0],1:[[0,6],[8,10],22,27,82,83,85],2:[0,1,[3,6],11,12,25,26,[81,83]],3:[[0,4],22,24,[26,29],81,82],4:[[0,2],11,21,24,[81,83]],5:[[0,3],[21,23]],6:[[0,2],21,24,[81,83]],7:[[0,3],23,26,27,[81,84]],8:[[0,3],22,24,25,81],9:[[0,3],21,22],10:[[0,4],[21,24],81,82],11:[[0,2],[21,27],81]},34:{0:[0],1:[[0,4],11,[21,24],81],2:[[0,4],7,8,[21,23],25],3:[[0,4],11,[21,23]],4:[[0,6],21],5:[[0,4],6,[21,23]],6:[[0,4],21],7:[[0,3],11,21],8:[[0,3],11,[22,28],81],10:[[0,4],[21,24]],11:[[0,3],22,[24,26],81,82],12:[[0,4],21,22,25,26,82],13:[[0,2],[21,24]],14:[[0,2],[21,24]],15:[[0,3],[21,25]],16:[[0,2],[21,23]],17:[[0,2],[21,23]],18:[[0,2],[21,25],81]},35:{0:[0],1:[[0,5],11,[21,25],28,81,82],2:[[0,6],[11,13]],3:[[0,5],22],4:[[0,3],21,[23,30],81],5:[[0,5],21,[24,27],[81,83]],6:[[0,3],[22,29],81],7:[[0,2],[21,25],[81,84]],8:[[0,2],[21,25],81],9:[[0,2],[21,26],81,82]},36:{0:[0],1:[[0,5],11,[21,24]],2:[[0,3],22,81],3:[[0,2],13,[21,23]],4:[[0,3],21,[23,30],81,82],5:[[0,2],21],6:[[0,2],22,81],7:[[0,2],[21,35],81,82],8:[[0,3],[21,30],81],9:[[0,2],[21,26],[81,83]],10:[[0,2],[21,30]],11:[[0,2],[21,30],81]},37:{0:[0],1:[[0,5],12,13,[24,26],81],2:[[0,3],5,[11,14],[81,85]],3:[[0,6],[21,23]],4:[[0,6],81],5:[[0,3],[21,23]],6:[[0,2],[11,13],34,[81,87]],7:[[0,5],24,25,[81,86]],8:[[0,2],11,[26,32],[81,83]],9:[[0,3],11,21,23,82,83],10:[[0,2],[81,83]],11:[[0,3],21,22],12:[[0,3]],13:[[0,2],11,12,[21,29]],14:[[0,2],[21,28],81,82],15:[[0,2],[21,26],81],16:[[0,2],[21,26]],17:[[0,2],[21,28]]},41:{0:[0],1:[[0,6],8,22,[81,85]],2:[[0,5],11,[21,25]],3:[[0,7],11,[22,29],81],4:[[0,4],11,[21,23],25,81,82],5:[[0,3],5,6,22,23,26,27,81],6:[[0,3],11,21,22],7:[[0,4],11,21,[24,28],81,82],8:[[0,4],11,[21,23],25,[81,83]],9:[[0,2],22,23,[26,28]],10:[[0,2],[23,25],81,82],11:[[0,4],[21,23]],12:[[0,2],21,22,24,81,82],13:[[0,3],[21,30],81],14:[[0,3],[21,26],81],15:[[0,3],[21,28]],16:[[0,2],[21,28],81],17:[[0,2],[21,29]],90:[0,1]},42:{0:[0],1:[[0,7],[11,17]],2:[[0,5],22,81],3:[[0,3],[21,25],81],5:[[0,6],[25,29],[81,83]],6:[[0,2],6,7,[24,26],[82,84]],7:[[0,4]],8:[[0,2],4,21,22,81],9:[[0,2],[21,23],81,82,84],10:[[0,3],[22,24],81,83,87],11:[[0,2],[21,27],81,82],12:[[0,2],[21,24],81],13:[[0,3],21,81],28:[[0,2],22,23,[25,28]],90:[0,[4,6],21]},43:{0:[0],1:[[0,5],11,12,21,22,24,81],2:[[0,4],11,21,[23,25],81],3:[[0,2],4,21,81,82],4:[0,1,[5,8],12,[21,24],26,81,82],5:[[0,3],11,[21,25],[27,29],81],6:[[0,3],11,21,23,24,26,81,82],7:[[0,3],[21,26],81],8:[[0,2],11,21,22],9:[[0,3],[21,23],81],10:[[0,3],[21,28],81],11:[[0,3],[21,29]],12:[[0,2],[21,30],81],13:[[0,2],21,22,81,82],31:[0,1,[22,27],30]},44:{0:[0],1:[[0,7],[11,16],83,84],2:[[0,5],21,22,24,29,32,33,81,82],3:[0,1,[3,8]],4:[[0,4]],5:[0,1,[6,15],23,82,83],6:[0,1,[4,8]],7:[0,1,[3,5],81,[83,85]],8:[[0,4],11,23,25,[81,83]],9:[[0,3],23,[81,83]],12:[[0,3],[23,26],83,84],13:[[0,3],[22,24],81],14:[[0,2],[21,24],26,27,81],15:[[0,2],21,23,81],16:[[0,2],[21,25]],17:[[0,2],21,23,81],18:[[0,3],21,23,[25,27],81,82],19:[0],20:[0],51:[[0,3],21,22],52:[[0,3],21,22,24,81],53:[[0,2],[21,23],81]},45:{0:[0],1:[[0,9],[21,27]],2:[[0,5],[21,26]],3:[[0,5],11,12,[21,32]],4:[0,1,[3,6],11,[21,23],81],5:[[0,3],12,21],6:[[0,3],21,81],7:[[0,3],21,22],8:[[0,4],21,81],9:[[0,3],[21,24],81],10:[[0,2],[21,31]],11:[[0,2],[21,23]],12:[[0,2],[21,29],81],13:[[0,2],[21,24],81],14:[[0,2],[21,25],81]},46:{0:[0],1:[0,1,[5,8]],2:[0,1],3:[0,[21,23]],90:[[0,3],[5,7],[21,39]]},50:{0:[0],1:[[0,19]],2:[0,[22,38],[40,43]],3:[0,[81,84]]},51:{0:[0],1:[0,1,[4,8],[12,15],[21,24],29,31,32,[81,84]],3:[[0,4],11,21,22],4:[[0,3],11,21,22],5:[[0,4],21,22,24,25],6:[0,1,3,23,26,[81,83]],7:[0,1,3,4,[22,27],81],8:[[0,2],11,12,[21,24]],9:[[0,4],[21,23]],10:[[0,2],11,24,25,28],11:[[0,2],[11,13],23,24,26,29,32,33,81],13:[[0,4],[21,25],81],14:[[0,2],[21,25]],15:[[0,3],[21,29]],16:[[0,3],[21,23],81],17:[[0,3],[21,25],81],18:[[0,3],[21,27]],19:[[0,3],[21,23]],20:[[0,2],21,22,81],32:[0,[21,33]],33:[0,[21,38]],34:[0,1,[22,37]]},52:{0:[0],1:[[0,3],[11,15],[21,23],81],2:[0,1,3,21,22],3:[[0,3],[21,30],81,82],4:[[0,2],[21,25]],5:[[0,2],[21,27]],6:[[0,3],[21,28]],22:[0,1,[22,30]],23:[0,1,[22,28]],24:[0,1,[22,28]],26:[0,1,[22,36]],27:[[0,2],22,23,[25,32]]},53:{0:[0],1:[[0,3],[11,14],21,22,[24,29],81],3:[[0,2],[21,26],28,81],4:[[0,2],[21,28]],5:[[0,2],[21,24]],6:[[0,2],[21,30]],7:[[0,2],[21,24]],8:[[0,2],[21,29]],9:[[0,2],[21,27]],23:[0,1,[22,29],31],25:[[0,4],[22,32]],26:[0,1,[21,28]],27:[0,1,[22,30]],28:[0,1,22,23],29:[0,1,[22,32]],31:[0,2,3,[22,24]],34:[0,[21,23]],33:[0,21,[23,25]],35:[0,[21,28]]},54:{0:[0],1:[[0,2],[21,27]],21:[0,[21,29],32,33],22:[0,[21,29],[31,33]],23:[0,1,[22,38]],24:[0,[21,31]],25:[0,[21,27]],26:[0,[21,27]]},61:{0:[0],1:[[0,4],[11,16],22,[24,26]],2:[[0,4],22],3:[[0,4],[21,24],[26,31]],4:[[0,4],[22,31],81],5:[[0,2],[21,28],81,82],6:[[0,2],[21,32]],7:[[0,2],[21,30]],8:[[0,2],[21,31]],9:[[0,2],[21,29]],10:[[0,2],[21,26]]},62:{0:[0],1:[[0,5],11,[21,23]],2:[0,1],3:[[0,2],21],4:[[0,3],[21,23]],5:[[0,3],[21,25]],6:[[0,2],[21,23]],7:[[0,2],[21,25]],8:[[0,2],[21,26]],9:[[0,2],[21,24],81,82],10:[[0,2],[21,27]],11:[[0,2],[21,26]],12:[[0,2],[21,28]],24:[0,21,[24,29]],26:[0,21,[23,30]],29:[0,1,[21,27]],30:[0,1,[21,27]]},63:{0:[0],1:[[0,5],[21,23]],2:[0,2,[21,25]],21:[0,[21,23],[26,28]],22:[0,[21,24]],23:[0,[21,24]],25:[0,[21,25]],26:[0,[21,26]],27:[0,1,[21,26]],28:[[0,2],[21,23]]},64:{0:[0],1:[0,1,[4,6],21,22,81],2:[[0,3],5,[21,23]],3:[[0,3],[21,24],81],4:[[0,2],[21,25]],5:[[0,2],21,22]},65:{0:[0],1:[[0,9],21],2:[[0,5]],21:[0,1,22,23],22:[0,1,22,23],23:[[0,3],[23,25],27,28],28:[0,1,[22,29]],29:[0,1,[22,29]],30:[0,1,[22,24]],31:[0,1,[21,31]],32:[0,1,[21,27]],40:[0,2,3,[21,28]],42:[[0,2],21,[23,26]],43:[0,1,[21,26]],90:[[0,4]],27:[[0,2],22,23]},71:{0:[0]},81:{0:[0]},82:{0:[0]}};var a=parseInt(s.substr(0,2),10);var n=parseInt(s.substr(2,2),10);var l=parseInt(s.substr(4,2),10);if(!e[a]||!e[a][n]){return{meta:{},valid:false}}var i=false;var u=e[a][n];var o;for(o=0;o<u.length;o++){if(Array.isArray(u[o])&&u[o][0]<=l&&l<=u[o][1]||!Array.isArray(u[o])&&l===u[o]){i=true;break}}if(!i){return{meta:{},valid:false}}var f;if(s.length===18){f=s.substr(6,8)}else{f="19".concat(s.substr(6,6))}var c=parseInt(f.substr(0,4),10);var p=parseInt(f.substr(4,2),10);var d=parseInt(f.substr(6,2),10);if(!t$Y(c,p,d)){return{meta:{},valid:false}}if(s.length===18){var _t=[7,9,10,5,8,4,2,1,6,3,7,9,10,5,8,4,2];var _r=0;for(o=0;o<17;o++){_r+=parseInt(s.charAt(o),10)*_t[o]}_r=(12-_r%11)%11;var _e=s.charAt(17).toUpperCase()!=="X"?parseInt(s.charAt(17),10):10;return{meta:{},valid:_e===_r}}return{meta:{},valid:true}}function t$N(t){var e=t.replace(/\./g,"").replace("-","");if(!/^\d{8,16}$/.test(e)){return{meta:{},valid:false}}var r=e.length;var a=[3,7,13,17,19,23,29,37,41,43,47,53,59,67,71];var l=0;for(var _t=r-2;_t>=0;_t--){l+=parseInt(e.charAt(_t),10)*a[_t]}l=l%11;if(l>=2){l=11-l}return{meta:{},valid:"".concat(l)===e.substr(r-1)}}function e$v(e){if(!/^\d{9,10}$/.test(e)){return{meta:{},valid:false}}var r=1900+parseInt(e.substr(0,2),10);var s=parseInt(e.substr(2,2),10)%50%20;var a=parseInt(e.substr(4,2),10);if(e.length===9){if(r>=1980){r-=100}if(r>1953){return{meta:{},valid:false}}}else if(r<1954){r+=100}if(!t$Y(r,s,a)){return{meta:{},valid:false}}if(e.length===10){var _t=parseInt(e.substr(0,9),10)%11;if(r<1985){_t=_t%10}return{meta:{},valid:"".concat(_t)===e.substr(9,1)}}return{meta:{},valid:true}}function e$u(e){if(!/^[0-9]{6}[-]{0,1}[0-9]{4}$/.test(e)){return{meta:{},valid:false}}var a=e.replace(/-/g,"");var r=parseInt(a.substr(0,2),10);var s=parseInt(a.substr(2,2),10);var n=parseInt(a.substr(4,2),10);switch(true){case"5678".indexOf(a.charAt(6))!==-1&&n>=58:n+=1800;break;case"0123".indexOf(a.charAt(6))!==-1:case"49".indexOf(a.charAt(6))!==-1&&n>=37:n+=1900;break;default:n+=2e3;break}return{meta:{},valid:t$Y(n,s,r)}}function t$M(t){var e=/^[0-9]{8}[-]{0,1}[A-HJ-NP-TV-Z]$/.test(t);var s=/^[XYZ][-]{0,1}[0-9]{7}[-]{0,1}[A-HJ-NP-TV-Z]$/.test(t);var n=/^[A-HNPQS][-]{0,1}[0-9]{7}[-]{0,1}[0-9A-J]$/.test(t);if(!e&&!s&&!n){return{meta:{},valid:false}}var r=t.replace(/-/g,"");var l;var a;var f=true;if(e||s){a="DNI";var _t="XYZ".indexOf(r.charAt(0));if(_t!==-1){r=_t+r.substr(1)+"";a="NIE"}l=parseInt(r.substr(0,8),10);l="TRWAGMYFPDXBNJZSQVHLCKE"[l%23];return{meta:{type:a},valid:l===r.substr(8,1)}}else{l=r.substr(1,7);a="CIF";var _t2=r[0];var _e=r.substr(-1);var _s=0;for(var _t3=0;_t3<l.length;_t3++){if(_t3%2!==0){_s+=parseInt(l[_t3],10)}else{var _e2=""+parseInt(l[_t3],10)*2;_s+=parseInt(_e2[0],10);if(_e2.length===2){_s+=parseInt(_e2[1],10)}}}var _n=_s-Math.floor(_s/10)*10;if(_n!==0){_n=10-_n}if("KQS".indexOf(_t2)!==-1){f=_e==="JABCDEFGHI"[_n]}else if("ABEH".indexOf(_t2)!==-1){f=_e===""+_n}else{f=_e===""+_n||_e==="JABCDEFGHI"[_n]}return{meta:{type:a},valid:f}}}function s$7(s){if(!/^[0-9]{6}[-+A][0-9]{3}[0-9ABCDEFHJKLMNPRSTUVWXY]$/.test(s)){return{meta:{},valid:false}}var r=parseInt(s.substr(0,2),10);var a=parseInt(s.substr(2,2),10);var e=parseInt(s.substr(4,2),10);var n={"+":1800,"-":1900,A:2e3};e=n[s.charAt(6)]+e;if(!t$Y(e,a,r)){return{meta:{},valid:false}}var u=parseInt(s.substr(7,3),10);if(u<2){return{meta:{},valid:false}}var i=parseInt(s.substr(0,6)+s.substr(7,3)+"",10);return{meta:{},valid:"0123456789ABCDEFHJKLMNPRSTUVWXY".charAt(i%31)===s.charAt(10)}}function t$L(t){var s=t.toUpperCase();if(!/^(1|2)\d{2}\d{2}(\d{2}|\d[A-Z]|\d{3})\d{2,3}\d{3}\d{2}$/.test(s)){return{meta:{},valid:false}}var e=s.substr(5,2);switch(true){case/^\d{2}$/.test(e):s=t;break;case e==="2A":s="".concat(t.substr(0,5),"19").concat(t.substr(7));break;case e==="2B":s="".concat(t.substr(0,5),"18").concat(t.substr(7));break;default:return{meta:{},valid:false}}var r=97-parseInt(s.substr(0,13),10)%97;var a=r<10?"0".concat(r):"".concat(r);return{meta:{},valid:a===s.substr(13)}}function t$K(t){var e=t.toUpperCase();if(!/^[A-MP-Z]{1,2}[0-9]{6}[0-9A]$/.test(e)){return{meta:{},valid:false}}var s="ABCDEFGHIJKLMNOPQRSTUVWXYZ";var n=e.charAt(0);var r=e.charAt(1);var a=0;var c=e;if(/^[A-Z]$/.test(r)){a+=9*(10+s.indexOf(n));a+=8*(10+s.indexOf(r));c=e.substr(2)}else{a+=9*36;a+=8*(10+s.indexOf(n));c=e.substr(1)}var o=c.length;for(var _t=0;_t<o-1;_t++){a+=(7-_t)*parseInt(c.charAt(_t),10)}var f=a%11;var l=f===0?"0":11-f===10?"A":"".concat(11-f);return{meta:{},valid:l===c.charAt(o-1)}}function o$1(o){return{meta:{},valid:/^[0-9]{11}$/.test(o)&&t$14(o)}}function e$t(e){if(!/^[2-9]\d{11}$/.test(e)){return{meta:{},valid:false}}var r=e.split("").map((function(t){return parseInt(t,10)}));return{meta:{},valid:t$12(r)}}function t$J(t){if(!/^\d{7}[A-W][AHWTX]?$/.test(t)){return{meta:{},valid:false}}var r=function r(t){var r=t;while(r.length<7){r="0".concat(r)}var e="WABCDEFGHIJKLMNOPQRSTUV";var s=0;for(var _t=0;_t<7;_t++){s+=parseInt(r.charAt(_t),10)*(8-_t)}s+=9*e.indexOf(r.substr(7));return e[s%23]};var e=t.length===9&&("A"===t.charAt(8)||"H"===t.charAt(8))?t.charAt(7)===r(t.substr(0,7)+t.substr(8)+""):t.charAt(7)===r(t.substr(0,7));return{meta:{},valid:e}}function e$s(e){if(!/^\d{1,9}$/.test(e)){return{meta:{},valid:false}}return{meta:{},valid:t$15(e)}}function e$r(e){if(!/^[0-9]{6}[-]{0,1}[0-9]{4}$/.test(e)){return{meta:{},valid:false}}var r=e.replace(/-/g,"");var s=parseInt(r.substr(0,2),10);var a=parseInt(r.substr(2,2),10);var n=parseInt(r.substr(4,2),10);var l=parseInt(r.charAt(9),10);n=l===9?1900+n:(20+l)*100+n;if(!t$Y(n,a,s,true)){return{meta:{},valid:false}}var c=[3,2,7,6,5,4,3,2];var i=0;for(var _t=0;_t<8;_t++){i+=parseInt(r.charAt(_t),10)*c[_t]}i=11-i%11;return{meta:{},valid:"".concat(i)===r.charAt(8)}}function e$q(e){var a=e.replace("-","");if(!/^\d{13}$/.test(a)){return{meta:{},valid:false}}var s=a.charAt(6);var r=parseInt(a.substr(0,2),10);var c=parseInt(a.substr(2,2),10);var n=parseInt(a.substr(4,2),10);switch(s){case"1":case"2":case"5":case"6":r+=1900;break;case"3":case"4":case"7":case"8":r+=2e3;break;default:r+=1800;break}if(!t$Y(r,c,n)){return{meta:{},valid:false}}var l=[2,3,4,5,6,7,8,9,2,3,4,5];var o=a.length;var i=0;for(var _t=0;_t<o-1;_t++){i+=l[_t]*parseInt(a.charAt(_t),10)}var u=(11-i%11)%10;return{meta:{},valid:"".concat(u)===a.charAt(o-1)}}function r$8(r){if(!/^[0-9]{11}$/.test(r)){return{meta:{},valid:false}}var e=parseInt(r.charAt(0),10);var a=parseInt(r.substr(1,2),10);var s=parseInt(r.substr(3,2),10);var n=parseInt(r.substr(5,2),10);var i=e%2===0?17+e/2:17+(e+1)/2;a=i*100+a;if(!t$Y(a,s,n,true)){return{meta:{},valid:false}}var l=[1,2,3,4,5,6,7,8,9,1];var f=0;var u;for(u=0;u<10;u++){f+=parseInt(r.charAt(u),10)*l[u]}f=f%11;if(f!==10){return{meta:{},valid:"".concat(f)===r.charAt(10)}}f=0;l=[3,4,5,6,7,8,9,1,2,3];for(u=0;u<10;u++){f+=parseInt(r.charAt(u),10)*l[u]}f=f%11;if(f===10){f=0}return{meta:{},valid:"".concat(f)===r.charAt(10)}}function e$p(e){if(!/^[0-9]{6}[-]{0,1}[0-9]{5}$/.test(e)){return{meta:{},valid:false}}var r=e.replace(/\D/g,"");var a=parseInt(r.substr(0,2),10);var s=parseInt(r.substr(2,2),10);var n=parseInt(r.substr(4,2),10);n=n+1800+parseInt(r.charAt(6),10)*100;if(!t$Y(n,s,a,true)){return{meta:{},valid:false}}var l=0;var i=[10,5,8,4,2,1,6,3,7,9];for(var _t=0;_t<10;_t++){l+=parseInt(r.charAt(_t),10)*i[_t]}l=(l+1)%11%10;return{meta:{},valid:"".concat(l)===r.charAt(10)}}function r$7(r){return{meta:{},valid:t$Q(r,"ME")}}function r$6(r){return{meta:{},valid:t$Q(r,"MK")}}function O(O){var t=O.toUpperCase();if(!/^[A-Z]{4}\d{6}[A-Z]{6}[0-9A-Z]\d$/.test(t)){return{meta:{},valid:false}}var C=["BACA","BAKA","BUEI","BUEY","CACA","CACO","CAGA","CAGO","CAKA","CAKO","COGE","COGI","COJA","COJE","COJI","COJO","COLA","CULO","FALO","FETO","GETA","GUEI","GUEY","JETA","JOTO","KACA","KACO","KAGA","KAGO","KAKA","KAKO","KOGE","KOGI","KOJA","KOJE","KOJI","KOJO","KOLA","KULO","LILO","LOCA","LOCO","LOKA","LOKO","MAME","MAMO","MEAR","MEAS","MEON","MIAR","MION","MOCO","MOKO","MULA","MULO","NACA","NACO","PEDA","PEDO","PENE","PIPI","PITO","POPO","PUTA","PUTO","QULO","RATA","ROBA","ROBE","ROBO","RUIN","SENO","TETA","VACA","VAGA","VAGO","VAKA","VUEI","VUEY","WUEI","WUEY"];var e=t.substr(0,4);if(C.indexOf(e)>=0){return{meta:{},valid:false}}var s=parseInt(t.substr(4,2),10);var r=parseInt(t.substr(6,2),10);var a=parseInt(t.substr(6,2),10);if(/^[0-9]$/.test(t.charAt(16))){s+=1900}else{s+=2e3}if(!t$Y(s,r,a)){return{meta:{},valid:false}}var E=t.charAt(10);if(E!=="H"&&E!=="M"){return{meta:{},valid:false}}var n=t.substr(11,2);var K=["AS","BC","BS","CC","CH","CL","CM","CS","DF","DG","GR","GT","HG","JC","MC","MN","MS","NE","NL","NT","OC","PL","QR","QT","SL","SP","SR","TC","TL","TS","VZ","YN","ZS"];if(K.indexOf(n)===-1){return{meta:{},valid:false}}var i="0123456789ABCDEFGHIJKLMN&OPQRSTUVWXYZ";var L=0;var l=t.length;for(var _A=0;_A<l-1;_A++){L+=(18-_A)*i.indexOf(t.charAt(_A))}L=(10-L%10)%10;return{meta:{},valid:"".concat(L)===t.charAt(l-1)}}function s$6(s){if(!/^\d{12}$/.test(s)){return{meta:{},valid:false}}var e=parseInt(s.substr(0,2),10);var r=parseInt(s.substr(2,2),10);var a=parseInt(s.substr(4,2),10);if(!t$Y(e+1900,r,a)&&!t$Y(e+2e3,r,a)){return{meta:{},valid:false}}var n=s.substr(6,2);var i=["17","18","19","20","69","70","73","80","81","94","95","96","97"];return{meta:{},valid:i.indexOf(n)===-1}}function e$o(e){if(e.length<8){return{meta:{},valid:false}}var t=e;if(t.length===8){t="0".concat(t)}if(!/^[0-9]{4}[.]{0,1}[0-9]{2}[.]{0,1}[0-9]{3}$/.test(t)){return{meta:{},valid:false}}t=t.replace(/\./g,"");if(parseInt(t,10)===0){return{meta:{},valid:false}}var a=0;var l=t.length;for(var _e=0;_e<l-1;_e++){a+=(9-_e)*parseInt(t.charAt(_e),10)}a=a%11;if(a===10){a=0}return{meta:{},valid:"".concat(a)===t.charAt(l-1)}}function t$I(t){if(!/^\d{11}$/.test(t)){return{meta:{},valid:false}}var r=function r(t){var r=[3,7,6,1,8,9,4,5,2];var e=0;for(var n=0;n<9;n++){e+=r[n]*parseInt(t.charAt(n),10)}return 11-e%11};var e=function e(t){var r=[5,4,3,2,7,6,5,4,3,2];var e=0;for(var n=0;n<10;n++){e+=r[n]*parseInt(t.charAt(n),10)}return 11-e%11};return{meta:{},valid:"".concat(r(t))===t.substr(-2,1)&&"".concat(e(t))===t.substr(-1)}}function t$H(t){if(!/^\d{8}[0-9A-Z]*$/.test(t)){return{meta:{},valid:false}}if(t.length===8){return{meta:{},valid:true}}var e=[3,2,7,6,5,4,3,2];var r=0;for(var _a=0;_a<8;_a++){r+=e[_a]*parseInt(t.charAt(_a),10)}var a=r%11;var n=[6,5,4,3,2,1,1,0,9,8,7][a];var c="KJIHGFEDCBA".charAt(a);return{meta:{},valid:t.charAt(8)==="".concat(n)||t.charAt(8)===c}}function t$G(t){if(!/^[0-9]{11}$/.test(t)){return{meta:{},valid:false}}var e=0;var a=t.length;var r=[1,3,7,9,1,3,7,9,1,3,7];for(var n=0;n<a-1;n++){e+=r[n]*parseInt(t.charAt(n),10)}e=e%10;if(e===0){e=10}e=10-e;return{meta:{},valid:"".concat(e)===t.charAt(a-1)}}function e$n(e){if(!/^[0-9]{13}$/.test(e)){return{meta:{},valid:false}}var a=parseInt(e.charAt(0),10);if(a===0||a===7||a===8){return{meta:{},valid:false}}var r=parseInt(e.substr(1,2),10);var s=parseInt(e.substr(3,2),10);var n=parseInt(e.substr(5,2),10);var i={1:1900,2:1900,3:1800,4:1800,5:2e3,6:2e3};if(n>31&&s>12){return{meta:{},valid:false}}if(a!==9){r=i[a+""]+r;if(!t$Y(r,s,n)){return{meta:{},valid:false}}}var l=0;var f=[2,7,9,1,4,6,3,5,8,2,7,9];var o=e.length;for(var _t=0;_t<o-1;_t++){l+=parseInt(e.charAt(_t),10)*f[_t]}l=l%11;if(l===10){l=1}return{meta:{},valid:"".concat(l)===e.charAt(o-1)}}function r$5(r){return{meta:{},valid:t$Q(r,"RS")}}function r$4(r){if(!/^[0-9]{10}$/.test(r)&&!/^[0-9]{6}[-|+][0-9]{4}$/.test(r)){return{meta:{},valid:false}}var e=r.replace(/[^0-9]/g,"");var a=parseInt(e.substr(0,2),10)+1900;var n=parseInt(e.substr(2,2),10);var i=parseInt(e.substr(4,2),10);if(!t$Y(a,n,i)){return{meta:{},valid:false}}return{meta:{},valid:t$15(e)}}function r$3(r){return{meta:{},valid:t$Q(r,"SI")}}function t$F(t){return{meta:{},valid:/^\d{5}$/.test(t)}}function t$E(t){if(t.length!==13){return{meta:{},valid:false}}var e=0;for(var a=0;a<12;a++){e+=parseInt(t.charAt(a),10)*(13-a)}return{meta:{},valid:(11-e%11)%10===parseInt(t.charAt(12),10)}}function t$D(t){if(t.length!==11){return{meta:{},valid:false}}var e=0;for(var a=0;a<10;a++){e+=parseInt(t.charAt(a),10)}return{meta:{},valid:e%10===parseInt(t.charAt(10),10)}}function t$C(t){var e=t.toUpperCase();if(!/^[A-Z][12][0-9]{8}$/.test(e)){return{meta:{},valid:false}}var a=e.length;var n="ABCDEFGHJKLMNPQRSTUVXYWZIO";var r=n.indexOf(e.charAt(0))+10;var o=Math.floor(r/10)+r%10*(a-1);var s=0;for(var _t=1;_t<a-1;_t++){s+=parseInt(e.charAt(_t),10)*(a-1-_t)}return{meta:{},valid:(o+s+parseInt(e.charAt(a-1),10))%10===0}}function t$B(t){if(!/^\d{8}$/.test(t)){return{meta:{},valid:false}}var e=[2,9,8,7,6,3,4];var a=0;for(var r=0;r<7;r++){a+=parseInt(t.charAt(r),10)*e[r]}a=a%10;if(a>0){a=10-a}return{meta:{},valid:"".concat(a)===t.charAt(7)}}function r$2(r){if(!/^[0-9]{10}[0|1][8|9][0-9]$/.test(r)){return{meta:{},valid:false}}var s=parseInt(r.substr(0,2),10);var a=(new Date).getFullYear()%100;var l=parseInt(r.substr(2,2),10);var n=parseInt(r.substr(4,2),10);s=s>=a?s+1900:s+2e3;if(!t$Y(s,l,n)){return{meta:{},valid:false}}return{meta:{},valid:t$15(r)}}function F(){var F=["AR","BA","BG","BR","CH","CL","CN","CO","CZ","DK","EE","ES","FI","FR","HK","HR","ID","IE","IL","IS","KR","LT","LV","ME","MK","MX","MY","NL","NO","PE","PL","RO","RS","SE","SI","SK","SM","TH","TR","TW","UY","ZA"];return{validate:function validate(P){if(P.value===""){return{valid:true}}var Y=Object.assign({},{message:""},P.options);var Z=P.value.substr(0,2);if("function"===typeof Y.country){Z=Y.country.call(this)}else{Z=Y.country}if(F.indexOf(Z)===-1){return{valid:true}}var G={meta:{},valid:true};switch(Z.toLowerCase()){case"ar":G=t$R(P.value);break;case"ba":G=r$a(P.value);break;case"bg":G=e$x(P.value);break;case"br":G=t$P(P.value);break;case"ch":G=t$O(P.value);break;case"cl":G=e$w(P.value);break;case"cn":G=r$9(P.value);break;case"co":G=t$N(P.value);break;case"cz":G=e$v(P.value);break;case"dk":G=e$u(P.value);break;case"ee":G=r$8(P.value);break;case"es":G=t$M(P.value);break;case"fi":G=s$7(P.value);break;case"fr":G=t$L(P.value);break;case"hk":G=t$K(P.value);break;case"hr":G=o$1(P.value);break;case"id":G=e$t(P.value);break;case"ie":G=t$J(P.value);break;case"il":G=e$s(P.value);break;case"is":G=e$r(P.value);break;case"kr":G=e$q(P.value);break;case"lt":G=r$8(P.value);break;case"lv":G=e$p(P.value);break;case"me":G=r$7(P.value);break;case"mk":G=r$6(P.value);break;case"mx":G=O(P.value);break;case"my":G=s$6(P.value);break;case"nl":G=e$o(P.value);break;case"no":G=t$I(P.value);break;case"pe":G=t$H(P.value);break;case"pl":G=t$G(P.value);break;case"ro":G=e$n(P.value);break;case"rs":G=r$5(P.value);break;case"se":G=r$4(P.value);break;case"si":G=r$3(P.value);break;case"sk":G=e$v(P.value);break;case"sm":G=t$F(P.value);break;case"th":G=t$E(P.value);break;case"tr":G=t$D(P.value);break;case"tw":G=t$C(P.value);break;case"uy":G=t$B(P.value);break;case"za":G=r$2(P.value);break}var V=r$d(P.l10n&&P.l10n.id?Y.message||P.l10n.id.country:Y.message,P.l10n&&P.l10n.id&&P.l10n.id.countries?P.l10n.id.countries[Z.toUpperCase()]:Z.toUpperCase());return Object.assign({},{message:V},G)}}}function t$A(){return{validate:function validate(t){if(t.value===""){return{valid:true}}switch(true){case/^\d{15}$/.test(t.value):case/^\d{2}-\d{6}-\d{6}-\d{1}$/.test(t.value):case/^\d{2}\s\d{6}\s\d{6}\s\d{1}$/.test(t.value):return{valid:t$15(t.value.replace(/[^0-9]/g,""))};case/^\d{14}$/.test(t.value):case/^\d{16}$/.test(t.value):case/^\d{2}-\d{6}-\d{6}(|-\d{2})$/.test(t.value):case/^\d{2}\s\d{6}\s\d{6}(|\s\d{2})$/.test(t.value):return{valid:true};default:return{valid:false}}}}}function e$m(){return{validate:function validate(e){if(e.value===""){return{valid:true}}if(!/^IMO \d{7}$/i.test(e.value)){return{valid:false}}var t=e.value.replace(/^.*(\d{7})$/,"$1");var r=0;for(var _e=6;_e>=1;_e--){r+=parseInt(t.slice(6-_e,-_e),10)*(_e+1)}return{valid:r%10===parseInt(t.charAt(6),10)}}}}function e$l(){return{validate:function validate(e){if(e.value===""){return{meta:{type:null},valid:true}}var t;switch(true){case/^\d{9}[\dX]$/.test(e.value):case e.value.length===13&&/^(\d+)-(\d+)-(\d+)-([\dX])$/.test(e.value):case e.value.length===13&&/^(\d+)\s(\d+)\s(\d+)\s([\dX])$/.test(e.value):t="ISBN10";break;case/^(978|979)\d{9}[\dX]$/.test(e.value):case e.value.length===17&&/^(978|979)-(\d+)-(\d+)-(\d+)-([\dX])$/.test(e.value):case e.value.length===17&&/^(978|979)\s(\d+)\s(\d+)\s(\d+)\s([\dX])$/.test(e.value):t="ISBN13";break;default:return{meta:{type:null},valid:false}}var a=e.value.replace(/[^0-9X]/gi,"").split("");var l=a.length;var s=0;var d;var u;switch(t){case"ISBN10":s=0;for(d=0;d<l-1;d++){s+=parseInt(a[d],10)*(10-d)}u=11-s%11;if(u===11){u=0}else if(u===10){u="X"}return{meta:{type:t},valid:"".concat(u)===a[l-1]};case"ISBN13":s=0;for(d=0;d<l-1;d++){s+=d%2===0?parseInt(a[d],10):parseInt(a[d],10)*3}u=10-s%10;if(u===10){u="0"}return{meta:{type:t},valid:"".concat(u)===a[l-1]}}}}}function M(){var M="AF|AX|AL|DZ|AS|AD|AO|AI|AQ|AG|AR|AM|AW|AU|AT|AZ|BS|BH|BD|BB|BY|BE|BZ|BJ|BM|BT|BO|BQ|BA|BW|"+"BV|BR|IO|BN|BG|BF|BI|KH|CM|CA|CV|KY|CF|TD|CL|CN|CX|CC|CO|KM|CG|CD|CK|CR|CI|HR|CU|CW|CY|CZ|DK|DJ|DM|DO|EC|EG|"+"SV|GQ|ER|EE|ET|FK|FO|FJ|FI|FR|GF|PF|TF|GA|GM|GE|DE|GH|GI|GR|GL|GD|GP|GU|GT|GG|GN|GW|GY|HT|HM|VA|HN|HK|HU|IS|"+"IN|ID|IR|IQ|IE|IM|IL|IT|JM|JP|JE|JO|KZ|KE|KI|KP|KR|KW|KG|LA|LV|LB|LS|LR|LY|LI|LT|LU|MO|MK|MG|MW|MY|MV|ML|MT|"+"MH|MQ|MR|MU|YT|MX|FM|MD|MC|MN|ME|MS|MA|MZ|MM|NA|NR|NP|NL|NC|NZ|NI|NE|NG|NU|NF|MP|NO|OM|PK|PW|PS|PA|PG|PY|PE|"+"PH|PN|PL|PT|PR|QA|RE|RO|RU|RW|BL|SH|KN|LC|MF|PM|VC|WS|SM|ST|SA|SN|RS|SC|SL|SG|SX|SK|SI|SB|SO|ZA|GS|SS|ES|LK|"+"SD|SR|SJ|SZ|SE|CH|SY|TW|TJ|TZ|TH|TL|TG|TK|TO|TT|TN|TR|TM|TC|TV|UG|UA|AE|GB|US|UM|UY|UZ|VU|VE|VN|VG|VI|WF|EH|"+"YE|ZM|ZW";return{validate:function validate(t){if(t.value===""){return{valid:true}}var S=t.value.toUpperCase();var A=new RegExp("^("+M+")[0-9A-Z]{10}$");if(!A.test(t.value)){return{valid:false}}var G=S.length;var C="";var T;for(T=0;T<G-1;T++){var _M=S.charCodeAt(T);C+=_M>57?(_M-55).toString():S.charAt(T)}var e="";var B=C.length;var E=B%2!==0?0:1;for(T=0;T<B;T++){e+=parseInt(C[T],10)*(T%2===E?2:1)+""}var N=0;for(T=0;T<e.length;T++){N+=parseInt(e.charAt(T),10)}N=(10-N%10)%10;return{valid:"".concat(N)===S.charAt(G-1)}}}}function e$k(){return{validate:function validate(e){if(e.value===""){return{meta:null,valid:true}}var t;switch(true){case/^M\d{9}$/.test(e.value):case/^M-\d{4}-\d{4}-\d{1}$/.test(e.value):case/^M\s\d{4}\s\d{4}\s\d{1}$/.test(e.value):t="ISMN10";break;case/^9790\d{9}$/.test(e.value):case/^979-0-\d{4}-\d{4}-\d{1}$/.test(e.value):case/^979\s0\s\d{4}\s\d{4}\s\d{1}$/.test(e.value):t="ISMN13";break;default:return{meta:null,valid:false}}var a=e.value;if("ISMN10"===t){a="9790".concat(a.substr(1))}a=a.replace(/[^0-9]/gi,"");var s=0;var l=a.length;var d=[1,3];for(var _e=0;_e<l-1;_e++){s+=parseInt(a.charAt(_e),10)*d[_e%2]}s=(10-s%10)%10;return{meta:{type:t},valid:"".concat(s)===a.charAt(l-1)}}}}function e$j(){return{validate:function validate(e){if(e.value===""){return{valid:true}}if(!/^\d{4}-\d{3}[\dX]$/.test(e.value)){return{valid:false}}var t=e.value.replace(/[^0-9X]/gi,"").split("");var l=t.length;var r=0;if(t[7]==="X"){t[7]="10"}for(var _e=0;_e<l;_e++){r+=parseInt(t[_e],10)*(8-_e)}return{valid:r%11===0}}}}function a$1(){return{validate:function validate(a){return{valid:a.value===""||/^([0-9A-Fa-f]{2}[:-]){5}([0-9A-Fa-f]{2})$/.test(a.value)||/^([0-9A-Fa-f]{4}\.){2}([0-9A-Fa-f]{4})$/.test(a.value)}}}}function e$i(){return{validate:function validate(e){if(e.value===""){return{valid:true}}var r=e.value;if(/^[0-9A-F]{15}$/i.test(r)||/^[0-9A-F]{2}[- ][0-9A-F]{6}[- ][0-9A-F]{6}[- ][0-9A-F]$/i.test(r)||/^\d{19}$/.test(r)||/^\d{5}[- ]\d{5}[- ]\d{4}[- ]\d{4}[- ]\d$/.test(r)){var _e=r.charAt(r.length-1).toUpperCase();r=r.replace(/[- ]/g,"");if(r.match(/^\d*$/i)){return{valid:t$15(r)}}r=r.slice(0,-1);var a="";var i;for(i=1;i<=13;i+=2){a+=(parseInt(r.charAt(i),16)*2).toString(16)}var l=0;for(i=0;i<a.length;i++){l+=parseInt(a.charAt(i),16)}return{valid:l%10===0?_e==="0":_e===((Math.floor((l+10)/10)*10-l)*2).toString(16).toUpperCase()}}if(/^[0-9A-F]{14}$/i.test(r)||/^[0-9A-F]{2}[- ][0-9A-F]{6}[- ][0-9A-F]{6}$/i.test(r)||/^\d{18}$/.test(r)||/^\d{5}[- ]\d{5}[- ]\d{4}[- ]\d{4}$/.test(r)){return{valid:true}}return{valid:false}}}}function e$h(){var e=["AE","BG","BR","CN","CZ","DE","DK","ES","FR","GB","IN","MA","NL","PK","RO","RU","SK","TH","US","VE"];return{validate:function validate(t){if(t.value===""){return{valid:true}}var a=Object.assign({},{message:""},t.options);var d=t.value.trim();var r=d.substr(0,2);if("function"===typeof a.country){r=a.country.call(this)}else{r=a.country}if(!r||e.indexOf(r.toUpperCase())===-1){return{valid:true}}var c=true;switch(r.toUpperCase()){case"AE":c=/^(((\+|00)?971[\s.-]?(\(0\)[\s.-]?)?|0)(\(5(0|2|5|6)\)|5(0|2|5|6)|2|3|4|6|7|9)|60)([\s.-]?[0-9]){7}$/.test(d);break;case"BG":c=/^(0|359|00)(((700|900)[0-9]{5}|((800)[0-9]{5}|(800)[0-9]{4}))|(87|88|89)([0-9]{7})|((2[0-9]{7})|(([3-9][0-9])(([0-9]{6})|([0-9]{5})))))$/.test(d.replace(/\+|\s|-|\/|\(|\)/gi,""));break;case"BR":c=/^(([\d]{4}[-.\s]{1}[\d]{2,3}[-.\s]{1}[\d]{2}[-.\s]{1}[\d]{2})|([\d]{4}[-.\s]{1}[\d]{3}[-.\s]{1}[\d]{4})|((\(?\+?[0-9]{2}\)?\s?)?(\(?\d{2}\)?\s?)?\d{4,5}[-.\s]?\d{4}))$/.test(d);break;case"CN":c=/^((00|\+)?(86(?:-| )))?((\d{11})|(\d{3}[- ]{1}\d{4}[- ]{1}\d{4})|((\d{2,4}[- ]){1}(\d{7,8}|(\d{3,4}[- ]{1}\d{4}))([- ]{1}\d{1,4})?))$/.test(d);break;case"CZ":c=/^(((00)([- ]?)|\+)(420)([- ]?))?((\d{3})([- ]?)){2}(\d{3})$/.test(d);break;case"DE":c=/^(((((((00|\+)49[ \-/]?)|0)[1-9][0-9]{1,4})[ \-/]?)|((((00|\+)49\()|\(0)[1-9][0-9]{1,4}\)[ \-/]?))[0-9]{1,7}([ \-/]?[0-9]{1,5})?)$/.test(d);break;case"DK":c=/^(\+45|0045|\(45\))?\s?[2-9](\s?\d){7}$/.test(d);break;case"ES":c=/^(?:(?:(?:\+|00)34\D?))?(?:5|6|7|8|9)(?:\d\D?){8}$/.test(d);break;case"FR":c=/^(?:(?:(?:\+|00)33[ ]?(?:\(0\)[ ]?)?)|0){1}[1-9]{1}([ .-]?)(?:\d{2}\1?){3}\d{2}$/.test(d);break;case"GB":c=/^\(?(?:(?:0(?:0|11)\)?[\s-]?\(?|\+)44\)?[\s-]?\(?(?:0\)?[\s-]?\(?)?|0)(?:\d{2}\)?[\s-]?\d{4}[\s-]?\d{4}|\d{3}\)?[\s-]?\d{3}[\s-]?\d{3,4}|\d{4}\)?[\s-]?(?:\d{5}|\d{3}[\s-]?\d{3})|\d{5}\)?[\s-]?\d{4,5}|8(?:00[\s-]?11[\s-]?11|45[\s-]?46[\s-]?4\d))(?:(?:[\s-]?(?:x|ext\.?\s?|#)\d+)?)$/.test(d);break;case"IN":c=/((\+?)((0[ -]+)*|(91 )*)(\d{12}|\d{10}))|\d{5}([- ]*)\d{6}/.test(d);break;case"MA":c=/^(?:(?:(?:\+|00)212[\s]?(?:[\s]?\(0\)[\s]?)?)|0){1}(?:5[\s.-]?[2-3]|6[\s.-]?[13-9]){1}[0-9]{1}(?:[\s.-]?\d{2}){3}$/.test(d);break;case"NL":c=/^((\+|00(\s|\s?-\s?)?)31(\s|\s?-\s?)?(\(0\)[-\s]?)?|0)[1-9]((\s|\s?-\s?)?[0-9])((\s|\s?-\s?)?[0-9])((\s|\s?-\s?)?[0-9])\s?[0-9]\s?[0-9]\s?[0-9]\s?[0-9]\s?[0-9]$/gm.test(d);break;case"PK":c=/^0?3[0-9]{2}[0-9]{7}$/.test(d);break;case"RO":c=/^(\+4|)?(07[0-8]{1}[0-9]{1}|02[0-9]{2}|03[0-9]{2}){1}?(\s|\.|-)?([0-9]{3}(\s|\.|-|)){2}$/g.test(d);break;case"RU":c=/^((8|\+7|007)[-./ ]?)?([(/.]?\d{3}[)/.]?[-./ ]?)?[\d\-./ ]{7,10}$/g.test(d);break;case"SK":c=/^(((00)([- ]?)|\+)(421)([- ]?))?((\d{3})([- ]?)){2}(\d{3})$/.test(d);break;case"TH":c=/^0\(?([6|8-9]{2})*-([0-9]{3})*-([0-9]{4})$/.test(d);break;case"VE":c=/^0(?:2(?:12|4[0-9]|5[1-9]|6[0-9]|7[0-8]|8[1-35-8]|9[1-5]|3[45789])|4(?:1[246]|2[46]))\d{7}$/.test(d);break;case"US":default:c=/^(?:(1-?)|(\+1 ?))?\(?\d{3}\)?[-.\s]?\d{3}[-.\s]?\d{4}$/.test(d);break}return{message:r$d(t.l10n&&t.l10n.phone?a.message||t.l10n.phone.country:a.message,t.l10n&&t.l10n.phone&&t.l10n.phone.countries?t.l10n.phone.countries[r]:r),valid:c}}}}function e$g(){return{validate:function validate(e){if(e.value===""){return{valid:true}}if(!/^\d{9}$/.test(e.value)){return{valid:false}}var t=0;for(var a=0;a<e.value.length;a+=3){t+=parseInt(e.value.charAt(a),10)*3+parseInt(e.value.charAt(a+1),10)*7+parseInt(e.value.charAt(a+2),10)}return{valid:t!==0&&t%10===0}}}}function t$z(){return{validate:function validate(t){if(t.value===""){return{valid:true}}var e=t.value.toUpperCase();if(!/^[0-9A-Z]{7}$/.test(e)){return{valid:false}}var r=[1,3,1,7,3,9,1];var a=e.length;var l=0;for(var _t=0;_t<a-1;_t++){l+=r[_t]*parseInt(e.charAt(_t),36)}l=(10-l%10)%10;return{valid:"".concat(l)===e.charAt(a-1)}}}}function e$f(){return{validate:function validate(e){return{valid:e.value===""||/^\d{9}$/.test(e.value)&&t$15(e.value)}}}}function e$e(){return{validate:function validate(e){if(e.value===""){return{valid:true}}var t=e.value.length;var l=0;var r;for(var a=0;a<t;a++){r=parseInt(e.value.charAt(a),10);if(a%2===0){r=r*2;if(r>9){r-=9}}l+=r}return{valid:l%10===0}}}}function e$d(){var e=function e(t,_e){var s=Math.pow(10,_e);var a=t*s;var n;switch(true){case a===0:n=0;break;case a>0:n=1;break;case a<0:n=-1;break}var r=a%1===.5*n;return r?(Math.floor(a)+(n>0?1:0))/s:Math.round(a)/s};var s=function s(t,_s){if(_s===0){return 1}var a="".concat(t).split(".");var n="".concat(_s).split(".");var r=(a.length===1?0:a[1].length)+(n.length===1?0:n[1].length);return e(t-_s*Math.floor(t/_s),r)};return{validate:function validate(e){if(e.value===""){return{valid:true}}var a=parseFloat(e.value);if(isNaN(a)||!isFinite(a)){return{valid:false}}var n=Object.assign({},{baseValue:0,message:"",step:1},e.options);var r=s(a-n.baseValue,n.step);return{message:r$d(e.l10n?n.message||e.l10n.step["default"]:n.message,"".concat(n.step)),valid:r===0||r===n.step}}}}function s$5(){return{validate:function validate(s){if(s.value===""){return{valid:true}}var A=Object.assign({},{message:""},s.options);var i={3:/^[0-9A-F]{8}-[0-9A-F]{4}-3[0-9A-F]{3}-[0-9A-F]{4}-[0-9A-F]{12}$/i,4:/^[0-9A-F]{8}-[0-9A-F]{4}-4[0-9A-F]{3}-[89AB][0-9A-F]{3}-[0-9A-F]{12}$/i,5:/^[0-9A-F]{8}-[0-9A-F]{4}-5[0-9A-F]{3}-[89AB][0-9A-F]{3}-[0-9A-F]{12}$/i,all:/^[0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12}$/i};var n=A.version?"".concat(A.version):"all";return{message:A.version?r$d(s.l10n?A.message||s.l10n.uuid.version:A.message,A.version):s.l10n?s.l10n.uuid["default"]:A.message,valid:null===i[n]?true:i[n].test(s.value)}}}}function t$y(t){var e=t.replace("-","");if(/^AR[0-9]{11}$/.test(e)){e=e.substr(2)}if(!/^[0-9]{11}$/.test(e)){return{meta:{},valid:false}}var r=[5,4,3,2,7,6,5,4,3,2];var a=0;for(var _t=0;_t<10;_t++){a+=parseInt(e.charAt(_t),10)*r[_t]}a=11-a%11;if(a===11){a=0}return{meta:{},valid:"".concat(a)===e.substr(10)}}function t$x(t){var e=t;if(/^ATU[0-9]{8}$/.test(e)){e=e.substr(2)}if(!/^U[0-9]{8}$/.test(e)){return{meta:{},valid:false}}e=e.substr(1);var r=[1,2,1,2,1,2,1];var s=0;var a=0;for(var _t=0;_t<7;_t++){a=parseInt(e.charAt(_t),10)*r[_t];if(a>9){a=Math.floor(a/10)+a%10}s+=a}s=10-(s+4)%10;if(s===10){s=0}return{meta:{},valid:"".concat(s)===e.substr(7,1)}}function t$w(t){var e=t;if(/^BE[0]?[0-9]{9}$/.test(e)){e=e.substr(2)}if(!/^[0]?[0-9]{9}$/.test(e)){return{meta:{},valid:false}}if(e.length===9){e="0".concat(e)}if(e.substr(1,1)==="0"){return{meta:{},valid:false}}var s=parseInt(e.substr(0,8),10)+parseInt(e.substr(8,2),10);return{meta:{},valid:s%97===0}}function r$1(r){var e=r;if(/^BG[0-9]{9,10}$/.test(e)){e=e.substr(2)}if(!/^[0-9]{9,10}$/.test(e)){return{meta:{},valid:false}}var s=0;var n=0;if(e.length===9){for(n=0;n<8;n++){s+=parseInt(e.charAt(n),10)*(n+1)}s=s%11;if(s===10){s=0;for(n=0;n<8;n++){s+=parseInt(e.charAt(n),10)*(n+3)}s=s%11}s=s%10;return{meta:{},valid:"".concat(s)===e.substr(8)}}else{var _r=function _r(r){var e=parseInt(r.substr(0,2),10)+1900;var s=parseInt(r.substr(2,2),10);var n=parseInt(r.substr(4,2),10);if(s>40){e+=100;s-=40}else if(s>20){e-=100;s-=20}if(!t$Y(e,s,n)){return false}var a=[2,4,8,5,10,9,7,3,6];var l=0;for(var _t=0;_t<9;_t++){l+=parseInt(r.charAt(_t),10)*a[_t]}l=l%11%10;return"".concat(l)===r.substr(9,1)};var _s=function _s(t){var r=[21,19,17,13,11,9,7,3,1];var e=0;for(var _s2=0;_s2<9;_s2++){e+=parseInt(t.charAt(_s2),10)*r[_s2]}e=e%10;return"".concat(e)===t.substr(9,1)};var _n=function _n(t){var r=[4,3,2,7,6,5,4,3,2];var e=0;for(var _s3=0;_s3<9;_s3++){e+=parseInt(t.charAt(_s3),10)*r[_s3]}e=11-e%11;if(e===10){return false}if(e===11){e=0}return"".concat(e)===t.substr(9,1)};return{meta:{},valid:_r(e)||_s(e)||_n(e)}}}function t$v(t){if(t===""){return{meta:{},valid:true}}var e=t.replace(/[^\d]+/g,"");if(e===""||e.length!==14){return{meta:{},valid:false}}if(e==="00000000000000"||e==="11111111111111"||e==="22222222222222"||e==="33333333333333"||e==="44444444444444"||e==="55555555555555"||e==="66666666666666"||e==="77777777777777"||e==="88888888888888"||e==="99999999999999"){return{meta:{},valid:false}}var r=e.length-2;var a=e.substring(0,r);var l=e.substring(r);var n=0;var i=r-7;var s;for(s=r;s>=1;s--){n+=parseInt(a.charAt(r-s),10)*i--;if(i<2){i=9}}var f=n%11<2?0:11-n%11;if(f!==parseInt(l.charAt(0),10)){return{meta:{},valid:false}}r=r+1;a=e.substring(0,r);n=0;i=r-7;for(s=r;s>=1;s--){n+=parseInt(a.charAt(r-s),10)*i--;if(i<2){i=9}}f=n%11<2?0:11-n%11;return{meta:{},valid:f===parseInt(l.charAt(1),10)}}function t$u(t){var e=t;if(/^CHE[0-9]{9}(MWST|TVA|IVA|TPV)?$/.test(e)){e=e.substr(2)}if(!/^E[0-9]{9}(MWST|TVA|IVA|TPV)?$/.test(e)){return{meta:{},valid:false}}e=e.substr(1);var r=[5,4,3,2,7,6,5,4];var s=0;for(var _t=0;_t<8;_t++){s+=parseInt(e.charAt(_t),10)*r[_t]}s=11-s%11;if(s===10){return{meta:{},valid:false}}if(s===11){s=0}return{meta:{},valid:"".concat(s)===e.substr(8,1)}}function t$t(t){var e=t;if(/^CY[0-5|9][0-9]{7}[A-Z]$/.test(e)){e=e.substr(2)}if(!/^[0-5|9][0-9]{7}[A-Z]$/.test(e)){return{meta:{},valid:false}}if(e.substr(0,2)==="12"){return{meta:{},valid:false}}var r=0;var s={0:1,1:0,2:5,3:7,4:9,5:13,6:15,7:17,8:19,9:21};for(var _t=0;_t<8;_t++){var a=parseInt(e.charAt(_t),10);if(_t%2===0){a=s["".concat(a)]}r+=a}return{meta:{},valid:"".concat("ABCDEFGHIJKLMNOPQRSTUVWXYZ"[r%26])===e.substr(8,1)}}function e$c(e){var r=e;if(/^CZ[0-9]{8,10}$/.test(r)){r=r.substr(2)}if(!/^[0-9]{8,10}$/.test(r)){return{meta:{},valid:false}}var a=0;var s=0;if(r.length===8){if("".concat(r.charAt(0))==="9"){return{meta:{},valid:false}}a=0;for(s=0;s<7;s++){a+=parseInt(r.charAt(s),10)*(8-s)}a=11-a%11;if(a===10){a=0}if(a===11){a=1}return{meta:{},valid:"".concat(a)===r.substr(7,1)}}else if(r.length===9&&"".concat(r.charAt(0))==="6"){a=0;for(s=0;s<7;s++){a+=parseInt(r.charAt(s+1),10)*(8-s)}a=11-a%11;if(a===10){a=0}if(a===11){a=1}a=[8,7,6,5,4,3,2,1,0,9,10][a-1];return{meta:{},valid:"".concat(a)===r.substr(8,1)}}else if(r.length===9||r.length===10){var _e=1900+parseInt(r.substr(0,2),10);var _a=parseInt(r.substr(2,2),10)%50%20;var _s=parseInt(r.substr(4,2),10);if(r.length===9){if(_e>=1980){_e-=100}if(_e>1953){return{meta:{},valid:false}}}else if(_e<1954){_e+=100}if(!t$Y(_e,_a,_s)){return{meta:{},valid:false}}if(r.length===10){var _t=parseInt(r.substr(0,9),10)%11;if(_e<1985){_t=_t%10}return{meta:{},valid:"".concat(_t)===r.substr(9,1)}}return{meta:{},valid:true}}return{meta:{},valid:false}}function e$b(e){var r=e;if(/^DE[0-9]{9}$/.test(r)){r=r.substr(2)}if(!/^[0-9]{9}$/.test(r)){return{meta:{},valid:false}}return{meta:{},valid:t$14(r)}}function t$s(t){var e=t;if(/^DK[0-9]{8}$/.test(e)){e=e.substr(2)}if(!/^[0-9]{8}$/.test(e)){return{meta:{},valid:false}}var r=0;var a=[2,7,6,5,4,3,2,1];for(var _t=0;_t<8;_t++){r+=parseInt(e.charAt(_t),10)*a[_t]}return{meta:{},valid:r%11===0}}function t$r(t){var e=t;if(/^EE[0-9]{9}$/.test(e)){e=e.substr(2)}if(!/^[0-9]{9}$/.test(e)){return{meta:{},valid:false}}var r=0;var a=[3,7,1,3,7,1,3,7,1];for(var _t=0;_t<9;_t++){r+=parseInt(e.charAt(_t),10)*a[_t]}return{meta:{},valid:r%10===0}}function t$q(t){var e=t;if(/^ES[0-9A-Z][0-9]{7}[0-9A-Z]$/.test(e)){e=e.substr(2)}if(!/^[0-9A-Z][0-9]{7}[0-9A-Z]$/.test(e)){return{meta:{},valid:false}}var s=function s(t){var e=parseInt(t.substr(0,8),10);return"".concat("TRWAGMYFPDXBNJZSQVHLCKE"[e%23])===t.substr(8,1)};var r=function r(t){var e=["XYZ".indexOf(t.charAt(0)),t.substr(1)].join("");var s="TRWAGMYFPDXBNJZSQVHLCKE"[parseInt(e,10)%23];return"".concat(s)===t.substr(8,1)};var n=function n(t){var e=t.charAt(0);var s;if("KLM".indexOf(e)!==-1){s=parseInt(t.substr(1,8),10);s="TRWAGMYFPDXBNJZSQVHLCKE"[s%23];return"".concat(s)===t.substr(8,1)}else if("ABCDEFGHJNPQRSUVW".indexOf(e)!==-1){var _e=[2,1,2,1,2,1,2];var _s=0;var _r=0;for(var _n=0;_n<7;_n++){_r=parseInt(t.charAt(_n+1),10)*_e[_n];if(_r>9){_r=Math.floor(_r/10)+_r%10}_s+=_r}_s=10-_s%10;if(_s===10){_s=0}return"".concat(_s)===t.substr(8,1)||"JABCDEFGHI"[_s]===t.substr(8,1)}return false};var a=e.charAt(0);if(/^[0-9]$/.test(a)){return{meta:{type:"DNI"},valid:s(e)}}else if(/^[XYZ]$/.test(a)){return{meta:{type:"NIE"},valid:r(e)}}else{return{meta:{type:"CIF"},valid:n(e)}}}function t$p(t){var e=t;if(/^FI[0-9]{8}$/.test(e)){e=e.substr(2)}if(!/^[0-9]{8}$/.test(e)){return{meta:{},valid:false}}var r=[7,9,10,5,8,4,2,1];var a=0;for(var _t=0;_t<8;_t++){a+=parseInt(e.charAt(_t),10)*r[_t]}return{meta:{},valid:a%11===0}}function e$a(e){var r=e;if(/^FR[0-9A-Z]{2}[0-9]{9}$/.test(r)){r=r.substr(2)}if(!/^[0-9A-Z]{2}[0-9]{9}$/.test(r)){return{meta:{},valid:false}}if(r.substr(2,4)!=="000"){return{meta:{},valid:t$15(r.substr(2))}}if(/^[0-9]{2}$/.test(r.substr(0,2))){return{meta:{},valid:r.substr(0,2)==="".concat(parseInt(r.substr(2)+"12",10)%97)}}else{var _t="0123456789ABCDEFGHJKLMNPQRSTUVWXYZ";var _e;if(/^[0-9]$/.test(r.charAt(0))){_e=_t.indexOf(r.charAt(0))*24+_t.indexOf(r.charAt(1))-10}else{_e=_t.indexOf(r.charAt(0))*34+_t.indexOf(r.charAt(1))-100}return{meta:{},valid:(parseInt(r.substr(2),10)+1+Math.floor(_e/11))%11===_e%11}}}function t$o(t){var s=t;if(/^GB[0-9]{9}$/.test(s)||/^GB[0-9]{12}$/.test(s)||/^GBGD[0-9]{3}$/.test(s)||/^GBHA[0-9]{3}$/.test(s)||/^GB(GD|HA)8888[0-9]{5}$/.test(s)){s=s.substr(2)}if(!/^[0-9]{9}$/.test(s)&&!/^[0-9]{12}$/.test(s)&&!/^GD[0-9]{3}$/.test(s)&&!/^HA[0-9]{3}$/.test(s)&&!/^(GD|HA)8888[0-9]{5}$/.test(s)){return{meta:{},valid:false}}var e=s.length;if(e===5){var _t=s.substr(0,2);var _e=parseInt(s.substr(2),10);return{meta:{},valid:"GD"===_t&&_e<500||"HA"===_t&&_e>=500}}else if(e===11&&("GD8888"===s.substr(0,6)||"HA8888"===s.substr(0,6))){if("GD"===s.substr(0,2)&&parseInt(s.substr(6,3),10)>=500||"HA"===s.substr(0,2)&&parseInt(s.substr(6,3),10)<500){return{meta:{},valid:false}}return{meta:{},valid:parseInt(s.substr(6,3),10)%97===parseInt(s.substr(9,2),10)}}else if(e===9||e===12){var _t2=[8,7,6,5,4,3,2,10,1];var _e2=0;for(var _r=0;_r<9;_r++){_e2+=parseInt(s.charAt(_r),10)*_t2[_r]}_e2=_e2%97;var r=parseInt(s.substr(0,3),10)>=100?_e2===0||_e2===42||_e2===55:_e2===0;return{meta:{},valid:r}}return{meta:{},valid:true}}function t$n(t){var e=t;if(/^(GR|EL)[0-9]{9}$/.test(e)){e=e.substr(2)}if(!/^[0-9]{9}$/.test(e)){return{meta:{},valid:false}}if(e.length===8){e="0".concat(e)}var r=[256,128,64,32,16,8,4,2];var s=0;for(var _t=0;_t<8;_t++){s+=parseInt(e.charAt(_t),10)*r[_t]}s=s%11%10;return{meta:{},valid:"".concat(s)===e.substr(8,1)}}function e$9(e){var r=e;if(/^HR[0-9]{11}$/.test(r)){r=r.substr(2)}if(!/^[0-9]{11}$/.test(r)){return{meta:{},valid:false}}return{meta:{},valid:t$14(r)}}function t$m(t){var e=t;if(/^HU[0-9]{8}$/.test(e)){e=e.substr(2)}if(!/^[0-9]{8}$/.test(e)){return{meta:{},valid:false}}var r=[9,7,3,1,9,7,3,1];var a=0;for(var _t=0;_t<8;_t++){a+=parseInt(e.charAt(_t),10)*r[_t]}return{meta:{},valid:a%10===0}}function t$l(t){var e=t;if(/^IE[0-9][0-9A-Z*+][0-9]{5}[A-Z]{1,2}$/.test(e)){e=e.substr(2)}if(!/^[0-9][0-9A-Z*+][0-9]{5}[A-Z]{1,2}$/.test(e)){return{meta:{},valid:false}}var r=function r(t){var e=t;while(e.length<7){e="0".concat(e)}var r="WABCDEFGHIJKLMNOPQRSTUV";var s=0;for(var _t=0;_t<7;_t++){s+=parseInt(e.charAt(_t),10)*(8-_t)}s+=9*r.indexOf(e.substr(7));return r[s%23]};if(/^[0-9]+$/.test(e.substr(0,7))){return{meta:{},valid:e.charAt(7)===r("".concat(e.substr(0,7)).concat(e.substr(8)))}}else if("ABCDEFGHIJKLMNOPQRSTUVWXYZ+*".indexOf(e.charAt(1))!==-1){return{meta:{},valid:e.charAt(7)===r("".concat(e.substr(2,5)).concat(e.substr(0,1)))}}return{meta:{},valid:true}}function t$k(t){var e=t;if(/^IS[0-9]{5,6}$/.test(e)){e=e.substr(2)}return{meta:{},valid:/^[0-9]{5,6}$/.test(e)}}function e$8(e){var r=e;if(/^IT[0-9]{11}$/.test(r)){r=r.substr(2)}if(!/^[0-9]{11}$/.test(r)){return{meta:{},valid:false}}if(parseInt(r.substr(0,7),10)===0){return{meta:{},valid:false}}var a=parseInt(r.substr(7,3),10);if(a<1||a>201&&a!==999&&a!==888){return{meta:{},valid:false}}return{meta:{},valid:t$15(r)}}function t$j(t){var e=t;if(/^LT([0-9]{7}1[0-9]|[0-9]{10}1[0-9])$/.test(e)){e=e.substr(2)}if(!/^([0-9]{7}1[0-9]|[0-9]{10}1[0-9])$/.test(e)){return{meta:{},valid:false}}var r=e.length;var a=0;var l;for(l=0;l<r-1;l++){a+=parseInt(e.charAt(l),10)*(1+l%9)}var f=a%11;if(f===10){a=0;for(l=0;l<r-1;l++){a+=parseInt(e.charAt(l),10)*(1+(l+2)%9)}}f=f%11%10;return{meta:{},valid:"".concat(f)===e.charAt(r-1)}}function t$i(t){var e=t;if(/^LU[0-9]{8}$/.test(e)){e=e.substr(2)}if(!/^[0-9]{8}$/.test(e)){return{meta:{},valid:false}}return{meta:{},valid:"".concat(parseInt(e.substr(0,6),10)%89)===e.substr(6,2)}}function e$7(e){var r=e;if(/^LV[0-9]{11}$/.test(r)){r=r.substr(2)}if(!/^[0-9]{11}$/.test(r)){return{meta:{},valid:false}}var s=parseInt(r.charAt(0),10);var a=r.length;var n=0;var l=[];var i;if(s>3){n=0;l=[9,1,4,8,3,10,2,5,7,6,1];for(i=0;i<a;i++){n+=parseInt(r.charAt(i),10)*l[i]}n=n%11;return{meta:{},valid:n===3}}else{var _e=parseInt(r.substr(0,2),10);var _s=parseInt(r.substr(2,2),10);var f=parseInt(r.substr(4,2),10);f=f+1800+parseInt(r.charAt(6),10)*100;if(!t$Y(f,_s,_e)){return{meta:{},valid:false}}n=0;l=[10,5,8,4,2,1,6,3,7,9];for(i=0;i<a-1;i++){n+=parseInt(r.charAt(i),10)*l[i]}n=(n+1)%11%10;return{meta:{},valid:"".concat(n)===r.charAt(a-1)}}}function t$h(t){var e=t;if(/^MT[0-9]{8}$/.test(e)){e=e.substr(2)}if(!/^[0-9]{8}$/.test(e)){return{meta:{},valid:false}}var r=[3,4,6,7,8,9,10,1];var a=0;for(var _t=0;_t<8;_t++){a+=parseInt(e.charAt(_t),10)*r[_t]}return{meta:{},valid:a%37===0}}function t$g(t){return t.split("").map((function(t){var n=t.charCodeAt(0);return n>=65&&n<=90?n-55:t})).join("").split("").map((function(t){return parseInt(t,10)}))}function n(n){var e=t$g(n);var r=0;var o=e.length;for(var _t=0;_t<o-1;++_t){r=(r+e[_t])*10%97}r+=e[o-1];return r%97===1}function e$6(e){var i=e;if(/^NL[0-9]{9}B[0-9]{2}$/.test(i)){i=i.substr(2)}if(!/^[0-9]{9}B[0-9]{2}$/.test(i)){return{meta:{},valid:false}}var o=i.substr(0,9);return{meta:{},valid:e$o(o).valid||n("NL".concat(i))}}function t$f(t){var e=t;if(/^NO[0-9]{9}$/.test(e)){e=e.substr(2)}if(!/^[0-9]{9}$/.test(e)){return{meta:{},valid:false}}var r=[3,2,7,6,5,4,3,2];var s=0;for(var _t=0;_t<8;_t++){s+=parseInt(e.charAt(_t),10)*r[_t]}s=11-s%11;if(s===11){s=0}return{meta:{},valid:"".concat(s)===e.substr(8,1)}}function t$e(t){var e=t;if(/^PL[0-9]{10}$/.test(e)){e=e.substr(2)}if(!/^[0-9]{10}$/.test(e)){return{meta:{},valid:false}}var r=[6,5,7,2,3,4,5,6,7,-1];var a=0;for(var _t=0;_t<10;_t++){a+=parseInt(e.charAt(_t),10)*r[_t]}return{meta:{},valid:a%11===0}}function t$d(t){var e=t;if(/^PT[0-9]{9}$/.test(e)){e=e.substr(2)}if(!/^[0-9]{9}$/.test(e)){return{meta:{},valid:false}}var r=[9,8,7,6,5,4,3,2];var s=0;for(var _t=0;_t<8;_t++){s+=parseInt(e.charAt(_t),10)*r[_t]}s=11-s%11;if(s>9){s=0}return{meta:{},valid:"".concat(s)===e.substr(8,1)}}function t$c(t){var e=t;if(/^RO[1-9][0-9]{1,9}$/.test(e)){e=e.substr(2)}if(!/^[1-9][0-9]{1,9}$/.test(e)){return{meta:{},valid:false}}var s=e.length;var r=[7,5,3,2,1,7,5,3,2].slice(10-s);var l=0;for(var _t=0;_t<s-1;_t++){l+=parseInt(e.charAt(_t),10)*r[_t]}l=10*l%11%10;return{meta:{},valid:"".concat(l)===e.substr(s-1,1)}}function t$b(t){var e=t;if(/^RS[0-9]{9}$/.test(e)){e=e.substr(2)}if(!/^[0-9]{9}$/.test(e)){return{meta:{},valid:false}}var r=10;var a=0;for(var _t=0;_t<8;_t++){a=(parseInt(e.charAt(_t),10)+r)%10;if(a===0){a=10}r=2*a%11}return{meta:{},valid:(r+parseInt(e.substr(8,1),10))%10===1}}function t$a(t){var e=t;if(/^RU([0-9]{10}|[0-9]{12})$/.test(e)){e=e.substr(2)}if(!/^([0-9]{10}|[0-9]{12})$/.test(e)){return{meta:{},valid:false}}var r=0;if(e.length===10){var _t=[2,4,10,3,5,9,4,6,8,0];var s=0;for(r=0;r<10;r++){s+=parseInt(e.charAt(r),10)*_t[r]}s=s%11;if(s>9){s=s%10}return{meta:{},valid:"".concat(s)===e.substr(9,1)}}else if(e.length===12){var _t2=[7,2,4,10,3,5,9,4,6,8,0];var _s=[3,7,2,4,10,3,5,9,4,6,8,0];var a=0;var l=0;for(r=0;r<11;r++){a+=parseInt(e.charAt(r),10)*_t2[r];l+=parseInt(e.charAt(r),10)*_s[r]}a=a%11;if(a>9){a=a%10}l=l%11;if(l>9){l=l%10}return{meta:{},valid:"".concat(a)===e.substr(10,1)&&"".concat(l)===e.substr(11,1)}}return{meta:{},valid:true}}function e$5(e){var r=e;if(/^SE[0-9]{10}01$/.test(r)){r=r.substr(2)}if(!/^[0-9]{10}01$/.test(r)){return{meta:{},valid:false}}r=r.substr(0,10);return{meta:{},valid:t$15(r)}}function t$9(t){var e=t.match(/^(SI)?([1-9][0-9]{7})$/);if(!e){return{meta:{},valid:false}}var r=e[1]?t.substr(2):t;var a=[8,7,6,5,4,3,2];var s=0;for(var _t=0;_t<7;_t++){s+=parseInt(r.charAt(_t),10)*a[_t]}s=11-s%11;if(s===10){s=0}return{meta:{},valid:"".concat(s)===r.substr(7,1)}}function t$8(t){var e=t;if(/^SK[1-9][0-9][(2-4)|(6-9)][0-9]{7}$/.test(e)){e=e.substr(2)}if(!/^[1-9][0-9][(2-4)|(6-9)][0-9]{7}$/.test(e)){return{meta:{},valid:false}}return{meta:{},valid:parseInt(e,10)%11===0}}function t$7(t){var e=t;if(/^VE[VEJPG][0-9]{9}$/.test(e)){e=e.substr(2)}if(!/^[VEJPG][0-9]{9}$/.test(e)){return{meta:{},valid:false}}var r={E:8,G:20,J:12,P:16,V:4};var s=[3,2,7,6,5,4,3,2];var a=r[e.charAt(0)];for(var _t=0;_t<8;_t++){a+=parseInt(e.charAt(_t+1),10)*s[_t]}a=11-a%11;if(a===11||a===10){a=0}return{meta:{},valid:"".concat(a)===e.substr(9,1)}}function t$6(t){var e=t;if(/^ZA4[0-9]{9}$/.test(e)){e=e.substr(2)}return{meta:{},valid:/^4[0-9]{9}$/.test(e)}}function x(){var x=["AR","AT","BE","BG","BR","CH","CY","CZ","DE","DK","EE","EL","ES","FI","FR","GB","GR","HR","HU","IE","IS","IT","LT","LU","LV","MT","NL","NO","PL","PT","RO","RU","RS","SE","SK","SI","VE","ZA"];return{validate:function validate(D){var F=D.value;if(F===""){return{valid:true}}var K=Object.assign({},{message:""},D.options);var N=F.substr(0,2);if("function"===typeof K.country){N=K.country.call(this)}else{N=K.country}if(x.indexOf(N)===-1){return{valid:true}}var P={meta:{},valid:true};switch(N.toLowerCase()){case"ar":P=t$y(F);break;case"at":P=t$x(F);break;case"be":P=t$w(F);break;case"bg":P=r$1(F);break;case"br":P=t$v(F);break;case"ch":P=t$u(F);break;case"cy":P=t$t(F);break;case"cz":P=e$c(F);break;case"de":P=e$b(F);break;case"dk":P=t$s(F);break;case"ee":P=t$r(F);break;case"el":P=t$n(F);break;case"es":P=t$q(F);break;case"fi":P=t$p(F);break;case"fr":P=e$a(F);break;case"gb":P=t$o(F);break;case"gr":P=t$n(F);break;case"hr":P=e$9(F);break;case"hu":P=t$m(F);break;case"ie":P=t$l(F);break;case"is":P=t$k(F);break;case"it":P=e$8(F);break;case"lt":P=t$j(F);break;case"lu":P=t$i(F);break;case"lv":P=e$7(F);break;case"mt":P=t$h(F);break;case"nl":P=e$6(F);break;case"no":P=t$f(F);break;case"pl":P=t$e(F);break;case"pt":P=t$d(F);break;case"ro":P=t$c(F);break;case"rs":P=t$b(F);break;case"ru":P=t$a(F);break;case"se":P=e$5(F);break;case"si":P=t$9(F);break;case"sk":P=t$8(F);break;case"ve":P=t$7(F);break;case"za":P=t$6(F);break}var Z=r$d(D.l10n&&D.l10n.vat?K.message||D.l10n.vat.country:K.message,D.l10n&&D.l10n.vat&&D.l10n.vat.countries?D.l10n.vat.countries[N.toUpperCase()]:N.toUpperCase());return Object.assign({},{message:Z},P)}}}function t$5(){return{validate:function validate(t){if(t.value===""){return{valid:true}}if(!/^[a-hj-npr-z0-9]{8}[0-9xX][a-hj-npr-z0-9]{8}$/i.test(t.value)){return{valid:false}}var e=t.value.toUpperCase();var r={A:1,B:2,C:3,D:4,E:5,F:6,G:7,H:8,J:1,K:2,L:3,M:4,N:5,P:7,R:9,S:2,T:3,U:4,V:5,W:6,X:7,Y:8,Z:9,0:0,1:1,2:2,3:3,4:4,5:5,6:6,7:7,8:8,9:9};var a=[8,7,6,5,4,3,2,10,0,9,8,7,6,5,4,3,2];var l=e.length;var n=0;for(var _t=0;_t<l;_t++){n+=r["".concat(e.charAt(_t))]*a[_t]}var u="".concat(n%11);if(u==="10"){u="X"}return{valid:u===e.charAt(8)}}}}function s$4(){var s=["AT","BG","BR","CA","CH","CZ","DE","DK","ES","FR","GB","IE","IN","IT","MA","NL","PL","PT","RO","RU","SE","SG","SK","US"];var a=function a(e){var s="[ABCDEFGHIJKLMNOPRSTUWYZ]";var a="[ABCDEFGHKLMNOPQRSTUVWXY]";var t="[ABCDEFGHJKPMNRSTUVWXY]";var r="[ABEHMNPRVWXY]";var u="[ABDEFGHJLNPQRSTUWXYZ]";var c=[new RegExp("^(".concat(s,"{1}").concat(a,"?[0-9]{1,2})(\\s*)([0-9]{1}").concat(u,"{2})$"),"i"),new RegExp("^(".concat(s,"{1}[0-9]{1}").concat(t,"{1})(\\s*)([0-9]{1}").concat(u,"{2})$"),"i"),new RegExp("^(".concat(s,"{1}").concat(a,"{1}?[0-9]{1}").concat(r,"{1})(\\s*)([0-9]{1}").concat(u,"{2})$"),"i"),new RegExp("^(BF1)(\\s*)([0-6]{1}[ABDEFGHJLNPQRST]{1}[ABDEFGHJLNPQRSTUWZYZ]{1})$","i"),/^(GIR)(\s*)(0AA)$/i,/^(BFPO)(\s*)([0-9]{1,4})$/i,/^(BFPO)(\s*)(c\/o\s*[0-9]{1,3})$/i,/^([A-Z]{4})(\s*)(1ZZ)$/i,/^(AI-2640)$/i];for(var _i=0,_c=c;_i<_c.length;_i++){var _s=_c[_i];if(_s.test(e)){return true}}return false};return{validate:function validate(t){var r=Object.assign({},{message:""},t.options);if(t.value===""||!r.country){return{valid:true}}var u=t.value.substr(0,2);if("function"===typeof r.country){u=r.country.call(this)}else{u=r.country}if(!u||s.indexOf(u.toUpperCase())===-1){return{valid:true}}var c=false;u=u.toUpperCase();switch(u){case"AT":c=/^([1-9]{1})(\d{3})$/.test(t.value);break;case"BG":c=/^([1-9]{1}[0-9]{3})$/.test(t.value);break;case"BR":c=/^(\d{2})([.]?)(\d{3})([-]?)(\d{3})$/.test(t.value);break;case"CA":c=/^(?:A|B|C|E|G|H|J|K|L|M|N|P|R|S|T|V|X|Y){1}[0-9]{1}(?:A|B|C|E|G|H|J|K|L|M|N|P|R|S|T|V|W|X|Y|Z){1}\s?[0-9]{1}(?:A|B|C|E|G|H|J|K|L|M|N|P|R|S|T|V|W|X|Y|Z){1}[0-9]{1}$/i.test(t.value);break;case"CH":c=/^([1-9]{1})(\d{3})$/.test(t.value);break;case"CZ":c=/^(\d{3})([ ]?)(\d{2})$/.test(t.value);break;case"DE":c=/^(?!01000|99999)(0[1-9]\d{3}|[1-9]\d{4})$/.test(t.value);break;case"DK":c=/^(DK(-|\s)?)?\d{4}$/i.test(t.value);break;case"ES":c=/^(?:0[1-9]|[1-4][0-9]|5[0-2])\d{3}$/.test(t.value);break;case"FR":c=/^[0-9]{5}$/i.test(t.value);break;case"GB":c=a(t.value);break;case"IN":c=/^\d{3}\s?\d{3}$/.test(t.value);break;case"IE":c=/^(D6W|[ACDEFHKNPRTVWXY]\d{2})\s[0-9ACDEFHKNPRTVWXY]{4}$/.test(t.value);break;case"IT":c=/^(I-|IT-)?\d{5}$/i.test(t.value);break;case"MA":c=/^[1-9][0-9]{4}$/i.test(t.value);break;case"NL":c=/^[1-9][0-9]{3} ?(?!sa|sd|ss)[a-z]{2}$/i.test(t.value);break;case"PL":c=/^[0-9]{2}-[0-9]{3}$/.test(t.value);break;case"PT":c=/^[1-9]\d{3}-\d{3}$/.test(t.value);break;case"RO":c=/^(0[1-8]{1}|[1-9]{1}[0-5]{1})?[0-9]{4}$/i.test(t.value);break;case"RU":c=/^[0-9]{6}$/i.test(t.value);break;case"SE":c=/^(S-)?\d{3}\s?\d{2}$/i.test(t.value);break;case"SG":c=/^([0][1-9]|[1-6][0-9]|[7]([0-3]|[5-9])|[8][0-2])(\d{4})$/i.test(t.value);break;case"SK":c=/^(\d{3})([ ]?)(\d{2})$/.test(t.value);break;case"US":default:c=/^\d{4,5}([-]?\d{4})?$/.test(t.value);break}return{message:r$d(t.l10n&&t.l10n.zipCode?r.message||t.l10n.zipCode.country:r.message,t.l10n&&t.l10n.zipCode&&t.l10n.zipCode.countries?t.l10n.zipCode.countries[u]:u),valid:c}}}}var s$3={between:s$9,blank:t$10,callback:o$4,choice:t$_,creditCard:l$2,date:n$1,different:o$3,digits:e$G,emailAddress:t$X,file:e$F,greaterThan:a$7,identical:o$2,integer:a$6,ip:d,lessThan:s$8,notEmpty:t$W,numeric:a$5,promise:r$c,regexp:e$E,remote:a$4,stringCase:e$C,stringLength:t$U,uri:t$T,base64:a$3,bic:a$2,color:e$B,cusip:t$S,ean:e$A,ein:e$z,grid:r$b,hex:e$y,iban:Z,id:F,imei:t$A,imo:e$m,isbn:e$l,isin:M,ismn:e$k,issn:e$j,mac:a$1,meid:e$i,phone:e$h,rtn:e$g,sedol:t$z,siren:e$f,siret:e$e,step:e$d,uuid:s$5,vat:x,vin:t$5,zipCode:s$4};var l$1=function(){function l(i,s){_classCallCheck(this,l);this.elements={};this.ee=s$a();this.filter=t$11();this.plugins={};this.results=new Map;this.validators={};this.form=i;this.fields=s}_createClass(l,[{key:"on",value:function on(e,t){this.ee.on(e,t);return this}},{key:"off",value:function off(e,t){this.ee.off(e,t);return this}},{key:"emit",value:function emit(e){var _this$ee;for(var _len=arguments.length,t=new Array(_len>1?_len-1:0),_key=1;_key<_len;_key++){t[_key-1]=arguments[_key]}(_this$ee=this.ee).emit.apply(_this$ee,[e].concat(t));return this}},{key:"registerPlugin",value:function registerPlugin(e,t){if(this.plugins[e]){throw new Error("The plguin ".concat(e," is registered"))}t.setCore(this);t.install();this.plugins[e]=t;return this}},{key:"deregisterPlugin",value:function deregisterPlugin(e){var t=this.plugins[e];if(t){t.uninstall()}delete this.plugins[e];return this}},{key:"registerValidator",value:function registerValidator(e,t){if(this.validators[e]){throw new Error("The validator ".concat(e," is registered"))}this.validators[e]=t;return this}},{key:"registerFilter",value:function registerFilter(e,t){this.filter.add(e,t);return this}},{key:"deregisterFilter",value:function deregisterFilter(e,t){this.filter.remove(e,t);return this}},{key:"executeFilter",value:function executeFilter(e,t,i){return this.filter.execute(e,t,i)}},{key:"addField",value:function addField(e,t){var i=Object.assign({},{selector:"",validators:{}},t);this.fields[e]=this.fields[e]?{selector:i.selector||this.fields[e].selector,validators:Object.assign({},this.fields[e].validators,i.validators)}:i;this.elements[e]=this.queryElements(e);this.emit("core.field.added",{elements:this.elements[e],field:e,options:this.fields[e]});return this}},{key:"removeField",value:function removeField(e){if(!this.fields[e]){throw new Error("The field ".concat(e," validators are not defined. Please ensure the field is added first"))}var t=this.elements[e];var i=this.fields[e];delete this.elements[e];delete this.fields[e];this.emit("core.field.removed",{elements:t,field:e,options:i});return this}},{key:"validate",value:function validate(){var _this=this;this.emit("core.form.validating",{formValidation:this});return this.filter.execute("validate-pre",Promise.resolve(),[]).then((function(){return Promise.all(Object.keys(_this.fields).map((function(e){return _this.validateField(e)}))).then((function(e){switch(true){case e.indexOf("Invalid")!==-1:_this.emit("core.form.invalid",{formValidation:_this});return Promise.resolve("Invalid");case e.indexOf("NotValidated")!==-1:_this.emit("core.form.notvalidated",{formValidation:_this});return Promise.resolve("NotValidated");default:_this.emit("core.form.valid",{formValidation:_this});return Promise.resolve("Valid")}}))}))}},{key:"validateField",value:function validateField(e){var _this2=this;var t=this.results.get(e);if(t==="Valid"||t==="Invalid"){return Promise.resolve(t)}this.emit("core.field.validating",e);var i=this.elements[e];if(i.length===0){this.emit("core.field.valid",e);return Promise.resolve("Valid")}var s=i[0].getAttribute("type");if("radio"===s||"checkbox"===s||i.length===1){return this.validateElement(e,i[0])}else{return Promise.all(i.map((function(t){return _this2.validateElement(e,t)}))).then((function(t){switch(true){case t.indexOf("Invalid")!==-1:_this2.emit("core.field.invalid",e);_this2.results.set(e,"Invalid");return Promise.resolve("Invalid");case t.indexOf("NotValidated")!==-1:_this2.emit("core.field.notvalidated",e);_this2.results["delete"](e);return Promise.resolve("NotValidated");default:_this2.emit("core.field.valid",e);_this2.results.set(e,"Valid");return Promise.resolve("Valid")}}))}}},{key:"validateElement",value:function validateElement(e,t){var _this3=this;this.results["delete"](e);var i=this.elements[e];var s=this.filter.execute("element-ignored",false,[e,t,i]);if(s){this.emit("core.element.ignored",{element:t,elements:i,field:e});return Promise.resolve("Ignored")}var _l=this.fields[e].validators;this.emit("core.element.validating",{element:t,elements:i,field:e});var r=Object.keys(_l).map((function(i){return function(){return _this3.executeValidator(e,t,i,_l[i])}}));return this.waterfall(r).then((function(s){var _l2=s.indexOf("Invalid")===-1;_this3.emit("core.element.validated",{element:t,elements:i,field:e,valid:_l2});var r=t.getAttribute("type");if("radio"===r||"checkbox"===r||i.length===1){_this3.emit(_l2?"core.field.valid":"core.field.invalid",e)}return Promise.resolve(_l2?"Valid":"Invalid")}))["catch"]((function(s){_this3.emit("core.element.notvalidated",{element:t,elements:i,field:e});return Promise.resolve(s)}))}},{key:"executeValidator",value:function executeValidator(e,t,i,s){var _this4=this;var _l3=this.elements[e];var r=this.filter.execute("validator-name",i,[i,e]);s.message=this.filter.execute("validator-message",s.message,[this.locale,e,r]);if(!this.validators[r]||s.enabled===false){this.emit("core.validator.validated",{element:t,elements:_l3,field:e,result:this.normalizeResult(e,r,{valid:true}),validator:r});return Promise.resolve("Valid")}var a=this.validators[r];var d=this.getElementValue(e,t,r);var o=this.filter.execute("field-should-validate",true,[e,t,d,i]);if(!o){this.emit("core.validator.notvalidated",{element:t,elements:_l3,field:e,validator:i});return Promise.resolve("NotValidated")}this.emit("core.validator.validating",{element:t,elements:_l3,field:e,validator:i});var n=a().validate({element:t,elements:_l3,field:e,l10n:this.localization,options:s,value:d});var h="function"===typeof n["then"];if(h){return n.then((function(s){var r=_this4.normalizeResult(e,i,s);_this4.emit("core.validator.validated",{element:t,elements:_l3,field:e,result:r,validator:i});return r.valid?"Valid":"Invalid"}))}else{var _s=this.normalizeResult(e,i,n);this.emit("core.validator.validated",{element:t,elements:_l3,field:e,result:_s,validator:i});return Promise.resolve(_s.valid?"Valid":"Invalid")}}},{key:"getElementValue",value:function getElementValue(e,t,s){var _l4=e$H(this.form,e,t,this.elements[e]);return this.filter.execute("field-value",_l4,[_l4,e,t,s])}},{key:"getElements",value:function getElements(e){return this.elements[e]}},{key:"getFields",value:function getFields(){return this.fields}},{key:"getFormElement",value:function getFormElement(){return this.form}},{key:"getLocale",value:function getLocale(){return this.locale}},{key:"getPlugin",value:function getPlugin(e){return this.plugins[e]}},{key:"updateFieldStatus",value:function updateFieldStatus(e,t,i){var _this5=this;var s=this.elements[e];var _l5=s[0].getAttribute("type");var r="radio"===_l5||"checkbox"===_l5?[s[0]]:s;r.forEach((function(s){return _this5.updateElementStatus(e,s,t,i)}));if(!i){switch(t){case"NotValidated":this.emit("core.field.notvalidated",e);this.results["delete"](e);break;case"Validating":this.emit("core.field.validating",e);this.results["delete"](e);break;case"Valid":this.emit("core.field.valid",e);this.results.set(e,"Valid");break;case"Invalid":this.emit("core.field.invalid",e);this.results.set(e,"Invalid");break}}else if(t==="Invalid"){this.emit("core.field.invalid",e);this.results.set(e,"Invalid")}return this}},{key:"updateElementStatus",value:function updateElementStatus(e,t,i,s){var _this6=this;var _l6=this.elements[e];var r=this.fields[e].validators;var a=s?[s]:Object.keys(r);switch(i){case"NotValidated":a.forEach((function(i){return _this6.emit("core.validator.notvalidated",{element:t,elements:_l6,field:e,validator:i})}));this.emit("core.element.notvalidated",{element:t,elements:_l6,field:e});break;case"Validating":a.forEach((function(i){return _this6.emit("core.validator.validating",{element:t,elements:_l6,field:e,validator:i})}));this.emit("core.element.validating",{element:t,elements:_l6,field:e});break;case"Valid":a.forEach((function(i){return _this6.emit("core.validator.validated",{element:t,elements:_l6,field:e,result:{message:r[i].message,valid:true},validator:i})}));this.emit("core.element.validated",{element:t,elements:_l6,field:e,valid:true});break;case"Invalid":a.forEach((function(i){return _this6.emit("core.validator.validated",{element:t,elements:_l6,field:e,result:{message:r[i].message,valid:false},validator:i})}));this.emit("core.element.validated",{element:t,elements:_l6,field:e,valid:false});break}return this}},{key:"resetForm",value:function resetForm(e){var _this7=this;Object.keys(this.fields).forEach((function(t){return _this7.resetField(t,e)}));this.emit("core.form.reset",{formValidation:this,reset:e});return this}},{key:"resetField",value:function resetField(e,t){if(t){var _t=this.elements[e];var _i=_t[0].getAttribute("type");_t.forEach((function(e){if("radio"===_i||"checkbox"===_i){e.removeAttribute("selected");e.removeAttribute("checked");e.checked=false}else{e.setAttribute("value","");if(e instanceof HTMLInputElement||e instanceof HTMLTextAreaElement){e.value=""}}}))}this.updateFieldStatus(e,"NotValidated");this.emit("core.field.reset",{field:e,reset:t});return this}},{key:"revalidateField",value:function revalidateField(e){if(!this.fields[e]){return Promise.resolve("Ignored")}this.updateFieldStatus(e,"NotValidated");return this.validateField(e)}},{key:"disableValidator",value:function disableValidator(e,t){if(!this.fields[e]){return this}var i=this.elements[e];this.toggleValidator(false,e,t);this.emit("core.validator.disabled",{elements:i,field:e,formValidation:this,validator:t});return this}},{key:"enableValidator",value:function enableValidator(e,t){if(!this.fields[e]){return this}var i=this.elements[e];this.toggleValidator(true,e,t);this.emit("core.validator.enabled",{elements:i,field:e,formValidation:this,validator:t});return this}},{key:"updateValidatorOption",value:function updateValidatorOption(e,t,i,s){if(this.fields[e]&&this.fields[e].validators&&this.fields[e].validators[t]){this.fields[e].validators[t][i]=s}return this}},{key:"setFieldOptions",value:function setFieldOptions(e,t){this.fields[e]=t;return this}},{key:"destroy",value:function destroy(){var _this8=this;Object.keys(this.plugins).forEach((function(e){return _this8.plugins[e].uninstall()}));this.ee.clear();this.filter.clear();this.results.clear();this.plugins={};return this}},{key:"setLocale",value:function setLocale(e,t){this.locale=e;this.localization=t;return this}},{key:"waterfall",value:function waterfall(e){return e.reduce((function(e,t){return e.then((function(e){return t().then((function(t){e.push(t);return e}))}))}),Promise.resolve([]))}},{key:"queryElements",value:function queryElements(e){var t=this.fields[e].selector?"#"===this.fields[e].selector.charAt(0)?'[id="'.concat(this.fields[e].selector.substring(1),'"]'):this.fields[e].selector:'[name="'.concat(e,'"]');return[].slice.call(this.form.querySelectorAll(t))}},{key:"normalizeResult",value:function normalizeResult(e,t,i){var s=this.fields[e].validators[t];return Object.assign({},i,{message:i.message||(s?s.message:"")||(this.localization&&this.localization[t]&&this.localization[t]["default"]?this.localization[t]["default"]:"")||"The field ".concat(e," is not valid")})}},{key:"toggleValidator",value:function toggleValidator(e,t,i){var _this9=this;var s=this.fields[t].validators;if(i&&s&&s[i]){this.fields[t].validators[i].enabled=e}else if(!i){Object.keys(s).forEach((function(i){return _this9.fields[t].validators[i].enabled=e}))}return this.updateFieldStatus(t,"NotValidated",i)}}]);return l}();function r(e,t){var i=Object.assign({},{fields:{},locale:"en_US",plugins:{},init:function init(e){}},t);var r=new l$1(e,i.fields);r.setLocale(i.locale,i.localization);Object.keys(i.plugins).forEach((function(e){return r.registerPlugin(e,i.plugins[e])}));Object.keys(s$3).forEach((function(e){return r.registerValidator(e,s$3[e])}));i.init(r);Object.keys(i.fields).forEach((function(e){return r.addField(e,i.fields[e])}));return r}var t$4=function(){function t(_t){_classCallCheck(this,t);this.opts=_t}_createClass(t,[{key:"setCore",value:function setCore(_t2){this.core=_t2;return this}},{key:"install",value:function install(){}},{key:"uninstall",value:function uninstall(){}}]);return t}();var index$2={getFieldValue:e$H};var e$4=function(_t){_inherits(e,_t);var _super=_createSuper(e);function e(t){var _this;_classCallCheck(this,e);_this=_super.call(this,t);_this.opts=t||{};_this.validatorNameFilter=_this.getValidatorName.bind(_assertThisInitialized(_this));return _this}_createClass(e,[{key:"install",value:function install(){this.core.registerFilter("validator-name",this.validatorNameFilter)}},{key:"uninstall",value:function uninstall(){this.core.deregisterFilter("validator-name",this.validatorNameFilter)}},{key:"getValidatorName",value:function getValidatorName(t,_e){return this.opts[t]||t}}]);return e}(t$4);var i$3=function(_e){_inherits(i,_e);var _super=_createSuper(i);function i(){var _this;_classCallCheck(this,i);_this=_super.call(this,{});_this.elementValidatedHandler=_this.onElementValidated.bind(_assertThisInitialized(_this));_this.fieldValidHandler=_this.onFieldValid.bind(_assertThisInitialized(_this));_this.fieldInvalidHandler=_this.onFieldInvalid.bind(_assertThisInitialized(_this));_this.messageDisplayedHandler=_this.onMessageDisplayed.bind(_assertThisInitialized(_this));return _this}_createClass(i,[{key:"install",value:function install(){this.core.on("core.field.valid",this.fieldValidHandler).on("core.field.invalid",this.fieldInvalidHandler).on("core.element.validated",this.elementValidatedHandler).on("plugins.message.displayed",this.messageDisplayedHandler)}},{key:"uninstall",value:function uninstall(){this.core.off("core.field.valid",this.fieldValidHandler).off("core.field.invalid",this.fieldInvalidHandler).off("core.element.validated",this.elementValidatedHandler).off("plugins.message.displayed",this.messageDisplayedHandler)}},{key:"onElementValidated",value:function onElementValidated(e){if(e.valid){e.element.setAttribute("aria-invalid","false");e.element.removeAttribute("aria-describedby")}}},{key:"onFieldValid",value:function onFieldValid(e){var _i=this.core.getElements(e);if(_i){_i.forEach((function(e){e.setAttribute("aria-invalid","false");e.removeAttribute("aria-describedby")}))}}},{key:"onFieldInvalid",value:function onFieldInvalid(e){var _i2=this.core.getElements(e);if(_i2){_i2.forEach((function(e){return e.setAttribute("aria-invalid","true")}))}}},{key:"onMessageDisplayed",value:function onMessageDisplayed(e){e.messageElement.setAttribute("role","alert");e.messageElement.setAttribute("aria-hidden","false");var _i3=this.core.getElements(e.field);var t=_i3.indexOf(e.element);var l="js-fv-".concat(e.field,"-").concat(t,"-").concat(Date.now(),"-message");e.messageElement.setAttribute("id",l);e.element.setAttribute("aria-describedby",l);var a=e.element.getAttribute("type");if("radio"===a||"checkbox"===a){_i3.forEach((function(e){return e.setAttribute("aria-describedby",l)}))}}}]);return i}(t$4);var t$3=function(_e){_inherits(t,_e);var _super=_createSuper(t);function t(e){var _this;_classCallCheck(this,t);_this=_super.call(this,e);_this.addedFields=new Map;_this.opts=Object.assign({},{html5Input:false,pluginPrefix:"data-fvp-",prefix:"data-fv-"},e);_this.fieldAddedHandler=_this.onFieldAdded.bind(_assertThisInitialized(_this));_this.fieldRemovedHandler=_this.onFieldRemoved.bind(_assertThisInitialized(_this));return _this}_createClass(t,[{key:"install",value:function install(){var _this2=this;this.parsePlugins();var e=this.parseOptions();Object.keys(e).forEach((function(_t){if(!_this2.addedFields.has(_t)){_this2.addedFields.set(_t,true)}_this2.core.addField(_t,e[_t])}));this.core.on("core.field.added",this.fieldAddedHandler).on("core.field.removed",this.fieldRemovedHandler)}},{key:"uninstall",value:function uninstall(){this.addedFields.clear();this.core.off("core.field.added",this.fieldAddedHandler).off("core.field.removed",this.fieldRemovedHandler)}},{key:"onFieldAdded",value:function onFieldAdded(e){var _this3=this;var _t2=e.elements;if(!_t2||_t2.length===0||this.addedFields.has(e.field)){return}this.addedFields.set(e.field,true);_t2.forEach((function(_t3){var s=_this3.parseElement(_t3);if(!_this3.isEmptyOption(s)){var _t12={selector:e.options.selector,validators:Object.assign({},e.options.validators||{},s.validators)};_this3.core.setFieldOptions(e.field,_t12)}}))}},{key:"onFieldRemoved",value:function onFieldRemoved(e){if(e.field&&this.addedFields.has(e.field)){this.addedFields["delete"](e.field)}}},{key:"parseOptions",value:function parseOptions(){var _this4=this;var e=this.opts.prefix;var _t5={};var s=this.core.getFields();var a=this.core.getFormElement();var i=[].slice.call(a.querySelectorAll("[name], [".concat(e,"field]")));i.forEach((function(s){var a=_this4.parseElement(s);if(!_this4.isEmptyOption(a)){var _i=s.getAttribute("name")||s.getAttribute("".concat(e,"field"));_t5[_i]=Object.assign({},_t5[_i],a)}}));Object.keys(_t5).forEach((function(e){Object.keys(_t5[e].validators).forEach((function(a){_t5[e].validators[a].enabled=_t5[e].validators[a].enabled||false;if(s[e]&&s[e].validators&&s[e].validators[a]){Object.assign(_t5[e].validators[a],s[e].validators[a])}}))}));return Object.assign({},s,_t5)}},{key:"createPluginInstance",value:function createPluginInstance(e,_t6){var s=e.split(".");var a=window||this;for(var _e2=0,_t13=s.length;_e2<_t13;_e2++){a=a[s[_e2]]}if(typeof a!=="function"){throw new Error("the plugin ".concat(e," doesn't exist"))}return new a(_t6)}},{key:"parsePlugins",value:function parsePlugins(){var _this5=this;var e=this.core.getFormElement();var _t8=new RegExp("^".concat(this.opts.pluginPrefix,"([a-z0-9-]+)(___)*([a-z0-9-]+)*$"));var s=e.attributes.length;var a={};for(var i=0;i<s;i++){var _s=e.attributes[i].name;var n=e.attributes[i].value;var r=_t8.exec(_s);if(r&&r.length===4){var _e3=this.toCamelCase(r[1]);a[_e3]=Object.assign({},r[3]?_defineProperty({},this.toCamelCase(r[3]),n):{enabled:""===n||"true"===n},a[_e3])}}Object.keys(a).forEach((function(e){var _t9=a[e];var s=_t9["enabled"];var i=_t9["class"];if(s&&i){delete _t9["enabled"];delete _t9["clazz"];var _s2=_this5.createPluginInstance(i,_t9);_this5.core.registerPlugin(e,_s2)}}))}},{key:"isEmptyOption",value:function isEmptyOption(e){var _t10=e.validators;return Object.keys(_t10).length===0&&_t10.constructor===Object}},{key:"parseElement",value:function parseElement(e){var _t11=new RegExp("^".concat(this.opts.prefix,"([a-z0-9-]+)(___)*([a-z0-9-]+)*$"));var s=e.attributes.length;var a={};var i=e.getAttribute("type");for(var n=0;n<s;n++){var _s3=e.attributes[n].name;var r=e.attributes[n].value;if(this.opts.html5Input){switch(true){case"minlength"===_s3:a["stringLength"]=Object.assign({},{enabled:true,min:parseInt(r,10)},a["stringLength"]);break;case"maxlength"===_s3:a["stringLength"]=Object.assign({},{enabled:true,max:parseInt(r,10)},a["stringLength"]);break;case"pattern"===_s3:a["regexp"]=Object.assign({},{enabled:true,regexp:r},a["regexp"]);break;case"required"===_s3:a["notEmpty"]=Object.assign({},{enabled:true},a["notEmpty"]);break;case"type"===_s3&&"color"===r:a["color"]=Object.assign({},{enabled:true,type:"hex"},a["color"]);break;case"type"===_s3&&"email"===r:a["emailAddress"]=Object.assign({},{enabled:true},a["emailAddress"]);break;case"type"===_s3&&"url"===r:a["uri"]=Object.assign({},{enabled:true},a["uri"]);break;case"type"===_s3&&"range"===r:a["between"]=Object.assign({},{enabled:true,max:parseFloat(e.getAttribute("max")),min:parseFloat(e.getAttribute("min"))},a["between"]);break;case"min"===_s3&&i!=="date"&&i!=="range":a["greaterThan"]=Object.assign({},{enabled:true,min:parseFloat(r)},a["greaterThan"]);break;case"max"===_s3&&i!=="date"&&i!=="range":a["lessThan"]=Object.assign({},{enabled:true,max:parseFloat(r)},a["lessThan"]);break}}var l=_t11.exec(_s3);if(l&&l.length===4){var _e4=this.toCamelCase(l[1]);a[_e4]=Object.assign({},l[3]?_defineProperty({},this.toCamelCase(l[3]),this.normalizeValue(r)):{enabled:""===r||"true"===r},a[_e4])}}return{validators:a}}},{key:"normalizeValue",value:function normalizeValue(e){return e==="true"?true:e==="false"?false:e}},{key:"toUpperCase",value:function toUpperCase(e){return e.charAt(1).toUpperCase()}},{key:"toCamelCase",value:function toCamelCase(e){return e.replace(/-./g,this.toUpperCase)}}]);return t}(t$4);var o=function(_t){_inherits(o,_t);var _super=_createSuper(o);function o(){var _this;_classCallCheck(this,o);_this=_super.call(this,{});_this.onValidHandler=_this.onFormValid.bind(_assertThisInitialized(_this));return _this}_createClass(o,[{key:"install",value:function install(){var t=this.core.getFormElement();if(t.querySelectorAll('[type="submit"][name="submit"]').length){throw new Error("Do not use `submit` for the name attribute of submit button")}this.core.on("core.form.valid",this.onValidHandler)}},{key:"uninstall",value:function uninstall(){this.core.off("core.form.valid",this.onValidHandler)}},{key:"onFormValid",value:function onFormValid(){var t=this.core.getFormElement();if(t instanceof HTMLFormElement){t.submit()}}}]);return o}(t$4);var e$3=function(_t){_inherits(e,_t);var _super=_createSuper(e);function e(t){var _this;_classCallCheck(this,e);_this=_super.call(this,t);_this.opts=t||{};_this.triggerExecutedHandler=_this.onTriggerExecuted.bind(_assertThisInitialized(_this));return _this}_createClass(e,[{key:"install",value:function install(){this.core.on("plugins.trigger.executed",this.triggerExecutedHandler)}},{key:"uninstall",value:function uninstall(){this.core.off("plugins.trigger.executed",this.triggerExecutedHandler)}},{key:"onTriggerExecuted",value:function onTriggerExecuted(t){if(this.opts[t.field]){var _e3=this.opts[t.field].split(" ");var _iterator=_createForOfIteratorHelper(_e3),_step;try{for(_iterator.s();!(_step=_iterator.n()).done;){var _t2=_step.value;var _e4=_t2.trim();if(this.opts[_e4]){this.core.revalidateField(_e4)}}}catch(err){_iterator.e(err)}finally{_iterator.f()}}}}]);return e}(t$4);var e$2=function(_t){_inherits(e,_t);var _super=_createSuper(e);function e(t){var _this;_classCallCheck(this,e);_this=_super.call(this,t);_this.opts=Object.assign({},{excluded:e.defaultIgnore},t);_this.ignoreValidationFilter=_this.ignoreValidation.bind(_assertThisInitialized(_this));return _this}_createClass(e,[{key:"install",value:function install(){this.core.registerFilter("element-ignored",this.ignoreValidationFilter)}},{key:"uninstall",value:function uninstall(){this.core.deregisterFilter("element-ignored",this.ignoreValidationFilter)}},{key:"ignoreValidation",value:function ignoreValidation(t,_e2,i){return this.opts.excluded.apply(this,[t,_e2,i])}}],[{key:"defaultIgnore",value:function defaultIgnore(t,_e,i){var r=!!(_e.offsetWidth||_e.offsetHeight||_e.getClientRects().length);var n=_e.getAttribute("disabled");return n===""||n==="disabled"||_e.getAttribute("type")==="hidden"||!r}}]);return e}(t$4);var t$2=function(_e){_inherits(t,_e);var _super=_createSuper(t);function t(e){var _this;_classCallCheck(this,t);_this=_super.call(this,e);_this.statuses=new Map;_this.opts=Object.assign({},{onStatusChanged:function onStatusChanged(){}},e);_this.elementValidatingHandler=_this.onElementValidating.bind(_assertThisInitialized(_this));_this.elementValidatedHandler=_this.onElementValidated.bind(_assertThisInitialized(_this));_this.elementNotValidatedHandler=_this.onElementNotValidated.bind(_assertThisInitialized(_this));_this.elementIgnoredHandler=_this.onElementIgnored.bind(_assertThisInitialized(_this));_this.fieldAddedHandler=_this.onFieldAdded.bind(_assertThisInitialized(_this));_this.fieldRemovedHandler=_this.onFieldRemoved.bind(_assertThisInitialized(_this));return _this}_createClass(t,[{key:"install",value:function install(){this.core.on("core.element.validating",this.elementValidatingHandler).on("core.element.validated",this.elementValidatedHandler).on("core.element.notvalidated",this.elementNotValidatedHandler).on("core.element.ignored",this.elementIgnoredHandler).on("core.field.added",this.fieldAddedHandler).on("core.field.removed",this.fieldRemovedHandler)}},{key:"uninstall",value:function uninstall(){this.statuses.clear();this.core.off("core.element.validating",this.elementValidatingHandler).off("core.element.validated",this.elementValidatedHandler).off("core.element.notvalidated",this.elementNotValidatedHandler).off("core.element.ignored",this.elementIgnoredHandler).off("core.field.added",this.fieldAddedHandler).off("core.field.removed",this.fieldRemovedHandler)}},{key:"areFieldsValid",value:function areFieldsValid(){return Array.from(this.statuses.values()).every((function(e){return e==="Valid"||e==="NotValidated"||e==="Ignored"}))}},{key:"getStatuses",value:function getStatuses(){return this.statuses}},{key:"onFieldAdded",value:function onFieldAdded(e){this.statuses.set(e.field,"NotValidated")}},{key:"onFieldRemoved",value:function onFieldRemoved(e){if(this.statuses.has(e.field)){this.statuses["delete"](e.field)}this.opts.onStatusChanged(this.areFieldsValid())}},{key:"onElementValidating",value:function onElementValidating(e){this.statuses.set(e.field,"Validating");this.opts.onStatusChanged(false)}},{key:"onElementValidated",value:function onElementValidated(e){this.statuses.set(e.field,e.valid?"Valid":"Invalid");if(e.valid){this.opts.onStatusChanged(this.areFieldsValid())}else{this.opts.onStatusChanged(false)}}},{key:"onElementNotValidated",value:function onElementNotValidated(e){this.statuses.set(e.field,"NotValidated");this.opts.onStatusChanged(false)}},{key:"onElementIgnored",value:function onElementIgnored(e){this.statuses.set(e.field,"Ignored");this.opts.onStatusChanged(this.areFieldsValid())}}]);return t}(t$4);function s$2(s,a){a.split(" ").forEach((function(a){if(s.classList){s.classList.add(a)}else if(" ".concat(s.className," ").indexOf(" ".concat(a," "))){s.className+=" ".concat(a)}}))}function a(s,a){a.split(" ").forEach((function(a){s.classList?s.classList.remove(a):s.className=s.className.replace(a,"")}))}function c(c,e){var t=[];var f=[];Object.keys(e).forEach((function(s){if(s){e[s]?t.push(s):f.push(s)}}));f.forEach((function(s){return a(c,s)}));t.forEach((function(a){return s$2(c,a)}))}function e$1(e,t){var l=e.matches||e.webkitMatchesSelector||e["mozMatchesSelector"]||e["msMatchesSelector"];if(l){return l.call(e,t)}var c=[].slice.call(e.parentElement.querySelectorAll(t));return c.indexOf(e)>=0}function t$1(t,l){var c=t;while(c){if(e$1(c,l)){break}c=c.parentElement}return c}var s$1=function(_e){_inherits(s,_e);var _super=_createSuper(s);function s(e){var _this;_classCallCheck(this,s);_this=_super.call(this,e);_this.useDefaultContainer=false;_this.messages=new Map;_this.defaultContainer=document.createElement("div");_this.useDefaultContainer=!e||!e.container;_this.opts=Object.assign({},{container:function container(e,t){return _this.defaultContainer}},e);_this.elementIgnoredHandler=_this.onElementIgnored.bind(_assertThisInitialized(_this));_this.fieldAddedHandler=_this.onFieldAdded.bind(_assertThisInitialized(_this));_this.fieldRemovedHandler=_this.onFieldRemoved.bind(_assertThisInitialized(_this));_this.validatorValidatedHandler=_this.onValidatorValidated.bind(_assertThisInitialized(_this));_this.validatorNotValidatedHandler=_this.onValidatorNotValidated.bind(_assertThisInitialized(_this));return _this}_createClass(s,[{key:"install",value:function install(){if(this.useDefaultContainer){this.core.getFormElement().appendChild(this.defaultContainer)}this.core.on("core.element.ignored",this.elementIgnoredHandler).on("core.field.added",this.fieldAddedHandler).on("core.field.removed",this.fieldRemovedHandler).on("core.validator.validated",this.validatorValidatedHandler).on("core.validator.notvalidated",this.validatorNotValidatedHandler)}},{key:"uninstall",value:function uninstall(){if(this.useDefaultContainer){this.core.getFormElement().removeChild(this.defaultContainer)}this.messages.forEach((function(e){return e.parentNode.removeChild(e)}));this.messages.clear();this.core.off("core.element.ignored",this.elementIgnoredHandler).off("core.field.added",this.fieldAddedHandler).off("core.field.removed",this.fieldRemovedHandler).off("core.validator.validated",this.validatorValidatedHandler).off("core.validator.notvalidated",this.validatorNotValidatedHandler)}},{key:"onFieldAdded",value:function onFieldAdded(e){var _this2=this;var t=e.elements;if(t){t.forEach((function(e){var t=_this2.messages.get(e);if(t){t.parentNode.removeChild(t);_this2.messages["delete"](e)}}));this.prepareFieldContainer(e.field,t)}}},{key:"onFieldRemoved",value:function onFieldRemoved(e){var _this3=this;if(!e.elements.length||!e.field){return}var t=e.elements[0].getAttribute("type");var _s2="radio"===t||"checkbox"===t?[e.elements[0]]:e.elements;_s2.forEach((function(e){if(_this3.messages.has(e)){var _t=_this3.messages.get(e);_t.parentNode.removeChild(_t);_this3.messages["delete"](e)}}))}},{key:"prepareFieldContainer",value:function prepareFieldContainer(e,t){var _this4=this;if(t.length){var _s12=t[0].getAttribute("type");if("radio"===_s12||"checkbox"===_s12){this.prepareElementContainer(e,t[0],t)}else{t.forEach((function(_s4){return _this4.prepareElementContainer(e,_s4,t)}))}}}},{key:"prepareElementContainer",value:function prepareElementContainer(e,_s5,i){var a;if("string"===typeof this.opts.container){var _e2="#"===this.opts.container.charAt(0)?'[id="'.concat(this.opts.container.substring(1),'"]'):this.opts.container;a=this.core.getFormElement().querySelector(_e2)}else{a=this.opts.container(e,_s5)}var l=document.createElement("div");a.appendChild(l);c(l,{"fv-plugins-message-container":true});this.core.emit("plugins.message.placed",{element:_s5,elements:i,field:e,messageElement:l});this.messages.set(_s5,l)}},{key:"getMessage",value:function getMessage(e){return typeof e.message==="string"?e.message:e.message[this.core.getLocale()]}},{key:"onValidatorValidated",value:function onValidatorValidated(e){var _s6=e.elements;var i=e.element.getAttribute("type");var a=("radio"===i||"checkbox"===i)&&_s6.length>0?_s6[0]:e.element;if(this.messages.has(a)){var _s13=this.messages.get(a);var _i=_s13.querySelector('[data-field="'.concat(e.field,'"][data-validator="').concat(e.validator,'"]'));if(!_i&&!e.result.valid){var _i2=document.createElement("div");_i2.innerHTML=this.getMessage(e.result);_i2.setAttribute("data-field",e.field);_i2.setAttribute("data-validator",e.validator);if(this.opts.clazz){c(_i2,_defineProperty({},this.opts.clazz,true))}_s13.appendChild(_i2);this.core.emit("plugins.message.displayed",{element:e.element,field:e.field,message:e.result.message,messageElement:_i2,meta:e.result.meta,validator:e.validator})}else if(_i&&!e.result.valid){_i.innerHTML=this.getMessage(e.result);this.core.emit("plugins.message.displayed",{element:e.element,field:e.field,message:e.result.message,messageElement:_i,meta:e.result.meta,validator:e.validator})}else if(_i&&e.result.valid){_s13.removeChild(_i)}}}},{key:"onValidatorNotValidated",value:function onValidatorNotValidated(e){var t=e.elements;var _s8=e.element.getAttribute("type");var i="radio"===_s8||"checkbox"===_s8?t[0]:e.element;if(this.messages.has(i)){var _t3=this.messages.get(i);var _s14=_t3.querySelector('[data-field="'.concat(e.field,'"][data-validator="').concat(e.validator,'"]'));if(_s14){_t3.removeChild(_s14)}}}},{key:"onElementIgnored",value:function onElementIgnored(e){var t=e.elements;var _s10=e.element.getAttribute("type");var i="radio"===_s10||"checkbox"===_s10?t[0]:e.element;if(this.messages.has(i)){var _t4=this.messages.get(i);var _s15=[].slice.call(_t4.querySelectorAll('[data-field="'.concat(e.field,'"]')));_s15.forEach((function(e){_t4.removeChild(e)}))}}}],[{key:"getClosestContainer",value:function getClosestContainer(e,t,_s){var i=e;while(i){if(i===t){break}i=i.parentElement;if(_s.test(i.className)){break}}return i}}]);return s}(t$4);var l=function(_e){_inherits(l,_e);var _super=_createSuper(l);function l(e){var _this;_classCallCheck(this,l);_this=_super.call(this,e);_this.results=new Map;_this.containers=new Map;_this.MESSAGE_PLUGIN="___frameworkMessage";_this.opts=Object.assign({},{defaultMessageContainer:true,eleInvalidClass:"",eleValidClass:"",rowClasses:"",rowValidatingClass:""},e);_this.elementIgnoredHandler=_this.onElementIgnored.bind(_assertThisInitialized(_this));_this.elementValidatingHandler=_this.onElementValidating.bind(_assertThisInitialized(_this));_this.elementValidatedHandler=_this.onElementValidated.bind(_assertThisInitialized(_this));_this.elementNotValidatedHandler=_this.onElementNotValidated.bind(_assertThisInitialized(_this));_this.iconPlacedHandler=_this.onIconPlaced.bind(_assertThisInitialized(_this));_this.fieldAddedHandler=_this.onFieldAdded.bind(_assertThisInitialized(_this));_this.fieldRemovedHandler=_this.onFieldRemoved.bind(_assertThisInitialized(_this));_this.messagePlacedHandler=_this.onMessagePlaced.bind(_assertThisInitialized(_this));return _this}_createClass(l,[{key:"install",value:function install(){var _t,_this2=this;c(this.core.getFormElement(),(_t={},_defineProperty(_t,this.opts.formClass,true),_defineProperty(_t,"fv-plugins-framework",true),_t));this.core.on("core.element.ignored",this.elementIgnoredHandler).on("core.element.validating",this.elementValidatingHandler).on("core.element.validated",this.elementValidatedHandler).on("core.element.notvalidated",this.elementNotValidatedHandler).on("plugins.icon.placed",this.iconPlacedHandler).on("core.field.added",this.fieldAddedHandler).on("core.field.removed",this.fieldRemovedHandler);if(this.opts.defaultMessageContainer){this.core.registerPlugin(this.MESSAGE_PLUGIN,new s$1({clazz:this.opts.messageClass,container:function container(e,t){var _l="string"===typeof _this2.opts.rowSelector?_this2.opts.rowSelector:_this2.opts.rowSelector(e,t);var a=t$1(t,_l);return s$1.getClosestContainer(t,a,_this2.opts.rowPattern)}}));this.core.on("plugins.message.placed",this.messagePlacedHandler)}}},{key:"uninstall",value:function uninstall(){var _t2;this.results.clear();this.containers.clear();c(this.core.getFormElement(),(_t2={},_defineProperty(_t2,this.opts.formClass,false),_defineProperty(_t2,"fv-plugins-framework",false),_t2));this.core.off("core.element.ignored",this.elementIgnoredHandler).off("core.element.validating",this.elementValidatingHandler).off("core.element.validated",this.elementValidatedHandler).off("core.element.notvalidated",this.elementNotValidatedHandler).off("plugins.icon.placed",this.iconPlacedHandler).off("core.field.added",this.fieldAddedHandler).off("core.field.removed",this.fieldRemovedHandler);if(this.opts.defaultMessageContainer){this.core.deregisterPlugin(this.MESSAGE_PLUGIN);this.core.off("plugins.message.placed",this.messagePlacedHandler)}}},{key:"onIconPlaced",value:function onIconPlaced(e){}},{key:"onMessagePlaced",value:function onMessagePlaced(e){}},{key:"onFieldAdded",value:function onFieldAdded(e){var _this3=this;var s=e.elements;if(s){s.forEach((function(e){var s=_this3.containers.get(e);if(s){var _t3;c(s,(_t3={},_defineProperty(_t3,_this3.opts.rowInvalidClass,false),_defineProperty(_t3,_this3.opts.rowValidatingClass,false),_defineProperty(_t3,_this3.opts.rowValidClass,false),_defineProperty(_t3,"fv-plugins-icon-container",false),_t3));_this3.containers["delete"](e)}}));this.prepareFieldContainer(e.field,s)}}},{key:"onFieldRemoved",value:function onFieldRemoved(e){var _this4=this;e.elements.forEach((function(e){var s=_this4.containers.get(e);if(s){var _t4;c(s,(_t4={},_defineProperty(_t4,_this4.opts.rowInvalidClass,false),_defineProperty(_t4,_this4.opts.rowValidatingClass,false),_defineProperty(_t4,_this4.opts.rowValidClass,false),_t4))}}))}},{key:"prepareFieldContainer",value:function prepareFieldContainer(e,t){var _this5=this;if(t.length){var _s=t[0].getAttribute("type");if("radio"===_s||"checkbox"===_s){this.prepareElementContainer(e,t[0])}else{t.forEach((function(t){return _this5.prepareElementContainer(e,t)}))}}}},{key:"prepareElementContainer",value:function prepareElementContainer(e,i){var _l2="string"===typeof this.opts.rowSelector?this.opts.rowSelector:this.opts.rowSelector(e,i);var a=t$1(i,_l2);if(a!==i){var _t5;c(a,(_t5={},_defineProperty(_t5,this.opts.rowClasses,true),_defineProperty(_t5,"fv-plugins-icon-container",true),_t5));this.containers.set(i,a)}}},{key:"onElementValidating",value:function onElementValidating(e){var s=e.elements;var i=e.element.getAttribute("type");var _l3="radio"===i||"checkbox"===i?s[0]:e.element;var a=this.containers.get(_l3);if(a){var _t6;c(a,(_t6={},_defineProperty(_t6,this.opts.rowInvalidClass,false),_defineProperty(_t6,this.opts.rowValidatingClass,true),_defineProperty(_t6,this.opts.rowValidClass,false),_t6))}}},{key:"onElementNotValidated",value:function onElementNotValidated(e){this.removeClasses(e.element,e.elements)}},{key:"onElementIgnored",value:function onElementIgnored(e){this.removeClasses(e.element,e.elements)}},{key:"removeClasses",value:function removeClasses(e,s){var _this6=this;var i=e.getAttribute("type");var _l4="radio"===i||"checkbox"===i?s[0]:e;s.forEach((function(e){var _t7;c(e,(_t7={},_defineProperty(_t7,_this6.opts.eleValidClass,false),_defineProperty(_t7,_this6.opts.eleInvalidClass,false),_t7))}));var a=this.containers.get(_l4);if(a){var _t8;c(a,(_t8={},_defineProperty(_t8,this.opts.rowInvalidClass,false),_defineProperty(_t8,this.opts.rowValidatingClass,false),_defineProperty(_t8,this.opts.rowValidClass,false),_t8))}}},{key:"onElementValidated",value:function onElementValidated(e){var _this7=this;var s=e.elements;var i=e.element.getAttribute("type");var _l5="radio"===i||"checkbox"===i?s[0]:e.element;s.forEach((function(s){var _t9;c(s,(_t9={},_defineProperty(_t9,_this7.opts.eleValidClass,e.valid),_defineProperty(_t9,_this7.opts.eleInvalidClass,!e.valid),_t9))}));var a=this.containers.get(_l5);if(a){if(!e.valid){var _t10;this.results.set(_l5,false);c(a,(_t10={},_defineProperty(_t10,this.opts.rowInvalidClass,true),_defineProperty(_t10,this.opts.rowValidatingClass,false),_defineProperty(_t10,this.opts.rowValidClass,false),_t10))}else{this.results["delete"](_l5);var _e2=true;this.containers.forEach((function(t,s){if(t===a&&_this7.results.get(s)===false){_e2=false}}));if(_e2){var _t11;c(a,(_t11={},_defineProperty(_t11,this.opts.rowInvalidClass,false),_defineProperty(_t11,this.opts.rowValidatingClass,false),_defineProperty(_t11,this.opts.rowValidClass,true),_t11))}}}}}]);return l}(t$4);var i$2=function(_e){_inherits(i,_e);var _super=_createSuper(i);function i(e){var _this;_classCallCheck(this,i);_this=_super.call(this,e);_this.icons=new Map;_this.opts=Object.assign({},{invalid:"fv-plugins-icon--invalid",onPlaced:function onPlaced(){},onSet:function onSet(){},valid:"fv-plugins-icon--valid",validating:"fv-plugins-icon--validating"},e);_this.elementValidatingHandler=_this.onElementValidating.bind(_assertThisInitialized(_this));_this.elementValidatedHandler=_this.onElementValidated.bind(_assertThisInitialized(_this));_this.elementNotValidatedHandler=_this.onElementNotValidated.bind(_assertThisInitialized(_this));_this.elementIgnoredHandler=_this.onElementIgnored.bind(_assertThisInitialized(_this));_this.fieldAddedHandler=_this.onFieldAdded.bind(_assertThisInitialized(_this));return _this}_createClass(i,[{key:"install",value:function install(){this.core.on("core.element.validating",this.elementValidatingHandler).on("core.element.validated",this.elementValidatedHandler).on("core.element.notvalidated",this.elementNotValidatedHandler).on("core.element.ignored",this.elementIgnoredHandler).on("core.field.added",this.fieldAddedHandler)}},{key:"uninstall",value:function uninstall(){this.icons.forEach((function(e){return e.parentNode.removeChild(e)}));this.icons.clear();this.core.off("core.element.validating",this.elementValidatingHandler).off("core.element.validated",this.elementValidatedHandler).off("core.element.notvalidated",this.elementNotValidatedHandler).off("core.element.ignored",this.elementIgnoredHandler).off("core.field.added",this.fieldAddedHandler)}},{key:"onFieldAdded",value:function onFieldAdded(e){var _this2=this;var t=e.elements;if(t){t.forEach((function(e){var t=_this2.icons.get(e);if(t){t.parentNode.removeChild(t);_this2.icons["delete"](e)}}));this.prepareFieldIcon(e.field,t)}}},{key:"prepareFieldIcon",value:function prepareFieldIcon(e,t){var _this3=this;if(t.length){var _i8=t[0].getAttribute("type");if("radio"===_i8||"checkbox"===_i8){this.prepareElementIcon(e,t[0])}else{t.forEach((function(t){return _this3.prepareElementIcon(e,t)}))}}}},{key:"prepareElementIcon",value:function prepareElementIcon(e,_i2){var n=document.createElement("i");n.setAttribute("data-field",e);_i2.parentNode.insertBefore(n,_i2.nextSibling);c(n,{"fv-plugins-icon":true});var l={classes:{invalid:this.opts.invalid,valid:this.opts.valid,validating:this.opts.validating},element:_i2,field:e,iconElement:n};this.core.emit("plugins.icon.placed",l);this.opts.onPlaced(l);this.icons.set(_i2,n)}},{key:"onElementValidating",value:function onElementValidating(e){var _this$setClasses;var t=this.setClasses(e.field,e.element,e.elements,(_this$setClasses={},_defineProperty(_this$setClasses,this.opts.invalid,false),_defineProperty(_this$setClasses,this.opts.valid,false),_defineProperty(_this$setClasses,this.opts.validating,true),_this$setClasses));var _i3={element:e.element,field:e.field,iconElement:t,status:"Validating"};this.core.emit("plugins.icon.set",_i3);this.opts.onSet(_i3)}},{key:"onElementValidated",value:function onElementValidated(e){var _this$setClasses2;var t=this.setClasses(e.field,e.element,e.elements,(_this$setClasses2={},_defineProperty(_this$setClasses2,this.opts.invalid,!e.valid),_defineProperty(_this$setClasses2,this.opts.valid,e.valid),_defineProperty(_this$setClasses2,this.opts.validating,false),_this$setClasses2));var _i4={element:e.element,field:e.field,iconElement:t,status:e.valid?"Valid":"Invalid"};this.core.emit("plugins.icon.set",_i4);this.opts.onSet(_i4)}},{key:"onElementNotValidated",value:function onElementNotValidated(e){var _this$setClasses3;var t=this.setClasses(e.field,e.element,e.elements,(_this$setClasses3={},_defineProperty(_this$setClasses3,this.opts.invalid,false),_defineProperty(_this$setClasses3,this.opts.valid,false),_defineProperty(_this$setClasses3,this.opts.validating,false),_this$setClasses3));var _i5={element:e.element,field:e.field,iconElement:t,status:"NotValidated"};this.core.emit("plugins.icon.set",_i5);this.opts.onSet(_i5)}},{key:"onElementIgnored",value:function onElementIgnored(e){var _this$setClasses4;var t=this.setClasses(e.field,e.element,e.elements,(_this$setClasses4={},_defineProperty(_this$setClasses4,this.opts.invalid,false),_defineProperty(_this$setClasses4,this.opts.valid,false),_defineProperty(_this$setClasses4,this.opts.validating,false),_this$setClasses4));var _i6={element:e.element,field:e.field,iconElement:t,status:"Ignored"};this.core.emit("plugins.icon.set",_i6);this.opts.onSet(_i6)}},{key:"setClasses",value:function setClasses(e,_i7,n,l){var s=_i7.getAttribute("type");var d="radio"===s||"checkbox"===s?n[0]:_i7;if(this.icons.has(d)){var _e2=this.icons.get(d);c(_e2,l);return _e2}else{return null}}}]);return i}(t$4);var i$1=function(_e){_inherits(i,_e);var _super=_createSuper(i);function i(e){var _this;_classCallCheck(this,i);_this=_super.call(this,e);_this.invalidFields=new Map;_this.opts=Object.assign({},{enabled:true},e);_this.validatorHandler=_this.onValidatorValidated.bind(_assertThisInitialized(_this));_this.shouldValidateFilter=_this.shouldValidate.bind(_assertThisInitialized(_this));_this.fieldAddedHandler=_this.onFieldAdded.bind(_assertThisInitialized(_this));_this.elementNotValidatedHandler=_this.onElementNotValidated.bind(_assertThisInitialized(_this));_this.elementValidatingHandler=_this.onElementValidating.bind(_assertThisInitialized(_this));return _this}_createClass(i,[{key:"install",value:function install(){this.core.on("core.validator.validated",this.validatorHandler).on("core.field.added",this.fieldAddedHandler).on("core.element.notvalidated",this.elementNotValidatedHandler).on("core.element.validating",this.elementValidatingHandler).registerFilter("field-should-validate",this.shouldValidateFilter)}},{key:"uninstall",value:function uninstall(){this.invalidFields.clear();this.core.off("core.validator.validated",this.validatorHandler).off("core.field.added",this.fieldAddedHandler).off("core.element.notvalidated",this.elementNotValidatedHandler).off("core.element.validating",this.elementValidatingHandler).deregisterFilter("field-should-validate",this.shouldValidateFilter)}},{key:"shouldValidate",value:function shouldValidate(e,_i,t,l){var d=(this.opts.enabled===true||this.opts.enabled[e]===true)&&this.invalidFields.has(_i)&&!!this.invalidFields.get(_i).length&&this.invalidFields.get(_i).indexOf(l)===-1;return!d}},{key:"onValidatorValidated",value:function onValidatorValidated(e){var _i2=this.invalidFields.has(e.element)?this.invalidFields.get(e.element):[];var t=_i2.indexOf(e.validator);if(e.result.valid&&t>=0){_i2.splice(t,1)}else if(!e.result.valid&&t===-1){_i2.push(e.validator)}this.invalidFields.set(e.element,_i2)}},{key:"onFieldAdded",value:function onFieldAdded(e){if(e.elements){this.clearInvalidFields(e.elements)}}},{key:"onElementNotValidated",value:function onElementNotValidated(e){this.clearInvalidFields(e.elements)}},{key:"onElementValidating",value:function onElementValidating(e){this.clearInvalidFields(e.elements)}},{key:"clearInvalidFields",value:function clearInvalidFields(e){var _this2=this;e.forEach((function(e){return _this2.invalidFields["delete"](e)}))}}]);return i}(t$4);var e=function(_t){_inherits(e,_t);var _super=_createSuper(e);function e(t){var _this;_classCallCheck(this,e);_this=_super.call(this,t);_this.isFormValid=false;_this.opts=Object.assign({},{aspNetButton:false,buttons:function buttons(t){return[].slice.call(t.querySelectorAll('[type="submit"]:not([formnovalidate])'))}},t);_this.submitHandler=_this.handleSubmitEvent.bind(_assertThisInitialized(_this));_this.buttonClickHandler=_this.handleClickEvent.bind(_assertThisInitialized(_this));return _this}_createClass(e,[{key:"install",value:function install(){var _this2=this;if(!(this.core.getFormElement()instanceof HTMLFormElement)){return}var t=this.core.getFormElement();this.submitButtons=this.opts.buttons(t);t.setAttribute("novalidate","novalidate");t.addEventListener("submit",this.submitHandler);this.hiddenClickedEle=document.createElement("input");this.hiddenClickedEle.setAttribute("type","hidden");t.appendChild(this.hiddenClickedEle);this.submitButtons.forEach((function(t){t.addEventListener("click",_this2.buttonClickHandler)}))}},{key:"uninstall",value:function uninstall(){var _this3=this;var t=this.core.getFormElement();if(t instanceof HTMLFormElement){t.removeEventListener("submit",this.submitHandler)}this.submitButtons.forEach((function(t){t.removeEventListener("click",_this3.buttonClickHandler)}));this.hiddenClickedEle.parentElement.removeChild(this.hiddenClickedEle)}},{key:"handleSubmitEvent",value:function handleSubmitEvent(t){this.validateForm(t)}},{key:"handleClickEvent",value:function handleClickEvent(t){var _e=t.currentTarget;if(_e instanceof HTMLElement){if(this.opts.aspNetButton&&this.isFormValid===true);else{var _e3=this.core.getFormElement();_e3.removeEventListener("submit",this.submitHandler);this.clickedButton=t.target;var i=this.clickedButton.getAttribute("name");var s=this.clickedButton.getAttribute("value");if(i&&s){this.hiddenClickedEle.setAttribute("name",i);this.hiddenClickedEle.setAttribute("value",s)}this.validateForm(t)}}}},{key:"validateForm",value:function validateForm(t){var _this4=this;t.preventDefault();this.core.validate().then((function(t){if(t==="Valid"&&_this4.opts.aspNetButton&&!_this4.isFormValid&&_this4.clickedButton){_this4.isFormValid=true;_this4.clickedButton.removeEventListener("click",_this4.buttonClickHandler);_this4.clickedButton.click()}}))}}]);return e}(t$4);var i=function(_t){_inherits(i,_t);var _super=_createSuper(i);function i(t){var _this;_classCallCheck(this,i);_this=_super.call(this,t);_this.messages=new Map;_this.opts=Object.assign({},{placement:"top",trigger:"click"},t);_this.iconPlacedHandler=_this.onIconPlaced.bind(_assertThisInitialized(_this));_this.validatorValidatedHandler=_this.onValidatorValidated.bind(_assertThisInitialized(_this));_this.elementValidatedHandler=_this.onElementValidated.bind(_assertThisInitialized(_this));_this.documentClickHandler=_this.onDocumentClicked.bind(_assertThisInitialized(_this));return _this}_createClass(i,[{key:"install",value:function install(){this.tip=document.createElement("div");c(this.tip,_defineProperty({"fv-plugins-tooltip":true},"fv-plugins-tooltip--".concat(this.opts.placement),true));document.body.appendChild(this.tip);this.core.on("plugins.icon.placed",this.iconPlacedHandler).on("core.validator.validated",this.validatorValidatedHandler).on("core.element.validated",this.elementValidatedHandler);if("click"===this.opts.trigger){document.addEventListener("click",this.documentClickHandler)}}},{key:"uninstall",value:function uninstall(){this.messages.clear();document.body.removeChild(this.tip);this.core.off("plugins.icon.placed",this.iconPlacedHandler).off("core.validator.validated",this.validatorValidatedHandler).off("core.element.validated",this.elementValidatedHandler);if("click"===this.opts.trigger){document.removeEventListener("click",this.documentClickHandler)}}},{key:"onIconPlaced",value:function onIconPlaced(t){var _this2=this;c(t.iconElement,{"fv-plugins-tooltip-icon":true});switch(this.opts.trigger){case"hover":t.iconElement.addEventListener("mouseenter",(function(e){return _this2.show(t.element,e)}));t.iconElement.addEventListener("mouseleave",(function(t){return _this2.hide()}));break;case"click":default:t.iconElement.addEventListener("click",(function(e){return _this2.show(t.element,e)}));break}}},{key:"onValidatorValidated",value:function onValidatorValidated(t){if(!t.result.valid){var _e2=t.elements;var _i4=t.element.getAttribute("type");var s="radio"===_i4||"checkbox"===_i4?_e2[0]:t.element;var o=typeof t.result.message==="string"?t.result.message:t.result.message[this.core.getLocale()];this.messages.set(s,o)}}},{key:"onElementValidated",value:function onElementValidated(t){if(t.valid){var _e3=t.elements;var _i5=t.element.getAttribute("type");var s="radio"===_i5||"checkbox"===_i5?_e3[0]:t.element;this.messages["delete"](s)}}},{key:"onDocumentClicked",value:function onDocumentClicked(t){this.hide()}},{key:"show",value:function show(t,_i3){_i3.preventDefault();_i3.stopPropagation();if(!this.messages.has(t)){return}c(this.tip,{"fv-plugins-tooltip--hide":false});this.tip.innerHTML='<div class="fv-plugins-tooltip__content">'.concat(this.messages.get(t),"</div>");var s=_i3.target;var o=s.getBoundingClientRect();var _this$tip$getBounding=this.tip.getBoundingClientRect(),l=_this$tip$getBounding.height,n=_this$tip$getBounding.width;var a=0;var d=0;switch(this.opts.placement){case"bottom":a=o.top+o.height;d=o.left+o.width/2-n/2;break;case"bottom-left":a=o.top+o.height;d=o.left;break;case"bottom-right":a=o.top+o.height;d=o.left+o.width-n;break;case"left":a=o.top+o.height/2-l/2;d=o.left-n;break;case"right":a=o.top+o.height/2-l/2;d=o.left+o.width;break;case"top-left":a=o.top-l;d=o.left;break;case"top-right":a=o.top-l;d=o.left+o.width-n;break;case"top":default:a=o.top-l;d=o.left+o.width/2-n/2;break}var c$1=window.pageYOffset||document.documentElement.scrollTop||document.body.scrollTop||0;var r=window.pageXOffset||document.documentElement.scrollLeft||document.body.scrollLeft||0;a=a+c$1;d=d+r;this.tip.setAttribute("style","top: ".concat(a,"px; left: ").concat(d,"px"))}},{key:"hide",value:function hide(){c(this.tip,{"fv-plugins-tooltip--hide":true})}}]);return i}(t$4);var t=function(_e){_inherits(t,_e);var _super=_createSuper(t);function t(e){var _this;_classCallCheck(this,t);_this=_super.call(this,e);_this.handlers=[];_this.timers=new Map;var _t=document.createElement("div");_this.defaultEvent=!("oninput"in _t)?"keyup":"input";_this.opts=Object.assign({},{delay:0,event:_this.defaultEvent,threshold:0},e);_this.fieldAddedHandler=_this.onFieldAdded.bind(_assertThisInitialized(_this));_this.fieldRemovedHandler=_this.onFieldRemoved.bind(_assertThisInitialized(_this));return _this}_createClass(t,[{key:"install",value:function install(){this.core.on("core.field.added",this.fieldAddedHandler).on("core.field.removed",this.fieldRemovedHandler)}},{key:"uninstall",value:function uninstall(){this.handlers.forEach((function(e){return e.element.removeEventListener(e.event,e.handler)}));this.handlers=[];this.timers.forEach((function(e){return window.clearTimeout(e)}));this.timers.clear();this.core.off("core.field.added",this.fieldAddedHandler).off("core.field.removed",this.fieldRemovedHandler)}},{key:"prepareHandler",value:function prepareHandler(e,_t2){var _this2=this;_t2.forEach((function(_t3){var i=[];if(!!_this2.opts.event&&_this2.opts.event[e]===false){i=[]}else if(!!_this2.opts.event&&!!_this2.opts.event[e]&&typeof _this2.opts.event[e]!=="function"){i=_this2.opts.event[e].split(" ")}else if("string"===typeof _this2.opts.event&&_this2.opts.event!==_this2.defaultEvent){i=_this2.opts.event.split(" ")}else{var _e2=_t3.getAttribute("type");var s=_t3.tagName.toLowerCase();var n="radio"===_e2||"checkbox"===_e2||"file"===_e2||"select"===s?"change":_this2.ieVersion>=10&&_t3.getAttribute("placeholder")?"keyup":_this2.defaultEvent;i=[n]}i.forEach((function(i){var s=function s(i){return _this2.handleEvent(i,e,_t3)};_this2.handlers.push({element:_t3,event:i,field:e,handler:s});_t3.addEventListener(i,s)}))}))}},{key:"handleEvent",value:function handleEvent(e,_t4,i){var _this3=this;if(this.exceedThreshold(_t4,i)&&this.core.executeFilter("plugins-trigger-should-validate",true,[_t4,i])){var s=function s(){return _this3.core.validateElement(_t4,i).then((function(s){_this3.core.emit("plugins.trigger.executed",{element:i,event:e,field:_t4})}))};var n=this.opts.delay[_t4]||this.opts.delay;if(n===0){s()}else{var _e3=this.timers.get(i);if(_e3){window.clearTimeout(_e3)}this.timers.set(i,window.setTimeout(s,n*1e3))}}}},{key:"onFieldAdded",value:function onFieldAdded(e){this.handlers.filter((function(_t5){return _t5.field===e.field})).forEach((function(e){return e.element.removeEventListener(e.event,e.handler)}));this.prepareHandler(e.field,e.elements)}},{key:"onFieldRemoved",value:function onFieldRemoved(e){this.handlers.filter((function(_t6){return _t6.field===e.field&&e.elements.indexOf(_t6.element)>=0})).forEach((function(e){return e.element.removeEventListener(e.event,e.handler)}))}},{key:"exceedThreshold",value:function exceedThreshold(e,_t7){var i=this.opts.threshold[e]===0||this.opts.threshold===0?false:this.opts.threshold[e]||this.opts.threshold;if(!i){return true}var s=_t7.getAttribute("type");if(["button","checkbox","file","hidden","image","radio","reset","submit"].indexOf(s)!==-1){return true}var n=this.core.getElementValue(e,_t7);return n.length>=i}}]);return t}(t$4);var index$1={Alias:e$4,Aria:i$3,Declarative:t$3,DefaultSubmit:o,Dependency:e$3,Excluded:e$2,FieldStatus:t$2,Framework:l,Icon:i$2,Message:s$1,Sequence:i$1,SubmitButton:e,Tooltip:i,Trigger:t};function s(s,t){return s.classList?s.classList.contains(t):new RegExp("(^| )".concat(t,"( |$)"),"gi").test(s.className)}var index={call:t$$,classSet:c,closest:t$1,fetch:t$V,format:r$d,hasClass:s,isValidDate:t$Y};var p={};exports.Plugin=t$4;exports.algorithms=index$3;exports.filters=index$2;exports.formValidation=r;exports.locales=p;exports.plugins=index$1;exports.utils=index;exports.validators=s$3;Object.defineProperty(exports,"__esModule",{value:true})}));
(function(global,factory){typeof exports==="object"&&typeof module!=="undefined"?module.exports=factory():typeof define==="function"&&define.amd?define(factory):(global=typeof globalThis!=="undefined"?globalThis:global||self,global.FormValidation=global.FormValidation||{},global.FormValidation.plugins=global.FormValidation.plugins||{},global.FormValidation.plugins.Bootstrap5=factory())})(this,(function(){"use strict";function _classCallCheck(instance,Constructor){if(!(instance instanceof Constructor)){throw new TypeError("Cannot call a class as a function")}}function _defineProperties(target,props){for(var i=0;i<props.length;i++){var descriptor=props[i];descriptor.enumerable=descriptor.enumerable||false;descriptor.configurable=true;if("value"in descriptor)descriptor.writable=true;Object.defineProperty(target,descriptor.key,descriptor)}}function _createClass(Constructor,protoProps,staticProps){if(protoProps)_defineProperties(Constructor.prototype,protoProps);if(staticProps)_defineProperties(Constructor,staticProps);Object.defineProperty(Constructor,"prototype",{writable:false});return Constructor}function _inherits(subClass,superClass){if(typeof superClass!=="function"&&superClass!==null){throw new TypeError("Super expression must either be null or a function")}subClass.prototype=Object.create(superClass&&superClass.prototype,{constructor:{value:subClass,writable:true,configurable:true}});Object.defineProperty(subClass,"prototype",{writable:false});if(superClass)_setPrototypeOf(subClass,superClass)}function _getPrototypeOf(o){_getPrototypeOf=Object.setPrototypeOf?Object.getPrototypeOf.bind():function _getPrototypeOf(o){return o.__proto__||Object.getPrototypeOf(o)};return _getPrototypeOf(o)}function _setPrototypeOf(o,p){_setPrototypeOf=Object.setPrototypeOf?Object.setPrototypeOf.bind():function _setPrototypeOf(o,p){o.__proto__=p;return o};return _setPrototypeOf(o,p)}function _isNativeReflectConstruct(){if(typeof Reflect==="undefined"||!Reflect.construct)return false;if(Reflect.construct.sham)return false;if(typeof Proxy==="function")return true;try{Boolean.prototype.valueOf.call(Reflect.construct(Boolean,[],(function(){})));return true}catch(e){return false}}function _assertThisInitialized(self){if(self===void 0){throw new ReferenceError("this hasn't been initialised - super() hasn't been called")}return self}function _possibleConstructorReturn(self,call){if(call&&(typeof call==="object"||typeof call==="function")){return call}else if(call!==void 0){throw new TypeError("Derived constructors may only return object or undefined")}return _assertThisInitialized(self)}function _createSuper(Derived){var hasNativeReflectConstruct=_isNativeReflectConstruct();return function _createSuperInternal(){var Super=_getPrototypeOf(Derived),result;if(hasNativeReflectConstruct){var NewTarget=_getPrototypeOf(this).constructor;result=Reflect.construct(Super,arguments,NewTarget)}else{result=Super.apply(this,arguments)}return _possibleConstructorReturn(this,result)}}function _superPropBase(object,property){while(!Object.prototype.hasOwnProperty.call(object,property)){object=_getPrototypeOf(object);if(object===null)break}return object}function _get(){if(typeof Reflect!=="undefined"&&Reflect.get){_get=Reflect.get.bind()}else{_get=function _get(target,property,receiver){var base=_superPropBase(target,property);if(!base)return;var desc=Object.getOwnPropertyDescriptor(base,property);if(desc.get){return desc.get.call(arguments.length<3?target:receiver)}return desc.value}}return _get.apply(this,arguments)}var e=FormValidation.utils.classSet;var t=FormValidation.utils.hasClass;var n=FormValidation.plugins.Framework;var l=function(_n){_inherits(l,_n);var _super=_createSuper(l);function l(e){var _this;_classCallCheck(this,l);_this=_super.call(this,Object.assign({},{eleInvalidClass:"is-invalid",eleValidClass:"is-valid",formClass:"fv-plugins-bootstrap5",rowInvalidClass:"fv-plugins-bootstrap5-row-invalid",rowPattern:/^(.*)(col|offset)(-(sm|md|lg|xl))*-[0-9]+(.*)$/,rowSelector:".row",rowValidClass:"fv-plugins-bootstrap5-row-valid"},e));_this.eleValidatedHandler=_this.handleElementValidated.bind(_assertThisInitialized(_this));return _this}_createClass(l,[{key:"install",value:function install(){_get(_getPrototypeOf(l.prototype),"install",this).call(this);this.core.on("core.element.validated",this.eleValidatedHandler)}},{key:"uninstall",value:function uninstall(){_get(_getPrototypeOf(l.prototype),"install",this).call(this);this.core.off("core.element.validated",this.eleValidatedHandler)}},{key:"handleElementValidated",value:function handleElementValidated(n){var _l=n.element.getAttribute("type");if(("checkbox"===_l||"radio"===_l)&&n.elements.length>1&&t(n.element,"form-check-input")){var _l5=n.element.parentElement;if(t(_l5,"form-check")&&t(_l5,"form-check-inline")){e(_l5,{"is-invalid":!n.valid,"is-valid":n.valid})}}}},{key:"onIconPlaced",value:function onIconPlaced(n){e(n.element,{"fv-plugins-icon-input":true});var _l3=n.element.parentElement;if(t(_l3,"input-group")){_l3.parentElement.insertBefore(n.iconElement,_l3.nextSibling);if(n.element.nextElementSibling&&t(n.element.nextElementSibling,"input-group-text")){e(n.iconElement,{"fv-plugins-icon-input-group":true})}}var i=n.element.getAttribute("type");if("checkbox"===i||"radio"===i){var _i=_l3.parentElement;if(t(_l3,"form-check")){e(n.iconElement,{"fv-plugins-icon-check":true});_l3.parentElement.insertBefore(n.iconElement,_l3.nextSibling)}else if(t(_l3.parentElement,"form-check")){e(n.iconElement,{"fv-plugins-icon-check":true});_i.parentElement.insertBefore(n.iconElement,_i.nextSibling)}}}},{key:"onMessagePlaced",value:function onMessagePlaced(n){n.messageElement.classList.add("invalid-feedback");var _l4=n.element.parentElement;if(t(_l4,"input-group")){_l4.appendChild(n.messageElement);e(_l4,{"has-validation":true});return}var i=n.element.getAttribute("type");if(("checkbox"===i||"radio"===i)&&t(n.element,"form-check-input")&&!t(_l4,"form-check")){n.elements[n.elements.length-1].parentElement.appendChild(n.messageElement)}}}]);return l}(n);return l}));
/* ==========================================================
 * bootstrap-maxlength.js v1.10.0
 *
 * Copyright (c) 2013-2021 Maurizio Napoleoni;
 *
 * Licensed under the terms of the MIT license.
 * See: https://github.com/mimo84/bootstrap-maxlength/blob/master/LICENSE
 * ========================================================== */
/*global  jQuery*/

(function ($) {
    'use strict';
    /**
     * We need an event when the elements are destroyed
     * because if an input is removed, we have to remove the
     * maxlength object associated (if any).
     * From:
     * http://stackoverflow.com/questions/2200494/jquery-trigger-event-when-an-element-is-removed-from-the-dom
     */
    if (!$.event.special.destroyed) {
        $.event.special.destroyed = {
            remove: function (o) {
                if (o.handler) {
                    o.handler();
                }
            }
        };
    }


    $.fn.extend({
        maxlength: function (options, callback) {
            var documentBody = $('body'),
                defaults = {
                    showOnReady: false, // true to always show when indicator is ready
                    alwaysShow: true, // if true the indicator it's always shown.
                    threshold: 0, // Represents how many chars left are needed to show up the counter
                    warningClass: 'small form-text text-muted',
                    limitReachedClass: 'small form-text text-danger',
                    limitExceededClass: '',
                    separator: ' / ',
                    preText: '',
                    postText: '',
                    showMaxLength: true,
                    placement: 'bottom-right-inside',
                    message: null, // an alternative way to provide the message text
                    showCharsTyped: true, // show the number of characters typed and not the number of characters remaining
                    validate: false, // if the browser doesn't support the maxlength attribute, attempt to type more than the indicated chars, will be prevented.
                    utf8: false, // counts using bytesize rather than length. eg: '£' is counted as 2 characters.
                    appendToParent: false, // append the indicator to the input field's parent instead of body
                    twoCharLinebreak: true, // count linebreak as 2 characters to match IE/Chrome textarea validation. As well as DB storage.
                    customMaxAttribute: null, // null = use maxlength attribute and browser functionality, string = use specified attribute instead.
                    customMaxClass: 'overmax', // Class to add to the input field when the maxlength is exceeded.
                    allowOverMax: false, // Form submit validation is handled on your own.  when maxlength has been exceeded 'overmax' class added to element
                    zIndex: 1099
                };

            if ($.isFunction(options) && !callback) {
                callback = options;
                options = {};
            }
            options = $.extend(defaults, options);


            /**
             * Return the byte count of the specified character in UTF8 encoding.
             * Note: This won't cover UTF-8 characters that are 4 bytes long.
             *
             * @param input
             * @return {number}
             */
            function utf8CharByteCount(character) {
                var c = character.charCodeAt();
                // Not c then 0, else c < 128 then 1, else c < 2048 then 2, else 3
                return !c ? 0 : c < 128 ? 1 : c < 2048 ? 2 : 3;
            }

            /**
             * Return the length of the specified input in UTF8 encoding.
             *
             * @param input
             * @return {number}
             */
            function utf8Length(string) {
                return string.split("")
                    .map(utf8CharByteCount)
                    // Prevent reduce from throwing an error if the string is empty.
                    .concat(0)
                    .reduce(function (sum, val) {
                        return sum + val;
                    });
            }

            /**
             * Return the length of the specified input.
             *
             * @param input
             * @return {number}
             */
            function inputLength(input) {
                var text = input.val();

                if (options.twoCharLinebreak) {
                    // Count all line breaks as 2 characters
                    text = text.replace(/\r(?!\n)|\n(?!\r)/g, '\r\n');
                } else {
                    // Remove all double-character (\r\n) linebreaks, so they're counted only once.
                    text = text.replace(/(?:\r\n|\r|\n)/g, '\n');
                }

                var currentLength = 0;

                if (options.utf8) {
                    currentLength = utf8Length(text);
                } else {
                    currentLength = text.length;
                }

                // Remove "C:\fakepath\" from counter when using file input
                // Fix https://github.com/mimo84/bootstrap-maxlength/issues/146
                if (input.prop("type") === "file" && input.val() !== "") {
                    currentLength -= 12;
                }

                return currentLength;
            }

            /**
             * Truncate the text of the specified input.
             *
             * @param input
             * @param limit
             */
            function truncateChars(input, maxlength) {
                var text = input.val();

                if (options.twoCharLinebreak) {
                    text = text.replace(/\r(?!\n)|\n(?!\r)/g, '\r\n');

                    if (text[text.length - 1] === '\n') {
                        maxlength -= text.length % 2;
                    }
                }

                if (options.utf8) {
                    var indexedSize = text.split("").map(utf8CharByteCount);
                    for (
                        var removedBytes = 0,
                            bytesPastMax = utf8Length(text) - maxlength; removedBytes < bytesPastMax; removedBytes += indexedSize.pop()
                    );
                    maxlength -= (maxlength - indexedSize.length);
                }

                input.val(text.substr(0, maxlength));
            }

            /**
             * Return true if the indicator should be showing up.
             *
             * @param input
             * @param threshold
             * @param maxlength
             * @return {number}
             */
            function charsLeftThreshold(input, threshold, maxlength) {
                var output = true;
                if (!options.alwaysShow && (maxlength - inputLength(input) > threshold)) {
                    output = false;
                }
                return output;
            }

            /**
             * Returns how many chars are left to complete the fill up of the form.
             *
             * @param input
             * @param maxlength
             * @return {number}
             */
            function remainingChars(input, maxlength) {
                var length = maxlength - inputLength(input);
                return length;
            }

            /**
             * When called displays the indicator.
             *
             * @param indicator
             */
            function showRemaining(currentInput, indicator) {
                indicator.css({
                    display: 'block'
                });
                currentInput.trigger('maxlength.shown');
            }

            /**
             * When called shows the indicator.
             *
             * @param indicator
             */
            function hideRemaining(currentInput, indicator) {

                if (options.alwaysShow) {
                    return;
                }

                indicator.css({
                    display: 'none'
                });
                currentInput.trigger('maxlength.hidden');
            }

            /**
             * This function updates the value in the indicator
             *
             * @param maxLengthThisInput
             * @param typedChars
             * @return String
             */
            function updateMaxLengthHTML(currentInputText, maxLengthThisInput, typedChars) {
                var output = '';
                if (options.message) {
                    if (typeof options.message === 'function') {
                        output = options.message(currentInputText, maxLengthThisInput);
                    } else {
                        output = options.message.replace('%charsTyped%', typedChars)
                            .replace('%charsRemaining%', maxLengthThisInput - typedChars)
                            .replace('%charsTotal%', maxLengthThisInput);
                    }
                } else {
                    if (options.preText) {
                        output += options.preText;
                    }
                    if (!options.showCharsTyped) {
                        output += maxLengthThisInput - typedChars;
                    } else {
                        output += typedChars;
                    }
                    if (options.showMaxLength) {
                        output += options.separator + maxLengthThisInput;
                    }
                    if (options.postText) {
                        output += options.postText;
                    }
                }
                return output;
            }

            /**
             * This function updates the value of the counter in the indicator.
             * Wants as parameters: the number of remaining chars, the element currently managed,
             * the maxLength for the current input and the indicator generated for it.
             *
             * @param remaining
             * @param currentInput
             * @param maxLengthCurrentInput
             * @param maxLengthIndicator
             */
            function manageRemainingVisibility(remaining, currentInput, maxLengthCurrentInput, maxLengthIndicator) {
                if (maxLengthIndicator) {
                    maxLengthIndicator.html(updateMaxLengthHTML(currentInput.val(), maxLengthCurrentInput, (maxLengthCurrentInput - remaining)));

                    if (remaining > 0) {
                        if (charsLeftThreshold(currentInput, options.threshold, maxLengthCurrentInput)) {
                            showRemaining(currentInput, maxLengthIndicator.removeClass(options.limitReachedClass + ' ' + options.limitExceededClass).addClass(options.warningClass));
                        } else {
                            hideRemaining(currentInput, maxLengthIndicator);
                        }
                    } else {
                        if (!options.limitExceededClass) {
                            showRemaining(currentInput, maxLengthIndicator.removeClass(options.warningClass).addClass(options.limitReachedClass));
                        } else {
                            if (remaining === 0) {
                                showRemaining(currentInput, maxLengthIndicator.removeClass(options.warningClass + ' ' + options.limitExceededClass).addClass(options.limitReachedClass));
                            } else {
                                showRemaining(currentInput, maxLengthIndicator.removeClass(options.warningClass + ' ' + options.limitReachedClass).addClass(options.limitExceededClass));
                            }
                        }
                    }
                }

                if (options.customMaxAttribute) {
                    // class to use for form validation on custom maxlength attribute
                    if (remaining < 0) {
                        currentInput.addClass(options.customMaxClass);
                    } else {
                        currentInput.removeClass(options.customMaxClass);
                    }
                }
            }

            /**
             * This function returns an object containing all the
             * informations about the position of the current input
             *
             * @param currentInput
             * @return object {bottom height left right top width}
             *
             */
            function getPosition(currentInput) {
                var el = currentInput[0];
                return $.extend({}, (typeof el.getBoundingClientRect === 'function') ? el.getBoundingClientRect() : {
                    width: el.offsetWidth,
                    height: el.offsetHeight
                }, currentInput.offset());
            }

            /**
             * This function places the maxLengthIndicator based on placement config object.
             *
             * @param {object} placement
             * @param {$} maxLengthIndicator
             * @return null
             *
             */
            function placeWithCSS(placement, maxLengthIndicator) {
                if (!placement || !maxLengthIndicator) {
                    return;
                }

                var POSITION_KEYS = [
                    'top',
                    'bottom',
                    'left',
                    'right',
                    'position'
                ];

                var cssPos = {};

                // filter css properties to position
                $.each(POSITION_KEYS, function (i, key) {
                    var val = options.placement[key];
                    if (typeof val !== 'undefined') {
                        cssPos[key] = val;
                    }
                });

                maxLengthIndicator.css(cssPos);

                return;
            }


            /**
             * This function places the maxLengthIndicator at the
             * top / bottom / left / right of the currentInput
             *
             * @param currentInput
             * @param maxLengthIndicator
             * @return null
             *
             */
            function place(currentInput, maxLengthIndicator) {
                var pos = getPosition(currentInput);

                // Supports custom placement handler
                if ($.type(options.placement) === 'function') {
                    options.placement(currentInput, maxLengthIndicator, pos);
                    return;
                }

                // Supports custom placement via css positional properties
                if ($.isPlainObject(options.placement)) {
                    placeWithCSS(options.placement, maxLengthIndicator);
                    return;
                }

                var inputOuter = currentInput.outerWidth(),
                    outerWidth = maxLengthIndicator.outerWidth(),
                    actualWidth = maxLengthIndicator.width(),
                    actualHeight = maxLengthIndicator.height();

                // get the right position if the indicator is appended to the input's parent
                if (options.appendToParent) {
                    pos.top -= currentInput.parent().offset().top;
                    pos.left -= currentInput.parent().offset().left;
                }

                switch (options.placement) {
                    case 'bottom':
                        maxLengthIndicator.css({
                            top: pos.top + pos.height,
                            left: pos.left + pos.width / 2 - actualWidth / 2
                        });
                        break;
                    case 'top':
                        maxLengthIndicator.css({
                            top: pos.top - actualHeight,
                            left: pos.left + pos.width / 2 - actualWidth / 2
                        });
                        break;
                    case 'left':
                        maxLengthIndicator.css({
                            top: pos.top + pos.height / 2 - actualHeight / 2,
                            left: pos.left - actualWidth
                        });
                        break;
                    case 'right':
                        maxLengthIndicator.css({
                            top: pos.top + pos.height / 2 - actualHeight / 2,
                            left: pos.left + pos.width
                        });
                        break;
                    case 'bottom-right':
                        maxLengthIndicator.css({
                            top: pos.top + pos.height,
                            left: pos.left + pos.width
                        });
                        break;
                    case 'top-right':
                        maxLengthIndicator.css({
                            top: pos.top - actualHeight,
                            left: pos.left + inputOuter
                        });
                        break;
                    case 'top-left':
                        maxLengthIndicator.css({
                            top: pos.top - actualHeight,
                            left: pos.left - outerWidth
                        });
                        break;
                    case 'bottom-left':
                        maxLengthIndicator.css({
                            top: pos.top + currentInput.outerHeight(),
                            left: pos.left - outerWidth
                        });
                        break;
                    case 'centered-right':
                        maxLengthIndicator.css({
                            top: pos.top + (actualHeight / 2),
                            left: pos.left + inputOuter - outerWidth - 3
                        });
                        break;

                    // Some more options for placements
                    case 'bottom-right-inside':
                        maxLengthIndicator.css({
                            top: pos.top + pos.height,
                            left: pos.left + pos.width - outerWidth
                        });
                        break;
                    case 'top-right-inside':
                        maxLengthIndicator.css({
                            top: pos.top - actualHeight,
                            left: pos.left + inputOuter - outerWidth
                        });
                        break;
                    case 'top-left-inside':
                        maxLengthIndicator.css({
                            top: pos.top - actualHeight,
                            left: pos.left
                        });
                        break;
                    case 'bottom-left-inside':
                        maxLengthIndicator.css({
                            top: pos.top + currentInput.outerHeight(),
                            left: pos.left
                        });
                        break;
                }
            }

            /**
             * This function returns true if the indicator position needs to
             * be recalculated when the currentInput changes
             *
             * @return {boolean}
             *
             */
            function isPlacementMutable() {
                return options.placement === 'bottom-right-inside' || options.placement === 'top-right-inside' || typeof options.placement === 'function' || (options.message && typeof options.message === 'function');
            }

            /**
             * This function retrieves the maximum length of currentInput
             *
             * @param currentInput
             * @return {number}
             *
             */
            function getMaxLength(currentInput) {
                var max = currentInput.attr('maxlength') || options.customMaxAttribute;

                if (options.customMaxAttribute && !options.allowOverMax) {
                    var custom = currentInput.attr(options.customMaxAttribute);
                    if (!max || custom < max) {
                        max = custom;
                    }
                }

                if (!max) {
                    max = currentInput.attr('size');
                }
                return max;
            }

            return this.each(function () {

                var currentInput = $(this),
                    maxLengthCurrentInput,
                    maxLengthIndicator;

                $(window).resize(function () {
                    if (maxLengthIndicator) {
                        place(currentInput, maxLengthIndicator);
                    }
                });

                function firstInit() {
                    var maxlengthContent = updateMaxLengthHTML(currentInput.val(), maxLengthCurrentInput, '0');
                    maxLengthCurrentInput = getMaxLength(currentInput);

                    if (!maxLengthIndicator) {
                        maxLengthIndicator = $('<span class="bootstrap-maxlength"></span>').css({
                            display: 'none',
                            position: 'absolute',
                            whiteSpace: 'nowrap',
                            zIndex: options.zIndex
                        }).html(maxlengthContent);
                    }

                    // We need to detect resizes if we are dealing with a textarea:
                    if (currentInput.is('textarea')) {
                        currentInput.data('maxlenghtsizex', currentInput.outerWidth());
                        currentInput.data('maxlenghtsizey', currentInput.outerHeight());

                        currentInput.mouseup(function () {
                            if (currentInput.outerWidth() !== currentInput.data('maxlenghtsizex') || currentInput.outerHeight() !== currentInput.data('maxlenghtsizey')) {
                                place(currentInput, maxLengthIndicator);
                            }

                            currentInput.data('maxlenghtsizex', currentInput.outerWidth());
                            currentInput.data('maxlenghtsizey', currentInput.outerHeight());
                        });
                    }

                    if (options.appendToParent) {
                        currentInput.parent().append(maxLengthIndicator);
                        currentInput.parent().css('position', 'relative');
                    } else {
                        documentBody.append(maxLengthIndicator);
                    }

                    var remaining = remainingChars(currentInput, getMaxLength(currentInput));
                    manageRemainingVisibility(remaining, currentInput, maxLengthCurrentInput, maxLengthIndicator);
                    place(currentInput, maxLengthIndicator);
                }

                if (options.showOnReady) {
                    currentInput.ready(function () {
                        firstInit();
                    });
                } else {
                    currentInput.focus(function () {
                        firstInit();
                    });
                }

                currentInput.on('maxlength.reposition', function () {
                    place(currentInput, maxLengthIndicator);
                });


                currentInput.on('destroyed', function () {
                    if (maxLengthIndicator) {
                        maxLengthIndicator.remove();
                    }
                });

                currentInput.on('blur', function () {
                    if (maxLengthIndicator && !options.showOnReady) {
                        maxLengthIndicator.remove();
                    }
                });

                currentInput.on('input', function () {
                    var maxlength = getMaxLength(currentInput),
                        remaining = remainingChars(currentInput, maxlength),
                        output = true;

                    if (options.validate && remaining < 0) {
                        truncateChars(currentInput, maxlength);
                        output = false;
                    } else {
                        manageRemainingVisibility(remaining, currentInput, maxLengthCurrentInput, maxLengthIndicator);
                    }

                    // if (isPlacementMutable()) {
                    //   place(currentInput, maxLengthIndicator);
                    // }

                    return output;
                });
            });
        }
    });
}(jQuery));
/**
 * @version: 3.1
 * @author: Dan Grossman http://www.dangrossman.info/
 * @copyright: Copyright (c) 2012-2019 Dan Grossman. All rights reserved.
 * @license: Licensed under the MIT license. See http://www.opensource.org/licenses/mit-license.php
 * @website: http://www.daterangepicker.com/
 */
// Following the UMD template https://github.com/umdjs/umd/blob/master/templates/returnExportsGlobal.js
(function (root, factory) {
    if (typeof define === 'function' && define.amd) {
        // AMD. Make globaly available as well
        define(['moment', 'jquery'], function (moment, jquery) {
            if (!jquery.fn) jquery.fn = {}; // webpack server rendering
            if (typeof moment !== 'function' && moment.hasOwnProperty('default')) moment = moment['default']
            return factory(moment, jquery);
        });
    } else if (typeof module === 'object' && module.exports) {
        // Node / Browserify
        //isomorphic issue
        var jQuery = (typeof window != 'undefined') ? window.jQuery : undefined;
        if (!jQuery) {
            jQuery = require('jquery');
            if (!jQuery.fn) jQuery.fn = {};
        }
        var moment = (typeof window != 'undefined' && typeof window.moment != 'undefined') ? window.moment : require('moment');
        module.exports = factory(moment, jQuery);
    } else {
        // Browser globals
        root.daterangepicker = factory(root.moment, root.jQuery);
    }
}(this, function(moment, $) {
    var DateRangePicker = function(element, options, cb) {

        //default settings for options
        this.parentEl = 'body';
        this.element = $(element);
        this.startDate = moment().startOf('day');
        this.endDate = moment().endOf('day');
        this.minDate = false;
        this.maxDate = false;
        this.maxSpan = false;
        this.autoApply = false;
        this.singleDatePicker = false;
        this.showDropdowns = false;
        this.minYear = moment().subtract(100, 'year').format('YYYY');
        this.maxYear = moment().add(100, 'year').format('YYYY');
        this.showWeekNumbers = false;
        this.showISOWeekNumbers = false;
        this.showCustomRangeLabel = true;
        this.timePicker = false;
        this.timePicker24Hour = false;
        this.timePickerIncrement = 1;
        this.timePickerSeconds = false;
        this.linkedCalendars = true;
        this.autoUpdateInput = true;
        this.alwaysShowCalendars = false;
        this.ranges = {};

        this.opens = 'right';
        if (this.element.hasClass('pull-right'))
            this.opens = 'left';

        this.drops = 'down';
        if (this.element.hasClass('dropup'))
            this.drops = 'up';

        this.buttonClasses = 'btn btn-sm';
        this.applyButtonClasses = 'btn-primary';
        this.cancelButtonClasses = 'btn-default';

        this.locale = {
            direction: 'ltr',
            format: moment.localeData().longDateFormat('L'),
            separator: ' - ',
            applyLabel: 'Apply',
            cancelLabel: 'Cancel',
            weekLabel: 'W',
            customRangeLabel: 'Custom Range',
            daysOfWeek: moment.weekdaysMin(),
            monthNames: moment.monthsShort(),
            firstDay: moment.localeData().firstDayOfWeek()
        };

        this.callback = function() { };

        //some state information
        this.isShowing = false;
        this.leftCalendar = {};
        this.rightCalendar = {};

        //custom options from user
        if (typeof options !== 'object' || options === null)
            options = {};

        //allow setting options with data attributes
        //data-api options will be overwritten with custom javascript options
        options = $.extend(this.element.data(), options);

        //html template for the picker UI
        if (typeof options.template !== 'string' && !(options.template instanceof $))
            options.template =
                '<div class="daterangepicker">' +
                '<div class="ranges"></div>' +
                '<div class="drp-calendar left">' +
                '<div class="calendar-table"></div>' +
                '<div class="calendar-time"></div>' +
                '</div>' +
                '<div class="drp-calendar right">' +
                '<div class="calendar-table"></div>' +
                '<div class="calendar-time"></div>' +
                '</div>' +
                '<div class="drp-buttons">' +
                '<span class="drp-selected"></span>' +
                '<button class="cancelBtn" type="button"></button>' +
                '<button class="applyBtn" disabled="disabled" type="button"></button> ' +
                '</div>' +
                '</div>';

        this.parentEl = (options.parentEl && $(options.parentEl).length) ? $(options.parentEl) : $(this.parentEl);
        this.container = $(options.template).appendTo(this.parentEl);

        //
        // handle all the possible options overriding defaults
        //

        if (typeof options.locale === 'object') {

            if (typeof options.locale.direction === 'string')
                this.locale.direction = options.locale.direction;

            if (typeof options.locale.format === 'string')
                this.locale.format = options.locale.format;

            if (typeof options.locale.separator === 'string')
                this.locale.separator = options.locale.separator;

            if (typeof options.locale.daysOfWeek === 'object')
                this.locale.daysOfWeek = options.locale.daysOfWeek.slice();

            if (typeof options.locale.monthNames === 'object')
                this.locale.monthNames = options.locale.monthNames.slice();

            if (typeof options.locale.firstDay === 'number')
                this.locale.firstDay = options.locale.firstDay;

            if (typeof options.locale.applyLabel === 'string')
                this.locale.applyLabel = options.locale.applyLabel;

            if (typeof options.locale.cancelLabel === 'string')
                this.locale.cancelLabel = options.locale.cancelLabel;

            if (typeof options.locale.weekLabel === 'string')
                this.locale.weekLabel = options.locale.weekLabel;

            if (typeof options.locale.customRangeLabel === 'string'){
                //Support unicode chars in the custom range name.
                var elem = document.createElement('textarea');
                elem.innerHTML = options.locale.customRangeLabel;
                var rangeHtml = elem.value;
                this.locale.customRangeLabel = rangeHtml;
            }
        }
        this.container.addClass(this.locale.direction);

        if (typeof options.startDate === 'string')
            this.startDate = moment(options.startDate, this.locale.format);

        if (typeof options.endDate === 'string')
            this.endDate = moment(options.endDate, this.locale.format);

        if (typeof options.minDate === 'string')
            this.minDate = moment(options.minDate, this.locale.format);

        if (typeof options.maxDate === 'string')
            this.maxDate = moment(options.maxDate, this.locale.format);

        if (typeof options.startDate === 'object')
            this.startDate = moment(options.startDate);

        if (typeof options.endDate === 'object')
            this.endDate = moment(options.endDate);

        if (typeof options.minDate === 'object')
            this.minDate = moment(options.minDate);

        if (typeof options.maxDate === 'object')
            this.maxDate = moment(options.maxDate);

        // sanity check for bad options
        if (this.minDate && this.startDate.isBefore(this.minDate))
            this.startDate = this.minDate.clone();

        // sanity check for bad options
        if (this.maxDate && this.endDate.isAfter(this.maxDate))
            this.endDate = this.maxDate.clone();

        if (typeof options.applyButtonClasses === 'string')
            this.applyButtonClasses = options.applyButtonClasses;

        if (typeof options.applyClass === 'string') //backwards compat
            this.applyButtonClasses = options.applyClass;

        if (typeof options.cancelButtonClasses === 'string')
            this.cancelButtonClasses = options.cancelButtonClasses;

        if (typeof options.cancelClass === 'string') //backwards compat
            this.cancelButtonClasses = options.cancelClass;

        if (typeof options.maxSpan === 'object')
            this.maxSpan = options.maxSpan;

        if (typeof options.dateLimit === 'object') //backwards compat
            this.maxSpan = options.dateLimit;

        if (typeof options.opens === 'string')
            this.opens = options.opens;

        if (typeof options.drops === 'string')
            this.drops = options.drops;

        if (typeof options.showWeekNumbers === 'boolean')
            this.showWeekNumbers = options.showWeekNumbers;

        if (typeof options.showISOWeekNumbers === 'boolean')
            this.showISOWeekNumbers = options.showISOWeekNumbers;

        if (typeof options.buttonClasses === 'string')
            this.buttonClasses = options.buttonClasses;

        if (typeof options.buttonClasses === 'object')
            this.buttonClasses = options.buttonClasses.join(' ');

        if (typeof options.showDropdowns === 'boolean')
            this.showDropdowns = options.showDropdowns;

        if (typeof options.minYear === 'number')
            this.minYear = options.minYear;

        if (typeof options.maxYear === 'number')
            this.maxYear = options.maxYear;

        if (typeof options.showCustomRangeLabel === 'boolean')
            this.showCustomRangeLabel = options.showCustomRangeLabel;

        if (typeof options.singleDatePicker === 'boolean') {
            this.singleDatePicker = options.singleDatePicker;
            if (this.singleDatePicker)
                this.endDate = this.startDate.clone();
        }

        if (typeof options.timePicker === 'boolean')
            this.timePicker = options.timePicker;

        if (typeof options.timePickerSeconds === 'boolean')
            this.timePickerSeconds = options.timePickerSeconds;

        if (typeof options.timePickerIncrement === 'number')
            this.timePickerIncrement = options.timePickerIncrement;

        if (typeof options.timePicker24Hour === 'boolean')
            this.timePicker24Hour = options.timePicker24Hour;

        if (typeof options.autoApply === 'boolean')
            this.autoApply = options.autoApply;

        if (typeof options.autoUpdateInput === 'boolean')
            this.autoUpdateInput = options.autoUpdateInput;

        if (typeof options.linkedCalendars === 'boolean')
            this.linkedCalendars = options.linkedCalendars;

        if (typeof options.isInvalidDate === 'function')
            this.isInvalidDate = options.isInvalidDate;

        if (typeof options.isCustomDate === 'function')
            this.isCustomDate = options.isCustomDate;

        if (typeof options.alwaysShowCalendars === 'boolean')
            this.alwaysShowCalendars = options.alwaysShowCalendars;

        // update day names order to firstDay
        if (this.locale.firstDay != 0) {
            var iterator = this.locale.firstDay;
            while (iterator > 0) {
                this.locale.daysOfWeek.push(this.locale.daysOfWeek.shift());
                iterator--;
            }
        }

        var start, end, range;

        //if no start/end dates set, check if an input element contains initial values
        if (typeof options.startDate === 'undefined' && typeof options.endDate === 'undefined') {
            if ($(this.element).is(':text')) {
                var val = $(this.element).val(),
                    split = val.split(this.locale.separator);

                start = end = null;

                if (split.length == 2) {
                    start = moment(split[0], this.locale.format);
                    end = moment(split[1], this.locale.format);
                } else if (this.singleDatePicker && val !== "") {
                    start = moment(val, this.locale.format);
                    end = moment(val, this.locale.format);
                }
                if (start !== null && end !== null) {
                    this.setStartDate(start);
                    this.setEndDate(end);
                }
            }
        }

        if (typeof options.ranges === 'object') {
            for (range in options.ranges) {

                if (typeof options.ranges[range][0] === 'string')
                    start = moment(options.ranges[range][0], this.locale.format);
                else
                    start = moment(options.ranges[range][0]);

                if (typeof options.ranges[range][1] === 'string')
                    end = moment(options.ranges[range][1], this.locale.format);
                else
                    end = moment(options.ranges[range][1]);

                // If the start or end date exceed those allowed by the minDate or maxSpan
                // options, shorten the range to the allowable period.
                if (this.minDate && start.isBefore(this.minDate))
                    start = this.minDate.clone();

                var maxDate = this.maxDate;
                if (this.maxSpan && maxDate && start.clone().add(this.maxSpan).isAfter(maxDate))
                    maxDate = start.clone().add(this.maxSpan);
                if (maxDate && end.isAfter(maxDate))
                    end = maxDate.clone();

                // If the end of the range is before the minimum or the start of the range is
                // after the maximum, don't display this range option at all.
                if ((this.minDate && end.isBefore(this.minDate, this.timepicker ? 'minute' : 'day'))
                    || (maxDate && start.isAfter(maxDate, this.timepicker ? 'minute' : 'day')))
                    continue;

                //Support unicode chars in the range names.
                var elem = document.createElement('textarea');
                elem.innerHTML = range;
                var rangeHtml = elem.value;

                this.ranges[rangeHtml] = [start, end];
            }

            var list = '<ul>';
            for (range in this.ranges) {
                list += '<li data-range-key="' + range + '">' + range + '</li>';
            }
            if (this.showCustomRangeLabel) {
                list += '<li data-range-key="' + this.locale.customRangeLabel + '">' + this.locale.customRangeLabel + '</li>';
            }
            list += '</ul>';
            this.container.find('.ranges').prepend(list);
        }

        if (typeof cb === 'function') {
            this.callback = cb;
        }

        if (!this.timePicker) {
            this.startDate = this.startDate.startOf('day');
            this.endDate = this.endDate.endOf('day');
            this.container.find('.calendar-time').hide();
        }

        //can't be used together for now
        if (this.timePicker && this.autoApply)
            this.autoApply = false;

        if (this.autoApply) {
            this.container.addClass('auto-apply');
        }

        if (typeof options.ranges === 'object')
            this.container.addClass('show-ranges');

        if (this.singleDatePicker) {
            this.container.addClass('single');
            this.container.find('.drp-calendar.left').addClass('single');
            this.container.find('.drp-calendar.left').show();
            this.container.find('.drp-calendar.right').hide();
            if (!this.timePicker && this.autoApply) {
                this.container.addClass('auto-apply');
            }
        }

        if ((typeof options.ranges === 'undefined' && !this.singleDatePicker) || this.alwaysShowCalendars) {
            this.container.addClass('show-calendar');
        }

        this.container.addClass('opens' + this.opens);

        //apply CSS classes and labels to buttons
        this.container.find('.applyBtn, .cancelBtn').addClass(this.buttonClasses);
        if (this.applyButtonClasses.length)
            this.container.find('.applyBtn').addClass(this.applyButtonClasses);
        if (this.cancelButtonClasses.length)
            this.container.find('.cancelBtn').addClass(this.cancelButtonClasses);
        this.container.find('.applyBtn').html(this.locale.applyLabel);
        this.container.find('.cancelBtn').html(this.locale.cancelLabel);

        //
        // event listeners
        //

        this.container.find('.drp-calendar')
            .on('click.daterangepicker', '.prev', $.proxy(this.clickPrev, this))
            .on('click.daterangepicker', '.next', $.proxy(this.clickNext, this))
            .on('mousedown.daterangepicker', 'td.available', $.proxy(this.clickDate, this))
            .on('mouseenter.daterangepicker', 'td.available', $.proxy(this.hoverDate, this))
            .on('change.daterangepicker', 'select.yearselect', $.proxy(this.monthOrYearChanged, this))
            .on('change.daterangepicker', 'select.monthselect', $.proxy(this.monthOrYearChanged, this))
            .on('change.daterangepicker', 'select.hourselect,select.minuteselect,select.secondselect,select.ampmselect', $.proxy(this.timeChanged, this));

        this.container.find('.ranges')
            .on('click.daterangepicker', 'li', $.proxy(this.clickRange, this));

        this.container.find('.drp-buttons')
            .on('click.daterangepicker', 'button.applyBtn', $.proxy(this.clickApply, this))
            .on('click.daterangepicker', 'button.cancelBtn', $.proxy(this.clickCancel, this));

        if (this.element.is('input') || this.element.is('button')) {
            this.element.on({
                'click.daterangepicker': $.proxy(this.show, this),
                'focus.daterangepicker': $.proxy(this.show, this),
                'keyup.daterangepicker': $.proxy(this.elementChanged, this),
                'keydown.daterangepicker': $.proxy(this.keydown, this) //IE 11 compatibility
            });
        } else {
            this.element.on('click.daterangepicker', $.proxy(this.toggle, this));
            this.element.on('keydown.daterangepicker', $.proxy(this.toggle, this));
        }

        //
        // if attached to a text input, set the initial value
        //

        this.updateElement();

    };

    DateRangePicker.prototype = {

        constructor: DateRangePicker,

        setStartDate: function(startDate) {
            if (typeof startDate === 'string')
                this.startDate = moment(startDate, this.locale.format);

            if (typeof startDate === 'object')
                this.startDate = moment(startDate);

            if (!this.timePicker)
                this.startDate = this.startDate.startOf('day');

            if (this.timePicker && this.timePickerIncrement)
                this.startDate.minute(Math.round(this.startDate.minute() / this.timePickerIncrement) * this.timePickerIncrement);

            if (this.minDate && this.startDate.isBefore(this.minDate)) {
                this.startDate = this.minDate.clone();
                if (this.timePicker && this.timePickerIncrement)
                    this.startDate.minute(Math.round(this.startDate.minute() / this.timePickerIncrement) * this.timePickerIncrement);
            }

            if (this.maxDate && this.startDate.isAfter(this.maxDate)) {
                this.startDate = this.maxDate.clone();
                if (this.timePicker && this.timePickerIncrement)
                    this.startDate.minute(Math.floor(this.startDate.minute() / this.timePickerIncrement) * this.timePickerIncrement);
            }

            if (!this.isShowing)
                this.updateElement();

            this.updateMonthsInView();
        },

        setEndDate: function(endDate) {
            if (typeof endDate === 'string')
                this.endDate = moment(endDate, this.locale.format);

            if (typeof endDate === 'object')
                this.endDate = moment(endDate);

            if (!this.timePicker)
                this.endDate = this.endDate.endOf('day');

            if (this.timePicker && this.timePickerIncrement)
                this.endDate.minute(Math.round(this.endDate.minute() / this.timePickerIncrement) * this.timePickerIncrement);

            if (this.endDate.isBefore(this.startDate))
                this.endDate = this.startDate.clone();

            if (this.maxDate && this.endDate.isAfter(this.maxDate))
                this.endDate = this.maxDate.clone();

            if (this.maxSpan && this.startDate.clone().add(this.maxSpan).isBefore(this.endDate))
                this.endDate = this.startDate.clone().add(this.maxSpan);

            this.previousRightTime = this.endDate.clone();

            this.container.find('.drp-selected').html(this.startDate.format(this.locale.format) + this.locale.separator + this.endDate.format(this.locale.format));

            if (!this.isShowing)
                this.updateElement();

            this.updateMonthsInView();
        },

        isInvalidDate: function() {
            return false;
        },

        isCustomDate: function() {
            return false;
        },

        updateView: function() {
            if (this.timePicker) {
                this.renderTimePicker('left');
                this.renderTimePicker('right');
                if (!this.endDate) {
                    this.container.find('.right .calendar-time select').prop('disabled', true).addClass('disabled');
                } else {
                    this.container.find('.right .calendar-time select').prop('disabled', false).removeClass('disabled');
                }
            }
            if (this.endDate)
                this.container.find('.drp-selected').html(this.startDate.format(this.locale.format) + this.locale.separator + this.endDate.format(this.locale.format));
            this.updateMonthsInView();
            this.updateCalendars();
            this.updateFormInputs();
        },

        updateMonthsInView: function() {
            if (this.endDate) {

                //if both dates are visible already, do nothing
                if (!this.singleDatePicker && this.leftCalendar.month && this.rightCalendar.month &&
                    (this.startDate.format('YYYY-MM') == this.leftCalendar.month.format('YYYY-MM') || this.startDate.format('YYYY-MM') == this.rightCalendar.month.format('YYYY-MM'))
                    &&
                    (this.endDate.format('YYYY-MM') == this.leftCalendar.month.format('YYYY-MM') || this.endDate.format('YYYY-MM') == this.rightCalendar.month.format('YYYY-MM'))
                ) {
                    return;
                }

                this.leftCalendar.month = this.startDate.clone().date(2);
                if (!this.linkedCalendars && (this.endDate.month() != this.startDate.month() || this.endDate.year() != this.startDate.year())) {
                    this.rightCalendar.month = this.endDate.clone().date(2);
                } else {
                    this.rightCalendar.month = this.startDate.clone().date(2).add(1, 'month');
                }

            } else {
                if (this.leftCalendar.month.format('YYYY-MM') != this.startDate.format('YYYY-MM') && this.rightCalendar.month.format('YYYY-MM') != this.startDate.format('YYYY-MM')) {
                    this.leftCalendar.month = this.startDate.clone().date(2);
                    this.rightCalendar.month = this.startDate.clone().date(2).add(1, 'month');
                }
            }
            if (this.maxDate && this.linkedCalendars && !this.singleDatePicker && this.rightCalendar.month > this.maxDate) {
                this.rightCalendar.month = this.maxDate.clone().date(2);
                this.leftCalendar.month = this.maxDate.clone().date(2).subtract(1, 'month');
            }
        },

        updateCalendars: function() {

            if (this.timePicker) {
                var hour, minute, second;
                if (this.endDate) {
                    hour = parseInt(this.container.find('.left .hourselect').val(), 10);
                    minute = parseInt(this.container.find('.left .minuteselect').val(), 10);
                    if (isNaN(minute)) {
                        minute = parseInt(this.container.find('.left .minuteselect option:last').val(), 10);
                    }
                    second = this.timePickerSeconds ? parseInt(this.container.find('.left .secondselect').val(), 10) : 0;
                    if (!this.timePicker24Hour) {
                        var ampm = this.container.find('.left .ampmselect').val();
                        if (ampm === 'PM' && hour < 12)
                            hour += 12;
                        if (ampm === 'AM' && hour === 12)
                            hour = 0;
                    }
                } else {
                    hour = parseInt(this.container.find('.right .hourselect').val(), 10);
                    minute = parseInt(this.container.find('.right .minuteselect').val(), 10);
                    if (isNaN(minute)) {
                        minute = parseInt(this.container.find('.right .minuteselect option:last').val(), 10);
                    }
                    second = this.timePickerSeconds ? parseInt(this.container.find('.right .secondselect').val(), 10) : 0;
                    if (!this.timePicker24Hour) {
                        var ampm = this.container.find('.right .ampmselect').val();
                        if (ampm === 'PM' && hour < 12)
                            hour += 12;
                        if (ampm === 'AM' && hour === 12)
                            hour = 0;
                    }
                }
                this.leftCalendar.month.hour(hour).minute(minute).second(second);
                this.rightCalendar.month.hour(hour).minute(minute).second(second);
            }

            this.renderCalendar('left');
            this.renderCalendar('right');

            //highlight any predefined range matching the current start and end dates
            this.container.find('.ranges li').removeClass('active');
            if (this.endDate == null) return;

            this.calculateChosenLabel();
        },

        renderCalendar: function(side) {

            //
            // Build the matrix of dates that will populate the calendar
            //

            var calendar = side == 'left' ? this.leftCalendar : this.rightCalendar;
            var month = calendar.month.month();
            var year = calendar.month.year();
            var hour = calendar.month.hour();
            var minute = calendar.month.minute();
            var second = calendar.month.second();
            var daysInMonth = moment([year, month]).daysInMonth();
            var firstDay = moment([year, month, 1]);
            var lastDay = moment([year, month, daysInMonth]);
            var lastMonth = moment(firstDay).subtract(1, 'month').month();
            var lastYear = moment(firstDay).subtract(1, 'month').year();
            var daysInLastMonth = moment([lastYear, lastMonth]).daysInMonth();
            var dayOfWeek = firstDay.day();

            //initialize a 6 rows x 7 columns array for the calendar
            var calendar = [];
            calendar.firstDay = firstDay;
            calendar.lastDay = lastDay;

            for (var i = 0; i < 6; i++) {
                calendar[i] = [];
            }

            //populate the calendar with date objects
            var startDay = daysInLastMonth - dayOfWeek + this.locale.firstDay + 1;
            if (startDay > daysInLastMonth)
                startDay -= 7;

            if (dayOfWeek == this.locale.firstDay)
                startDay = daysInLastMonth - 6;

            var curDate = moment([lastYear, lastMonth, startDay, 12, minute, second]);

            var col, row;
            for (var i = 0, col = 0, row = 0; i < 42; i++, col++, curDate = moment(curDate).add(24, 'hour')) {
                if (i > 0 && col % 7 === 0) {
                    col = 0;
                    row++;
                }
                calendar[row][col] = curDate.clone().hour(hour).minute(minute).second(second);
                curDate.hour(12);

                if (this.minDate && calendar[row][col].format('YYYY-MM-DD') == this.minDate.format('YYYY-MM-DD') && calendar[row][col].isBefore(this.minDate) && side == 'left') {
                    calendar[row][col] = this.minDate.clone();
                }

                if (this.maxDate && calendar[row][col].format('YYYY-MM-DD') == this.maxDate.format('YYYY-MM-DD') && calendar[row][col].isAfter(this.maxDate) && side == 'right') {
                    calendar[row][col] = this.maxDate.clone();
                }

            }

            //make the calendar object available to hoverDate/clickDate
            if (side == 'left') {
                this.leftCalendar.calendar = calendar;
            } else {
                this.rightCalendar.calendar = calendar;
            }

            //
            // Display the calendar
            //

            var minDate = side == 'left' ? this.minDate : this.startDate;
            var maxDate = this.maxDate;
            var selected = side == 'left' ? this.startDate : this.endDate;
            var arrow = this.locale.direction == 'ltr' ? {left: 'chevron-left', right: 'chevron-right'} : {left: 'chevron-right', right: 'chevron-left'};

            var html = '<table class="table-condensed">';
            html += '<thead>';
            html += '<tr>';

            // add empty cell for week number
            if (this.showWeekNumbers || this.showISOWeekNumbers)
                html += '<th></th>';

            if ((!minDate || minDate.isBefore(calendar.firstDay)) && (!this.linkedCalendars || side == 'left')) {
                html += '<th class="prev available"><span></span></th>';
            } else {
                html += '<th></th>';
            }

            var dateHtml = this.locale.monthNames[calendar[1][1].month()] + calendar[1][1].format(" YYYY");

            if (this.showDropdowns) {
                var currentMonth = calendar[1][1].month();
                var currentYear = calendar[1][1].year();
                var maxYear = (maxDate && maxDate.year()) || (this.maxYear);
                var minYear = (minDate && minDate.year()) || (this.minYear);
                var inMinYear = currentYear == minYear;
                var inMaxYear = currentYear == maxYear;

                var monthHtml = '<select class="monthselect">';
                for (var m = 0; m < 12; m++) {
                    if ((!inMinYear || (minDate && m >= minDate.month())) && (!inMaxYear || (maxDate && m <= maxDate.month()))) {
                        monthHtml += "<option value='" + m + "'" +
                            (m === currentMonth ? " selected='selected'" : "") +
                            ">" + this.locale.monthNames[m] + "</option>";
                    } else {
                        monthHtml += "<option value='" + m + "'" +
                            (m === currentMonth ? " selected='selected'" : "") +
                            " disabled='disabled'>" + this.locale.monthNames[m] + "</option>";
                    }
                }
                monthHtml += "</select>";

                var yearHtml = '<select class="yearselect">';
                for (var y = minYear; y <= maxYear; y++) {
                    yearHtml += '<option value="' + y + '"' +
                        (y === currentYear ? ' selected="selected"' : '') +
                        '>' + y + '</option>';
                }
                yearHtml += '</select>';

                dateHtml = monthHtml + yearHtml;
            }

            html += '<th colspan="5" class="month">' + dateHtml + '</th>';
            if ((!maxDate || maxDate.isAfter(calendar.lastDay)) && (!this.linkedCalendars || side == 'right' || this.singleDatePicker)) {
                html += '<th class="next available"><span></span></th>';
            } else {
                html += '<th></th>';
            }

            html += '</tr>';
            html += '<tr>';

            // add week number label
            if (this.showWeekNumbers || this.showISOWeekNumbers)
                html += '<th class="week">' + this.locale.weekLabel + '</th>';

            $.each(this.locale.daysOfWeek, function(index, dayOfWeek) {
                html += '<th>' + dayOfWeek + '</th>';
            });

            html += '</tr>';
            html += '</thead>';
            html += '<tbody>';

            //adjust maxDate to reflect the maxSpan setting in order to
            //grey out end dates beyond the maxSpan
            if (this.endDate == null && this.maxSpan) {
                var maxLimit = this.startDate.clone().add(this.maxSpan).endOf('day');
                if (!maxDate || maxLimit.isBefore(maxDate)) {
                    maxDate = maxLimit;
                }
            }

            for (var row = 0; row < 6; row++) {
                html += '<tr>';

                // add week number
                if (this.showWeekNumbers)
                    html += '<td class="week">' + calendar[row][0].week() + '</td>';
                else if (this.showISOWeekNumbers)
                    html += '<td class="week">' + calendar[row][0].isoWeek() + '</td>';

                for (var col = 0; col < 7; col++) {

                    var classes = [];

                    //highlight today's date
                    if (calendar[row][col].isSame(new Date(), "day"))
                        classes.push('today');

                    //highlight weekends
                    if (calendar[row][col].isoWeekday() > 5)
                        classes.push('weekend');

                    //grey out the dates in other months displayed at beginning and end of this calendar
                    if (calendar[row][col].month() != calendar[1][1].month())
                        classes.push('off', 'ends');

                    //don't allow selection of dates before the minimum date
                    if (this.minDate && calendar[row][col].isBefore(this.minDate, 'day'))
                        classes.push('off', 'disabled');

                    //don't allow selection of dates after the maximum date
                    if (maxDate && calendar[row][col].isAfter(maxDate, 'day'))
                        classes.push('off', 'disabled');

                    //don't allow selection of date if a custom function decides it's invalid
                    if (this.isInvalidDate(calendar[row][col]))
                        classes.push('off', 'disabled');

                    //highlight the currently selected start date
                    if (calendar[row][col].format('YYYY-MM-DD') == this.startDate.format('YYYY-MM-DD'))
                        classes.push('active', 'start-date');

                    //highlight the currently selected end date
                    if (this.endDate != null && calendar[row][col].format('YYYY-MM-DD') == this.endDate.format('YYYY-MM-DD'))
                        classes.push('active', 'end-date');

                    //highlight dates in-between the selected dates
                    if (this.endDate != null && calendar[row][col] > this.startDate && calendar[row][col] < this.endDate)
                        classes.push('in-range');

                    //apply custom classes for this date
                    var isCustom = this.isCustomDate(calendar[row][col]);
                    if (isCustom !== false) {
                        if (typeof isCustom === 'string')
                            classes.push(isCustom);
                        else
                            Array.prototype.push.apply(classes, isCustom);
                    }

                    var cname = '', disabled = false;
                    for (var i = 0; i < classes.length; i++) {
                        cname += classes[i] + ' ';
                        if (classes[i] == 'disabled')
                            disabled = true;
                    }
                    if (!disabled)
                        cname += 'available';

                    html += '<td class="' + cname.replace(/^\s+|\s+$/g, '') + '" data-title="' + 'r' + row + 'c' + col + '">' + calendar[row][col].date() + '</td>';

                }
                html += '</tr>';
            }

            html += '</tbody>';
            html += '</table>';

            this.container.find('.drp-calendar.' + side + ' .calendar-table').html(html);

        },

        renderTimePicker: function(side) {

            // Don't bother updating the time picker if it's currently disabled
            // because an end date hasn't been clicked yet
            if (side == 'right' && !this.endDate) return;

            var html, selected, minDate, maxDate = this.maxDate;

            if (this.maxSpan && (!this.maxDate || this.startDate.clone().add(this.maxSpan).isBefore(this.maxDate)))
                maxDate = this.startDate.clone().add(this.maxSpan);

            if (side == 'left') {
                selected = this.startDate.clone();
                minDate = this.minDate;
            } else if (side == 'right') {
                selected = this.endDate.clone();
                minDate = this.startDate;

                //Preserve the time already selected
                var timeSelector = this.container.find('.drp-calendar.right .calendar-time');
                if (timeSelector.html() != '') {

                    selected.hour(!isNaN(selected.hour()) ? selected.hour() : timeSelector.find('.hourselect option:selected').val());
                    selected.minute(!isNaN(selected.minute()) ? selected.minute() : timeSelector.find('.minuteselect option:selected').val());
                    selected.second(!isNaN(selected.second()) ? selected.second() : timeSelector.find('.secondselect option:selected').val());

                    if (!this.timePicker24Hour) {
                        var ampm = timeSelector.find('.ampmselect option:selected').val();
                        if (ampm === 'PM' && selected.hour() < 12)
                            selected.hour(selected.hour() + 12);
                        if (ampm === 'AM' && selected.hour() === 12)
                            selected.hour(0);
                    }

                }

                if (selected.isBefore(this.startDate))
                    selected = this.startDate.clone();

                if (maxDate && selected.isAfter(maxDate))
                    selected = maxDate.clone();

            }

            //
            // hours
            //

            html = '<select class="hourselect">';

            var start = this.timePicker24Hour ? 0 : 1;
            var end = this.timePicker24Hour ? 23 : 12;

            for (var i = start; i <= end; i++) {
                var i_in_24 = i;
                if (!this.timePicker24Hour)
                    i_in_24 = selected.hour() >= 12 ? (i == 12 ? 12 : i + 12) : (i == 12 ? 0 : i);

                var time = selected.clone().hour(i_in_24);
                var disabled = false;
                if (minDate && time.minute(59).isBefore(minDate))
                    disabled = true;
                if (maxDate && time.minute(0).isAfter(maxDate))
                    disabled = true;

                if (i_in_24 == selected.hour() && !disabled) {
                    html += '<option value="' + i + '" selected="selected">' + i + '</option>';
                } else if (disabled) {
                    html += '<option value="' + i + '" disabled="disabled" class="disabled">' + i + '</option>';
                } else {
                    html += '<option value="' + i + '">' + i + '</option>';
                }
            }

            html += '</select> ';

            //
            // minutes
            //

            html += ': <select class="minuteselect">';

            for (var i = 0; i < 60; i += this.timePickerIncrement) {
                var padded = i < 10 ? '0' + i : i;
                var time = selected.clone().minute(i);

                var disabled = false;
                if (minDate && time.second(59).isBefore(minDate))
                    disabled = true;
                if (maxDate && time.second(0).isAfter(maxDate))
                    disabled = true;

                if (selected.minute() == i && !disabled) {
                    html += '<option value="' + i + '" selected="selected">' + padded + '</option>';
                } else if (disabled) {
                    html += '<option value="' + i + '" disabled="disabled" class="disabled">' + padded + '</option>';
                } else {
                    html += '<option value="' + i + '">' + padded + '</option>';
                }
            }

            html += '</select> ';

            //
            // seconds
            //

            if (this.timePickerSeconds) {
                html += ': <select class="secondselect">';

                for (var i = 0; i < 60; i++) {
                    var padded = i < 10 ? '0' + i : i;
                    var time = selected.clone().second(i);

                    var disabled = false;
                    if (minDate && time.isBefore(minDate))
                        disabled = true;
                    if (maxDate && time.isAfter(maxDate))
                        disabled = true;

                    if (selected.second() == i && !disabled) {
                        html += '<option value="' + i + '" selected="selected">' + padded + '</option>';
                    } else if (disabled) {
                        html += '<option value="' + i + '" disabled="disabled" class="disabled">' + padded + '</option>';
                    } else {
                        html += '<option value="' + i + '">' + padded + '</option>';
                    }
                }

                html += '</select> ';
            }

            //
            // AM/PM
            //

            if (!this.timePicker24Hour) {
                html += '<select class="ampmselect">';

                var am_html = '';
                var pm_html = '';

                if (minDate && selected.clone().hour(12).minute(0).second(0).isBefore(minDate))
                    am_html = ' disabled="disabled" class="disabled"';

                if (maxDate && selected.clone().hour(0).minute(0).second(0).isAfter(maxDate))
                    pm_html = ' disabled="disabled" class="disabled"';

                if (selected.hour() >= 12) {
                    html += '<option value="AM"' + am_html + '>AM</option><option value="PM" selected="selected"' + pm_html + '>PM</option>';
                } else {
                    html += '<option value="AM" selected="selected"' + am_html + '>AM</option><option value="PM"' + pm_html + '>PM</option>';
                }

                html += '</select>';
            }

            this.container.find('.drp-calendar.' + side + ' .calendar-time').html(html);

        },

        updateFormInputs: function() {

            if (this.singleDatePicker || (this.endDate && (this.startDate.isBefore(this.endDate) || this.startDate.isSame(this.endDate)))) {
                this.container.find('button.applyBtn').prop('disabled', false);
            } else {
                this.container.find('button.applyBtn').prop('disabled', true);
            }

        },

        move: function() {
            var parentOffset = { top: 0, left: 0 },
                containerTop,
                drops = this.drops;

            var parentRightEdge = $(window).width();
            if (!this.parentEl.is('body')) {
                parentOffset = {
                    top: this.parentEl.offset().top - this.parentEl.scrollTop(),
                    left: this.parentEl.offset().left - this.parentEl.scrollLeft()
                };
                parentRightEdge = this.parentEl[0].clientWidth + this.parentEl.offset().left;
            }

            switch (drops) {
                case 'auto':
                    containerTop = this.element.offset().top + this.element.outerHeight() - parentOffset.top;
                    if (containerTop + this.container.outerHeight() >= this.parentEl[0].scrollHeight) {
                        containerTop = this.element.offset().top - this.container.outerHeight() - parentOffset.top;
                        drops = 'up';
                    }
                    break;
                case 'up':
                    containerTop = this.element.offset().top - this.container.outerHeight() - parentOffset.top;
                    break;
                default:
                    containerTop = this.element.offset().top + this.element.outerHeight() - parentOffset.top;
                    break;
            }

            // Force the container to it's actual width
            this.container.css({
                top: 0,
                left: 0,
                right: 'auto'
            });
            var containerWidth = this.container.outerWidth();

            this.container.toggleClass('drop-up', drops == 'up');

            if (this.opens == 'left') {
                var containerRight = parentRightEdge - this.element.offset().left - this.element.outerWidth();
                if (containerWidth + containerRight > $(window).width()) {
                    this.container.css({
                        top: containerTop,
                        right: 'auto',
                        left: 9
                    });
                } else {
                    this.container.css({
                        top: containerTop,
                        right: containerRight,
                        left: 'auto'
                    });
                }
            } else if (this.opens == 'center') {
                var containerLeft = this.element.offset().left - parentOffset.left + this.element.outerWidth() / 2
                    - containerWidth / 2;
                if (containerLeft < 0) {
                    this.container.css({
                        top: containerTop,
                        right: 'auto',
                        left: 9
                    });
                } else if (containerLeft + containerWidth > $(window).width()) {
                    this.container.css({
                        top: containerTop,
                        left: 'auto',
                        right: 0
                    });
                } else {
                    this.container.css({
                        top: containerTop,
                        left: containerLeft,
                        right: 'auto'
                    });
                }
            } else {
                var containerLeft = this.element.offset().left - parentOffset.left;
                if (containerLeft + containerWidth > $(window).width()) {
                    this.container.css({
                        top: containerTop,
                        left: 'auto',
                        right: 0
                    });
                } else {
                    this.container.css({
                        top: containerTop,
                        left: containerLeft,
                        right: 'auto'
                    });
                }
            }
        },

        show: function(e) {
            if (this.isShowing) return;

            // Create a click proxy that is private to this instance of datepicker, for unbinding
            this._outsideClickProxy = $.proxy(function(e) { this.outsideClick(e); }, this);

            // Bind global datepicker mousedown for hiding and
            $(document)
                .on('mousedown.daterangepicker', this._outsideClickProxy)
                // also support mobile devices
                .on('touchend.daterangepicker', this._outsideClickProxy)
                // also explicitly play nice with Bootstrap dropdowns, which stopPropagation when clicking them
                .on('click.daterangepicker', '[data-toggle=dropdown]', this._outsideClickProxy)
                // and also close when focus changes to outside the picker (eg. tabbing between controls)
                .on('focusin.daterangepicker', this._outsideClickProxy);

            // Reposition the picker if the window is resized while it's open
            $(window).on('resize.daterangepicker', $.proxy(function(e) { this.move(e); }, this));

            this.oldStartDate = this.startDate.clone();
            this.oldEndDate = this.endDate.clone();
            this.previousRightTime = this.endDate.clone();

            this.updateView();
            this.container.show();
            this.move();
            this.element.trigger('show.daterangepicker', this);
            this.isShowing = true;
        },

        hide: function(e) {
            if (!this.isShowing) return;

            //incomplete date selection, revert to last values
            if (!this.endDate) {
                this.startDate = this.oldStartDate.clone();
                this.endDate = this.oldEndDate.clone();
            }

            //if a new date range was selected, invoke the user callback function
            if (!this.startDate.isSame(this.oldStartDate) || !this.endDate.isSame(this.oldEndDate))
                this.callback(this.startDate.clone(), this.endDate.clone(), this.chosenLabel);

            //if picker is attached to a text input, update it
            this.updateElement();

            $(document).off('.daterangepicker');
            $(window).off('.daterangepicker');
            this.container.hide();
            this.element.trigger('hide.daterangepicker', this);
            this.isShowing = false;
        },

        toggle: function(e) {
            if (this.isShowing) {
                this.hide();
            } else {
                this.show();
            }
        },

        outsideClick: function(e) {
            var target = $(e.target);
            // if the page is clicked anywhere except within the daterangerpicker/button
            // itself then call this.hide()
            if (
                // ie modal dialog fix
                e.type == "focusin" ||
                target.closest(this.element).length ||
                target.closest(this.container).length ||
                target.closest('.calendar-table').length
            ) return;
            this.hide();
            this.element.trigger('outsideClick.daterangepicker', this);
        },

        showCalendars: function() {
            this.container.addClass('show-calendar');
            this.move();
            this.element.trigger('showCalendar.daterangepicker', this);
        },

        hideCalendars: function() {
            this.container.removeClass('show-calendar');
            this.element.trigger('hideCalendar.daterangepicker', this);
        },

        clickRange: function(e) {
            var label = e.target.getAttribute('data-range-key');
            this.chosenLabel = label;
            if (label == this.locale.customRangeLabel) {
                this.showCalendars();
            } else {
                var dates = this.ranges[label];
                this.startDate = dates[0];
                this.endDate = dates[1];

                if (!this.timePicker) {
                    this.startDate.startOf('day');
                    this.endDate.endOf('day');
                }

                if (!this.alwaysShowCalendars)
                    this.hideCalendars();
                this.clickApply();
            }
        },

        clickPrev: function(e) {
            var cal = $(e.target).parents('.drp-calendar');
            if (cal.hasClass('left')) {
                this.leftCalendar.month.subtract(1, 'month');
                if (this.linkedCalendars)
                    this.rightCalendar.month.subtract(1, 'month');
            } else {
                this.rightCalendar.month.subtract(1, 'month');
            }
            this.updateCalendars();
        },

        clickNext: function(e) {
            var cal = $(e.target).parents('.drp-calendar');
            if (cal.hasClass('left')) {
                this.leftCalendar.month.add(1, 'month');
            } else {
                this.rightCalendar.month.add(1, 'month');
                if (this.linkedCalendars)
                    this.leftCalendar.month.add(1, 'month');
            }
            this.updateCalendars();
        },

        hoverDate: function(e) {

            //ignore dates that can't be selected
            if (!$(e.target).hasClass('available')) return;

            var title = $(e.target).attr('data-title');
            var row = title.substr(1, 1);
            var col = title.substr(3, 1);
            var cal = $(e.target).parents('.drp-calendar');
            var date = cal.hasClass('left') ? this.leftCalendar.calendar[row][col] : this.rightCalendar.calendar[row][col];

            //highlight the dates between the start date and the date being hovered as a potential end date
            var leftCalendar = this.leftCalendar;
            var rightCalendar = this.rightCalendar;
            var startDate = this.startDate;
            if (!this.endDate) {
                this.container.find('.drp-calendar tbody td').each(function(index, el) {

                    //skip week numbers, only look at dates
                    if ($(el).hasClass('week')) return;

                    var title = $(el).attr('data-title');
                    var row = title.substr(1, 1);
                    var col = title.substr(3, 1);
                    var cal = $(el).parents('.drp-calendar');
                    var dt = cal.hasClass('left') ? leftCalendar.calendar[row][col] : rightCalendar.calendar[row][col];

                    if ((dt.isAfter(startDate) && dt.isBefore(date)) || dt.isSame(date, 'day')) {
                        $(el).addClass('in-range');
                    } else {
                        $(el).removeClass('in-range');
                    }

                });
            }

        },

        clickDate: function(e) {

            if (!$(e.target).hasClass('available')) return;

            var title = $(e.target).attr('data-title');
            var row = title.substr(1, 1);
            var col = title.substr(3, 1);
            var cal = $(e.target).parents('.drp-calendar');
            var date = cal.hasClass('left') ? this.leftCalendar.calendar[row][col] : this.rightCalendar.calendar[row][col];

            //
            // this function needs to do a few things:
            // * alternate between selecting a start and end date for the range,
            // * if the time picker is enabled, apply the hour/minute/second from the select boxes to the clicked date
            // * if autoapply is enabled, and an end date was chosen, apply the selection
            // * if single date picker mode, and time picker isn't enabled, apply the selection immediately
            // * if one of the inputs above the calendars was focused, cancel that manual input
            //

            if (this.endDate || date.isBefore(this.startDate, 'day')) { //picking start
                if (this.timePicker) {
                    var hour = parseInt(this.container.find('.left .hourselect').val(), 10);
                    if (!this.timePicker24Hour) {
                        var ampm = this.container.find('.left .ampmselect').val();
                        if (ampm === 'PM' && hour < 12)
                            hour += 12;
                        if (ampm === 'AM' && hour === 12)
                            hour = 0;
                    }
                    var minute = parseInt(this.container.find('.left .minuteselect').val(), 10);
                    if (isNaN(minute)) {
                        minute = parseInt(this.container.find('.left .minuteselect option:last').val(), 10);
                    }
                    var second = this.timePickerSeconds ? parseInt(this.container.find('.left .secondselect').val(), 10) : 0;
                    date = date.clone().hour(hour).minute(minute).second(second);
                }
                this.endDate = null;
                this.setStartDate(date.clone());
            } else if (!this.endDate && date.isBefore(this.startDate)) {
                //special case: clicking the same date for start/end,
                //but the time of the end date is before the start date
                this.setEndDate(this.startDate.clone());
            } else { // picking end
                if (this.timePicker) {
                    var hour = parseInt(this.container.find('.right .hourselect').val(), 10);
                    if (!this.timePicker24Hour) {
                        var ampm = this.container.find('.right .ampmselect').val();
                        if (ampm === 'PM' && hour < 12)
                            hour += 12;
                        if (ampm === 'AM' && hour === 12)
                            hour = 0;
                    }
                    var minute = parseInt(this.container.find('.right .minuteselect').val(), 10);
                    if (isNaN(minute)) {
                        minute = parseInt(this.container.find('.right .minuteselect option:last').val(), 10);
                    }
                    var second = this.timePickerSeconds ? parseInt(this.container.find('.right .secondselect').val(), 10) : 0;
                    date = date.clone().hour(hour).minute(minute).second(second);
                }
                this.setEndDate(date.clone());
                if (this.autoApply) {
                    this.calculateChosenLabel();
                    this.clickApply();
                }
            }

            if (this.singleDatePicker) {
                this.setEndDate(this.startDate);
                if (!this.timePicker && this.autoApply)
                    this.clickApply();
            }

            this.updateView();

            //This is to cancel the blur event handler if the mouse was in one of the inputs
            e.stopPropagation();

        },

        calculateChosenLabel: function () {
            var customRange = true;
            var i = 0;
            for (var range in this.ranges) {
                if (this.timePicker) {
                    var format = this.timePickerSeconds ? "YYYY-MM-DD HH:mm:ss" : "YYYY-MM-DD HH:mm";
                    //ignore times when comparing dates if time picker seconds is not enabled
                    if (this.startDate.format(format) == this.ranges[range][0].format(format) && this.endDate.format(format) == this.ranges[range][1].format(format)) {
                        customRange = false;
                        this.chosenLabel = this.container.find('.ranges li:eq(' + i + ')').addClass('active').attr('data-range-key');
                        break;
                    }
                } else {
                    //ignore times when comparing dates if time picker is not enabled
                    if (this.startDate.format('YYYY-MM-DD') == this.ranges[range][0].format('YYYY-MM-DD') && this.endDate.format('YYYY-MM-DD') == this.ranges[range][1].format('YYYY-MM-DD')) {
                        customRange = false;
                        this.chosenLabel = this.container.find('.ranges li:eq(' + i + ')').addClass('active').attr('data-range-key');
                        break;
                    }
                }
                i++;
            }
            if (customRange) {
                if (this.showCustomRangeLabel) {
                    this.chosenLabel = this.container.find('.ranges li:last').addClass('active').attr('data-range-key');
                } else {
                    this.chosenLabel = null;
                }
                this.showCalendars();
            }
        },

        clickApply: function(e) {
            this.hide();
            this.element.trigger('apply.daterangepicker', this);
        },

        clickCancel: function(e) {
            this.startDate = this.oldStartDate;
            this.endDate = this.oldEndDate;
            this.hide();
            this.element.trigger('cancel.daterangepicker', this);
        },

        monthOrYearChanged: function(e) {
            var isLeft = $(e.target).closest('.drp-calendar').hasClass('left'),
                leftOrRight = isLeft ? 'left' : 'right',
                cal = this.container.find('.drp-calendar.'+leftOrRight);

            // Month must be Number for new moment versions
            var month = parseInt(cal.find('.monthselect').val(), 10);
            var year = cal.find('.yearselect').val();

            if (!isLeft) {
                if (year < this.startDate.year() || (year == this.startDate.year() && month < this.startDate.month())) {
                    month = this.startDate.month();
                    year = this.startDate.year();
                }
            }

            if (this.minDate) {
                if (year < this.minDate.year() || (year == this.minDate.year() && month < this.minDate.month())) {
                    month = this.minDate.month();
                    year = this.minDate.year();
                }
            }

            if (this.maxDate) {
                if (year > this.maxDate.year() || (year == this.maxDate.year() && month > this.maxDate.month())) {
                    month = this.maxDate.month();
                    year = this.maxDate.year();
                }
            }

            if (isLeft) {
                this.leftCalendar.month.month(month).year(year);
                if (this.linkedCalendars)
                    this.rightCalendar.month = this.leftCalendar.month.clone().add(1, 'month');
            } else {
                this.rightCalendar.month.month(month).year(year);
                if (this.linkedCalendars)
                    this.leftCalendar.month = this.rightCalendar.month.clone().subtract(1, 'month');
            }
            this.updateCalendars();
        },

        timeChanged: function(e) {

            var cal = $(e.target).closest('.drp-calendar'),
                isLeft = cal.hasClass('left');

            var hour = parseInt(cal.find('.hourselect').val(), 10);
            var minute = parseInt(cal.find('.minuteselect').val(), 10);
            if (isNaN(minute)) {
                minute = parseInt(cal.find('.minuteselect option:last').val(), 10);
            }
            var second = this.timePickerSeconds ? parseInt(cal.find('.secondselect').val(), 10) : 0;

            if (!this.timePicker24Hour) {
                var ampm = cal.find('.ampmselect').val();
                if (ampm === 'PM' && hour < 12)
                    hour += 12;
                if (ampm === 'AM' && hour === 12)
                    hour = 0;
            }

            if (isLeft) {
                var start = this.startDate.clone();
                start.hour(hour);
                start.minute(minute);
                start.second(second);
                this.setStartDate(start);
                if (this.singleDatePicker) {
                    this.endDate = this.startDate.clone();
                } else if (this.endDate && this.endDate.format('YYYY-MM-DD') == start.format('YYYY-MM-DD') && this.endDate.isBefore(start)) {
                    this.setEndDate(start.clone());
                }
            } else if (this.endDate) {
                var end = this.endDate.clone();
                end.hour(hour);
                end.minute(minute);
                end.second(second);
                this.setEndDate(end);
            }

            //update the calendars so all clickable dates reflect the new time component
            this.updateCalendars();

            //update the form inputs above the calendars with the new time
            this.updateFormInputs();

            //re-render the time pickers because changing one selection can affect what's enabled in another
            this.renderTimePicker('left');
            this.renderTimePicker('right');

        },

        elementChanged: function() {
            if (!this.element.is('input')) return;
            if (!this.element.val().length) return;

            var dateString = this.element.val().split(this.locale.separator),
                start = null,
                end = null;

            if (dateString.length === 2) {
                start = moment(dateString[0], this.locale.format);
                end = moment(dateString[1], this.locale.format);
            }

            if (this.singleDatePicker || start === null || end === null) {
                start = moment(this.element.val(), this.locale.format);
                end = start;
            }

            if (!start.isValid() || !end.isValid()) return;

            this.setStartDate(start);
            this.setEndDate(end);
            this.updateView();
        },

        keydown: function(e) {
            //hide on tab or enter
            if ((e.keyCode === 9) || (e.keyCode === 13)) {
                this.hide();
            }

            //hide on esc and prevent propagation
            if (e.keyCode === 27) {
                e.preventDefault();
                e.stopPropagation();

                this.hide();
            }
        },

        updateElement: function() {
            if (this.element.is('input') && this.autoUpdateInput) {
                var newValue = this.startDate.format(this.locale.format);
                if (!this.singleDatePicker) {
                    newValue += this.locale.separator + this.endDate.format(this.locale.format);
                }
                if (newValue !== this.element.val()) {
                    this.element.val(newValue).trigger('change');
                }
            }
        },

        remove: function() {
            this.container.remove();
            this.element.off('.daterangepicker');
            this.element.removeData();
        }

    };

    $.fn.daterangepicker = function(options, callback) {
        var implementOptions = $.extend(true, {}, $.fn.daterangepicker.defaultOptions, options);
        this.each(function() {
            var el = $(this);
            if (el.data('daterangepicker'))
                el.data('daterangepicker').remove();
            el.data('daterangepicker', new DateRangePicker(el, implementOptions, callback));
        });
        return this;
    };

    return DateRangePicker;

}));

/*!
 * dist/inputmask
 * https://github.com/RobinHerbots/Inputmask
 * Copyright (c) 2010 - 2021 Robin Herbots
 * Licensed under the MIT license
 * Version: 5.0.7
 */
!function(e, t) {
    if ("object" == typeof exports && "object" == typeof module) module.exports = t(); else if ("function" == typeof define && define.amd) define([], t); else {
        var i = t();
        for (var a in i) ("object" == typeof exports ? exports : e)[a] = i[a];
    }
}(self, (function() {
    return function() {
        "use strict";
        var e = {
            8741: function(e, t) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.default = void 0;
                var i = !("undefined" == typeof window || !window.document || !window.document.createElement);
                t.default = i;
            },
            3976: function(e, t, i) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.default = void 0;
                var a, n = (a = i(5581)) && a.__esModule ? a : {
                    default: a
                };
                var r = {
                    _maxTestPos: 500,
                    placeholder: "_",
                    optionalmarker: [ "[", "]" ],
                    quantifiermarker: [ "{", "}" ],
                    groupmarker: [ "(", ")" ],
                    alternatormarker: "|",
                    escapeChar: "\\",
                    mask: null,
                    regex: null,
                    oncomplete: function() {},
                    onincomplete: function() {},
                    oncleared: function() {},
                    repeat: 0,
                    greedy: !1,
                    autoUnmask: !1,
                    removeMaskOnSubmit: !1,
                    clearMaskOnLostFocus: !0,
                    insertMode: !0,
                    insertModeVisual: !0,
                    clearIncomplete: !1,
                    alias: null,
                    onKeyDown: function() {},
                    onBeforeMask: null,
                    onBeforePaste: function(e, t) {
                        return "function" == typeof t.onBeforeMask ? t.onBeforeMask.call(this, e, t) : e;
                    },
                    onBeforeWrite: null,
                    onUnMask: null,
                    showMaskOnFocus: !0,
                    showMaskOnHover: !0,
                    onKeyValidation: function() {},
                    skipOptionalPartCharacter: " ",
                    numericInput: !1,
                    rightAlign: !1,
                    undoOnEscape: !0,
                    radixPoint: "",
                    _radixDance: !1,
                    groupSeparator: "",
                    keepStatic: null,
                    positionCaretOnTab: !0,
                    tabThrough: !1,
                    supportsInputType: [ "text", "tel", "url", "password", "search" ],
                    ignorables: [ n.default.BACKSPACE, n.default.TAB, n.default["PAUSE/BREAK"], n.default.ESCAPE, n.default.PAGE_UP, n.default.PAGE_DOWN, n.default.END, n.default.HOME, n.default.LEFT, n.default.UP, n.default.RIGHT, n.default.DOWN, n.default.INSERT, n.default.DELETE, 93, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 0, 229 ],
                    isComplete: null,
                    preValidation: null,
                    postValidation: null,
                    staticDefinitionSymbol: void 0,
                    jitMasking: !1,
                    nullable: !0,
                    inputEventOnly: !1,
                    noValuePatching: !1,
                    positionCaretOnClick: "lvp",
                    casing: null,
                    inputmode: "text",
                    importDataAttributes: !0,
                    shiftPositions: !0,
                    usePrototypeDefinitions: !0,
                    validationEventTimeOut: 3e3,
                    substitutes: {}
                };
                t.default = r;
            },
            7392: function(e, t) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.default = void 0;
                t.default = {
                    9: {
                        validator: "[0-9\uff10-\uff19]",
                        definitionSymbol: "*"
                    },
                    a: {
                        validator: "[A-Za-z\u0410-\u044f\u0401\u0451\xc0-\xff\xb5]",
                        definitionSymbol: "*"
                    },
                    "*": {
                        validator: "[0-9\uff10-\uff19A-Za-z\u0410-\u044f\u0401\u0451\xc0-\xff\xb5]"
                    }
                };
            },
            253: function(e, t) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.default = function(e, t, i) {
                    if (void 0 === i) return e.__data ? e.__data[t] : null;
                    e.__data = e.__data || {}, e.__data[t] = i;
                };
            },
            3776: function(e, t, i) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.Event = void 0, t.off = function(e, t) {
                    var i, a;
                    function n(e, t, n) {
                        if (e in i == !0) if (a.removeEventListener ? a.removeEventListener(e, n, !1) : a.detachEvent && a.detachEvent("on" + e, n),
                        "global" === t) for (var r in i[e]) i[e][r].splice(i[e][r].indexOf(n), 1); else i[e][t].splice(i[e][t].indexOf(n), 1);
                    }
                    function r(e, a) {
                        var n, r, o = [];
                        if (e.length > 0) if (void 0 === t) for (n = 0, r = i[e][a].length; n < r; n++) o.push({
                            ev: e,
                            namespace: a && a.length > 0 ? a : "global",
                            handler: i[e][a][n]
                        }); else o.push({
                            ev: e,
                            namespace: a && a.length > 0 ? a : "global",
                            handler: t
                        }); else if (a.length > 0) for (var l in i) for (var s in i[l]) if (s === a) if (void 0 === t) for (n = 0,
                                                                                                                                r = i[l][s].length; n < r; n++) o.push({
                            ev: l,
                            namespace: s,
                            handler: i[l][s][n]
                        }); else o.push({
                            ev: l,
                            namespace: s,
                            handler: t
                        });
                        return o;
                    }
                    if (u(this[0]) && e) {
                        i = this[0].eventRegistry, a = this[0];
                        for (var o = e.split(" "), l = 0; l < o.length; l++) for (var s = o[l].split("."), c = r(s[0], s[1]), f = 0, d = c.length; f < d; f++) n(c[f].ev, c[f].namespace, c[f].handler);
                    }
                    return this;
                }, t.on = function(e, t) {
                    function i(e, i) {
                        n.addEventListener ? n.addEventListener(e, t, !1) : n.attachEvent && n.attachEvent("on" + e, t),
                            a[e] = a[e] || {}, a[e][i] = a[e][i] || [], a[e][i].push(t);
                    }
                    if (u(this[0])) for (var a = this[0].eventRegistry, n = this[0], r = e.split(" "), o = 0; o < r.length; o++) {
                        var l = r[o].split("."), s = l[0], c = l[1] || "global";
                        i(s, c);
                    }
                    return this;
                }, t.trigger = function(e) {
                    if (u(this[0])) for (var t = this[0].eventRegistry, i = this[0], a = "string" == typeof e ? e.split(" ") : [ e.type ], r = 0; r < a.length; r++) {
                        var l = a[r].split("."), s = l[0], c = l[1] || "global";
                        if (void 0 !== document && "global" === c) {
                            var f, d, p = {
                                bubbles: !0,
                                cancelable: !0,
                                detail: arguments[1]
                            };
                            if (document.createEvent) {
                                try {
                                    if ("input" === s) p.inputType = "insertText", f = new InputEvent(s, p); else f = new CustomEvent(s, p);
                                } catch (e) {
                                    (f = document.createEvent("CustomEvent")).initCustomEvent(s, p.bubbles, p.cancelable, p.detail);
                                }
                                e.type && (0, n.default)(f, e), i.dispatchEvent(f);
                            } else (f = document.createEventObject()).eventType = s, f.detail = arguments[1],
                            e.type && (0, n.default)(f, e), i.fireEvent("on" + f.eventType, f);
                        } else if (void 0 !== t[s]) if (arguments[0] = arguments[0].type ? arguments[0] : o.default.Event(arguments[0]),
                            arguments[0].detail = arguments.slice(1), "global" === c) for (var h in t[s]) for (d = 0; d < t[s][h].length; d++) t[s][h][d].apply(i, arguments); else for (d = 0; d < t[s][c].length; d++) t[s][c][d].apply(i, arguments);
                    }
                    return this;
                };
                var a, n = s(i(600)), r = s(i(9380)), o = s(i(4963)), l = s(i(8741));
                function s(e) {
                    return e && e.__esModule ? e : {
                        default: e
                    };
                }
                function u(e) {
                    return e instanceof Element;
                }
                t.Event = a, "function" == typeof r.default.CustomEvent ? t.Event = a = r.default.CustomEvent : l.default && (t.Event = a = function(e, t) {
                    t = t || {
                        bubbles: !1,
                        cancelable: !1,
                        detail: void 0
                    };
                    var i = document.createEvent("CustomEvent");
                    return i.initCustomEvent(e, t.bubbles, t.cancelable, t.detail), i;
                }, a.prototype = r.default.Event.prototype);
            },
            600: function(e, t) {
                function i(e) {
                    return i = "function" == typeof Symbol && "symbol" == typeof Symbol.iterator ? function(e) {
                        return typeof e;
                    } : function(e) {
                        return e && "function" == typeof Symbol && e.constructor === Symbol && e !== Symbol.prototype ? "symbol" : typeof e;
                    }, i(e);
                }
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.default = function e() {
                    var t, a, n, r, o, l, s = arguments[0] || {}, u = 1, c = arguments.length, f = !1;
                    "boolean" == typeof s && (f = s, s = arguments[u] || {}, u++);
                    "object" !== i(s) && "function" != typeof s && (s = {});
                    for (;u < c; u++) if (null != (t = arguments[u])) for (a in t) n = s[a], r = t[a],
                    s !== r && (f && r && ("[object Object]" === Object.prototype.toString.call(r) || (o = Array.isArray(r))) ? (o ? (o = !1,
                        l = n && Array.isArray(n) ? n : []) : l = n && "[object Object]" === Object.prototype.toString.call(n) ? n : {},
                        s[a] = e(f, l, r)) : void 0 !== r && (s[a] = r));
                    return s;
                };
            },
            4963: function(e, t, i) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.default = void 0;
                var a = l(i(600)), n = l(i(9380)), r = l(i(253)), o = i(3776);
                function l(e) {
                    return e && e.__esModule ? e : {
                        default: e
                    };
                }
                var s = n.default.document;
                function u(e) {
                    return e instanceof u ? e : this instanceof u ? void (null != e && e !== n.default && (this[0] = e.nodeName ? e : void 0 !== e[0] && e[0].nodeName ? e[0] : s.querySelector(e),
                    void 0 !== this[0] && null !== this[0] && (this[0].eventRegistry = this[0].eventRegistry || {}))) : new u(e);
                }
                u.prototype = {
                    on: o.on,
                    off: o.off,
                    trigger: o.trigger
                }, u.extend = a.default, u.data = r.default, u.Event = o.Event;
                var c = u;
                t.default = c;
            },
            9845: function(e, t, i) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.ua = t.mobile = t.iphone = t.iemobile = t.ie = void 0;
                var a, n = (a = i(9380)) && a.__esModule ? a : {
                    default: a
                };
                var r = n.default.navigator && n.default.navigator.userAgent || "", o = r.indexOf("MSIE ") > 0 || r.indexOf("Trident/") > 0, l = "ontouchstart" in n.default, s = /iemobile/i.test(r), u = /iphone/i.test(r) && !s;
                t.iphone = u, t.iemobile = s, t.mobile = l, t.ie = o, t.ua = r;
            },
            7184: function(e, t) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.default = function(e) {
                    return e.replace(i, "\\$1");
                };
                var i = new RegExp("(\\" + [ "/", ".", "*", "+", "?", "|", "(", ")", "[", "]", "{", "}", "\\", "$", "^" ].join("|\\") + ")", "gim");
            },
            6030: function(e, t, i) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.EventHandlers = void 0;
                var a, n = i(8711), r = (a = i(5581)) && a.__esModule ? a : {
                    default: a
                }, o = i(9845), l = i(7215), s = i(7760), u = i(4713);
                function c(e, t) {
                    var i = "undefined" != typeof Symbol && e[Symbol.iterator] || e["@@iterator"];
                    if (!i) {
                        if (Array.isArray(e) || (i = function(e, t) {
                            if (!e) return;
                            if ("string" == typeof e) return f(e, t);
                            var i = Object.prototype.toString.call(e).slice(8, -1);
                            "Object" === i && e.constructor && (i = e.constructor.name);
                            if ("Map" === i || "Set" === i) return Array.from(e);
                            if ("Arguments" === i || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(i)) return f(e, t);
                        }(e)) || t && e && "number" == typeof e.length) {
                            i && (e = i);
                            var a = 0, n = function() {};
                            return {
                                s: n,
                                n: function() {
                                    return a >= e.length ? {
                                        done: !0
                                    } : {
                                        done: !1,
                                        value: e[a++]
                                    };
                                },
                                e: function(e) {
                                    throw e;
                                },
                                f: n
                            };
                        }
                        throw new TypeError("Invalid attempt to iterate non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.");
                    }
                    var r, o = !0, l = !1;
                    return {
                        s: function() {
                            i = i.call(e);
                        },
                        n: function() {
                            var e = i.next();
                            return o = e.done, e;
                        },
                        e: function(e) {
                            l = !0, r = e;
                        },
                        f: function() {
                            try {
                                o || null == i.return || i.return();
                            } finally {
                                if (l) throw r;
                            }
                        }
                    };
                }
                function f(e, t) {
                    (null == t || t > e.length) && (t = e.length);
                    for (var i = 0, a = new Array(t); i < t; i++) a[i] = e[i];
                    return a;
                }
                var d = {
                    keydownEvent: function(e) {
                        var t = this.inputmask, i = t.opts, a = t.dependencyLib, c = t.maskset, f = this, d = a(f), p = e.keyCode, h = n.caret.call(t, f), v = i.onKeyDown.call(this, e, n.getBuffer.call(t), h, i);
                        if (void 0 !== v) return v;
                        if (p === r.default.BACKSPACE || p === r.default.DELETE || o.iphone && p === r.default.BACKSPACE_SAFARI || e.ctrlKey && p === r.default.X && !("oncut" in f)) e.preventDefault(),
                            l.handleRemove.call(t, f, p, h), (0, s.writeBuffer)(f, n.getBuffer.call(t, !0), c.p, e, f.inputmask._valueGet() !== n.getBuffer.call(t).join("")); else if (p === r.default.END || p === r.default.PAGE_DOWN) {
                            e.preventDefault();
                            var m = n.seekNext.call(t, n.getLastValidPosition.call(t));
                            n.caret.call(t, f, e.shiftKey ? h.begin : m, m, !0);
                        } else p === r.default.HOME && !e.shiftKey || p === r.default.PAGE_UP ? (e.preventDefault(),
                            n.caret.call(t, f, 0, e.shiftKey ? h.begin : 0, !0)) : i.undoOnEscape && p === r.default.ESCAPE && !0 !== e.altKey ? ((0,
                            s.checkVal)(f, !0, !1, t.undoValue.split("")), d.trigger("click")) : p !== r.default.INSERT || e.shiftKey || e.ctrlKey || void 0 !== t.userOptions.insertMode ? !0 === i.tabThrough && p === r.default.TAB ? !0 === e.shiftKey ? (h.end = n.seekPrevious.call(t, h.end, !0),
                        !0 === u.getTest.call(t, h.end - 1).match.static && h.end--, h.begin = n.seekPrevious.call(t, h.end, !0),
                        h.begin >= 0 && h.end > 0 && (e.preventDefault(), n.caret.call(t, f, h.begin, h.end))) : (h.begin = n.seekNext.call(t, h.begin, !0),
                            h.end = n.seekNext.call(t, h.begin, !0), h.end < c.maskLength && h.end--, h.begin <= c.maskLength && (e.preventDefault(),
                            n.caret.call(t, f, h.begin, h.end))) : e.shiftKey || i.insertModeVisual && !1 === i.insertMode && (p === r.default.RIGHT ? setTimeout((function() {
                            var e = n.caret.call(t, f);
                            n.caret.call(t, f, e.begin);
                        }), 0) : p === r.default.LEFT && setTimeout((function() {
                            var e = n.translatePosition.call(t, f.inputmask.caretPos.begin);
                            n.translatePosition.call(t, f.inputmask.caretPos.end);
                            t.isRTL ? n.caret.call(t, f, e + (e === c.maskLength ? 0 : 1)) : n.caret.call(t, f, e - (0 === e ? 0 : 1));
                        }), 0)) : l.isSelection.call(t, h) ? i.insertMode = !i.insertMode : (i.insertMode = !i.insertMode,
                            n.caret.call(t, f, h.begin, h.begin));
                        t.ignorable = i.ignorables.includes(p);
                    },
                    keypressEvent: function(e, t, i, a, o) {
                        var u = this.inputmask || this, c = u.opts, f = u.dependencyLib, d = u.maskset, p = u.el, h = f(p), v = e.keyCode;
                        if (!(!0 === t || e.ctrlKey && e.altKey) && (e.ctrlKey || e.metaKey || u.ignorable)) return v === r.default.ENTER && u.undoValue !== u._valueGet(!0) && (u.undoValue = u._valueGet(!0),
                            setTimeout((function() {
                                h.trigger("change");
                            }), 0)), u.skipInputEvent = !0, !0;
                        if (v) {
                            44 !== v && 46 !== v || 3 !== e.location || "" === c.radixPoint || (v = c.radixPoint.charCodeAt(0));
                            var m, g = t ? {
                                begin: o,
                                end: o
                            } : n.caret.call(u, p), k = String.fromCharCode(v);
                            k = c.substitutes[k] || k, d.writeOutBuffer = !0;
                            var y = l.isValid.call(u, g, k, a, void 0, void 0, void 0, t);
                            if (!1 !== y && (n.resetMaskSet.call(u, !0), m = void 0 !== y.caret ? y.caret : n.seekNext.call(u, y.pos.begin ? y.pos.begin : y.pos),
                                d.p = m), m = c.numericInput && void 0 === y.caret ? n.seekPrevious.call(u, m) : m,
                            !1 !== i && (setTimeout((function() {
                                c.onKeyValidation.call(p, v, y);
                            }), 0), d.writeOutBuffer && !1 !== y)) {
                                var b = n.getBuffer.call(u);
                                (0, s.writeBuffer)(p, b, m, e, !0 !== t);
                            }
                            if (e.preventDefault(), t) return !1 !== y && (y.forwardPosition = m), y;
                        }
                    },
                    keyupEvent: function(e) {
                        var t = this.inputmask;
                        !t.isComposing || e.keyCode !== r.default.KEY_229 && e.keyCode !== r.default.ENTER || t.$el.trigger("input");
                    },
                    pasteEvent: function(e) {
                        var t, i = this.inputmask, a = i.opts, r = i._valueGet(!0), o = n.caret.call(i, this);
                        i.isRTL && (t = o.end, o.end = n.translatePosition.call(i, o.begin), o.begin = n.translatePosition.call(i, t));
                        var l = r.substr(0, o.begin), u = r.substr(o.end, r.length);
                        if (l == (i.isRTL ? n.getBufferTemplate.call(i).slice().reverse() : n.getBufferTemplate.call(i)).slice(0, o.begin).join("") && (l = ""),
                        u == (i.isRTL ? n.getBufferTemplate.call(i).slice().reverse() : n.getBufferTemplate.call(i)).slice(o.end).join("") && (u = ""),
                        window.clipboardData && window.clipboardData.getData) r = l + window.clipboardData.getData("Text") + u; else {
                            if (!e.clipboardData || !e.clipboardData.getData) return !0;
                            r = l + e.clipboardData.getData("text/plain") + u;
                        }
                        var f = r;
                        if (i.isRTL) {
                            f = f.split("");
                            var d, p = c(n.getBufferTemplate.call(i));
                            try {
                                for (p.s(); !(d = p.n()).done; ) {
                                    var h = d.value;
                                    f[0] === h && f.shift();
                                }
                            } catch (e) {
                                p.e(e);
                            } finally {
                                p.f();
                            }
                            f = f.join("");
                        }
                        if ("function" == typeof a.onBeforePaste) {
                            if (!1 === (f = a.onBeforePaste.call(i, f, a))) return !1;
                            f || (f = r);
                        }
                        (0, s.checkVal)(this, !0, !1, f.toString().split(""), e), e.preventDefault();
                    },
                    inputFallBackEvent: function(e) {
                        var t = this.inputmask, i = t.opts, a = t.dependencyLib;
                        var l = this, c = l.inputmask._valueGet(!0), f = (t.isRTL ? n.getBuffer.call(t).slice().reverse() : n.getBuffer.call(t)).join(""), p = n.caret.call(t, l, void 0, void 0, !0);
                        if (f !== c) {
                            c = function(e, i, a) {
                                if (o.iemobile) {
                                    var r = i.replace(n.getBuffer.call(t).join(""), "");
                                    if (1 === r.length) {
                                        var l = i.split("");
                                        l.splice(a.begin, 0, r), i = l.join("");
                                    }
                                }
                                return i;
                            }(0, c, p);
                            var h = function(e, a, r) {
                                for (var o, l, s, c = e.substr(0, r.begin).split(""), f = e.substr(r.begin).split(""), d = a.substr(0, r.begin).split(""), p = a.substr(r.begin).split(""), h = c.length >= d.length ? c.length : d.length, v = f.length >= p.length ? f.length : p.length, m = "", g = [], k = "~"; c.length < h; ) c.push(k);
                                for (;d.length < h; ) d.push(k);
                                for (;f.length < v; ) f.unshift(k);
                                for (;p.length < v; ) p.unshift(k);
                                var y = c.concat(f), b = d.concat(p);
                                for (l = 0, o = y.length; l < o; l++) switch (s = u.getPlaceholder.call(t, n.translatePosition.call(t, l)),
                                    m) {
                                    case "insertText":
                                        b[l - 1] === y[l] && r.begin == y.length - 1 && g.push(y[l]), l = o;
                                        break;

                                    case "insertReplacementText":
                                    case "deleteContentBackward":
                                        y[l] === k ? r.end++ : l = o;
                                        break;

                                    default:
                                        y[l] !== b[l] && (y[l + 1] !== k && y[l + 1] !== s && void 0 !== y[l + 1] || (b[l] !== s || b[l + 1] !== k) && b[l] !== k ? b[l + 1] === k && b[l] === y[l + 1] ? (m = "insertText",
                                            g.push(y[l]), r.begin--, r.end--) : y[l] !== s && y[l] !== k && (y[l + 1] === k || b[l] !== y[l] && b[l + 1] === y[l + 1]) ? (m = "insertReplacementText",
                                            g.push(y[l]), r.begin--) : y[l] === k ? (m = "deleteContentBackward", (n.isMask.call(t, n.translatePosition.call(t, l), !0) || b[l] === i.radixPoint) && r.end++) : l = o : (m = "insertText",
                                            g.push(y[l]), r.begin--, r.end--));
                                }
                                return {
                                    action: m,
                                    data: g,
                                    caret: r
                                };
                            }(c, f, p);
                            switch ((l.inputmask.shadowRoot || l.ownerDocument).activeElement !== l && l.focus(),
                                (0, s.writeBuffer)(l, n.getBuffer.call(t)), n.caret.call(t, l, p.begin, p.end, !0),
                                h.action) {
                                case "insertText":
                                case "insertReplacementText":
                                    h.data.forEach((function(e, i) {
                                        var n = new a.Event("keypress");
                                        n.keyCode = e.charCodeAt(0), t.ignorable = !1, d.keypressEvent.call(l, n);
                                    })), setTimeout((function() {
                                        t.$el.trigger("keyup");
                                    }), 0);
                                    break;

                                case "deleteContentBackward":
                                    var v = new a.Event("keydown");
                                    v.keyCode = r.default.BACKSPACE, d.keydownEvent.call(l, v);
                                    break;

                                default:
                                    (0, s.applyInputValue)(l, c);
                            }
                            e.preventDefault();
                        }
                    },
                    compositionendEvent: function(e) {
                        var t = this.inputmask;
                        t.isComposing = !1, t.$el.trigger("input");
                    },
                    setValueEvent: function(e) {
                        var t = this.inputmask, i = this, a = e && e.detail ? e.detail[0] : arguments[1];
                        void 0 === a && (a = i.inputmask._valueGet(!0)), (0, s.applyInputValue)(i, a), (e.detail && void 0 !== e.detail[1] || void 0 !== arguments[2]) && n.caret.call(t, i, e.detail ? e.detail[1] : arguments[2]);
                    },
                    focusEvent: function(e) {
                        var t = this.inputmask, i = t.opts, a = this, r = a.inputmask._valueGet();
                        i.showMaskOnFocus && r !== n.getBuffer.call(t).join("") && (0, s.writeBuffer)(a, n.getBuffer.call(t), n.seekNext.call(t, n.getLastValidPosition.call(t))),
                        !0 !== i.positionCaretOnTab || !1 !== t.mouseEnter || l.isComplete.call(t, n.getBuffer.call(t)) && -1 !== n.getLastValidPosition.call(t) || d.clickEvent.apply(a, [ e, !0 ]),
                            t.undoValue = t._valueGet(!0);
                    },
                    invalidEvent: function(e) {
                        this.inputmask.validationEvent = !0;
                    },
                    mouseleaveEvent: function() {
                        var e = this.inputmask, t = e.opts, i = this;
                        e.mouseEnter = !1, t.clearMaskOnLostFocus && (i.inputmask.shadowRoot || i.ownerDocument).activeElement !== i && (0,
                            s.HandleNativePlaceholder)(i, e.originalPlaceholder);
                    },
                    clickEvent: function(e, t) {
                        var i = this.inputmask, a = this;
                        if ((a.inputmask.shadowRoot || a.ownerDocument).activeElement === a) {
                            var r = n.determineNewCaretPosition.call(i, n.caret.call(i, a), t);
                            void 0 !== r && n.caret.call(i, a, r);
                        }
                    },
                    cutEvent: function(e) {
                        var t = this.inputmask, i = t.maskset, a = this, o = n.caret.call(t, a), u = t.isRTL ? n.getBuffer.call(t).slice(o.end, o.begin) : n.getBuffer.call(t).slice(o.begin, o.end), c = t.isRTL ? u.reverse().join("") : u.join("");
                        window.navigator.clipboard ? window.navigator.clipboard.writeText(c) : window.clipboardData && window.clipboardData.getData && window.clipboardData.setData("Text", c),
                            l.handleRemove.call(t, a, r.default.DELETE, o), (0, s.writeBuffer)(a, n.getBuffer.call(t), i.p, e, t.undoValue !== t._valueGet(!0));
                    },
                    blurEvent: function(e) {
                        var t = this.inputmask, i = t.opts, a = (0, t.dependencyLib)(this), r = this;
                        if (r.inputmask) {
                            (0, s.HandleNativePlaceholder)(r, t.originalPlaceholder);
                            var o = r.inputmask._valueGet(), u = n.getBuffer.call(t).slice();
                            "" !== o && (i.clearMaskOnLostFocus && (-1 === n.getLastValidPosition.call(t) && o === n.getBufferTemplate.call(t).join("") ? u = [] : s.clearOptionalTail.call(t, u)),
                            !1 === l.isComplete.call(t, u) && (setTimeout((function() {
                                a.trigger("incomplete");
                            }), 0), i.clearIncomplete && (n.resetMaskSet.call(t), u = i.clearMaskOnLostFocus ? [] : n.getBufferTemplate.call(t).slice())),
                                (0, s.writeBuffer)(r, u, void 0, e)), t.undoValue !== t._valueGet(!0) && (t.undoValue = t._valueGet(!0),
                                a.trigger("change"));
                        }
                    },
                    mouseenterEvent: function() {
                        var e = this.inputmask, t = e.opts, i = this;
                        if (e.mouseEnter = !0, (i.inputmask.shadowRoot || i.ownerDocument).activeElement !== i) {
                            var a = (e.isRTL ? n.getBufferTemplate.call(e).slice().reverse() : n.getBufferTemplate.call(e)).join("");
                            e.placeholder !== a && i.placeholder !== e.originalPlaceholder && (e.originalPlaceholder = i.placeholder),
                            t.showMaskOnHover && (0, s.HandleNativePlaceholder)(i, a);
                        }
                    },
                    submitEvent: function() {
                        var e = this.inputmask, t = e.opts;
                        e.undoValue !== e._valueGet(!0) && e.$el.trigger("change"), -1 === n.getLastValidPosition.call(e) && e._valueGet && e._valueGet() === n.getBufferTemplate.call(e).join("") && e._valueSet(""),
                        t.clearIncomplete && !1 === l.isComplete.call(e, n.getBuffer.call(e)) && e._valueSet(""),
                        t.removeMaskOnSubmit && (e._valueSet(e.unmaskedvalue(), !0), setTimeout((function() {
                            (0, s.writeBuffer)(e.el, n.getBuffer.call(e));
                        }), 0));
                    },
                    resetEvent: function() {
                        var e = this.inputmask;
                        e.refreshValue = !0, setTimeout((function() {
                            (0, s.applyInputValue)(e.el, e._valueGet(!0));
                        }), 0);
                    }
                };
                t.EventHandlers = d;
            },
            9716: function(e, t, i) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.EventRuler = void 0;
                var a = l(i(2394)), n = l(i(5581)), r = i(8711), o = i(7760);
                function l(e) {
                    return e && e.__esModule ? e : {
                        default: e
                    };
                }
                var s = {
                    on: function(e, t, i) {
                        var l = e.inputmask.dependencyLib, s = function(t) {
                            t.originalEvent && (t = t.originalEvent || t, arguments[0] = t);
                            var s, u = this, c = u.inputmask, f = c ? c.opts : void 0;
                            if (void 0 === c && "FORM" !== this.nodeName) {
                                var d = l.data(u, "_inputmask_opts");
                                l(u).off(), d && new a.default(d).mask(u);
                            } else {
                                if ([ "submit", "reset", "setvalue" ].includes(t.type) || "FORM" === this.nodeName || !(u.disabled || u.readOnly && !("keydown" === t.type && t.ctrlKey && 67 === t.keyCode || !1 === f.tabThrough && t.keyCode === n.default.TAB))) {
                                    switch (t.type) {
                                        case "input":
                                            if (!0 === c.skipInputEvent || t.inputType && "insertCompositionText" === t.inputType) return c.skipInputEvent = !1,
                                                t.preventDefault();
                                            break;

                                        case "keydown":
                                            c.skipKeyPressEvent = !1, c.skipInputEvent = c.isComposing = t.keyCode === n.default.KEY_229;
                                            break;

                                        case "keyup":
                                        case "compositionend":
                                            c.isComposing && (c.skipInputEvent = !1);
                                            break;

                                        case "keypress":
                                            if (!0 === c.skipKeyPressEvent) return t.preventDefault();
                                            c.skipKeyPressEvent = !0;
                                            break;

                                        case "click":
                                        case "focus":
                                            return c.validationEvent ? (c.validationEvent = !1, e.blur(), (0, o.HandleNativePlaceholder)(e, (c.isRTL ? r.getBufferTemplate.call(c).slice().reverse() : r.getBufferTemplate.call(c)).join("")),
                                                setTimeout((function() {
                                                    e.focus();
                                                }), f.validationEventTimeOut), !1) : (s = arguments, setTimeout((function() {
                                                e.inputmask && i.apply(u, s);
                                            }), 0), !1);
                                    }
                                    var p = i.apply(u, arguments);
                                    return !1 === p && (t.preventDefault(), t.stopPropagation()), p;
                                }
                                t.preventDefault();
                            }
                        };
                        [ "submit", "reset" ].includes(t) ? (s = s.bind(e), null !== e.form && l(e.form).on(t, s)) : l(e).on(t, s),
                            e.inputmask.events[t] = e.inputmask.events[t] || [], e.inputmask.events[t].push(s);
                    },
                    off: function(e, t) {
                        if (e.inputmask && e.inputmask.events) {
                            var i = e.inputmask.dependencyLib, a = e.inputmask.events;
                            for (var n in t && ((a = [])[t] = e.inputmask.events[t]), a) {
                                for (var r = a[n]; r.length > 0; ) {
                                    var o = r.pop();
                                    [ "submit", "reset" ].includes(n) ? null !== e.form && i(e.form).off(n, o) : i(e).off(n, o);
                                }
                                delete e.inputmask.events[n];
                            }
                        }
                    }
                };
                t.EventRuler = s;
            },
            219: function(e, t, i) {
                var a = d(i(2394)), n = d(i(5581)), r = d(i(7184)), o = i(8711), l = i(4713);
                function s(e) {
                    return s = "function" == typeof Symbol && "symbol" == typeof Symbol.iterator ? function(e) {
                        return typeof e;
                    } : function(e) {
                        return e && "function" == typeof Symbol && e.constructor === Symbol && e !== Symbol.prototype ? "symbol" : typeof e;
                    }, s(e);
                }
                function u(e, t) {
                    return function(e) {
                        if (Array.isArray(e)) return e;
                    }(e) || function(e, t) {
                        var i = null == e ? null : "undefined" != typeof Symbol && e[Symbol.iterator] || e["@@iterator"];
                        if (null == i) return;
                        var a, n, r = [], o = !0, l = !1;
                        try {
                            for (i = i.call(e); !(o = (a = i.next()).done) && (r.push(a.value), !t || r.length !== t); o = !0) ;
                        } catch (e) {
                            l = !0, n = e;
                        } finally {
                            try {
                                o || null == i.return || i.return();
                            } finally {
                                if (l) throw n;
                            }
                        }
                        return r;
                    }(e, t) || function(e, t) {
                        if (!e) return;
                        if ("string" == typeof e) return c(e, t);
                        var i = Object.prototype.toString.call(e).slice(8, -1);
                        "Object" === i && e.constructor && (i = e.constructor.name);
                        if ("Map" === i || "Set" === i) return Array.from(e);
                        if ("Arguments" === i || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(i)) return c(e, t);
                    }(e, t) || function() {
                        throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.");
                    }();
                }
                function c(e, t) {
                    (null == t || t > e.length) && (t = e.length);
                    for (var i = 0, a = new Array(t); i < t; i++) a[i] = e[i];
                    return a;
                }
                function f(e, t) {
                    for (var i = 0; i < t.length; i++) {
                        var a = t[i];
                        a.enumerable = a.enumerable || !1, a.configurable = !0, "value" in a && (a.writable = !0),
                            Object.defineProperty(e, a.key, a);
                    }
                }
                function d(e) {
                    return e && e.__esModule ? e : {
                        default: e
                    };
                }
                var p = a.default.dependencyLib, h = function() {
                    function e(t, i, a) {
                        !function(e, t) {
                            if (!(e instanceof t)) throw new TypeError("Cannot call a class as a function");
                        }(this, e), this.mask = t, this.format = i, this.opts = a, this._date = new Date(1, 0, 1),
                            this.initDateObject(t, this.opts);
                    }
                    var t, i, a;
                    return t = e, (i = [ {
                        key: "date",
                        get: function() {
                            return void 0 === this._date && (this._date = new Date(1, 0, 1), this.initDateObject(void 0, this.opts)),
                                this._date;
                        }
                    }, {
                        key: "initDateObject",
                        value: function(e, t) {
                            var i;
                            for (P(t).lastIndex = 0; i = P(t).exec(this.format); ) {
                                var a = new RegExp("\\d+$").exec(i[0]), n = a ? i[0][0] + "x" : i[0], r = void 0;
                                if (void 0 !== e) {
                                    if (a) {
                                        var o = P(t).lastIndex, l = O(i.index, t);
                                        P(t).lastIndex = o, r = e.slice(0, e.indexOf(l.nextMatch[0]));
                                    } else r = e.slice(0, n.length);
                                    e = e.slice(r.length);
                                }
                                Object.prototype.hasOwnProperty.call(g, n) && this.setValue(this, r, n, g[n][2], g[n][1]);
                            }
                        }
                    }, {
                        key: "setValue",
                        value: function(e, t, i, a, n) {
                            if (void 0 !== t && (e[a] = "ampm" === a ? t : t.replace(/[^0-9]/g, "0"), e["raw" + a] = t.replace(/\s/g, "_")),
                            void 0 !== n) {
                                var r = e[a];
                                ("day" === a && 29 === parseInt(r) || "month" === a && 2 === parseInt(r)) && (29 !== parseInt(e.day) || 2 !== parseInt(e.month) || "" !== e.year && void 0 !== e.year || e._date.setFullYear(2012, 1, 29)),
                                "day" === a && (m = !0, 0 === parseInt(r) && (r = 1)), "month" === a && (m = !0),
                                "year" === a && (m = !0, r.length < 4 && (r = _(r, 4, !0))), "" === r || isNaN(r) || n.call(e._date, r),
                                "ampm" === a && n.call(e._date, r);
                            }
                        }
                    }, {
                        key: "reset",
                        value: function() {
                            this._date = new Date(1, 0, 1);
                        }
                    }, {
                        key: "reInit",
                        value: function() {
                            this._date = void 0, this.date;
                        }
                    } ]) && f(t.prototype, i), a && f(t, a), Object.defineProperty(t, "prototype", {
                        writable: !1
                    }), e;
                }(), v = (new Date).getFullYear(), m = !1, g = {
                    d: [ "[1-9]|[12][0-9]|3[01]", Date.prototype.setDate, "day", Date.prototype.getDate ],
                    dd: [ "0[1-9]|[12][0-9]|3[01]", Date.prototype.setDate, "day", function() {
                        return _(Date.prototype.getDate.call(this), 2);
                    } ],
                    ddd: [ "" ],
                    dddd: [ "" ],
                    m: [ "[1-9]|1[012]", function(e) {
                        var t = e ? parseInt(e) : 0;
                        return t > 0 && t--, Date.prototype.setMonth.call(this, t);
                    }, "month", function() {
                        return Date.prototype.getMonth.call(this) + 1;
                    } ],
                    mm: [ "0[1-9]|1[012]", function(e) {
                        var t = e ? parseInt(e) : 0;
                        return t > 0 && t--, Date.prototype.setMonth.call(this, t);
                    }, "month", function() {
                        return _(Date.prototype.getMonth.call(this) + 1, 2);
                    } ],
                    mmm: [ "" ],
                    mmmm: [ "" ],
                    yy: [ "[0-9]{2}", Date.prototype.setFullYear, "year", function() {
                        return _(Date.prototype.getFullYear.call(this), 2);
                    } ],
                    yyyy: [ "[0-9]{4}", Date.prototype.setFullYear, "year", function() {
                        return _(Date.prototype.getFullYear.call(this), 4);
                    } ],
                    h: [ "[1-9]|1[0-2]", Date.prototype.setHours, "hours", Date.prototype.getHours ],
                    hh: [ "0[1-9]|1[0-2]", Date.prototype.setHours, "hours", function() {
                        return _(Date.prototype.getHours.call(this), 2);
                    } ],
                    hx: [ function(e) {
                        return "[0-9]{".concat(e, "}");
                    }, Date.prototype.setHours, "hours", function(e) {
                        return Date.prototype.getHours;
                    } ],
                    H: [ "1?[0-9]|2[0-3]", Date.prototype.setHours, "hours", Date.prototype.getHours ],
                    HH: [ "0[0-9]|1[0-9]|2[0-3]", Date.prototype.setHours, "hours", function() {
                        return _(Date.prototype.getHours.call(this), 2);
                    } ],
                    Hx: [ function(e) {
                        return "[0-9]{".concat(e, "}");
                    }, Date.prototype.setHours, "hours", function(e) {
                        return function() {
                            return _(Date.prototype.getHours.call(this), e);
                        };
                    } ],
                    M: [ "[1-5]?[0-9]", Date.prototype.setMinutes, "minutes", Date.prototype.getMinutes ],
                    MM: [ "0[0-9]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9]", Date.prototype.setMinutes, "minutes", function() {
                        return _(Date.prototype.getMinutes.call(this), 2);
                    } ],
                    s: [ "[1-5]?[0-9]", Date.prototype.setSeconds, "seconds", Date.prototype.getSeconds ],
                    ss: [ "0[0-9]|1[0-9]|2[0-9]|3[0-9]|4[0-9]|5[0-9]", Date.prototype.setSeconds, "seconds", function() {
                        return _(Date.prototype.getSeconds.call(this), 2);
                    } ],
                    l: [ "[0-9]{3}", Date.prototype.setMilliseconds, "milliseconds", function() {
                        return _(Date.prototype.getMilliseconds.call(this), 3);
                    } ],
                    L: [ "[0-9]{2}", Date.prototype.setMilliseconds, "milliseconds", function() {
                        return _(Date.prototype.getMilliseconds.call(this), 2);
                    } ],
                    t: [ "[ap]", y, "ampm", b, 1 ],
                    tt: [ "[ap]m", y, "ampm", b, 2 ],
                    T: [ "[AP]", y, "ampm", b, 1 ],
                    TT: [ "[AP]M", y, "ampm", b, 2 ],
                    Z: [ ".*", void 0, "Z", function() {
                        var e = this.toString().match(/\((.+)\)/)[1];
                        e.includes(" ") && (e = (e = e.replace("-", " ").toUpperCase()).split(" ").map((function(e) {
                            return u(e, 1)[0];
                        })).join(""));
                        return e;
                    } ],
                    o: [ "" ],
                    S: [ "" ]
                }, k = {
                    isoDate: "yyyy-mm-dd",
                    isoTime: "HH:MM:ss",
                    isoDateTime: "yyyy-mm-dd'T'HH:MM:ss",
                    isoUtcDateTime: "UTC:yyyy-mm-dd'T'HH:MM:ss'Z'"
                };
                function y(e) {
                    var t = this.getHours();
                    e.toLowerCase().includes("p") ? this.setHours(t + 12) : e.toLowerCase().includes("a") && t >= 12 && this.setHours(t - 12);
                }
                function b() {
                    var e = this.getHours();
                    return (e = e || 12) >= 12 ? "PM" : "AM";
                }
                function x(e) {
                    var t = new RegExp("\\d+$").exec(e[0]);
                    if (t && void 0 !== t[0]) {
                        var i = g[e[0][0] + "x"].slice("");
                        return i[0] = i[0](t[0]), i[3] = i[3](t[0]), i;
                    }
                    if (g[e[0]]) return g[e[0]];
                }
                function P(e) {
                    if (!e.tokenizer) {
                        var t = [], i = [];
                        for (var a in g) if (/\.*x$/.test(a)) {
                            var n = a[0] + "\\d+";
                            -1 === i.indexOf(n) && i.push(n);
                        } else -1 === t.indexOf(a[0]) && t.push(a[0]);
                        e.tokenizer = "(" + (i.length > 0 ? i.join("|") + "|" : "") + t.join("+|") + ")+?|.",
                            e.tokenizer = new RegExp(e.tokenizer, "g");
                    }
                    return e.tokenizer;
                }
                function E(e, t, i) {
                    if (!m) return !0;
                    if (void 0 === e.rawday || !isFinite(e.rawday) && new Date(e.date.getFullYear(), isFinite(e.rawmonth) ? e.month : e.date.getMonth() + 1, 0).getDate() >= e.day || "29" == e.day && (!isFinite(e.rawyear) || void 0 === e.rawyear || "" === e.rawyear) || new Date(e.date.getFullYear(), isFinite(e.rawmonth) ? e.month : e.date.getMonth() + 1, 0).getDate() >= e.day) return t;
                    if ("29" == e.day) {
                        var a = O(t.pos, i);
                        if ("yyyy" === a.targetMatch[0] && t.pos - a.targetMatchIndex == 2) return t.remove = t.pos + 1,
                            t;
                    } else if ("02" == e.month && "30" == e.day && void 0 !== t.c) return e.day = "03",
                        e.date.setDate(3), e.date.setMonth(1), t.insert = [ {
                        pos: t.pos,
                        c: "0"
                    }, {
                        pos: t.pos + 1,
                        c: t.c
                    } ], t.caret = o.seekNext.call(this, t.pos + 1), t;
                    return !1;
                }
                function S(e, t, i, a) {
                    var n, o, l = "";
                    for (P(i).lastIndex = 0; n = P(i).exec(e); ) {
                        if (void 0 === t) if (o = x(n)) l += "(" + o[0] + ")"; else switch (n[0]) {
                            case "[":
                                l += "(";
                                break;

                            case "]":
                                l += ")?";
                                break;

                            default:
                                l += (0, r.default)(n[0]);
                        } else if (o = x(n)) if (!0 !== a && o[3]) l += o[3].call(t.date); else o[2] ? l += t["raw" + o[2]] : l += n[0]; else l += n[0];
                    }
                    return l;
                }
                function _(e, t, i) {
                    for (e = String(e), t = t || 2; e.length < t; ) e = i ? e + "0" : "0" + e;
                    return e;
                }
                function w(e, t, i) {
                    return "string" == typeof e ? new h(e, t, i) : e && "object" === s(e) && Object.prototype.hasOwnProperty.call(e, "date") ? e : void 0;
                }
                function M(e, t) {
                    return S(t.inputFormat, {
                        date: e
                    }, t);
                }
                function O(e, t) {
                    var i, a, n = 0, r = 0;
                    for (P(t).lastIndex = 0; a = P(t).exec(t.inputFormat); ) {
                        var o = new RegExp("\\d+$").exec(a[0]);
                        if ((n += r = o ? parseInt(o[0]) : a[0].length) >= e + 1) {
                            i = a, a = P(t).exec(t.inputFormat);
                            break;
                        }
                    }
                    return {
                        targetMatchIndex: n - r,
                        nextMatch: a,
                        targetMatch: i
                    };
                }
                a.default.extendAliases({
                    datetime: {
                        mask: function(e) {
                            return e.numericInput = !1, g.S = e.i18n.ordinalSuffix.join("|"), e.inputFormat = k[e.inputFormat] || e.inputFormat,
                                e.displayFormat = k[e.displayFormat] || e.displayFormat || e.inputFormat, e.outputFormat = k[e.outputFormat] || e.outputFormat || e.inputFormat,
                                e.placeholder = "" !== e.placeholder ? e.placeholder : e.inputFormat.replace(/[[\]]/, ""),
                                e.regex = S(e.inputFormat, void 0, e), e.min = w(e.min, e.inputFormat, e), e.max = w(e.max, e.inputFormat, e),
                                null;
                        },
                        placeholder: "",
                        inputFormat: "isoDateTime",
                        displayFormat: null,
                        outputFormat: null,
                        min: null,
                        max: null,
                        skipOptionalPartCharacter: "",
                        i18n: {
                            dayNames: [ "Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday", "Sunday" ],
                            monthNames: [ "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec", "January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December" ],
                            ordinalSuffix: [ "st", "nd", "rd", "th" ]
                        },
                        preValidation: function(e, t, i, a, n, r, o, l) {
                            if (l) return !0;
                            if (isNaN(i) && e[t] !== i) {
                                var s = O(t, n);
                                if (s.nextMatch && s.nextMatch[0] === i && s.targetMatch[0].length > 1) {
                                    var u = g[s.targetMatch[0]][0];
                                    if (new RegExp(u).test("0" + e[t - 1])) return e[t] = e[t - 1], e[t - 1] = "0",
                                        {
                                            fuzzy: !0,
                                            buffer: e,
                                            refreshFromBuffer: {
                                                start: t - 1,
                                                end: t + 1
                                            },
                                            pos: t + 1
                                        };
                                }
                            }
                            return !0;
                        },
                        postValidation: function(e, t, i, a, n, r, o, s) {
                            var u, c;
                            if (o) return !0;
                            if (!1 === a && (((u = O(t + 1, n)).targetMatch && u.targetMatchIndex === t && u.targetMatch[0].length > 1 && void 0 !== g[u.targetMatch[0]] || (u = O(t + 2, n)).targetMatch && u.targetMatchIndex === t + 1 && u.targetMatch[0].length > 1 && void 0 !== g[u.targetMatch[0]]) && (c = g[u.targetMatch[0]][0]),
                            void 0 !== c && (void 0 !== r.validPositions[t + 1] && new RegExp(c).test(i + "0") ? (e[t] = i,
                                e[t + 1] = "0", a = {
                                pos: t + 2,
                                caret: t
                            }) : new RegExp(c).test("0" + i) && (e[t] = "0", e[t + 1] = i, a = {
                                pos: t + 2
                            })), !1 === a)) return a;
                            if (a.fuzzy && (e = a.buffer, t = a.pos), (u = O(t, n)).targetMatch && u.targetMatch[0] && void 0 !== g[u.targetMatch[0]]) {
                                var f = g[u.targetMatch[0]];
                                c = f[0];
                                var d = e.slice(u.targetMatchIndex, u.targetMatchIndex + u.targetMatch[0].length);
                                if (!1 === new RegExp(c).test(d.join("")) && 2 === u.targetMatch[0].length && r.validPositions[u.targetMatchIndex] && r.validPositions[u.targetMatchIndex + 1] && (r.validPositions[u.targetMatchIndex + 1].input = "0"),
                                "year" == f[2]) for (var p = l.getMaskTemplate.call(this, !1, 1, void 0, !0), h = t + 1; h < e.length; h++) e[h] = p[h],
                                    delete r.validPositions[h];
                            }
                            var m = a, k = w(e.join(""), n.inputFormat, n);
                            return m && k.date.getTime() == k.date.getTime() && (n.prefillYear && (m = function(e, t, i) {
                                if (e.year !== e.rawyear) {
                                    var a = v.toString(), n = e.rawyear.replace(/[^0-9]/g, ""), r = a.slice(0, n.length), o = a.slice(n.length);
                                    if (2 === n.length && n === r) {
                                        var l = new Date(v, e.month - 1, e.day);
                                        e.day == l.getDate() && (!i.max || i.max.date.getTime() >= l.getTime()) && (e.date.setFullYear(v),
                                            e.year = a, t.insert = [ {
                                            pos: t.pos + 1,
                                            c: o[0]
                                        }, {
                                            pos: t.pos + 2,
                                            c: o[1]
                                        } ]);
                                    }
                                }
                                return t;
                            }(k, m, n)), m = function(e, t, i, a, n) {
                                if (!t) return t;
                                if (t && i.min && i.min.date.getTime() == i.min.date.getTime()) {
                                    var r;
                                    for (e.reset(), P(i).lastIndex = 0; r = P(i).exec(i.inputFormat); ) {
                                        var o;
                                        if ((o = x(r)) && o[3]) {
                                            for (var l = o[1], s = e[o[2]], u = i.min[o[2]], c = i.max ? i.max[o[2]] : u, f = [], d = !1, p = 0; p < u.length; p++) void 0 !== a.validPositions[p + r.index] || d ? (f[p] = s[p],
                                                d = d || s[p] > u[p]) : (f[p] = u[p], "year" === o[2] && s.length - 1 == p && u != c && (f = (parseInt(f.join("")) + 1).toString().split("")),
                                            "ampm" === o[2] && u != c && i.min.date.getTime() > e.date.getTime() && (f[p] = c[p]));
                                            l.call(e._date, f.join(""));
                                        }
                                    }
                                    t = i.min.date.getTime() <= e.date.getTime(), e.reInit();
                                }
                                return t && i.max && i.max.date.getTime() == i.max.date.getTime() && (t = i.max.date.getTime() >= e.date.getTime()),
                                    t;
                            }(k, m = E.call(this, k, m, n), n, r)), void 0 !== t && m && a.pos !== t ? {
                                buffer: S(n.inputFormat, k, n).split(""),
                                refreshFromBuffer: {
                                    start: t,
                                    end: a.pos
                                },
                                pos: a.caret || a.pos
                            } : m;
                        },
                        onKeyDown: function(e, t, i, a) {
                            e.ctrlKey && e.keyCode === n.default.RIGHT && (this.inputmask._valueSet(M(new Date, a)),
                                p(this).trigger("setvalue"));
                        },
                        onUnMask: function(e, t, i) {
                            return t ? S(i.outputFormat, w(e, i.inputFormat, i), i, !0) : t;
                        },
                        casing: function(e, t, i, a) {
                            return 0 == t.nativeDef.indexOf("[ap]") ? e.toLowerCase() : 0 == t.nativeDef.indexOf("[AP]") ? e.toUpperCase() : e;
                        },
                        onBeforeMask: function(e, t) {
                            return "[object Date]" === Object.prototype.toString.call(e) && (e = M(e, t)), e;
                        },
                        insertMode: !1,
                        shiftPositions: !1,
                        keepStatic: !1,
                        inputmode: "numeric",
                        prefillYear: !0
                    }
                });
            },
            3851: function(e, t, i) {
                var a, n = (a = i(2394)) && a.__esModule ? a : {
                    default: a
                }, r = i(8711), o = i(4713);
                n.default.extendDefinitions({
                    A: {
                        validator: "[A-Za-z\u0410-\u044f\u0401\u0451\xc0-\xff\xb5]",
                        casing: "upper"
                    },
                    "&": {
                        validator: "[0-9A-Za-z\u0410-\u044f\u0401\u0451\xc0-\xff\xb5]",
                        casing: "upper"
                    },
                    "#": {
                        validator: "[0-9A-Fa-f]",
                        casing: "upper"
                    }
                });
                var l = new RegExp("25[0-5]|2[0-4][0-9]|[01][0-9][0-9]");
                function s(e, t, i, a, n) {
                    return i - 1 > -1 && "." !== t.buffer[i - 1] ? (e = t.buffer[i - 1] + e, e = i - 2 > -1 && "." !== t.buffer[i - 2] ? t.buffer[i - 2] + e : "0" + e) : e = "00" + e,
                        l.test(e);
                }
                n.default.extendAliases({
                    cssunit: {
                        regex: "[+-]?[0-9]+\\.?([0-9]+)?(px|em|rem|ex|%|in|cm|mm|pt|pc)"
                    },
                    url: {
                        regex: "(https?|ftp)://.*",
                        autoUnmask: !1,
                        keepStatic: !1,
                        tabThrough: !0
                    },
                    ip: {
                        mask: "i{1,3}.j{1,3}.k{1,3}.l{1,3}",
                        definitions: {
                            i: {
                                validator: s
                            },
                            j: {
                                validator: s
                            },
                            k: {
                                validator: s
                            },
                            l: {
                                validator: s
                            }
                        },
                        onUnMask: function(e, t, i) {
                            return e;
                        },
                        inputmode: "decimal",
                        substitutes: {
                            ",": "."
                        }
                    },
                    email: {
                        mask: function(e) {
                            var t = "*{1,64}[.*{1,64}][.*{1,64}][.*{1,63}]@-{1,63}.-{1,63}[.-{1,63}][.-{1,63}]", i = t;
                            if (e.separator) for (var a = 0; a < e.quantifier; a++) i += "[".concat(e.separator).concat(t, "]");
                            return i;
                        },
                        greedy: !1,
                        casing: "lower",
                        separator: null,
                        quantifier: 5,
                        skipOptionalPartCharacter: "",
                        onBeforePaste: function(e, t) {
                            return (e = e.toLowerCase()).replace("mailto:", "");
                        },
                        definitions: {
                            "*": {
                                validator: "[0-9\uff11-\uff19A-Za-z\u0410-\u044f\u0401\u0451\xc0-\xff\xb5!#$%&'*+/=?^_`{|}~-]"
                            },
                            "-": {
                                validator: "[0-9A-Za-z-]"
                            }
                        },
                        onUnMask: function(e, t, i) {
                            return e;
                        },
                        inputmode: "email"
                    },
                    mac: {
                        mask: "##:##:##:##:##:##"
                    },
                    vin: {
                        mask: "V{13}9{4}",
                        definitions: {
                            V: {
                                validator: "[A-HJ-NPR-Za-hj-npr-z\\d]",
                                casing: "upper"
                            }
                        },
                        clearIncomplete: !0,
                        autoUnmask: !0
                    },
                    ssn: {
                        mask: "999-99-9999",
                        postValidation: function(e, t, i, a, n, l, s) {
                            var u = o.getMaskTemplate.call(this, !0, r.getLastValidPosition.call(this), !0, !0);
                            return /^(?!219-09-9999|078-05-1120)(?!666|000|9.{2}).{3}-(?!00).{2}-(?!0{4}).{4}$/.test(u.join(""));
                        }
                    }
                });
            },
            207: function(e, t, i) {
                var a = l(i(2394)), n = l(i(5581)), r = l(i(7184)), o = i(8711);
                function l(e) {
                    return e && e.__esModule ? e : {
                        default: e
                    };
                }
                var s = a.default.dependencyLib;
                function u(e, t) {
                    for (var i = "", n = 0; n < e.length; n++) a.default.prototype.definitions[e.charAt(n)] || t.definitions[e.charAt(n)] || t.optionalmarker[0] === e.charAt(n) || t.optionalmarker[1] === e.charAt(n) || t.quantifiermarker[0] === e.charAt(n) || t.quantifiermarker[1] === e.charAt(n) || t.groupmarker[0] === e.charAt(n) || t.groupmarker[1] === e.charAt(n) || t.alternatormarker === e.charAt(n) ? i += "\\" + e.charAt(n) : i += e.charAt(n);
                    return i;
                }
                function c(e, t, i, a) {
                    if (e.length > 0 && t > 0 && (!i.digitsOptional || a)) {
                        var n = e.indexOf(i.radixPoint), r = !1;
                        i.negationSymbol.back === e[e.length - 1] && (r = !0, e.length--), -1 === n && (e.push(i.radixPoint),
                            n = e.length - 1);
                        for (var o = 1; o <= t; o++) isFinite(e[n + o]) || (e[n + o] = "0");
                    }
                    return r && e.push(i.negationSymbol.back), e;
                }
                function f(e, t) {
                    var i = 0;
                    if ("+" === e) {
                        for (i in t.validPositions) ;
                        i = o.seekNext.call(this, parseInt(i));
                    }
                    for (var a in t.tests) if ((a = parseInt(a)) >= i) for (var n = 0, r = t.tests[a].length; n < r; n++) if ((void 0 === t.validPositions[a] || "-" === e) && t.tests[a][n].match.def === e) return a + (void 0 !== t.validPositions[a] && "-" !== e ? 1 : 0);
                    return i;
                }
                function d(e, t) {
                    var i = -1;
                    for (var a in t.validPositions) {
                        var n = t.validPositions[a];
                        if (n && n.match.def === e) {
                            i = parseInt(a);
                            break;
                        }
                    }
                    return i;
                }
                function p(e, t, i, a, n) {
                    var r = t.buffer ? t.buffer.indexOf(n.radixPoint) : -1, o = (-1 !== r || a && n.jitMasking) && new RegExp(n.definitions[9].validator).test(e);
                    return n._radixDance && -1 !== r && o && null == t.validPositions[r] ? {
                        insert: {
                            pos: r === i ? r + 1 : r,
                            c: n.radixPoint
                        },
                        pos: i
                    } : o;
                }
                a.default.extendAliases({
                    numeric: {
                        mask: function(e) {
                            e.repeat = 0, e.groupSeparator === e.radixPoint && e.digits && "0" !== e.digits && ("." === e.radixPoint ? e.groupSeparator = "," : "," === e.radixPoint ? e.groupSeparator = "." : e.groupSeparator = ""),
                            " " === e.groupSeparator && (e.skipOptionalPartCharacter = void 0), e.placeholder.length > 1 && (e.placeholder = e.placeholder.charAt(0)),
                            "radixFocus" === e.positionCaretOnClick && "" === e.placeholder && (e.positionCaretOnClick = "lvp");
                            var t = "0", i = e.radixPoint;
                            !0 === e.numericInput && void 0 === e.__financeInput ? (t = "1", e.positionCaretOnClick = "radixFocus" === e.positionCaretOnClick ? "lvp" : e.positionCaretOnClick,
                                e.digitsOptional = !1, isNaN(e.digits) && (e.digits = 2), e._radixDance = !1, i = "," === e.radixPoint ? "?" : "!",
                            "" !== e.radixPoint && void 0 === e.definitions[i] && (e.definitions[i] = {}, e.definitions[i].validator = "[" + e.radixPoint + "]",
                                e.definitions[i].placeholder = e.radixPoint, e.definitions[i].static = !0, e.definitions[i].generated = !0)) : (e.__financeInput = !1,
                                e.numericInput = !0);
                            var a, n = "[+]";
                            if (n += u(e.prefix, e), "" !== e.groupSeparator ? (void 0 === e.definitions[e.groupSeparator] && (e.definitions[e.groupSeparator] = {},
                                e.definitions[e.groupSeparator].validator = "[" + e.groupSeparator + "]", e.definitions[e.groupSeparator].placeholder = e.groupSeparator,
                                e.definitions[e.groupSeparator].static = !0, e.definitions[e.groupSeparator].generated = !0),
                                n += e._mask(e)) : n += "9{+}", void 0 !== e.digits && 0 !== e.digits) {
                                var o = e.digits.toString().split(",");
                                isFinite(o[0]) && o[1] && isFinite(o[1]) ? n += i + t + "{" + e.digits + "}" : (isNaN(e.digits) || parseInt(e.digits) > 0) && (e.digitsOptional || e.jitMasking ? (a = n + i + t + "{0," + e.digits + "}",
                                    e.keepStatic = !0) : n += i + t + "{" + e.digits + "}");
                            } else e.inputmode = "numeric";
                            return n += u(e.suffix, e), n += "[-]", a && (n = [ a + u(e.suffix, e) + "[-]", n ]),
                                e.greedy = !1, function(e) {
                                void 0 === e.parseMinMaxOptions && (null !== e.min && (e.min = e.min.toString().replace(new RegExp((0,
                                    r.default)(e.groupSeparator), "g"), ""), "," === e.radixPoint && (e.min = e.min.replace(e.radixPoint, ".")),
                                    e.min = isFinite(e.min) ? parseFloat(e.min) : NaN, isNaN(e.min) && (e.min = Number.MIN_VALUE)),
                                null !== e.max && (e.max = e.max.toString().replace(new RegExp((0, r.default)(e.groupSeparator), "g"), ""),
                                "," === e.radixPoint && (e.max = e.max.replace(e.radixPoint, ".")), e.max = isFinite(e.max) ? parseFloat(e.max) : NaN,
                                isNaN(e.max) && (e.max = Number.MAX_VALUE)), e.parseMinMaxOptions = "done");
                            }(e), "" !== e.radixPoint && (e.substitutes["." == e.radixPoint ? "," : "."] = e.radixPoint),
                                n;
                        },
                        _mask: function(e) {
                            return "(" + e.groupSeparator + "999){+|1}";
                        },
                        digits: "*",
                        digitsOptional: !0,
                        enforceDigitsOnBlur: !1,
                        radixPoint: ".",
                        positionCaretOnClick: "radixFocus",
                        _radixDance: !0,
                        groupSeparator: "",
                        allowMinus: !0,
                        negationSymbol: {
                            front: "-",
                            back: ""
                        },
                        prefix: "",
                        suffix: "",
                        min: null,
                        max: null,
                        SetMaxOnOverflow: !1,
                        step: 1,
                        inputType: "text",
                        unmaskAsNumber: !1,
                        roundingFN: Math.round,
                        inputmode: "decimal",
                        shortcuts: {
                            k: "1000",
                            m: "1000000"
                        },
                        placeholder: "0",
                        greedy: !1,
                        rightAlign: !0,
                        insertMode: !0,
                        autoUnmask: !1,
                        skipOptionalPartCharacter: "",
                        usePrototypeDefinitions: !1,
                        stripLeadingZeroes: !0,
                        definitions: {
                            0: {
                                validator: p
                            },
                            1: {
                                validator: p,
                                definitionSymbol: "9"
                            },
                            9: {
                                validator: "[0-9\uff10-\uff19\u0660-\u0669\u06f0-\u06f9]",
                                definitionSymbol: "*"
                            },
                            "+": {
                                validator: function(e, t, i, a, n) {
                                    return n.allowMinus && ("-" === e || e === n.negationSymbol.front);
                                }
                            },
                            "-": {
                                validator: function(e, t, i, a, n) {
                                    return n.allowMinus && e === n.negationSymbol.back;
                                }
                            }
                        },
                        preValidation: function(e, t, i, a, n, r, o, l) {
                            if (!1 !== n.__financeInput && i === n.radixPoint) return !1;
                            var s = e.indexOf(n.radixPoint), u = t;
                            if (t = function(e, t, i, a, n) {
                                return n._radixDance && n.numericInput && t !== n.negationSymbol.back && e <= i && (i > 0 || t == n.radixPoint) && (void 0 === a.validPositions[e - 1] || a.validPositions[e - 1].input !== n.negationSymbol.back) && (e -= 1),
                                    e;
                            }(t, i, s, r, n), "-" === i || i === n.negationSymbol.front) {
                                if (!0 !== n.allowMinus) return !1;
                                var c = !1, p = d("+", r), h = d("-", r);
                                return -1 !== p && (c = [ p, h ]), !1 !== c ? {
                                    remove: c,
                                    caret: u - n.negationSymbol.back.length
                                } : {
                                    insert: [ {
                                        pos: f.call(this, "+", r),
                                        c: n.negationSymbol.front,
                                        fromIsValid: !0
                                    }, {
                                        pos: f.call(this, "-", r),
                                        c: n.negationSymbol.back,
                                        fromIsValid: void 0
                                    } ],
                                    caret: u + n.negationSymbol.back.length
                                };
                            }
                            if (i === n.groupSeparator) return {
                                caret: u
                            };
                            if (l) return !0;
                            if (-1 !== s && !0 === n._radixDance && !1 === a && i === n.radixPoint && void 0 !== n.digits && (isNaN(n.digits) || parseInt(n.digits) > 0) && s !== t) return {
                                caret: n._radixDance && t === s - 1 ? s + 1 : s
                            };
                            if (!1 === n.__financeInput) if (a) {
                                if (n.digitsOptional) return {
                                    rewritePosition: o.end
                                };
                                if (!n.digitsOptional) {
                                    if (o.begin > s && o.end <= s) return i === n.radixPoint ? {
                                        insert: {
                                            pos: s + 1,
                                            c: "0",
                                            fromIsValid: !0
                                        },
                                        rewritePosition: s
                                    } : {
                                        rewritePosition: s + 1
                                    };
                                    if (o.begin < s) return {
                                        rewritePosition: o.begin - 1
                                    };
                                }
                            } else if (!n.showMaskOnHover && !n.showMaskOnFocus && !n.digitsOptional && n.digits > 0 && "" === this.__valueGet.call(this.el)) return {
                                rewritePosition: s
                            };
                            return {
                                rewritePosition: t
                            };
                        },
                        postValidation: function(e, t, i, a, n, r, o) {
                            if (!1 === a) return a;
                            if (o) return !0;
                            if (null !== n.min || null !== n.max) {
                                var l = n.onUnMask(e.slice().reverse().join(""), void 0, s.extend({}, n, {
                                    unmaskAsNumber: !0
                                }));
                                if (null !== n.min && l < n.min && (l.toString().length > n.min.toString().length || l < 0)) return !1;
                                if (null !== n.max && l > n.max) return !!n.SetMaxOnOverflow && {
                                    refreshFromBuffer: !0,
                                    buffer: c(n.max.toString().replace(".", n.radixPoint).split(""), n.digits, n).reverse()
                                };
                            }
                            return a;
                        },
                        onUnMask: function(e, t, i) {
                            if ("" === t && !0 === i.nullable) return t;
                            var a = e.replace(i.prefix, "");
                            return a = (a = a.replace(i.suffix, "")).replace(new RegExp((0, r.default)(i.groupSeparator), "g"), ""),
                            "" !== i.placeholder.charAt(0) && (a = a.replace(new RegExp(i.placeholder.charAt(0), "g"), "0")),
                                i.unmaskAsNumber ? ("" !== i.radixPoint && -1 !== a.indexOf(i.radixPoint) && (a = a.replace(r.default.call(this, i.radixPoint), ".")),
                                    a = (a = a.replace(new RegExp("^" + (0, r.default)(i.negationSymbol.front)), "-")).replace(new RegExp((0,
                                        r.default)(i.negationSymbol.back) + "$"), ""), Number(a)) : a;
                        },
                        isComplete: function(e, t) {
                            var i = (t.numericInput ? e.slice().reverse() : e).join("");
                            return i = (i = (i = (i = (i = i.replace(new RegExp("^" + (0, r.default)(t.negationSymbol.front)), "-")).replace(new RegExp((0,
                                r.default)(t.negationSymbol.back) + "$"), "")).replace(t.prefix, "")).replace(t.suffix, "")).replace(new RegExp((0,
                                r.default)(t.groupSeparator) + "([0-9]{3})", "g"), "$1"), "," === t.radixPoint && (i = i.replace((0,
                                r.default)(t.radixPoint), ".")), isFinite(i);
                        },
                        onBeforeMask: function(e, t) {
                            var i = t.radixPoint || ",";
                            isFinite(t.digits) && (t.digits = parseInt(t.digits)), "number" != typeof e && "number" !== t.inputType || "" === i || (e = e.toString().replace(".", i));
                            var a = "-" === e.charAt(0) || e.charAt(0) === t.negationSymbol.front, n = e.split(i), o = n[0].replace(/[^\-0-9]/g, ""), l = n.length > 1 ? n[1].replace(/[^0-9]/g, "") : "", s = n.length > 1;
                            e = o + ("" !== l ? i + l : l);
                            var u = 0;
                            if ("" !== i && (u = t.digitsOptional ? t.digits < l.length ? t.digits : l.length : t.digits,
                            "" !== l || !t.digitsOptional)) {
                                var f = Math.pow(10, u || 1);
                                e = e.replace((0, r.default)(i), "."), isNaN(parseFloat(e)) || (e = (t.roundingFN(parseFloat(e) * f) / f).toFixed(u)),
                                    e = e.toString().replace(".", i);
                            }
                            if (0 === t.digits && -1 !== e.indexOf(i) && (e = e.substring(0, e.indexOf(i))),
                            null !== t.min || null !== t.max) {
                                var d = e.toString().replace(i, ".");
                                null !== t.min && d < t.min ? e = t.min.toString().replace(".", i) : null !== t.max && d > t.max && (e = t.max.toString().replace(".", i));
                            }
                            return a && "-" !== e.charAt(0) && (e = "-" + e), c(e.toString().split(""), u, t, s).join("");
                        },
                        onBeforeWrite: function(e, t, i, a) {
                            function n(e, t) {
                                if (!1 !== a.__financeInput || t) {
                                    var i = e.indexOf(a.radixPoint);
                                    -1 !== i && e.splice(i, 1);
                                }
                                if ("" !== a.groupSeparator) for (;-1 !== (i = e.indexOf(a.groupSeparator)); ) e.splice(i, 1);
                                return e;
                            }
                            var o, l;
                            if (a.stripLeadingZeroes && (l = function(e, t) {
                                var i = new RegExp("(^" + ("" !== t.negationSymbol.front ? (0, r.default)(t.negationSymbol.front) + "?" : "") + (0,
                                    r.default)(t.prefix) + ")(.*)(" + (0, r.default)(t.suffix) + ("" != t.negationSymbol.back ? (0,
                                    r.default)(t.negationSymbol.back) + "?" : "") + "$)").exec(e.slice().reverse().join("")), a = i ? i[2] : "", n = !1;
                                return a && (a = a.split(t.radixPoint.charAt(0))[0], n = new RegExp("^[0" + t.groupSeparator + "]*").exec(a)),
                                !(!n || !(n[0].length > 1 || n[0].length > 0 && n[0].length < a.length)) && n;
                            }(t, a))) for (var u = t.join("").lastIndexOf(l[0].split("").reverse().join("")) - (l[0] == l.input ? 0 : 1), f = l[0] == l.input ? 1 : 0, d = l[0].length - f; d > 0; d--) delete this.maskset.validPositions[u + d],
                                delete t[u + d];
                            if (e) switch (e.type) {
                                case "blur":
                                case "checkval":
                                    if (null !== a.min) {
                                        var p = a.onUnMask(t.slice().reverse().join(""), void 0, s.extend({}, a, {
                                            unmaskAsNumber: !0
                                        }));
                                        if (null !== a.min && p < a.min) return {
                                            refreshFromBuffer: !0,
                                            buffer: c(a.min.toString().replace(".", a.radixPoint).split(""), a.digits, a).reverse()
                                        };
                                    }
                                    if (t[t.length - 1] === a.negationSymbol.front) {
                                        var h = new RegExp("(^" + ("" != a.negationSymbol.front ? (0, r.default)(a.negationSymbol.front) + "?" : "") + (0,
                                            r.default)(a.prefix) + ")(.*)(" + (0, r.default)(a.suffix) + ("" != a.negationSymbol.back ? (0,
                                            r.default)(a.negationSymbol.back) + "?" : "") + "$)").exec(n(t.slice(), !0).reverse().join(""));
                                        0 == (h ? h[2] : "") && (o = {
                                            refreshFromBuffer: !0,
                                            buffer: [ 0 ]
                                        });
                                    } else if ("" !== a.radixPoint) {
                                        t.indexOf(a.radixPoint) === a.suffix.length && (o && o.buffer ? o.buffer.splice(0, 1 + a.suffix.length) : (t.splice(0, 1 + a.suffix.length),
                                            o = {
                                                refreshFromBuffer: !0,
                                                buffer: n(t)
                                            }));
                                    }
                                    if (a.enforceDigitsOnBlur) {
                                        var v = (o = o || {}) && o.buffer || t.slice().reverse();
                                        o.refreshFromBuffer = !0, o.buffer = c(v, a.digits, a, !0).reverse();
                                    }
                            }
                            return o;
                        },
                        onKeyDown: function(e, t, i, a) {
                            var r, o, l = s(this), u = String.fromCharCode(e.keyCode).toLowerCase();
                            if ((o = a.shortcuts && a.shortcuts[u]) && o.length > 1) return this.inputmask.__valueSet.call(this, parseFloat(this.inputmask.unmaskedvalue()) * parseInt(o)),
                                l.trigger("setvalue"), !1;
                            if (e.ctrlKey) switch (e.keyCode) {
                                case n.default.UP:
                                    return this.inputmask.__valueSet.call(this, parseFloat(this.inputmask.unmaskedvalue()) + parseInt(a.step)),
                                        l.trigger("setvalue"), !1;

                                case n.default.DOWN:
                                    return this.inputmask.__valueSet.call(this, parseFloat(this.inputmask.unmaskedvalue()) - parseInt(a.step)),
                                        l.trigger("setvalue"), !1;
                            }
                            if (!e.shiftKey && (e.keyCode === n.default.DELETE || e.keyCode === n.default.BACKSPACE || e.keyCode === n.default.BACKSPACE_SAFARI) && i.begin !== t.length) {
                                if (t[e.keyCode === n.default.DELETE ? i.begin - 1 : i.end] === a.negationSymbol.front) return r = t.slice().reverse(),
                                "" !== a.negationSymbol.front && r.shift(), "" !== a.negationSymbol.back && r.pop(),
                                    l.trigger("setvalue", [ r.join(""), i.begin ]), !1;
                                if (!0 === a._radixDance) {
                                    var f = t.indexOf(a.radixPoint);
                                    if (a.digitsOptional) {
                                        if (0 === f) return (r = t.slice().reverse()).pop(), l.trigger("setvalue", [ r.join(""), i.begin >= r.length ? r.length : i.begin ]),
                                            !1;
                                    } else if (-1 !== f && (i.begin < f || i.end < f || e.keyCode === n.default.DELETE && i.begin === f)) return i.begin !== i.end || e.keyCode !== n.default.BACKSPACE && e.keyCode !== n.default.BACKSPACE_SAFARI || i.begin++,
                                        (r = t.slice().reverse()).splice(r.length - i.begin, i.begin - i.end + 1), r = c(r, a.digits, a).join(""),
                                        l.trigger("setvalue", [ r, i.begin >= r.length ? f + 1 : i.begin ]), !1;
                                }
                            }
                        }
                    },
                    currency: {
                        prefix: "",
                        groupSeparator: ",",
                        alias: "numeric",
                        digits: 2,
                        digitsOptional: !1
                    },
                    decimal: {
                        alias: "numeric"
                    },
                    integer: {
                        alias: "numeric",
                        inputmode: "numeric",
                        digits: 0
                    },
                    percentage: {
                        alias: "numeric",
                        min: 0,
                        max: 100,
                        suffix: " %",
                        digits: 0,
                        allowMinus: !1
                    },
                    indianns: {
                        alias: "numeric",
                        _mask: function(e) {
                            return "(" + e.groupSeparator + "99){*|1}(" + e.groupSeparator + "999){1|1}";
                        },
                        groupSeparator: ",",
                        radixPoint: ".",
                        placeholder: "0",
                        digits: 2,
                        digitsOptional: !1
                    }
                });
            },
            9380: function(e, t, i) {
                var a;
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.default = void 0;
                var n = ((a = i(8741)) && a.__esModule ? a : {
                    default: a
                }).default ? window : {};
                t.default = n;
            },
            7760: function(e, t, i) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.HandleNativePlaceholder = function(e, t) {
                    var i = e ? e.inputmask : this;
                    if (s.ie) {
                        if (e.inputmask._valueGet() !== t && (e.placeholder !== t || "" === e.placeholder)) {
                            var a = o.getBuffer.call(i).slice(), n = e.inputmask._valueGet();
                            if (n !== t) {
                                var r = o.getLastValidPosition.call(i);
                                -1 === r && n === o.getBufferTemplate.call(i).join("") ? a = [] : -1 !== r && f.call(i, a),
                                    p(e, a);
                            }
                        }
                    } else e.placeholder !== t && (e.placeholder = t, "" === e.placeholder && e.removeAttribute("placeholder"));
                }, t.applyInputValue = c, t.checkVal = d, t.clearOptionalTail = f, t.unmaskedvalue = function(e) {
                    var t = e ? e.inputmask : this, i = t.opts, a = t.maskset;
                    if (e) {
                        if (void 0 === e.inputmask) return e.value;
                        e.inputmask && e.inputmask.refreshValue && c(e, e.inputmask._valueGet(!0));
                    }
                    var n = [], r = a.validPositions;
                    for (var l in r) r[l] && r[l].match && (1 != r[l].match.static || Array.isArray(a.metadata) && !0 !== r[l].generatedInput) && n.push(r[l].input);
                    var s = 0 === n.length ? "" : (t.isRTL ? n.reverse() : n).join("");
                    if ("function" == typeof i.onUnMask) {
                        var u = (t.isRTL ? o.getBuffer.call(t).slice().reverse() : o.getBuffer.call(t)).join("");
                        s = i.onUnMask.call(t, u, s, i);
                    }
                    return s;
                }, t.writeBuffer = p;
                var a, n = (a = i(5581)) && a.__esModule ? a : {
                    default: a
                }, r = i(4713), o = i(8711), l = i(7215), s = i(9845), u = i(6030);
                function c(e, t) {
                    var i = e ? e.inputmask : this, a = i.opts;
                    e.inputmask.refreshValue = !1, "function" == typeof a.onBeforeMask && (t = a.onBeforeMask.call(i, t, a) || t),
                        d(e, !0, !1, t = t.toString().split("")), i.undoValue = i._valueGet(!0), (a.clearMaskOnLostFocus || a.clearIncomplete) && e.inputmask._valueGet() === o.getBufferTemplate.call(i).join("") && -1 === o.getLastValidPosition.call(i) && e.inputmask._valueSet("");
                }
                function f(e) {
                    e.length = 0;
                    for (var t, i = r.getMaskTemplate.call(this, !0, 0, !0, void 0, !0); void 0 !== (t = i.shift()); ) e.push(t);
                    return e;
                }
                function d(e, t, i, a, n) {
                    var s = e ? e.inputmask : this, c = s.maskset, f = s.opts, d = s.dependencyLib, h = a.slice(), v = "", m = -1, g = void 0, k = f.skipOptionalPartCharacter;
                    f.skipOptionalPartCharacter = "", o.resetMaskSet.call(s), c.tests = {}, m = f.radixPoint ? o.determineNewCaretPosition.call(s, {
                        begin: 0,
                        end: 0
                    }, !1, !1 === f.__financeInput ? "radixFocus" : void 0).begin : 0, c.p = m, s.caretPos = {
                        begin: m
                    };
                    var y = [], b = s.caretPos;
                    if (h.forEach((function(e, t) {
                        if (void 0 !== e) {
                            var a = new d.Event("_checkval");
                            a.keyCode = e.toString().charCodeAt(0), v += e;
                            var n = o.getLastValidPosition.call(s, void 0, !0);
                            !function(e, t) {
                                for (var i = r.getMaskTemplate.call(s, !0, 0).slice(e, o.seekNext.call(s, e, !1, !1)).join("").replace(/'/g, ""), a = i.indexOf(t); a > 0 && " " === i[a - 1]; ) a--;
                                var n = 0 === a && !o.isMask.call(s, e) && (r.getTest.call(s, e).match.nativeDef === t.charAt(0) || !0 === r.getTest.call(s, e).match.static && r.getTest.call(s, e).match.nativeDef === "'" + t.charAt(0) || " " === r.getTest.call(s, e).match.nativeDef && (r.getTest.call(s, e + 1).match.nativeDef === t.charAt(0) || !0 === r.getTest.call(s, e + 1).match.static && r.getTest.call(s, e + 1).match.nativeDef === "'" + t.charAt(0)));
                                if (!n && a > 0 && !o.isMask.call(s, e, !1, !0)) {
                                    var l = o.seekNext.call(s, e);
                                    s.caretPos.begin < l && (s.caretPos = {
                                        begin: l
                                    });
                                }
                                return n;
                            }(m, v) ? (g = u.EventHandlers.keypressEvent.call(s, a, !0, !1, i, s.caretPos.begin)) && (m = s.caretPos.begin + 1,
                                v = "") : g = u.EventHandlers.keypressEvent.call(s, a, !0, !1, i, n + 1), g ? (void 0 !== g.pos && c.validPositions[g.pos] && !0 === c.validPositions[g.pos].match.static && void 0 === c.validPositions[g.pos].alternation && (y.push(g.pos),
                            s.isRTL || (g.forwardPosition = g.pos + 1)), p.call(s, void 0, o.getBuffer.call(s), g.forwardPosition, a, !1),
                                s.caretPos = {
                                    begin: g.forwardPosition,
                                    end: g.forwardPosition
                                }, b = s.caretPos) : void 0 === c.validPositions[t] && h[t] === r.getPlaceholder.call(s, t) && o.isMask.call(s, t, !0) ? s.caretPos.begin++ : s.caretPos = b;
                        }
                    })), y.length > 0) {
                        var x, P, E = o.seekNext.call(s, -1, void 0, !1);
                        if (!l.isComplete.call(s, o.getBuffer.call(s)) && y.length <= E || l.isComplete.call(s, o.getBuffer.call(s)) && y.length > 0 && y.length !== E && 0 === y[0]) for (var S = E; void 0 !== (x = y.shift()); ) {
                            var _ = new d.Event("_checkval");
                            if ((P = c.validPositions[x]).generatedInput = !0, _.keyCode = P.input.charCodeAt(0),
                            (g = u.EventHandlers.keypressEvent.call(s, _, !0, !1, i, S)) && void 0 !== g.pos && g.pos !== x && c.validPositions[g.pos] && !0 === c.validPositions[g.pos].match.static) y.push(g.pos); else if (!g) break;
                            S++;
                        }
                    }
                    t && p.call(s, e, o.getBuffer.call(s), g ? g.forwardPosition : s.caretPos.begin, n || new d.Event("checkval"), n && ("input" === n.type && s.undoValue !== o.getBuffer.call(s).join("") || "paste" === n.type)),
                        f.skipOptionalPartCharacter = k;
                }
                function p(e, t, i, a, r) {
                    var s = e ? e.inputmask : this, u = s.opts, c = s.dependencyLib;
                    if (a && "function" == typeof u.onBeforeWrite) {
                        var f = u.onBeforeWrite.call(s, a, t, i, u);
                        if (f) {
                            if (f.refreshFromBuffer) {
                                var d = f.refreshFromBuffer;
                                l.refreshFromBuffer.call(s, !0 === d ? d : d.start, d.end, f.buffer || t), t = o.getBuffer.call(s, !0);
                            }
                            void 0 !== i && (i = void 0 !== f.caret ? f.caret : i);
                        }
                    }
                    if (void 0 !== e && (e.inputmask._valueSet(t.join("")), void 0 === i || void 0 !== a && "blur" === a.type || o.caret.call(s, e, i, void 0, void 0, void 0 !== a && "keydown" === a.type && (a.keyCode === n.default.DELETE || a.keyCode === n.default.BACKSPACE)),
                    !0 === r)) {
                        var p = c(e), h = e.inputmask._valueGet();
                        e.inputmask.skipInputEvent = !0, p.trigger("input"), setTimeout((function() {
                            h === o.getBufferTemplate.call(s).join("") ? p.trigger("cleared") : !0 === l.isComplete.call(s, t) && p.trigger("complete");
                        }), 0);
                    }
                }
            },
            2394: function(e, t, i) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.default = void 0, i(7149), i(3194);
                var a = i(157), n = m(i(4963)), r = m(i(9380)), o = i(2391), l = i(4713), s = i(8711), u = i(7215), c = i(7760), f = i(9716), d = m(i(7392)), p = m(i(3976)), h = m(i(8741));
                function v(e) {
                    return v = "function" == typeof Symbol && "symbol" == typeof Symbol.iterator ? function(e) {
                        return typeof e;
                    } : function(e) {
                        return e && "function" == typeof Symbol && e.constructor === Symbol && e !== Symbol.prototype ? "symbol" : typeof e;
                    }, v(e);
                }
                function m(e) {
                    return e && e.__esModule ? e : {
                        default: e
                    };
                }
                var g = r.default.document, k = "_inputmask_opts";
                function y(e, t, i) {
                    if (h.default) {
                        if (!(this instanceof y)) return new y(e, t, i);
                        this.dependencyLib = n.default, this.el = void 0, this.events = {}, this.maskset = void 0,
                        !0 !== i && ("[object Object]" === Object.prototype.toString.call(e) ? t = e : (t = t || {},
                        e && (t.alias = e)), this.opts = n.default.extend(!0, {}, this.defaults, t), this.noMasksCache = t && void 0 !== t.definitions,
                            this.userOptions = t || {}, b(this.opts.alias, t, this.opts)), this.refreshValue = !1,
                            this.undoValue = void 0, this.$el = void 0, this.skipKeyPressEvent = !1, this.skipInputEvent = !1,
                            this.validationEvent = !1, this.ignorable = !1, this.maxLength, this.mouseEnter = !1,
                            this.originalPlaceholder = void 0, this.isComposing = !1;
                    }
                }
                function b(e, t, i) {
                    var a = y.prototype.aliases[e];
                    return a ? (a.alias && b(a.alias, void 0, i), n.default.extend(!0, i, a), n.default.extend(!0, i, t),
                        !0) : (null === i.mask && (i.mask = e), !1);
                }
                y.prototype = {
                    dataAttribute: "data-inputmask",
                    defaults: p.default,
                    definitions: d.default,
                    aliases: {},
                    masksCache: {},
                    get isRTL() {
                        return this.opts.isRTL || this.opts.numericInput;
                    },
                    mask: function(e) {
                        var t = this;
                        return "string" == typeof e && (e = g.getElementById(e) || g.querySelectorAll(e)),
                            (e = e.nodeName ? [ e ] : Array.isArray(e) ? e : Array.from(e)).forEach((function(e, i) {
                                var l = n.default.extend(!0, {}, t.opts);
                                if (function(e, t, i, a) {
                                    function o(t, n) {
                                        var o = "" === a ? t : a + "-" + t;
                                        null !== (n = void 0 !== n ? n : e.getAttribute(o)) && ("string" == typeof n && (0 === t.indexOf("on") ? n = r.default[n] : "false" === n ? n = !1 : "true" === n && (n = !0)),
                                            i[t] = n);
                                    }
                                    if (!0 === t.importDataAttributes) {
                                        var l, s, u, c, f = e.getAttribute(a);
                                        if (f && "" !== f && (f = f.replace(/'/g, '"'), s = JSON.parse("{" + f + "}")),
                                            s) for (c in u = void 0, s) if ("alias" === c.toLowerCase()) {
                                            u = s[c];
                                            break;
                                        }
                                        for (l in o("alias", u), i.alias && b(i.alias, i, t), t) {
                                            if (s) for (c in u = void 0, s) if (c.toLowerCase() === l.toLowerCase()) {
                                                u = s[c];
                                                break;
                                            }
                                            o(l, u);
                                        }
                                    }
                                    n.default.extend(!0, t, i), ("rtl" === e.dir || t.rightAlign) && (e.style.textAlign = "right");
                                    ("rtl" === e.dir || t.numericInput) && (e.dir = "ltr", e.removeAttribute("dir"),
                                        t.isRTL = !0);
                                    return Object.keys(i).length;
                                }(e, l, n.default.extend(!0, {}, t.userOptions), t.dataAttribute)) {
                                    var s = (0, o.generateMaskSet)(l, t.noMasksCache);
                                    void 0 !== s && (void 0 !== e.inputmask && (e.inputmask.opts.autoUnmask = !0, e.inputmask.remove()),
                                        e.inputmask = new y(void 0, void 0, !0), e.inputmask.opts = l, e.inputmask.noMasksCache = t.noMasksCache,
                                        e.inputmask.userOptions = n.default.extend(!0, {}, t.userOptions), e.inputmask.el = e,
                                        e.inputmask.$el = (0, n.default)(e), e.inputmask.maskset = s, n.default.data(e, k, t.userOptions),
                                        a.mask.call(e.inputmask));
                                }
                            })), e && e[0] && e[0].inputmask || this;
                    },
                    option: function(e, t) {
                        return "string" == typeof e ? this.opts[e] : "object" === v(e) ? (n.default.extend(this.userOptions, e),
                        this.el && !0 !== t && this.mask(this.el), this) : void 0;
                    },
                    unmaskedvalue: function(e) {
                        if (this.maskset = this.maskset || (0, o.generateMaskSet)(this.opts, this.noMasksCache),
                        void 0 === this.el || void 0 !== e) {
                            var t = ("function" == typeof this.opts.onBeforeMask && this.opts.onBeforeMask.call(this, e, this.opts) || e).split("");
                            c.checkVal.call(this, void 0, !1, !1, t), "function" == typeof this.opts.onBeforeWrite && this.opts.onBeforeWrite.call(this, void 0, s.getBuffer.call(this), 0, this.opts);
                        }
                        return c.unmaskedvalue.call(this, this.el);
                    },
                    remove: function() {
                        if (this.el) {
                            n.default.data(this.el, k, null);
                            var e = this.opts.autoUnmask ? (0, c.unmaskedvalue)(this.el) : this._valueGet(this.opts.autoUnmask);
                            e !== s.getBufferTemplate.call(this).join("") ? this._valueSet(e, this.opts.autoUnmask) : this._valueSet(""),
                                f.EventRuler.off(this.el), Object.getOwnPropertyDescriptor && Object.getPrototypeOf ? Object.getOwnPropertyDescriptor(Object.getPrototypeOf(this.el), "value") && this.__valueGet && Object.defineProperty(this.el, "value", {
                                get: this.__valueGet,
                                set: this.__valueSet,
                                configurable: !0
                            }) : g.__lookupGetter__ && this.el.__lookupGetter__("value") && this.__valueGet && (this.el.__defineGetter__("value", this.__valueGet),
                                this.el.__defineSetter__("value", this.__valueSet)), this.el.inputmask = void 0;
                        }
                        return this.el;
                    },
                    getemptymask: function() {
                        return this.maskset = this.maskset || (0, o.generateMaskSet)(this.opts, this.noMasksCache),
                            s.getBufferTemplate.call(this).join("");
                    },
                    hasMaskedValue: function() {
                        return !this.opts.autoUnmask;
                    },
                    isComplete: function() {
                        return this.maskset = this.maskset || (0, o.generateMaskSet)(this.opts, this.noMasksCache),
                            u.isComplete.call(this, s.getBuffer.call(this));
                    },
                    getmetadata: function() {
                        if (this.maskset = this.maskset || (0, o.generateMaskSet)(this.opts, this.noMasksCache),
                            Array.isArray(this.maskset.metadata)) {
                            var e = l.getMaskTemplate.call(this, !0, 0, !1).join("");
                            return this.maskset.metadata.forEach((function(t) {
                                return t.mask !== e || (e = t, !1);
                            })), e;
                        }
                        return this.maskset.metadata;
                    },
                    isValid: function(e) {
                        if (this.maskset = this.maskset || (0, o.generateMaskSet)(this.opts, this.noMasksCache),
                            e) {
                            var t = ("function" == typeof this.opts.onBeforeMask && this.opts.onBeforeMask.call(this, e, this.opts) || e).split("");
                            c.checkVal.call(this, void 0, !0, !1, t);
                        } else e = this.isRTL ? s.getBuffer.call(this).slice().reverse().join("") : s.getBuffer.call(this).join("");
                        for (var i = s.getBuffer.call(this), a = s.determineLastRequiredPosition.call(this), n = i.length - 1; n > a && !s.isMask.call(this, n); n--) ;
                        return i.splice(a, n + 1 - a), u.isComplete.call(this, i) && e === (this.isRTL ? s.getBuffer.call(this).slice().reverse().join("") : s.getBuffer.call(this).join(""));
                    },
                    format: function(e, t) {
                        this.maskset = this.maskset || (0, o.generateMaskSet)(this.opts, this.noMasksCache);
                        var i = ("function" == typeof this.opts.onBeforeMask && this.opts.onBeforeMask.call(this, e, this.opts) || e).split("");
                        c.checkVal.call(this, void 0, !0, !1, i);
                        var a = this.isRTL ? s.getBuffer.call(this).slice().reverse().join("") : s.getBuffer.call(this).join("");
                        return t ? {
                            value: a,
                            metadata: this.getmetadata()
                        } : a;
                    },
                    setValue: function(e) {
                        this.el && (0, n.default)(this.el).trigger("setvalue", [ e ]);
                    },
                    analyseMask: o.analyseMask
                }, y.extendDefaults = function(e) {
                    n.default.extend(!0, y.prototype.defaults, e);
                }, y.extendDefinitions = function(e) {
                    n.default.extend(!0, y.prototype.definitions, e);
                }, y.extendAliases = function(e) {
                    n.default.extend(!0, y.prototype.aliases, e);
                }, y.format = function(e, t, i) {
                    return y(t).format(e, i);
                }, y.unmask = function(e, t) {
                    return y(t).unmaskedvalue(e);
                }, y.isValid = function(e, t) {
                    return y(t).isValid(e);
                }, y.remove = function(e) {
                    "string" == typeof e && (e = g.getElementById(e) || g.querySelectorAll(e)), (e = e.nodeName ? [ e ] : e).forEach((function(e) {
                        e.inputmask && e.inputmask.remove();
                    }));
                }, y.setValue = function(e, t) {
                    "string" == typeof e && (e = g.getElementById(e) || g.querySelectorAll(e)), (e = e.nodeName ? [ e ] : e).forEach((function(e) {
                        e.inputmask ? e.inputmask.setValue(t) : (0, n.default)(e).trigger("setvalue", [ t ]);
                    }));
                }, y.dependencyLib = n.default, r.default.Inputmask = y;
                var x = y;
                t.default = x;
            },
            5296: function(e, t, i) {
                function a(e) {
                    return a = "function" == typeof Symbol && "symbol" == typeof Symbol.iterator ? function(e) {
                        return typeof e;
                    } : function(e) {
                        return e && "function" == typeof Symbol && e.constructor === Symbol && e !== Symbol.prototype ? "symbol" : typeof e;
                    }, a(e);
                }
                var n = h(i(9380)), r = h(i(2394)), o = h(i(8741));
                function l(e, t) {
                    for (var i = 0; i < t.length; i++) {
                        var a = t[i];
                        a.enumerable = a.enumerable || !1, a.configurable = !0, "value" in a && (a.writable = !0),
                            Object.defineProperty(e, a.key, a);
                    }
                }
                function s(e, t) {
                    if (t && ("object" === a(t) || "function" == typeof t)) return t;
                    if (void 0 !== t) throw new TypeError("Derived constructors may only return object or undefined");
                    return function(e) {
                        if (void 0 === e) throw new ReferenceError("this hasn't been initialised - super() hasn't been called");
                        return e;
                    }(e);
                }
                function u(e) {
                    var t = "function" == typeof Map ? new Map : void 0;
                    return u = function(e) {
                        if (null === e || (i = e, -1 === Function.toString.call(i).indexOf("[native code]"))) return e;
                        var i;
                        if ("function" != typeof e) throw new TypeError("Super expression must either be null or a function");
                        if (void 0 !== t) {
                            if (t.has(e)) return t.get(e);
                            t.set(e, a);
                        }
                        function a() {
                            return c(e, arguments, p(this).constructor);
                        }
                        return a.prototype = Object.create(e.prototype, {
                            constructor: {
                                value: a,
                                enumerable: !1,
                                writable: !0,
                                configurable: !0
                            }
                        }), d(a, e);
                    }, u(e);
                }
                function c(e, t, i) {
                    return c = f() ? Reflect.construct : function(e, t, i) {
                        var a = [ null ];
                        a.push.apply(a, t);
                        var n = new (Function.bind.apply(e, a));
                        return i && d(n, i.prototype), n;
                    }, c.apply(null, arguments);
                }
                function f() {
                    if ("undefined" == typeof Reflect || !Reflect.construct) return !1;
                    if (Reflect.construct.sham) return !1;
                    if ("function" == typeof Proxy) return !0;
                    try {
                        return Boolean.prototype.valueOf.call(Reflect.construct(Boolean, [], (function() {}))),
                            !0;
                    } catch (e) {
                        return !1;
                    }
                }
                function d(e, t) {
                    return d = Object.setPrototypeOf || function(e, t) {
                        return e.__proto__ = t, e;
                    }, d(e, t);
                }
                function p(e) {
                    return p = Object.setPrototypeOf ? Object.getPrototypeOf : function(e) {
                        return e.__proto__ || Object.getPrototypeOf(e);
                    }, p(e);
                }
                function h(e) {
                    return e && e.__esModule ? e : {
                        default: e
                    };
                }
                var v = n.default.document;
                if (o.default && v && v.head && v.head.attachShadow && n.default.customElements && void 0 === n.default.customElements.get("input-mask")) {
                    var m = function(e) {
                        !function(e, t) {
                            if ("function" != typeof t && null !== t) throw new TypeError("Super expression must either be null or a function");
                            Object.defineProperty(e, "prototype", {
                                value: Object.create(t && t.prototype, {
                                    constructor: {
                                        value: e,
                                        writable: !0,
                                        configurable: !0
                                    }
                                }),
                                writable: !1
                            }), t && d(e, t);
                        }(c, e);
                        var t, i, a, n, o, u = (t = c, i = f(), function() {
                            var e, a = p(t);
                            if (i) {
                                var n = p(this).constructor;
                                e = Reflect.construct(a, arguments, n);
                            } else e = a.apply(this, arguments);
                            return s(this, e);
                        });
                        function c() {
                            var e;
                            !function(e, t) {
                                if (!(e instanceof t)) throw new TypeError("Cannot call a class as a function");
                            }(this, c);
                            var t = (e = u.call(this)).getAttributeNames(), i = e.attachShadow({
                                mode: "closed"
                            }), a = v.createElement("input");
                            for (var n in a.type = "text", i.appendChild(a), t) Object.prototype.hasOwnProperty.call(t, n) && a.setAttribute(t[n], e.getAttribute(t[n]));
                            var o = new r.default;
                            return o.dataAttribute = "", o.mask(a), a.inputmask.shadowRoot = i, e;
                        }
                        return a = c, n && l(a.prototype, n), o && l(a, o), Object.defineProperty(a, "prototype", {
                            writable: !1
                        }), a;
                    }(u(HTMLElement));
                    n.default.customElements.define("input-mask", m);
                }
            },
            2391: function(e, t, i) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.analyseMask = function(e, t, i) {
                    var a, o, l, s, u, c, f = /(?:[?*+]|\{[0-9+*]+(?:,[0-9+*]*)?(?:\|[0-9+*]*)?\})|[^.?*+^${[]()|\\]+|./g, d = /\[\^?]?(?:[^\\\]]+|\\[\S\s]?)*]?|\\(?:0(?:[0-3][0-7]{0,2}|[4-7][0-7]?)?|[1-9][0-9]*|x[0-9A-Fa-f]{2}|u[0-9A-Fa-f]{4}|c[A-Za-z]|[\S\s]?)|\((?:\?[:=!]?)?|(?:[?*+]|\{[0-9]+(?:,[0-9]*)?\})\??|[^.?*+^${[()|\\]+|./g, p = !1, h = new n.default, v = [], m = [], g = !1;
                    function k(e, a, n) {
                        n = void 0 !== n ? n : e.matches.length;
                        var o = e.matches[n - 1];
                        if (t) 0 === a.indexOf("[") || p && /\\d|\\s|\\w/i.test(a) || "." === a ? e.matches.splice(n++, 0, {
                            fn: new RegExp(a, i.casing ? "i" : ""),
                            static: !1,
                            optionality: !1,
                            newBlockMarker: void 0 === o ? "master" : o.def !== a,
                            casing: null,
                            def: a,
                            placeholder: void 0,
                            nativeDef: a
                        }) : (p && (a = a[a.length - 1]), a.split("").forEach((function(t, a) {
                            o = e.matches[n - 1], e.matches.splice(n++, 0, {
                                fn: /[a-z]/i.test(i.staticDefinitionSymbol || t) ? new RegExp("[" + (i.staticDefinitionSymbol || t) + "]", i.casing ? "i" : "") : null,
                                static: !0,
                                optionality: !1,
                                newBlockMarker: void 0 === o ? "master" : o.def !== t && !0 !== o.static,
                                casing: null,
                                def: i.staticDefinitionSymbol || t,
                                placeholder: void 0 !== i.staticDefinitionSymbol ? t : void 0,
                                nativeDef: (p ? "'" : "") + t
                            });
                        }))), p = !1; else {
                            var l = i.definitions && i.definitions[a] || i.usePrototypeDefinitions && r.default.prototype.definitions[a];
                            l && !p ? e.matches.splice(n++, 0, {
                                fn: l.validator ? "string" == typeof l.validator ? new RegExp(l.validator, i.casing ? "i" : "") : new function() {
                                    this.test = l.validator;
                                } : new RegExp("."),
                                static: l.static || !1,
                                optionality: l.optional || !1,
                                newBlockMarker: void 0 === o || l.optional ? "master" : o.def !== (l.definitionSymbol || a),
                                casing: l.casing,
                                def: l.definitionSymbol || a,
                                placeholder: l.placeholder,
                                nativeDef: a,
                                generated: l.generated
                            }) : (e.matches.splice(n++, 0, {
                                fn: /[a-z]/i.test(i.staticDefinitionSymbol || a) ? new RegExp("[" + (i.staticDefinitionSymbol || a) + "]", i.casing ? "i" : "") : null,
                                static: !0,
                                optionality: !1,
                                newBlockMarker: void 0 === o ? "master" : o.def !== a && !0 !== o.static,
                                casing: null,
                                def: i.staticDefinitionSymbol || a,
                                placeholder: void 0 !== i.staticDefinitionSymbol ? a : void 0,
                                nativeDef: (p ? "'" : "") + a
                            }), p = !1);
                        }
                    }
                    function y() {
                        if (v.length > 0) {
                            if (k(s = v[v.length - 1], o), s.isAlternator) {
                                u = v.pop();
                                for (var e = 0; e < u.matches.length; e++) u.matches[e].isGroup && (u.matches[e].isGroup = !1);
                                v.length > 0 ? (s = v[v.length - 1]).matches.push(u) : h.matches.push(u);
                            }
                        } else k(h, o);
                    }
                    function b(e) {
                        var t = new n.default(!0);
                        return t.openGroup = !1, t.matches = e, t;
                    }
                    function x() {
                        if ((l = v.pop()).openGroup = !1, void 0 !== l) if (v.length > 0) {
                            if ((s = v[v.length - 1]).matches.push(l), s.isAlternator) {
                                for (var e = (u = v.pop()).matches[0].matches ? u.matches[0].matches.length : 1, t = 0; t < u.matches.length; t++) u.matches[t].isGroup = !1,
                                    u.matches[t].alternatorGroup = !1, null === i.keepStatic && e < (u.matches[t].matches ? u.matches[t].matches.length : 1) && (i.keepStatic = !0),
                                    e = u.matches[t].matches ? u.matches[t].matches.length : 1;
                                v.length > 0 ? (s = v[v.length - 1]).matches.push(u) : h.matches.push(u);
                            }
                        } else h.matches.push(l); else y();
                    }
                    function P(e) {
                        var t = e.pop();
                        return t.isQuantifier && (t = b([ e.pop(), t ])), t;
                    }
                    t && (i.optionalmarker[0] = void 0, i.optionalmarker[1] = void 0);
                    for (;a = t ? d.exec(e) : f.exec(e); ) {
                        if (o = a[0], t) {
                            switch (o.charAt(0)) {
                                case "?":
                                    o = "{0,1}";
                                    break;

                                case "+":
                                case "*":
                                    o = "{" + o + "}";
                                    break;

                                case "|":
                                    if (0 === v.length) {
                                        var E = b(h.matches);
                                        E.openGroup = !0, v.push(E), h.matches = [], g = !0;
                                    }
                            }
                            if ("\\d" === o) o = "[0-9]";
                        }
                        if (p) y(); else switch (o.charAt(0)) {
                            case "$":
                            case "^":
                                t || y();
                                break;

                            case i.escapeChar:
                                p = !0, t && y();
                                break;

                            case i.optionalmarker[1]:
                            case i.groupmarker[1]:
                                x();
                                break;

                            case i.optionalmarker[0]:
                                v.push(new n.default(!1, !0));
                                break;

                            case i.groupmarker[0]:
                                v.push(new n.default(!0));
                                break;

                            case i.quantifiermarker[0]:
                                var S = new n.default(!1, !1, !0), _ = (o = o.replace(/[{}?]/g, "")).split("|"), w = _[0].split(","), M = isNaN(w[0]) ? w[0] : parseInt(w[0]), O = 1 === w.length ? M : isNaN(w[1]) ? w[1] : parseInt(w[1]), T = isNaN(_[1]) ? _[1] : parseInt(_[1]);
                                "*" !== M && "+" !== M || (M = "*" === O ? 0 : 1), S.quantifier = {
                                    min: M,
                                    max: O,
                                    jit: T
                                };
                                var C = v.length > 0 ? v[v.length - 1].matches : h.matches;
                                if ((a = C.pop()).isAlternator) {
                                    C.push(a), C = a.matches;
                                    var A = new n.default(!0), D = C.pop();
                                    C.push(A), C = A.matches, a = D;
                                }
                                a.isGroup || (a = b([ a ])), C.push(a), C.push(S);
                                break;

                            case i.alternatormarker:
                                if (v.length > 0) {
                                    var j = (s = v[v.length - 1]).matches[s.matches.length - 1];
                                    c = s.openGroup && (void 0 === j.matches || !1 === j.isGroup && !1 === j.isAlternator) ? v.pop() : P(s.matches);
                                } else c = P(h.matches);
                                if (c.isAlternator) v.push(c); else if (c.alternatorGroup ? (u = v.pop(), c.alternatorGroup = !1) : u = new n.default(!1, !1, !1, !0),
                                    u.matches.push(c), v.push(u), c.openGroup) {
                                    c.openGroup = !1;
                                    var B = new n.default(!0);
                                    B.alternatorGroup = !0, v.push(B);
                                }
                                break;

                            default:
                                y();
                        }
                    }
                    g && x();
                    for (;v.length > 0; ) l = v.pop(), h.matches.push(l);
                    h.matches.length > 0 && (!function e(a) {
                        a && a.matches && a.matches.forEach((function(n, r) {
                            var o = a.matches[r + 1];
                            (void 0 === o || void 0 === o.matches || !1 === o.isQuantifier) && n && n.isGroup && (n.isGroup = !1,
                            t || (k(n, i.groupmarker[0], 0), !0 !== n.openGroup && k(n, i.groupmarker[1]))),
                                e(n);
                        }));
                    }(h), m.push(h));
                    (i.numericInput || i.isRTL) && function e(t) {
                        for (var a in t.matches = t.matches.reverse(), t.matches) if (Object.prototype.hasOwnProperty.call(t.matches, a)) {
                            var n = parseInt(a);
                            if (t.matches[a].isQuantifier && t.matches[n + 1] && t.matches[n + 1].isGroup) {
                                var r = t.matches[a];
                                t.matches.splice(a, 1), t.matches.splice(n + 1, 0, r);
                            }
                            void 0 !== t.matches[a].matches ? t.matches[a] = e(t.matches[a]) : t.matches[a] = ((o = t.matches[a]) === i.optionalmarker[0] ? o = i.optionalmarker[1] : o === i.optionalmarker[1] ? o = i.optionalmarker[0] : o === i.groupmarker[0] ? o = i.groupmarker[1] : o === i.groupmarker[1] && (o = i.groupmarker[0]),
                                o);
                        }
                        var o;
                        return t;
                    }(m[0]);
                    return m;
                }, t.generateMaskSet = function(e, t) {
                    var i;
                    function n(e, i, n) {
                        var o, l, s = !1;
                        if (null !== e && "" !== e || ((s = null !== n.regex) ? e = (e = n.regex).replace(/^(\^)(.*)(\$)$/, "$2") : (s = !0,
                            e = ".*")), 1 === e.length && !1 === n.greedy && 0 !== n.repeat && (n.placeholder = ""),
                        n.repeat > 0 || "*" === n.repeat || "+" === n.repeat) {
                            var u = "*" === n.repeat ? 0 : "+" === n.repeat ? 1 : n.repeat;
                            e = n.groupmarker[0] + e + n.groupmarker[1] + n.quantifiermarker[0] + u + "," + n.repeat + n.quantifiermarker[1];
                        }
                        return l = s ? "regex_" + n.regex : n.numericInput ? e.split("").reverse().join("") : e,
                        null !== n.keepStatic && (l = "ks_" + n.keepStatic + l), void 0 === r.default.prototype.masksCache[l] || !0 === t ? (o = {
                            mask: e,
                            maskToken: r.default.prototype.analyseMask(e, s, n),
                            validPositions: {},
                            _buffer: void 0,
                            buffer: void 0,
                            tests: {},
                            excludes: {},
                            metadata: i,
                            maskLength: void 0,
                            jitOffset: {}
                        }, !0 !== t && (r.default.prototype.masksCache[l] = o, o = a.default.extend(!0, {}, r.default.prototype.masksCache[l]))) : o = a.default.extend(!0, {}, r.default.prototype.masksCache[l]),
                            o;
                    }
                    "function" == typeof e.mask && (e.mask = e.mask(e));
                    if (Array.isArray(e.mask)) {
                        if (e.mask.length > 1) {
                            null === e.keepStatic && (e.keepStatic = !0);
                            var o = e.groupmarker[0];
                            return (e.isRTL ? e.mask.reverse() : e.mask).forEach((function(t) {
                                o.length > 1 && (o += e.alternatormarker), void 0 !== t.mask && "function" != typeof t.mask ? o += t.mask : o += t;
                            })), n(o += e.groupmarker[1], e.mask, e);
                        }
                        e.mask = e.mask.pop();
                    }
                    i = e.mask && void 0 !== e.mask.mask && "function" != typeof e.mask.mask ? n(e.mask.mask, e.mask, e) : n(e.mask, e.mask, e);
                    null === e.keepStatic && (e.keepStatic = !1);
                    return i;
                };
                var a = o(i(4963)), n = o(i(9695)), r = o(i(2394));
                function o(e) {
                    return e && e.__esModule ? e : {
                        default: e
                    };
                }
            },
            157: function(e, t, i) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.mask = function() {
                    var e = this, t = this.opts, i = this.el, a = this.dependencyLib;
                    l.EventRuler.off(i);
                    var f = function(t, i) {
                        "textarea" !== t.tagName.toLowerCase() && i.ignorables.push(n.default.ENTER);
                        var s = t.getAttribute("type"), u = "input" === t.tagName.toLowerCase() && i.supportsInputType.includes(s) || t.isContentEditable || "textarea" === t.tagName.toLowerCase();
                        if (!u) if ("input" === t.tagName.toLowerCase()) {
                            var c = document.createElement("input");
                            c.setAttribute("type", s), u = "text" === c.type, c = null;
                        } else u = "partial";
                        return !1 !== u ? function(t) {
                            var n, s;
                            function u() {
                                return this.inputmask ? this.inputmask.opts.autoUnmask ? this.inputmask.unmaskedvalue() : -1 !== r.getLastValidPosition.call(e) || !0 !== i.nullable ? (this.inputmask.shadowRoot || this.ownerDocument).activeElement === this && i.clearMaskOnLostFocus ? (e.isRTL ? o.clearOptionalTail.call(e, r.getBuffer.call(e).slice()).reverse() : o.clearOptionalTail.call(e, r.getBuffer.call(e).slice())).join("") : n.call(this) : "" : n.call(this);
                            }
                            function c(e) {
                                s.call(this, e), this.inputmask && (0, o.applyInputValue)(this, e);
                            }
                            if (!t.inputmask.__valueGet) {
                                if (!0 !== i.noValuePatching) {
                                    if (Object.getOwnPropertyDescriptor) {
                                        var f = Object.getPrototypeOf ? Object.getOwnPropertyDescriptor(Object.getPrototypeOf(t), "value") : void 0;
                                        f && f.get && f.set ? (n = f.get, s = f.set, Object.defineProperty(t, "value", {
                                            get: u,
                                            set: c,
                                            configurable: !0
                                        })) : "input" !== t.tagName.toLowerCase() && (n = function() {
                                            return this.textContent;
                                        }, s = function(e) {
                                            this.textContent = e;
                                        }, Object.defineProperty(t, "value", {
                                            get: u,
                                            set: c,
                                            configurable: !0
                                        }));
                                    } else document.__lookupGetter__ && t.__lookupGetter__("value") && (n = t.__lookupGetter__("value"),
                                        s = t.__lookupSetter__("value"), t.__defineGetter__("value", u), t.__defineSetter__("value", c));
                                    t.inputmask.__valueGet = n, t.inputmask.__valueSet = s;
                                }
                                t.inputmask._valueGet = function(t) {
                                    return e.isRTL && !0 !== t ? n.call(this.el).split("").reverse().join("") : n.call(this.el);
                                }, t.inputmask._valueSet = function(t, i) {
                                    s.call(this.el, null == t ? "" : !0 !== i && e.isRTL ? t.split("").reverse().join("") : t);
                                }, void 0 === n && (n = function() {
                                    return this.value;
                                }, s = function(e) {
                                    this.value = e;
                                }, function(t) {
                                    if (a.valHooks && (void 0 === a.valHooks[t] || !0 !== a.valHooks[t].inputmaskpatch)) {
                                        var n = a.valHooks[t] && a.valHooks[t].get ? a.valHooks[t].get : function(e) {
                                            return e.value;
                                        }, l = a.valHooks[t] && a.valHooks[t].set ? a.valHooks[t].set : function(e, t) {
                                            return e.value = t, e;
                                        };
                                        a.valHooks[t] = {
                                            get: function(t) {
                                                if (t.inputmask) {
                                                    if (t.inputmask.opts.autoUnmask) return t.inputmask.unmaskedvalue();
                                                    var a = n(t);
                                                    return -1 !== r.getLastValidPosition.call(e, void 0, void 0, t.inputmask.maskset.validPositions) || !0 !== i.nullable ? a : "";
                                                }
                                                return n(t);
                                            },
                                            set: function(e, t) {
                                                var i = l(e, t);
                                                return e.inputmask && (0, o.applyInputValue)(e, t), i;
                                            },
                                            inputmaskpatch: !0
                                        };
                                    }
                                }(t.type), function(t) {
                                    l.EventRuler.on(t, "mouseenter", (function() {
                                        var t = this.inputmask._valueGet(!0);
                                        t !== (e.isRTL ? r.getBuffer.call(e).reverse() : r.getBuffer.call(e)).join("") && (0,
                                            o.applyInputValue)(this, t);
                                    }));
                                }(t));
                            }
                        }(t) : t.inputmask = void 0, u;
                    }(i, t);
                    if (!1 !== f) {
                        e.originalPlaceholder = i.placeholder, e.maxLength = void 0 !== i ? i.maxLength : void 0,
                        -1 === e.maxLength && (e.maxLength = void 0), "inputMode" in i && null === i.getAttribute("inputmode") && (i.inputMode = t.inputmode,
                            i.setAttribute("inputmode", t.inputmode)), !0 === f && (t.showMaskOnFocus = t.showMaskOnFocus && -1 === [ "cc-number", "cc-exp" ].indexOf(i.autocomplete),
                        s.iphone && (t.insertModeVisual = !1), l.EventRuler.on(i, "submit", c.EventHandlers.submitEvent),
                            l.EventRuler.on(i, "reset", c.EventHandlers.resetEvent), l.EventRuler.on(i, "blur", c.EventHandlers.blurEvent),
                            l.EventRuler.on(i, "focus", c.EventHandlers.focusEvent), l.EventRuler.on(i, "invalid", c.EventHandlers.invalidEvent),
                            l.EventRuler.on(i, "click", c.EventHandlers.clickEvent), l.EventRuler.on(i, "mouseleave", c.EventHandlers.mouseleaveEvent),
                            l.EventRuler.on(i, "mouseenter", c.EventHandlers.mouseenterEvent), l.EventRuler.on(i, "paste", c.EventHandlers.pasteEvent),
                            l.EventRuler.on(i, "cut", c.EventHandlers.cutEvent), l.EventRuler.on(i, "complete", t.oncomplete),
                            l.EventRuler.on(i, "incomplete", t.onincomplete), l.EventRuler.on(i, "cleared", t.oncleared),
                        !0 !== t.inputEventOnly && (l.EventRuler.on(i, "keydown", c.EventHandlers.keydownEvent),
                            l.EventRuler.on(i, "keypress", c.EventHandlers.keypressEvent), l.EventRuler.on(i, "keyup", c.EventHandlers.keyupEvent)),
                        (s.mobile || t.inputEventOnly) && i.removeAttribute("maxLength"), l.EventRuler.on(i, "input", c.EventHandlers.inputFallBackEvent),
                            l.EventRuler.on(i, "compositionend", c.EventHandlers.compositionendEvent)), l.EventRuler.on(i, "setvalue", c.EventHandlers.setValueEvent),
                            r.getBufferTemplate.call(e).join(""), e.undoValue = e._valueGet(!0);
                        var d = (i.inputmask.shadowRoot || i.ownerDocument).activeElement;
                        if ("" !== i.inputmask._valueGet(!0) || !1 === t.clearMaskOnLostFocus || d === i) {
                            (0, o.applyInputValue)(i, i.inputmask._valueGet(!0), t);
                            var p = r.getBuffer.call(e).slice();
                            !1 === u.isComplete.call(e, p) && t.clearIncomplete && r.resetMaskSet.call(e), t.clearMaskOnLostFocus && d !== i && (-1 === r.getLastValidPosition.call(e) ? p = [] : o.clearOptionalTail.call(e, p)),
                            (!1 === t.clearMaskOnLostFocus || t.showMaskOnFocus && d === i || "" !== i.inputmask._valueGet(!0)) && (0,
                                o.writeBuffer)(i, p), d === i && r.caret.call(e, i, r.seekNext.call(e, r.getLastValidPosition.call(e)));
                        }
                    }
                };
                var a, n = (a = i(5581)) && a.__esModule ? a : {
                    default: a
                }, r = i(8711), o = i(7760), l = i(9716), s = i(9845), u = i(7215), c = i(6030);
            },
            9695: function(e, t) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.default = function(e, t, i, a) {
                    this.matches = [], this.openGroup = e || !1, this.alternatorGroup = !1, this.isGroup = e || !1,
                        this.isOptional = t || !1, this.isQuantifier = i || !1, this.isAlternator = a || !1,
                        this.quantifier = {
                            min: 1,
                            max: 1
                        };
                };
            },
            3194: function() {
                Array.prototype.includes || Object.defineProperty(Array.prototype, "includes", {
                    value: function(e, t) {
                        if (null == this) throw new TypeError('"this" is null or not defined');
                        var i = Object(this), a = i.length >>> 0;
                        if (0 === a) return !1;
                        for (var n = 0 | t, r = Math.max(n >= 0 ? n : a - Math.abs(n), 0); r < a; ) {
                            if (i[r] === e) return !0;
                            r++;
                        }
                        return !1;
                    }
                });
            },
            7149: function() {
                function e(t) {
                    return e = "function" == typeof Symbol && "symbol" == typeof Symbol.iterator ? function(e) {
                        return typeof e;
                    } : function(e) {
                        return e && "function" == typeof Symbol && e.constructor === Symbol && e !== Symbol.prototype ? "symbol" : typeof e;
                    }, e(t);
                }
                "function" != typeof Object.getPrototypeOf && (Object.getPrototypeOf = "object" === e("test".__proto__) ? function(e) {
                    return e.__proto__;
                } : function(e) {
                    return e.constructor.prototype;
                });
            },
            8711: function(e, t, i) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.caret = function(e, t, i, a, n) {
                    var r, o = this, l = this.opts;
                    if (void 0 === t) return "selectionStart" in e && "selectionEnd" in e ? (t = e.selectionStart,
                        i = e.selectionEnd) : window.getSelection ? (r = window.getSelection().getRangeAt(0)).commonAncestorContainer.parentNode !== e && r.commonAncestorContainer !== e || (t = r.startOffset,
                        i = r.endOffset) : document.selection && document.selection.createRange && (r = document.selection.createRange(),
                        t = 0 - r.duplicate().moveStart("character", -e.inputmask._valueGet().length), i = t + r.text.length),
                        {
                            begin: a ? t : u.call(o, t),
                            end: a ? i : u.call(o, i)
                        };
                    if (Array.isArray(t) && (i = o.isRTL ? t[0] : t[1], t = o.isRTL ? t[1] : t[0]),
                    void 0 !== t.begin && (i = o.isRTL ? t.begin : t.end, t = o.isRTL ? t.end : t.begin),
                    "number" == typeof t) {
                        t = a ? t : u.call(o, t), i = "number" == typeof (i = a ? i : u.call(o, i)) ? i : t;
                        var s = parseInt(((e.ownerDocument.defaultView || window).getComputedStyle ? (e.ownerDocument.defaultView || window).getComputedStyle(e, null) : e.currentStyle).fontSize) * i;
                        if (e.scrollLeft = s > e.scrollWidth ? s : 0, e.inputmask.caretPos = {
                            begin: t,
                            end: i
                        }, l.insertModeVisual && !1 === l.insertMode && t === i && (n || i++), e === (e.inputmask.shadowRoot || e.ownerDocument).activeElement) if ("setSelectionRange" in e) e.setSelectionRange(t, i); else if (window.getSelection) {
                            if (r = document.createRange(), void 0 === e.firstChild || null === e.firstChild) {
                                var c = document.createTextNode("");
                                e.appendChild(c);
                            }
                            r.setStart(e.firstChild, t < e.inputmask._valueGet().length ? t : e.inputmask._valueGet().length),
                                r.setEnd(e.firstChild, i < e.inputmask._valueGet().length ? i : e.inputmask._valueGet().length),
                                r.collapse(!0);
                            var f = window.getSelection();
                            f.removeAllRanges(), f.addRange(r);
                        } else e.createTextRange && ((r = e.createTextRange()).collapse(!0), r.moveEnd("character", i),
                            r.moveStart("character", t), r.select());
                    }
                }, t.determineLastRequiredPosition = function(e) {
                    var t, i, r = this, l = this.maskset, s = this.dependencyLib, u = a.getMaskTemplate.call(r, !0, o.call(r), !0, !0), c = u.length, f = o.call(r), d = {}, p = l.validPositions[f], h = void 0 !== p ? p.locator.slice() : void 0;
                    for (t = f + 1; t < u.length; t++) i = a.getTestTemplate.call(r, t, h, t - 1), h = i.locator.slice(),
                        d[t] = s.extend(!0, {}, i);
                    var v = p && void 0 !== p.alternation ? p.locator[p.alternation] : void 0;
                    for (t = c - 1; t > f && (((i = d[t]).match.optionality || i.match.optionalQuantifier && i.match.newBlockMarker || v && (v !== d[t].locator[p.alternation] && 1 != i.match.static || !0 === i.match.static && i.locator[p.alternation] && n.checkAlternationMatch.call(r, i.locator[p.alternation].toString().split(","), v.toString().split(",")) && "" !== a.getTests.call(r, t)[0].def)) && u[t] === a.getPlaceholder.call(r, t, i.match)); t--) c--;
                    return e ? {
                        l: c,
                        def: d[c] ? d[c].match : void 0
                    } : c;
                }, t.determineNewCaretPosition = function(e, t, i) {
                    var n = this, u = this.maskset, c = this.opts;
                    t && (n.isRTL ? e.end = e.begin : e.begin = e.end);
                    if (e.begin === e.end) {
                        switch (i = i || c.positionCaretOnClick) {
                            case "none":
                                break;

                            case "select":
                                e = {
                                    begin: 0,
                                    end: r.call(n).length
                                };
                                break;

                            case "ignore":
                                e.end = e.begin = s.call(n, o.call(n));
                                break;

                            case "radixFocus":
                                if (function(e) {
                                    if ("" !== c.radixPoint && 0 !== c.digits) {
                                        var t = u.validPositions;
                                        if (void 0 === t[e] || t[e].input === a.getPlaceholder.call(n, e)) {
                                            if (e < s.call(n, -1)) return !0;
                                            var i = r.call(n).indexOf(c.radixPoint);
                                            if (-1 !== i) {
                                                for (var o in t) if (t[o] && i < o && t[o].input !== a.getPlaceholder.call(n, o)) return !1;
                                                return !0;
                                            }
                                        }
                                    }
                                    return !1;
                                }(e.begin)) {
                                    var f = r.call(n).join("").indexOf(c.radixPoint);
                                    e.end = e.begin = c.numericInput ? s.call(n, f) : f;
                                    break;
                                }

                            default:
                                var d = e.begin, p = o.call(n, d, !0), h = s.call(n, -1 !== p || l.call(n, 0) ? p : -1);
                                if (d <= h) e.end = e.begin = l.call(n, d, !1, !0) ? d : s.call(n, d); else {
                                    var v = u.validPositions[p], m = a.getTestTemplate.call(n, h, v ? v.match.locator : void 0, v), g = a.getPlaceholder.call(n, h, m.match);
                                    if ("" !== g && r.call(n)[h] !== g && !0 !== m.match.optionalQuantifier && !0 !== m.match.newBlockMarker || !l.call(n, h, c.keepStatic, !0) && m.match.def === g) {
                                        var k = s.call(n, h);
                                        (d >= k || d === h) && (h = k);
                                    }
                                    e.end = e.begin = h;
                                }
                        }
                        return e;
                    }
                }, t.getBuffer = r, t.getBufferTemplate = function() {
                    var e = this.maskset;
                    void 0 === e._buffer && (e._buffer = a.getMaskTemplate.call(this, !1, 1), void 0 === e.buffer && (e.buffer = e._buffer.slice()));
                    return e._buffer;
                }, t.getLastValidPosition = o, t.isMask = l, t.resetMaskSet = function(e) {
                    var t = this.maskset;
                    t.buffer = void 0, !0 !== e && (t.validPositions = {}, t.p = 0);
                }, t.seekNext = s, t.seekPrevious = function(e, t) {
                    var i = this, n = e - 1;
                    if (e <= 0) return 0;
                    for (;n > 0 && (!0 === t && (!0 !== a.getTest.call(i, n).match.newBlockMarker || !l.call(i, n, void 0, !0)) || !0 !== t && !l.call(i, n, void 0, !0)); ) n--;
                    return n;
                }, t.translatePosition = u;
                var a = i(4713), n = i(7215);
                function r(e) {
                    var t = this.maskset;
                    return void 0 !== t.buffer && !0 !== e || (t.buffer = a.getMaskTemplate.call(this, !0, o.call(this), !0),
                    void 0 === t._buffer && (t._buffer = t.buffer.slice())), t.buffer;
                }
                function o(e, t, i) {
                    var a = this.maskset, n = -1, r = -1, o = i || a.validPositions;
                    for (var l in void 0 === e && (e = -1), o) {
                        var s = parseInt(l);
                        o[s] && (t || !0 !== o[s].generatedInput) && (s <= e && (n = s), s >= e && (r = s));
                    }
                    return -1 === n || n == e ? r : -1 == r || e - n < r - e ? n : r;
                }
                function l(e, t, i) {
                    var n = this, r = this.maskset, o = a.getTestTemplate.call(n, e).match;
                    if ("" === o.def && (o = a.getTest.call(n, e).match), !0 !== o.static) return o.fn;
                    if (!0 === i && void 0 !== r.validPositions[e] && !0 !== r.validPositions[e].generatedInput) return !0;
                    if (!0 !== t && e > -1) {
                        if (i) {
                            var l = a.getTests.call(n, e);
                            return l.length > 1 + ("" === l[l.length - 1].match.def ? 1 : 0);
                        }
                        var s = a.determineTestTemplate.call(n, e, a.getTests.call(n, e)), u = a.getPlaceholder.call(n, e, s.match);
                        return s.match.def !== u;
                    }
                    return !1;
                }
                function s(e, t, i) {
                    var n = this;
                    void 0 === i && (i = !0);
                    for (var r = e + 1; "" !== a.getTest.call(n, r).match.def && (!0 === t && (!0 !== a.getTest.call(n, r).match.newBlockMarker || !l.call(n, r, void 0, !0)) || !0 !== t && !l.call(n, r, void 0, i)); ) r++;
                    return r;
                }
                function u(e) {
                    var t = this.opts, i = this.el;
                    return !this.isRTL || "number" != typeof e || t.greedy && "" === t.placeholder || !i || (e = Math.abs(this._valueGet().length - e)),
                        e;
                }
            },
            4713: function(e, t, i) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.determineTestTemplate = u, t.getDecisionTaker = o, t.getMaskTemplate = function(e, t, i, a, n) {
                    var r = this, o = this.opts, c = this.maskset, f = o.greedy;
                    n && o.greedy && (o.greedy = !1, r.maskset.tests = {});
                    t = t || 0;
                    var p, h, v, m, g = [], k = 0;
                    do {
                        if (!0 === e && c.validPositions[k]) v = n && c.validPositions[k].match.optionality && void 0 === c.validPositions[k + 1] && (!0 === c.validPositions[k].generatedInput || c.validPositions[k].input == o.skipOptionalPartCharacter && k > 0) ? u.call(r, k, d.call(r, k, p, k - 1)) : c.validPositions[k],
                            h = v.match, p = v.locator.slice(), g.push(!0 === i ? v.input : !1 === i ? h.nativeDef : l.call(r, k, h)); else {
                            v = s.call(r, k, p, k - 1), h = v.match, p = v.locator.slice();
                            var y = !0 !== a && (!1 !== o.jitMasking ? o.jitMasking : h.jit);
                            (m = (m && h.static && h.def !== o.groupSeparator && null === h.fn || c.validPositions[k - 1] && h.static && h.def !== o.groupSeparator && null === h.fn) && c.tests[k] && 1 === c.tests[k].length) || !1 === y || void 0 === y || "number" == typeof y && isFinite(y) && y > k ? g.push(!1 === i ? h.nativeDef : l.call(r, k, h)) : m = !1;
                        }
                        k++;
                    } while (!0 !== h.static || "" !== h.def || t > k);
                    "" === g[g.length - 1] && g.pop();
                    !1 === i && void 0 !== c.maskLength || (c.maskLength = k - 1);
                    return o.greedy = f, g;
                }, t.getPlaceholder = l, t.getTest = c, t.getTestTemplate = s, t.getTests = d, t.isSubsetOf = f;
                var a, n = (a = i(2394)) && a.__esModule ? a : {
                    default: a
                };
                function r(e, t) {
                    var i = (null != e.alternation ? e.mloc[o(e)] : e.locator).join("");
                    if ("" !== i) for (;i.length < t; ) i += "0";
                    return i;
                }
                function o(e) {
                    var t = e.locator[e.alternation];
                    return "string" == typeof t && t.length > 0 && (t = t.split(",")[0]), void 0 !== t ? t.toString() : "";
                }
                function l(e, t, i) {
                    var a = this.opts, n = this.maskset;
                    if (void 0 !== (t = t || c.call(this, e).match).placeholder || !0 === i) return "function" == typeof t.placeholder ? t.placeholder(a) : t.placeholder;
                    if (!0 === t.static) {
                        if (e > -1 && void 0 === n.validPositions[e]) {
                            var r, o = d.call(this, e), l = [];
                            if (o.length > 1 + ("" === o[o.length - 1].match.def ? 1 : 0)) for (var s = 0; s < o.length; s++) if ("" !== o[s].match.def && !0 !== o[s].match.optionality && !0 !== o[s].match.optionalQuantifier && (!0 === o[s].match.static || void 0 === r || !1 !== o[s].match.fn.test(r.match.def, n, e, !0, a)) && (l.push(o[s]),
                            !0 === o[s].match.static && (r = o[s]), l.length > 1 && /[0-9a-bA-Z]/.test(l[0].match.def))) return a.placeholder.charAt(e % a.placeholder.length);
                        }
                        return t.def;
                    }
                    return a.placeholder.charAt(e % a.placeholder.length);
                }
                function s(e, t, i) {
                    return this.maskset.validPositions[e] || u.call(this, e, d.call(this, e, t ? t.slice() : t, i));
                }
                function u(e, t) {
                    var i = this.opts, a = function(e, t) {
                        var i = 0, a = !1;
                        t.forEach((function(e) {
                            e.match.optionality && (0 !== i && i !== e.match.optionality && (a = !0), (0 === i || i > e.match.optionality) && (i = e.match.optionality));
                        })), i && (0 == e || 1 == t.length ? i = 0 : a || (i = 0));
                        return i;
                    }(e, t);
                    e = e > 0 ? e - 1 : 0;
                    var n, o, l, s = r(c.call(this, e));
                    i.greedy && t.length > 1 && "" === t[t.length - 1].match.def && t.pop();
                    for (var u = 0; u < t.length; u++) {
                        var f = t[u];
                        n = r(f, s.length);
                        var d = Math.abs(n - s);
                        (void 0 === o || "" !== n && d < o || l && !i.greedy && l.match.optionality && l.match.optionality - a > 0 && "master" === l.match.newBlockMarker && (!f.match.optionality || f.match.optionality - a < 1 || !f.match.newBlockMarker) || l && !i.greedy && l.match.optionalQuantifier && !f.match.optionalQuantifier) && (o = d,
                            l = f);
                    }
                    return l;
                }
                function c(e, t) {
                    var i = this.maskset;
                    return i.validPositions[e] ? i.validPositions[e] : (t || d.call(this, e))[0];
                }
                function f(e, t, i) {
                    function a(e) {
                        for (var t, i = [], a = -1, n = 0, r = e.length; n < r; n++) if ("-" === e.charAt(n)) for (t = e.charCodeAt(n + 1); ++a < t; ) i.push(String.fromCharCode(a)); else a = e.charCodeAt(n),
                            i.push(e.charAt(n));
                        return i.join("");
                    }
                    return e.match.def === t.match.nativeDef || !(!(i.regex || e.match.fn instanceof RegExp && t.match.fn instanceof RegExp) || !0 === e.match.static || !0 === t.match.static) && -1 !== a(t.match.fn.toString().replace(/[[\]/]/g, "")).indexOf(a(e.match.fn.toString().replace(/[[\]/]/g, "")));
                }
                function d(e, t, i) {
                    var a, r, o = this, l = this.dependencyLib, s = this.maskset, c = this.opts, d = this.el, p = s.maskToken, h = t ? i : 0, v = t ? t.slice() : [ 0 ], m = [], g = !1, k = t ? t.join("") : "";
                    function y(t, i, r, o) {
                        function l(r, o, u) {
                            function p(e, t) {
                                var i = 0 === t.matches.indexOf(e);
                                return i || t.matches.every((function(a, n) {
                                    return !0 === a.isQuantifier ? i = p(e, t.matches[n - 1]) : Object.prototype.hasOwnProperty.call(a, "matches") && (i = p(e, a)),
                                        !i;
                                })), i;
                            }
                            function v(e, t, i) {
                                var a, n;
                                if ((s.tests[e] || s.validPositions[e]) && (s.tests[e] || [ s.validPositions[e] ]).every((function(e, r) {
                                    if (e.mloc[t]) return a = e, !1;
                                    var o = void 0 !== i ? i : e.alternation, l = void 0 !== e.locator[o] ? e.locator[o].toString().indexOf(t) : -1;
                                    return (void 0 === n || l < n) && -1 !== l && (a = e, n = l), !0;
                                })), a) {
                                    var r = a.locator[a.alternation];
                                    return (a.mloc[t] || a.mloc[r] || a.locator).slice((void 0 !== i ? i : a.alternation) + 1);
                                }
                                return void 0 !== i ? v(e, t) : void 0;
                            }
                            function b(e, t) {
                                var i = e.alternation, a = void 0 === t || i === t.alternation && -1 === e.locator[i].toString().indexOf(t.locator[i]);
                                if (!a && i > t.alternation) for (var n = t.alternation; n < i; n++) if (e.locator[n] !== t.locator[n]) {
                                    i = n, a = !0;
                                    break;
                                }
                                if (a) {
                                    e.mloc = e.mloc || {};
                                    var r = e.locator[i];
                                    if (void 0 !== r) {
                                        if ("string" == typeof r && (r = r.split(",")[0]), void 0 === e.mloc[r] && (e.mloc[r] = e.locator.slice()),
                                        void 0 !== t) {
                                            for (var o in t.mloc) "string" == typeof o && (o = o.split(",")[0]), void 0 === e.mloc[o] && (e.mloc[o] = t.mloc[o]);
                                            e.locator[i] = Object.keys(e.mloc).join(",");
                                        }
                                        return !0;
                                    }
                                    e.alternation = void 0;
                                }
                                return !1;
                            }
                            function x(e, t) {
                                if (e.locator.length !== t.locator.length) return !1;
                                for (var i = e.alternation + 1; i < e.locator.length; i++) if (e.locator[i] !== t.locator[i]) return !1;
                                return !0;
                            }
                            if (h > e + c._maxTestPos) throw "Inputmask: There is probably an error in your mask definition or in the code. Create an issue on github with an example of the mask you are using. " + s.mask;
                            if (h === e && void 0 === r.matches) {
                                if (m.push({
                                    match: r,
                                    locator: o.reverse(),
                                    cd: k,
                                    mloc: {}
                                }), !r.optionality || void 0 !== u || !(c.definitions && c.definitions[r.nativeDef] && c.definitions[r.nativeDef].optional || n.default.prototype.definitions[r.nativeDef] && n.default.prototype.definitions[r.nativeDef].optional)) return !0;
                                g = !0, h = e;
                            } else if (void 0 !== r.matches) {
                                if (r.isGroup && u !== r) {
                                    if (r = l(t.matches[t.matches.indexOf(r) + 1], o, u)) return !0;
                                } else if (r.isOptional) {
                                    var P = r, E = m.length;
                                    if (r = y(r, i, o, u)) {
                                        if (m.forEach((function(e, t) {
                                            t >= E && (e.match.optionality = e.match.optionality ? e.match.optionality + 1 : 1);
                                        })), a = m[m.length - 1].match, void 0 !== u || !p(a, P)) return !0;
                                        g = !0, h = e;
                                    }
                                } else if (r.isAlternator) {
                                    var S, _ = r, w = [], M = m.slice(), O = o.length, T = !1, C = i.length > 0 ? i.shift() : -1;
                                    if (-1 === C || "string" == typeof C) {
                                        var A, D = h, j = i.slice(), B = [];
                                        if ("string" == typeof C) B = C.split(","); else for (A = 0; A < _.matches.length; A++) B.push(A.toString());
                                        if (void 0 !== s.excludes[e]) {
                                            for (var R = B.slice(), L = 0, I = s.excludes[e].length; L < I; L++) {
                                                var F = s.excludes[e][L].toString().split(":");
                                                o.length == F[1] && B.splice(B.indexOf(F[0]), 1);
                                            }
                                            0 === B.length && (delete s.excludes[e], B = R);
                                        }
                                        (!0 === c.keepStatic || isFinite(parseInt(c.keepStatic)) && D >= c.keepStatic) && (B = B.slice(0, 1));
                                        for (var N = 0; N < B.length; N++) {
                                            A = parseInt(B[N]), m = [], i = "string" == typeof C && v(h, A, O) || j.slice();
                                            var V = _.matches[A];
                                            if (V && l(V, [ A ].concat(o), u)) r = !0; else if (0 === N && (T = !0), V && V.matches && V.matches.length > _.matches[0].matches.length) break;
                                            S = m.slice(), h = D, m = [];
                                            for (var G = 0; G < S.length; G++) {
                                                var H = S[G], K = !1;
                                                H.match.jit = H.match.jit || T, H.alternation = H.alternation || O, b(H);
                                                for (var U = 0; U < w.length; U++) {
                                                    var $ = w[U];
                                                    if ("string" != typeof C || void 0 !== H.alternation && B.includes(H.locator[H.alternation].toString())) {
                                                        if (H.match.nativeDef === $.match.nativeDef) {
                                                            K = !0, b($, H);
                                                            break;
                                                        }
                                                        if (f(H, $, c)) {
                                                            b(H, $) && (K = !0, w.splice(w.indexOf($), 0, H));
                                                            break;
                                                        }
                                                        if (f($, H, c)) {
                                                            b($, H);
                                                            break;
                                                        }
                                                        if (Z = $, !0 === (Q = H).match.static && !0 !== Z.match.static && Z.match.fn.test(Q.match.def, s, e, !1, c, !1)) {
                                                            x(H, $) || void 0 !== d.inputmask.userOptions.keepStatic ? b(H, $) && (K = !0, w.splice(w.indexOf($), 0, H)) : c.keepStatic = !0;
                                                            break;
                                                        }
                                                    }
                                                }
                                                K || w.push(H);
                                            }
                                        }
                                        m = M.concat(w), h = e, g = m.length > 0, r = w.length > 0, i = j.slice();
                                    } else r = l(_.matches[C] || t.matches[C], [ C ].concat(o), u);
                                    if (r) return !0;
                                } else if (r.isQuantifier && u !== t.matches[t.matches.indexOf(r) - 1]) for (var q = r, z = i.length > 0 ? i.shift() : 0; z < (isNaN(q.quantifier.max) ? z + 1 : q.quantifier.max) && h <= e; z++) {
                                    var W = t.matches[t.matches.indexOf(q) - 1];
                                    if (r = l(W, [ z ].concat(o), W)) {
                                        if ((a = m[m.length - 1].match).optionalQuantifier = z >= q.quantifier.min, a.jit = (z + 1) * (W.matches.indexOf(a) + 1) > q.quantifier.jit,
                                        a.optionalQuantifier && p(a, W)) {
                                            g = !0, h = e;
                                            break;
                                        }
                                        return a.jit && (s.jitOffset[e] = W.matches.length - W.matches.indexOf(a)), !0;
                                    }
                                } else if (r = y(r, i, o, u)) return !0;
                            } else h++;
                            var Q, Z;
                        }
                        for (var u = i.length > 0 ? i.shift() : 0; u < t.matches.length; u++) if (!0 !== t.matches[u].isQuantifier) {
                            var p = l(t.matches[u], [ u ].concat(r), o);
                            if (p && h === e) return p;
                            if (h > e) break;
                        }
                    }
                    if (e > -1) {
                        if (void 0 === t) {
                            for (var b, x = e - 1; void 0 === (b = s.validPositions[x] || s.tests[x]) && x > -1; ) x--;
                            void 0 !== b && x > -1 && (v = function(e, t) {
                                var i, a = [];
                                return Array.isArray(t) || (t = [ t ]), t.length > 0 && (void 0 === t[0].alternation || !0 === c.keepStatic ? 0 === (a = u.call(o, e, t.slice()).locator.slice()).length && (a = t[0].locator.slice()) : t.forEach((function(e) {
                                    "" !== e.def && (0 === a.length ? (i = e.alternation, a = e.locator.slice()) : e.locator[i] && -1 === a[i].toString().indexOf(e.locator[i]) && (a[i] += "," + e.locator[i]));
                                }))), a;
                            }(x, b), k = v.join(""), h = x);
                        }
                        if (s.tests[e] && s.tests[e][0].cd === k) return s.tests[e];
                        for (var P = v.shift(); P < p.length; P++) {
                            if (y(p[P], v, [ P ]) && h === e || h > e) break;
                        }
                    }
                    return (0 === m.length || g) && m.push({
                        match: {
                            fn: null,
                            static: !0,
                            optionality: !1,
                            casing: null,
                            def: "",
                            placeholder: ""
                        },
                        locator: [],
                        mloc: {},
                        cd: k
                    }), void 0 !== t && s.tests[e] ? r = l.extend(!0, [], m) : (s.tests[e] = l.extend(!0, [], m),
                        r = s.tests[e]), m.forEach((function(e) {
                        e.match.optionality = !1;
                    })), r;
                }
            },
            7215: function(e, t, i) {
                Object.defineProperty(t, "__esModule", {
                    value: !0
                }), t.alternate = s, t.checkAlternationMatch = function(e, t, i) {
                    for (var a, n = this.opts.greedy ? t : t.slice(0, 1), r = !1, o = void 0 !== i ? i.split(",") : [], l = 0; l < o.length; l++) -1 !== (a = e.indexOf(o[l])) && e.splice(a, 1);
                    for (var s = 0; s < e.length; s++) if (n.includes(e[s])) {
                        r = !0;
                        break;
                    }
                    return r;
                }, t.handleRemove = function(e, t, i, a, l) {
                    var u = this, c = this.maskset, f = this.opts;
                    if ((f.numericInput || u.isRTL) && (t === r.default.BACKSPACE ? t = r.default.DELETE : t === r.default.DELETE && (t = r.default.BACKSPACE),
                        u.isRTL)) {
                        var d = i.end;
                        i.end = i.begin, i.begin = d;
                    }
                    var p, h = o.getLastValidPosition.call(u, void 0, !0);
                    i.end >= o.getBuffer.call(u).length && h >= i.end && (i.end = h + 1);
                    t === r.default.BACKSPACE ? i.end - i.begin < 1 && (i.begin = o.seekPrevious.call(u, i.begin)) : t === r.default.DELETE && i.begin === i.end && (i.end = o.isMask.call(u, i.end, !0, !0) ? i.end + 1 : o.seekNext.call(u, i.end) + 1);
                    if (!1 !== (p = m.call(u, i))) {
                        if (!0 !== a && !1 !== f.keepStatic || null !== f.regex && -1 !== n.getTest.call(u, i.begin).match.def.indexOf("|")) {
                            var v = s.call(u, !0);
                            if (v) {
                                var g = void 0 !== v.caret ? v.caret : v.pos ? o.seekNext.call(u, v.pos.begin ? v.pos.begin : v.pos) : o.getLastValidPosition.call(u, -1, !0);
                                (t !== r.default.DELETE || i.begin > g) && i.begin;
                            }
                        }
                        !0 !== a && (c.p = t === r.default.DELETE ? i.begin + p : i.begin, c.p = o.determineNewCaretPosition.call(u, {
                            begin: c.p,
                            end: c.p
                        }, !1, !1 === f.insertMode && t === r.default.BACKSPACE ? "none" : void 0).begin);
                    }
                }, t.isComplete = c, t.isSelection = f, t.isValid = d, t.refreshFromBuffer = h,
                    t.revalidateMask = m;
                var a, n = i(4713), r = (a = i(5581)) && a.__esModule ? a : {
                    default: a
                }, o = i(8711), l = i(6030);
                function s(e, t, i, a, r, l) {
                    var u, c, f, p, h, v, m, g, k, y, b, x = this, P = this.dependencyLib, E = this.opts, S = x.maskset, _ = P.extend(!0, {}, S.validPositions), w = P.extend(!0, {}, S.tests), M = !1, O = !1, T = void 0 !== r ? r : o.getLastValidPosition.call(x);
                    if (l && (y = l.begin, b = l.end, l.begin > l.end && (y = l.end, b = l.begin)),
                    -1 === T && void 0 === r) u = 0, c = (p = n.getTest.call(x, u)).alternation; else for (;T >= 0; T--) if ((f = S.validPositions[T]) && void 0 !== f.alternation) {
                        if (p && p.locator[f.alternation] !== f.locator[f.alternation]) break;
                        u = T, c = S.validPositions[u].alternation, p = f;
                    }
                    if (void 0 !== c) {
                        m = parseInt(u), S.excludes[m] = S.excludes[m] || [], !0 !== e && S.excludes[m].push((0,
                            n.getDecisionTaker)(p) + ":" + p.alternation);
                        var C = [], A = -1;
                        for (h = m; h < o.getLastValidPosition.call(x, void 0, !0) + 1; h++) -1 === A && e <= h && void 0 !== t && (C.push(t),
                            A = C.length - 1), (v = S.validPositions[h]) && !0 !== v.generatedInput && (void 0 === l || h < y || h >= b) && C.push(v.input),
                            delete S.validPositions[h];
                        for (-1 === A && void 0 !== t && (C.push(t), A = C.length - 1); void 0 !== S.excludes[m] && S.excludes[m].length < 10; ) {
                            for (S.tests = {}, o.resetMaskSet.call(x, !0), M = !0, h = 0; h < C.length && (g = M.caret || o.getLastValidPosition.call(x, void 0, !0) + 1,
                                k = C[h], M = d.call(x, g, k, !1, a, !0)); h++) h === A && (O = M), 1 == e && M && (O = {
                                caretPos: h
                            });
                            if (M) break;
                            if (o.resetMaskSet.call(x), p = n.getTest.call(x, m), S.validPositions = P.extend(!0, {}, _),
                                S.tests = P.extend(!0, {}, w), !S.excludes[m]) {
                                O = s.call(x, e, t, i, a, m - 1, l);
                                break;
                            }
                            var D = (0, n.getDecisionTaker)(p);
                            if (-1 !== S.excludes[m].indexOf(D + ":" + p.alternation)) {
                                O = s.call(x, e, t, i, a, m - 1, l);
                                break;
                            }
                            for (S.excludes[m].push(D + ":" + p.alternation), h = m; h < o.getLastValidPosition.call(x, void 0, !0) + 1; h++) delete S.validPositions[h];
                        }
                    }
                    return O && !1 === E.keepStatic || delete S.excludes[m], O;
                }
                function u(e, t, i) {
                    var a = this.opts, n = this.maskset;
                    switch (a.casing || t.casing) {
                        case "upper":
                            e = e.toUpperCase();
                            break;

                        case "lower":
                            e = e.toLowerCase();
                            break;

                        case "title":
                            var o = n.validPositions[i - 1];
                            e = 0 === i || o && o.input === String.fromCharCode(r.default.SPACE) ? e.toUpperCase() : e.toLowerCase();
                            break;

                        default:
                            if ("function" == typeof a.casing) {
                                var l = Array.prototype.slice.call(arguments);
                                l.push(n.validPositions), e = a.casing.apply(this, l);
                            }
                    }
                    return e;
                }
                function c(e) {
                    var t = this, i = this.opts, a = this.maskset;
                    if ("function" == typeof i.isComplete) return i.isComplete(e, i);
                    if ("*" !== i.repeat) {
                        var r = !1, l = o.determineLastRequiredPosition.call(t, !0), s = o.seekPrevious.call(t, l.l);
                        if (void 0 === l.def || l.def.newBlockMarker || l.def.optionality || l.def.optionalQuantifier) {
                            r = !0;
                            for (var u = 0; u <= s; u++) {
                                var c = n.getTestTemplate.call(t, u).match;
                                if (!0 !== c.static && void 0 === a.validPositions[u] && !0 !== c.optionality && !0 !== c.optionalQuantifier || !0 === c.static && e[u] !== n.getPlaceholder.call(t, u, c)) {
                                    r = !1;
                                    break;
                                }
                            }
                        }
                        return r;
                    }
                }
                function f(e) {
                    var t = this.opts.insertMode ? 0 : 1;
                    return this.isRTL ? e.begin - e.end > t : e.end - e.begin > t;
                }
                function d(e, t, i, a, r, l, p) {
                    var g = this, k = this.dependencyLib, y = this.opts, b = g.maskset;
                    i = !0 === i;
                    var x = e;
                    function P(e) {
                        if (void 0 !== e) {
                            if (void 0 !== e.remove && (Array.isArray(e.remove) || (e.remove = [ e.remove ]),
                                e.remove.sort((function(e, t) {
                                    return t.pos - e.pos;
                                })).forEach((function(e) {
                                    m.call(g, {
                                        begin: e,
                                        end: e + 1
                                    });
                                })), e.remove = void 0), void 0 !== e.insert && (Array.isArray(e.insert) || (e.insert = [ e.insert ]),
                                e.insert.sort((function(e, t) {
                                    return e.pos - t.pos;
                                })).forEach((function(e) {
                                    "" !== e.c && d.call(g, e.pos, e.c, void 0 === e.strict || e.strict, void 0 !== e.fromIsValid ? e.fromIsValid : a);
                                })), e.insert = void 0), e.refreshFromBuffer && e.buffer) {
                                var t = e.refreshFromBuffer;
                                h.call(g, !0 === t ? t : t.start, t.end, e.buffer), e.refreshFromBuffer = void 0;
                            }
                            void 0 !== e.rewritePosition && (x = e.rewritePosition, e = !0);
                        }
                        return e;
                    }
                    function E(t, i, r) {
                        var l = !1;
                        return n.getTests.call(g, t).every((function(s, c) {
                            var d = s.match;
                            if (o.getBuffer.call(g, !0), !1 !== (l = (!d.jit || void 0 !== b.validPositions[o.seekPrevious.call(g, t)]) && (null != d.fn ? d.fn.test(i, b, t, r, y, f.call(g, e)) : (i === d.def || i === y.skipOptionalPartCharacter) && "" !== d.def && {
                                c: n.getPlaceholder.call(g, t, d, !0) || d.def,
                                pos: t
                            }))) {
                                var p = void 0 !== l.c ? l.c : i, h = t;
                                return p = p === y.skipOptionalPartCharacter && !0 === d.static ? n.getPlaceholder.call(g, t, d, !0) || d.def : p,
                                !0 !== (l = P(l)) && void 0 !== l.pos && l.pos !== t && (h = l.pos), !0 !== l && void 0 === l.pos && void 0 === l.c ? !1 : (!1 === m.call(g, e, k.extend({}, s, {
                                    input: u.call(g, p, d, h)
                                }), a, h) && (l = !1), !1);
                            }
                            return !0;
                        })), l;
                    }
                    void 0 !== e.begin && (x = g.isRTL ? e.end : e.begin);
                    var S = !0, _ = k.extend(!0, {}, b.validPositions);
                    if (!1 === y.keepStatic && void 0 !== b.excludes[x] && !0 !== r && !0 !== a) for (var w = x; w < (g.isRTL ? e.begin : e.end); w++) void 0 !== b.excludes[w] && (b.excludes[w] = void 0,
                        delete b.tests[w]);
                    if ("function" == typeof y.preValidation && !0 !== a && !0 !== l && (S = P(S = y.preValidation.call(g, o.getBuffer.call(g), x, t, f.call(g, e), y, b, e, i || r))),
                    !0 === S) {
                        if (S = E(x, t, i), (!i || !0 === a) && !1 === S && !0 !== l) {
                            var M = b.validPositions[x];
                            if (!M || !0 !== M.match.static || M.match.def !== t && t !== y.skipOptionalPartCharacter) {
                                if (y.insertMode || void 0 === b.validPositions[o.seekNext.call(g, x)] || e.end > x) {
                                    var O = !1;
                                    if (b.jitOffset[x] && void 0 === b.validPositions[o.seekNext.call(g, x)] && !1 !== (S = d.call(g, x + b.jitOffset[x], t, !0, !0)) && (!0 !== r && (S.caret = x),
                                        O = !0), e.end > x && (b.validPositions[x] = void 0), !O && !o.isMask.call(g, x, y.keepStatic && 0 === x)) for (var T = x + 1, C = o.seekNext.call(g, x, !1, 0 !== x); T <= C; T++) if (!1 !== (S = E(T, t, i))) {
                                        S = v.call(g, x, void 0 !== S.pos ? S.pos : T) || S, x = T;
                                        break;
                                    }
                                }
                            } else S = {
                                caret: o.seekNext.call(g, x)
                            };
                        }
                        !1 !== S || !y.keepStatic || !c.call(g, o.getBuffer.call(g)) && 0 !== x || i || !0 === r ? f.call(g, e) && b.tests[x] && b.tests[x].length > 1 && y.keepStatic && !i && !0 !== r && (S = s.call(g, !0)) : S = s.call(g, x, t, i, a, void 0, e),
                        !0 === S && (S = {
                            pos: x
                        });
                    }
                    if ("function" == typeof y.postValidation && !0 !== a && !0 !== l) {
                        var A = y.postValidation.call(g, o.getBuffer.call(g, !0), void 0 !== e.begin ? g.isRTL ? e.end : e.begin : e, t, S, y, b, i, p);
                        void 0 !== A && (S = !0 === A ? S : A);
                    }
                    S && void 0 === S.pos && (S.pos = x), !1 === S || !0 === l ? (o.resetMaskSet.call(g, !0),
                        b.validPositions = k.extend(!0, {}, _)) : v.call(g, void 0, x, !0);
                    var D = P(S);
                    void 0 !== g.maxLength && (o.getBuffer.call(g).length > g.maxLength && !a && (o.resetMaskSet.call(g, !0),
                        b.validPositions = k.extend(!0, {}, _), D = !1));
                    return D;
                }
                function p(e, t, i) {
                    for (var a = this.maskset, r = !1, o = n.getTests.call(this, e), l = 0; l < o.length; l++) {
                        if (o[l].match && (o[l].match.nativeDef === t.match[i.shiftPositions ? "def" : "nativeDef"] && (!i.shiftPositions || !t.match.static) || o[l].match.nativeDef === t.match.nativeDef || i.regex && !o[l].match.static && o[l].match.fn.test(t.input))) {
                            r = !0;
                            break;
                        }
                        if (o[l].match && o[l].match.def === t.match.nativeDef) {
                            r = void 0;
                            break;
                        }
                    }
                    return !1 === r && void 0 !== a.jitOffset[e] && (r = p.call(this, e + a.jitOffset[e], t, i)),
                        r;
                }
                function h(e, t, i) {
                    var a, n, r = this, s = this.maskset, u = this.opts, c = this.dependencyLib, f = u.skipOptionalPartCharacter, d = r.isRTL ? i.slice().reverse() : i;
                    if (u.skipOptionalPartCharacter = "", !0 === e) o.resetMaskSet.call(r), s.tests = {},
                        e = 0, t = i.length, n = o.determineNewCaretPosition.call(r, {
                        begin: 0,
                        end: 0
                    }, !1).begin; else {
                        for (a = e; a < t; a++) delete s.validPositions[a];
                        n = e;
                    }
                    var p = new c.Event("keypress");
                    for (a = e; a < t; a++) {
                        p.keyCode = d[a].toString().charCodeAt(0), r.ignorable = !1;
                        var h = l.EventHandlers.keypressEvent.call(r, p, !0, !1, !1, n);
                        !1 !== h && void 0 !== h && (n = h.forwardPosition);
                    }
                    u.skipOptionalPartCharacter = f;
                }
                function v(e, t, i) {
                    var a = this, r = this.maskset, l = this.dependencyLib;
                    if (void 0 === e) for (e = t - 1; e > 0 && !r.validPositions[e]; e--) ;
                    for (var s = e; s < t; s++) {
                        if (void 0 === r.validPositions[s] && !o.isMask.call(a, s, !1)) if (0 == s ? n.getTest.call(a, s) : r.validPositions[s - 1]) {
                            var u = n.getTests.call(a, s).slice();
                            "" === u[u.length - 1].match.def && u.pop();
                            var c, f = n.determineTestTemplate.call(a, s, u);
                            if (f && (!0 !== f.match.jit || "master" === f.match.newBlockMarker && (c = r.validPositions[s + 1]) && !0 === c.match.optionalQuantifier) && ((f = l.extend({}, f, {
                                input: n.getPlaceholder.call(a, s, f.match, !0) || f.match.def
                            })).generatedInput = !0, m.call(a, s, f, !0), !0 !== i)) {
                                var p = r.validPositions[t].input;
                                return r.validPositions[t] = void 0, d.call(a, t, p, !0, !0);
                            }
                        }
                    }
                }
                function m(e, t, i, a) {
                    var r = this, l = this.maskset, s = this.opts, u = this.dependencyLib;
                    function c(e, t, i) {
                        var a = t[e];
                        if (void 0 !== a && !0 === a.match.static && !0 !== a.match.optionality && (void 0 === t[0] || void 0 === t[0].alternation)) {
                            var n = i.begin <= e - 1 ? t[e - 1] && !0 === t[e - 1].match.static && t[e - 1] : t[e - 1], r = i.end > e + 1 ? t[e + 1] && !0 === t[e + 1].match.static && t[e + 1] : t[e + 1];
                            return n && r;
                        }
                        return !1;
                    }
                    var f = 0, h = void 0 !== e.begin ? e.begin : e, v = void 0 !== e.end ? e.end : e, m = !0;
                    if (e.begin > e.end && (h = e.end, v = e.begin), a = void 0 !== a ? a : h, h !== v || s.insertMode && void 0 !== l.validPositions[a] && void 0 === i || void 0 === t || t.match.optionalQuantifier || t.match.optionality) {
                        var g, k = u.extend(!0, {}, l.validPositions), y = o.getLastValidPosition.call(r, void 0, !0);
                        for (l.p = h, g = y; g >= h; g--) delete l.validPositions[g], void 0 === t && delete l.tests[g + 1];
                        var b, x, P = a, E = P;
                        for (t && (l.validPositions[a] = u.extend(!0, {}, t), E++, P++), g = t ? v : v - 1; g <= y; g++) {
                            if (void 0 !== (b = k[g]) && !0 !== b.generatedInput && (g >= v || g >= h && c(g, k, {
                                begin: h,
                                end: v
                            }))) {
                                for (;"" !== n.getTest.call(r, E).match.def; ) {
                                    if (!1 !== (x = p.call(r, E, b, s)) || "+" === b.match.def) {
                                        "+" === b.match.def && o.getBuffer.call(r, !0);
                                        var S = d.call(r, E, b.input, "+" !== b.match.def, !0);
                                        if (m = !1 !== S, P = (S.pos || E) + 1, !m && x) break;
                                    } else m = !1;
                                    if (m) {
                                        void 0 === t && b.match.static && g === e.begin && f++;
                                        break;
                                    }
                                    if (!m && o.getBuffer.call(r), E > l.maskLength) break;
                                    E++;
                                }
                                "" == n.getTest.call(r, E).match.def && (m = !1), E = P;
                            }
                            if (!m) break;
                        }
                        if (!m) return l.validPositions = u.extend(!0, {}, k), o.resetMaskSet.call(r, !0),
                            !1;
                    } else t && n.getTest.call(r, a).match.cd === t.match.cd && (l.validPositions[a] = u.extend(!0, {}, t));
                    return o.resetMaskSet.call(r, !0), f;
                }
            },
            5581: function(e) {
                e.exports = JSON.parse('{"BACKSPACE":8,"BACKSPACE_SAFARI":127,"DELETE":46,"DOWN":40,"END":35,"ENTER":13,"ESCAPE":27,"HOME":36,"INSERT":45,"LEFT":37,"PAGE_DOWN":34,"PAGE_UP":33,"RIGHT":39,"SPACE":32,"TAB":9,"UP":38,"X":88,"Z":90,"CONTROL":17,"PAUSE/BREAK":19,"WINDOWS_LEFT":91,"WINDOWS_RIGHT":92,"KEY_229":229}');
            }
        }, t = {};
        function i(a) {
            var n = t[a];
            if (void 0 !== n) return n.exports;
            var r = t[a] = {
                exports: {}
            };
            return e[a](r, r.exports, i), r.exports;
        }
        var a = {};
        return function() {
            var e, t = a;
            Object.defineProperty(t, "__esModule", {
                value: !0
            }), t.default = void 0, i(3851), i(219), i(207), i(5296);
            var n = ((e = i(2394)) && e.__esModule ? e : {
                default: e
            }).default;
            t.default = n;
        }(), a;
    }();
}));
/*
 Input Mask plugin binding
 http://github.com/RobinHerbots/jquery.inputmask
 Copyright (c) Robin Herbots
 Licensed under the MIT license
 */
(function (factory) {
    factory(jQuery, window.Inputmask, window);
}
(function ($, Inputmask, window) {
    $(window.document).ajaxComplete(function (event, xmlHttpRequest, ajaxOptions) {
        if ($.inArray("html", ajaxOptions.dataTypes) !== -1) {
            $(".inputmask, [data-inputmask], [data-inputmask-mask], [data-inputmask-alias], [data-inputmask-regex]").each(function (ndx, lmnt) {
                if (lmnt.inputmask === undefined) {
                    Inputmask().mask(lmnt);
                }
            });
        }
    }).ready(function () {
        $(".inputmask, [data-inputmask], [data-inputmask-mask], [data-inputmask-alias],[data-inputmask-regex]").each(function (ndx, lmnt) {
            if (lmnt.inputmask === undefined) {
                Inputmask().mask(lmnt);
            }
        });
    });
}));

var tns=function(){var t=window,Ai=t.requestAnimationFrame||t.webkitRequestAnimationFrame||t.mozRequestAnimationFrame||t.msRequestAnimationFrame||function(t){return setTimeout(t,16)},e=window,Ni=e.cancelAnimationFrame||e.mozCancelAnimationFrame||function(t){clearTimeout(t)};function Li(){for(var t,e,n,i=arguments[0]||{},a=1,r=arguments.length;a<r;a++)if(null!==(t=arguments[a]))for(e in t)i!==(n=t[e])&&void 0!==n&&(i[e]=n);return i}function Bi(t){return 0<=["true","false"].indexOf(t)?JSON.parse(t):t}function Si(t,e,n,i){if(i)try{t.setItem(e,n)}catch(t){}return n}function Hi(){var t=document,e=t.body;return e||((e=t.createElement("body")).fake=!0),e}var n=document.documentElement;function Oi(t){var e="";return t.fake&&(e=n.style.overflow,t.style.background="",t.style.overflow=n.style.overflow="hidden",n.appendChild(t)),e}function Di(t,e){t.fake&&(t.remove(),n.style.overflow=e,n.offsetHeight)}function ki(t,e,n,i){"insertRule"in t?t.insertRule(e+"{"+n+"}",i):t.addRule(e,n,i)}function Ri(t){return("insertRule"in t?t.cssRules:t.rules).length}function Ii(t,e,n){for(var i=0,a=t.length;i<a;i++)e.call(n,t[i],i)}var i="classList"in document.createElement("_"),Pi=i?function(t,e){return t.classList.contains(e)}:function(t,e){return 0<=t.className.indexOf(e)},zi=i?function(t,e){Pi(t,e)||t.classList.add(e)}:function(t,e){Pi(t,e)||(t.className+=" "+e)},Wi=i?function(t,e){Pi(t,e)&&t.classList.remove(e)}:function(t,e){Pi(t,e)&&(t.className=t.className.replace(e,""))};function qi(t,e){return t.hasAttribute(e)}function Fi(t,e){return t.getAttribute(e)}function r(t){return void 0!==t.item}function ji(t,e){if(t=r(t)||t instanceof Array?t:[t],"[object Object]"===Object.prototype.toString.call(e))for(var n=t.length;n--;)for(var i in e)t[n].setAttribute(i,e[i])}function Vi(t,e){t=r(t)||t instanceof Array?t:[t];for(var n=(e=e instanceof Array?e:[e]).length,i=t.length;i--;)for(var a=n;a--;)t[i].removeAttribute(e[a])}function Gi(t){for(var e=[],n=0,i=t.length;n<i;n++)e.push(t[n]);return e}function Qi(t,e){"none"!==t.style.display&&(t.style.display="none")}function Xi(t,e){"none"===t.style.display&&(t.style.display="")}function Yi(t){return"none"!==window.getComputedStyle(t).display}function Ki(e){if("string"==typeof e){var n=[e],i=e.charAt(0).toUpperCase()+e.substr(1);["Webkit","Moz","ms","O"].forEach(function(t){"ms"===t&&"transform"!==e||n.push(t+i)}),e=n}for(var t=document.createElement("fakeelement"),a=(e.length,0);a<e.length;a++){var r=e[a];if(void 0!==t.style[r])return r}return!1}function Ji(t,e){var n=!1;return/^Webkit/.test(t)?n="webkit"+e+"End":/^O/.test(t)?n="o"+e+"End":t&&(n=e.toLowerCase()+"end"),n}var a=!1;try{var o=Object.defineProperty({},"passive",{get:function(){a=!0}});window.addEventListener("test",null,o)}catch(t){}var u=!!a&&{passive:!0};function Ui(t,e,n){for(var i in e){var a=0<=["touchstart","touchmove"].indexOf(i)&&!n&&u;t.addEventListener(i,e[i],a)}}function _i(t,e){for(var n in e){var i=0<=["touchstart","touchmove"].indexOf(n)&&u;t.removeEventListener(n,e[n],i)}}function Zi(){return{topics:{},on:function(t,e){this.topics[t]=this.topics[t]||[],this.topics[t].push(e)},off:function(t,e){if(this.topics[t])for(var n=0;n<this.topics[t].length;n++)if(this.topics[t][n]===e){this.topics[t].splice(n,1);break}},emit:function(e,n){n.type=e,this.topics[e]&&this.topics[e].forEach(function(t){t(n,e)})}}}Object.keys||(Object.keys=function(t){var e=[];for(var n in t)Object.prototype.hasOwnProperty.call(t,n)&&e.push(n);return e}),"remove"in Element.prototype||(Element.prototype.remove=function(){this.parentNode&&this.parentNode.removeChild(this)});var $i=function(H){H=Li({container:".slider",mode:"carousel",axis:"horizontal",items:1,gutter:0,edgePadding:0,fixedWidth:!1,autoWidth:!1,viewportMax:!1,slideBy:1,center:!1,controls:!0,controlsPosition:"top",controlsText:["prev","next"],controlsContainer:!1,prevButton:!1,nextButton:!1,nav:!0,navPosition:"top",navContainer:!1,navAsThumbnails:!1,arrowKeys:!1,speed:300,autoplay:!1,autoplayPosition:"top",autoplayTimeout:5e3,autoplayDirection:"forward",autoplayText:["start","stop"],autoplayHoverPause:!1,autoplayButton:!1,autoplayButtonOutput:!0,autoplayResetOnVisibility:!0,animateIn:"tns-fadeIn",animateOut:"tns-fadeOut",animateNormal:"tns-normal",animateDelay:!1,loop:!0,rewind:!1,autoHeight:!1,responsive:!1,lazyload:!1,lazyloadSelector:".tns-lazy-img",touch:!0,mouseDrag:!1,swipeAngle:15,nested:!1,preventActionWhenRunning:!1,preventScrollOnTouch:!1,freezable:!0,onInit:!1,useLocalStorage:!0,nonce:!1},H||{});var O=document,m=window,a={ENTER:13,SPACE:32,LEFT:37,RIGHT:39},e={},n=H.useLocalStorage;if(n){var t=navigator.userAgent,i=new Date;try{(e=m.localStorage)?(e.setItem(i,i),n=e.getItem(i)==i,e.removeItem(i)):n=!1,n||(e={})}catch(t){n=!1}n&&(e.tnsApp&&e.tnsApp!==t&&["tC","tPL","tMQ","tTf","t3D","tTDu","tTDe","tADu","tADe","tTE","tAE"].forEach(function(t){e.removeItem(t)}),localStorage.tnsApp=t)}var y=e.tC?Bi(e.tC):Si(e,"tC",function(){var t=document,e=Hi(),n=Oi(e),i=t.createElement("div"),a=!1;e.appendChild(i);try{for(var r,o="(10px * 10)",u=["calc"+o,"-moz-calc"+o,"-webkit-calc"+o],l=0;l<3;l++)if(r=u[l],i.style.width=r,100===i.offsetWidth){a=r.replace(o,"");break}}catch(t){}return e.fake?Di(e,n):i.remove(),a}(),n),g=e.tPL?Bi(e.tPL):Si(e,"tPL",function(){var t,e=document,n=Hi(),i=Oi(n),a=e.createElement("div"),r=e.createElement("div"),o="";a.className="tns-t-subp2",r.className="tns-t-ct";for(var u=0;u<70;u++)o+="<div></div>";return r.innerHTML=o,a.appendChild(r),n.appendChild(a),t=Math.abs(a.getBoundingClientRect().left-r.children[67].getBoundingClientRect().left)<2,n.fake?Di(n,i):a.remove(),t}(),n),D=e.tMQ?Bi(e.tMQ):Si(e,"tMQ",function(){if(window.matchMedia||window.msMatchMedia)return!0;var t,e=document,n=Hi(),i=Oi(n),a=e.createElement("div"),r=e.createElement("style"),o="@media all and (min-width:1px){.tns-mq-test{position:absolute}}";return r.type="text/css",a.className="tns-mq-test",n.appendChild(r),n.appendChild(a),r.styleSheet?r.styleSheet.cssText=o:r.appendChild(e.createTextNode(o)),t=window.getComputedStyle?window.getComputedStyle(a).position:a.currentStyle.position,n.fake?Di(n,i):a.remove(),"absolute"===t}(),n),r=e.tTf?Bi(e.tTf):Si(e,"tTf",Ki("transform"),n),o=e.t3D?Bi(e.t3D):Si(e,"t3D",function(t){if(!t)return!1;if(!window.getComputedStyle)return!1;var e,n=document,i=Hi(),a=Oi(i),r=n.createElement("p"),o=9<t.length?"-"+t.slice(0,-9).toLowerCase()+"-":"";return o+="transform",i.insertBefore(r,null),r.style[t]="translate3d(1px,1px,1px)",e=window.getComputedStyle(r).getPropertyValue(o),i.fake?Di(i,a):r.remove(),void 0!==e&&0<e.length&&"none"!==e}(r),n),x=e.tTDu?Bi(e.tTDu):Si(e,"tTDu",Ki("transitionDuration"),n),u=e.tTDe?Bi(e.tTDe):Si(e,"tTDe",Ki("transitionDelay"),n),b=e.tADu?Bi(e.tADu):Si(e,"tADu",Ki("animationDuration"),n),l=e.tADe?Bi(e.tADe):Si(e,"tADe",Ki("animationDelay"),n),s=e.tTE?Bi(e.tTE):Si(e,"tTE",Ji(x,"Transition"),n),c=e.tAE?Bi(e.tAE):Si(e,"tAE",Ji(b,"Animation"),n),f=m.console&&"function"==typeof m.console.warn,d=["container","controlsContainer","prevButton","nextButton","navContainer","autoplayButton"],v={};if(d.forEach(function(t){if("string"==typeof H[t]){var e=H[t],n=O.querySelector(e);if(v[t]=e,!n||!n.nodeName)return void(f&&console.warn("Can't find",H[t]));H[t]=n}}),!(H.container.children.length<1)){var k=H.responsive,R=H.nested,I="carousel"===H.mode;if(k){0 in k&&(H=Li(H,k[0]),delete k[0]);var p={};for(var h in k){var w=k[h];w="number"==typeof w?{items:w}:w,p[h]=w}k=p,p=null}if(I||function t(e){for(var n in e)I||("slideBy"===n&&(e[n]="page"),"edgePadding"===n&&(e[n]=!1),"autoHeight"===n&&(e[n]=!1)),"responsive"===n&&t(e[n])}(H),!I){H.axis="horizontal",H.slideBy="page",H.edgePadding=!1;var P=H.animateIn,z=H.animateOut,C=H.animateDelay,W=H.animateNormal}var M,q,F="horizontal"===H.axis,T=O.createElement("div"),j=O.createElement("div"),V=H.container,E=V.parentNode,A=V.outerHTML,G=V.children,Q=G.length,X=rn(),Y=!1;k&&En(),I&&(V.className+=" tns-vpfix");var N,L,B,S,K,J,U,_,Z,$=H.autoWidth,tt=sn("fixedWidth"),et=sn("edgePadding"),nt=sn("gutter"),it=un(),at=sn("center"),rt=$?1:Math.floor(sn("items")),ot=sn("slideBy"),ut=H.viewportMax||H.fixedWidthViewportWidth,lt=sn("arrowKeys"),st=sn("speed"),ct=H.rewind,ft=!ct&&H.loop,dt=sn("autoHeight"),vt=sn("controls"),pt=sn("controlsText"),ht=sn("nav"),mt=sn("touch"),yt=sn("mouseDrag"),gt=sn("autoplay"),xt=sn("autoplayTimeout"),bt=sn("autoplayText"),wt=sn("autoplayHoverPause"),Ct=sn("autoplayResetOnVisibility"),Mt=(U=null,_=sn("nonce"),Z=document.createElement("style"),U&&Z.setAttribute("media",U),_&&Z.setAttribute("nonce",_),document.querySelector("head").appendChild(Z),Z.sheet?Z.sheet:Z.styleSheet),Tt=H.lazyload,Et=H.lazyloadSelector,At=[],Nt=ft?(K=function(){{if($||tt&&!ut)return Q-1;var t=tt?"fixedWidth":"items",e=[];if((tt||H[t]<Q)&&e.push(H[t]),k)for(var n in k){var i=k[n][t];i&&(tt||i<Q)&&e.push(i)}return e.length||e.push(0),Math.ceil(tt?ut/Math.min.apply(null,e):Math.max.apply(null,e))}}(),J=I?Math.ceil((5*K-Q)/2):4*K-Q,J=Math.max(K,J),ln("edgePadding")?J+1:J):0,Lt=I?Q+2*Nt:Q+Nt,Bt=!(!tt&&!$||ft),St=tt?_n():null,Ht=!I||!ft,Ot=F?"left":"top",Dt="",kt="",Rt=tt?function(){return at&&!ft?Q-1:Math.ceil(-St/(tt+nt))}:$?function(){for(var t=0;t<Lt;t++)if(N[t]>=-St)return t}:function(){return at&&I&&!ft?Q-1:ft||I?Math.max(0,Lt-Math.ceil(rt)):Lt-1},It=en(sn("startIndex")),Pt=It,zt=(tn(),0),Wt=$?null:Rt(