<?php

namespace ViartasShop\Products\Controllers\Accounting;

use ViartasCore\Core\Forms\Common\ImagesUploadForm;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use ViartasCore\Core\Controllers\AccountingController;
use ViartasShop\Products\Forms\Accounting\ProductEditForm;
use ViartasShop\Products\Models\Accounting\Product;
use ViartasShop\Products\Tables\Accounting\ProductsTable;

class ProductsController extends AccountingController
{
    /**
     * @param ProductsTable $table
     * @param Request $request
     * @return string|array|View|\Illuminate\Foundation\Application|Factory|Application
     */
    public function index(ProductsTable $table, Request $request): string|array|View|\Illuminate\Foundation\Application|Factory|Application
    {
        return $this->tableBuilder(
            $table, $request, theme_view('accounting.products.index', compact('table'))
        );
    }

    /**
     * @param ProductEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function create(ProductEditForm $form): Application|Factory|View|\Illuminate\Foundation\Application
    {
        return theme_view('accounting.products.edit', [
            'form' => $form->build(),
            'element' => new Product(),
        ]);
    }

    /**
     * @param ProductEditForm $form
     * @return RedirectResponse
     */
    public function store(ProductEditForm $form): RedirectResponse
    {
        $page = Product::create($form->all());

        return theme_redirect()->route('accounting.products.edit', [
            'id' => $page->id,
        ])->with('success', __('product.created'));
    }

    /**
     * @param string $id
     * @param ProductEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function edit(string $id, ProductEditForm $form): Application|Factory|View|\Illuminate\Foundation\Application
    {
        $product_category = Product::findOrFail($id);
        $form = $form->setAction('update')->build($product_category);

        return theme_view('accounting.products.edit', [
            'form' => $form,
            'element' => $product_category,
        ]);
    }

    /**
     * @param string $id
     * @param ProductEditForm $form
     * @return RedirectResponse
     */
    public function update(string $id, ProductEditForm $form): RedirectResponse
    {
        Product::findOrFail($id)->update($form->all());

        return theme_redirect()->route('accounting.products.edit', [
            'id' => $id,
        ])->with('success', __('product.updated'));
    }
    /**
     * @param string $id
     * @param Request $request
     * @return void
     */
    public function updateOneField(string $id, Request $request): void
    {
        Product::findOrFail($id)->update([
            $request->get('field') => $request->get('value')
        ]);
    }

    /**
     * @param string $id
     * @param Request $request
     * @return bool|RedirectResponse
     */
    public function destroy(string $id, Request $request): bool|RedirectResponse
    {
        if ($id !== '0') {
            Product::findOrFail($id)->delete();

            return theme_redirect()->route('accounting.products.index')
                ->with('success', __('product.deleted'));
        }

        $this->destroySelected($request);

        return true;
    }

    /**
     * @param Request $request
     * @return void
     */
    public function destroySelected(Request $request): void
    {
        foreach ($request->elements as $element) {
            Product::find($element)->delete();
        }

        Session::flash('success', __('elements.selected.deleted'));
    }

    public function uploadImages($id, ImagesUploadForm $images)
    {
        $page = Product::findOrFail($id);
        $images = $images->build();

        $images->field('images')
            ->setModel(Product::class)
            ->setPrimaryKey('id')
            ->setPrimaryValue($id);

        return theme_view('accounting.products.common', [
            'form' => $images,
            'element' => $page,
        ]);
    }

}
