<?php

namespace ViartasShop\ProductCategories\Models\Accounting;

use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Route;
use ViartasCore\Core\Drivers\RouteDriver;
use ViartasCore\Core\Facades\Viartas;
use ViartasCore\Core\Models\BaseModel;
use ViartasCore\Core\Traits\DefinesAsCacheItem;
use ViartasCore\Core\Traits\HasMetaTags;
use ViartasShop\Attributes\Models\Accounting\Attribute;

class ProductCategory extends BaseModel
{
    use DefinesAsCacheItem, HasMetaTags;

    /**
     * @var string[]
     */
    protected $fillable = [
        'title', 'content', 'slug',
        'user_id', 'product_category_id',
        'is_visible', 'layout', 'show_level',
    ];

    /**
     * @var string
     */
    protected string $module = 'product-categories';

    /**
     * @var string
     */
    protected string $layout = 'default';

    /**
     * @var string
     */
    protected string $element_title = 'Product category';

    /**
     * @return string
     */
    public function getParentKeyName(): string
    {
        return 'product_category_id';
    }

    /**
     * @param Collection $menu
     * @return bool
     */
    public function menuRouteActive(Collection $menu): bool
    {
        if (Route::currentRouteName() == $menu->get('route')->getRouteName()) {
            return true;
        }

        if ($menu->get('route')->getRouteName() == 'accounting.product-categories.attributes') {

            if (Route::currentRouteName() == 'accounting.product-categories.attributes-settings') {
                return true;
            }

            if (Route::currentRouteName() == 'accounting.product-categories.attributes-settings') {
                return true;
            }

            if (Route::currentRouteName() == 'accounting.product-categories.attributes-order') {
                return true;
            }
        }

        return false;
    }

    /**
     * @return RouteDriver
     */
    public function routeAttributes(): RouteDriver
    {
        $route = sprintf('accounting.%s.attributes', $this->module);

        return new RouteDriver($route, [
            'id' => $this->id,
        ]);
    }

    /**
     * @return RouteDriver
     */
    public function routeAttributesSettings(): RouteDriver
    {
        $route = sprintf('accounting.%s.attributes-settings', $this->module);

        return new RouteDriver($route, [
            'id' => $this->id,
        ]);
    }

    /**
     * @return RouteDriver
     */
    public function routeAttributesOrder(): RouteDriver
    {
        $route = sprintf('accounting.%s.attributes-order', $this->module);

        return new RouteDriver($route, [
            'id' => $this->id,
        ]);
    }

    /**
     * @return array
     */
    protected function menuConfig(): array
    {
        $config = [
            'edit' => $this->menuItem(__('Main settings'), $this->routeEdit()),
            'attributes' => $this->menuItem(__('Attributes'), $this->routeAttributes()),
            'upload_documents' => $this->menuItem(__('Upload documents'), $this->routeEdit()),
            'documents' => $this->menuItem(__('Documents'), $this->routeEdit()),
            'comments' => $this->menuItem(__('Comments'), $this->routeEdit()),
            'activity' => $this->menuItem(__('Activity'), $this->routeEdit()),
        ];

        if ($this->usesMedia()) {
            $config['upload_images'] = $this->menuItem(__('Upload images'), $this->routeUploadImages());
            $config['images'] = $this->menuItem(__('Images'), $this->routeUploadImages());
        }

        return $config;
    }

    /**
     * @return BelongsToMany
     */
    public function attrs(): BelongsToMany
    {
        return $this->belongsToMany(Attribute::class, 'attribute_product_category')
            ->using(AttributeProductCategory::class)
            ->withPivot([
                'sort',
                'is_visible_in_product',
                'is_visible_in_category',
                'is_filterable_lg',
                'is_filterable_sm',
            ])->withTimestamps();
    }

    /**
     * @return \Illuminate\Foundation\Application|View|Factory|Application
     */
    public function view(): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $app = Viartas::driver()->app()->current()->tag;
        //load views
        //check if template has
        //load view
        $element = $this;
        //make in base model, return app.module.default
        return theme_view($app.'.'.$this->module.'.'.$this->layout, compact('element'));
    }

    /**
     * @return string
     */
    public function controller(): string
    {
        return 'ViartasShop\ProductCategories\Controllers\Web\ProductCategoriesController@index';
    }

    /**
     * @param string|null $locale
     * @return string
     */
    public function generateRoute(?string $locale = null): string
    {
        $segments = [];

        if ($locale) {
            if (Viartas::driver()->locale()->default()->tag != $locale) {
                $segments[] = $locale;
            }
        }

        $child = $this->ancestorsAndSelf()->get()->toTree()->first();

        while ($child !== false) {
            $element = $child;

            if (! $locale) {
                $locale = Viartas::driver()->locale()->current()->tag;
            }

            $segments[] = $element->getTranslation('slug', $locale);

            if ($element->children->count()) {
                $child = $element->children->first();
            } else {
                $child = false;
            }
        }

        return implode('/', $segments);
    }
}
