<?php

namespace ViartasShop\ProductCategories\Controllers\Accounting;

use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use ViartasCore\Core\Controllers\AccountingController;
use ViartasCore\Core\Forms\Common\ImagesUploadForm;
use ViartasShop\Attributes\Models\Accounting\Attribute;
use ViartasShop\ProductCategories\Forms\Accounting\AttributesSettingsForm;
use ViartasShop\ProductCategories\Forms\Accounting\ProductCategoriesEditForm;
use ViartasShop\ProductCategories\Models\Accounting\ProductCategory;
use ViartasShop\ProductCategories\Tables\Accounting\ProductCategoriesTable;
use ViartasSupport\Number\Number;

class ProductCategoriesController extends AccountingController
{
    /**
     * @param ProductCategoriesTable $table
     * @param Request $request
     * @return string|array|View|\Illuminate\Foundation\Application|Factory|Application
     */
    public function index(ProductCategoriesTable $table, Request $request): string|array|View|\Illuminate\Foundation\Application|Factory|Application
    {
        return $this->tableBuilder(
            $table, $request, theme_view('accounting.product-categories.index', compact('table'))
        );
    }

    /**
     * @param ProductCategoriesEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function create(ProductCategoriesEditForm $form): View|Factory|Application|\Illuminate\Foundation\Application
    {
        return theme_view('accounting.product-categories.edit', [
            'form' => $form->build(),
            'element' => new ProductCategory(),
        ]);
    }

    /**
     * @param ProductCategoriesEditForm $form
     * @return RedirectResponse
     */
    public function store(ProductCategoriesEditForm $form): RedirectResponse
    {
        $category = ProductCategory::create($form->all());

        return theme_redirect()->route('accounting.product-categories.edit', [
            'id' => $category->id,
        ])->with('success', __('category.created'));
    }

    /**
     * @param string $id
     * @param ProductCategoriesEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function edit(string $id, ProductCategoriesEditForm $form): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $category = ProductCategory::findOrFail($id);
        $form = $form->setAction('update')->build($category);

        return theme_view('accounting.product-categories.edit', [
            'form' => $form,
            'element' => $category,
        ]);
    }

    /**
     * @param string $id
     * @param ProductCategoriesEditForm $form
     * @return RedirectResponse
     */
    public function update(string $id, ProductCategoriesEditForm $form): RedirectResponse
    {
        ProductCategory::findOrFail($id)->update($form->all());

        return theme_redirect()->route('accounting.product-categories.edit', [
            'id' => $id,
        ])->with('success', __('category.updated'));
    }

    /**
     * @param string $id
     * @param Request $request
     * @return void
     */
    public function updateOneField(string $id, Request $request): void
    {
        ProductCategory::findOrFail($id)->update([
            $request->get('field') => $request->get('value')
        ]);
    }

    /**
     * @param string $id
     * @param ProductCategoriesEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function attributes(string $id, ProductCategoriesEditForm $form): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $category = ProductCategory::findOrFail($id);
        $attributes = Attribute::query()->with('group')
            ->get()
            ->groupBy(fn ($attr) => $attr->group->title ?? 'Ungrouped')
            ->sortKeys();

        $form = $form->setAction('update')->build($category);

        return theme_view('accounting.product-categories.attributes', [
            'form' => $form,
            'element' => $category,
            'attributes' => $attributes,
        ]);
    }

    /**
     * @param int $id
     * @param Request $request
     * @return RedirectResponse
     */
    public function assignAttributes(int $id, Request $request): RedirectResponse
    {
        $category = ProductCategory::findOrFail($id);
        $category->attrs()->sync(
            collect($request->input('kt_ecommerce_add_product_options', []))
                ->pluck('product_option')
                ->filter()
                ->map(fn ($v) => (int) $v)
                ->unique()
                ->values()
                ->all()
        );

        return theme_redirect()->route('accounting.product-categories.attributes', [
            'id' => $id,
        ])->with('success', __('attributes.updated'));
    }

    /**
     * @param string $id
     * @param AttributesSettingsForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function attributesSettings(string $id, AttributesSettingsForm $form): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $category = ProductCategory::findOrFail($id);
        $attributes = Attribute::query()->with('group')
            ->get()
            ->groupBy(fn ($attr) => $attr->group->title ?? 'Ungrouped')
            ->sortKeys();

        $attrs = [];
        foreach ($category->attrs as $attribute) {
            $attrs['settings.'.$attribute->id.'.is_visible_in_product'] = $attribute->pivot->is_visible_in_product ?? false;
            $attrs['settings.'.$attribute->id.'.is_visible_in_category'] = $attribute->pivot->is_visible_in_category ?? false;
            $attrs['settings.'.$attribute->id.'.is_filterable_lg'] = $attribute->pivot->is_filterable_lg ?? false;
            $attrs['settings.'.$attribute->id.'.is_filterable_sm'] = $attribute->pivot->is_filterable_sm ?? false;
        }

        $category->settings = $attrs;

        $form = $form->setAction('update')->build($category);

        return theme_view('accounting.product-categories.attributes-settings', [
            'form' => $form,
            'element' => $category,
            'attributes' => $attributes,
        ]);
    }

    /**
     * @param string $id
     * @param AttributesSettingsForm $form
     * @return RedirectResponse
     */
    public function attributesSettingsSave(string $id, AttributesSettingsForm $form): RedirectResponse
    {
        $category = ProductCategory::findOrFail($id);

        foreach ($form->get('settings') as $attribute_id => $settings) {
            $category->attrs()->updateExistingPivot($attribute_id, [
                'is_visible_in_product' => $settings['is_visible_in_product'] ?? false,
                'is_visible_in_category' => $settings['is_visible_in_category'] ?? false,
                'is_filterable_lg' => $settings['is_filterable_lg'] ?? false,
                'is_filterable_sm' => $settings['is_filterable_sm'] ?? false,
            ]);
        }

        return theme_redirect()->route('accounting.product-categories.attributes-settings', [
            'id' => $id,
        ])->with('success', __('attribute.settings.updated'));
    }

    /**
     * @param string $id
     * @param ProductCategoriesEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function attributesOrder(string $id, ProductCategoriesEditForm $form): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $category = ProductCategory::findOrFail($id);
        $attributes = Attribute::query()->with('group')
            ->get()
            ->groupBy(fn ($attr) => $attr->group->title ?? 'Ungrouped')
            ->sortKeys();

        $form = $form->setAction('update')->build($category);

        return theme_view('accounting.product-categories.attributes-order', [
            'form' => $form,
            'element' => $category,
            'attributes' => $attributes,
        ]);
    }

    /**
     * @param string $id
     * @param Request $request
     * @return void
     */
    public function attributesOrderSave(string $id, Request $request): void
    {
        $productCategory = ProductCategory::findOrFail($id);

        foreach (json_decode($request->list) as $sort => $attribute) {
            $productCategory->attrs()->updateExistingPivot($attribute->id, [
                'sort' => (int) $sort,
            ]);
        }

        session()->flash('success', __('attributes.order.updated'));
    }

    /**
     * @param string $id
     * @param Request $request
     * @return bool|RedirectResponse
     */
    public function destroy(string $id, Request $request): bool|RedirectResponse
    {
        if ($id !== '0') {
            ProductCategory::findOrFail($id)->delete();

            return theme_redirect()->route('accounting.product-categories.index')
                ->with('success', __('category.deleted'));
        }

        $this->destroySelected($request);

        return true;
    }

    /**
     * @param Request $request
     * @return void
     */
    public function destroySelected(Request $request): void
    {
        foreach ($request->elements as $element) {
            ProductCategory::find($element)->delete();
        }

        Session::flash('success', __('elements.selected.deleted'));
    }

    /**
     * @param $id
     * @param ImagesUploadForm $images
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function uploadImages($id, ImagesUploadForm $images): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $category = ProductCategory::findOrFail($id);
        $images = $images->build();

        $images->field('images')
            ->setModel(ProductCategory::class)
            ->setPrimaryKey('id')
            ->setPrimaryValue($id);

        return theme_view('accounting.product-categories.common', [
            'form' => $images,
            'element' => $category,
        ]);
    }

}
