<?php

namespace ViartasShop\Cart;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cookie;
use ViartasShop\Cart\Models\Accounting\Cart;
use ViartasShop\Products\Models\Accounting\Product;

class ViartasCart
{
    /**
     * @var string
     */
    protected string $cookie_name = 'viartas_cart_test_2';

    /**
     * @var Cart
     */
    private Cart $cart;

    /**
     * @var int
     */
    //protected int $cookie_lifetime = 60 * 24 * 30;
    protected int $cookie_lifetime = 60;

    /**
     *
     */
    public function __construct()
    {

    }

    /**
     * @return string
     */
    protected function create(): string
    {
        $cart = Cart::create([
            'user_id' => Auth::id(),
        ]);

        $cart->hash = $cart->id.'-'.\Str::random(32);
        $cart->save();

        // Queue the cookie (it will be encrypted by EncryptCookies middleware)
        Cookie::queue(
            cookie(
                name: $this->cookie_name,
                value: $cart->hash,
                minutes: $this->cookie_lifetime,
                path: '/',
                domain: null,              // e.g. '.example.com' if needed
                secure: request()->isSecure(),
                httpOnly: true,
                raw: false,
                sameSite: 'lax'            // use 'none' + secure=true if cross-site
            )
        );

        return $cart->hash;

        // NOTE: Do not try to read it now; it becomes available next request.
    }

    /**
     * @param string $hash
     * @return $this
     */
    public function access(string $hash): self
    {
        $this->cart = Cart::where('hash', $hash)->firstOrFail();
        return $this;
    }

    /**
     * @return $this
     */
    public function init(): self
    {
        if (! request()->hasCookie($this->cookie_name)) {
            $hash = $this->create();
        } else {
            $hash = request()->cookie($this->cookie_name);
        }

        return $this->access($hash);
    }

    /**
     * @return mixed
     */
    public function hash(): string
    {
        return $this->cart->hash;
    }

    /**
     * @return mixed
     */
    public function user(): mixed
    {
        return $this->cart->user;
    }

    /**
     * @return mixed
     */
    public function id(): mixed
    {
        return $this->cart->id;
    }

    /**
     * @return mixed
     */
    public function created_at(): mixed
    {
        return $this->cart->created_at;
    }

    /**
     * @return mixed
     */
    public function updated_at(): mixed
    {
        return $this->cart->updated_at;
    }

    /**
     * @return mixed
     */
    public function products(): mixed
    {
        return $this->cart->cartProducts;
    }

    /**
     * @return int
     */
    public function totalQuantity(): int
    {
        return $this->cart->cartProducts()->sum('quantity');
    }

    /**
     * @return float
     */
    public function totalPrice(): float
    {
        return $this->cart->cartProducts()->sum('price');
    }

    public function isEmpty(): bool
    {
        return $this->cart->cartProducts()->count() === 0;

    }

    /**
     * @return $this
     */
    public function facade(): self
    {
        return $this;
    }

    /**
     * @param Product $product
     * @param int $quantity
     * @param array $attributes
     * @param array $options
     * @param array $modifiers
     * @return Model
     */
    public function add(Product $product, int $quantity = 1, string $comments = '', array $attributes = [], array $options = [], array $modifiers = []): Model
    {
        $cartProduct = $this->cart->cartProducts()->where('product_id', $product->id)->first();
        if (! $cartProduct) {
            return $this->cart->cartProducts()->create([
                'product_id' => $product->id,
                'quantity' => $quantity,
                'price' => $product->price,
                'comment' => $comments,
            ]);
        }

        $cartProduct->quantity = $cartProduct->quantity->add($quantity);
        $cartProduct->save();

        return $cartProduct;
    }
}
