<?php

namespace ViartasShop\Cart\Tables\Accounting;

use ViartasBuilders\Tables\DataTables;
use ViartasBuilders\Tables\DataTablesButton;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use ViartasBuilders\Tables\DataTablesRow;
use ViartasCore\Core\Facades\Viartas;
use ViartasShop\Products\Models\Accounting\Product;

class CartTable extends DataTables
{
    public function __construct(Request $request)
    {
        parent::__construct(
            new Product(),
            $request
        );

        $this->table();

        $this->addRows([
            $this->fieldId(),
            $this->fieldTitle(),
            $this->fieldParent(),
            $this->fieldUser(),
            $this->fieldCreatedAt(),
            $this->button()
                ->add((new DataTablesButton())->editButton())
                ->add((new DataTablesButton())->destroyButton()),
        ]);
    }

    public function table(): self
    {
        $this->setRoute(route('accounting.products.index'));
        $this->setFilters(route('accounting.products.index'));
        $this->setId('products-table');
        $this->setClass('products-table');
        $this->setTitle(__('Products'));
        $this->setIndexColumn('id');

        return $this;
    }
    public function fieldId(): DataTablesRow
    {
        $id = new DataTablesRow();

        $id->field('id')
            ->title('ID')
            ->setSearchable()
            ->render(function (Product $product) {
                return '#'.$product->id;
            })
            ->order(function (Builder $builder, string $direction) {
                return $builder->orderBy('id', $direction);
            });

        return $id;
    }
    public function fieldTitle(): DataTablesRow
    {
        $title = new DataTablesRow();

        $title->field('title')
            ->title(__('Title'))
            ->setSearchable()
            ->order(function (Builder $builder, string $direction) {
                $builder->orderBy('title->'.Viartas::driver()->locale()->current()->tag, $direction);
            });

        return $title;
    }
    public function fieldParent(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('product_category_id')
            ->title(__('Parent'))
            ->setSelectable()
            ->render(function (Product $product) {
                if ($product->ancestors->count()) {
                    $parents = '';
                    foreach ($product->ancestors as $ancestor) {
                        $parents .= '<span class="badge badge-primary mx-1">'.$ancestor->title.'</span>';

                        if ($ancestor->ancestors->count()) {
                            $parents .= '<i class="fas fa-arrow-alt-circle-left"></i>';
                        }
                    }

                    return $parents;
                }

                return '<span class="badge badge-secondary mx-1">'.__('Root').'</span>';
            })
            ->order(function (Builder $builder, string $direction) {
                return $builder->withAggregate('parent','title')
                    ->orderBy('parent_title', $direction);
            })
            ->filters(function (Product $product) {
                return array_merge([[
                    'value' => '0',
                    'label' => __('Root'),
                    'values' => [],
                ]], self::treeToFilters(
                    $product::orderBy('product_category_id')->tree()->get()->toTree(),
                    'id',
                    'title'
                ));
            })
            ->filter(function (Builder $builder, $searches) {
                foreach ($searches as $value => $key) {
                    if ($value == 0) {
                        $builder->whereNull('product_category_id');
                    } else {
                        $builder->where($key, 'like', '%'.$value.'%');
                    }
                }

                return $builder;
            });
    }
    public function fieldCreatedAt(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('created_at')
            ->title(__('Created at'))
            ->setRange(true)
            ->render(function (Product $product) {
                return $product->created_at->format("Y-m-d H:i:s");
            });
    }
}
