<?php

namespace ViartasShop\Cart\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Symfony\Component\Intl\Countries;
use Symfony\Component\Intl\Languages;

class ImportCountriesCommand extends Command
{
    protected $signature = 'countries:import
        {--vat= : Path to vat_rates.json (optional)}
        {--vat-only : Update only VAT fields for existing countries}';

    protected $description = 'Import countries with ISO, localized names, emoji flag, calling codes, IDD, and optional VAT rates. Use --vat-only to refresh VAT only.';

    public function handle(): int
    {
        // Optional VAT dataset (you maintain this; keyed by ISO2)
        $vatPath  = $this->option('vat') ?: __DIR__ . '/../../database/seeders/data/vat_rates.json';
        $vatData  = (is_file($vatPath) ? json_decode(file_get_contents($vatPath), true) : []) ?: [];

        if ($this->option('vat-only')) {
            // ---------- VAT-ONLY FAST UPDATE ----------
            if (empty($vatData)) {
                $this->warn('No VAT data found. Provide --vat=path/to/file.json');
                return self::SUCCESS;
            }

            $updated = 0; $missing = 0;
            foreach ($vatData as $iso2 => $entry) {
                $iso2 = strtoupper($iso2);
                $payload = [
                    'vat'                 => json_encode(($entry['rates'] ?? $entry['vat'] ?? $entry), JSON_UNESCAPED_UNICODE),
                    'vat_effective_date'  => $entry['effective_date'] ?? $entry['date'] ?? null,
                    'vat_source_url'      => $entry['source'] ?? $entry['source_url'] ?? null,
                    'updated_at'          => now(),
                ];

                $affected = DB::table('countries')->where('iso2', $iso2)->update($payload);
                if ($affected > 0) {
                    $updated += $affected;
                } else {
                    $missing++;
                }
            }

            $this->info("VAT updated for {$updated} countries. Missing (not in DB): {$missing}.");
            return self::SUCCESS;
        }

        // ---------- FULL IMPORT (idempotent; also updates VAT) ----------
        $candidates = [
            base_path('vendor/mledoze/countries/dist/countries.json'),
            base_path('vendor/mledoze/countries/data/countries.json'),
            base_path('vendor/mledoze/countries/countries.json'),
        ];
        $file = collect($candidates)->first(fn ($p) => is_file($p));
        if (!$file) {
            $this->error('countries.json not found (mledoze/countries).');
            return self::FAILURE;
        }
        $raw = json_decode(file_get_contents($file), true, 512, JSON_THROW_ON_ERROR);

        // Build translations from CLDR
        $languageCodes = array_keys(Languages::getNames());

        $rows = [];
        foreach ($raw as $c) {
            $iso2 = strtoupper($c['cca2'] ?? '');
            $iso3 = strtoupper($c['cca3'] ?? '');
            $num  = isset($c['ccn3']) && $c['ccn3'] !== '' ? (int) $c['ccn3'] : null;
            if (strlen($iso2) !== 2 || strlen($iso3) !== 3) continue;

            // Localized names
            $titles = [];
            foreach ($languageCodes as $lang) {
                try {
                    $name = Countries::getName($iso2, $lang);
                    if ($name) $titles[$lang] = $name;
                } catch (\Throwable $e) {}
            }
            if (!isset($titles['en'])) {
                try { $fallback = Countries::getName($iso2, 'en'); if ($fallback) $titles['en'] = $fallback; } catch (\Throwable $e) {}
            }
            if (empty($titles) && isset($c['name']['common'])) {
                $titles['en'] = $c['name']['common'];
            }

            // Emoji flag
            $emojiFlag = $c['flag'] ?? null;

            // IDD & calling codes
            $idd = [
                'root'     => $c['idd']['root'] ?? null,
                'suffixes' => $c['idd']['suffixes'] ?? [],
            ];
            $callingCodes = $c['callingCodes'] ?? null;
            if (!$callingCodes) {
                $callingCodes = [];
                if (!empty($idd['root']) && !empty($idd['suffixes'])) {
                    foreach ($idd['suffixes'] as $suf) {
                        $callingCodes[] = "{$idd['root']}{$suf}";
                    }
                }
                $callingCodes = array_values(array_unique(array_filter($callingCodes)));
            }

            // VAT (optional; merged if provided)
            $vatEntry    = $vatData[$iso2] ?? null;
            $vatPayload  = $vatEntry ? ($vatEntry['rates'] ?? $vatEntry['vat'] ?? $vatEntry) : null;
            $vatEffective= $vatEntry['effective_date'] ?? $vatEntry['date'] ?? null;
            $vatSource   = $vatEntry['source'] ?? $vatEntry['source_url'] ?? null;

            $rows[] = [
                'iso2'                => $iso2,
                'iso3'                => $iso3,
                'iso_numeric'         => $num,
                'title'               => json_encode($titles, JSON_UNESCAPED_UNICODE),
                'emoji_flag'          => $emojiFlag,
                'calling_codes'       => json_encode($callingCodes, JSON_UNESCAPED_UNICODE),
                'idd'                 => json_encode($idd, JSON_UNESCAPED_UNICODE),
                'vat'                 => $vatPayload ? json_encode($vatPayload, JSON_UNESCAPED_UNICODE) : null,
                'vat_effective_date'  => $vatEffective,
                'vat_source_url'      => $vatSource,
                'created_at'          => now(),
                'updated_at'          => now(),
            ];
        }

        foreach (array_chunk($rows, 500) as $chunk) {
            DB::table('countries')->upsert(
                $chunk,
                ['iso2'], // unique
                ['iso3','iso_numeric','title','emoji_flag','calling_codes','idd','vat','vat_effective_date','vat_source_url','updated_at']
            );
        }

        $this->info('Full import finished: '.count($rows).' countries processed.');
        return self::SUCCESS;
    }
}
