<?php

namespace ViartasShop\Brands\Tables\Accounting;

use ViartasBuilders\Tables\DataTables;
use ViartasBuilders\Tables\DataTablesButton;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use ViartasBuilders\Tables\DataTablesRow;
use ViartasCore\Core\Facades\Viartas;
use ViartasShop\Brands\Models\Accounting\Brand;
use ViartasShop\Brands\Models\Accounting\Model;
use ViartasShop\ProductCategories\Models\Accounting\ProductCategory;

class ModelsTable extends DataTables
{
    public function __construct(Request $request)
    {
        parent::__construct(
            new Model(),
            $request
        );

        $this->table();

        $this->addRows([
            $this->fieldUser(),
            $this->fieldId(),
            $this->fieldTitle(),
            $this->fieldSeries(),
            $this->fieldBrands(),
            $this->fieldCategories(),
            $this->fieldCreatedAt(),
            $this->button()
                ->add((new DataTablesButton())->editButton())
                ->add((new DataTablesButton())->destroyButton()),
        ]);
    }

    public function table(): self
    {
        $this->setRoute(route('accounting.models.index'));
        $this->setFilters(route('accounting.models.index'));
        $this->setId('models-table');
        $this->setClass('models-table');
        $this->setTitle(__('Models'));
        $this->setIndexColumn('id');

        return $this;
    }
    public function fieldId(): DataTablesRow
    {
        $id = new DataTablesRow();

        $id->field('id')
            ->title('ID')
            ->setSearchable()
            ->render(function (Model $model) {
                return '#'.$model->id;
            })
            ->order(function (Builder $builder, string $direction) {
                return $builder->orderBy('id', $direction);
            });

        return $id;
    }
    public function fieldTitle(): DataTablesRow
    {
        $title = new DataTablesRow();

        $title->field('title')
            ->title(__('Title'))
            ->setSearchable()
            ->order(function (Builder $builder, string $direction) {
                $builder->orderBy('title->'.Viartas::driver()->locale()->current()->tag, $direction);
            });

        return $title;
    }

    public function fieldSeries(): DataTablesRow
    {
        $title = new DataTablesRow();

        $title->field('series')
            ->title(__('Series'))
            ->setSearchable()
            ->order(function (Builder $builder, string $direction) {
                $builder->orderBy('series->'.Viartas::driver()->locale()->current()->tag, $direction);
            });

        return $title;
    }
    public function fieldBrands(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('brand_id')
            ->title(__('Brand'))
            ->setSelectable()
            ->render(function (Model $model) {
                return '<span class="badge badge-primary mx-1">'.$model->brand->title.'</span>';
            })
            ->order(function (Builder $builder, string $direction) {
                return $builder->withAggregate('parent','title')
                    ->orderBy('parent_title', $direction);
            })
            ->filters(function () {
                return array_merge([[
                    'value' => '0',
                    'label' => __('Root'),
                    'values' => [],
                ]], Brand::orderBy('title')
                    ->pluck('title', 'id')
                    ->map(fn ($title, $id) => [
                        'value'  => $id,
                        'label'  => $title,
                        'values' => [],
                    ])
                    ->values()
                    ->toArray()
                );
            })
            ->filter(function (Builder $builder, $searches) {
                foreach ($searches as $value => $key) {
                    if ($value == 0) {
                        $builder->whereNull('brand_id');
                    } else {
                        $builder->where($key, 'like', '%'.$value.'%');
                    }
                }

                return $builder;
            });
    }

    public function fieldCategories(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('product_category_id')
            ->title(__('Category'))
            ->setSelectable()
            ->render(function (Model $model) {
                if ($model->category) {
                    $parents = '<span class="badge badge-primary mx-1">'.$model->category->title.'</span>';

                    if ($model->category->ancestors->count()) {
                        $parents .= '<i class="fas fa-arrow-alt-circle-left"></i>';
                    }

                    foreach ($model->category->ancestors as $category) {
                        $parents .= '<span class="badge badge-primary mx-1">'.$category->title.'</span>';

                        if ($category->ancestors->count()) {
                            $parents .= '<i class="fas fa-arrow-alt-circle-left"></i>';
                        }
                    }

                    return $parents;
                }

                return '<span class="badge badge-secondary mx-1">'.__('Root').'</span>';
            })
            ->order(function (Builder $builder, string $direction) {
                return $builder->withAggregate('parent','title')
                    ->orderBy('parent_title', $direction);
            })
            ->filters(function (Model $model) {
                return array_merge([[
                    'value' => '0',
                    'label' => __('Root'),
                    'values' => [],
                ]], self::treeToFilters(
                    ProductCategory::orderBy('title')->tree()->get()->toTree(),
                    'id',
                    'title'
                ));
            })
            ->filter(function (Builder $builder, $searches) {
                foreach ($searches as $value => $key) {
                    if ($value == 0) {
                        $builder->whereNull('product_category_id');
                    } else {
                        $builder->where($key, 'like', '%'.$value.'%');
                    }
                }

                return $builder;
            });
    }

    public function fieldCreatedAt(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('created_at')
            ->title(__('Created at'))
            ->setRange(true)
            ->render(function (Model $model) {
                return $model->created_at->format("Y-m-d H:i:s");
            });
    }
}
