<?php

namespace ViartasShop\Brands\Models\Accounting;

use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Database\Eloquent\Relations\HasMany;
use ViartasCore\Core\Facades\Viartas;
use ViartasCore\Core\Models\BaseModel;
use ViartasCore\Core\Traits\DefinesAsCacheItem;
use ViartasCore\Core\Traits\HasMetaTags;

class Brand extends BaseModel
{
    use DefinesAsCacheItem, HasMetaTags;

    /**
     * @var string[]
     */
    protected $fillable = [
        'title', 'content', 'slug',
        'user_id',
        'is_visible', 'layout',
    ];

    /**
     * @var string
     */
    protected string $module = 'brands';

    /**
     * @var string
     */
    protected string $layout = 'default';

    /**
     * @var string
     */
    protected string $element_title = 'Brand';

    /**
     * @var bool
     */
    protected bool $usesRecursiveRelations = false;

    /**
     * @return string
     */
    public function getParentKeyName(): string
    {
        return 'product_category_id';
    }

    /**
     * @return HasMany
     */
    public function models(): HasMany
    {
        return $this->hasMany(Model::class, 'brand_id', 'id');
    }

    /**
     * @return void
     */
    public function ensureDefaultModel(): void
    {
        // Skip if any model already exists
        if ($this->models()->exists()) {
            return;
        }

        $defaultTitle = [
            'en' => 'Default',
            'lt' => 'Numatytasis',
            'ru' => 'По умолчанию',
        ];

        $baseTag = $this->tag
            ?? \Str::slug(data_get($this->title, 'en', 'brand'));

        $this->models()->create([
            'title'      => $defaultTitle,
            'tag'        => "{$baseTag}_default",
            'is_default' => true,
            'is_visible' => false,
        ]);
    }

    /**
     * Auto-create the default model after the brand is created.
     * You can opt out by setting $brand->skip_default_model = true before save.
     */
    protected static function booted(): void
    {
        static::created(function (Brand $brand) {
            if ($brand->getAttribute('skip_default_model')) {
                return; // opt out flag
            }

            $brand->ensureDefaultModel();
        });
    }

    /**
     * @return \Illuminate\Foundation\Application|View|Factory|Application
     */
    public function view(): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $app = Viartas::driver()->app()->current()->tag;
        //load views
        //check if template has
        //load view
        $element = $this;
        //make in base model, return app.module.default
        return theme_view($app.'.'.$this->module.'.'.$this->layout, compact('element'));
    }

    /**
     * @param string|null $locale
     * @return string
     */
    public function generateRoute(?string $locale = null): string
    {
        $segments = [];

        if ($locale) {
            if (Viartas::driver()->locale()->default()->tag != $locale) {
                $segments[] = $locale;
            }
        }

        //var_dump($segments);

        $child = $this->ancestorsAndSelf()->get()->toTree()->first();

        while ($child !== false) {
            $element = $child;

            if (! $locale) {
                $locale = Viartas::driver()->locale()->current()->tag;
            }

            $segments[] = $element->getTranslation('slug', $locale);

            if ($element->children->count()) {
                $child = $element->children->first();
            } else {
                $child = false;
            }
        }

        return implode('/', $segments);
    }
}
