<?php

namespace ViartasShop\Brands\Controllers\Accounting;

use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use ViartasCore\Core\Controllers\AccountingController;
use ViartasCore\Core\Forms\Common\ImagesUploadForm;
use ViartasShop\Brands\Forms\Accounting\ModelsEditForm;
use ViartasShop\Brands\Models\Accounting\Model;
use ViartasShop\Brands\Tables\Accounting\ModelsTable;

class ModelsController extends AccountingController
{
    /**
     * @param ModelsTable $table
     * @param Request $request
     * @return string|array|View|\Illuminate\Foundation\Application|Factory|Application
     */
    public function index(ModelsTable $table, Request $request): string|array|View|\Illuminate\Foundation\Application|Factory|Application
    {
        return $this->tableBuilder(
            $table, $request, theme_view('accounting.models.index', compact('table'))
        );
    }

    /**
     * @param ModelsEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function create(ModelsEditForm $form): View|Factory|Application|\Illuminate\Foundation\Application
    {
        return theme_view('accounting.models.edit', [
            'form' => $form->build(),
            'element' => new Model(),
        ]);
    }

    /**
     * @param ModelsEditForm $form
     * @return RedirectResponse
     */
    public function store(ModelsEditForm $form): RedirectResponse
    {
        $category = Model::create($form->all());

        return theme_redirect()->route('accounting.models.edit', [
            'id' => $category->id,
        ])->with('success', __('brand.created'));
    }

    /**
     * @param string $id
     * @param ModelsEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function edit(string $id, ModelsEditForm $form): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $category = Model::findOrFail($id);
        $form = $form->setAction('update')->build($category);

        return theme_view('accounting.models.edit', [
            'form' => $form,
            'element' => $category,
        ]);
    }

    /**
     * @param string $id
     * @param ModelsEditForm $form
     * @return RedirectResponse
     */
    public function update(string $id, ModelsEditForm $form): RedirectResponse
    {
        Model::findOrFail($id)->update($form->all());

        return theme_redirect()->route('accounting.models.edit', [
            'id' => $id,
        ])->with('success', __('brand.updated'));
    }

    /**
     * @param string $id
     * @param Request $request
     * @return void
     */
    public function updateOneField(string $id, Request $request): void
    {
        Model::findOrFail($id)->update([
            $request->get('field') => $request->get('value')
        ]);
    }

    /**
     * @param string $id
     * @param Request $request
     * @return bool|RedirectResponse
     */
    public function destroy(string $id, Request $request): bool|RedirectResponse
    {
        if ($id !== '0') {
            Model::findOrFail($id)->delete();

            return theme_redirect()->route('accounting.models.index')
                ->with('success', __('brand.deleted'));
        }

        $this->destroySelected($request);

        return true;
    }

    /**
     * @param Request $request
     * @return void
     */
    public function destroySelected(Request $request): void
    {
        foreach ($request->elements as $element) {
            Model::find($element)->delete();
        }

        Session::flash('success', __('elements.selected.deleted'));
    }

    /**
     * @param $id
     * @param ImagesUploadForm $images
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function uploadImages($id, ImagesUploadForm $images): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $category = Model::findOrFail($id);
        $images = $images->build();

        $images->field('images')
            ->setModel(Model::class)
            ->setPrimaryKey('id')
            ->setPrimaryValue($id);

        return theme_view('accounting.models.common', [
            'form' => $images,
            'element' => $category,
        ]);
    }

}
