<?php

namespace ViartasShop\Brands\Controllers\Accounting;

use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use ViartasCore\Core\Controllers\AccountingController;
use ViartasCore\Core\Forms\Common\ImagesUploadForm;
use ViartasShop\Brands\Models\Accounting\Brand;
use ViartasShop\Brands\Forms\Accounting\BrandsEditForm;
use ViartasShop\Brands\Tables\Accounting\BrandsTable;

class BrandsController extends AccountingController
{
    /**
     * @param BrandsTable $table
     * @param Request $request
     * @return string|array|View|\Illuminate\Foundation\Application|Factory|Application
     */
    public function index(BrandsTable $table, Request $request): string|array|View|\Illuminate\Foundation\Application|Factory|Application
    {
        return $this->tableBuilder(
            $table, $request, theme_view('accounting.brands.index', compact('table'))
        );
    }

    /**
     * @param BrandsEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function create(BrandsEditForm $form): View|Factory|Application|\Illuminate\Foundation\Application
    {
        return theme_view('accounting.brands.edit', [
            'form' => $form->build(),
            'element' => new Brand(),
        ]);
    }

    /**
     * @param BrandsEditForm $form
     * @return RedirectResponse
     */
    public function store(BrandsEditForm $form): RedirectResponse
    {
        $category = Brand::create($form->all());

        return theme_redirect()->route('accounting.brands.edit', [
            'id' => $category->id,
        ])->with('success', __('brand.created'));
    }

    /**
     * @param string $id
     * @param BrandsEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function edit(string $id, BrandsEditForm $form): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $category = Brand::findOrFail($id);
        $form = $form->setAction('update')->build($category);

        return theme_view('accounting.brands.edit', [
            'form' => $form,
            'element' => $category,
        ]);
    }

    /**
     * @param string $id
     * @param BrandsEditForm $form
     * @return RedirectResponse
     */
    public function update(string $id, BrandsEditForm $form): RedirectResponse
    {
        Brand::findOrFail($id)->update($form->all());

        return theme_redirect()->route('accounting.brands.edit', [
            'id' => $id,
        ])->with('success', __('brand.updated'));
    }

    /**
     * @param string $id
     * @param Request $request
     * @return void
     */
    public function updateOneField(string $id, Request $request): void
    {
        Brand::findOrFail($id)->update([
            $request->get('field') => $request->get('value')
        ]);
    }

    /**
     * @param string $id
     * @param Request $request
     * @return bool|RedirectResponse
     */
    public function destroy(string $id, Request $request): bool|RedirectResponse
    {
        if ($id !== '0') {
            Brand::findOrFail($id)->delete();

            return theme_redirect()->route('accounting.brands.index')
                ->with('success', __('brand.deleted'));
        }

        $this->destroySelected($request);

        return true;
    }

    /**
     * @param Request $request
     * @return void
     */
    public function destroySelected(Request $request): void
    {
        foreach ($request->elements as $element) {
            Brand::find($element)->delete();
        }

        Session::flash('success', __('elements.selected.deleted'));
    }

    /**
     * @param $id
     * @param ImagesUploadForm $images
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function uploadImages($id, ImagesUploadForm $images): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $category = Brand::findOrFail($id);
        $images = $images->build();

        $images->field('images')
            ->setModel(Brand::class)
            ->setPrimaryKey('id')
            ->setPrimaryValue($id);

        return theme_view('accounting.brands.common', [
            'form' => $images,
            'element' => $category,
        ]);
    }

}
