<?php

namespace ViartasShop\Attributes\Tables\Accounting;

use Illuminate\Database\Eloquent\Model;
use ViartasBuilders\Tables\DataTables;
use ViartasBuilders\Tables\DataTablesButton;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use ViartasBuilders\Tables\DataTablesRow;
use ViartasCore\Core\Facades\Viartas;
use ViartasShop\Attributes\Models\Accounting\Attribute;
use ViartasShop\Attributes\Models\Accounting\AttributeGroup;
use ViartasShop\Attributes\Models\Accounting\AttributeValue;

class ValuesTable extends DataTables
{
    public function __construct(Request $request)
    {
        parent::__construct(
            new AttributeValue(),
            $request
        );

        $this->table($request->id);

        $this->addRows([
            $this->fieldId(),
            //$this->fieldAttributeGroup(),
            $this->fieldAttribute(),
            $this->fieldTitle(),
            $this->fieldTag(),
            $this->fieldUser(),
            $this->fieldCreatedAt(),
            $this->button()
                ->add((new DataTablesButton())->editButton())
                ->add((new DataTablesButton())->destroyButton()),
        ]);
    }

    public function table(int $id): self
    {
        $this->setRoute(route('accounting.values.index', ['id' => $id]));
        $this->setFilters(route('accounting.values.index', ['id' => $id]));
        $this->setId('values-table');
        $this->setClass('values-table');
        $this->setTitle(__('Values'));
        $this->setIndexColumn('id');

        return $this;
    }
    public function fieldId(): DataTablesRow
    {
        $id = new DataTablesRow();

        $id->field('id')
            ->title('ID')
            ->setSearchable()
            ->render(function (Model $model) {
                return '#'.$model->id;
            })
            ->order(function (Builder $builder, string $direction) {
                return $builder->orderBy('id', $direction);
            });

        return $id;
    }
    public function fieldTitle(): DataTablesRow
    {
        $title = new DataTablesRow();

        $title->field('title')
            ->title(__('Title'))
            ->setSearchable()
            ->order(function (Builder $builder, string $direction) {
                $builder->orderBy('title->'.Viartas::driver()->locale()->current()->tag, $direction);
            });

        return $title;
    }

    public function fieldTag(): DataTablesRow
    {
        $title = new DataTablesRow();

        $title->field('tag')
            ->title(__('Tag'))
            ->setSearchable()
            ->order(function (Builder $builder, string $direction) {
                $builder->orderBy('tag', $direction);
            });

        return $title;
    }

    public function fieldAttribute(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('attribute_id')
            ->title(__('Attribute'))
            ->setSelectable()
            ->render(function (AttributeValue $value) {
                return '<span class="badge badge-primary mx-1">'.$value->attr->title.'</span>';
            })
            ->order(function (Builder $builder, string $direction) {
                return $builder->withAggregate('attr','title')
                    ->orderBy('attr_title', $direction);
            })
            ->filters(function () {
                return array_merge([[
                    'value' => '0',
                    'label' => __('Root'),
                    'values' => [],
                ]], Attribute::orderBy('title')
                    ->pluck('title', 'id')
                    ->map(fn ($title, $id) => [
                        'value'  => $id,
                        'label'  => $title,
                        'values' => [],
                    ])
                    ->values()
                    ->toArray()
                );
            })
            ->filter(function (Builder $builder, $searches) {
                foreach ($searches as $value => $key) {
                    if ($value == 0) {
                        $builder->whereNull('attribute_id');
                    } else {
                        $builder->where($key, 'like', '%'.$value.'%');
                    }
                }

                return $builder;
            });
    }

    public function fieldGroup(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('attribute_group_id')
            ->title(__('Group'))
            ->setSelectable()
            ->render(function (Attribute $attribute) {
                return '<span class="badge badge-primary mx-1">'.$attribute->group->title.'</span>';
            })
            ->order(function (Builder $builder, string $direction) {
                return $builder->withAggregate('group','title')
                    ->orderBy('group_title', $direction);
            })
            ->filters(function () {
                return array_merge([[
                    'value' => '0',
                    'label' => __('Root'),
                    'values' => [],
                ]], AttributeGroup::orderBy('title')
                    ->pluck('title', 'id')
                    ->map(fn ($title, $id) => [
                        'value'  => $id,
                        'label'  => $title,
                        'values' => [],
                    ])
                    ->values()
                    ->toArray()
                );
            })
            ->filter(function (Builder $builder, $searches) {
                foreach ($searches as $value => $key) {
                    if ($value == 0) {
                        $builder->whereNull('attribute_group_id');
                    } else {
                        $builder->where($key, 'like', '%'.$value.'%');
                    }
                }

                return $builder;
            });
    }

    public function fieldCreatedAt(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('created_at')
            ->title(__('Created at'))
            ->setRange(true)
            ->render(function (Model $model) {
                return $model->created_at->format("Y-m-d H:i:s");
            });
    }
}
