<?php

namespace ViartasShop\Attributes\Models\Accounting;

use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Route;
use ViartasCore\Core\Drivers\RouteDriver;
use ViartasCore\Core\Facades\Viartas;
use ViartasCore\Core\Models\BaseModel;
use ViartasCore\Core\Traits\DefinesAsCacheItem;
use ViartasCore\Core\Traits\HasMetaTags;
use ViartasShop\ProductCategories\Models\Accounting\AttributeProductCategory;
use ViartasShop\ProductCategories\Models\Accounting\ProductCategory;

class Attribute extends BaseModel
{
    use DefinesAsCacheItem;

    /**
     * @var string[]
     */
    protected $fillable = [
        'title', 'content', 'slug',
        'user_id',
        'attribute_group_id',
        'tag',
        'is_visible',
    ];

    /**
     * @var string
     */
    protected string $module = 'attributes';

    /**
     * @var string
     */
    protected string $layout = 'default';

    /**
     * @var string
     */
    protected string $element_title = 'Attribute';

    /**
     * @var bool
     */
    protected bool $usesRecursiveRelations = false;

    /**
     * @return string
     */
    public function getParentKeyName(): string
    {
        return 'title';
    }

    /**
     * @return BelongsToMany
     */
    public function categories(): BelongsToMany
    {
        return $this->belongsToMany(ProductCategory::class, 'attribute_product_category')
            ->using(AttributeProductCategory::class)
            ->withPivot([
                'sort',
                'is_visible_in_product',
                'is_visible_in_category',
                'is_filterable_lg',
                'is_filterable_sm',
            ])->withTimestamps();
    }

    /**
     * @return array
     */
    protected function menuConfig(): array
    {
        $config = [
            'edit' => $this->menuItem(__('Main settings'), $this->routeEdit()),
            'values' => $this->menuItem(__('Values'), $this->routeValues()),
            'activity' => $this->menuItem(__('Activity'), $this->routeEdit()),
        ];

        if ($this->usesMedia()) {
            $config['upload_images'] = $this->menuItem(__('Upload images'), $this->routeUploadImages());
            $config['images'] = $this->menuItem(__('Images'), $this->routeUploadImages());
        }

        return $config;
    }


    /**
     * @param Collection $menu
     * @return bool
     */
    public function menuRouteActive(Collection $menu): bool
    {
        if (Route::currentRouteName() == $menu->get('route')->getRouteName()) {
            return true;
        }

        if ($menu->get('route')->getRouteName() == 'accounting.values.index') {

            if ($menu->get('model')) {
                return true;
            }

            if (request()->route('value')) {
                return true;
            }

            if (Route::currentRouteName() == 'accounting.values.create') {
                return true;
            }
        }

        return false;
    }

    /**
     * @return RouteDriver
     */
    public function routeEdit(): RouteDriver
    {
        $route = sprintf('accounting.attributes.edit');

        return new RouteDriver($route, [
            'id' => $this->id,
        ]);
    }

    /**
     * @return RouteDriver
     */
    public function routeValues(): RouteDriver
    {
        $route = sprintf('accounting.values.index');

        return new RouteDriver($route, [
            'id' => $this->id,
        ]);
    }

    /**
     * @return BelongsTo
     */
    public function group(): BelongsTo
    {
        return $this->belongsTo(AttributeGroup::class, 'attribute_group_id', 'id');
    }

    /**
     * @return \Illuminate\Foundation\Application|View|Factory|Application
     */
    public function view(): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $app = Viartas::driver()->app()->current()->tag;
        $element = $this;
        return theme_view($app.'.'.$this->module.'.'.$this->layout, compact('element'));
    }
}
