<?php

namespace ViartasShop\Attributes\Controllers\Accounting;

use ViartasCore\Core\Forms\Common\ImagesUploadForm;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use ViartasCore\Core\Controllers\AccountingController;
use ViartasShop\Attributes\Forms\Accounting\ValuesEditForm;
use ViartasShop\Attributes\Models\Accounting\Attribute;
use ViartasShop\Attributes\Models\Accounting\AttributeValue;
use ViartasShop\Attributes\Tables\Accounting\ValuesTable;

class ValuesController extends AccountingController
{
    /**
     * @param int $id
     * @param ValuesTable $table
     * @param Request $request
     * @return string|array|View|\Illuminate\Foundation\Application|Factory|Application
     */
    public function index(int $id, ValuesTable $table, Request $request): string|array|View|\Illuminate\Foundation\Application|Factory|Application
    {
        $element = Attribute::findOrFail($id);
        $value = new AttributeValue();

        return $this->tableBuilder(
            $table, $request, theme_view('accounting.values.index', compact('table', 'element', 'value'))
        );
    }

    /**
     * @param int $id
     * @param ValuesEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function create(int $id, ValuesEditForm $form): Application|Factory|View|\Illuminate\Foundation\Application
    {
        return theme_view('accounting.values.edit', [
            'form' => $form->build(),
            'element' => Attribute::findOrFail($id),
            'value' => new AttributeValue(),
        ]);
    }

    /**
     * @param $id
     * @param ValuesEditForm $form
     * @return RedirectResponse
     */
    public function store($id, ValuesEditForm $form): RedirectResponse
    {
        $value = AttributeValue::create(array_merge($form->all(), [
            'attribute_id' => $id,
        ]));

        return theme_redirect()->route('accounting.values.edit', [
            'id' => $id,
            'value' => $value->id,
        ])->with('success', __('value.created'));
    }

    /**
     * @param int $id
     * @param int $value
     * @param ValuesEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function edit(int $id, int $value, ValuesEditForm $form): Application|Factory|View|\Illuminate\Foundation\Application
    {
        $attribute = Attribute::findOrFail($id);
        $value = AttributeValue::findOrFail($value);

        $form = $form->setAction('update')->build($value);

        return theme_view('accounting.values.edit', [
            'form' => $form,
            'element' => $attribute,
            'value' => $value,
        ]);
    }

    /**
     * @param int $id
     * @param int $value
     * @param ValuesEditForm $form
     * @return RedirectResponse
     */
    public function update(int $id, int $value, ValuesEditForm $form): RedirectResponse
    {
        AttributeValue::findOrFail($value)->update($form->all());

        return theme_redirect()->route('accounting.values.edit', [
            'id' => $id,
            'value' => $value,
        ])->with('success', __('value.updated'));
    }

    /**
     * @param int $id
     * @param int $value
     * @param Request $request
     * @return void
     */
    public function updateOneField(int $id, int $value, Request $request): void
    {
        AttributeValue::findOrFail($value)->update([
            $request->get('field') => $request->get('value')
        ]);
    }

    /**
     * @param string $id
     * @param Request $request
     * @return bool|RedirectResponse
     */
    public function destroy(string $id, Request $request): bool|RedirectResponse
    {
        if ($id !== '0') {
            AttributeValue::findOrFail($id)->delete();

            return theme_redirect()->route('accounting.values.index')
                ->with('success', __('value.deleted'));
        }

        $this->destroySelected($request);

        return true;
    }

    /**
     * @param Request $request
     * @return void
     */
    public function destroySelected(Request $request): void
    {
        foreach ($request->elements as $element) {
            AttributeValue::find($element)->delete();
        }

        Session::flash('success', __('elements.selected.deleted'));
    }

    public function uploadImages($id, ImagesUploadForm $images)
    {
        $page = AttributeValue::findOrFail($id);
        $images = $images->build();

        $images->field('images')
            ->setModel(AttributeValue::class)
            ->setPrimaryKey('id')
            ->setPrimaryValue($id);

        return theme_view('accounting.values.common', [
            'form' => $images,
            'element' => $page,
        ]);
    }

}
