<?php

namespace ViartasShop\Account\Controllers;

use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use Illuminate\View\Factory;
use Illuminate\View\View;
use ViartasCore\Core\Controllers\WebController;
use ViartasCore\Core\Facades\Viartas;
use ViartasShop\Account\Forms\CustomerForm;
use ViartasShop\Cart\Models\Accounting\Customer;

class CustomersController extends WebController
{
    /**
     * AccountController constructor.
     */
    public function __construct()
    {
        $route = Route::currentRouteName();
        $route = str_replace('.'.Viartas::driver()->locale()->path(), '', $route);

        view()->share('route', $route);

        parent::__construct();
    }
    /**
     * @return Factory|View
     */
    public function index(CustomerForm $form): Factory|View
    {
        $user = auth()->user();
        view()->share('user', $user);
        $customers = Customer::query()
            ->where('user_id', Auth::id())
            ->get();

        return theme_view('web.account.customers.index', [
            'form' => $form->build(),
            'customers' => $customers,
        ]);
    }

    /**
     * @param string $id
     * @param CustomerForm $form
     * @return string
     */
    public function edit(string $id, CustomerForm $form): string
    {
        $customer = Customer::findOrFail($id);
        if ($customer->user_id !== Auth::id()) {
            abort(404);
        }

        $form = $form->setAction('update')->build($customer);

        return $form->renderForm();
    }

    /**
     * @param string $id
     * @param CustomerForm $form
     * @return ?RedirectResponse
     */
    public function update(string $id, CustomerForm $form): ?RedirectResponse
    {
        $customer = Customer::findOrFail($id);
        if ($customer->user_id !== Auth::id()) {
            abort(404);
        }

        $customer->update($form->validated());

        if ($form->expectsJson()) {
            return session()->flash('success', __('customer.updated'));
        }

        return theme_redirect()->route('web.account.customers')
            ->with('success', __('customer.updated'));
    }

    /**
     * @param CustomerForm $form
     * @return Factory|View
     */
    public function create(CustomerForm $form): Factory|View
    {
        return theme_view('web.account.customers.index', [
            'form' => $form->build(),
        ]);
    }

    /**
     * @param CustomerForm $form
     * @return ?RedirectResponse
     */
    public function store(CustomerForm $form): ?RedirectResponse
    {
        $customer = Customer::query()
            ->create($form->all());

        if ($form->expectsJson()) {
            return session()->flash('success', __('customer.created'));
        }

        return theme_redirect()->route('web.account.customers', [
            'id' => $customer->id,
        ])->with('success', __('customer.created'));
    }

    /**
     * @param string $id
     * @param CustomerForm $form
     * @return ?RedirectResponse
     */
    public function delete(string $id,): ?RedirectResponse
    {
        $customer = Customer::findOrFail($id);
        if ($customer->user_id !== Auth::id()) {
            abort(404);
        }

        $customer->delete();

        return theme_redirect()->route('web.account.customers')
            ->with('success', __('customer.deleted'));
    }
}
