<?php

namespace ViartasModules\Posts\Tables\Accounting;

use ViartasBuilders\Tables\DataTables;
use ViartasBuilders\Tables\DataTablesButton;
use ViartasModules\PostCategories\Models\Accounting\PostCategory;
use ViartasModules\Posts\Models\Accounting\Post;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use ViartasBuilders\Tables\DataTablesRow;

class PostsTable extends DataTables
{
    public function __construct(Request $request)
    {
        parent::__construct(
            new Post(),
            $request
        );

        $this->table();

        $this->addRows([
            $this->fieldId(),
            $this->fieldTitle(),
            $this->fieldParent(),
            $this->fieldUser(),
            $this->fieldCreatedAt(),
            $this->button()
                ->add((new DataTablesButton())->editButton())
                ->add((new DataTablesButton())->destroyButton()),
        ]);
    }

    public function table(): self
    {
        $this->setRoute(route('accounting.posts.index'));
        $this->setFilters(route('accounting.posts.index'));
        $this->setId('posts-table');
        $this->setClass('posts-table');
        $this->setTitle(__('Posts'));
        $this->setIndexColumn('id');

        return $this;
    }
    public function fieldId(): DataTablesRow
    {
        $id = new DataTablesRow();

        $id->field('id')
            ->title('ID')
            ->setSearchable()
            ->render(function (Post $post) {
                return '#'.$post->id;
            })
            ->order(function (Builder $builder, string $direction) {
                return $builder->orderBy('id', $direction);
            });

        return $id;
    }
    public function fieldTitle(): DataTablesRow
    {
        $title = new DataTablesRow();

        $title->field('title')
            ->title(__('Title'))
            ->setSearchable()
            ->order(function (Builder $builder, string $direction) {
                $builder->orderBy('title->'.Asp::getLocale(), $direction);
            });

        return $title;
    }
    public function fieldParent(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('post_category_id')
            ->title(__('Parent'))
            ->setSelectable()
            ->render(function (Post $post) {
                if ($post->category) {
                    $parents = '<span class="badge badge-primary mx-1">' . $post->category->title . '</span>';

                    if ($post->category->ancestors->count()) {
                        $parents .= '<i class="fas fa-arrow-alt-circle-left"></i>';
                    }

                    if ($post->category->ancestors->count()) {
                        foreach ($post->category->ancestors as $ancestor) {
                            $parents .= '<span class="badge badge-primary mx-1">' . $ancestor->title . '</span>';

                            if ($ancestor->ancestors->count()) {
                                $parents .= '<i class="fas fa-arrow-alt-circle-left"></i>';
                            }
                        }
                    }

                    return $parents;
                }

                return '<span class="badge badge-secondary mx-1">'.__('Root').'</span>';
            })
            ->order(function (Builder $builder, string $direction) {
                return $builder->withAggregate('parent','title')
                    ->orderBy('parent_title', $direction);
            })
            ->filters(function (Post $post) {
                return array_merge([[
                    'value' => '0',
                    'label' => __('Root'),
                    'values' => [],
                ]], self::treeToFilters(
                    PostCategory::orderBy('post_category_id')->tree()->get()->toTree(),
                    'id',
                    'title'
                ));
            })
            ->filter(function (Builder $builder, $searches) {
                foreach ($searches as $value => $key) {
                    if ($value == 0) {
                        $builder->whereNull('post_category_id');
                    } else {
                        $builder->where($key, 'like', '%'.$value.'%');
                    }
                }

                return $builder;
            });
    }
    public function fieldCreatedAt(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('created_at')
            ->title(__('Created at'))
            ->setRange(true)
            ->render(function (Post $post) {
                return $post->created_at->format("Y-m-d H:i:s");
            });
    }
}
