<?php

namespace ViartasModules\Posts\Controllers\Accounting;

use App\Forms\Common\ImagesUploadForm;
use ViartasModules\Posts\Forms\Accounting\PostEditForm;
use ViartasModules\Posts\Models\Accounting\Post;
use ViartasModules\Posts\Tables\Accounting\PostsTable;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use ViartasCore\Core\Controllers\AccountingController;

class PostsController extends AccountingController
{
    /**
     * @param PostsTable $table
     * @param Request $request
     * @return string|array|View|\Illuminate\Foundation\Application|Factory|Application
     */
    public function index(PostsTable $table, Request $request): string|array|View|\Illuminate\Foundation\Application|Factory|Application
    {
        return $this->tableBuilder(
            $table, $request, theme_view('accounting.posts.index', compact('table'))
        );
    }

    /**
     * @param PostEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function create(PostEditForm $form): Application|Factory|View|\Illuminate\Foundation\Application
    {
        return theme_view('accounting.posts.edit', [
            'form' => $form->build(),
            'element' => new Post(),
        ]);
    }

    /**
     * @param PostEditForm $form
     * @return RedirectResponse
     */
    public function store(PostEditForm $form): RedirectResponse
    {
        $page = Post::create($form->all());

        return theme_redirect()->route('accounting.posts.edit', [
            'id' => $page->id,
        ])->with('success', __('post.category.created'));
    }

    /**
     * @param string $id
     * @param PostEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function edit(string $id, PostEditForm $form): Application|Factory|View|\Illuminate\Foundation\Application
    {
        $post_category = Post::findOrFail($id);
        $form = $form->setAction('update')->build($post_category);

        return theme_view('accounting.posts.edit', [
            'form' => $form,
            'element' => $post_category,
        ]);
    }

    /**
     * @param string $id
     * @param PostEditForm $form
     * @return RedirectResponse
     */
    public function update(string $id, PostEditForm $form): RedirectResponse
    {
        Post::findOrFail($id)->update($form->all());

        return theme_redirect()->route('accounting.posts.edit', [
            'id' => $id,
        ])->with('success', __('post.category.updated'));
    }
    /**
     * @param string $id
     * @param Request $request
     * @return void
     */
    public function updateOneField(string $id, Request $request): void
    {
        Post::findOrFail($id)->update([
            $request->get('field') => $request->get('value')
        ]);
    }

    /**
     * @param string $id
     * @param Request $request
     * @return bool|RedirectResponse
     */
    public function destroy(string $id, Request $request): bool|RedirectResponse
    {
        if ($id !== '0') {
            Post::findOrFail($id)->delete();

            return theme_redirect()->route('accounting.posts.index')
                ->with('success', __('post.deleted'));
        }

        $this->destroySelected($request);

        return true;
    }

    /**
     * @param Request $request
     * @return void
     */
    public function destroySelected(Request $request): void
    {
        foreach ($request->elements as $element) {
            Post::find($element)->delete();
        }

        Session::flash('success', __('elements.selected.deleted'));
    }

    public function uploadImages($id, ImagesUploadForm $images)
    {
        $page = Post::findOrFail($id);
        $images = $images->build();

        $images->field('images')
            ->setModel(Post::class)
            ->setPrimaryKey('id')
            ->setPrimaryValue($id);

        return theme_view('accounting.posts.common', [
            'form' => $images,
            'element' => $page,
        ]);
    }

}
