<?php

namespace ViartasModules\PostCategories\Tables\Accounting;

use ViartasBuilders\Tables\DataTables;
use ViartasBuilders\Tables\DataTablesButton;
use ViartasBuilders\Tables\DataTablesRow;
use ViartasModules\PostCategories\Models\Accounting\PostCategory;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use ViartasCore\Core\Facades\Viartas;

class MenuPostCategoriesTable extends DataTables
{
    public function __construct(Request $request)
    {
        parent::__construct(
            new PostCategory(),
            $request
        );

        $this->table();

        $this->addRows([
            $this->fieldId(),
            $this->fieldTitle(),
            $this->fieldParent(),
            $this->fieldSwitch([
                ['label' => __('Visible'), 'value' => "1",],
                ['label' => __('Invisible'), 'value' => "0",]
            ])
                ->field('is_visible')
                ->title(__('Visibility')),
            $this->fieldCheckboxMassActions(),
            $this->button()
                ->add($this->addOneItemButton())
                ->add($this->addManyItemsButton())
                ->add((new DataTablesButton())->editButton()->setColor('warning'))
        ]);
    }

    public function addOneItemButton (): DataTablesButton
    {
        return (new DataTablesButton())
            ->setColor('primary')
            ->setTitle(__('Add one item'))
            ->setIcon('fa-plus-circle')
            ->setClass('add-one-menu-item')
            ->getRoute();
    }

    public function addManyItemsButton (): DataTablesButton
    {
        return (new DataTablesButton())
            ->setColor('primary')
            ->setTitle(__('Add many items button'))
            ->setIcon('fa-layer-group')
            ->setClass('add-one-menu-item-with-children')
            ->getRoute();
    }

    public function table(): self
    {
        $this->setRoute(route('accounting.menus.items.post-categories'));
        $this->setFilters(route('accounting.menus.items.post-categories'));
        $this->setId('post-categories-table');
        $this->setClass('post-categories-table');
        $this->setType('simple');
        $this->setTitle(__('Post Categories'));
        $this->setIndexColumn('id');

        return $this;
    }
    public function fieldId(): DataTablesRow
    {
        $id = new DataTablesRow();

        $id->field('id')
            ->title('ID')
            ->setSearchable()
            ->render(function (PostCategory $postCategory) {
                return '#'.$postCategory->id;
            })
            ->order(function (Builder $builder, string $direction) {
                return $builder->orderBy('id', $direction);
            });

        return $id;
    }
    public function fieldTitle(): DataTablesRow
    {
        $title = new DataTablesRow();

        $title->field('title')
            ->title(__('Title'))
            ->setSearchable()
            ->order(function (Builder $builder, string $direction) {
                $builder->orderBy('title->'.Viartas::driver()->locale()->current()->tag, $direction);
            });

        return $title;
    }
    public function fieldParent(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('post_category_id')
            ->title(__('Parent'))
            ->setSelectable()
            ->render(function (PostCategory $postCategory) {
                if ($postCategory->ancestors->count()) {
                    $parents = '';
                    foreach ($postCategory->ancestors as $ancestor) {
                        $parents .= '<span class="badge badge-primary mx-1">'.$ancestor->title.'</span>';

                        if ($ancestor->ancestors->count()) {
                            $parents .= '<i class="fas fa-arrow-alt-circle-left"></i>';
                        }
                    }

                    return $parents;
                }

                return '<span class="badge badge-secondary mx-1">'.__('Root').'</span>';
            })
            ->order(function (Builder $builder, string $direction) {
                return $builder->withAggregate('parent','title')
                    ->orderBy('parent_title', $direction);
            })
            ->filters(function (PostCategory $postCategory) {
                return array_merge([[
                    'value' => '0',
                    'label' => __('Root'),
                    'values' => [],
                ]], self::treeToFilters(
                    $postCategory::orderBy('post_category_id')->tree()->get()->toTree(),
                    'id',
                    'title'
                ));
            })
            ->filter(function (Builder $builder, $searches) {
                foreach ($searches as $value => $key) {
                    if ($value == 0) {
                        $builder->whereNull('post_category_id');
                    } else {
                        $builder->where($key, 'like', '%'.$value.'%');
                    }
                }

                return $builder;
            });
    }
    public function fieldCreatedAt(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('created_at')
            ->title(__('Created at'))
            ->setRange(true)
            ->render(function (PostCategory $postCategory) {
                return $postCategory->created_at->format("Y-m-d H:i:s");
            });
    }
}
