<?php

namespace ViartasModules\Pages\Controllers\Accounting;

use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use ViartasCore\Core\Controllers\AccountingController;
use ViartasCore\Core\Forms\Common\ImagesUploadForm;
use ViartasModules\Pages\Forms\Accounting\PageEditForm;
use ViartasModules\Pages\Models\Accounting\Page;
use ViartasModules\Pages\Tables\Accounting\PagesTable;

class PagesController extends AccountingController
{
    /**
     * @param PagesTable $table
     * @param Request $request
     * @return string|array|View|\Illuminate\Foundation\Application|Factory|Application
     */
    public function index(PagesTable $table, Request $request): string|array|View|\Illuminate\Foundation\Application|Factory|Application
    {
        return $this->tableBuilder(
            $table, $request, theme_view('accounting.pages.index', compact('table'))
        );
    }

    /**
     * @param PageEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function create(PageEditForm $form): View|Factory|Application|\Illuminate\Foundation\Application
    {
        return theme_view('accounting.pages.edit', [
            'form' => $form->build(),
            'element' => new Page(),
        ]);
    }

    /**
     * @param PageEditForm $form
     * @return RedirectResponse
     */
    public function store(PageEditForm $form): RedirectResponse
    {
        $page = Page::create($form->all());

        return theme_redirect()->route('accounting.pages.edit', [
            'id' => $page->id,
        ])->with('success', __('page.created'));
    }

    /**
     * @param string $id
     * @param PageEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function edit(string $id, PageEditForm $form): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $page = Page::findOrFail($id);
        $form = $form->setAction('update')->build($page);

        return theme_view('accounting.pages.edit', [
            'form' => $form,
            'element' => $page,
        ]);
    }

    /**
     * @param string $id
     * @param PageEditForm $form
     * @return RedirectResponse
     */
    public function update(string $id, PageEditForm $form): RedirectResponse
    {
        Page::findOrFail($id)->update($form->all());

        return theme_redirect()->route('accounting.pages.edit', [
            'id' => $id,
        ])->with('success', __('page.updated'));
    }

    /**
     * @param string $id
     * @param Request $request
     * @return void
     */
    public function updateOneField(string $id, Request $request): void
    {
        Page::findOrFail($id)->update([
            $request->get('field') => $request->get('value')
        ]);
    }

    /**
     * @param string $id
     * @param Request $request
     * @return bool|RedirectResponse
     */
    public function destroy(string $id, Request $request): bool|RedirectResponse
    {
        if ($id !== '0') {
            Page::findOrFail($id)->delete();

            return theme_redirect()->route('accounting.pages.index')
                ->with('success', __('page.deleted'));
        }

        $this->destroySelected($request);

        return true;
    }

    /**
     * @param Request $request
     * @return void
     */
    public function destroySelected(Request $request): void
    {
        foreach ($request->elements as $element) {
            Page::find($element)->delete();
        }

        Session::flash('success', __('elements.selected.deleted'));
    }

    /**
     * @param $id
     * @param ImagesUploadForm $images
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function uploadImages($id, ImagesUploadForm $images): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $page = Page::findOrFail($id);
        $images = $images->build();

        $images->field('images')
            ->setModel(Page::class)
            ->setPrimaryKey('id')
            ->setPrimaryValue($id);

        return theme_view('accounting.pages.common', [
            'form' => $images,
            'element' => $page,
        ]);
    }

}
