(function(window, document) {
    var factory = function($, dataTable) {
        'use strict';

        var
            /**
             * Create a new Column instance
             *
             * @param {DataTable} dataTable       The DataTables object
             * @param {integer}   index           The index of the column
             * @param {DataTable} dataTableColumn The DataTables Column object
             * @param {object}    options         The columnfilter options for this column
             *
             * @returns {Column}
             */
            Column = function(dataTable, index, dataTableColumn, options) {
                var
                    self = this,
                    defaultOptions,
                    methods = [
                        'dom',
                        'bindEvents',
                        'request'
                    ]
                ;

                if (options.type in ColumnFilter.filter) {
                    defaultOptions = $.extend({}, ColumnFilter.filter[options.type]);
                } else {
                    defaultOptions = {};
                }
                self.options = $.extend({}, defaultOptions, options);

                $.each(methods, function(ii, method){
                    if (method in self.options) {
                        self[method] = self.options[method];
                    }
                })

                self.dataTable = dataTable;
                self.dataTableColumn = dataTableColumn;
                self.index = index;
            },
            /**
             * Create a new ColumnFilter instance
             *
             * @param {DataTable} dataTable The DataTables object
             * @param {object}    options   The columnfilter options
             *
             * @returns {ColumnFilter}
             */
            ColumnFilter = function(dataTable, options) {
                var self = this;

                self.columns = [];
                self.dataTable = null;
                self.init(dataTable, options);
            }
        ;

        Column.prototype = {
            /**
             * Build the form DOM elements
             *
             * @param {type} th The th element where to put the elements
             *
             * @returns {jQuery}
             */
            dom: function(th){},
            /**
             * Binds event to the DOM elements
             *
             * @returns {void}
             */
            bindEvents: function(){},
            /**
             * Return the searched string
             *
             * @returns {string}
             */
            request: function(){},
            /**
             * Trigger the datatable search with the request
             *
             * @returns {void}
             */
            search: function(){
                var self = this;

                self
                    .dataTableColumn
                    .search(self.request())
                    .draw()
                ;
            }
        }

        ColumnFilter.prototype = {
            /**
             * Initialize the filter pluggin
             *
             * @param {DataTable} dataTable
             * @param {object}    options
             *
             * @returns {void}
             */
            init: function(dataTable, options){

                var
                    self = this,
                    tr
                ;

                self.dataTable = dataTable;

                tr = $('<tr>').appendTo(self.dataTable.api().table().header());

                if (!self.dataTable.api().columns().eq(0)) return;

                self.dataTable.api().columns().eq(0).each(function(index){
                    var
                        className = self.dataTable.api().column(index).header().className,
                        never = className.match(/\bnever\b/),
                        columnOptions,
                        column,
                        th
                    ;

                    if (never && ('responsive' in self.dataTable)) {
                        return;
                    }

                    columnOptions = index in options ? options[index] : {};
                    column = new Column(
                        self.dataTable,
                        index,
                        self.dataTable.api().column(index),
                        columnOptions
                    );
                    th = $('<th>').appendTo(tr);
                    self.columns.push(column);

                    column.dom(th);
                    column.bindEvents();
                });

                // Hide and Show column filter th according to datatable build-in columns visibility
                $(self.dataTable.api().table().node()).on('column-visibility.dt', function (e, settings, column, state) {
                    if (state) {
                        $('th', tr).eq(column - 1).show()
                    } else {
                        $('th', tr).eq(column - 1).hide()
                    }
                });
            },
            /**
             * Add a custom filter
             *
             * @param {string} name
             * @param {object} filter
             *
             * @returns {void}
             */
            addFilter: function(name, filter){
                ColumnFilter.filter[name] = filter;
            }
        };

        /**
         * Default Column configuration
         */
        ColumnFilter.default = {
            type: 'text'
        };

        ColumnFilter.filter = {
            text: {
                /**
                 * Build the form DOM elements
                 *
                 * @param {type} th The th element where to put the elements
                 *
                 * @returns {jQuery}
                 */
                dom: function(th){
                    var self = this;

                    self.elements = $('<input>', {
                        type: 'text'
                    }).appendTo(th);

                    $.each(self.options.attr, function(key, value){
                        self.elements.attr(key,value);
                    });

                    if (typeof self.options.width !== 'undefined') {
                        self.elements.css('width', self.options.width);
                    } else {
                        self.elements.css('width', '100%');
                    }

                    return self.elements;
                },
                /**
                 * Binds event to the DOM elements
                 *
                 * @returns {void}
                 */
                bindEvents: function(){
                    var
                        self = this,
                        time = 200,
                        timeOutId = 0
                    ;

                    if ('time' in self.options) {
                        time = self.options.time;
                    }

                    self.elements.keyup(function(){
                        clearTimeout(timeOutId);
                        timeOutId = window.setTimeout(function(){
                            self.search();
                        }, time);
                    });
                },
                /**
                 * Return the searched string
                 *
                 * @returns {string}
                 */
                request: function(){
                    var self = this;

                    return self.elements.val();
                }
            },
            select: {
                dom: function(th){
                    var self = this, select;

                    select = $('<select>').append('<option></option>');

                    select.addClass(self.options.cssClass);

                    $.each(self.options.values, function(ii, value){
                        $('<option>').val(value.value).text(value.label).appendTo(select);
                    });

                    self.elements = select.appendTo(th);

                    $.each(self.options.attr, function(key, value){
                        self.elements.attr(key,value);
                    });

                    if (typeof self.options.width !== 'undefined') {
                        self.elements.css('width', self.options.width);
                    } else {
                        self.elements.css('width', '100%');
                    }

                    return self.elements;
                },

                bindEvents: function(){
                    var self = this;

                    self.elements.on('change', function(){
                        self.search();
                    });
                },
                request: function(){
                    var self = this;

                    return self.elements.val();
                }
            },
            checkbox: {
                separator: '~',
                /**
                 * Build the form DOM elements
                 *
                 * @param {type} th The th element where to put the elements
                 *
                 * @returns {jQuery}
                 */
                dom: function(th){
                    var self = this;

                    self.elements = $('<a href="#" class="btn btn-xs btn-icon btn-light-primary table-check-all"><i class="fas fa-square-check" /></a>')
                        .add($('<a href="#" class="btn btn-xs btn-icon btn-light-primary table-uncheck-all"><i class="fas fa-square" /></a>'))
                        .appendTo(th);

                    self.elements.css('margin', '0 3px 0 0');

                    /*if (typeof self.options.width !== 'undefined') {
                        self.elements.css('width', self.options.width);
                    } else {
                        self.elements.css('width', '45%');
                        self.elements.css('display', 'inline-block');
                        self.elements.css('margin', '3px');
                        self.elements.addClass('form-control form-control-solid daterangepicker-input');
                    }*/

                    return self.elements;
                },
                /**
                 * Binds event to the DOM elements
                 *
                 * @returns {void}
                 */
                /*bindEvents: function(){
                    var self = this;

                    self.elements.change(function(){
                        self.search();
                    });
                },*/
                /**
                 * Return the searched string
                 *
                 * @returns {string}
                 */
                /*request: function(){
                    var
                        self = this,
                        request = []
                    ;

                    self.elements.each(function(){
                        request.push($(this).val());
                    });

                    return request.join(self.options.separator);
                }*/
            },
            dateRange: {
                separator: '~',
                /**
                 * Build the form DOM elements
                 *
                 * @param {type} th The th element where to put the elements
                 *
                 * @returns {jQuery}
                 */
                dom: function(th){
                    var self = this;

                    self.elements = $('<input>', {
                        type: 'text'
                    }).add($('<input>', {
                        type: 'text'
                    })).appendTo(th);

                    if (typeof self.options.width !== 'undefined') {
                        self.elements.css('width', self.options.width);
                    } else {
                        self.elements.css('width', '45%');
                        self.elements.css('display', 'inline-block');
                        self.elements.css('margin', '0 3px');
                        self.elements.css('padding', '6px');
                        self.elements.addClass('form-control form-control-solid daterangepicker-input');
                    }

                    return self.elements;
                },
                /**
                 * Binds event to the DOM elements
                 *
                 * @returns {void}
                 */
                bindEvents: function(){
                    var self = this;

                    self.elements.change(function(){
                        self.search();
                    });
                },
                /**
                 * Return the searched string
                 *
                 * @returns {string}
                 */
                request: function(){
                    var
                        self = this,
                        request = []
                    ;

                    self.elements.each(function(){
                        request.push($(this).val());
                    });

                    return request.join(self.options.separator);
                }
            }
        };

        $.fn.dataTable.ColumnFilter = ColumnFilter;

        return ColumnFilter;
    };

    factory($, $.fn.dataTable);
})(window, document);
