<?php

namespace ViartasCore\Core\Tables;

use ViartasBuilders\Tables\DataTables;
use ViartasBuilders\Tables\DataTablesButton;
use ViartasBuilders\Tables\DataTablesRow;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use ViartasCore\Core\Facades\Viartas;
use ViartasCore\Core\Models\Module;
use ViartasCore\Core\Models\ModuleAction;

class MenuModulesTable extends DataTables
{
    /**
     * @var array|string[]
     */
    protected array $guards = [
        'accounting', 'web', 'b2b', 'api',
    ];

    /**
     * @var array|string[]
     */
    protected array $blacklist = [
        'login', 'logout', 'password',
    ];

    /**
     * @return array
     */
    public function appRoutes(): array
    {
        $locales = Viartas::driver()->locale()->all()->pluck('tag')->toArray();

        $permissions = [];
        $response = [];
        $routes = Route::getRoutes();

        foreach ($routes as $route) {

            $parts = explode(".", $route->getName());
            $guard = $parts[0];
            if (in_array($guard, $this->guards)) {
                $module = $parts[1];

                if (in_array($module, $locales)) {
                    continue;
                }

                unset($parts[0]);
                unset($parts[1]);

                if (count($parts) && !in_array($module, $this->blacklist) && !in_array($guard, $this->blacklist)) {
                    $permission = implode(".", $parts);

                    if (in_array('GET', $route->methods())) {
                        $response[$guard][$module][$route->getName()] = [
                            'name' => $permission,
                            'route_name' => $route->getName(),
                            'route_attributes' => $route->parameterNames(),
                        ];

                        $permissions[$guard][] = $route->getName();
                    }
                }
            }
        }

        foreach ($response as $guard => $modules) {
            foreach ($modules as $module => $actions) {

                $exists_module = Module::query()
                    ->where('guard', $guard)
                    ->where('name', $module)
                    ->first();

                if (!$exists_module) {
                    $exists_module = Module::create([
                        'guard' => $guard,
                        'name' => $module,
                    ]);
                }

                foreach ($actions as $action) {
                    $exists_action = $exists_module->actions()
                        ->where('route_name', $action['route_name'])
                        ->first();

                    if (!$exists_action) {
                        $action['route_attributes'] = json_encode($action['route_attributes']);
                        $exists_action = $exists_module->actions()
                            ->create($action);
                    }
                }
            }
        }

        return $response;
    }

    /**
     * @param Request $request
     */
    public function __construct(Request $request)
    {
        $this->appRoutes();

        parent::__construct(
            new ModuleAction(),
            $request
        );

        $this->table();

        $this->addRows([
            $this->fieldId(),
            $this->fieldModuleId(),
            //$this->fieldName(),
            $this->fieldModuleRoute(),
            $this->fieldSwitch([
                ['label' => __('Visible'), 'value' => "1",],
                ['label' => __('Invisible'), 'value' => "0",]
            ])
            ->field('is_visible')
            ->title(__('Visibility')),
            $this->fieldSwitch([
                ['label' => __('Active'), 'value' => "1",],
                ['label' => __('Inactive'), 'value' => "0",]
            ])
                ->field('is_active')
                ->title(__('Active')),
            $this->fieldSwitch([
                ['label' => __('Deleted'), 'value' => "1",],
                ['label' => __('Available'), 'value' => "0",]
            ])
                ->field('is_deleted')
                ->title(__('Deleted')),
            $this->button()
                ->add($this->addOneItemButton())
                ->add((new DataTablesButton())
                    ->getRoute()
                    ->setTitle(__('Edit'))
                    ->setColor('warning')
                    ->setRoute(function (ModuleAction $moduleAction) {
                        return $moduleAction->routeEdit();
                    })
                )
        ]);
    }

    /**
     * @return $this
     */
    public function table(): self
    {
        $this->setRoute(route('accounting.menus.items.modules'));
        $this->setFilters(route('accounting.menus.items.modules'));
        $this->setId('modules-table');
        $this->setClass('modules-table');
        $this->setType('simple');
        $this->setTitle(__('Modules'));
        $this->setIndexColumn('id');

        return $this;
    }

    /**
     * @return DataTablesButton
     */
    public function addOneItemButton (): DataTablesButton
    {
        return (new DataTablesButton())
            ->setColor('primary')
            ->setTitle(__('Add one item'))
            ->setElementTitle('route_name')
            ->setIcon('fa-plus-circle')
            ->setClass('add-one-menu-item')
            ->getRoute();
    }

    /**
     * @return DataTablesRow
     */
    public function fieldName(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('name')
            ->title(__('View'))
            ->setSearchable();
    }

    public function fieldModuleRoute(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('route_name')
            ->setSearchable()
            ->title(__('route'));
            //->setVisible(false);
    }

    /**
     * @return DataTablesRow
     */
    public function fieldModuleId(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('module_id')
            ->title(__('Module name'))
            ->setSelectable()
            ->filters(function (Model $model) {
                $array = [];
                foreach (Module::query()->distinct('name')->orderBy('name')->get() as $module) {
                    $array[] = [
                        'value' => $module->id,
                        'label' => $module->name,
                    ];
                }

                return $array;
            })
            ->render(function (ModuleAction $moduleAction) {
                return $moduleAction->module->name;
            });
    }

    /**
     * @return DataTablesRow
     */
    public function fieldGuard(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('guard')
            ->title(__('Guard'))
            ->setSelectable();
    }
}
