<?php

namespace ViartasCore\Core\Models;

use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use Spatie\Activitylog\LogOptions;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Image\Enums\Fit;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Exceptions\FileDoesNotExist;
use Spatie\MediaLibrary\MediaCollections\Exceptions\FileIsTooBig;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use Spatie\Sluggable\HasTranslatableSlug;
use Spatie\Sluggable\SlugOptions;
use Spatie\Translatable\HasTranslations;
use Staudenmeir\LaravelAdjacencyList\Eloquent\HasRecursiveRelationships;
use ViartasCore\Core\Drivers\RouteDriver;
use ViartasCore\Core\Facades\Viartas;
use ViartasCore\Core\Traits\HasMetaTags;

class BaseModel extends Model implements HasMedia
{
    use HasFactory,
        HasTranslations, HasTranslatableSlug,
        InteractsWithMedia,
        HasRecursiveRelationships,
        LogsActivity;

    /**
     * @var array|Collection|string[]
     */
    protected array $guard_name = ['web', 'accounting', 'api', 'b2b'];

    /**
     * @var bool
     */
    protected bool $usesRecursiveRelations = true;

    /**
     * @var bool
     */
    protected bool $usesMedia = true;

    /**
     * @var string
     */
    protected string $module = 'pages';

    /**
     * @var string
     */
    protected string $layout = 'default';

    /**
     * @var string
     */
    protected string $element_title = '';

    /**
     * @var string[]
     */
    public array $translatable = [
        'title', 'content', 'slug',
    ];

    /**
     * @var array
     */
    static array $packagesMenus = [];

    /**
     *
     */
    public function __construct()
    {
        /*foreach ($this->translatable as $key => $translatable) {
            if (!in_array($translatable, $this->fillable)) {
                unset($this->translatable[$key]);
            }
        }*/

        //$this->guard_name = Application::all()->pluck('tag');

        parent::__construct();
    }

    /**
     * @return string
     */
    public function getModule(): string
    {
        return $this->module;
    }

    /**
     * @return void
     */
    protected static function boot(): void
    {
        parent::boot();

        static::creating(function ($model) {
            if (in_array('user_id', $model->fillable)) {
                $model->user_id = Auth::id() ?? User::query()->orderBy('id')->first()->id;
            }
        });
    }

    /**
     * @return string
     */
    public function elementTitle(): string
    {
        return __($this->element_title);
    }

    /**
     * @param $attr
     * @return bool
     */
    public function hasAttribute($attr): bool
    {
        return array_key_exists($attr, $this->attributes);
    }

    /**
     * @param $field
     * @return bool
     */
    public function hasField($field): bool
    {
        return in_array($field, $this->fillable);
    }

    /**
     * @return bool
     */
    public function usesRecursiveRelations(): bool
    {
        return $this->usesRecursiveRelations;
    }

    /**
     * @return bool
     */
    public function usesMedia(): bool
    {
        return $this->usesMedia;
    }

    /**
     * @return string
     */
    public function getParentKeyName(): string
    {
        return 'parent_id';
    }

    /**
     * @return string
     */
    public function getLocalKeyName(): string
    {
        return 'id';
    }

    /**
     * @return int
     */
    public function isRoot(): int
    {
        return $this->rootAncestor()->count();
    }

    /**
     * @return LogOptions
     */
    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly($this->fillable);
    }

    /**
     * @return BelongsTo
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * @return \Illuminate\Foundation\Application|View|Factory|Application
     */
    public function view(): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $app = Viartas::driver()->app()->current()->tag;

        $element = $this;

        return theme_view($app.'.'.$this->module.'.'.$this->layout, compact('element'));
    }

    /**
     * @return SlugOptions
     */
    public function getSlugOptions(): SlugOptions
    {
        $locales = Viartas::driver()->locale()->all()->pluck('tag')->toArray();

        return SlugOptions::createWithLocales($locales)
            ->generateSlugsFrom(function($model, $locale) {
                return "{$locale} {$model->id}";
            })
            ->generateSlugsFrom(
                in_array('slug', $this->fillable)
                    ? 'title'
                    : 'no_slug_required'
            )
            ->saveSlugsTo('slug')
            ->doNotGenerateSlugsOnUpdate();
    }

    /*    public function getSlugOptions() : SlugOptions
        {
            return SlugOptions::createWithLocales(['en', 'nl'])
                ->generateSlugsFrom(function($model, $locale) {
                    return "{$locale} {$model->id}";
                })
                ->saveSlugsTo('slug');
        }*/

    /**
     * Get the route key for the model.
     *
     * @return string
     */
    public function getRouteKeyName(): string
    {
        return 'slug';
    }

    /**
     * @param Media|null $media
     * @return void
     */
    public function registerMediaConversions(?\Spatie\MediaLibrary\MediaCollections\Models\Media $media = null): void
    {
        $this
            ->addMediaConversion('preview')
            ->fit(Fit::Contain, 300, 300)
            ->nonQueued();
    }


    /**
     * @param mixed $pathToFile
     * @param string $collectionName
     * @param string $disk
     * @return Media
     * @throws FileDoesNotExist
     * @throws FileIsTooBig
     */
    public function uploadMedia(mixed $pathToFile, string $collectionName = 'images', string $disk = 'public'): Media
    {
        return $this->addMedia($pathToFile)
            ->toMediaCollection($collectionName, $disk);
    }

    /**
     * @param string $title
     * @param RouteDriver $route
     * @param BaseModel|null $model
     * @return Collection
     */
    protected static function menuItem(string $title, RouteDriver $route, BaseModel|null $model = null): Collection
    {
        return collect([
            'title' => $title,
            'route' => $route,
            'model' => $model,
        ]);
    }

    /**
     * @return array
     */
    protected function menuConfig(): array
    {
        $config = [
            'edit' => $this->menuItem(__('Main settings'), $this->routeEdit()),
            'upload_documents' => $this->menuItem(__('Upload documents'), $this->routeEdit()),
            'documents' => $this->menuItem(__('Documents'), $this->routeEdit()),
            'comments' => $this->menuItem(__('Comments'), $this->routeEdit()),
            'activity' => $this->menuItem(__('Activity'), $this->routeEdit()),
        ];

        if ($this->usesMedia()) {
            $config['upload_images'] = $this->menuItem(__('Upload images'), $this->routeUploadImages());
            $config['images'] = $this->menuItem(__('Images'), $this->routeUploadImages());
        }

        return $config;
    }

    /**
     * @param string $key
     * @param string $title
     * @param string $routeName
     * @param array $routeAttributes
     * @return void
     */
    protected static function addPackageMenuItem(string $key, string $title, string $routeName, array $routeAttributes = []): void
    {
        self::$packagesMenus[$key] = [
            'key' => $key,
            'title' => $title,
            'routeName' => $routeName,
            'routeAttributes' => $routeAttributes,
        ];
    }

    /**
     * @return array
     */
    public function menu(): array
    {
        $menus = [];

        foreach ($this->menuConfig() as $name => $menu) {
            if (! Route::has($menu->get('route')->getRouteName())) {
                continue;
            }

            if ($menu->get('route')->isAvailable()) {
                $menus[$name] = $menu;
            }
        }

        $packagesMenu = [];

        foreach (self::$packagesMenus as $key => $packageMenu) {
            if (! method_exists($this, $packageMenu['routeName'])) {
                continue;
            }

            $packagesMenu[$key] = $this->menuItem(
                $packageMenu['title'],
                $this->{$packageMenu['routeName']}()
            );
        }

        foreach ($packagesMenu as $name => $menu) {
            if ($menu->get('route')->isAvailable()) {
                $menus[$name] = $menu;
            }
        }

        return $menus;
    }

    /**
     * @return array
     */
    protected function buttonsConfig(): array
    {
        $buttons = [
            'list' => collect([
                'title' => __('List'),
                'route' => $this->routeIndex(),
                'icon' => 'fa-list',
                'size' => 'sm',
                'color' => 'primary',
                'class' => '',
                'type' => [
                    'forms', 'summaries',
                ],
            ]),
            'create' => collect([
                'title' => __('Create'),
                'route' => $this->routeCreate(),
                'icon' => 'fa-plus',
                'size' => 'sm',
                'color' => 'success',
                'class' => '',
                'type' => [
                    'forms', 'tables', 'summaries',
                ],
            ]),
            'delete' => collect([
                'title' => __('Delete'),
                'route' => $this->routeDestroy(),
                'icon' => 'fa-trash',
                'size' => 'sm',
                'color' => 'danger',
                'class' => 'confirm-form',
                'type' => [
                    'forms', 'summaries',
                ],
            ]),
            'delete-selected' => collect([
                'title' => __('Delete selected'),
                'route' => $this->routeDestroySelected(),
                'icon' => 'fa-trash',
                'size' => 'sm',
                'color' => 'danger',
                'class' => 'destroy-selected-button',
                'type' => [
                    'tables',
                ],
            ]),
        ];

        if (!$this->id) {
            unset($buttons['delete']);
        }

        return $buttons;
    }

    /**
     * @return array
     */
    public function buttons(): array
    {
        $buttons = [];

        foreach ($this->buttonsConfig() as $name => $button) {

            if (! Route::has($button->get('route')->getRouteName())) {
                continue;
            }

            if ($button->get('route')->isAvailable()) {
                $buttons[$name] = $button;
            }
        }

        return $buttons;
    }

    /**
     * @param Collection $menu
     * @return bool
     */
    public function menuRouteActive(Collection $menu): bool
    {
        return Route::currentRouteName() == $menu->get('route')->getRouteName();
    }

    /**
     * @return RouteDriver
     */
    public function routeEdit(): RouteDriver
    {
        $route = sprintf('accounting.%s.edit', $this->module);

        return new RouteDriver($route, [
            'id' => $this->id,
        ]);
    }

    /**
     * @param string $field
     * @return RouteDriver
     */
    public function routeFieldChange(string $field): RouteDriver
    {
        $route = sprintf('accounting.%s.edit', $this->module);

        return new RouteDriver($route, [
            'id' => $this->id,
            'field' => $field,
        ], 'PATCH');
    }

    /**
     * @return RouteDriver
     */
    public function routeDestroy(): RouteDriver
    {
        $route = sprintf('accounting.%s.destroy', $this->module);

        return new RouteDriver($route, [
            'id' => $this->id ?? 0,
        ], 'POST');
    }

    /**
     * @return RouteDriver
     */
    public function routeDestroySelected(): RouteDriver
    {
        $route = sprintf('accounting.%s.destroy', $this->module);

        return new RouteDriver($route, [
            'id' => 0,
        ], 'POST');
    }

    /**
     * @return RouteDriver
     */
    public function routeCreate(): RouteDriver
    {
        $route = sprintf('accounting.%s.create', $this->module);

        return new RouteDriver($route);
    }

    /**
     * @return RouteDriver
     */
    public function routeIndex(): RouteDriver
    {
        $route = sprintf('accounting.%s.index', $this->module);

        return new RouteDriver($route);
    }

    /**
     * @return RouteDriver
     */
    public function routeUploadImages(): RouteDriver
    {
        $route = sprintf('accounting.%s.upload.images', $this->module);

        return new RouteDriver($route, [
            'id' => $this->id,
        ]);
    }
}
