<?php

namespace ViartasCore\Core\Drivers;

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\URL;
use Illuminate\View\Factory;
use Illuminate\View\View;
use ViartasCore\Core\Facades\Viartas;
use ViartasCore\Core\Models\Application;
use ViartasCore\Core\Models\ModuleAction;
use ViartasCore\Core\Models\Theme;

class ThemeDriver
{
    /**
     * @var array
     */
    private array $controllers = [];

    /**
     * @var array
     */
    private array $views = [];

    /**
     * @var Theme
     */
    private Theme $theme;

    /**
     * @var Application
     */
    private Application $application;

    /**
     * @var array
     */
    private array $packageJs = [];

    /**
     * @var array
     */
    private array $packageCss = [];

    /**
     * @var array
     */
    private array $disabledNodePackages = [];

    /**
     * @var array
     */
    private array $viteCssLibrary = [];
    /**
     * @var array
     */
    private array $viteCssModule = [];

    /**
     * @var array
     */
    private array $viteCssScript = [];

    /**
     * @var array
     */
    private array $viteJsLibrary = [];

    /**
     * @var array
     */
    private array $viteJsModule = [];

    /**
     * @var array
     */
    private array $viteJsScript = [];

    /**
     * @var string
     */
    protected string $publicPath = 'vendor/viartas-themes/%s/assets/';

    /**
     * @var string
     */
    private string $currentView = '';

    /**
     * @param Application $application
     */
    public function __construct(Application $application)
    {
        $this->application = $application;
    }

    /**
     * @param array|null $controllers
     * @return array
     */
    public function controllers(?array $controllers = null): array
    {
        if ($controllers) {
            $this->controllers = array_merge($this->controllers, $controllers);
        }

        return $this->controllers;
    }

    /**
     * @param null $view
     * @param array $data
     * @param array $mergeData
     * @return Factory|View
     */
    public function view($view = null, array $data = [], array $mergeData = []): Factory|View
    {
        if (! preg_match('/\.(any)\.(index)$/', Route::currentRouteName())) {
            $module = ModuleAction::where('route_name', Route::currentRouteName())->first();
            \Illuminate\Support\Facades\View::share('module', $module);
        }

        if (isset($data['element'])) {
            \Illuminate\Support\Facades\View::share('element', $data['element']);
            unset($data['element']);
        }

        $this->currentView = $this->views()[$view] ?? $view;

        foreach ($this->packageJs as $view => $packages) {
            $view = str_replace('*', '', $view);

            if (preg_match('/^('.$view.')/i', $this->currentView)) {
                foreach ($packages as $package) {
                    $this->{'viteJs' . ucfirst($package['type'])}($package['file'], 'vendor/'.$package['vendor'], $package['render_as']);
                }
            }
        }

        foreach ($this->packageCss as $view => $packages) {
            $view = str_replace('*', '', $view);

            if (preg_match('/^('.$view.')/i', $this->currentView)) {
                foreach ($packages as $package) {
                    $this->{'viteCss' . ucfirst($package['type'])}($package['file'], 'vendor/'.$package['vendor']);
                }
            }
        }

        return view($this->currentView, $data, $mergeData);
    }

    /**
     * @param array|null $views
     * @return array
     */
    public function views(?array $views = null): array
    {
        if ($views) {
            $this->views = array_merge($this->views, $views);
        }

        return $this->views;
    }

    /**
     * @return string
     */
    public function layout(): string
    {
        return sprintf(
            'theme.%s.layout',
            Viartas::driver()->app()->current()->tag
        );
    }

    /**
     * @param string|null $file
     * @return string
     */
    public function publicPath(?string $file): string
    {
        return URL::asset(sprintf(
            $this->publicPath,
            Viartas::driver()->theme()->current()->tag
        ). ($file ?? ''));
    }

    /**
     * @param string $part
     * @return string
     */
    public function partials(string $part): string
    {
        return sprintf(
            'theme.%s.partials.%s',
            Viartas::driver()->app()->current()->tag,
            $part
        );
    }

    /**
     * @param string $path
     * @return string
     */
    public function fixPath(string $path): string
    {
        return ltrim(rtrim($path, '/'), '/');
    }

    /**
     * @param array $packages
     * @return void
     */
    public function packageJs(array $packages): void
    {
        foreach ($packages as $view => $package) {
            if (!isset($this->packageJs[$view])) {
                $this->packageJs[$view] = [];
            }

            $this->packageJs[$view] = array_merge($this->packageJs[$view], $package);
        }
    }

    /**
     * @param array $packages
     * @return void
     */
    public function packageCss(array $packages): void
    {
        foreach ($packages as $view => $package) {
            if (!isset($this->packageCss[$view])) {
                $this->packageCss[$view] = [];
            }

            $this->packageCss[$view] = array_merge($this->packageCss[$view], $package);
        }
    }

    /**
     * @param array $packages
     * @return void
     */
    public function disabledNodePackages(array $packages): void
    {
        foreach ($packages as $view => $package) {
            if (!isset($this->disabledNodePackages[$view])) {
                $this->disabledNodePackages[$view] = [];
            }

            $this->disabledNodePackages[$view] = array_merge($this->disabledNodePackages[$view], $package);
        }
    }

    /**
     * @param string|null $file
     * @param string|null $vendor_path
     * @return array|string
     */
    public function viteCssLibrary(?string $file = null, ?string $vendor_path = null): array|string
    {
        if ($file) {
            $fullPath = $this->fixPath($vendor_path) . '/' . $file;

            if (!isset($this->viteCssLibrary[$fullPath])) {
                $this->viteCssLibrary[$fullPath] = [
                    'file' => $file,
                    'vendor_path' => $vendor_path,
                ];
            }

            return '';
        }

        return $this->viteCssLibrary;
    }

    /**
     * @param string|null $file
     * @param string|null $vendor_path
     * @return array|string
     */
    public function viteCssModule(?string $file = null, ?string $vendor_path = null): array|string
    {
        if ($file) {
            $fullPath = $this->fixPath($vendor_path) . '/' . $file;

            if (!isset($this->viteCssModule[$fullPath])) {
                $this->viteCssModule[$fullPath] = [
                    'file' => $file,
                    'vendor_path' => $vendor_path,
                ];
            }

            return '';
        }

        return $this->viteCssModule;
    }

    /**
     * @param string|null $file
     * @param string|null $vendor_path
     * @return array|string
     */
    public function viteCssScript(?string $file = null, ?string $vendor_path = null): array|string
    {
        if ($file) {
            $fullPath = $this->fixPath($vendor_path) . '/' . $file;

            if (!isset($this->viteCssScript[$fullPath])) {
                $this->viteCssScript[$fullPath] = [
                    'file' => $file,
                    'vendor_path' => $vendor_path,
                ];
            }

            return '';
        }

        return $this->viteCssScript;
    }

    /**
     * @param string|null $file
     * @param string|null $vendor_path
     * @param string $type
     * @return array|string
     */
    public function viteJsLibrary(?string $file = null, ?string $vendor_path = null, string $type = 'module'): array|string
    {
        if ($file) {
            $fullPath = $this->fixPath($vendor_path) . '/' . $file;

            if (!isset($this->viteJsLibrary[$fullPath])) {
                $this->viteJsLibrary[$fullPath] = [
                    'file' => $file,
                    'vendor_path' => $vendor_path,
                    'type' => $type,
                ];
            }

            return '';
        }

        return $this->viteJsLibrary;
    }

    /**
     * @param string|null $file
     * @param string|null $vendor_path
     * @param string $type
     * @return array|string
     */
    public function viteJsModule(?string $file = null, ?string $vendor_path = null, string $type = 'module'): array|string
    {
        if ($file) {
            $fullPath = $this->fixPath($vendor_path) . '/' . $file;

            if (!isset($this->viteJsModule[$fullPath])) {
                $this->viteJsModule[$fullPath] = [
                    'file' => $file,
                    'vendor_path' => $vendor_path,
                    'type' => $type,
                ];
            }

            return '';
        }

        return $this->viteJsModule;
    }

    /**
     * @param string|null $file
     * @param string|null $vendor_path
     * @param string $type
     * @return array|string
     */
    public function viteJsScript(?string $file = null, ?string $vendor_path = null, string $type = 'module'): array|string
    {
        if ($file) {
            $fullPath = $this->fixPath($vendor_path) . '/' . $file;

            if (!isset($this->viteJsScript[$fullPath])) {
                $this->viteJsScript[$fullPath] = [
                    'file' => $file,
                    'vendor_path' => $vendor_path,
                    'type' => $type,
                ];
            }

            return '';
        }

        return $this->viteJsScript;
    }

    /**
     * @param string $type
     * @return array
     */
    public function viteScripts(string $type): array
    {
        $type = 'viteJs'.ucfirst($type);
        $scripts = $this->{$type};

        foreach ($this->disabledNodePackages as $view => $packages) {
            $view = str_replace('*', '', $view);

            if (preg_match('/^('.$view.')/i', $this->currentView)) {
                foreach ($packages as $package) {
                    foreach ($scripts as $vendor => $script) {
                        $package_pattern = str_replace('/', '\/', $package);
                        if (preg_match('/^(vendor)\/('.$package_pattern.')/i', $vendor)) {
                            unset($scripts[$vendor]);
                        }
                    }
                }
            }
        }

        return $scripts;
    }

    /**
     * @param string $type
     * @return array
     */
    public function viteStyles(string $type): array
    {
        $type = 'viteCss'.ucfirst($type);
        return $this->{$type};
    }

    /**
     * @return Theme
     */
    public function current(): Theme
    {
        return $this->theme ?? $this->theme = Theme::where('application_id', $this->application->id)
            ->where('is_default', 1)
            ->where('is_active', 1)
            ->first();
    }

    public function title()
    {
        return 'metronic';
    }
}
