<?php

namespace ViartasCore\Core\Drivers;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Spatie\Permission\Models\Role;
use ViartasCore\Core\Facades\Viartas;

class RouteDriver
{
    /**
     * @var string
     */
    protected string $defaultRoute = 'accounting.empty.route';

    /**
     * @var string|Model
     */
    private string|Model $routeName = '';
    /**
     * @var array
     */
    private array $routeAttributes = [];

    /**
     * @var string
     */
    private string $routeMethod = 'GET';

    /**
     * @var Model
     */
    private Model $model;

    /**
     * @param string|Model $routeName
     * @param array $routeAttributes
     * @param string $routeMethod
     */
    public function __construct (string|Model $routeName = '', array $routeAttributes = [], string $routeMethod = 'GET')
    {
        $this->routeAttributes = $routeAttributes;
        $this->routeMethod = $routeMethod;

        if ($routeName instanceof Model) {
            return $this->set('routeName', $routeName);
        }

        $this->routeName = (
        $routeName == ''
            ? $this->defaultRoute
            : $routeName
        );

        return $this;
    }

    /**
     * @param string $variable
     * @param mixed $value
     * @return self
     */
    private function set(string $variable, mixed $value): self
    {
        $this->{$variable} = $value;
        return $this;
    }

    /**
     * @return string
     */
    private function guessRouteGate(): string
    {
        if ($this->routeName instanceof Model) {
            return Viartas::driver()->app()->current()->tag;
        }

        $parts = explode(".", $this->routeName);
        return $parts[0];
    }

    /**
     * @return array
     */
    public function getRouteAttributes(): array
    {
        return $this->routeAttributes;
    }

    /**
     * @param array $routeAttributes
     * @return self
     */
    public function setRouteAttributes(array $routeAttributes): self
    {
        return $this->set('routeAttributes', $routeAttributes);
    }

    /**
     * @param array $routeAttributes
     * @return self
     */
    public function addRouteAttributes(array $routeAttributes): self
    {
        return $this->setRouteAttributes(array_merge(
            $this->getRouteAttributes(), $routeAttributes
        ));
    }

    /**
     * @param string $routeName
     * @return self
     */
    public function setRouteName(string $routeName): self
    {
        return $this->set('routeName', $routeName);
    }

    /**
     * @return string|Model
     */
    public function getRouteName(): string|Model
    {
        return $this->routeName;
    }

    /**
     * @param string $routeMethod
     * @return self
     */
    public function setRouteMethod(string $routeMethod): self
    {
        return $this->set('routeMethod', $routeMethod);
    }

    /**
     * @return string
     */
    public function getRouteMethod(): string
    {
        return $this->routeMethod;
    }

    /**
     * @return string
     */
    public function getRoutePath(): string
    {
        return
            $this->routeName instanceof Model
            ? $this->route()
            : route($this->stringRoute($this->routeName), $this->routeAttributes);
    }

    public function stringRoute(string $route): string
    {
        $parts = explode(".", $route);
        $route = [];

        foreach ($parts as $key => $value) {
            $route[] = $value;

            if ($key == 0) {
                if (Viartas::driver()->locale()->path() != '') {
                    $route[] = Viartas::driver()->locale()->path();
                }
            }
        }

        return implode('.', $route);
    }

    /**
     * @return string
     */
    private function route(): string
    {
        $segments = [];
        $segments[] = Viartas::driver()->app()->path();
        $segments[] = Viartas::driver()->locale()->path();

        foreach ($segments as $key => $value) {
            if ($value === '') {
                unset($segments[$key]);
            }
        }

        $app_path = '/';
        if (! empty($segments)) {
            $app_path .= implode('/', $segments) . '/';
        }

        return $app_path . $this->routeName->generateRoute();
    }

    /**
     * @return bool
     */
    public function isAvailable(): bool
    {
        $key = 'route_available_' . $this->guessRouteGate() . '_' . $this->getRouteName() . '_'. (auth()?->user()->id ?? 'guest');
        return Cache::remember($key, 60, function () {
            if (Auth::Guest()) {
                $guestRole = Role::findByName('guest');

                if ($this->routeName instanceof Model) {
                    $route = $this->guessRouteGate().'.'.($this->routeName->getModule()).'.index';
                    return $guestRole->hasPermissionTo($route, $this->guessRouteGate());
                }

                return $guestRole->hasPermissionTo($this->getRouteName(), $this->guessRouteGate());
            }

            if ($this->routeName instanceof Model) {
                $route = $this->guessRouteGate().'.'.($this->routeName->getModule()).'.index';
                return auth()->user()->can($route, $this->guessRouteGate());
            }

            return auth()->user()?->can($this->getRouteName(), $this->guessRouteGate()) ?? false;
        });
    }
}
