<?php

namespace ViartasCore\Core\Controllers;

use ViartasCore\Core\Forms\Common\ImagesUploadForm;
use ViartasCore\Core\Forms\Modules\ModuleEditForm;
use ViartasCore\Core\Tables\ModulesTable;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use ViartasCore\Core\Models\Module;

class ModulesController extends AccountingController
{
    /**
     * @param ModulesTable $table
     * @param Request $request
     * @return string|array|View|\Illuminate\Foundation\Application|Factory|Application
     */
    public function index(ModulesTable $table, Request $request): string|array|View|\Illuminate\Foundation\Application|Factory|Application
    {
        return $this->tableBuilder(
            $table, $request, theme_view('accounting.modules.index', compact('table'))
        );
    }

    /**
     * @param $id
     * @return \Illuminate\Foundation\Application|View|Factory|Application
     */
    public function list($id): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $module = Module::findOrFail($id);

        return theme_view('accounting.modules.list', [
            'element' => $module,
        ]);
    }

    /**
     * @param string $id
     * @param ModuleEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function edit(string $id, ModuleEditForm $form): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $module = Module::findOrFail($id);
        $form = $form->setAction('update')->build($module);

        return theme_view('accounting.modules.edit', [
            'form' => $form,
            'element' => $module,
        ]);
    }

    /**
     * @param string $id
     * @param ModuleEditForm $form
     * @return RedirectResponse
     */
    public function update(string $id, ModuleEditForm $form): RedirectResponse
    {
        Module::findOrFail($id)->update($form->all());

        return theme_redirect()->route('accounting.modules.edit', [
            'id' => $id,
        ])->with('success', __('module.updated'));
    }

    /**
     * @param string $id
     * @param Request $request
     * @return void
     */
    public function updateOneField(string $id, Request $request): void
    {
        Module::findOrFail($id)->update([
            $request->get('field') => $request->get('value')
        ]);
    }

    /**
     * @param string $id
     * @param Request $request
     * @return bool|RedirectResponse
     */
    public function destroy(string $id, Request $request): bool|RedirectResponse
    {
        if ($id !== '0') {
            $module = Module::findOrFail($id);
            $module->is_deleted = true;
            $module->save();

            return theme_redirect()->route('accounting.modules.index')
                ->with('success', __('module.deleted'));
        }

        $this->destroySelected($request);

        return true;
    }

    /**
     * @param Request $request
     * @return void
     */
    public function destroySelected(Request $request): void
    {
        foreach ($request->elements as $element) {
            $module = Module::findOrFail($element);
            $module->is_deleted = true;
            $module->save();
        }

        Session::flash('success', __('elements.selected.deleted'));
    }

    /**
     * @param $id
     * @param ImagesUploadForm $images
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function uploadImages($id, ImagesUploadForm $images): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $module = Module::findOrFail($id);
        $images = $images->build();

        $images->field('images')
            ->setModel(Module::class)
            ->setPrimaryKey('id')
            ->setPrimaryValue($id);

        return theme_view('accounting.modules.common', [
            'form' => $images,
            'element' => $module,
        ]);
    }

}
