<?php

namespace ViartasCore\Core\Providers;

use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Foundation\Support\Providers\RouteServiceProvider as ServiceProvider;
use Illuminate\Support\Facades\Blade;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\View;
use Schema;
use ViartasCore\Core\Facades\Viartas;
use function PHPUnit\Framework\directoryExists;

class BasePackageServiceProvider extends ServiceProvider
{
    /**
     * @var string
     */
    public string $package_directory = __DIR__;

    /**
     * @var array
     */
    public array $commands = [];

    /**
     * @var array|string[]
     */
    protected array $guards = [
        'api' => 'api',
        'default' => 'web',
    ];

    /**
     * @var string
     */
    protected string $config_tag_for_publishing = 'viartas-config';

    /**
     * @var string
     */
    public string $vendor = '';

    /**
     * @var string
     */
    public string $package = '';

    /**
     * @var string
     * %s - package directory
     */
    protected string $assets_vendor = '%s/../../resources/theme/%s/assets';

    /**
     * @var string
     * %s - vendor
     * %s - package
     */
    protected string $assets_public = 'vendor/%s/%s/assets';

    /**
     * @var string
     * %s - package directory
     * %s - vendor
     * %s - package
     */
    protected string $build_vendor = '%s/../../public/vendor/%s/%s';

    /**
     * @var string
     * %s - vendor
     * %s - package
     */
    protected string $build_public = 'vendor/%s/%s';

    /**
     * @return void
     */
    public function register(): void
    {
        parent::register();

        $this->addProviders(
            $this->package_directory
        );

        $this->app->booted($this->addScheduleCommands());
    }

    /**
     * Define your route model bindings, pattern filters, and other route configuration.
     * @return void
     */
    public function boot(): void
    {
        $this->definePackageVars();

        if ($this->app->runningInConsole()) {
            $this->addConfigs(
                $this->package_directory
            );
        }

        $this->addRoutes(
            $this->package_directory
        );

        if ($this->app->runningInConsole()) {
            $this->addMigrations(
                $this->package_directory
            );

            $this->addCommands(
                $this->artisanCommands()
            );
        }

        $this->addViews(
            $this->package_directory
        );

        $this->addComponents(
            $this->package_directory
        );

        $this->addAssets(
            $this->package_directory
        );

        $this->addPackageJs(
            $this->javascript()
        );

        $this->addPackageCss(
            $this->css()
        );

        $this->disableNodePackages(
            $this->disabledNodePackages()
        );
    }

    /**
     * @return void
     */
    public function definePackageVars(): void
    {
        $parts = explode("/", $this->package_directory);
        $this->package = $parts[count($parts) - 3];
        $this->vendor = $parts[count($parts) - 4];
    }

    /**
     * @param array $packages
     * @return void
     */
    public function disableNodePackages(array $packages): void
    {
        if (! $this->app->runningInConsole()) {
            Viartas::driver()->theme()->disabledNodePackages($packages);
        }
    }

    /**
     * @param array $scripts
     * @return void
     */
    public function addPackageJs(array $scripts): void
    {
        if (! $this->app->runningInConsole()) {
            Viartas::driver()->theme()->packageJs($scripts);
        }
    }

    /**
     * @param array $scripts
     * @return void
     */
    public function addPackageCss(array $scripts): void
    {
        if (! $this->app->runningInConsole()) {
            Viartas::driver()->theme()->packageCss($scripts);
        }
    }

    /**
     * @param $package_directory
     * @return void
     */
    public function addAssets($package_directory): void
    {
        if (! Schema::hasTable('applications')) {
            return;
        }

        if ($this->app->runningInConsole()) {

            $assets_public = sprintf(
                $this->assets_public,
                $this->vendor,
                $this->package,
            );

            $build_vendor = sprintf(
                $this->build_vendor,
                $package_directory,
                $this->vendor,
                $this->package,
            );

            $build_public = sprintf(
                $this->build_public,
                $this->vendor,
                $this->package,
            );

            $publishes = [];

            foreach (Viartas::driver()->app()->all() as $app) {

                $assets_vendor = sprintf(
                    $this->assets_vendor,
                    $package_directory,
                    $app->tag,
                );

                if (is_dir($assets_vendor)) {
                    $publishes = array_merge($publishes, [
                        $assets_vendor => public_path($assets_public),
                    ]);
                }
            }

            if (is_dir($build_vendor)) {
                $publishes = array_merge($publishes, [
                    $build_vendor => public_path($build_public),
                ]);
            }

            $this->publishes($publishes, 'viartas-config');
        }
    }

    /**
     * @param $package_directory
     * @return void
     */
    private function addRoutes($package_directory): void
    {
        $guards = $this->guards;

        $this->routes(function () use ($package_directory, $guards) {

            foreach (Viartas::driver()->app()->all() as $app) {

                $file = sprintf(
                    $package_directory . '/../../routes/%s.php',
                    $app->tag
                );

                $guard = $guards[$app->tag] ?? $guards['default'];

                $file_exists = Cache::remember($file.'-exists', '1200', function () use ($file) {
                    return file_exists($file);
                });

                if ($file_exists === false) {
                    continue;
                }

                $file_contents = Cache::remember($file.'-contents', '1200', function () use ($file) {
                    return file_get_contents($file);
                });

                $prefix = $app->tag;
                $locale = '';

                if (! $this->app->runningInConsole()) {
                    $prefix = $app->is_default
                        ? ''
                        : $app->tag;

                    if (! Viartas::driver()->locale()->current()->is_default) {
                        $locale = Viartas::driver()->locale()->current()->tag;
                    }
                }

                $locales = Viartas::driver()->locale()->all();

                Route::middleware($guard)
                    ->prefix($prefix)
                    ->as($app->tag.'.')
                    ->group(function () use ($file, $guard, $locales, $file_contents) {
                        if ($guard === 'api') {
                            Route::middleware('auth:sanctum')->group($file);
                        } else {
                            Route::prefix('')->group(function () use ($file_contents) {
                                eval('?>'.$file_contents);
                            });

                            foreach ($locales as $language) {
                                if (! $language->is_default) {
                                    Route::prefix($language->tag)->as($language->tag.'.')->group(function () use ($file_contents) {
                                        eval('?>'.$file_contents);
                                    });
                                }
                            }
                            //Route::middleware('auth')->group($file);
                        }
                    });
            }
        });
    }

    /**
     * @param $package_directory
     * @return void
     */
    public function addMigrations($package_directory): void
    {
        if (directoryExists($package_directory . '/../../database/migrations/')) {
            $this->loadMigrationsFrom(
                $package_directory . '/../../database/migrations/'
            );
        }
    }

    /**
     * @param array|null $commands
     * @return void
     */
    private function addCommands(?array $commands): void
    {
        $this->commands($commands ?? $this->commands);
    }

    /**
     * @return Callable
     */
    private function addScheduleCommands(): Callable
    {
        return function (): void
        {
            $schedule = $this->app->make(Schedule::class);
            $this->scheduleCommands($schedule);
        };
    }

    /**
     * @param $package_directory
     * @return void
     */
    private function addConfigs($package_directory): void
    {
        foreach ($this->configs() as $package_name => $app_name) {
            if (file_exists($package_directory . '/../../config/'.$package_name.'.php')) {
                $this->publishes([
                    $package_directory . '/../../config/'.$package_name.'.php' => config_path($app_name.'.php'),
                ], [$this->config_tag_for_publishing]);
            }
        }
    }

    /**
     * @param $package_directory
     * @return void
     */
    private function addProviders($package_directory): void
    {
        foreach ($this->providers() as $provider) {
            if (file_exists($package_directory . '/../../providers/'.$provider.'.php')) {
                $this->publishes([
                    $package_directory . '/../../providers/'.$provider.'.php' => $package_directory . '/../../../../../app/Providers/'.$provider.'.php',
                ], [$this->config_tag_for_publishing]);
            }
        }
    }

    /**
     * @param $package_directory
     * @return void
     */
    private function addViews($package_directory): void
    {
        if (directoryExists($package_directory . '/../../resources/')) {
            View::addLocation($package_directory . '/../../resources/');
        }
    }

    protected function addComponents($package_directory): void
    {
        if ($this->app->runningInConsole()) {
            return;
        }

        $app = Viartas::driver()->app()->current()->tag;

        if (directoryExists($package_directory . '/../../components/' . $app . '/')) {
            /**
             * Works like <x:package::folder.script />
             */

            Blade::anonymousComponentPath($package_directory . '/../../components/' . $app . '/', $this->package);

            /**
             * Works like
             * @component(package::folder.script)
             * @endcomponent
             */
            //$this->loadViewsFrom($package_directory . '/../../components/' . $app . '/', $package);
        }
    }

    /**
     * @param mixed $schedule
     * @return void
     */
    public function scheduleCommands(mixed $schedule): void
    {

    }

    /**
     * @return string[]
     */
    public function artisanCommands(): array
    {
        return [

        ];
    }

    /**
     * @return array
     */
    public function configs(): array
    {
        return [

        ];
    }

    /**
     * @return array
     */
    public function providers(): array
    {
        return [

        ];
    }

    /**
     * @return array
     */
    public function javascript(): array
    {
        return [

        ];
    }

    /**
     * @return array
     */
    public function css(): array
    {
        return [

        ];
    }

    /**
     * @return array
     */
    public function disabledNodePackages(): array
    {
        return [

        ];
    }
}
