<?php

namespace ViartasCore\Core\Drivers;

use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Collection;
use Schema;
use ViartasCore\Core\Contracts\RouteGuesserContract;
use ViartasCore\Core\Models\Application;
use ViartasCore\Core\Models\Locale;
use ViartasCore\Core\Models\Segment;

class AppDriver
{
    /**
     * @var Application|null
     */
    private ?Application $application = null;

    /**
     * @var Locale|null
     */
    private ?Locale $locale = null;

    /**
     * @var array
     */
    private array $packageRouteGuessers = [];

    private string $slug = '';

    /**
     *
     */
    public function __construct()
    {
        $this->boot();
    }

    /**
     * @return self
     */
    private function boot(): self
    {
        if (!app()->runningInConsole()) {
            $this->setDefaults();
            $this->detect();
        }

        return $this;
    }

    /**
     * @return void
     */
    private function setDefaults(): void
    {
        $this->application = Application::default();
        $this->locale = Locale::default();
    }

    /**
     * @param Application|Locale $detected
     * @return void
     */
    private function setDetected (Application|Locale $detected): void
    {
        $detected instanceof Application
            ? $this->application = $detected
            : $this->locale = $detected;
    }

    /**
     * @return RedirectResponse|self
     */
    private function detect(): self|RedirectResponse
    {
        $availableSegments = Segment::available()->get();

        $position = 1;
        $segments = request()->segments();
        $segment = request()->segment($position);

        $slug = $segments;

        foreach ($availableSegments as $availableSegment) {

            $detected = $availableSegment->getDetected($segment);

            if ($detected) {
                if ($detected->is_default) {
                    unset($segments[$position-1]);
                    return $this->redirect($segments);
                }

                if ($detected->is_active) {
                    $this->setDetected($detected);

                    unset($slug[$position-1]);
                    $position++;
                }
            }

            $segment = request()->segment($position);
        }

        $this->slug = implode("/", $slug);

        app()->setLocale($this->locale->tag);

        return $this;
    }

    /**
     * @param array $segments
     * @return RedirectResponse
     */
    private function redirect(array $segments): RedirectResponse
    {
        return redirect(implode('/', $segments) . $this->getQueryString())->send();
    }

    /**
     * @return string
     */
    public function getQueryString(): string
    {
        return str_replace(request()->url(), '', request()->fullUrl());
    }

    /**
     * @return Application|null
     */
    public function current(): ?Application
    {
        return $this->application;
    }

    /**
     * @return Collection
     */
    public function all(): Collection
    {
        if (! Schema::hasTable('applications')) {
            return collect([]);
        }

        return Application::query()->get();
    }

    /**
     * @return Locale|null
     */
    public function locale(): ?Locale
    {
        return $this->locale;
    }

    /**
     * @return string
     */
    public function path(): string
    {
        return $this->current()?->path() ?? '';
    }

    /**
     * @return string
     */
    public function route(): string
    {
        $tag = $this->current()->tag;
        return ($tag === 'web' ? '' : '/'.$tag);
    }


    /**
     * @return string
     */
    public function getSlug(): string
    {
        return $this->slug;
    }

    /**
     * @return string
     */
    public function getSlugLastSegment(): string
    {
        $slug = $this->getSlug();
        $parts = explode("/", $slug);
        return $parts[count($parts)-1];
    }

    /**
     * @return string
     */
    public function logout(): string
    {
        $tag = $this->current()->tag;
        return $tag . '.logout.index';
    }

    /**
     * @return string
     */
    public function login(): string
    {
        $tag = $this->current()->tag;
        return $tag . '.login.index';
        //return ($tag === 'web' ? '' : $tag . '.') . 'login';
    }

    /**
     * @param RouteGuesserContract $routeGuesser
     * @return void
     */
    public function addPackageRouteGuesser(RouteGuesserContract $routeGuesser): void
    {
        $priority = $routeGuesser->priority();
        while (isset($this->packageRouteGuessers[$priority])) {
            $priority ++;
        }

        $this->packageRouteGuessers[$priority] = $routeGuesser;
    }

    /**
     * @return array
     */
    public function getPackagesRouteGuessers(): array
    {
        ksort($this->packageRouteGuessers);

        return $this->packageRouteGuessers;
    }
}
