<?php

namespace ViartasCore\Core\Controllers;

use ViartasCore\Core\Forms\Menus\MenuEditForm;
use ViartasModules\CustomLinks\Tables\Accounting\MenuCustomLinksTable;
use ViartasCore\Core\Tables\MenuModulesTable;
use ViartasModules\Pages\Tables\Accounting\MenuPagesTable;
use ViartasModules\PostCategories\Tables\Accounting\MenuPostCategoriesTable;
use ViartasCore\Core\Tables\MenusTable;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use ViartasCore\Core\Models\Menu;
use ViartasCore\Core\Models\MenuHistory;
use ViartasCore\Core\Models\MenuItem;
use ViartasShop\Brands\Tables\Accounting\MenuBrandsTable;
use ViartasShop\ProductCategories\Tables\Accounting\MenuProductCategoriesTable;

class MenusController extends AccountingController
{
    /**
     * @param MenusTable $table
     * @param Request $request
     * @return string|array|View|\Illuminate\Foundation\Application|Factory|Application
     */
    public function index(MenusTable $table, Request $request): string|array|View|\Illuminate\Foundation\Application|Factory|Application
    {
        return $this->tableBuilder(
            $table, $request, theme_view('accounting.menus.index', compact('table'))
        );
    }

    /**
     * @param MenuEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function create(MenuEditForm $form): View|Factory|Application|\Illuminate\Foundation\Application
    {
        return theme_view('accounting.menus.edit', [
            'form' => $form->build(),
            'element' => new Menu(),
        ]);
    }

    /**
     * @param MenuEditForm $form
     * @return RedirectResponse
     */
    public function store(MenuEditForm $form): RedirectResponse
    {
        $menu = Menu::query()->create($form->all());

        return theme_redirect()
            ->route('accounting.menus.edit', ['id' => $menu->id,])
            ->with('success', __('menu.created'));
    }

    /**
     * @param string $id
     * @param MenuEditForm $form
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function edit(string $id, MenuEditForm $form): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $menu = Menu::findOrFail($id);
        $form = $form->setAction('update')->build($menu);

        return theme_view('accounting.menus.edit', [
            'form' => $form,
            'element' => $menu,
        ]);
    }

    /**
     * @param string $id
     * @param MenuEditForm $form
     * @return RedirectResponse
     */
    public function update(string $id, MenuEditForm $form): RedirectResponse
    {
        Menu::findOrFail($id)->update($form->all());

        return theme_redirect()->route('accounting.menus.edit', [
            'id' => $id,
        ])->with('success', __('menu.updated'));
    }

    /**
     * @param string $id
     * @param Request $request
     * @return void
     */
    public function updateOneField(string $id, Request $request): void
    {
        Menu::findOrFail($id)->update([
            $request->get('field') => $request->get('value')
        ]);
    }

    /**
     * @param string $id
     * @param MenuModulesTable $modules
     * @param MenuPagesTable $pages
     * @param MenuPostCategoriesTable $post_categories
     * @param MenuCustomLinksTable $custom_links
     * @param MenuProductCategoriesTable $product_categories
     * @param MenuBrandsTable $brands
     * @return \Illuminate\Foundation\Application|View|Factory|Application
     */
    public function list(
        string $id,
        MenuModulesTable $modules,
        MenuPagesTable $pages,
        MenuPostCategoriesTable $post_categories,
        MenuCustomLinksTable $custom_links,
        MenuProductCategoriesTable $product_categories,
        MenuBrandsTable $brands,
    ): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $menu = Menu::findOrFail($id);

        return theme_view('accounting.menus.list', [
            'element' => $menu,
            'menu_items' => [
                'modules' => [
                    'title' => __('Modules'),
                    'table' => $modules,
                ],
                'pages' => [
                    'title' => __('Pages'),
                    'table' => $pages,
                ],
                'post_categories' => [
                    'title' => __('Post categories'),
                    'table' => $post_categories,
                ],
                'custom_links' => [
                    'title' => __('Custom links'),
                    'table' => $custom_links,
                ],
                'product_categories' => [
                    'title' => __('Custom links'),
                    'table' => $product_categories,
                ],
                'brands' => [
                    'title' => __('Custom links'),
                    'table' => $brands,
                ]
            ],
        ]);
    }

    /**
     * @param string $id
     * @param Request $request
     * @return void
     */
    public function save(string $id, Request $request): void
    {
        $latest = MenuHistory::query()
            ->where('menu_id', $id)
            ->max('history_id') ?? 0;

        $latest++;

        $menuItems = MenuItem::query()
            ->where('menu_id', $id)
            ->get();

        foreach ($menuItems as $item) {
            MenuHistory::query()->create(array_merge(
                $item->toArray(),
                ['history_id' => $latest]
            ));
        }

        MenuItem::query()->where('menu_id', $id)->delete();

        $list = json_decode($request->get('list'));

        if (json_last_error() === JSON_ERROR_NONE && is_iterable($list)) {
            $this->saveMenuItem($id, $list);
        } else {
            // Consider logging or throwing an exception here
            throw new \InvalidArgumentException('Invalid JSON list provided.');
        }

        session()->flash('success', __('menu.saved'));
    }

    /**
     * @param string $menu_id
     * @param mixed $elements
     * @param int|null $parent
     * @return void
     */
    public function saveMenuItem(string $menu_id, iterable $elements, ?int $parent = null): void
    {
        $sort = 0;

        foreach ($elements as $element) {
            $sort++;

            $decodedId = base64_decode($element->id, true);
            if ($decodedId === false) {
                continue; // Skip invalid ID
            }

            $hash = explode('-', $decodedId);
            if (count($hash) < 3) {
                continue; // Skip if the expected parts are not present
            }

            $menuElementData = [
                'menu_id' => $menu_id,
                'element' => (int)$hash[0],
                'class' => $hash[1],
                'has_children' => $hash[2],
                'sort' => $sort,
            ];

            if ($parent !== null) {
                $menuElementData['menu_item_id'] = $parent;
            }

            $menuItem = MenuItem::query()->create($menuElementData);

            if (!empty($element->children)) {
                $this->saveMenuItem($menu_id, $element->children, $menuItem->id);
            }
        }
    }

    /**
     * @param string $id
     * @param Request $request
     * @return bool|RedirectResponse
     */
    public function destroy(string $id, Request $request): bool|RedirectResponse
    {
        if ($id !== '0') {
            Menu::findOrFail($id)->delete();

            return theme_redirect()->route('accounting.menus.index')
                ->with('success', __('menu.deleted'));
        }

        $this->destroySelected($request);

        return true;
    }

    /**
     * @param Request $request
     * @return void
     */
    public function destroySelected(Request $request): void
    {
        foreach ($request->elements as $element) {
            Menu::find($element)->delete();
        }

        Session::flash('success', __('elements.selected.deleted'));
    }

    /**
     * @param MenuModulesTable $table
     * @param Request $request
     * @return string|array|View|\Illuminate\Foundation\Application|Factory|Application
     */
    public function modules(MenuModulesTable $table, Request $request): string|array|View|\Illuminate\Foundation\Application|Factory|Application
    {
        return $this->tableBuilder(
            $table, $request, theme_view('accounting.modules.index', compact('table'))
        );
    }

    /**
     * @param MenuPagesTable $table
     * @param Request $request
     * @return string|array|View|\Illuminate\Foundation\Application|Factory|Application
     */
    public function pages(MenuPagesTable $table, Request $request): string|array|View|\Illuminate\Foundation\Application|Factory|Application
    {
        return $this->tableBuilder(
            $table, $request, theme_view('accounting.pages.index', compact('table'))
        );
    }

    /**
     * @param MenuPostCategoriesTable $table
     * @param Request $request
     * @return string|array|View|\Illuminate\Foundation\Application|Factory|Application
     */
    public function post_categories(MenuPostCategoriesTable $table, Request $request): string|array|View|\Illuminate\Foundation\Application|Factory|Application
    {
        return $this->tableBuilder(
            $table, $request, theme_view('accounting.post-categories.index', compact('table'))
        );
    }

    /**
     * @param MenuCustomLinksTable $table
     * @param Request $request
     * @return string|array|View|\Illuminate\Foundation\Application|Factory|Application
     */
    public function custom_links(MenuCustomLinksTable $table, Request $request): string|array|View|\Illuminate\Foundation\Application|Factory|Application
    {
        return $this->tableBuilder(
            $table, $request, theme_view('accounting.custom-links.index', compact('table'))
        );
    }

    /**
     * @param MenuProductCategoriesTable $table
     * @param Request $request
     * @return string|array|View|\Illuminate\Foundation\Application|Factory|Application
     */
    public function product_categories(MenuProductCategoriesTable $table, Request $request): string|array|View|\Illuminate\Foundation\Application|Factory|Application
    {
        return $this->tableBuilder(
            $table, $request, theme_view('accounting.product-categories.index', compact('table'))
        );
    }

    /**
     * @param MenuBrandsTable $table
     * @param Request $request
     * @return string|array|View|\Illuminate\Foundation\Application|Factory|Application
     */
    public function brands(MenuBrandsTable $table, Request $request): string|array|View|\Illuminate\Foundation\Application|Factory|Application
    {
        return $this->tableBuilder(
            $table, $request, theme_view('accounting.brands.index', compact('table'))
        );
    }
}
