<?php

namespace ViartasBuilders\Tables;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Blade;
use ViartasCore\Core\Facades\Viartas;
use ViartasCore\Core\Models\User;

class DataTables
{
    private Builder $query;
    private Model $model;
    private string $type = 'card';
    private string $id = "table-id";
    private string $class = "table-class";
    private string $title = "example";
    private string  $filters = "";
    private string  $route = "";
    private array $request;
    private Builder|Collection $data;

    protected array $emptyValues = [
        'n/a', null,
        //0, 1,
        0,
        //true, false,
        false,
    ];

    private string $indexColumn = 'id';
    private array $rows = [];

    private int $rowsCount;

    public function __construct(Model $model, Request $request)
    {
        $this->model = $model;
        $this->query = $model::query();
        $this->request = $request->toArray();
        return $this;
    }

    public function setIndexColumn (string $indexColumn): self
    {
        $this->indexColumn = $indexColumn;
        return $this;
    }

    public function addRow (DataTablesRow $row): self
    {
        $this->rows[] = $row;
        return $this;
    }

    public function addRows (array $rows): self
    {
        foreach ($rows as $row) {
            $this->addRow($row);
        }

        return $this;
    }

    public function getRow(string $field): mixed
    {
        foreach ($this->getRows() as $row) {
            if ($row->get('field') == $field) {
                return $row;
            }
        }

        return false;
    }

    public function getRows()
    {
        $rows = [];
        foreach ($this->rows as $row) {
            if ($row->visible()) {
                $rows[] = $row;
            }
        }
        return $rows;
    }

    public function getAllRows()
    {
        return $this->rows;
    }

    public function build (): self
    {
        $this->countRows();
        $this->selectRows();
        $this->filter();
        $this->search();

        return $this;
    }

    private function countRows (): void
    {
        $this->rowsCount = $this->query->select($this->indexColumn)->get()->count();
    }

    private function selectRows (): void
    {
        $rows = [];
        foreach ($this->rows as $row) {
            if ($row->get('getable')) {
                $rows[] = $row->get('field');
            }
        }

        $this->query->select($rows);
    }

    private function isRequestColumnFilterable (int $columnIndex): bool
    {
        return
            isset($this->request['columns'][$columnIndex])
            && $this->request['columns'][$columnIndex]['searchable'] == 'true'
            && $this->request['columns'][$columnIndex]['search']['value'] != '';
    }

    private function isRequestColumnHasRange (int $columnIndex): bool
    {
        return preg_match('/\~/', $this->request['columns'][$columnIndex]['search']['value']);
    }

    private function isRequestColumnHasEmptyValue ($searchValue): bool
    {
        return in_array($searchValue, $this->emptyValues);
    }

    /**
     * @return Model
     */
    public function model(): Model
    {
        return $this->model;
    }

    private function filter (): self
    {
        //for ($i=0;$i<count($this->getRows());$i++) {
        foreach ($this->getRows() as $i => $row) {
            //$row = $this->getRows()[$i];

            if (!$row->get('selectable') && !$row->get('searchable') && !$row->get('range')) {
                continue;
            }

            $search_value = $this->request['columns'][$i]['search']['value'];

            if (trim($search_value) == '') {
                continue;
            }

            $searches = [];

            $exp = explode(" ", $search_value);
            foreach ($exp as $v) {
                if (trim($v) != '') {
                    $searches[$v] = $row->get('field');
                }
            }

            if ($row->dt_filter !== false) {
                if (is_callable([$row, 'dt_filter'])) {
                    $row->{'dt_filter'}($this->query, $searches);
                }

                continue;
            }

            if (!$this->isRequestColumnFilterable($i)) {
                continue;
            }

            if ($this->isRequestColumnHasRange($i)) {
                $exp = explode("~", $search_value);

                if ($exp[0] != '') {
                    $this->query->where($row->get('field'), '>=', $exp[0]);
                }
                if ($exp[1] != '') {
                    $this->query->where($row->get('field'), '<=', $exp[1]);
                }

                continue;
            }

            if ($this->isRequestColumnHasEmptyValue($search_value)) {
                $this->query->where($row->get('field'), '=', '');

                continue;
            }

            foreach ($searches as $key => $value) {
                $this->query->where($value, 'like', '%' . $key . '%');
            }
        }

        return $this;
    }

    private function search ()
    {
        if ($this->request['search']['value'] != '') {

            $rows = $this->getRows();

            $exp = explode(" ", $this->request['search']['value']);
            foreach ($exp as $word) {
                if ($word != '') {
                    $this->query->where(function ($builder) use ($rows, $word) {
                        foreach ($rows as $i => $row) {
                        //for ($i = 0; $i < count($rows); $i++) {

                            //$row = $this->rows[$i];

                            if ($this->request['columns'][$i]['searchable'] == 'true' && $row->get('getable')) {
                                $builder->orWhere($row->get('field'), 'like', '%' . $word . '%');
                            }
                        }
                    });
                }
            }
        }

        return $this;
    }

    private function order(): self
    {
        $orderRow = $this->request['order'][0]['column'];
        $field = $this->rows[$orderRow] ?? false;
        $direction = $this->request['order'][0]['dir'] ?? 'asc';

        if ($field) {
            $this->query = $field->dt_order($this->query, $direction) ?? $this->query;
        }

        return $this;
    }

    public function fetchData (): array
    {
        $resultCount = $this->query->groupBy($this->indexColumn)->get()->count();

        $this->order();


        /*if ($this->rows['dbColumns'][$this->query['order'][0]['column']] != '')
            $this->builder->orderBy($this->rows['dbColumns'][$this->query['order'][0]['column']], $this->query['order'][0]['dir']);
        else {
            if (method_exists($this, 'dt_order_' . $this->rows['tableColumns'][$this->query['order'][0]['column']]))
                $this->{'dt_order_' . $this->rows['tableColumns'][$this->query['order'][0]['column']]}($this->query['order'][0]['dir']);
        }*/

        $this->data = $this->query
            ->groupBy($this->indexColumn)
            ->skip($this->request['start'])->take($this->request['length'])
            ->get();

        $data = [];

        foreach ($this->data as $datum) {

            $row = [];
            foreach ($this->getRows() as $field) {

                $value = $datum->{$field->get('field')};

                if (is_callable([$field, 'dt_render'])) {
                    $value = $field->{'dt_render'}($datum);
                }

                if (!empty($field->buttons())) {
                    foreach ($field->buttons() as $button) {
                        $btn = $field->button($button->title());
                        $btn->setModel($datum);
                        $btn->bindRoute($btn->{'dt_route'}($datum));

                        if (!$btn->route()->isAvailable()) {
                            $field->removeButton($button);
                        }
                    }

                    $value .= Blade::render('<x-buttons :field="$field"></x-buttons>', [
                        'field' => $field,
                    ]);
                }

                $row[] = $value;
            }

            $data[] = $row;
        }

        return [
            'draw' => $this->request['draw'],
            'recordsTotal' => $this->rowsCount,
            'recordsFiltered' => $resultCount,
            'data' => $data,
        ];
    }

    public function render()
    {
        $component = sprintf('<x-table-%s :table="$table"></x-table->', $this->getType());
        return Blade::render($component, [
            'table' => $this,
        ]);
    }

    /**
     * @param string $type
     * @return self
     */
    public function setType (string $type): self
    {
        return $this->setAttribute('type', $type);
    }

    public function getType(): string
    {
        return $this->type;
    }

    public function getFilters(): string
    {
        $filters = $this->getRow($this->request['filter'])->dt_filters($this->model);
        $response = '<option value="">All</option>';

        $response .= $this->renderFilters($filters);

        return $response;
    }

    private function renderFilters(array $filters, int $depth = -1): string
    {
        $prefix = str_repeat('-', $depth + 1);

        $response = '';
        foreach ($filters as $filter) {
            $response .= '<option value="'.$filter['value'].'">'.$prefix.$filter['label'].'</option>';
            if (!empty($filter['values'])) {
                $response .= $this->renderFilters($filter['values'], $depth ++);
            }
        }

        return $response;
    }


    public static function treeToFilters(\Illuminate\Support\Collection $tree, $key, $label): array
    {
        $response = [];

        foreach ($tree as $datum) {
            $response[] = [
                'label' => $datum->{$label},
                'value' => $datum->{$key},
                'values' => self::renderTree($datum->children, $key, $label),
            ];
        }

        return $response;
    }

    public static function renderTree (Collection $tree, $key, $label): array
    {
        if (!$tree->count()) {
            return [];
        }

        $response = [];

        foreach ($tree as $datum) {
            $response[] = [
                'label' => $datum->{$label},
                'value' => $datum->{$key},
                'values' => self::renderTree($datum->children, $key, $label),
            ];
        }

        return $response;
    }

    /**
     * @param string $key
     * @param mixed $value
     * @return self
     */
    private function setAttribute(string $key, mixed $value): self
    {
        $this->{$key} = $value;
        return $this;
    }

    /**
     * @param string $route
     * @return self
     */
    public function setRoute(string $route): self
    {
        return $this->setAttribute('route', $route);
    }

    /**
     * @return string
     */
    public function filters(): string
    {
        return $this->filters;
    }

    /**
     * @param string $filters
     * @return self
     */
    public function setFilters(string $filters): self
    {
        return $this->setAttribute('filters', $filters);
    }

    /**
     * @return string
     */
    public function route(): string
    {
        return $this->route;
    }

    /**
     * @param string $title
     * @return self
     */
    public function setTitle(string $title): self
    {
        return $this->setAttribute('title', $title);
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return $this->title;
    }

    /**
     * @param string $id
     * @return self
     */
    public function setId(string $id): self
    {
        return $this->setAttribute('id', $id);
    }

    /**
     * @return string
     */
    public function id(): string
    {
        return $this->id;
    }

    /**
     * @param string $class
     * @return self
     */
    public function setClass(string $class): self
    {
        return $this->setAttribute('class', $class);
    }

    /**
     * @return string
     */
    public function class(): string
    {
        return $this->class;
    }


    /**
     * @return DataTablesRow
     */
    public function button(string $field = 'action', string $title = 'Actions'): DataTablesRow
    {
        return (new DataTablesRow())
            ->field($field)
            ->title(__($title))
            ->setGetable(false)
            ->setOrderable(false);
    }

    public function fieldId(): DataTablesRow
    {
        $id = new DataTablesRow();

        $id->field('id')
            ->title('ID')
            ->setSearchable()
            ->render(function (Model $model) {
                return '#'.$model->id;
            })
            ->order(function (Builder $builder, string $direction) {
                return $builder->orderBy('id', $direction);
            });

        return $id;
    }

    public function fieldTitle(): DataTablesRow
    {
        $title = new DataTablesRow();

        $title->field('title')
            ->title(__('Title'))
            ->setSearchable()
            ->render(function (Model $model) {
                return $model->title;
            })
            ->order(function (Builder $builder, string $direction) {
                $builder->orderBy('title->'.Viartas::driver()->locale()->current()->tag, $direction);
            });

        return $title;
    }

    public function fieldUser(): DataTablesRow
    {
        $user = new DataTablesRow();

        $user->field('user_id')
            ->title(__('Author'))
            ->render(function (Model $model) {
                return $model->user->name;
            })
            ->setSelectable()
            ->filters(function (Model $model) {
                $response = [];

                $data = User::select(['id', 'name'])->groupBy('id')->get();

                foreach ($data as $datum) {
                    $response[] = [
                        'label' => $datum->name,
                        'value' => $datum->id,
                        'values' => [],
                    ];
                }

                return $response;
            })
            ->order(function (Builder $builder, string $direction) {
                return $builder->withAggregate('user','name')
                    ->orderBy('user_name', $direction);
            });

        return $user;
    }

    public function fieldCreatedAt(): DataTablesRow
    {
        return (new DataTablesRow())
            ->field('created_at')
            ->title(__('Created at'))
            ->setRange(true)
            ->render(function (Model $model) {
                return $model->created_at->format("Y-m-d H:i:s");
            });
    }

    public function fieldCheckboxMassActions(): DataTablesRow
    {
        $checkbox = new DataTablesRow();

        $checkbox->field('checkbox')
            ->title(__('Select'))
            ->setGetable(false)
            ->setCheckbox()
            ->setOrderable(false);

        $checkbox->render(function (Model $model) {
            return Blade::render('<x-checkbox :element="$element" />', [
                'element' => $model,
            ]);
        });

        return $checkbox;
    }

    public function fieldSwitch(array $filters = []): DataTablesRow
    {
        $switch = new DataTablesRow();

        $switch->field('is_visible')
            ->title(__('Visibility'))
            ->setSelectable()
            ->setOrderable();

        $switch->render(function (Model $model) use ($switch) {

            $field = $switch->get('field');
            $route = $model->routeFieldChange($field);

            return $route->isAvailable()
                    ? Blade::render('<x-switch :element="$element" :key="$key" />', [
                            'element' => $model,
                            'key' => $field,
                        ])
                    : (
                        $model->{$field} == 1
                            ? '<span class="badge badge-primary">'.__('Visible') .'</span>'
                            : '<span class="badge badge-secondary">'.__('Invisible') .'</span>'
                        );
        });

        $switch->filters(function (Model $model) use ($filters) {
            if (empty($filters)) {
                return [
                    ['label' => __('Visible'), 'value' => "1",],
                    ['label' => __('Invisible'), 'value' => "0",],
                ];
            }

            $response = [];
            foreach ($filters as $filter) {
                $response[] = [
                    'label' => $filter['label'],
                    'value' => $filter['value'],
                ];
            }

            return $response;
        });

        return $switch;
    }

    public function fieldActions(array $actions = []): DataTablesRow
    {
        $field = new DataTablesRow();

        $field->field('status')
            ->title(__('Status'))
            ->setGetable(false)
            ->setSelectable()
            ->setOrderable();

        $field->render(function (Model $model) use ($field, $actions) {

            $key = $field->get('field');
            $route = $model->routeFieldChange($key);

            return $route->isAvailable()
                ? Blade::render('<x-actions :element="$element" :field="$field" :actions="$actions" />', [
                    'element' => $model,
                    'field' => $field,
                    'actions' => $actions,
                ])
                : '<span class="badge badge-light-info">'.($model->{$key} ?? 'value').'</span>';
        });

        $field->filters(function (Model $model) use ($actions) {
            return $actions;
        });

        return $field;
    }
}
