<?php

namespace ViartasBuilders\Menus;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use ViartasCore\Core\Drivers\RouteDriver as Route;

class MenuItem
{
    /**
     * @var Model
     */
    private Model $model;

    /**
     * @var string
     */
    private string $title = '';

    /**
     * @var Route|null
     */
    private ?Route $route = null;

    /**
     * @var Collection
     */
    private Collection $children;

    /**
     * @param Model $model
     */
    public function __construct(Model $model)
    {
        $this->model = $model;
        $this->children = collect([]);

        if ($model instanceof \ViartasCore\Core\Models\MenuItem) {
            return $this->buildItem();
        }

        return $this->buildElement();
    }

    /**
     * @param string|null $title
     * @return string
     */
    public function title(?string $title = null): string
    {
        if (!is_null($title) && $title !== '') {
            $this->title = $title;
        }

        return $this->title;
    }

    /**
     * @param Route|null $route
     * @return Route|null
     */
    public function route(?Route $route = null): ?Route
    {
        if (!is_null($route)) {
            $this->route = $route;
        }

        return $this->route;
    }

    /**
     * @return self
     */
    private function buildItem(): self
    {
        $title = $this->model->item->title;

        $this->title(
            !empty($title) ? $title : ($this->model->item->route_name ?? '')
        );

        if (isset($this->model->item->route_name)) {
            $this->route(
                new Route(
                    $this->model->item->route_name,
                    json_decode($this->model->item->route_attributes, 1)
                )
            );
        } else {
            if (method_exists($this->model->item, 'generateRoute')) {
                $this->route(
                    new Route(
                        $this->model->item,
                    )
                );
            }
        }

        return $this->buildChildren();
    }

    /**
     * @return self
     */
    private function buildElement(): self
    {
        if (method_exists($this->model, 'generateRoute')) {
            $this->route(
                new Route(
                    $this->model,
                )
            );
        }

        $this->title($this->model->title ?? '');

        return $this->buildChildren();
    }

    /**
     * @return self
     */
    private function buildChildren(): self
    {
        if ($this->model instanceof \ViartasCore\Core\Models\MenuItem) {
            $items = $this->model->items;

            if ($this->model->has_children) {
                $items = $items->merge($this->model->item?->children ?? collect());
            }
        } else {
            $items = $this->model->children;
        }

        foreach ($items as $item) {

            if ($item->element === 0) {
                $model = (new $item->class);

                $rootItems = $model->whereNull($model->getParentKeyName())->orderBy('title')->get();

                foreach ($rootItems as $rootItem) {
                    $this->children->push(new MenuItem($rootItem));
                }

            } else {
                $this->children->push(new MenuItem($item));
            }
        }

        return $this;
    }

    /**
     * @return Collection
     */
    public function children(): Collection
    {
        return $this->children;
    }
}
