<?php

namespace ViartasBuilders\Forms;

use ViartasBuilders\Forms\Contracts\FieldContract;
use ViartasBuilders\Forms\Contracts\GroupContract;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Blade;
use JetBrains\PhpStorm\NoReturn;

class Form extends FormRequest
{
    private array $formGroups = [];
    public array $formFields = [];

    private string $title = "";
    private string $id = "";
    private string $class = "";
    private string $action = "default";
    private string $_method = "POST";
    private array $routes = [
        'default' => [
            'route' => '',
            'routeAttributes' => [],
        ],
    ];
    private array $routeAttributes = [];

    private Model $model;

    public function __construct()
    {
        parent::__construct();
        $this->buildOptions();
    }

    /**
     * @return array
     */
    protected function formOptions(): array
    {
        return [];
    }

    /**
     * @return $this
     */
    public function build(Model|null $model = null): self
    {
        if ($model && $model->id) {
            $this->model = $model;
        }

        if (empty($this->routeAttributes)) {

            if ($model && $model->id) {
                $this->setRouteAttributes([
                    'id' => $model->id
                ]);
            }

            $routeAttributes = [];

            if ($this->routes[$this->action]['routeAttributes']) {
                foreach ($this->routes[$this->action]['routeAttributes'] as $key => $value) {
                    $routeAttributes[$key] = $model->{$value} ?? $value;
                }

                $this->setRouteAttributes($routeAttributes);
            }

            $this->setMethod($this->routes[$this->action]['_method'] ?? $this->_method);
        }

        foreach ($this->fields() as $group_name => $fields) {

            $layout = $this->groupAttributes()[$group_name]['layout'] ?? Group::SIMPLE;

            $group = (new $layout($this->groupAttributes()[$group_name] ?? []))
                ->setName($group_name);

            if (isset($this->groupAttributes()[$group_name]['title'])) {
                $group = $group->setTitle($this->groupAttributes()[$group_name]['title']);
            }

            $this->addGroup($group);

            foreach ($fields as $name => $field) {

                if (is_array($field)) {
                    $field = (new $field['type']($field))
                        ->setName($name)->setGroup($group_name);
                }

                if ($field instanceof FieldContract) {
                    $field->setName($name)->setGroup($group_name);
                }

                $this->addField($field);

                if ($field->confirmation()) {

                    $confirmation = clone $field;

                    $confirmation->setName($name.'_confirmation');
                    $confirmation->setTitle($field->title().' '.__('confirmation'));
                    $confirmation->setRules(array_merge(
                        $field->rules(), ['same:'.$name]
                    ));

                    $this->addField($confirmation);
                }
            }
        }

        if (!$model) {
            return $this->buildRequest();
        }

        //return $this->fillFields($model->toArray() ?? [])->buildRequest();
        return $this->buildRequest()
            ->selfIgnoreFields()
            ->fillFields($model->toArray() ?? []);
    }

    private function selfIgnoreFields(): self
    {
        foreach ($this->formFields as $key => $formField) {

            $rules = $formField->rules();
            foreach ($rules as $k => $rule) {
                if (preg_match('/^(unique)\:/i', $rule)) {

                    $parts = explode(',', $rule);
                    if (isset($parts[2])) {
                        $field = $parts[2];
                        unset($parts[2]);

                        $rule = implode(',', $parts);

                        if (!$formField->model()) {
                            continue;
                        }

                        if (!$formField->primaryKey()) {
                            continue;
                        }

                        if (!$formField->primaryValue()) {
                            continue;
                        }

                        $model = $formField->model();
                        $model = new $model;

                        $model = $model->where($formField->primaryKey(), $formField->primaryValue())
                                ->firstOrFail();

                        $rule .= ','.$model->{$field};

                        $rules[$k] = $rule;
                    }
                }
            }

            $this->formFields[$key]->setRules($rules);
        }

        return $this;
    }

    /**
     * @return self
     */
    private function buildOptions(): self
    {
        foreach ($this->formOptions() as $key => $value) {
            $this->{$key} = $value;
        }

        return $this;
    }

    /**
     * @param string $key
     * @param mixed $value
     * @return self
     */
    private function setAttribute(string $key, mixed $value): self
    {
        $this->{$key} = $value;
        return $this;
    }

    /**
     * @param array $fieldsValues
     * @return $this
     */
    public function fillFields(array $fieldsValues): self
    {

        foreach ($this->formFields as $formKey => $field) {
            foreach ($fieldsValues as $valueKey => $value) {
                if ($field->request_name() == $valueKey) {
                    $this->formFields[$formKey]->setValue($value);
                }

                if (is_array($value)) {
                    foreach ($value as $k => $v) {
                        if ($field->request_name() == $k) {
                            $this->formFields[$formKey]->setValue($v);
                        }
                    }
                }
            }

            if ($field->primaryKey() !== null) {
                $this->formFields[$formKey]->setPrimaryValue($this->model->{$field->primaryKey()});
            }
        }

        return $this;
    }

    /**
     * @return self
     */
    private function buildRequest(): self
    {
        foreach ($this->formFields as $key => $value) {
            $this->formFields[$key]->buildRequest();
        }
        return $this;
    }

    /**
     * @return string
     */
    public function renderForm(): string
    {
        if (empty($this->gridAttributes())) {
            $response = sprintf(
                '<x-form :form="$form">%s</x-form>',
                $this->renderGroups()
            );
        } else {
            $response = sprintf(
                '<x-form :form="$form">%s</x-form>',
                $this->renderGrid()
            );
        }

        return Blade::render($response, [
            'form' => $this,
        ]);
    }

    public function renderGrid(): string
    {
        $response = '';
        foreach ($this->gridAttributes() as $key => $grid) {
            $response .= sprintf(
                Blade::render(
                    '<x-form-grid :grid="$grid">%s</x-form-grid>', [
                    'grid' => $grid,
                ]),
                $this->renderGroups($key)
            );
        }

        return $response;
    }

    /**
     * @param string|bool $grid
     * @return string
     */
    public function renderGroups(string|bool $grid = false): string
    {
        $response = '';

        foreach ($this->formGroups as $formGroup) {

            if ($grid && $formGroup->grid() != $grid) {
                continue;
            }

            $response .= $this->renderGroup($formGroup->name());
        }

        return $response;
    }

    /**
     * @param string $group
     * @return string
     */
    public function renderGroup(string $group): string
    {
        $group = $this->formGroups[$group];

        foreach ($this->formFields as $formField) {
            if ($formField->group() != $group->name()) {
                continue;
            }

            if (method_exists($formField, 'translatable') && $formField->translatable() === true) {
                $this->formGroups[$group->name()]->setTranslatable(true);
            }
        }

        $response = $group->render();

        $fields = $this->renderGroupFields($group->name());

        if ($fields == '') {
            return '';
        }

        return sprintf($response, $fields);
    }

    /**
     * @param array $fields
     * @return string
     */
    public function renderOnly(array $fields): string
    {
        return '';
    }

    /**
     * @return string
     */
    public function renderFields(): string
    {
        $response = '';

        foreach ($this->formFields as $formField) {
            $response .= $formField->render();
        }

        return $response;
    }

    /**
     * @param string $group
     * @return string
     */
    public function renderGroupFields(string $group): string
    {
        $response = '';

        foreach ($this->formFields as $formField) {
            if ($formField->group() != $group) {
                continue;
            }

            $response .= $formField->render();
        }

        return $response;
    }

    /**
     * @param string $name
     * @return FieldContract
     */
    public function field(string $name): FieldContract
    {
        return $this->formFields[$name];
    }

    /**
     * @return array
     */
    protected function fields(): array
    {
        return [];
    }

    /**
     * @param FieldContract $field
     * @return $this
     */
    public function addField(FieldContract $field): self
    {
        $this->formFields[$field->name()] = $field;
        return $this;
    }

    /**
     * @param GroupContract $group
     * @return $this
     */
    public function addGroup(GroupContract $group): self
    {
        $this->formGroups[$group->name()] = $group;
        return $this;
    }

    /**
     * @return array[]
     */
    protected function groupAttributes(): array
    {
        return [
            'group1' => [
                'size' => 12,
                'templates' => 'card',
            ],
        ];
    }

    /**
     * @return array[]
     */
    protected function gridAttributes(): array
    {
        return [];
    }

    private function editFieldBeforeValidate(): self
    {
        foreach ($this->formFields as $key => $formField) {
            $this->formFields[$key] = $formField->beforeValidation();
        }

        return $this;
    }

    /**
     * @return array
     */
    protected function rules(): array
    {
        return $this->build()->editFieldBeforeValidate()->mergeRules();
    }

    /**
     * @return array
     */
    protected function mergeRules(): array
    {
        $merged = [];

        $this->selfIgnoreFields();

        foreach ($this->formFields as $formField) {

            $name = $formField->name();

            if ($formField->multiple() || preg_match('/\[\]/', $name)) {
                $name = str_replace('[]', '', $name);
                $name = str_replace('[', '.*', $name);
                $name = str_replace(']', '', $name);
            }

            if ($formField->required()) {
                $merged[$name][] = 'required';
            }

            foreach ($formField->rules() as $rule) {
                $merged[$name][] = $rule;
            }

            foreach ($formField->customRules() as $key => $rules) {
                $merged[$key] = $rules;
            }
        }

        return $merged;
    }

    /**
     * @return array[]
     */
    protected function validationRules(): array
    {
        if ($this->request->all()) {
            return $this->rules();
        }

        return [];
    }

    /**
     * @return array
     */
    public function messages(): array
    {
        return [];
        /*return [
            'first_name' => [],
        ];*/
    }

    /**
     * @return array
     */
    public function attributes(): array
    {
        $attributes = [];
        foreach ($this->formFields as $formField) {
            $attributes[$formField->request_name()] = $formField->title();
        }

        return $attributes;
    }

    /**
     * @return void
     */
    #[NoReturn] protected function prepareForValidation(): void
    {
        $request = $this->filter_trim_recursive_array($this->request->all());

        $this->validationRules();

        foreach ($this->formFields as $field) {
            if (!isset($request[$field->name()])) {
                if (in_array('bool', $field->rules())) {
                    $request[$field->name()] = 0;
                }
            }
        }

        $this->replace($request);
    }

    /**
     * @param array $array
     * @return array
     */
    public function filter_trim_recursive_array(array $array = []): array
    {
        $callback = function ($item) {
            if (is_array($item)) {
                return $this->filter_trim_recursive_array($item);
            } elseif (is_string($item)) {
                return trim($item);
            }
            return $item;
        };

        $array = array_map($callback, $array);
        return array_filter($array);
    }

    /**
     * @param string $title
     * @return self
     */
    public function setTitle(string $title): self
    {
        return $this->setAttribute('title', $title);
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return $this->title;
    }

    /**
     * @param string $id
     * @return self
     */
    public function setId(string $id): self
    {
        return $this->setAttribute('id', $id);
    }

    /**
     * @return string
     */
    public function id(): string
    {
        return $this->id;
    }

    /**
     * @param string $class
     * @return self
     */
    public function setClass(string $class): self
    {
        return $this->setAttribute('class', $class);
    }

    /**
     * @return string
     */
    public function class(): string
    {
        return $this->class;
    }

    /**
     * @param string $action
     * @return self
     */
    public function setAction(string $action, array|bool $attributes = false): self
    {
        if ($attributes) {
            $this->setRouteAttributes($attributes);
        }

        return $this->setAttribute('action', $action);
    }

    /**
     * @return string
     */
    public function action(): string
    {
        return $this->action;
    }

    /**
     * @param string $method
     * @return self
     */
    public function setMethod(string $method): self
    {
        return $this->setAttribute('_method', $method);
    }

    /**
     * @return string
     */
    public function _method(): string
    {
        return $this->_method;
    }

    /**
     * @param array $routes
     * @return self
     */
    public function setRoutes(array $routes): self
    {
        return $this->setAttribute('routes', $routes);
    }

    /**
     * @return array|string[]
     */
    public function routes(): array
    {
        return $this->routes;
    }

    /**
     * @param array $attributes
     * @return self
     */
    public function setRouteAttributes(array $attributes): self
    {
        return $this->setAttribute('routeAttributes', $attributes);
    }

    /**
     * @return array
     */
    public function routeAttributes(): array
    {
        return $this->routeAttributes;
    }

    /**
     * @param string|null $action
     * @param array|null $attributes
     * @return string
     */
    public function _route(string|null $action = null, array|null $attributes = null): string
    {
        if (empty($this->routes)) {
            return '';
        }

        if (empty($this->routes[$action ?? $this->action]['route'])) {
            return '';
        }

        return route($this->routes[$action ?? $this->action]['route'], $attributes ?? $this->routeAttributes);
    }

    /**
     * @param $key
     * @param $default
     * @return mixed
     */
    public function validated($key = null, $default = null): mixed
    {
        $validated = parent::validated($key, $default);

        foreach ($this->formFields as $key => $field) {

            if ($field->disabled()) {
                unset($validated[$key]);
            }

            if ($field->readonly()) {
                unset($validated[$key]);
            }
        }

        return $validated;
    }

    /**
     * Map Laravel rule strings/objects to jQuery Validate rules.
     */
    protected function toJqueryValidate(array $rules): array
    {
        $out = [];

        foreach ($rules as $field => $fieldRules) {
            $fieldRules = is_array($fieldRules) ? $fieldRules : explode('|', $fieldRules);

            $isString  = in_array('string', $fieldRules, true);
            $isNumeric = in_array('numeric', $fieldRules, true) || in_array('integer', $fieldRules, true);

            $cfg = [];

            foreach ($fieldRules as $r) {
                // Normalize object rules (e.g., Rule::unique()) to string tokens
                if ($r instanceof Rule) {
                    $name = class_basename($r);
                    if ($name === 'Unique') {
                        // Client-side: use remote check instead of Unique
                        // You can add remote endpoint below (see comment).
                        $cfg['remote'] = $cfg['remote'] ?? [
                            'url'  => '/api/validate/unique', // implement if needed
                            'type' => 'post',
                            'data' => ['field' => $field],
                        ];
                    }
                    continue;
                }

                // String rule like: required, max:120, min:1, email, regex:/.../
                [$name, $param] = array_pad(explode(':', $r, 2), 2, null);

                switch ($name) {
                    case 'required':
                        $cfg['required'] = true; break;

                    case 'email':
                        $cfg['email'] = true; break;

                    case 'numeric':
                        $cfg['number'] = true; break;

                    case 'integer':
                        $cfg['digits'] = true; break;

                    case 'min':
                        if ($param !== null) {
                            if ($isNumeric) $cfg['min'] = (float)$param;
                            if ($isString)  $cfg['minlength'] = (int)$param;
                        }
                        break;

                    case 'max':
                        if ($param !== null) {
                            if ($isNumeric) $cfg['max'] = (float)$param;
                            if ($isString)  $cfg['maxlength'] = (int)$param;
                        }
                        break;

                    case 'between':
                        if ($param) {
                            [$a,$b] = array_map('trim', explode(',', $param));
                            if ($isNumeric) { $cfg['min'] = (float)$a; $cfg['max'] = (float)$b; }
                            if ($isString)  { $cfg['minlength'] = (int)$a; $cfg['maxlength'] = (int)$b; }
                        }
                        break;

                    case 'regex':
                        // Requires jQuery Validate additional-methods for `pattern`
                        // or add your own method. Expect format: regex:/^...$/
                        if ($param && str_starts_with($param, '/')) {
                            // remove leading/trailing /.../flags
                            $pattern = $this->extractJsPattern($param);
                            if ($pattern) $cfg['pattern'] = $pattern;
                        }
                        break;

                    case 'confirmed':
                        // Laravel expects `<field>_confirmation`
                        $cfg['equalTo'] = "[name='{$field}_confirmation']"; break;

                    // Ignore purely server-side rules client can’t safely emulate
                    // (e.g., 'date', 'exists', 'unique' handled via remote, etc.)
                }
            }

            if (!empty($cfg)) $out[$field] = $cfg;
        }

        return $out;
    }

    /**
     * Map Laravel messages to jQuery Validate message keys (maxlength vs max etc.).
     */
    protected function toJqueryMessages(array $rules, array $messages): array
    {
        $out = [];

        foreach ($rules as $field => $fieldRules) {
            $fieldRules = is_array($fieldRules) ? $fieldRules : explode('|', $fieldRules);
            $isString  = in_array('string', $fieldRules, true);
            $isNumeric = in_array('numeric', $fieldRules, true) || in_array('integer', $fieldRules, true);

            foreach ($fieldRules as $r) {
                [$name, $param] = array_pad(is_string($r) ? explode(':', $r, 2) : [null, null], 2, null);
                if (!$name) continue;

                // Determine jQuery rule key used above
                $jqKey = match ($name) {
                    'required' => 'required',
                    'email'    => 'email',
                    'numeric'  => 'number',
                    'integer'  => 'digits',
                    'min'      => $isNumeric ? 'min'       : 'minlength',
                    'max'      => $isNumeric ? 'max'       : 'maxlength',
                    'between'  => $isNumeric ? 'max'       : 'maxlength', // we’ll only map upper here (optional to expand)
                    'regex'    => 'pattern',
                    'confirmed'=> 'equalTo',
                    default    => null,
                };

                // Laravel message keys may be "field.rule"
                $laravelKey = "{$field}.{$name}";
                if ($jqKey && isset($messages[$laravelKey])) {
                    $out[$field][$jqKey] = $messages[$laravelKey];
                }
            }

            // Fallback generic messages like 'name.required' already covered,
            // but if you also use '*' messages, merge them here as needed.
        }

        return $out;
    }

    /**
     * Convert Laravel-like regex:/.../flags to a JS pattern string (without slashes).
     */
    protected function extractJsPattern(string $token): ?string
    {
        // token like '/^[A-Z0-9-]+$/i'
        if (preg_match('#^/(.*?)/[a-z]*$#', $token, $m)) {
            return $m[1];
        }
        return null;
    }
}
